/*
 * ***************************************************************************
 * MALOC = < Minimal Abstraction Layer for Object-oriented C >
 * Copyright (C) 1994--2000  Michael Holst
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * rcsid="$Id: vec3.h,v 1.9 2002/10/01 21:29:45 mholst Exp $"
 * ***************************************************************************
 */

/*
 * ***************************************************************************
 * File:     vec3.h    < vec3.c >
 *
 * Purpose:  Classes Vec3,Mat3: 3x1 vector and 3x3 dense matrix objects.
 *
 * Author:   Michael Holst
 * ***************************************************************************
 */

#ifndef _VEC3_H_
#define _VEC3_H_

#include <maloc/maloc_base.h>

#include <maloc/vsys.h>

/*
 * ***************************************************************************
 * Class Vec3,Mat3: Parameters and datatypes
 * ***************************************************************************
 */

/*
 * ***************************************************************************
 * Class Vec3,Mat3: Definition
 * ***************************************************************************
 */

typedef double Vec3[3];
typedef double Mat3[3][3];

/*
 * ***************************************************************************
 * Class Vec3,Mat3: Inlineable methods (mat3.c)
 * ***************************************************************************
 */

#if !defined(VINLINE_MALOC)
#else /* if defined(VINLINE_MALOC) */
#endif /* if !defined(VINLINE_MALOC) */

/*
 * ***************************************************************************
 * Class Vec3,Mat3: Non-inlineable methods (mat3.c)
 * ***************************************************************************
 */

VEXTERNC double Vec3_nrm1(Vec3 u);
VEXTERNC double Vec3_nrm2(Vec3 u);
VEXTERNC double Vec3_nrm8(Vec3 u);
VEXTERNC double Vec3_dif1(Vec3 u, Vec3 v);
VEXTERNC double Vec3_dif2(Vec3 u, Vec3 v);
VEXTERNC double Vec3_dif8(Vec3 u, Vec3 v);
VEXTERNC double Vec3_dot(Vec3 u, Vec3 v);

VEXTERNC void Vec3_init(Vec3 u, double val);
VEXTERNC void Vec3_scal(Vec3 u, double val);
VEXTERNC void Vec3_copy(Vec3 u, Vec3 v);
VEXTERNC void Vec3_axpy(Vec3 u, Vec3 v, double val);
VEXTERNC void Vec3_xcry(Vec3 u, Vec3 v, double *w);
VEXTERNC void Vec3_nrmlize(Vec3 u, double scale);
VEXTERNC void Vec3_nrmlizeNE(Vec3 u, double scale);
VEXTERNC void Vec3_print(Vec3 u, const char *name);
VEXTERNC void Vec3_mult(Vec3 u, Mat3 A, Vec3 v);
VEXTERNC void Vec3_getCol(Vec3 u, Mat3 A, int col);
VEXTERNC void Vec3_getRow(Vec3 u, Mat3 A, int row);

VEXTERNC double Mat3_nrm1(Mat3 A);
VEXTERNC double Mat3_nrm2(Mat3 A);
VEXTERNC double Mat3_nrm8(Mat3 A);
VEXTERNC double Mat3_dif1(Mat3 A, Mat3 B);
VEXTERNC double Mat3_dif2(Mat3 A, Mat3 B);
VEXTERNC double Mat3_dif8(Mat3 A, Mat3 B);
VEXTERNC double Mat3_nrm8Low(Mat3 A);

VEXTERNC void Mat3_eye(Mat3 A);
VEXTERNC void Mat3_init(Mat3 A, double val);
VEXTERNC void Mat3_scal(Mat3 A, double val);
VEXTERNC void Mat3_copy(Mat3 A, Mat3 B);
VEXTERNC void Mat3_axpy(Mat3 A, Mat3 B, double val);
VEXTERNC void Mat3_mult(Mat3 C, Mat3 A, Mat3 B);
VEXTERNC void Mat3_putCol(Mat3 A, Vec3 u, int col);
VEXTERNC void Mat3_putRow(Mat3 A, Vec3 u, int row);
VEXTERNC void Mat3_print(Mat3 A, const char *name);
VEXTERNC double Mat3_qri(Mat3 V, Mat3 D, Mat3 A);
VEXTERNC void Mat3_gramSch(Mat3 Q, Mat3 R, Mat3 A);
VEXTERNC void Mat3_qr(Mat3 Q, Mat3 R, Mat3 A);

VEXTERNC double Mat3_det(Mat3 A);
VEXTERNC void Mat3_inverse(Mat3 A, Mat3 Ainv);

#endif /* _VEC3_H_ */

