"""
    Routines for PDB2PQR

    This module contains the protein object used in PDB2PQR and methods
    used to correct, analyze, and optimize that protein.

    Based on C code from Jens Erik Nielsen
    UCSD/HHMI
    
    Ported to Python by Todd Dolinsky (todd@ccb.wustl.edu)
    Washington University in St. Louis

"""

__date__ = "22 October 2003"
__author__ = "Jens Erik Nielsen, Todd Dolinsky"

BUMP_DIST = 2.0
BONDED_SS_LIMIT = 2.5
LARGE_TORSION_ANGLE = 1000.0
PEPTIDE_DIST = 1.7
REPAIR_LIMIT = 10
REFATOM_SIZE = 3
HYDRO_BONDCOORDS = [[7.581,2.090,12.506],[6.458,2.162,13.159],[5.145,2.209,12.453]]
HYDRO_COORDS = [6.476, 2.186, 14.159]
NTERM_COORDS = [[-24.196, 48.790, -20.8], [-25.552, 49.881, -21.848], [-24.645, 49.491, -22.007]]
NTERM2_COORDS = [-24.001, 50.224, -22.226]
NTERM3_COORDS = [-24.869, 48.846, -22.770]
PEP_TRANS_N = [-1.252,1.877,0.883]
PEP_TRANS_CA = [-2.313,2.784,1.023]
OXT_COORDS = [-1.529,1.858,0.695]
HISTIDINES = ["HID","HIE","HIP","HSD","HSE","HSP"]
AAS = ["ALA","ARG","ASN","ASP","CYS","GLN","GLU","GLY","HIS","ILE","LEU",\
       "LYS","MET","PHE","PRO","SER","THR","TRP","TYR","VAL"]

import random
from pdb import *
from utilities import *
from forcefield import *
from structures import *
from protein import *


class Routines:
    def __init__(self, protein, verbose, definition=None):
        """
        """
        self.protein = protein
        self.definition = definition
        self.aadef = None
        self.verbose = verbose
        self.warnings = []
        if definition != None:
            self.aadef = definition.getAA()

    def write(self, message, indent=0):
        """
            Write a message to stderr for debugging if verbose

            Parameters
                message: The message to write (string)
                indent : The indent level (int, default=0)
        """
        out = ""
        if self.verbose:
           for i in range(indent):
               out += "\t"
           out += message
           sys.stderr.write(out)

    def getWarnings(self):
        """
            Get all warnings generated from routines
        """
        return self.warnings
           
    def applyForcefield(self, forcefield):
        """
            Apply the forcefield to the atoms within the protein

            Parameters
                forcefield: The forcefield object (forcefield)
            Returns
                hitlist:    A list of atoms that were found in
                            the forcefield (list)
                misslist:   A list of atoms that were not found in
                            the forcefield (list)
        """
        self.write("Applying the forcefield to the protein...")
        misslist = []
        hitlist = []
        for chain in self.protein.getChains():
            for residue in chain.get("residues"):
                for atom in residue.get("atoms"):
                    atomname = atom.get("name")
                    charge, radius = forcefield.getParams(residue, atomname)
                    if charge != None and radius != None:
                        atom.set("ffcharge", charge)
                        atom.set("radius", radius)
                        hitlist.append(atom)
                    else:
                        misslist.append(atom)  
        self.write("Done.\n")            
        return hitlist, misslist

    def updateResidueTypes(self):
        """
            Find the type of residue as notated in the Amino Acid definition
        """
        self.write("Updating Residue Types... ")
        for chain in self.protein.getChains():
            for residue in chain.get("residues"):
                name = residue.get("name")
                if name in AAS or name in HISTIDINES:
                    residue.set("type",1)
                elif name == "WAT":
                    residue.set("type",3)
                else: # Residue is a ligand or unknown
                    residue.set("type",2)
        self.write("Done\n")
            
    def updateSSbridges(self):
        """
            Check for SS-bridge partners, and if present, set appropriate
            partners
        """
        self.write("Updating SS bridges...\n")
        SGatoms = []
        SGpartners = []
        for atom in self.protein.getAtoms():
            if atom.name == "SG":
                SGatoms.append(atom)
                SGpartners.append([])

        for i in range(len(SGatoms)):
            for j in range(len(SGatoms)):
                dist = distance(SGatoms[i].getCoords(), SGatoms[j].getCoords())
                if i != j and dist < BONDED_SS_LIMIT:
                    SGpartners[i].append(j)
        
        for i in range(len(SGatoms)):
            res1 = SGatoms[i].get("residue")
            if len(SGpartners[i]) == 1:
                partner = SGpartners[i][0]
                if SGpartners[partner][0] == i:
                    res2 = SGatoms[partner].get("residue")
                    if i<partner:
                        self.write("CYS %4d - CYS %4d\n" % \
                                   (res1.get("resSeq"), res2.get("resSeq")), 1)
                    if res1.get("name") == "CYS":
                        res1.set("SSbonded", 1)
                        res1.set("SSbondpartner", SGatoms[partner])
                    else:
                        name = res1.get("name")
                        num = res1.get("resSeq")
                        error = "Fatal Error: tried to set SS bonding "
                        error += "for CYS %i, but residue is a %s!" % \
                                 (num, name)
                        raise ValueError, error
                
                else:
                    raise ValueError, "CYS %i unresolved!" % res1.get("resSeq")
            elif len(SGpartners[i]) > 1:
                error = "CYS %i has multiple potential " % res1.get("resSeq")
                error += "SS-bridge partners"
                raise ValueError, error
            elif len(SGpartners[i]) == 0:
                self.write("CYS %4d is a free cysteine\n" % res1.get("resSeq"), 1)
        self.write("Done.\n")

    def calculateChiangles(self):
        """
            Calculate the dihedral angle for every residue within the protein,
            using the Amino Acid definition.
        """
        self.write("Calculating all chiangles... ")
        for chain in self.protein.getChains():
            for residue in chain.get("residues"):
                residue.set("chiangles",[])
                name = residue.get("name")
                definitionres = self.aadef.getResidue(name)
                if name in HISTIDINES:
                    definitionres = self.aadef.getResidue("HIS")
                if definitionres != None:
                    defdihedrals = definitionres.get("dihedralatoms")
                    for i in range(0, len(defdihedrals), 4):       
                        atom1 = residue.getAtom(defdihedrals[i])
                        atom2 = residue.getAtom(defdihedrals[i+1])
                        atom3 = residue.getAtom(defdihedrals[i+2])
                        atom4 = residue.getAtom(defdihedrals[i+3])
                        
                        if atom1 == None or atom2 == None \
                               or atom3 == None or atom4 == None:
                            residue.addChiangle(LARGE_TORSION_ANGLE)
                        else:
                            residue.addChiangle(getDihedral(atom1.getCoords(),\
                                                            atom2.getCoords(),\
                                                            atom3.getCoords(),\
                                                            atom4.getCoords()))
             
                else:
                    if residue.get("type") == 1:
                        error = "Unable to find Amino Acid definition for "
                        error += "%s!" % name
                        raise ValueError, error
        self.write("Done.\n")

    def updateExtraBonds(self):
        """
            Update peptide bonds between amino acids.
        """
        self.write("Determining peptide bonds and termini... \n")
        for chain in self.protein.getChains():
            for i in range(chain.numResidues() - 1):
                residue1 = chain.get("residues")[i]
                residue2 = chain.get("residues")[i+1]
                if residue1.get("type") == 1 and residue2.get("type") == 1: 
                    atom1 = residue1.getAtom("C")
                    atom2 = residue2.getAtom("N")
                    if atom1 != None and atom2 != None:
                        if distance(atom1.getCoords(),
                                    atom2.getCoords()) < PEPTIDE_DIST:
                            atom1.addExtraBond(atom2)
                            atom2.addExtraBond(atom1)
                        else:
                            self.write("Gap in backbone detected in chain ",1)
                            self.write("%s between %s " % (chain.get("chainID"), \
                                                           residue1.get("name")))
                            self.write("%s and %s %s\n"%(residue1.get("resSeq"),\
                                                         residue2.get("name"),\
                                                         residue2.get("resSeq")))
                if residue1.get("type") == 1 and i == 0:
                    residue1.set("isNterm",1)
                elif residue1.get("type") == 1 and residue2.get("type") != 1 and \
                         residue2.get("name") not in ["ACE","HSP","HSE","HMS"]:
                    residue1.set("isCterm",1)
                elif residue2.get("type") == 1 and i+2 == chain.numResidues():
                    residue2.set("isCterm",1)
        self.write("Done.\n")

    def updateIntraBonds(self):
        """
            Update the bonds within a residue of the protein
        """
        for chain in self.protein.getChains():
            for residue in chain.get("residues"):
                if residue.get("type") != 1: return
                name = residue.get("name")
                defresidue = self.aadef.getResidue(name)
                if name in HISTIDINES:
                    defresidue = self.aadef.getResidue("HIS")
                if defresidue == None:
                    error = "Could not find definition for %s " % name
                    error += "even though it is type 1!"
                    raise ValueError, error
                residue.updateIntraBonds(defresidue)

    def correctNames(self):
        """
            Correct atom names so that they match those listed in
            the amino acid definition.  Handles C-Terminal Oxygens
            and various Hydrogen naming schemes.
        """
        self.write("Correcting all atom names... ")
        for chain in self.protein.getChains():
            for residue in chain.get("residues"):
                if residue.get("type") == 1:
                    residue.checkAtomNames()
                    if residue.get("isCterm") == 1:
                        atom = residue.getAtom("O\'\'")
                        if atom != None:
                            residue.renameAtom("O\'\'", "OXT")
                            self.write("\n")
                            self.write("Renaming O\'\' to OXT\n",1)

                elif residue.get("type") == 2:
                    if residue.get("name") in ["ACE"]: # Acetyl N-Terminus
                        residue.checkAtomNames()
                        
                elif residue.get("type") == 3:
                    residue.checkAtomNames()
                    name = residue.get("name")
                 
                    for atom in residue.get("atoms"):
                        atomname = atom.get("name")
                        if atomname == "O":
                            atom = residue.getAtom("O")
                            if atom == None:
                                error = "\tCannot Repair Water when " \
                                        "Oxygen is missing!: %s %i\n" % \
                                        (name, resSeq)
                                raise ValueError, error
                            
                else:   #residue is an unknown type
                    raise ValueError, "Unknown residue type!"

        self.write("Done.\n")

