#!/usr/bin/python2 -O

"""
    Driver for PDB2PQR

    This module takes a PDB file as input and performs optimizations
    before yielding a new PDB-style file as output.

    Ported to Python by Todd Dolinsky (todd@ccb.wustl.edu)
    Washington University in St. Louis

    Parsing utilities provided by Nathan A. Baker (baker@biochem.wustl.edu)
    Washington University in St. Louis
"""

__date__ = "23 September 2003"
__author__ = "Todd Dolinsky, Nathan Baker"

import string
import sys
import getopt
import os
import time
from pdb import *
from utilities import *
from structures import *
from forcefield import *
from routines import *
from protein import *
from StringIO import *

def usage(rc):
    """
        Print usage for this script to stdout.

        Parameters
            rc:  Exit status (int)
    """

    str = "\n"
    str = str + "pdb2pqr\n"
    str = str + "\n"
    str = str + "This module takes a PDB file as input and performs\n"
    str = str + "optimizations before yielding a new PDB-style file as\n"
    str = str + "output\n"
    str = str + "\n"
    str = str + "Usage: pdb2pqr.py [options] --ff=<forcefield> <path> [output-path]\n"
    str = str + "    Required Arguments:\n"
    str = str + "        <path>        :  The path to the PDB file or an ID\n"
    str = str + "                         to obtain from the PDB archive\n"
    str = str + "        <forcefield>  :  The forcefield to use - currently\n"
    str = str + "                         amber, charmm, and parse are supported.\n"
    str = str + "    Optional Arguments:\n"
    str = str + "        --verbose (-v):  Print information to stdout\n"
    str = str + "        --help    (-h):  Display the usage information\n"
    str = str + "    If no output-path is specified, the PQR file is\n"
    str = str + "    printed to stdout\n"
    str = str + "\n"
    sys.stderr.write(str)
    sys.exit(rc)

def printHeader(atomlist, reslist, charge, ff, warnings):
    """
        Print the header for the PQR file

        Parameters:
            atomlist: A list of atoms that were unable to have
                      charges assigned (list)
            reslist:  A list of residues with non-integral charges
                      (list)
            charge:   The total charge on the protein (float)
            ff:       The forcefield name (string)
            warnings: A list of warnings generated from routines (list)
        Returns
            header:   The header for the PQR file (string)
    """
    header = "REMARK   1 PQR file generated by PDB2PQR\n"
    header = header + "REMARK   1\n"
    header = header + "REMARK   1 Forcefield Used: %s\n" % ff
    header = header + "REMARK   1\n"

    for warning in warnings:
        header = header + "REMARK   5 " + warning 
    header = header + "REMARK   5\n"
    
    if len(atomlist) != 0:
        header += "REMARK   5 WARNING: PDB2PQR was unable to assign charges\n"
        header += "REMARK   5          to the following atoms (omitted below):\n"
        for atom in atomlist:
            header += "REMARK   5              %i %s in %s %i\n" % \
                      (atom.get("serial"), atom.get("name"), \
                       atom.get("residue").get("name"), \
                       atom.get("residue").get("resSeq"))
        header += "REMARK   5\n"
    if len(reslist) != 0:
        header += "REMARK   5 WARNING: Non-integral net charges were found in\n"
        header += "REMARK   5          the following residues:\n"
        for residue in reslist:
            header += "REMARK   5              %s %i - Residue Charge: %.4f\n" % \
                      (residue.get("name"), residue.get("resSeq"), \
                       residue.getCharge())
        header += "REMARK   5\n"
    header += "REMARK   6 Total charge on this protein: %.4f e\n" % charge
    header += "REMARK   6\n"

    return header
  
def runPDB2PQR(pdblist, verbose, ff):
    """
        Run the PDB2PQR Suite

        Parameters
            pdblist: The list of objects that was read from the PDB file
                     given as input (list)
            verbose: When 1, script will print information to stdout
                     When 0, no detailed information will be printed (int)
        Returns
            header:  The PQR file header (string)
            lines:   The PQR file atoms (list)
    """
    lines = []
    
    start = time.time()

    if verbose:
        print "Beginning PDB2PQR...\n"

    myProtein = Protein(pdblist)
    if verbose:
        print "Created protein object -"
        print "\tNumber of residues in protein: %s" % myProtein.numResidues()
        print "\tNumber of atoms in protein   : %s" % myProtein.numAtoms()


    myRoutines = Routines(myProtein, verbose)
    myRoutines.updateResidueTypes()
    myRoutines.updateSSbridges()
    myRoutines.updateExtraBonds()
    myRoutines.correctNames()

    myForcefield = Forcefield(ff)
    hitlist, misslist = myRoutines.applyForcefield(myForcefield)
    reslist, charge = myProtein.getCharge()

    header = printHeader(misslist, reslist, charge, ff, myRoutines.getWarnings())

    lines = myProtein.printAtoms(hitlist)
    if verbose:
        print "Total time taken: %.2f seconds\n" % (time.time() - start)

    return header, lines

def mainCommand():
    """
        Main driver for running program from the command line.
    """
    shortOptlist = "h,v"
    longOptlist = ["help","verbose","ff="]

    try: opts, args = getopt.getopt(sys.argv[1:], shortOptlist, longOptlist)
    except getopt.GetoptError, details:
        sys.stderr.write("GetoptError:  %s\n" % details)
        usage(2)

    if len(args) < 1 or len(args) > 2:
        sys.stderr.write("Incorrect number (%d) of arguments!\n" % len(args))
        usage(2)

    verbose = 0
    outpath = None
    ff = None
    debump = 1
    for o,a in opts:
        if o in ("-v","--verbose"):
            verbose = 1
        elif o in ("-h","--help"):
            usage(2)
            sys.exit()
        elif o == "--ff":
            if a in ["amber","AMBER","charmm","CHARMM","parse","PARSE"]:
                ff = string.lower(a)
            else:
                raise ValueError, "Invalid forcefield %s!" % a

    if ff == None:
        raise ValueError, "Forcefield not specified!"
            
    path = args[0]
    file = getFile(path)
    pdblist, errlist = readPDB(file)
    
    if len(pdblist) == 0 and len(errlist) == 0:
        print "Unable to find file %s!\n" % path
        os.remove(path)
        sys.exit(2)

    if len(errlist) != 0 and verbose:
        print "Warning: %s is a non-standard PDB file.\n" % path
        print errlist

    header, lines = runPDB2PQR(pdblist, verbose, ff)

    if len(args) == 2:
        outpath = args[1]
        outfile = open(outpath,"w")
        outfile.write(header)
        for line in lines:
            outfile.write(line)
        outfile.close()
    else:
        print header
        for line in lines:
            print line

def mainCGI():
    """
        Main driver for running PDB2PQR from a web page
    """
    import cgi
    import cgitb

    cgitb.enable()
    form = cgi.FieldStorage()

    pdbid = form["PDBID"].value
    pdb = form["PDB"].value
    ff = form["FF"].value
 
    if pdbid == "":
        file = StringIO(pdb)
    else:
        file = getFile(pdbid)

    pdblist, errlist = readPDB(file)    
    if len(pdblist) == 0 and len(errlist) == 0:
        print "Unable to find file %s!\n" % path
        os.remove(path)
        sys.exit(2)

    try:
        header, lines = runPDB2PQR(pdblist, 0, ff)
        print "Content-type: file\n"
        print header
        for line in lines:
            print string.strip(line)
        
    except StandardError, details:
        print "Content-type: text/html\n"
        print details
        
    
if __name__ == "__main__":
    """ Determine if called from command line or CGI """
    
    if not os.environ.has_key("REQUEST_METHOD"): mainCommand()    
    else:
        mainCGI()
