/**
 *  @file    femparse.h
 *  @ingroup NOsh
 *  @author  Nathan Baker
 *  @brief   Class NOsh parsing methods
 *  @version $Id: femparse.c,v 2.2 2004/01/19 21:01:36 apbs Exp $
 *  @attention
 *  @verbatim
 *
 * APBS -- Adaptive Poisson-Boltzmann Solver
 *
 * Nathan A. Baker (baker@biochem.wustl.edu)
 * Dept. of Biochemistry and Molecular Biophysics
 * Center for Computational Biology
 * Washington University in St. Louis
 *
 * Additional contributing authors listed in the code documentation.
 *
 * Copyright (c) 2002-2004.  Washington University in St. Louis.
 * All Rights Reserved.
 * Portions Copyright (c) 1999-2002.  The Regents of the University of
 * California.  
 * Portions Copyright (c) 1995.  Michael Holst.
 *
 * This file is part of APBS.
 *
 * APBS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * APBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with APBS; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 * @endverbatim
 */

#include "apbscfg.h"
#include "apbs/apbs.h"
#include "apbs/femparm.h"
#include "apbs/nosh.h"
#include "apbs/vstring.h"

/**
 * @brief  Set up a fe-manual calculation
 * @ingroup  NOsh
 * @author  Nathan Baker
 * @param  thee  NOsh (master) object
 * @param  feparm  FE parameter object
 * @param  pbeparm  PBE parameter object
 * @return  1 if successful, 0 otherwise */
VPRIVATE int NOsh_setupFEMANUAL(NOsh *thee, FEMparm *feparm, PBEparm *pbeparm);

VPUBLIC int NOsh_parseFEM(NOsh *thee, Vio *sock, FEMparm_CalcType type) {

    char tok[VMAX_BUFSIZE];
    FEMparm *feparm = VNULL;    
    PBEparm *pbeparm = VNULL;    
    int rc;
    /* Check the arguments */
    if (thee == VNULL) {
        Vnm_print(2, "NOsh_parseFEM:  Got NULL thee!\n");
        return 0;
    }
    if (sock == VNULL) {
        Vnm_print(2, "NOsh_parseFEM:  Got pointer to NULL socket!\n");
        return 0;
    }

    Vnm_print(0, "NOsh_parseFEM: Parsing parameters for FEM calculation\n");

    feparm  = FEMparm_ctor(type);
    pbeparm = PBEparm_ctor();

    /* Start snarfing tokens from the input stream */
    rc = 1;
    while (Vio_scanf(sock, "%s", tok) == 1) { 

        Vnm_print(0, "NOsh_parseFEM:  Parsing %s...\n", tok);

        /* See if it's an END token */
        if (Vstring_strcasecmp(tok, "end") == 0) {
            feparm->parsed = 1;
            pbeparm->parsed = 1;
            rc = 1;
            break;
        }

        /* Pass the token through a series of parsers */
        rc = PBEparm_parseToken(pbeparm, tok, sock);
        if (rc == -1) {
            Vnm_print(0, "NOsh_parseFEM:  parsePBE error!\n");
            break;
        } else if (rc == 0) {
            /* Pass the token to the generic MG parser */
            rc = FEMparm_parseToken(feparm, tok, sock);
            if (rc == -1) { 
               Vnm_print(0, "NOsh_parseFEM:  parseMG error!\n");
               break;
            } else if (rc == 0) {
                /* We ran out of parsers! */
                Vnm_print(2, "NOsh:  Unrecognized keyword: %s\n", tok);
                break;
            }
        }
    }

    /* Handle various errors arising in the token-snarfing loop -- these all
     * just result in simple returns right now */
    if (rc == -1) return 0;
    if (rc == 0) return 0;
   
    /* Check the status of the parameter objects */
    if ((!FEMparm_check(feparm)) || (!PBEparm_check(pbeparm))) {
        Vnm_print(2, "NOsh:  FEM parameters not set correctly!\n");
        return 0;
    }
   
    /* Now we're ready to whatever sorts of post-processing operations that are
     * necessary for the various types of calculations */
    if (type == FCT_MAN) return NOsh_setupFEMANUAL(thee, feparm, pbeparm);

    /* Should never get here */
    return 0;

}

VPUBLIC int NOsh_setupFEMANUAL(NOsh *thee, FEMparm *feparm, PBEparm *pbeparm) {

    if (thee == VNULL) {
        Vnm_print(2, "NOsh_setupFEMANUAL:  Got NULL thee!\n");
        return 0;
    }
    if (feparm == VNULL) {
        Vnm_print(2, "NOsh_setupFEMANUAL:  Got NULL feparm!\n");
        return 0;
    }
    if (pbeparm == VNULL) {
        Vnm_print(2, "NOsh_setupFEMANUAL:  Got NULL pbeparm!\n");
        return 0;
    }

    /* Check to see if he have any room left for this type of
     * calculation, if so: set the calculation type, update the number
     * of calculations of this type, and parse the rest of the section
     */
    if (thee->ncalc >= NOSH_MAXCALC) {
        Vnm_print(2, "NOsh:  Too many calculations in this run!\n");
        Vnm_print(2, "NOsh:  Current max is %d; ignoring this calculation\n",
          NOSH_MAXCALC);
        return 0;
    }

    /* Increment the number of calculations */
    (thee->ncalc)++;

    /* Set the calculation type to multigrid */
    thee->calc[thee->ncalc-1].calctype = 1;

    /* Associate ELEC statement with the calculation */
    thee->elec2calc[thee->nelec-1] = thee->ncalc-1;

    thee->calc[thee->ncalc-1].femparm = feparm;
    thee->calc[thee->ncalc-1].pbeparm = pbeparm;

    return 1;
}
