C
      PROGRAM SHELXE
C
C              Phasing and density modification
C              ================================
C
C       ** FORTRAN-95 version 2006/3 - February 2006 **
C
C IH(), IK() and IL() are the reflection indices h, k and l. FA() and
C SA() are experimental FA values and sigma(FA), FH() and PH() are the
C calculated heavy atom structure factor and phase in degrees. PD() is.
C the experimental phase shift (Phi(T)-Phi(A)); this is 90/270 for SAD
C phases (90 if F+>F-), 360/180 for SIR (360 if FPH>FP), and in the
C full range 1-360 degrees for MAD and SIRAS. FT() and ST() are the
C native Fo and sigma. SQ() stores 1/d^2 = (2sin(theta)/lambda)^2,
C EP() holds Sqrt(epsilon). EH(), EA() and ET() are normalized
C structure factors corresponding to FH(), FA() and FT(). PT() is the
C current protein phase. PS() is initial centroid protein phase and
C WS() is its figure of merit. IN flags omit status of a reflection
C for density modification. Atoms are stored in XA(), YA() and ZA()
C (crystal coordinates) with occupancies in OC(). A(2-7) holds the
C cell in Angstroms and degrees, A(8-13) are the coefficients for
C calculating distances, A(14-19) the coefficients for calculating
C 1/d^2 = (2sin(theta)/lambda)^2, A(20) the SHELX lattice code and
C A(21-(LY+11)) the symmetry operators as a 3x3 matrix followed by
C 3x1 vector. The rest of A is used as working space. LM is the
C maximum number of heavy atoms allowed (10000 should be adequate).
C Large arrays including the two FFT grids B and C are allocated
C dynamically, the maximum number of reflections N (which includes
C several working lists) may be set by the command line flag -lN
C (N is in units of 1000000). SHELXE has following modes of action:
C
C JP=-1: shelxe xx [reads xx.hkl and xx.ins, phases from atoms]
C JP=0: shelxe xx yy [normal mode: reads xx.hkl, yy.hkl, yy.res]
C JP=1: shelxe xx.phi [reads xx.phi, xx.hkl, xx.ins]
C JP=2: shelxe xx.fcf [reads only xx.fcf]
C JP=3: shelxe xx.phi yy [reads xx.phi, xx.hkl, xx.ins, yy.hkl]
C JP=4(later 5): shelxe xx.fcf yy [reads xx.fcf, yy.hkl, yy.res]
C
      PARAMETER(LM=1000000)
      REAL::XA(LM),YA(LM),ZA(LM),OC(LM),A(1000),D(67),E(23),BF(18)
      INTEGER::IA(8),IX(92),IY(92),IZ(92)
      INTEGER,ALLOCATABLE,DIMENSION(:)::IH,IK,IL,IN,IJ
      REAL,ALLOCATABLE,DIMENSION(:)::FA,SA,PD,FT,ST,FH,PH,PS,WS,WM,B,C
      REAL,ALLOCATABLE,DIMENSION(:)::SQ,EP,EH,EA,ET,PT,WA,WB,CP,PM,F
      CHARACTER::KR*80,KD*80,KA*80,KT*80,KK*80,KF*80,KU*80,KX(3)*20
      CHARACTER::KQ*4,KG(65)*4,KM(12)*3,KS*1,KE(94)*2
      COMMON/FFT/IQ,N1,N2,N3,N4,N5,N6,N7,AZ,EZ,RA,RC,RG,RQ,RS,RV,JX,
     +JP,US,VS,LI,LP,NA,IX,IY,IZ,A
      COMMON/SFUN/KF,KU
      DATA BF/20.83913,0.,37.37737,180.,42.59582,0.,79.18768,180.,
     +63.43495,0.,58.91670,-23.26954,58.91670,23.26954,
     +80.76185,-12.23077,80.76185,12.23077/
      DATA KM/'Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep',
     +'Oct','Nov','Dec'/
      DATA KG/'ACTA','AFIX','ANIS','BASF','BIND','BLOC','BOND','BUMP',
     +'CGLS','CHIV','CONF','CONN','DAMP','DANG','DEFS','DELU','DFIX',
     +'DISP','EADP','EQIV','ESEL','EXTI','EXYZ','FLAT','FMAP','FREE',
     +'FVAR','GRID','HFIX','HOPE','HTAB','ILSF','ISOR','LAUE','LIST',
     +'L.S.','MERG','MOLE','MORE','MPLA','NCSY','OMIT','PLAN','REM ',
     +'RESI','RTAB','SADI','SAME','SHEL','SIMU','SIZE','SKIP','SLIM',
     +'SPEC','SUMP','SWAT','TEMP','TEXP','TWIN','UNIT','WGHT','WPDB',
     +'STIR','UNIT','ZERR'/
      DATA KE/'H ','HE','LI','BE','B ','C ','N ','O ','F ','NE','NA',
     +'MG','AL','SI','P ','S ','CL','AR','K ','CA','SC','TI','V ',
     +'CR','MN','FE','CO','NI','CU','ZN','GA','GE','AS','SE','BR',
     +'KR','RB','SR','Y ','ZR','NB','MO','TC','RU','RH','PD','AG',
     +'CD','IN','SN','SB','TE','I ','XE','CS','BA','LA','CE','PR',
     +'ND','PM','SM','EU','GD','TB','DY','HO','ER','TM','YB','LU',
     +'HF','TA','W ','RE','OS','IR','PT','AU','HG','TL','PB','BI',
     +'PO','AT','RN','FR','RA','AC','TH','PA','U ','NP','PU'/
C
   1  FORMAT(/'  ',68('+')/'  +  SHELXE  -  PHASING AND DENSITY ',
     +'MODIFICATION  -  Version 2006/3  +'/'  +  Copyright (C) ',
     +' George M. Sheldrick 2001-6',23X,'+'/'  +  Started at ',A2,
     +':',A2,':',A2,' on ',A2,1X,A3,1X,A4,28X,'  +'/'  ',68('+')/)
C
C Set units for file input/output
C
      LR=2
      LP=3
      LI=4
C
C Output header
C
      CALL DATE_AND_TIME(KD,KK,KA,IA)
      WRITE(*,1)KK(1:2),KK(3:4),KK(5:6),KD(7:8),KM(IA(2)),KD(1:4)
C
C Default parameters
C
      KT=' '
      KA=' '
      KF='SFAC SE'
      KU='UNIT 400'
      RB=-2.
      RC=0.4
      RD=0.0
      RE=-1.
      JE=999999
      JF=0
      RG=1.1
      JH=0
      JI=0
      JJ=5
      JL=2
      JM=0
      JN=0
      JP=0
      RQ=4.5
      RR=3.0
      RS=0.45
      RV=1.5
      RW=0.2
      JX=0
      JZ=0
      NR=0
      IQ=1
      NE=0
      LL=0
C
C Interpret command line
C
      M=0
        DO 4 I=1,IARGC()
        CALL GETARG(I,KR)
        IF(KR(1:1).EQ.'-')GOTO 2
        M=M+1
        IF(M.EQ.1)KT=KR
        IF(M.EQ.2)KA=KR
        IF(M.GT.2)GOTO 7
        GOTO 4
   2    KS=KR(2:2)
        IF(KS.GE.'A'.AND.KS.LE.'Z')KS=CHAR(ICHAR(KS)+32)
        IF(INDEX('bcdefghijlmnqrsvwxz',KS).EQ.0)GOTO 7
        IF(KS.EQ.'b')RB=5.
        IF(KS.EQ.'i')JI=1
        IF(KS.EQ.'j')JJ=1
        IF(KS.EQ.'f')JF=1
        IF(KS.EQ.'h')JH=999999
        IF(KS.EQ.'n')JN=1
        IF(KS.EQ.'x')JX=1
        IF(KS.EQ.'z')JZ=1
        IF(KR(3:3).LE.' ')GOTO 4
        IF(INDEX('hjlmx',KS).EQ.0)GOTO 3
        READ(KR(3:80),*,ERR=4)N
        IF(KS.EQ.'h')JH=N
        IF(KS.EQ.'j')JJ=N
        IF(KS.EQ.'l')JL=N
        IF(KS.EQ.'m')JM=N
        IF(KS.EQ.'x')JX=N
        GOTO 4
   3    READ(KR(3:80),*,ERR=4)T
        IF(KS.EQ.'b')RB=T
        IF(KS.EQ.'c')RC=T
        IF(KS.EQ.'d')RD=T
        IF(KS.EQ.'e')RE=T
        IF(KS.EQ.'g')RG=T
        IF(KS.EQ.'r')RR=T
        IF(KS.EQ.'s')RS=T
        IF(KS.EQ.'v')RV=T
        IF(KS.EQ.'w')RW=T
   4    CONTINUE
      IF(M.EQ.0)GOTO 8
      LL=INDEX(KT,' ')-1
      IF(M.EQ.2)GOTO 5
      IF(RB.GT.-1.)GOTO 7
      JP=-1
      IF(LL.LT.5)GOTO 17
      IF(KT(LL-3:LL).EQ.'.phi')JP=1
      IF(KT(LL-3:LL).EQ.'.fcf')JP=2
      IF(JP.GT.0)LL=LL-4
      GOTO 17
   5  IF(LL.LT.5)GOTO 6
      IF(KT(LL-3:LL).EQ.'.phi')JP=3
      IF(KT(LL-3:LL).EQ.'.fcf')JP=4
      IF(JP.LT.3)GOTO 6
      IF(RB.LT.-1.)RB=5.
      LL=LL-4
   6  IF(KA.NE.KT)GOTO 17
   7  M=1
   8  WRITE(*,13)
      WRITE(*,14)
      WRITE(*,15)
      WRITE(*,16)
      IF(M.NE.0)WRITE(*,'(A/)')' ** Bad command line **'
      CALL EXIT(0)
      STOP' '
   9  WRITE(*,'(/A/)')' ** Unable to open file '//KR(1:N)//' **'
      CALL EXIT(0)
      STOP' '
C
C Output useful information for running the program
C
  10  FORMAT(' Listing file:  ',A/' Phase output file: ',A)
  11  FORMAT(' Revised heavy atom sites: ',A/
     +' Revised heavy atom phases: ',A)
  12  FORMAT(/' Summary of parameters to be employed:'/)
  13  FORMAT(' SHELXE has following modes of action (xx and yy',
     +' are filename stems):'//
     +' shelxe xx [reads xx.hkl and xx.ins, phases from atoms]'/
     +' shelxe xx yy [normal mode: reads xx.hkl, yy.hkl, yy.res]'/
     +' shelxe xx.phi [reads xx.phi, xx.hkl, xx.ins]'/
     +' shelxe xx.fcf [reads only xx.fcf]'/
     +' shelxe xx.phi yy [reads xx.phi, xx.hkl, xx.ins, yy.hkl]'/
     +' shelxe xx.fcf yy [reads xx.fcf, yy.hkl, yy.res]'//
     +' xx.hkl contains native data, yy.hkl contains FA and alpha',
     +' and should have'/' been created using SHELXC or XPREP.',
     +' xx.phi has .phs format but only the'/' starting phases are',
     +' read from it; if a .phi file is read, the cell and'/
     +' symmetry are read from xx.ins and the native F-values',
     +' are read from xx.hkl.'/' xx.fcf (from a SHELXL structure',
     +' refinement) provides cell, symmetry and'/' starting phases.',
     +' The output phases are written to xx.phs, the log file is'/
     +' written to xx.lst and, if -b is set, improved substructure',
     +' phases are'/' output to xx.pha and revised heavy atoms to',
     +' xx.hat.'//' The first two modes provide density modification',
     +' starting from atoms or')
  14  FORMAT(' phases, the third and',
     +' fourth modes are for phase extension, the fifth is'/
     +' an inverse cross-Fourier for finding heavy atoms for',
     +' a second derivative'/' with the same origin as the first,',
     +' and the last mode is useful to confirm'/
     +' the heavy atom substructure from the final refined',
     +' phases. For these last'/' two modes, the phases for the',
     +' inverse Fourier are phi(nat)-alpha, where '/' phi(nat)',
     +' may be refined (-m etc.) and alpha is taken from yy.hkl.'//
     +' SHELXE normally requires a few command line switches, e.g.'//
     +' shelxe xx yy -m20 -s0.45 -h8 -b'//' would do 20 cycles',
     +' density modification with a solvent content of 0.45,'/
     +' phasing from the first 8 heavy atoms in the yy.res',
     +' file assuming that they'/' are also present in the native',
     +' structure (-h8) and then use the modified'/' density to',
     +' generate improved heavy atoms (-b). The switch -i may be',
     +' added'/' to invert the substructure (and if necessary the',
     +' space group), this writes')
  15  FORMAT(' xx_i.phs instead of xx.phs',
     +' etc., and so may be run in parallel. The new'/' switch -e',
     +' may be used to extrapolate the data to the specified',
     +' resolution'/' (the ''free lunch algorithm''); -e1.0 can',
     +' produce spectacular results, but'/' since a large number',
     +' of cycles is required (-m400) and the ''contrast'' and'/
     +' ''connectivity'' become unreliable, it is best to establish',
     +' the substructure'/' enantiomorph without -e first. The',
     +' current values of all parameters are'/' output at the',
     +' start of the shelxe output, the default values will rarely'/
     +' need changing.'//' Please cite: G.M. Sheldrick (2002),',
     +' "Macromolecular phasing with SHELXE",'/' Z. Kristallogr.',
     +' 217, 644-650 whenever SHELXE proves useful.'/)
  16  FORMAT(1X,78('=')/)
  17  WRITE(*,13)
      WRITE(*,14)
      WRITE(*,15)
      WRITE(*,16)
C
C Instruction summary to console
C
      KR=KT(1:LL)//'.lst'
      N=LL+4
      LJ=INDEX(KA,' ')-1
      IF(JI.EQ.0)GOTO 18
      IF(JP.NE.0)GOTO 7
      KR=KT(1:LL)//'_i.lst'
      N=N+2
  18  IF(JP.EQ.-1)WRITE(*,'(A)')' Cell, symmetry and '
     +//'heavy atoms from:  '//KT(1:LL)//'.ins'
      IF(JP.EQ.2.OR.JP.EQ.4)WRITE(*,'(A)')' Cell, symmetry, data '
     +//'and phases from:  '//KT(1:LL+4)
      IF(JP.EQ.1.OR.JP.EQ.3)WRITE(*,'(A/A)')' Cell and symmetry only'
     +//' from:  '//KT(1:LL)//'.ins',' Phases from:  '//KT(1:LL+4)
      IF(JP.EQ.0)WRITE(*,'(A)')' Cell, symmetry and '
     +//'heavy atoms from:  '//KA(1:LJ)//'.res'
      IF(JP.EQ.0.OR.JP.GT.2)WRITE(*,'(A)')' DelF from:  '//
     +KA(1:LJ)//'.hkl'
      IF(JP.LT.2)WRITE(*,'(A)')' Native data from:  '//
     +KT(1:LL)//'.hkl'
      WRITE(*,10)KR(1:N),KR(1:LL)//'.phs'
      IF(RB.GT.-1.)WRITE(*,11)KR(1:LL)//'.hat',KR(1:LL)//'.pha'
      WRITE(*,12)
      IF(JP.GT.1)GOTO 19
      IF(JF.EQ.1)WRITE(*,'(A)')' -f read F instead of intensity from'
     +//' native .hkl file'
      IF(JF.EQ.0)WRITE(*,'(A)')' -f NOT SET: read intensity not F from'
     +//' native .hkl file'
      IF(JP.GT.0)GOTO 19
      IF(JI.EQ.1)WRITE(*,'(A)')' -i invert HA structure '//
     +'(and space group)'
      IF(JI.EQ.0)WRITE(*,'(A)')' -i NOT SET: no structure inversion'
      IF(JP.LT.0)GOTO 19
      IF(JN.EQ.0)WRITE(*,'(A)')' -n NOT SET: resolve 2-fold ambiguity '
     +//'if SAD or SIR'
      IF(JN.NE.0)WRITE(*,'(A)')' -n do not resolve 2-fold ambiguity'
      IF(JH.GT.999998)WRITE(*,'(A)')' -h heavy atoms present in '
     +//'native - use all'
      IF(JH.GT.0.AND.JH.LT.999999)WRITE(*,'(A,I5)')' -h heavy atoms '
     +//'present in native - use first',JH
      IF(JH.EQ.0)WRITE(*,'(A)')' -h NOT SET: heavy atoms not '
     +//'present in native'
      IF(RB.LT.-1.)WRITE(*,'(A)')' -b NOT SET: do not output revised '
     +//'heavy atom sites and phases'
  19  IF(JZ.EQ.1)WRITE(*,'(A)')' -z do not sharpen density maps'
      IF(JZ.EQ.0)WRITE(*,'(A)')' -z NOT SET: sharpen density maps'
      IF(JJ.EQ.0)WRITE(*,'(A)')' -j NOT SET: do not output '//
     +'pseudo-free CC except at end'
      IF(RE.LT.0.)WRITE(*,'(A)')' -e NOT SET: fill in missing data '
     +//'up to maximum resolution + 0.2 Angstroms'
      WRITE(*,'(/A,F6.3,A)')' -d',RD,'  high resolution limit to be'
     +//' applied to input data'
      IF(RE.GE.0.)WRITE(*,'(A,F6.3,A)')' -e',RE,'  high resolution'
     +//' limit for data extrapolation'
      WRITE(*,'(A,I6,A)')' -m',JM,'  cycles of density modification'
      IF(JJ.GT.0)WRITE(*,'(A,I6,A)')' -j',JJ,'  output pseudo-free CC'
     +//' before every Nth cycle'
      WRITE(*,'(A,F6.3,A)')' -s',RS,'  solvent fraction'
      WRITE(*,'(A,F6.3,A)')' -c',RC,'  fraction of pixels in '//
     +'crossover region'
      WRITE(*,'(A,F6.3,A)')' -g',RG,'  solvent gamma flipping factor'
      WRITE(*,'(A,F6.3,A)')' -v',RV,'  low density reduction factor'
      WRITE(*,'(A,F6.3,A)')' -w',RW,'  weight for retained '//
     +'experimental phases'
      WRITE(*,'(A,F6.2,A)')' -r',RR,'  map resolution (multiplies '//
     +'maximum indices)'
      IF(RB.LT.-1.)GOTO 20
      WRITE(*,'(A,F6.1,A)')' -b',RB,'  extra B for '//
     +'revised heavy atom sites'
      WRITE(*,'(A,F6.1,A)')' -q',RQ,'  minimum height/sigma for '//
     +'revised heavy atom sites'
  20  WRITE(*,'(A,I6,A,I9,A/)')' -l',JL,'  space for',
     +MAX0(JL*1000000,500000),' reflections'
C
C Open listing file and output parameter summary to it
C
      OPEN(UNIT=LI,FILE=KR(1:N),STATUS='OLD',IOSTAT=I)
      CLOSE(UNIT=LI,STATUS='DELETE',IOSTAT=I)
      OPEN(UNIT=LI,FILE=KR(1:N),STATUS='NEW',ERR=9)
      WRITE(LI,1)KK(1:2),KK(3:4),KK(5:6),KD(7:8),KM(IA(2)),KD(1:4)
      WRITE(LI,13)
      WRITE(LI,14)
      WRITE(LI,15)
      WRITE(LI,16)
      IF(JP.EQ.-1)WRITE(LI,'(A)')' Cell, symmetry and '
     +//'heavy atoms from:  '//KA(1:LJ)//'.ins'
      IF(JP.EQ.2.OR.JP.EQ.4)WRITE(LI,'(A)')' Cell, symmetry, data '
     +//'and phases from:  '//KT(1:LL+4)
      IF(JP.EQ.1.OR.JP.EQ.3)WRITE(LI,'(A/A)')' Cell and symmetry only'
     +//' from:  '//KT(1:LL)//'.ins',' Phases from:  '//KT(1:LL+4)
      IF(JP.EQ.0)WRITE(LI,'(A)')' Cell, symmetry and '
     +//'heavy atoms from:  '//KA(1:LJ)//'.res'
      IF(JP.EQ.0.OR.JP.EQ.3)WRITE(LI,'(A)')' DelF from:  '//
     +KA(1:LJ)//'.hkl'
      IF(JP.LT.2)WRITE(LI,'(A)')' Native data from:  '//
     +KT(1:LL)//'.hkl'
      WRITE(LI,10)KR(1:N),KR(1:LL)//'.phs'
      IF(RB.GT.-1.)WRITE(LI,11)KR(1:LL)//'.hat',KR(1:LL)//'.pha'
      WRITE(LI,12)
      IF(JP.GT.1)GOTO 21
      IF(JF.EQ.1)WRITE(LI,'(A)')' -f read F instead of intensity from'
     +//' native .hkl file'
      IF(JF.EQ.0)WRITE(LI,'(A)')' -f NOT SET: read intensity not F from'
     +//' native .hkl file'
      IF(JP.GT.0)GOTO 21
      IF(JI.EQ.1)WRITE(LI,'(A)')' -i invert HA structure '//
     +'(and space group)'
      IF(JI.EQ.0)WRITE(LI,'(A)')' -i NOT SET: no structure inversion'
      IF(JP.LT.0)GOTO 21
      IF(JN.EQ.0)WRITE(LI,'(A)')' -n NOT SET: resolve 2-fold ambiguity '
     +//'if SAD or SIR'
      IF(JN.NE.0)WRITE(LI,'(A)')' -n do not resolve 2-fold ambiguity'
      IF(JH.GT.999998)WRITE(LI,'(A)')' -h heavy atoms present in '
     +//'native - use all'
      IF(JH.GT.0.AND.JH.LT.999999)WRITE(LI,'(A,I5)')' -h heavy atoms '
     +//'present in native - use first',JH
      IF(JH.EQ.0)WRITE(LI,'(A)')' -h NOT SET: heavy atoms not '
     +//'present in native'
      IF(RB.LT.-1.)WRITE(LI,'(A)')' -b NOT SET: do not output revised '
     +//'heavy atom sites and phases'
  21  IF(JZ.EQ.1)WRITE(LI,'(A)')' -z do not sharpen density maps'
      IF(JZ.EQ.0)WRITE(LI,'(A)')' -z NOT SET: sharpen density maps'
      IF(JJ.EQ.0)WRITE(LI,'(A)')' -j NOT SET: do not output '//
     +'pseudo-free CC except at end'
      IF(RE.LT.0.)WRITE(LI,'(A)')' -e NOT SET: fill in missing data'//
     +' up to maximum resolution + 0.2 Angstroms'
      WRITE(LI,'(/A,F6.3,A)')' -d',RD,'  high resolution limit to be'
     +//' applied to input data'
      IF(RE.GE.0.)WRITE(LI,'(A,F6.3,A)')' -e',RE,'  high resolution'
     +//' limit for data extrapolation'
      WRITE(LI,'(A,I6,A)')' -m',JM,'  cycles of density modification'
      IF(JJ.GT.0)WRITE(LI,'(A,I6,A)')' -j',JJ,' output pseudo-free CC'
     +//' before every Nth cycle'
      WRITE(LI,'(A,F6.3,A)')' -s',RS,'  solvent fraction'
      WRITE(LI,'(A,F6.3,A)')' -c',RC,'  fraction of pixels in '//
     +'crossover region'
      WRITE(LI,'(A,F6.3,A)')' -g',RG,'  solvent gamma flipping factor'
      WRITE(LI,'(A,F6.3,A)')' -v',RV,'  low density reduction factor'
      WRITE(LI,'(A,F6.3,A)')' -w',RW,'  weight for retained '//
     +'experimental phases'
      WRITE(LI,'(A,F6.2,A)')' -r',RR,'  map resolution (multiplies '//
     +'maximum indices)'
      IF(RB.LT.-1.)GOTO 22
      WRITE(LI,'(A,F6.1,A)')' -b',RB,'  extra B for '//
     +'revised heavy atom sites'
      WRITE(LI,'(A,F6.1,A)')' -q',RQ,'  minimum height/sigma for '//
     +'revised heavy atom sites'
  22  WRITE(LI,'(A,I6,A,I9,A/)')' -l',JL,'  space for',
     +MAX0(JL*1000000,500000),' reflections'
C
C Remove left-over .fin file and allocate array space for reflections
C
      OPEN(UNIT=LP,FILE=KT(1:LL)//'.fin',STATUS='OLD',ERR=23)
      CLOSE(UNIT=LP,STATUS='DELETE',ERR=23)
  23  LD=MAX0(500000,JL*1000000)
      ALLOCATE(IH(LD),IK(LD),IL(LD),IN(LD),FA(LD),SA(LD),PD(LD),
     +FT(LD),ST(LD),FH(LD),PH(LD),PS(LD),WS(LD),SQ(LD),EP(LD),EH(LD),
     +EA(LD),ET(LD),PT(LD),WA(LD),WB(LD),CP(LD),WM(LD),PM(LD),F(LD),
     +STAT=I)
      IF(I.EQ.0)GOTO 24
      WRITE(*,'(A/)')' ** Not enough memory for reflection arrays; '
     +//'reduce -l setting or buy more RAM **'
      CALL EXIT(0)
      STOP' '
C
C Open .res, .ins or .fcf file
C
  24  KR=KT(1:LL)//'.ins'
      N=LL+4
      IF(IABS(JP).EQ.1.OR.JP.EQ.3)GOTO 25
      KR(LL+2:N)='fcf'
      IF(JP.NE.0)GOTO 25
      KR=KA(1:LJ)//'.res'
      N=LJ+4
  25  OPEN(UNIT=LR,FILE=KR(1:N),STATUS='OLD',ERR=9)
      N=0
      NA=0
      LY=21
        DO 26 I=1,32
        A(I)=0.
  26    CONTINUE
      A(20)=1.
      A(21)=1.
      A(25)=1.
      A(29)=1.
      ET(1)=0.
      ET(2)=0.
      ET(3)=0.
      ET(4)=1.
  27  KR=' '
      N=N+1
      READ(LR,'(A)',ERR=47,END=47)KR
      J=0
        DO 28 I=1,80
        KS=KR(I:I)
        IF(KS.LT.' ')KR(I:I)=' '
        IF(KS.GT.' ')J=I
        IF(KS.GE.'a'.AND.KS.LE.'z')KR(I:I)=CHAR(ICHAR(KS)-32)
  28    CONTINUE
      IF(J.EQ.0)GOTO 27
C
C Read cell from .fcf file
C
      IF(JP.NE.2.AND.JP.NE.4)GOTO 30
      IF(INDEX(KR,'_SYMMETRY_EQUIV_POS_AS_XYZ').EQ.0)GOTO 29
      A(20)=-1.
      LY=9
      NE=1
      GOTO 27
  29  I=INDEX(KR,'_CELL_LENGTH_A')
      IF(I.NE.0)READ(KR(I+14:80),*,ERR=27)A(2)
      I=INDEX(KR,'_CELL_LENGTH_B')
      IF(I.NE.0)READ(KR(I+14:80),*,ERR=27)A(3)
      I=INDEX(KR,'_CELL_LENGTH_C')
      IF(I.NE.0)READ(KR(I+14:80),*,ERR=27)A(4)
      I=INDEX(KR,'_CELL_ANGLE_ALPHA')
      IF(I.NE.0)READ(KR(I+17:80),*,ERR=27)A(5)
      I=INDEX(KR,'_CELL_ANGLE_BETA')
      IF(I.NE.0)READ(KR(I+16:80),*,ERR=27)A(6)
      I=INDEX(KR,'_CELL_ANGLE_GAMMA')
      IF(I.EQ.0)GOTO 34
      READ(KR(I+17:80),*,ERR=27)A(7)
      A(1)=1.
      GOTO 31
C
C Read cell from .ins or .res file
C
  30  IF(KR(1:1).EQ.' ')GOTO 27
      IF(KR(1:4).NE.'CELL')GOTO 34
      READ(KR(5:80),*,ERR=47,END=47)(A(I),I=1,7)
  31  U=A(2)*A(3)*A(4)
        DO 32 K=2,4
        IF(A(K).LT.0.1)GOTO 47
        IF(A(K+3).LT.10.)GOTO 47
        X=1.74533E-2*A(K+3)
        D(K)=COS(X)
        D(K+3)=SIN(X)
        A(K+9)=2.*U*D(K)/A(K)
        A(K+6)=A(K)**2
  32    CONTINUE
      V=1./(1.-D(2)**2-D(3)**2-D(4)**2+2.*D(2)*D(3)*D(4))
        DO 33 K=2,4
        A(K+12)=V*(D(K+3)/A(K))**2
  33    CONTINUE
      U=2.*V/U
      A(17)=U*A(2)*(D(3)*D(4)-D(2))
      A(18)=U*A(3)*(D(2)*D(4)-D(3))
      A(19)=U*A(4)*(D(2)*D(3)-D(4))
      GOTO 27
C
C Read symmetry operators from .fcf file
C
  34  IF(JP.NE.2.AND.JP.NE.4)GOTO 45
      IF(NE.NE.1)GOTO 37
      NE=0
      IF(INDEX(KR,'Y').EQ.0)GOTO 37
      NE=1
      LY=LY+12
        DO 35 I=LY,LY+11
        A(I)=0.
  35    CONTINUE
      K=INDEX(KR,'''')+1
      J=INDEX(KR,',')
      KX(1)='+'//KR(K:J-1)
      I=INDEX(KR(J+1:80),',')+J
      KX(2)='+'//KR(J+2:I-1)
      J=INDEX(KR(I+1:80),'''')+I-1
      KX(3)='+'//KR(I+2:J)
      M=LY
        DO 36 I=1,3
        K=INDEX(KX(I),'+X')
        IF(K.GT.0.AND.K.LT.20)A(M)=1.
        K=INDEX(KX(I),'-X')
        IF(K.GT.0.AND.K.LT.20)A(M)=-1.
        K=INDEX(KX(I),'+Y')
        IF(K.GT.0.AND.K.LT.20)A(M+1)=1.
        K=INDEX(KX(I),'-Y')
        IF(K.GT.0.AND.K.LT.20)A(M+1)=-1.
        K=INDEX(KX(I),'+Z')
        IF(K.GT.0.AND.K.LT.20)A(M+2)=1.
        K=INDEX(KX(I),'-Z')
        IF(K.GT.0.AND.K.LT.20)A(M+2)=-1.
        M=M+3
        J=INDEX(KX(I),'/')
        IF(J.EQ.0)GOTO 36
        READ(KX(I)(J-2:J+1),'(I2,1X,I1)')K,L
        A(I+LY+8)=REAL(K)/REAL(L)
  36    CONTINUE
      GOTO 27
C
C Separate lattice and symmetry operators
C
  37  IF(INDEX(KR,'_REFLN_PHASE_CALC').EQ.0)GOTO 27
      K=0
        DO 38 I=21,LY,12
        IF(ABS(A(I)-1.)+ABS(A(I+1))+ABS(A(I+2))+ABS(A(I+3))+
     +  ABS(A(I+4)-1.)+ABS(A(I+5))+ABS(A(I+6))+ABS(A(I+7))+
     +  ABS(A(I+8)-1.).GT.0.1)GOTO 38
        K=K+3
        EP(K-2)=A(I+9)
        EP(K-1)=A(I+10)
        EP(K)=A(I+11)
  38    CONTINUE
      M=K/3
      IF(M.NE.2)GOTO 39
      I=3
      IF(ABS(EP(1))+ABS(EP(2))+ABS(EP(3)).GT.0.1)I=0
      IF(ABS(EP(I+1)).LT.0.1)M=5
      IF(ABS(EP(I+2)).LT.0.1)M=6
      IF(ABS(EP(I+3)).LT.0.1)M=7
  39  A(20)=-REAL(M)
      IF(LY.EQ.21)GOTO 54
      L=21
  40  I=L
  41  I=I+12
      T=0.
        DO 42 J=0,8
        T=T+ABS(A(L+J)-A(I+J))
  42    CONTINUE
      IF(T.GT.0.1)GOTO 44
        DO 43 J=0,11
        A(I+J)=A(LY+J)
  43    CONTINUE
      LY=LY-12
      I=I-12
  44  IF(I.LT.LY)GOTO 41
      L=L+12
      IF(L.LT.LY)GOTO 40
      GOTO 54
C
C Read LATT/SYMM
C
  45  IF(KR(1:4).NE.'LATT')GOTO 46
      READ(KR(5:80),*,ERR=47,END=47)A(20)
      GOTO 27
  46  IF(KR(1:4).NE.'SYMM')GOTO 48
      LY=LY+12
      CALL XSYM(KR(5:80),A(LY))
      IF(A(LY).LT.998.)GOTO 27
  47  WRITE(*,'(A,I8,A/A/)')' ** Bad .ins, .res or .fcf file at line',
     +N,' **',' '//KR(1:77)
      CALL EXIT(0)
      STOP' '
C
C Lattice operators
C
  48  IF(KR(1:4).NE.'SFAC')GOTO 27
      KF=KR
      M=NINT(ABS(A(20)))
      K=3*M
      EP(1)=0.
      EP(2)=0.
      EP(3)=0.
        DO 49 I=4,12
        EP(I)=.5
  49    CONTINUE
      IF(K.LT.12)GOTO 51
      IF(K.GT.12)GOTO 53
        DO 50 I=4,12,4
        EP(I)=0.
  50    CONTINUE
  51  IF(K.NE.9)GOTO 54
        DO 52 I=4,9
        EP(I)=.6666667
  52    CONTINUE
      EP(5)=.3333333
      EP(6)=.3333333
      EP(7)=.3333333
      GOTO 54
  53  EP(M-1)=0.
      K=4
  54  LV=LY+9
        DO 55 I=1,K,3
        LV=LV+3
        A(LV)=EP(I)+99.5
        A(LV+1)=EP(I+1)+99.5
        A(LV+2)=EP(I+2)+99.5
  55    CONTINUE
      JE=LV+2
      J=4
  56  J=J+1
      IF(J.GT.78)GOTO 59
      IF(KF(J:J).LE.' ')GOTO 56
        DO 57 I=1,94
        IF(KE(I).EQ.KF(J:J+1))GOTO 58
  57    CONTINUE
      GOTO 47
  58  JE=JE+1
      A(JE)=REAL(I)
      J=J+1
      GOTO 56
C
C Check lattice and symmetry operators
C
  59  KR=' Bad or missing CELL'
      IF(A(7).LT.0.01)GOTO 47
      KR=' Error in LATT or SYMM (only chiral space groups allowed)'
      IF(A(20).GT.-0.9)GOTO 47
        DO 60 K=21,LY,12
        IF(ABS(A(K)*(A(K+4)*A(K+8)-A(K+5)*A(K+7))+
     +  A(K+1)*(A(K+5)*A(K+6)-A(K+3)*A(K+8))+
     +  A(K+2)*(A(K+3)*A(K+7)-A(K+4)*A(K+6))-1.).GT.0.1)GOTO 47
  60    CONTINUE
      IF(JP.GT.0)GOTO 78
      GOTO 62
C
C Open .res file to get atoms if generating anomalous Fourier from .fcf
C
  61  KR=KA(1:LJ)//'.res'
      N=LJ+4
      OPEN(UNIT=LR,FILE=KR(1:N),STATUS='OLD',ERR=9)
      N=0
      NA=0
C
C Extract atoms if required
C
  62  KR=' '
      N=N+1
      READ(LR,'(A)',ERR=73,END=73)KR
        DO 63 I=1,80
        KS=KR(I:I)
        IF(KS.LT.' ')KR(I:I)=' '
        IF(KS.GE.'a'.AND.KS.LE.'z')KR(I:I)=CHAR(ICHAR(KS)-32)
  63    CONTINUE
      IF(KR(1:1).LE.' ')GOTO 62
      KQ=KR(1:4)
      IF(KQ.EQ.'HKLF'.OR.KQ.EQ.'END ')GOTO 73
      IF(KQ.NE.'MOVE')GOTO 65
      READ(KR(5:80),*,ERR=64,END=64)(ET(I),I=1,4)
      GOTO 62
  64  ET(4)=1.
      READ(KR(5:80),*,ERR=47,END=47)(ET(I),I=1,3)
      GOTO 62
  65  IF(KR(1:4).NE.'SFAC')GOTO 69
      KF=KR
      JE=LV+2
      J=4
  66  J=J+1
      IF(J.GT.78)GOTO 62
      IF(KF(J:J).LE.' ')GOTO 66
        DO 67 I=1,94
        IF(KE(I).EQ.KF(J:J+1))GOTO 68
  67    CONTINUE
      GOTO 47
  68  JE=JE+1
      A(JE)=REAL(I)
      J=J+1
      GOTO 66
  69  IF(KQ.EQ.'UNIT')KU=KR
      IF(KQ.EQ.'UNIT'.AND.JP.EQ.4)JP=5
      IF(JP.EQ.4)GOTO 62
        DO 70 I=1,65
        IF(KG(I).EQ.KQ)GOTO 62
  70    CONTINUE
      READ(KR(5:80),*,ERR=62,END=62)(EP(I),I=1,5)
        DO 71 I=2,5
        EP(I)=EP(I)-10.*ANINT(0.1*EP(I))
  71    CONTINUE
      NA=NA+1
      XA(NA)=ET(1)+ET(4)*EP(2)
      YA(NA)=ET(2)+ET(4)*EP(3)
      ZA(NA)=ET(3)+ET(4)*EP(4)
      J=LV+2+NINT(EP(1))
      IF(J.GT.JE)GOTO 47
      OC(NA)=EP(5)*A(J)
      IF(JH.EQ.0.OR.NA.LT.JH)GOTO 62
  72  FORMAT(I6,' heavy atoms read from .',A,' file'/)
C
C Read file containing (FA or delF) and phase shifts
C
  73  CLOSE(UNIT=LR)
      KR='ins'
      IF(JP.GE.0)KR='res'
      WRITE(*,72)NA,KR(1:3)
      WRITE(LI,72)NA,KR(1:3)
      NE=-1
      IF(JP.EQ.-1)GOTO 86
  74  KR=KA(1:LJ)//'.hkl'
      N=LJ+4
      L=NR
      OPEN(UNIT=LR,FILE=KR(1:N),STATUS='OLD',ERR=9)
  75  J=NR+1
      IF(J.GT.LD)GOTO 77
      READ(LR,'(3I4,2F8.2,I4)',ERR=83,END=83)IH(J),IK(J),IL(J),
     +FA(J),SA(J),K
      IF(IABS(IH(J))+IABS(IK(J))+IABS(IL(J)).EQ.0)GOTO 83
      IF(K.EQ.90.OR.K.EQ.270)GOTO 76
      IF(MOD(K,180).NE.0)NE=1
      NE=MAX0(NE,0)
  76  PD(J)=REAL(K)
      FT(J)=-99.
      ST(J)=-99.
      PS(J)=-99.
      NR=J
      GOTO 75
  77  WRITE(*,'(A/)')' ** Too many reflections - use -l flag to '
     +//'increase array space **'
      CALL EXIT(0)
      STOP' '
C
C Read .phi file containing h, k, l, Fo, fom and phase
C
  78  N=LL+4
      KR=KT(1:N)
      L=0
      IF(JP.EQ.2.OR.JP.EQ.4)GOTO 81
      CLOSE(UNIT=LR)
      OPEN(UNIT=LR,FILE=KR(1:N),STATUS='OLD',ERR=9)
      NE=-2
  79  J=NR+1
      IF(J.GT.LD)GOTO 77
      READ(LR,*,ERR=80,END=80)IH(J),IK(J),IL(J),T,FA(J),PS(J)
      IF(IABS(IH(J))+IABS(IK(J))+IABS(IL(J)).EQ.0)GOTO 80
      SA(J)=-99.
      FT(J)=-99.
      ST(J)=-99
      PD(J)=-999.
      NR=J
      GOTO 79
  80  WRITE(*,82)NR,KR(1:N)
      WRITE(LI,82)NR,KR(1:N)
      CLOSE(UNIT=LR)
      IF(NR.EQ.0)GOTO 84
      IF(JP.EQ.3)GOTO 74
      GOTO 86
C
C Read reflection data from .fcf file
C
  81  J=NR+1
      IF(J.GT.LD)GOTO 77
      READ(LR,*,ERR=81,END=83)IH(J),IK(J),IL(J),FT(J),T,FA(J),PS(J)
      T=AMAX1(0.01,T)
      FT(J)=SQRT(AMAX1(0.,FT(J)))
      ST(J)=0.5*T/AMAX1(0.5*SQRT(T),FT(J))
      SA(J)=-99.
      PD(J)=-999.
      NR=J
      GOTO 81
  82  FORMAT(I9,' reflections read from file ',A/)
  83  WRITE(*,82)NR-L,KR(1:N)
      WRITE(LI,82)NR-L,KR(1:N)
      IF(NR.GT.0)GOTO 85
  84  WRITE(*,'(A/)')' ** Error - no data read **'
      CALL EXIT(0)
      STOP' '
  85  CLOSE(UNIT=LR)
      IF(JP.EQ.4)GOTO 61
      IF(JP.EQ.2.OR.JP.EQ.5)GOTO 90
C
C Read native intensities
C
  86  KR=KT(1:LL)//'.hkl'
      N=LL+4
      OPEN(UNIT=LR,FILE=KR(1:N),STATUS='OLD',ERR=9)
      K=NR
  87  J=NR+1
      READ(LR,'(3I4,2F8.2)',ERR=89,END=89)IH(J),IK(J),IL(J),FT(J),T
      IF(IABS(IH(J))+IABS(IK(J))+IABS(IL(J)).EQ.0)GOTO 89
      ST(J)=AMAX1(0.01,T)
      IF(JF.NE.0)GOTO 88
      FT(J)=SQRT(AMAX1(0.,FT(J)))
      ST(J)=0.5*ST(J)/AMAX1(0.5*SQRT(ST(J)),FT(J))
  88  FA(J)=-99.
      SA(J)=-99.
      PS(J)=-99.
      PD(J)=-999.
      NR=J
      GOTO 87
  89  CLOSE(UNIT=LR)
      J=NR-K
      WRITE(*,82)J,KR(1:N)
      WRITE(LI,82)J,KR(1:N)
      IF(J.EQ.0)GOTO 84
C
C Find equivalent reflections with standard indices, transform phases
C
  90    DO 95 I=1,NR
        U=REAL(IH(I))
        V=REAL(IK(I))
        W=REAL(IL(I))
        MH=IH(I)
        MK=IK(I)
        MN=IL(I)
        P=PS(I)
        Q=PD(I)
          DO 94 K=21,LY,12
          NH=NINT(U*A(K)+V*A(K+3)+W*A(K+6))
          NK=NINT(U*A(K+1)+V*A(K+4)+W*A(K+7))
          NL=NINT(U*A(K+2)+V*A(K+5)+W*A(K+8))
          T=1.
          IF(NL.GT.0)GOTO 92
          IF(NL.LT.0)GOTO 91
          IF(NK.GT.0)GOTO 92
          IF(NK.LT.0)GOTO 91
          IF(NH.GE.0)GOTO 92
  91      NH=-NH
          NK=-NK
          NL=-NL
          T=-1.
  92      IF(NL.LT.MN)GOTO 94
          IF(NL.GT.MN)GOTO 93
          IF(NK.LT.MK)GOTO 94
          IF(NK.GT.MK)GOTO 93
          IF(NH.LT.MH)GOTO 94
  93      MH=NH
          MK=NK
          MN=NL
          IF(P.GE.-98.)PS(I)=AMOD(719.99+T*AMOD(P-360.*(U*A(K+9)+
     +    V*A(K+10)+W*A(K+11)),360.),360.)+0.01
          IF(Q.GE.-98.)PD(I)=AMOD(719.99+Q*T,360.)+0.01
  94      CONTINUE
        IH(I)=MH
        IK(I)=MK
        IL(I)=MN
  95    CONTINUE
      CLOSE(UNIT=LR)
C
C Sort data into standard order so that equivalents are adjacent
C
      CALL SXIS(NR,IN,IH,IK,IL,FA,SA,PD,FT,ST,PS,
     +WS,FH,PH,SQ,EP,EH,EA,ET,PT)
      CALL SXIS(NR,IN,IK,IL,IH,FA,SA,PD,FT,ST,PS,
     +WS,FH,PH,SQ,EP,EH,EA,ET,PT)
      CALL SXIS(NR,IN,IL,IH,IK,FA,SA,PD,FT,ST,PS,
     +WS,FH,PH,SQ,EP,EH,EA,ET,PT)
C
C Make unique combined FA, Fo list, reject systematic absences, find
C epsilon and 1/d^2 (non-centrosymmetric space group assumed !)
C
      SM=0.
      M=0
      I=1
  96  IF(I.GT.NR)GOTO 106
      U=0.
      V=0.
      X=0.
      Y=0.
      Q=0.
      Z=0.
      P=-999.
      N=M+1
      WS(N)=-99.
      PS(N)=PS(I)
      K=I
      GOTO 98
  97  I=I+1
      IF(I.GT.NR)GOTO 101
      IF(IH(I).NE.IH(K))GOTO 101
      IF(IK(I).NE.IK(K))GOTO 101
      IF(IL(I).NE.IL(K))GOTO 101
  98  IF(SA(I).LT.1.E-8)GOTO 99
      W=1./SA(I)**2
      U=U+FA(I)*W
      X=X+W
      IF(PD(I).GE.0.)P=PD(I)
      GOTO 97
  99  IF(ST(I).LT.1.E-8)GOTO 100
      W=1./ST(I)**2
      V=V+FT(I)*W
      Y=Y+W
      IF(JP.NE.2.AND.JP.NE.5)GOTO 97
      Z=Z+1.
      Q=Q+FA(I)
 100  PS(N)=PS(I)
      WS(N)=FA(I)
      GOTO 97
 101  IF(Y.LT.1.E-8)GOTO 96
      IH(N)=IH(K)
      IK(N)=IK(K)
      IL(N)=IL(K)
      FT(N)=V/Y
      ST(N)=SQRT(1./Y)
      FA(N)=-99.
      SA(N)=-99.
      IF(PS(K).GT.-998.)PS(N)=PS(K)
      PT(N)=-99.
      IF(Z.GT.0.5)PT(N)=Q/Z
      IF(X.LT.1.E-8)GOTO 102
      FA(N)=U/X
      SA(N)=SQRT(1./X)
 102  PD(N)=P
      U=REAL(IH(N))
      V=REAL(IK(N))
      W=REAL(IL(N))
      SQ(N)=U**2*A(14)+V**2*A(15)+W**2*A(16)+V*W*A(17)+
     +U*W*A(18)+U*V*A(19)
      IF(SQ(N)*RD**2.GT.1.)GOTO 96
      CP(N)=-99.
      Q=1.
        DO 104 K=33,LY,12
        NH=NINT(U*A(K)+V*A(K+3)+W*A(K+6))
        NK=NINT(U*A(K+1)+V*A(K+4)+W*A(K+7))
        NL=NINT(U*A(K+2)+V*A(K+5)+W*A(K+8))
        T=AMOD(9.5+AMOD(U*A(K+9)+V*A(K+10)+W*A(K+11),1.),1.)-0.5
        IF(NH.NE.-IH(N))GOTO 103
        IF(NK.NE.-IK(N))GOTO 103
        IF(NL.NE.-IL(N))GOTO 103
        CP(N)=180.*T
 103    IF(NH.NE.IH(N))GOTO 104
        IF(NK.NE.IK(N))GOTO 104
        IF(NL.NE.IL(N))GOTO 104
        IF(ABS(T).GT.0.1)GOTO 96
        Q=Q+1.
 104    CONTINUE
      FH(N)=-99.
      EH(N)=-99.
      PH(N)=-999.
      EP(N)=SQRT(Q)
      SM=AMAX1(SM,SQ(N))
      M=N
      GOTO 96
C
C Add missing reflections and extend resolution
C
 105  FORMAT(' Maximum resolution of native data =',F7.3,' Angstroms')
 106  NB=M
      NR=M
      IF(NR.GE.LD)GOTO 76
      SM=1./SQRT(SM)
      WRITE(*,105)SM
      WRITE(LI,105)SM
      IF(RE.LE.0.)RE=SM+0.2
      RE=AMAX1(RE,AMIN1(SM,0.5))
      M=NINT(ABS(A(20)))
      MX=NINT(A(2)/RE+.5)
      MY=NINT(A(3)/RE+.5)
      MZ=NINT(A(4)/RE+.5)
      N=0
      NQ=0
      ML=-1
 107  ML=ML+1
      IF(ML.GT.MZ)GOTO 118
      W=REAL(ML)
      MK=-1-MY
      IF(ML.EQ.0)MK=-1
 108  MK=MK+1
      IF(MK.GT.MY)GOTO 107
      IF((M.EQ.4.OR.M.EQ.5).AND.MOD(MK+ML,2).NE.0)GOTO 108
      V=REAL(MK)
      MT=IABS(MK)+IABS(ML)
      MH=-1-MX
      IF(MT.EQ.0)MH=0
 109  MH=MH+1
      IF(MH.GT.MX)GOTO 108
      IF(M.EQ.2.AND.MOD(MH+MK+ML,2).NE.0)GOTO 109
      IF(M.EQ.3.AND.MOD(MK+ML-MH,3).NE.0)GOTO 109
      IF((M.EQ.4.OR.M.EQ.6).AND.MOD(MH+ML,2).NE.0)GOTO 109
      IF(M.EQ.7.AND.MOD(MH+MK,2).NE.0)GOTO 109
      IF(IABS(MH)+MT.EQ.0)GOTO 109
      U=REAL(MH)
        DO 112 NK=33,LY,12
        I=NINT(U*A(NK)+V*A(NK+3)+W*A(NK+6))
        J=NINT(U*A(NK+1)+V*A(NK+4)+W*A(NK+7))
        K=NINT(U*A(NK+2)+V*A(NK+5)+W*A(NK+8))
        IF(K.GT.0)GOTO 111
        IF(K.LT.0)GOTO 110
        IF(J.GT.0)GOTO 111
        IF(J.LT.0)GOTO 110
        IF(I.GE.0)GOTO 111
 110    I=-I
        J=-J
        K=-K
 111    IF(K.GT.ML)GOTO 109
        IF(K.LT.ML)GOTO 112
        IF(J.GT.MK)GOTO 109
        IF(J.LT.MK)GOTO 112
        IF(I.GT.MH)GOTO 109
 112    CONTINUE
 113  N=N+1
      IF(N.GT.NB)GOTO 114
      IF(ML.LT.IL(N))GOTO 114
      IF(ML.GT.IL(N))GOTO 113
      IF(MK.LT.IK(N))GOTO 114
      IF(MK.GT.IK(N))GOTO 113
      IF(MH.LT.IH(N))GOTO 114
      IF(MH.GT.IH(N))GOTO 113
      GOTO 109
 114  N=N-1
      Q=1.
        DO 116 K=33,LY,12
        NH=NINT(U*A(K)+V*A(K+3)+W*A(K+6))
        NK=NINT(U*A(K+1)+V*A(K+4)+W*A(K+7))
        NL=NINT(U*A(K+2)+V*A(K+5)+W*A(K+8))
        T=AMOD(9.5+AMOD(U*A(K+9)+V*A(K+10)+W*A(K+11),1.),1.)-0.5
        IF(NH.NE.-MH)GOTO 115
        IF(NK.NE.-MK)GOTO 115
        IF(NL.NE.-ML)GOTO 115
        CP(NR+1)=180.*T
 115    IF(NH.NE.MH)GOTO 116
        IF(NK.NE.MK)GOTO 116
        IF(NL.NE.ML)GOTO 116
        IF(ABS(T).GT.0.1)GOTO 109
        Q=Q+1.
 116    CONTINUE
      SQ(NR+1)=A(14)*U**2+A(15)*V**2+A(16)*W**2+A(17)*V*W+A(18)*U*W+
     +A(19)*U*V
      T=1./SQRT(SQ(NR+1))
      IF(T.LT.RE)GOTO 109
      IF(T.GE.SM+0.2)NQ=NQ+1
      CP(NR+1)=-99.
      NR=NR+1
      IF(NR.GE.LD)GOTO 76
      IH(NR)=MH
      IK(NR)=MK
      IL(NR)=ML
      FT(NR)=0.
      ST(NR)=0.
      FA(NR)=0.
      SA(NR)=0.
      FH(NR)=0.
      EH(NR)=0.
      PH(NR)=0.
      PT(NR)=-999.
      PD(NR)=0.
      EP(NR)=SQRT(Q)
      WS(NR)=0.
      PS(NR)=-999.
      WM(NR)=0.
      PM(NR)=-999.
      GOTO 109
 117  FORMAT(I6,' reflections with d >',F6.3,' and',I7,
     +' in range',F7.3,' > d >',F6.3,' added'/)
 118  Q=SM+0.2
      WRITE(*,117)NQ,Q,NR-NB-NQ,Q,AMIN1(RE,Q)
      WRITE(LI,117)NQ,Q,NR-NB-NQ,Q,AMIN1(RE,Q)
C
C Define Fourier grid
C
      T=1.
      L=INT(ABS(A(20)))
      IF(IABS(L-4).LT.3)T=.5
      IF(L.EQ.3)T=.3333333
      S=T
      AZ=99.
        DO 120 K=33,LY,12
        IF(ABS(A(K+6))+ABS(A(K+7)).GT.0.1)GOTO 120
        IF(A(20).GT.0.)GOTO 119
        IF(A(K+8).GT.0.)GOTO 119
        AZ=AMIN1(AZ,.5*ABS(AMOD(99.5+A(K+11),1.)-.5),
     +  .5*ABS(AMOD(99.5+A(K+11)-S,1.)-.5))
        GOTO 120
 119    U=AMIN1(ABS(AMOD(99.5+A(K+11),1.)-.5),ABS(AMOD(99.5-
     +  A(K+11),1.)-.5),ABS(AMOD(99.5+A(K+11)-S,1.)-.5),
     +  ABS(AMOD(99.5-A(K+11)-S,1.)-.5))
        IF(U.LT.0.05)GOTO 120
        T=AMIN1(T,U)
 120    CONTINUE
      EZ=.5*T+AZ
      IF(AZ.LT.98.)GOTO 121
      AZ=0.
      EZ=T
 121  L=1
      M=1
      N3=1
        DO 123 N=1,NB
          DO 122 K=21,LY,12
          L=MAX0(L,IABS(NINT(REAL(IH(N))*A(K)+
     +    REAL(IK(N))*A(K+3)+REAL(IL(N))*A(K+6))))
          M=MAX0(M,IABS(NINT(REAL(IH(N))*A(K+1)+
     +    REAL(IK(N))*A(K+4)+REAL(IL(N))*A(K+7))))
          N3=MAX0(N3,IABS(NINT(REAL(IH(N))*A(K+2)+
     +    REAL(IK(N))*A(K+5)+REAL(IL(N))*A(K+8))))
 122      CONTINUE
 123    CONTINUE
      T=ABS(RR)
      L=NINT(T*REAL(L))
      M=NINT(T*REAL(M))
      N3=NINT((EZ-AZ)*T*REAL(N3))
      N1=1
 124  N1=N1*2
      IF(N1.LT.L)GOTO 124
      N2=1
 125  N2=N2*2
      IF(N2.LT.M)GOTO 125
      N3=N3+1
      WRITE(*,130)N1,N2,N3,AZ,EZ
      N6=2*N1
      N4=N2*N6
      N5=N3*N4
C
C Set up grid offset table for triangulated spherical net
C
      N7=0
      T=(D(2)*D(3)-D(4))/(D(5)*D(6))
      S=SQRT(1.-T**2)
      D(13)=REAL(N1)
      D(14)=REAL(N2)
      D(15)=REAL(N3-1)/(EZ-AZ)
      D(8)=A(2)*D(6)*S
      D(9)=A(2)*D(6)*T
      D(10)=A(3)*D(5)
      D(11)=A(2)*D(3)
      D(12)=A(3)*D(2)
      M=1
      FH(1)=0.
      PH(1)=0.
      FH(47)=3.141593
      PH(47)=0.
        DO 127 I=1,17,2
        P=0.0174533*BF(I)
        V=BF(I+1)
          DO 126 J=1,5
          M=M+1
          FH(M)=P
          FH(M+46)=P+3.141593
          PH(M)=0.0174533*V
          PH(M+46)=PH(M)
          V=V+72.
 126      CONTINUE
 127    CONTINUE
      R=2.42
        DO 128 I=1,92
        U=R*SIN(FH(I))
        X=U*COS(PH(I))/D(8)
        Y=(U*SIN(PH(I))+X*D(9))/D(10)
        Z=(R*COS(FH(I))-X*D(11)-Y*D(12))/A(4)
        IX(I)=NINT(X*D(13))+N1
        IY(I)=NINT(Y*D(14))+N2
        IZ(I)=NINT(Z*D(15))
        N7=MAX0(N7,IABS(IZ(I)))
 128    CONTINUE
 129  FORMAT(' 92 point spherical net set up with radius 2.42A',
     +/I4,' extra Fourier layers will be generated'/)
      WRITE(*,129)N7*2
      WRITE(LI,129)N7*2
C
C Allocate memory for FFT arrays B and C and generate E-values
C
      LB=MAX0(N5,LD)
      LC=MAX0(N5/2,LD,(N3+2*N7)*N4/2)
      ALLOCATE(B(LB),C(LC),IJ(LC),STAT=I)
      IF(I.EQ.0)GOTO 131
      WRITE(*,'(A/)')' ** Not enough memory for FFT - buy more RAM **'
      CALL EXIT(0)
      STOP' '
 130  FORMAT(' Fourier grid = ',2(I5,' x'),I5,F10.3,' <= z <=',F6.3/)
C
C Take advantage of symmetry
C
 131  M=0
      NJ=N7*N1*N2
      DX=1./REAL(N1)
      DY=1./REAL(N2)
      DZ=(EZ-AZ)/REAL(N3-1)
        DO 136 NL=-N7,N3+N7-1
        Z=REAL(NL)*DZ+AZ
          DO 135 NK=0,N2-1
          Y=REAL(NK)*DY
            DO 134 NH=0,N1-1
            X=REAL(NH)*DX
            M=M+1
            IJ(M)=LC
            IF(NL.GE.0.AND.NL.LT.N3)IJ(M)=M-NJ
              DO 133 K=21,LY,12
              P=A(K)*X+A(K+1)*Y+A(K+2)*Z+A(K+9)+0.5
              Q=A(K+3)*X+A(K+4)*Y+A(K+5)*Z+A(K+10)+0.5
              R=A(K+6)*X+A(K+7)*Y+A(K+8)*Z+A(K+11)+0.5
                DO 132 L=LY+12,LV,3
                IF(K+L.EQ.LY+33)GOTO 132
                W=AMOD(R+A(L+2)-AZ,1.)/DZ
                N=NINT(W)
                IF(N.GE.N3)GOTO 132
                IF(ABS(W-REAL(N)).GT.0.01)GOTO 132
                V=AMOD(Q+A(L+1),1.)/DY
                J=NINT(V)
                IF(ABS(V-REAL(J)).GT.0.01)GOTO 132
                U=AMOD(P+A(L),1.)/DX
                I=NINT(U)
                IF(ABS(U-REAL(I)).GT.0.01)GOTO 132
                IJ(M)=MIN0(IJ(M),I+N1*(J+N2*N)+1)
 132            CONTINUE
 133          CONTINUE
 134        CONTINUE
 135      CONTINUE
 136    CONTINUE
      NK=N1*N2*N3+NJ
      IP=0
        DO 137 I=1,NJ
        IF(IJ(I).EQ.LC)IP=1
        IF(IJ(I+NK).EQ.LC)IP=1
 137    CONTINUE
C
C Read in .fcf file for diagnostics if -x switch set
C
      IF(JX.EQ.0)GOTO 146
        DO 138 I=1,NR
        F(I)=-999.
        WM(I)=1.
        PM(I)=0.
        IN(I)=-1
 138    CONTINUE
      KR=KT(1:LL)//'.fcf'
      N=LL+4
      OPEN(UNIT=LR,FILE=KR(1:N),STATUS='OLD',ERR=9)
 139  KK=' '
      READ(LR,'(A)',ERR=139,END=144)KK
      IF(INDEX(KK,'_refln_phase_calc').EQ.0)GOTO 139
      L=0
      M=0
      N=1
 140  KK=' '
      READ(LR,'(A)',ERR=140,END=145)KK
      READ(KK,*,ERR=140,END=140)I,J,K,S,T,U,V
      L=L+1
 141  IF(K.LT.IL(N))GOTO 140
      IF(K.GT.IL(N))GOTO 142
      IF(J.LT.IK(N))GOTO 140
      IF(J.GT.IK(N))GOTO 142
      IF(I.LT.IH(N))GOTO 140
      IF(I.GT.IH(N))GOTO 142
      M=M+1
      PM(N)=U
      F(N)=V
      IN(N)=0
      N=N+1
      IF(N.GT.NR)GOTO 145
      IF(N.EQ.NB+1)REWIND LR
      GOTO 140
 142  N=N+1
      IF(N.LE.NR)GOTO 141
      GOTO 145
 143  FORMAT(I7,' reflections read from ',A,',',I7,' phases assigned')
 144  WRITE(*,'(A)')' ** Corrupted file '//KR(1:LL+4)//' **'
      CALL EXIT(0)
      STOP' '
 145  CLOSE(UNIT=LR)
      WRITE(*,143)L,KR(1:LL+4),M
      WRITE(LI,143)L,KR(1:LL+4),M
C
C Normalize FT and FA
C
 146  CALL ENORM(NB,FT,ST,SQ,EP,ET)
        DO 147 I=1,NB
        WM(I)=0.
        PM(I)=-999.
 147    CONTINUE
      IF(JP.EQ.2)GOTO 181
      IF(JP.EQ.1)GOTO 186
      CALL ENORM(NB,FA,SA,SQ,EP,EA)
        DO 148 I=1,NB
        IF(ABS(FA(I)).LT.0.0001)SA(I)=-99.
        IF(SA(I).LT.-98.)GOTO 148
        SA(I)=SA(I)*EA(I)/AMAX1(0.0001,FA(I))
 148    CONTINUE
      IF(JP.EQ.3)GOTO 186
      IF(JP.EQ.5)GOTO 181
C
C Invert heavy atoms and space group if required
C
      IF(JI.EQ.0)GOTO 158
      LL=LL+2
      KT(LL-1:LL)='_i'
        DO 149 I=1,3
        PT(I)=1.
 149    CONTINUE
      J=0
        DO 150 K=33,LY,12
        IF(ABS(A(K+11)-0.3333).LT.0.02)J=3
        IF(ABS(A(K+11)-0.25).LT.0.02)J=4
 150    CONTINUE
      K=NINT(A(20))
      IF(J.NE.0.AND.K.EQ.-1)GOTO 154
      IF(J.NE.4)GOTO 156
      IF(K.NE.-2)GOTO 152
      IF(LY.NE.57)GOTO 151
      PT(2)=0.5
      GOTO 156
 151  IF(LY.NE.105)GOTO 156
      PT(2)=0.5
      PT(3)=0.25
      GOTO 156
 152  IF(K.NE.-4.OR.LY.NE.297)GOTO 156
      PT(1)=0.25
      PT(2)=0.25
      PT(3)=0.25
      GOTO 156
 153  FORMAT(' ** Space group converted to enantiomorph **'/)
 154  WRITE(*,153)
      WRITE(LI,153)
        DO 155 K=33,LY,12
        A(K+9)=1.-A(K+9)
        A(K+10)=1.-A(K+10)
        A(K+11)=1.-A(K+11)
 155    CONTINUE
 156    DO 157 I=1,NA
        XA(I)=PT(1)-XA(I)
        YA(I)=PT(2)-YA(I)
        ZA(I)=PT(3)-ZA(I)
 157    CONTINUE
C
C Calculate and normalize heavy-atom structure factors
C
 158  N=NA
        DO 160 J=1,NA
        X=XA(J)
        Y=YA(J)
        Z=ZA(J)
          DO 159 K=33,LY,12
          N=N+1
          XA(N)=X*A(K)+Y*A(K+1)+Z*A(K+2)+A(K+9)
          YA(N)=X*A(K+3)+Y*A(K+4)+Z*A(K+5)+A(K+10)
          ZA(N)=X*A(K+6)+Y*A(K+7)+Z*A(K+8)+A(K+11)
          OC(N)=OC(J)
 159      CONTINUE
 160    CONTINUE
        DO 162 I=1,NB
        WS(I)=0.
        PS(I)=-999.
        FH(I)=0.
        PH(I)=-999.
        X=6.2831853*REAL(IH(I))
        Y=6.2831853*REAL(IK(I))
        Z=6.2831853*REAL(IL(I))
        U=0.
        V=0.
          DO 161 K=1,N
          P=X*XA(K)+Y*YA(K)+Z*ZA(K)
          U=U+OC(K)*COS(P)
          V=V+OC(K)*SIN(P)
 161      CONTINUE
        FH(I)=SQRT(U**2+V**2)
        PH(I)=AMOD(720.+57.29578*ATAN2(V,U),360.)
 162    CONTINUE
      CALL ENORM(NB,FH,PH,SQ,EP,EH)
      IF(JP.EQ.-1)GOTO 176
      CALL CALCC(NB,EH,EA,SQ,SQ,D)
      WRITE(*,163)D(13)
      WRITE(LI,163)D(13)
      IF(D(13).GT.0.)GOTO 165
      WRITE(*,164)
      WRITE(LI,164)
      GOTO 165
 163  FORMAT(' Overall CC between Eobs (from delF) and Ecalc (from ',
     +'heavy atoms) =',F6.2,'%'/)
 164  FORMAT(10X,'** Heavy atom solution almost certainly wrong! **'/)
C
C Estimate starting centroid protein phases and weights
C
 165    DO 167 I=1,NB
        WS(I)=0.
        PS(I)=-999.
        IF(EH(I).LT.0.2)GOTO 167
        IF(EA(I).LT.0.3)GOTO 167
        P=D(1)*SQ(I)
        N=MIN0(INT(P),9)
        P=P-REAL(N)
        P=AMIN1(0.9,SQRT(AMAX1((1.-P)*D(N+2)+P*D(N+3),0.001))+0.1)
        P=P/((1.-P**2)*SQRT(1.+SA(I)**2))
        PS(I)=AMOD(720.+PH(I)+PD(I),360.)
        T=P*EA(I)
        IF(NE.EQ.1)T=1.25*T*EH(I)
        IF(CP(I).GT.-98.)GOTO 166
        WS(I)=AMIN1(T*(0.5658+T*(T*0.0106-0.1304)),T/(0.56+T))
        GOTO 167
 166    T=EXP(AMIN1(T,12.))
        WS(I)=(T-1.)/(T+1.)
 167    CONTINUE
      NN=0
      IF(JX.NE.0)CALL PHSTAT(NB,NB,LI,F,PS,WS,CP)
      CALL FOMOUT(NB,NN,LI,LY,IH,IK,IL,FT,WS,US,VS,A,
     +'for initial phases')
C
C Identify twofold ambiguities
C
      S=0.
        DO 168 I=1,NB
        PM(I)=PS(I)
        WM(I)=WS(I)
        IF(WS(I).GT.0.0001)S=AMAX1(S,SQ(I))
 168    CONTINUE
      IF(JN.NE.0)GOTO 175
      IF(NE.EQ.1)GOTO 175
      IQ=1
        DO 169 I=1,NR
        IN(I)=0
        IF(I.GT.NB)GOTO 169
        IF(SQ(I).GT.S)GOTO 169
        IF(ET(I).LT.0.3)GOTO 169
        IN(I)=1
 169    CONTINUE
C
C Resolve twofold ambiguities by low density elimination
C
      CALL DMFFT(LD,LB,LC,NR,LY,LV,IH,IK,IL,IN,FT,WS,PS,
     +WA,WB,B,C,XA,YA,ZA,OC,LM,IP,IJ)
        DO 170 I=1,NB
        PT(I)=-999.
        IF(IN(I).EQ.0)GOTO 170
        T=CP(I)
        IF(T.LT.-98.)T=PH(I)
        IF(NE.EQ.0)T=T+90.
        W=WA(I)*COS(0.0174533*T)+WB(I)*SIN(0.0174533*T)
        WA(I)=0.
        IF(ABS(W).LT.1.E-4)GOTO 170
        IF(W.LT.0.)T=180.+T
        PT(I)=AMOD(720.+T,360.)
        WA(I)=ABS(W)
 170    CONTINUE
      CALL ENORM(NB,WA,PT,SQ,EP,WB)
      CALL CALCC(NB,WB,ET,PT,SQ,D)
        DO 173 I=1,NB
        WA(I)=0.
        IF(PT(I).LT.-998.)GOTO 173
        P=D(1)*SQ(I)
        N=MIN0(INT(P),9)
        P=P-REAL(N)
        Q=1.-P
        P=AMIN1(0.9,SQRT(AMAX1(Q*D(N+2)+P*D(N+3),0.0001))+0.1)
        T=WB(I)*ET(I)*P/(1.-P**2)
        IF(CP(I).GT.-98.)GOTO 171
        W=AMIN1(T*(0.5658+T*(T*0.0106-0.1304)),T/(0.56+T))
        GOTO 172
 171    T=EXP(AMIN1(T,12.))
        W=(T-1.)/(T+1.)
 172    U=WS(I)*COS(0.0174533*PS(I))+W*COS(0.0174533*PT(I))
        V=WS(I)*SIN(0.0174533*PS(I))+W*SIN(0.0174533*PT(I))
        WS(I)=AMIN1(1.,SQRT(U**2+V**2))
        PS(I)=AMOD(720.+57.29578*ATAN2(V,U),360.)
        WA(I)=W
 173    CONTINUE
      NN=0
      IF(JX.EQ.0)GOTO 174
      CALL PHSTAT(NB,NB,LI,F,PT,WA,CP)
      CALL PHSTAT(NB,NB,LI,F,PS,WS,CP)
 174  CALL FOMOUT(NB,NN,LI,LY,IH,IK,IL,FT,WS,US,VS,A,
     +'after resolving 2-fold ambiguity')
C
C Add heavy atom contributions if present in native
C
 175  IF(JH.EQ.0)GOTO 186
 176  CALL CALCC(NB,EH,ET,FH,SQ,D)
        DO 179 I=1,NB
        P=D(1)*SQ(I)
        N=MIN0(INT(P),9)
        P=P-REAL(N)
        Q=1.-P
        P=AMIN1(0.9,SQRT(AMAX1(Q*D(N+2)+P*D(N+3),0.0001))+0.1)
        T=1.25*ET(I)*EH(I)*P/(1.-P**2)
        IF(CP(I).GT.-98.)GOTO 177
        W=AMIN1(T*(0.5658+T*(T*0.0106-0.1304)),T/(0.56+T))
        GOTO 178
 177    T=EXP(AMIN1(T,12.))
        W=(T-1.)/(T+1.)
 178    U=WS(I)*COS(0.0174533*PS(I))+W*COS(0.0174533*PH(I))
        V=WS(I)*SIN(0.0174533*PS(I))+W*SIN(0.0174533*PH(I))
        WS(I)=AMIN1(1.,SQRT(U**2+V**2))
        PS(I)=0.
        IF(WS(I).GT.0.0001)PS(I)=AMOD(720.+57.29578*ATAN2(V,U),360.)
        WA(I)=W
 179    CONTINUE
      NN=0
      IF(JX.EQ.0)GOTO 180
      CALL PHSTAT(NR,NB,LI,F,PH,WA,CP)
      CALL PHSTAT(NR,NB,LI,F,PS,WS,CP)
 180  CALL FOMOUT(NB,NN,LI,LY,IH,IK,IL,FT,WS,US,VS,A,
     +'after including heavy atoms')
      GOTO 186
C
C Sigma-A weights for phases from fcf file
C
 181  CALL ENORM(NB,PT,ST,SQ,EP,WA)
      CALL CALCC(NB,WA,ET,ST,SQ,D)
      U=0.
      V=0.
        DO 184 I=1,NB
        WS(I)=0.
        P=D(1)*SQ(I)
        N=MIN0(INT(P),9)
        P=P-REAL(N)
        Q=1.-P
        P=SQRT(AMAX1(Q*D(N+2)+P*D(N+3),0.0001))
        T=2.*ET(I)*WA(I)*P/(1.-P**2)
        IF(CP(I).GT.-98.)GOTO 182
        WS(I)=AMIN1(T*(0.5658+T*(T*0.0106-0.1304)),T/(0.56+T))
        GOTO 183
 182    T=EXP(AMIN1(2.*T,12.))
        WS(I)=(T-1.)/(T+1.)
 183    U=U+WS(I)
        V=V+1.
 184    CONTINUE
      W=U/V
      IF(W.LT.0.3.OR.JP.GT.4)GOTO 186
      W=0.3/W
        DO 185 I=1,NB
        WS(I)=WS(I)*W
 185    CONTINUE
C
C Function for estimating F from E
C
 186  IQ=2
      R=10.*SM**2
        DO 187 I=1,27
        D(I)=0.
 187    CONTINUE
        DO 188 I=1,NB
        P=R*SQ(I)
        N=MIN0(INT(P),9)
        P=P-REAL(N)
        Q=1.-P
        T=(FT(I)/AMAX1(ET(I),0.0001))**2/EP(I)
        D(N+1)=D(N+1)+Q
        D(N+2)=D(N+2)+P
        D(N+12)=D(N+12)+Q*T
        D(N+13)=D(N+13)+P*T
 188    CONTINUE
        DO 189 I=1,11
        T=D(I+11)/AMAX1(D(I),0.01)
        D(I)=T
        IF(I.EQ.1.OR.I.EQ.11)GOTO 189
        T=ALOG(T)
        S=REAL(I-1)/R
        D(23)=D(23)+1.
        D(24)=D(24)+S
        D(25)=D(25)+S**2
        D(26)=D(26)+T
        D(27)=D(27)+S*T
 189    CONTINUE
      T=D(23)*D(25)-D(24)**2
      AW=(D(26)*D(25)-D(24)*D(27))/T
      BW=(D(23)*D(27)-D(24)*D(26))/T
C
C Recycle phase refinement and calculate pseudo-free CC
C
      NN=0
 190  NN=NN+1
      IF(NN.GT.JM)GOTO 206
      IF(JJ.EQ.0)GOTO 193
      IF(MOD(NN,JJ).NE.0)GOTO 193
      IF(NN.EQ.JM.AND.JJ.GT.1)GOTO 193
      J=1
        DO 191 I=1,NR
        IN(I)=0
        IF(I.GT.NB)GOTO 191
        J=MOD(J*1366+150889,714025)
        IF(J.LT.71403)IN(I)=-1
        FA(I)=FT(I)
        IF(JZ.EQ.0)FA(I)=SQRT(ET(I)*FT(I)*EP(I))
 191    CONTINUE
      CALL DMFFT(LD,LB,LC,NR,LY,LV,IH,IK,IL,IN,FA,WS,PS,
     +WA,WB,B,C,XA,YA,ZA,OC,LM,IP,IJ)
        DO 192 I=1,NB
        PT(I)=-999.
        IF(IN(I).EQ.0)GOTO 192
        U=WA(I)
        V=WB(I)
        FA(I)=SQRT(U**2+V**2)
        IF(FA(I).GT.1.E-8)PT(I)=1.
 192    CONTINUE
      CALL ENORM(NB,FA,PT,SQ,EP,WB)
      CALL CALCC(NB,WB,ET,PT,SQ,D)
      WRITE(*,211)D(13)
      WRITE(LI,211)D(13)
C
C Refine phases
C
 193  Q=1./(SM+0.2)**2
        DO 194 I=1,NR
        IN(I)=0
        FA(I)=0.
        IF(NN.EQ.1.AND.I.GT.NB)GOTO 194
        IN(I)=1
        FA(I)=FT(I)
        IF(JZ.EQ.0)FA(I)=SQRT(ET(I)*FT(I)*EP(I))
 194    CONTINUE
      CALL DMFFT(LD,LB,LC,NR,LY,LV,IH,IK,IL,IN,FA,WS,PS,
     +WA,WB,B,C,XA,YA,ZA,OC,LM,IP,IJ)
        DO 195 I=1,NR
        PT(I)=-999.
        IF(IN(I).EQ.0)GOTO 195
        U=WA(I)
        V=WB(I)
        FA(I)=SQRT(U**2+V**2)
        IF(FA(I).GT.1.E-8)PT(I)=AMOD(720.+57.29578*ATAN2(V,U),360.)
 195    CONTINUE
      CALL ENORM(NR,FA,PT,SQ,EP,WB)
        DO 198 I=NB+1,NR
        WS(I)=0.
        PS(I)=PT(I)
        IF(PT(I).LT.-998.)GOTO 198
        W=EP(I)*SQRT(EXP(AW+BW*AMAX1(0.04,SQ(I))))
        FT(I)=WB(I)*W
        ST(I)=0.3*W
        ET(I)=1.
        W=0.
        W=2.*WB(I)**2
        IF(CP(I).GT.-98.)GOTO 196
        W=AMIN1(W*(0.5658+W*(W*0.0106-0.1304)),W/(0.56+W))
        GOTO 197
 196    W=EXP(AMIN1(2.*W,12.))
        W=(W-1.)/(W+1.)
 197    IF(NN.LT.JM)W=W*0.5
        FA(I)=W
        WS(I)=W
        IF(W.LT.0.0001)PS(I)=-999.
 198    CONTINUE
      CALL CALCC(NB,WB,ET,PT,SQ,D)
        DO 201 I=1,NB
        IF(PT(I).LT.-998.)GOTO 201
        V=0.
        P=D(1)*SQ(I)
        N=MIN0(INT(P),9)
        P=P-REAL(N)
        Q=1.-P
        P=AMIN1(0.9,SQRT(AMAX1(Q*D(N+2)+P*D(N+3),0.0001))+0.05)
        T=2.*WB(I)*ET(I)*P/(1.-P**2)
        IF(CP(I).GT.-98.)GOTO 199
        W=AMIN1(T*(0.5658+T*(T*0.0106-0.1304)),T/(0.56+T))
        GOTO 200
 199    T=EXP(AMIN1(2.*T,12.))
        W=(T-1.)/(T+1.)
 200    V=WM(I)*RW
        FA(I)=W
        W=W*0.4
        U=WS(I)*COS(0.0174533*PS(I))+W*COS(0.0174533*PT(I))+
     +  V*COS(0.0174533*PM(I))
        V=WS(I)*SIN(0.0174533*PS(I))+W*SIN(0.0174533*PT(I))+
     +  V*SIN(0.0174533*PM(I))
        WS(I)=AMIN1(1.,SQRT(U**2+V**2))
        PS(I)=-999.
        IF(WS(I).GT.0.0001)PS(I)=AMOD(720.+57.29578*ATAN2(V,U),360.)
 201    CONTINUE
C
C Renormalize weights
C
        DO 202 I=1,23
        E(I)=0.
 202    CONTINUE
      E(1)=10.*SM**2
      R=1./(SM+0.2)**2
        DO 203 I=1,NB
        P=E(1)*SQ(I)
        N=INT(P)
        IF(N.GT.9)N=9
        P=P-REAL(N)
        Q=1.-P
        E(N+2)=E(N+2)+Q*WS(I)
        E(N+3)=E(N+3)+P*WS(I)
        E(N+13)=E(N+13)+Q
        E(N+14)=E(N+14)+P
 203    CONTINUE
        DO 204 I=2,12
        E(I)=AMIN1(1.,0.3*E(I+11)/AMAX1(E(I),0.0001))
 204    CONTINUE
        DO 205 I=1,NB
        P=E(1)*SQ(I)
        N=MIN0(INT(P),9)
        P=P-REAL(N)
        Q=1.-P
        WS(I)=WS(I)*(Q*E(N+2)+P*E(N+3))
 205    CONTINUE
      IF(JX.NE.0)CALL PHSTAT(NR,NB,LI,F,PS,WS,CP)
      CALL FOMOUT(NB,NN,LI,LY,IH,IK,IL,FT,WS,US,VS,A,
     +'for dens.mod.')
      OPEN(UNIT=LP,FILE=KT(1:LL)//'.fin',STATUS='OLD',ERR=190)
      CLOSE(UNIT=LP,STATUS='DELETE',ERR=190)
      JM=NN+1
      GOTO 190
C
C Final weighting scheme and statistics
C
 206  J=1
        DO 207 I=1,NR
        IN(I)=0
        IF(I.GT.NB)GOTO 207
        J=MOD(J*1366+150889,714025)
        IF(J.LT.71403)IN(I)=-1
        FA(I)=FT(I)
        IF(JZ.EQ.0)FA(I)=SQRT(ET(I)*FT(I)*EP(I))
 207    CONTINUE
      CALL DMFFT(LD,LB,LC,NR,LY,LV,IH,IK,IL,IN,FA,WS,PS,
     +WA,WB,B,C,XA,YA,ZA,OC,LM,IP,IJ)
        DO 208 I=1,NB
        PT(I)=-999.
        IF(IN(I).EQ.0)GOTO 208
        U=WA(I)
        V=WB(I)
        FA(I)=SQRT(U**2+V**2)
        IF(FA(I).GT.1.E-8)PT(I)=AMOD(720.+57.29578*ATAN2(V,U),360.)
 208    CONTINUE
      CALL ENORM(NB,FA,PT,SQ,EP,WB)
      CALL CALCC(NB,WB,ET,PT,SQ,D)
        DO 209 I=1,NB
        P=D(1)*SQ(I)
        N=MIN0(INT(P),9)
        P=P-REAL(N)
        WS(I)=AMIN1(WS(I)*AMAX1(0.,(1.-P)*D(N+2)+P*D(N+3))**1.1/.3,1.)
 209    CONTINUE
      CALL FOMRES(NB,LI,LY,IH,IK,IL,FT,WS,SQ,B,A)
      WRITE(*,211)D(13)
      WRITE(LI,211)D(13)
        DO 210 I=1,NR
        FA(I)=0.
        IN(I)=0
 210    CONTINUE
 211  FORMAT(' Pseudo-free CC =',F6.2,' %')
C
C Output phases to .phs file
C
      KR=KT(1:LL)//'.phs'
      N=LL+4
      OPEN(UNIT=LP,FILE=KR(1:N),STATUS='OLD',IOSTAT=I)
      CLOSE(UNIT=LP,STATUS='DELETE',IOSTAT=I)
      OPEN(UNIT=LP,FILE=KR(1:N),STATUS='NEW',ERR=9)
      T=0.
        DO 212 I=1,NR
        T=AMAX1(T,FT(I),ST(I))
 212    CONTINUE
      T=9999./T
        DO 213 I=1,NR
        WRITE(LP,214)IH(I),IK(I),IL(I),FT(I)*T,AMAX1(WS(I),0.),
     +  AMAX1(PS(I),0.),ST(I)*T
 213    CONTINUE
      IF(RB.LT.-1.)GOTO 217
 214  FORMAT(3I4,F9.2,F8.4,F8.1,F8.2)
C
C Output revised heavy atoms to .hat file and their phases to .pha
C
      KR=KT(1:LL)//'.pha'
      N=LL+4
      CLOSE(UNIT=LP)
      OPEN(UNIT=LP,FILE=KR(1:N),STATUS='OLD',IOSTAT=I)
      CLOSE(UNIT=LP,STATUS='DELETE',IOSTAT=I)
      OPEN(UNIT=LP,FILE=KR(1:N),STATUS='NEW',ERR=9)
      T=0.
        DO 215 I=1,NB
        IF(SA(I).LT.0.)GOTO 215
        FA(I)=EA(I)*EP(I)
        SA(I)=SA(I)*EP(I)
        FA(I)=FA(I)*EXP(-RB*SQ(I))
        PS(I)=AMOD(720.+PS(I)-PD(I),360.)
        T=AMAX1(T,FA(I),SA(I))
 215    CONTINUE
      T=9999./T
        DO 216 I=1,NB
        IF(SA(I).LT.0.)GOTO 216
        WRITE(LP,214)IH(I),IK(I),IL(I),FA(I)*T,AMAX1(WS(I),0.),
     +  PS(I),SA(I)*T
 216    CONTINUE
      KR(LL+1:N)='.hat'
      CLOSE(UNIT=LP)
      OPEN(UNIT=LP,FILE=KR(1:N),STATUS='OLD',IOSTAT=I)
      CLOSE(UNIT=LP,STATUS='DELETE',IOSTAT=I)
      OPEN(UNIT=LP,FILE=KR(1:N),STATUS='NEW',ERR=9)
      IQ=4
      CALL DMFFT(LD,LB,LC,NR,LY,LV,IH,IK,IL,IN,FA,WS,PS,
     +WA,WB,B,C,XA,YA,ZA,OC,LM,IP,IJ)
 217  CALL CPU_TIME(T)
      CALL DATE_AND_TIME(KD,KK,KA,IA)
      WRITE(*,218)KK(1:2),KK(3:4),KK(5:6),T
      WRITE(LI,218)KK(1:2),KK(3:4),KK(5:6),T
 218  FORMAT(/'  ',68('+')/'  +  SHELXE finished at ',A2,':',A2,
     +':',A2,'      Total time:',F13.2,' secs  +'/'  ',68('+'))
      CLOSE(UNIT=LI)
      CALL EXIT(0)
      STOP ' '
      END
C
C ------------------------------------------------------------
C
      SUBROUTINE PHSTAT(NR,NB,LI,F,PH,WS,CP)
      REAL::D(67),F(NR),PH(NR),WS(NR),CP(NR)
C
C Output phasing statistics
C
        DO 1 I=1,11
        D(I)=0.
   1    CONTINUE
        DO 2 I=1,NB
        IF(PH(I).LT.-998.)GOTO 2
        IF(F(I).LT.-998.)GOTO 2
        S=ABS(AMOD(900.+F(I)-PH(I),360.)-180.)
        T=COS(0.0174533*S)
        R=S*WS(I)
        D(1)=D(1)+R
        D(7)=D(7)+1.
        D(8)=D(8)+S
        D(9)=D(9)+WS(I)
        D(10)=D(10)+R
        D(11)=D(11)+T
        IF(CP(I).LT.-98.)GOTO 2
        D(2)=D(2)+1.
        D(3)=D(3)+S
        D(4)=D(4)+WS(I)
        D(5)=D(5)+R
        D(6)=D(6)+T
   2    CONTINUE
      S=AMAX1(D(2),0.01)
      P=D(6)/S
      Q=D(3)/S
      R=0.
      IF(D(2).GT.0.1)R=D(5)/D(4)
      S=D(4)/S
      T=D(11)/D(7)
      U=D(8)/D(7)
      V=D(10)/D(9)
      W=D(9)/D(7)
      WRITE(*,3)T,P,W,S,U,Q,V,R
      WRITE(LI,3)T,P,W,S,U,Q,V,R
   3  FORMAT(' <cos>',F6.3,' /',F6.3,'  <fom>',F6.3,' /',
     +F6.3,'  MPE',F6.1,' /',F5.1,'  wMPE',F5.1,' /',F5.1)
      RETURN
      END
C
C ------------------------------------------------------------
C
      SUBROUTINE FOMRES(NR,LI,LY,IH,IK,IL,FT,WS,SQ,B,A)
      INTEGER::IH(NR),IK(NR),IL(NR)
      REAL::FT(NR),WS(NR),SQ(NR),B(4040),A(1000)
C
C Output mean fom and estimated mapCC as a function of resolution
C
        DO 1 I=1,4000
        B(I)=0.
   1    CONTINUE
      R=0.
        DO 5 I=1,NR
        X=REAL(IH(I))
        Y=REAL(IK(I))
        Z=REAL(IL(I))
        Q=1.
          DO 4 K=33,LY,12
          NH=NINT(X*A(K)+Y*A(K+3)+Z*A(K+6))
          NK=NINT(X*A(K+1)+Y*A(K+4)+Z*A(K+7))
          NL=NINT(X*A(K+2)+Y*A(K+5)+Z*A(K+8))
          IF(NH.NE.IH(I))GOTO 2
          IF(NK.NE.IK(I))GOTO 2
          IF(NL.EQ.IL(I))GOTO 3
   2      IF(NH.NE.-IH(I))GOTO 4
          IF(NK.NE.-IK(I))GOTO 4
          IF(NL.NE.-IL(I))GOTO 4
   3      Q=Q+1.
   4      CONTINUE
        J=MIN0(1000,1+INT(SQRT(10000./SQ(I))))
        B(J)=B(J)+1.
        B(J+1000)=B(J+1000)+WS(I)
        T=(FT(I)**2)/Q
        B(J+2000)=B(J+2000)+T*WS(I)**2
        B(J+3000)=B(J+3000)+T
        R=R+1.
   5    CONTINUE
      R=R*0.1
      P=0.
      M=1
      N=1001
   6  Q=0.
      U=0.
      V=0.
      W=0.
   7  N=N-1
      P=P+B(N)
      Q=Q+B(N)
      U=U+B(N+1000)
      V=V+B(N+2000)
      W=W+B(N+3000)
      IF(P.LT.R*REAL(M)-0.5)GOTO 7
      B(M+4000)=0.01*REAL(N)
      B(M+4010)=U/AMAX1(Q,0.1)
      B(M+4020)=SQRT(V/AMAX1(W,0.0001))
      B(M+4030)=Q
      M=M+1
      IF(M.LT.11)GOTO 6
      WRITE(*,8)(B(I),I=4001,4030),(NINT(B(I)),I=4031,4040)
      WRITE(LI,8)(B(I),I=4001,4030),(NINT(B(I)),I=4031,4040)
   8  FORMAT(/' Mean weight and estimated mapCC as a function of ',
     +'resolution'/' d    inf',10(' -',F5.2)/' <wt>  ',10F7.3/
     +' <mapCC>',F6.3,9F7.3/' N     ',10I7/)
      RETURN
      END
C
C ------------------------------------------------------------
C
      SUBROUTINE FOMOUT(NR,NN,LI,LY,IH,IK,IL,FT,WS,US,VS,A,KT)
      INTEGER::IH(NR),IK(NR),IL(NR)
      REAL::FT(NR),WS(NR),A(1000)
      CHARACTER(LEN=*)::KT
      CHARACTER::KM*8
C
C Output status report
C
      U=0.
      V=0.
      W=0.
        DO 4 I=1,NR
        X=REAL(IH(I))
        Y=REAL(IK(I))
        Z=REAL(IL(I))
        Q=1.
          DO 3 K=33,LY,12
          NH=NINT(X*A(K)+Y*A(K+3)+Z*A(K+6))
          NK=NINT(X*A(K+1)+Y*A(K+4)+Z*A(K+7))
          NL=NINT(X*A(K+2)+Y*A(K+5)+Z*A(K+8))
          IF(NH.NE.IH(I))GOTO 1
          IF(NK.NE.IK(I))GOTO 1
          IF(NL.EQ.IL(I))GOTO 2
   1      IF(NH.NE.-IH(I))GOTO 3
          IF(NK.NE.-IK(I))GOTO 3
          IF(NL.NE.-IL(I))GOTO 3
   2      Q=Q+1.
   3      CONTINUE
        T=FT(I)**2/Q
        U=U+T*WS(I)**2
        V=V+T
        W=W+WS(I)
   4    CONTINUE
      W=W/REAL(NR)
      T=SQRT(U/V)
      IF(NN.NE.0)GOTO 7
      WRITE(*,5)W,KT
      WRITE(LI,5)W,KT
      GOTO 10
   5  FORMAT(' <wt> =',F6.3,1X,A)
   6  FORMAT(' <wt> =',F6.3,', Contrast =',F6.3,', Connect. =',
     +F6.3,1X,A)
   7  WRITE(KM,'(I8)')NN
      L=1
        DO 8 I=1,8
        IF(KM(I:I).EQ.' ')L=I
   8    CONTINUE
      IF(VS.GT.-1.)GOTO 9
      WRITE(*,5)W,KT//' cycle'//KM(L:8)
      WRITE(LI,5)W,KT//' cycle'//KM(L:8)
      GOTO 10
   9  WRITE(*,6)W,VS,US,KT//' cycle'//KM(L:8)
      WRITE(LI,6)W,VS,US,KT//' cycle'//KM(L:8)
  10  RETURN
      END
C
C ------------------------------------------------------------
C
      SUBROUTINE CALCC(NR,A,B,C,SQ,D)
      REAL::A(NR),B(NR),C(NR),SQ(NR),D(67)
C
C Find correlation coefficient as a function of resolution
C
        DO 1 I=1,67
        D(I)=0.
   1    CONTINUE
      Q=0.0001
        DO 2 I=1,NR
        IF(AMIN1(A(I),B(I),C(I)).LT.-98.)GOTO 2
        Q=AMAX1(Q,SQ(I))
   2    CONTINUE
      D(1)=10./Q
        DO 3 I=1,NR
        IF(AMIN1(A(I),B(I),C(I)).LT.-98.)GOTO 3
        P=D(1)*SQ(I)
        N=MIN0(INT(P),9)
        P=P-REAL(N)
        Q=1.-P
        S=A(I)
        T=B(I)
        D(N+2)=D(N+2)+Q*T*S
        D(N+3)=D(N+3)+P*T*S
        D(N+13)=D(N+13)+Q
        D(N+14)=D(N+14)+P
        D(N+24)=D(N+24)+Q*S
        D(N+25)=D(N+25)+P*S
        D(N+35)=D(N+35)+Q*T
        D(N+36)=D(N+36)+P*T
        D(N+46)=D(N+46)+Q*S**2
        D(N+47)=D(N+47)+P*S**2
        D(N+57)=D(N+57)+Q*T**2
        D(N+58)=D(N+58)+P*T**2
   3    CONTINUE
      U=0.
      V=0.
      P=0.
      Q=0.
      R=0.
      S=0.
        DO 4 I=2,57,11
        D(I)=D(I)+0.7*D(I+1)
   4    CONTINUE
        DO 5 I=12,67,11
        D(I)=D(I)+0.7*D(I-1)
   5    CONTINUE
        DO 6 I=2,12
        U=U+D(I)
        V=V+D(I+11)
        P=P+D(I+22)
        Q=Q+D(I+33)
        R=R+D(I+44)
        S=S+D(I+55)
        D(I)=AMAX1(0.,D(I)*D(I+11)-D(I+22)*D(I+33))/
     +  SQRT(AMAX1(D(I+11)*D(I+44)-D(I+22)**2,0.0001)*
     +  AMAX1(D(I+11)*D(I+55)-D(I+33)**2,0.0001))
   6    CONTINUE
      D(13)=100.*(U*V-P*Q)/SQRT(AMAX1(V*R-P**2,0.0001)*
     +AMAX1(V*S-Q**2,0.0001))
      D(2)=0.7*D(2)
      RETURN
      END
C
C ------------------------------------------------------------
C
      SUBROUTINE ENORM(NR,F,S,SQ,EP,E)
      REAL::F(NR),S(NR),SQ(NR),EP(NR),E(NR),B(84)
C
C Estimate E-values
C
      R=0.
        DO 1 I=1,NR
        IF(S(I).LT.-98.)GOTO 1
        R=AMAX1(R,SQ(I))
   1    CONTINUE
      R=20./AMAX1(R,0.0001)
        DO 2 I=1,84
        B(I)=0.
   2    CONTINUE
        DO 3 I=1,NR
        IF(S(I).LT.-98.)GOTO 3
        P=R*SQ(I)
        N=MIN0(INT(P),19)
        P=P-REAL(N)
        Q=1.-P
        T=(F(I)/EP(I))**2
        B(N+1)=B(N+1)+Q*T
        B(N+2)=B(N+2)+P*T
        B(N+22)=B(N+22)+Q
        B(N+23)=B(N+23)+P
   3    CONTINUE
        DO 4 I=1,21
        B(I)=B(I+21)/AMAX1(B(I),0.0001)
        B(I+42)=B(I+63)/AMAX1(B(I+42),0.0001)
   4    CONTINUE
        DO 5 I=1,NR
        E(I)=-99.
        IF(S(I).LT.-98.)GOTO 5
        P=R*SQ(I)
        N=MIN0(INT(P),19)
        P=P-REAL(N)
        Q=1.-P
        T=SQRT(B(N+1)*Q+B(N+2)*P)/EP(I)
        E(I)=SQRT(SQRT(1./(.00390625+1./AMAX1(0.01,T*F(I))**4)))
   5    CONTINUE
      RETURN
      END
C
C ------------------------------------------------------------
C
      SUBROUTINE SXFT(D,N1,N2,IS)
C
C N1 x N2 2D-FFT of data in D. D is a real array of dimension 2*N1*N2
C and the data are stored in two-dimensional "complex" FORTRAN form
C (real before imaginary) with index 1..N1 changing more rapidly than
C 1..N2.  IS = +1 for the forward transform and -1 for the inverse.
C
      REAL::D(*)
      I=2
      J=I*N1
      K=J*N2
        DO 12 ND=1,2
        NJ=1
          DO 6 M=1,J,I
          IF(M.GE.NJ)GOTO 3
          NK=M+I-2
            DO 2 L=M,NK,2
              DO 1 N=L,K,J
              NL=NJ+N-M
              T=D(N)
              D(N)=D(NL)
              D(NL)=T
              T=D(N+1)
              D(N+1)=D(NL+1)
              D(NL+1)=T
   1          CONTINUE
   2        CONTINUE
   3      NI=J/2
   4      IF((NI.LT.I).OR.(NJ.LE.NI))GOTO 5
          NJ=NJ-NI
          NI=NI/2
          GOTO 4
   5      NJ=NJ+NI
   6      CONTINUE
        NI=I
   7    IF(NI.GE.J)GOTO 11
        NJ=2*NI
        T=6.2831853/REAL(IS*NJ/I)
        UR=-2.*SIN(0.5*T)**2
        UI=SIN(T)
        WR=1.
        WI=0.
          DO 10 N=1,NI,I
          NL=N+I-2
            DO 9 L=N,NL,2
              DO 8 M=L,K,NJ
              NK=M+NI
              TR=WR*D(NK)-WI*D(NK+1)
              TI=WR*D(NK+1)+WI*D(NK)
              D(NK)=D(M)-TR
              D(NK+1)=D(M+1)-TI
              D(M)=D(M)+TR
              D(M+1)=D(M+1)+TI
   8          CONTINUE
   9        CONTINUE
          T=WR
          WR=WR*UR-WI*UI+WR
          WI=WI*UR+T*UI+WI
  10      CONTINUE
        NI=NJ
        GOTO 7
  11    I=J
        J=K
  12    CONTINUE
      RETURN
      END
C
C --------------------------------------------------------------------
C
      SUBROUTINE SXIS(NR,IN,IH,IK,IL,FA,SA,PD,FT,ST,PT,
     +VA,VB,WA,WB,FH,PH,SQ,EP,EA)
C
C Sort-merge data using order N algorithm (no NlogN nonsense here !)
C in order of ascending IH(I).  The allowed range of values in IH(I)
C is detemined by the dimension of IN, and is here -1999 ... +1999.
C LARGE version.
C
      INTEGER::IN(3999),IH(NR),IK(NR),IL(NR)
      REAL::FA(NR),SA(NR),PD(NR),FT(NR),ST(NR),PT(NR),
     +VA(NR),VB(NR),WA(NR),WB(NR),FH(NR),PH(NR),SQ(NR),EP(NR),EA(NR)
        DO 1 I=1,3999
        IN(I)=0
   1    CONTINUE
        DO 3 I=1,NR
        J=IH(I)+2000
        IF(J.GT.0.OR.J.LT.4000)GOTO 2
        WRITE(*,'(/A/)')' ** REFLECTION INDEX OUTSIDE '
     +  //'RANGE -1999 TO +1999 **'
        CALL EXIT(0)
        STOP' '
   2    IN(J)=IN(J)+1
        EP(I)=REAL(IH(I))
        VA(I)=REAL(IK(I))
        VB(I)=REAL(IL(I))
        WA(I)=FA(I)
        WB(I)=SA(I)
        FH(I)=PD(I)
        PH(I)=FT(I)
        SQ(I)=ST(I)
        EA(I)=PT(I)
   3    CONTINUE
      J=0
        DO 4 I=1,3999
        K=J
        J=J+IN(I)
        IN(I)=K
   4    CONTINUE
        DO 5 I=1,NR
        J=INT(EP(I))+2000
        IN(J)=IN(J)+1
        J=IN(J)
        IH(J)=NINT(EP(I))
        IK(J)=NINT(VA(I))
        IL(J)=NINT(VB(I))
        FA(J)=WA(I)
        SA(J)=WB(I)
        PD(J)=FH(I)
        FT(J)=PH(I)
        ST(J)=SQ(I)
        PT(J)=EA(I)
   5    CONTINUE
      RETURN
      END
C
C -------------------------------------------------------------------
C
      SUBROUTINE XSYM(KR,A)
C
C Decode SHELX format symmetry operator in KR to A(1-12); set A(1)=999.
C if format error or determinant neither +1. nor -1.
C
      CHARACTER::K*1,KT*20
      CHARACTER(LEN=*)::KR
      REAL::A(12)
      FD=9.E9
        DO 1 I=1,12
        A(I)=0.
   1    CONTINUE
      M=1
      N=10
      I=0
   2  J=0
      KT=' '
   3  I=I+1
      IF(I.GT.LEN(KR))GOTO 6
      K=KR(I:I)
      IF(K.EQ.' ')GOTO 3
      IF(K.EQ.'!')GOTO 16
      IF(J.EQ.0.AND.K.EQ.'-')GOTO 5
      IF(K.EQ.'.')GOTO 5
      IF(K.GE.'0'.AND.K.LE.'9')GOTO 5
      P=1.
      IF(J.GT.1)GOTO 7
      IF(J.EQ.0)GOTO 10
      IF(KT(1:1).NE.'-')GOTO 7
      P=-1.
      GOTO 10
   4  J=0
      KT=' '
   5  IF(J.GT.19)GOTO 3
      J=J+1
      KT(J:J)=K
      GOTO 3
   6  IF(J.EQ.0)GOTO 16
   7    DO 8 L=J+1,12
        KT(L:L)='0'
   8    CONTINUE
      IF(INDEX(KT(1:J),'.').EQ.0)KT(J+1:J+1)='.'
      READ(KT,'(F20.0)',END=17,ERR=17)P
      IF(FD.LT.8.E9)GOTO 9
      IF(K.NE.'/')GOTO 10
      FD=P
      GOTO 2
   9  IF(ABS(P).LT.0.1)GOTO 17
      P=FD/P
      FD=9.E9
  10  IF(K.NE.'X')GOTO 11
      IF(ABS(A(M)).GT.0.0001)GOTO 17
      A(M)=P
      GOTO 14
  11  IF(K.NE.'Y')GOTO 12
      IF(ABS(A(M+1)).GT.0.0001)GOTO 17
      A(M+1)=P
      GOTO 14
  12  IF(K.NE.'Z')GOTO 13
      IF(ABS(A(M+2)).GT.0.0001)GOTO 17
      A(M+2)=P
      GOTO 14
  13  IF(J.EQ.0)GOTO 15
      IF(ABS(A(N)).GT.0.0001)GOTO 17
      A(N)=P
  14  IF(K.EQ.'-')GOTO 4
  15  IF(K.NE.',')GOTO 2
      N=N+1
      M=M+3
      IF(M.LT.8)GOTO 2
  16  IF(ABS(1.-ABS(A(1)*(A(5)*A(9)-A(6)*A(8))+A(2)*(A(6)*A(7)-
     +A(4)*A(9))+A(3)*(A(4)*A(8)-A(5)*A(7)))).LT.0.01)GOTO 18
  17  A(1)=999.
  18  RETURN
      END
C
C ------------------------------------------------------------
C
      SUBROUTINE DMFFT(LD,LB,LC,NR,LY,LV,IH,IK,IL,IN,WA,WB,PH,
     +D,E,B,C,XA,YA,ZA,OC,LM,IP,IJ)
      CHARACTER::KT*80,KF*80,KU*80
      INTEGER::IH(LD),IK(LD),IL(LD),IN(LD),IX(92),IY(92),IZ(92)
      INTEGER::II(92),IJ(LC)
      REAL::WA(LD),WB(LD),PH(LD),D(LD),E(LD),B(LB),C(LC),A(1000)
      REAL::XA(LM),YA(LM),ZA(LM),OC(LM)
      REAL(KIND=8)::DP,DT,DU,DV
      COMMON/FFT/IQ,N1,N2,N3,N4,N5,N6,N7,AZ,EZ,RA,RC,RG,RQ,RS,RV,JX,
     +JP,US,VS,LI,LP,NA,IX,IY,IZ,A
      COMMON/SFUN/KF,KU
C
C Expand to triclinic hemisphere, avoiding redundancy
C
      VS=-2.
      IF(NR+4029.LE.LD)GOTO 2
   1  WRITE(*,'(A/)')' ** Arrays too small for reflection lists: '
     +//'use -l flag to allocate more memory **'
      CALL EXIT(0)
      STOP' '
   2  DX=1./REAL(N1)
      DY=1./REAL(N2)
      DZ=(EZ-AZ)/REAL(N3-1)
      NS=NR+1
      NX=NR
        DO 7 I=1,NR
        IF(IN(I).LT.0)GOTO 7
        W=WA(I)*WB(I)
        IF(W.LT.1.E-8)GOTO 7
        X=REAL(IH(I))
        Y=REAL(IK(I))
        Z=REAL(IL(I))
        M=NX
          DO 6 K=21,LY,12
          NH=NINT(X*A(K)+Y*A(K+3)+Z*A(K+6))
          NK=NINT(X*A(K+1)+Y*A(K+4)+Z*A(K+7))
          NL=NINT(X*A(K+2)+Y*A(K+5)+Z*A(K+8))
          T=0.0174533*PH(I)-6.283185*(X*A(K+9)+Y*A(K+10)+Z*A(K+11))
          S=1.
          L=NL
          IF(L.EQ.0)L=NK
          IF(L.EQ.0)L=NH
          IF(L.GE.0)GOTO 3
          S=-S
          NH=-NH
          NK=-NK
          NL=-NL
   3      J=M
   4      J=J+1
          IF(J.GT.NX)GOTO 5
          IF(NH.NE.IH(J))GOTO 4
          IF(NK.NE.IK(J))GOTO 4
          IF(NL.NE.IL(J))GOTO 4
          GOTO 6
   5      IF(NX.GE.LD)GOTO 1
          NX=NX+1
          IH(NX)=NH
          IK(NX)=NK
          IL(NX)=NL
          WA(NX)=W*COS(T)
          WB(NX)=S*W*SIN(T)
   6      CONTINUE
   7    CONTINUE
C
C Sort expanded list on l (but not h and k)
C
        DO 8 I=NS,NX
        NH=IH(I)
        IF(NH.LT.0)NH=NH+N1
        NK=IK(I)
        IF(NK.LT.0)NK=NK+N2
        IK(I)=2*(NH+NK*N1)
   8    CONTINUE
      L=1-NS
        DO 9 I=NS,NR+3999
        IH(I)=0
   9    CONTINUE
        DO 10 I=NS,NX
        J=IL(I)+NR+2000
        IH(J)=IH(J)+1
        PH(I)=REAL(IK(I))
        IN(I)=IL(I)
        D(I+L)=WA(I)
        E(I+L)=WB(I)
  10    CONTINUE
      J=NR
        DO 11 I=NS,NR+3999
        K=J
        J=J+IH(I)
        IH(I)=K
  11    CONTINUE
        DO 12 I=NS,NX
        J=IN(I)+NR+2000
        IH(J)=IH(J)+1
        J=IH(J)
        IK(J)=NINT(PH(I))
        IL(J)=IN(I)
        WA(J)=D(I+L)
        WB(J)=E(I+L)
  12    CONTINUE
C
C If IQ>1 do FFT of extra layers first and store in C
C
      IF(IQ.LT.2)GOTO 21
      IF(IP.EQ.0)GOTO 21
      M=0
      Z=6.283185*(AZ-REAL(N7)*DZ)
        DO 20 NL=1-N7,N3+N7
        IF(NL.GT.0.AND.NL.LE.N3)GOTO 19
          DO 13 I=1,N4
          B(I)=0.
  13      CONTINUE
        IF(IQ.NE.4)GOTO 14
        IF(NL.NE.0.AND.NL.NE.N3+1)GOTO 17
  14    L=0
        U=1.
        V=0.
          DO 16 N=NS,NX
          IF(IL(N).EQ.L)GOTO 15
          L=IL(N)
          T=Z*REAL(L)
          U=COS(T)
          V=SIN(T)
  15      I=IK(N)+1
          B(I)=B(I)+U*WA(N)+V*WB(N)
          B(I+1)=B(I+1)+V*WA(N)-U*WB(N)
  16      CONTINUE
        CALL SXFT(B(1),N1,N2,1)
  17    J=M
          DO 18 I=1,N4,2
          J=J+1
          C(J)=B(I)
  18      CONTINUE
  19    M=M+N1*N2
        Z=Z+6.283185*DZ
  20    CONTINUE
C
C Fourier transform to asymmetric unit
C
  21    DO 22 I=1,N5
        B(I)=0.
  22    CONTINUE
      M=1
      Z=6.283185*AZ
        DO 25 NL=1,N3
        L=0
        U=1.
        V=0.
          DO 24 N=NS,NX
          IF(IL(N).EQ.L)GOTO 23
          L=IL(N)
          T=Z*REAL(L)
          U=COS(T)
          V=SIN(T)
  23      I=IK(N)+M
          B(I)=B(I)+U*WA(N)+V*WB(N)
          B(I+1)=B(I+1)+V*WA(N)-U*WB(N)
  24      CONTINUE
        CALL SXFT(B(M),N1,N2,1)
        M=M+N4
        Z=Z+6.283185*DZ
  25    CONTINUE
      DP=0.
        DO 26 I=1,N5,2
        DT=B(I)**2
        DP=DP+DT
  26    CONTINUE
      P=DP
      P=SQRT(0.5*REAL(N5)/P)
        DO 27 I=1,N5,2
        B(I)=B(I)*P
        B(I+1)=0.
  27    CONTINUE
C
C Update rest of C array
C
      IF(IQ.LT.0)GOTO 115
      IF(IQ.EQ.0)GOTO 105
      IF(IQ.EQ.1)GOTO 80
      J=N7*N1*N2
      M=N5/2+J
      IF(IP.EQ.0)GOTO 29
        DO 28 I=1,J
        C(I)=C(I)*P
        C(I+M)=C(I+M)*P
  28    CONTINUE
  29  K=J
        DO 30 I=1,N5,2
        K=K+1
        C(K)=B(I)
  30    CONTINUE
      IF(IP.NE.0)GOTO 32
        DO 31 I=1,J
        C(I)=C(IJ(I)+J)
        C(I+M)=C(IJ(I+M)+J)
  31    CONTINUE
C
C Density at atomic sites (IQ=4)
C
  32  IF(IQ.NE.4)GOTO 80
      IF(NA.EQ.0)GOTO 38
      WRITE(*,36)
      WRITE(LI,36)
      T=AZ+0.5*DZ*REAL(N3)
      NT=N1*N2
        DO 35 N=1,NA
        NH=0
        NK=0
        NL=0
        U=0.
        V=0.
        W=0.
        Q=9.E9
          DO 34 K=21,LY,12
            DO 33 L=LY+12,LV,3
            Z=AMOD(A(K+6)*XA(N)+A(K+7)*YA(N)+A(K+8)*ZA(N)+A(K+11)+
     +      A(L+2)-T,1.)-0.5
            IF(ABS(Z).GE.Q)GOTO 33
            Q=ABS(Z)
            Z=(Z+T-AZ)/DZ
            NL=NINT(Z)
            W=Z-REAL(NL)
            X=AMOD(A(K)*XA(N)+A(K+1)*YA(N)+A(K+2)*ZA(N)+A(K+9)+
     +      A(L)-0.5,1.)/DX
            NH=NINT(X)
            U=X-REAL(NH)
            IF(NH.GE.N1)NH=0
            Y=AMOD(A(K+3)*XA(N)+A(K+4)*YA(N)+A(K+5)*ZA(N)+A(K+10)+
     +      A(L+1)-0.5,1.)/DY
            NK=NINT(Y)
            V=Y-REAL(NK)
            IF(NK.GE.N2)NK=0
  33        CONTINUE
  34      CONTINUE
        I=N1*(N2*(N7+NL)+NK)+NH+1
        J=I-1
        IF(NH.EQ.0)J=J+N1
        K=I+1
        IF(NH.EQ.N1)K=K-N1
        L=I-N1
        IF(NK.EQ.0)L=L+NT
        M=I+N1
        IF(NK.EQ.N2)M=M-NT
        UU=U**2
        VV=V**2
        WW=W**2
        P=C(I)*(1.-UU-VV-WW)+0.5*(C(J)*(UU-U)+C(K)*(UU+U)+C(L)*
     +  (VV-V)+C(M)*(VV+V)+C(I-NT)*(WW-W)+C(I+NT)*(WW+W))
        WRITE(*,37)N,XA(N),YA(N),ZA(N),OC(N),P
        WRITE(LI,37)N,XA(N),YA(N),ZA(N),OC(N),P
  35    CONTINUE
  36  FORMAT(/' Density (in map sigma units) at input heavy atom sites'
     +//'  Site     x        y        z     occ*Z    density')
  37  FORMAT(I5,4F9.4,F9.2)
C
C Peaksearch (IQ=4)
C
  38  T=RQ
      M=N1*N2
      NT=M*N7
      N=NA
      NC=0
        DO 51 NL=0,N3-1
          DO 50 NK=0,N2-1
            DO 49 NH=0,N1-1
            NC=NC+1
            NT=NT+1
            IF(IJ(NT).LT.NC)GOTO 49
            P=C(NT)
            IF(P.LT.T)GOTO 39
            IF(P.LT.AMAX1(C(NT-M),C(NT+M)))GOTO 39
            J=NT+1
            IF(NH.EQ.N1-1)J=J-N1
            IF(P.LT.AMAX1(C(J),C(J-M),C(J+M)))GOTO 39
            I=NT-1
            IF(NH.EQ.0)I=I+N1
            IF(P.LT.AMAX1(C(I),C(I-M),C(I+M)))GOTO 39
            K=-N1
            IF(NK.EQ.0)K=K+M
            NI=NT+K
            IF(P.LT.AMAX1(C(NI),C(NI-M),C(NI+M)))GOTO 39
            L=N1
            IF(NK.EQ.N2-1)L=L-M
            NI=NT+L
            IF(P.LT.AMAX1(C(NI),C(NI-M),C(NI+M)))GOTO 39
            NI=I+K
            IF(P.LT.AMAX1(C(NI),C(NI-M),C(NI+M)))GOTO 39
            NI=I+L
            IF(P.LT.AMAX1(C(NI),C(NI-M),C(NI+M)))GOTO 39
            NI=J+K
            IF(P.LT.AMAX1(C(NI),C(NI-M),C(NI+M)))GOTO 39
            NI=J+L
            IF(P.LT.AMAX1(C(NI),C(NI-M),C(NI+M)))GOTO 39
            GOTO 40
  39        IF(P.GT.-T)GOTO 49
            IF(P.GT.AMIN1(C(NT-M),C(NT+M)))GOTO 49
            J=NT+1
            IF(NH.EQ.N1-1)J=J-N1
            IF(P.GT.AMIN1(C(J),C(J-M),C(J+M)))GOTO 49
            I=NT-1
            IF(NH.EQ.0)I=I+N1
            IF(P.GT.AMIN1(C(I),C(I-M),C(I+M)))GOTO 49
            K=-N1
            IF(NK.EQ.0)K=K+M
            NI=NT+K
            IF(P.GT.AMIN1(C(NI),C(NI-M),C(NI+M)))GOTO 49
            L=N1
            IF(NK.EQ.N2-1)L=L-M
            NI=NT+L
            IF(P.GT.AMIN1(C(NI),C(NI-M),C(NI+M)))GOTO 49
            NI=I+K
            IF(P.GT.AMIN1(C(NI),C(NI-M),C(NI+M)))GOTO 49
            NI=I+L
            IF(P.GT.AMIN1(C(NI),C(NI-M),C(NI+M)))GOTO 49
            NI=J+K
            IF(P.GT.AMIN1(C(NI),C(NI-M),C(NI+M)))GOTO 49
            NI=J+L
            IF(P.GT.AMIN1(C(NI),C(NI-M),C(NI+M)))GOTO 49
  40        U=C(I)-C(J)
            V=C(NT+K)-C(NT+L)
            W=C(NT-M)-C(NT+M)
            Q=U/(C(I)+C(J)-2.*P)
            R=V/(C(NT+K)+C(NT+L)-2.*P)
            S=W/(C(NT-M)+C(NT+M)-2.*P)
            P=P-0.125*(U*Q+V*R+W*S)
            U=DX*(REAL(NH)+0.5*Q)
            V=DY*(REAL(NK)+0.5*R)
            W=AZ+DZ*(REAL(NL)+0.5*S)
C
C Test for equivalent peaks
C
              DO 43 K=21,LY,12
              Q=A(K)*U+A(K+1)*V+A(K+2)*W+A(K+9)
              R=A(K+3)*U+A(K+4)*V+A(K+5)*W+A(K+10)
              S=A(K+6)*U+A(K+7)*V+A(K+8)*W+A(K+11)
                DO 42 L=LY+12,LV,3
                XX=Q+A(L)
                YY=R+A(L+1)
                ZZ=S+A(L+2)
                  DO 41 I=NA+1,N
                  X=AMOD(XX-XA(I),1.)-0.5
                  Y=AMOD(YY-YA(I),1.)-0.5
                  Z=AMOD(ZZ-ZA(I),1.)-0.5
                  IF(A(8)*X**2+A(9)*Y**2+A(10)*Z**2+A(11)*Y*Z+
     +            A(12)*X*Z+A(13)*X*Y.GT.0.1)GOTO 41
                  IF(ABS(P).LE.ABS(OC(I)))GOTO 48
                  GOTO 47
  41              CONTINUE
  42            CONTINUE
  43          CONTINUE
  44        IF(N.LT.LM)GOTO 46
            T=T+0.1
            IF(ABS(P).LT.T)GOTO 49
            NI=N
            NJ=NA
            N=NA
  45        NJ=NJ+1
            IF(NJ.GT.NI)GOTO 44
            IF(ABS(OC(NJ)).LT.T)GOTO 45
            N=N+1
            XA(N)=XA(NJ)
            YA(N)=YA(NJ)
            ZA(N)=ZA(NJ)
            OC(N)=OC(NJ)
            GOTO 45
  46        N=N+1
            I=N
  47        XA(I)=U
            YA(I)=V
            ZA(I)=W
            OC(I)=P
  48        C(NT)=C(NT)+0.0001
  49        CONTINUE
  50      CONTINUE
  51    CONTINUE
C
C Sort peaks
C
      K=N-NA
  52  K=INT(REAL(K)/1.3)
      IF(K.LT.1)K=1
      IF(K.EQ.9.OR.K.EQ.10)K=11
      M=0
        DO 53 I=NA+1,N-K
        Q=OC(I+K)
        IF(ABS(Q).LE.ABS(OC(I)))GOTO 53
        OC(I+K)=OC(I)
        OC(I)=Q
        Q=XA(I+K)
        XA(I+K)=XA(I)
        XA(I)=Q
        Q=YA(I+K)
        YA(I+K)=YA(I)
        YA(I)=Q
        Q=ZA(I+K)
        ZA(I+K)=ZA(I)
        ZA(I)=Q
        M=1
  53    CONTINUE
      IF(M+K.GT.1)GOTO 52
C
C Output .hat file
C
      WRITE(LP,'(A)')'TITL Revised heavy atom sites'
      WRITE(LP,'(A,F9.5,3F9.3,3F9.2)')'CELL',(A(I),I=1,7)
      WRITE(LP,'(A,I3)')'LATT',NINT(A(20))
        DO 60 K=33,LY,12
        KT='SYMM'
        M=5
        NJ=K+9
          DO 59 I=K,K+6,3
          L=M
          U=AMOD(9.01+A(NJ),1.)-0.01
          NJ=NJ+1
          IF(U.LT.0.01)GOTO 55
          WRITE(KT(73:80),'(F8.5)')U
          NK=1
            DO 54 J=74,80
            IF(KT(J:J).NE.'0')NK=J-73
  54        CONTINUE
          KT(M+1:M+NK)=KT(74:73+NK)
          M=M+NK
  55        DO 58 J=0,2
            IF(A(I+J).LT.-0.1)GOTO 56
            IF(A(I+J).LT.0.1)GOTO 58
            IF(M.EQ.L)GOTO 57
            M=M+1
            KT(M:M)='+'
            GOTO 57
  56        M=M+1
            KT(M:M)='-'
  57        M=M+1
            KT(M:M)=CHAR(J+88)
  58        CONTINUE
          M=M+2
          KT(M-1:M)=', '
  59      CONTINUE
        WRITE(LP,'(A)')KT(1:M-2)
  60    CONTINUE
      J=1
      K=1
        DO 61 I=1,80
        IF(KF(I:I).GT.' ')J=I
        IF(KU(I:I).GT.' ')K=I
  61    CONTINUE
      WRITE(LP,'(A/A)')KF(1:J),KU(1:K)
C
C Check special positions, write peaks to .hat file, analyze distances
C
      IF(JP.EQ.3)THEN
      WRITE(*,79)
      WRITE(LI,79)
      ELSE
      WRITE(*,78)
      WRITE(LI,78)
      ENDIF
        DO 75 I=NA+1,N
        U=XA(I)
        V=YA(I)
        W=ZA(I)
          DO 64 J=1,3
          R=0.
          S=0.
          T=0.
          P=1.
          NJ=LY+15
            DO 63 K=21,LY,12
              DO 62 L=NJ,LV,3
              X=AMOD(A(K)*U+A(K+1)*V+A(K+2)*W+A(K+9)+A(L)-U,1.)-0.5
              Y=AMOD(A(K+3)*U+A(K+4)*V+A(K+5)*W+A(K+10)+
     +        A(L+1)-V,1.)-0.5
              Z=AMOD(A(K+6)*U+A(K+7)*V+A(K+8)*W+A(K+11)+
     +        A(L+2)-W,1.)-0.5
              Q=A(8)*X**2+A(9)*Y**2+A(10)*Z**2+A(11)*Y*Z+
     +        A(12)*X*Z+A(13)*X*Y
              IF(Q.GT.0.1)GOTO 62
              R=R+X
              S=S+Y
              T=T+Z
              P=P+1.
  62          CONTINUE
            NJ=LY+12
  63        CONTINUE
          U=U+R/P
          V=V+S/P
          W=W+T/P
  64      CONTINUE
        XA(I)=U
        YA(I)=V
        ZA(I)=W
        M=1
        C(M)=999980.
          DO 71 K=21,LY,12
          Q=A(K)*U+A(K+1)*V+A(K+2)*W+A(K+9)
          R=A(K+3)*U+A(K+4)*V+A(K+5)*W+A(K+10)
          S=A(K+6)*U+A(K+7)*V+A(K+8)*W+A(K+11)
            DO 70 L=LY+12,LV,3
            XX=Q+A(L)
            YY=R+A(L+1)
            ZZ=S+A(L+2)
              DO 69 J=1,N
              X=AMOD(XX-XA(J),1.)-0.5
              Y=AMOD(YY-YA(J),1.)-0.5
              Z=AMOD(ZZ-ZA(J),1.)-0.5
              T=A(8)*X**2+A(9)*Y**2+A(10)*Z**2+A(11)*Y*Z+
     +        A(12)*X*Z+A(13)*X*Y
              IF(J.EQ.I.AND.T.LT.0.1)GOTO 69
              IF(J.GT.NA)GOTO 65
              IF(T.GE.C(1))GOTO 69
              C(1)=T
              IN(1)=J
              XA(I)=XA(J)+X
              YA(I)=YA(J)+Y
              ZA(I)=ZA(J)+Z
              GOTO 69
  65          NI=M+1
              NJ=NI
  66          NJ=NJ-1
              IF(NJ.LT.2)GOTO 67
              IF(T.LT.C(NJ))GOTO 66
  67          NJ=NJ+1
              M=MIN0(NI,5)
  68          C(NI)=C(NI-1)
              IN(NI)=IN(NI-1)
              NI=NI-1
              IF(NI.GT.NJ)GOTO 68
              C(NJ)=T
              IN(NJ)=J-NA
  69          CONTINUE
  70        CONTINUE
  71      CONTINUE
        KT=' '
        WRITE(KT,76)(IN(K),SQRT(C(K)),K=1,M)
        IF(JP.EQ.3)KT(1:16)='                '
        M=0
          DO 72 K=1,80
          IF(KT(K:K).EQ.' ')GOTO 72
          M=M+1
          KT(M:M)=KT(K:K)
          IF(KT(K:K).EQ.'#')KT(M:M)=' '
  72      CONTINUE
        M=MIN0(M,43)
        L=MIN0(I-NA,9999)
        WRITE(*,77)L,XA(I),YA(I),ZA(I),OC(I),KT(1:M)
        WRITE(LI,77)L,XA(I),YA(I),ZA(I),OC(I),KT(1:M)
        WRITE(KT,'(I4)')L
        KT(5:7)='  1'
        IF(L.LT.100)KT(2:2)='0'
        IF(L.LT.10)KT(3:3)='0'
        KT(1:1)='S'
        K=4
  73    K=K+1
        IF(K.GT.80)GOTO 74
        IF(KF(K:K).LE.' ')GOTO 73
        KT(1:1)=KF(K:K)
        IF(KF(K+1:K+1).NE.' '.AND.L.LT.100)KT(2:2)=KF(K+1:K+1)
  74    WRITE(LP,'(A,3F10.6,F8.4,A)')KT(1:7),XA(I),YA(I),ZA(I),
     +  OC(I)/(P*ABS(OC(NA+1))),'  0.2'
  75    CONTINUE
      WRITE(LP,'(A/A)')'HKLF 3','END '
      GOTO 115
  76  FORMAT(I5,'/',F8.2,'#',4('#',I5,'/',F8.2))
  77  FORMAT(I4,3F8.4,F6.1,2X,A)
  78  FORMAT(/' Site    x       y       z  h(sig) near old  ',
     +'near new')
  79  FORMAT(/' Site    x       y       z   h(sig)  near sites')
C
C Estimate density threshold
C
  80  P=9.E9
      Q=-9.E9
        DO 81 I=1,N5,2
        P=AMIN1(P,B(I))
        Q=AMAX1(Q,B(I))
  81    CONTINUE
      T=3998./AMAX1(Q-P,0.0001)
        DO 82 I=NS,NR+3999
        PH(I)=0.
  82    CONTINUE
      M=0
        DO 84 L=1,N3
        U=1.
        IF(L.EQ.1.OR.L.EQ.N3)U=0.5
          DO 83 J=1,N4,2
          I=NINT((B(M+J)-P)*T)+NS
          PH(I)=PH(I)+U
  83      CONTINUE
        M=M+N4
  84    CONTINUE
      R=0.07
      IF(IQ.GT.1)R=0.05
      R=0.5*(1.-R)*REAL(N4*(N3-1))
      U=0.
        DO 85 I=NS,NR+3999
        U=U+PH(I)
        T=P+REAL(I-NR)*(Q-P)/3999.
        IF(U.GE.R)GOTO 86
  85    CONTINUE
  86  IF(IQ.GT.1)GOTO 88
        DO 87 I=1,N5,2
        IF(B(I).LT.T)B(I)=0.
  87    CONTINUE
      GOTO 105
C
C Calculate variance of density at distance of 2.42A
C
  88  NJ=N1*N2*N7
      NT=NJ+1
      NC=0
      M=1
      NL=0
      DU=0.
      DV=0.
      R=0.
      S=0.
        DO 94 L=0,N3-1
          DO 93 K=0,N2-1
            DO 89 J=1,92
            II(J)=N1*(MOD(IY(J)+K,N2)+N2*(L+IZ(J)))+NT
  89        CONTINUE
            DO 92 I=0,N1-1
            NC=NC+1
            P=9.E9
            IF(B(M).GT.T)GOTO 91
            N=IJ(NC+NJ)
            P=B(2*N-1)
            IF(N.LT.NC)GOTO 91
            P=0.
            Q=0.
              DO 90 J=1,92
              N=MOD(IX(J)+I,N1)+II(J)
              P=P+C(N)
              Q=Q+C(N)**2
  90          CONTINUE
            P=SQRT(Q/92.-(P/92.)**2)
            DP=P
            DU=DU+DP
            DV=DV+DP**2
            NL=NL+1
  91        B(M)=P
            M=M+2
  92        CONTINUE
  93      CONTINUE
  94    CONTINUE
      V=DV
      U=DU
      VS=REAL(NL)*V/U**2-1.
C
C Sum for histogram
C
      P=9.E9
      Q=-9.E9
        DO 95 I=1,N5,2
        IF(B(I).GT.8.E9)GOTO 95
        P=AMIN1(P,B(I))
        Q=AMAX1(Q,B(I))
  95    CONTINUE
      T=3998./AMAX1(Q-P,0.0001)
        DO 96 I=NS,NR+3999
        PH(I)=0.
  96    CONTINUE
      M=0
        DO 98 L=1,N3
        U=1.
        IF(L.EQ.1.OR.L.EQ.N3)U=0.5
          DO 97 J=1,N4,2
          I=NINT(AMIN1(3998.,(B(M+J)-P)*T))+NS
          PH(I)=PH(I)+U
  97      CONTINUE
        M=M+N4
  98    CONTINUE
C
C Construct histogrammed mask
C
      S=0.5*REAL(N4*(N3-1))
      U=0.
      W=1./AMAX1(0.0001,0.5*RC)
        DO 99 I=NS,NR+3999
        U=U+PH(I)
        PH(I)=0.
        R=U/S-RS
        IF(R.GT.0.)PH(I)=1.
        R=R*W
        IF(ABS(R).LT.1.)PH(I)=SIN(0.7853982*(R+1.))**2
  99    CONTINUE
        DO 100 I=1,N5,2
        K=NINT(AMIN1(3998.,(B(I)-P)*T))+NS
        B(I)=PH(K)-0.5
 100    CONTINUE
C
C Calculate connectivity
C
      J=-1
      N=0
        DO 103 L=1,N3-1
          DO 102 K=1,N2
          NK=N6
          IF(K.EQ.N2)NK=N6-N4
          NL=N4-N6
          IF(K.EQ.1)NL=NL+N4
            DO 101 I=1,N1
            J=J+2
            NH=J+2
            IF(I.EQ.N1)NH=NH-N6
            NI=J-2
            IF(I.EQ.1)NI=NI+N6
            V=B(J)
            N=N+NINT(SIGN(1.,V*B(NH+NK+N4))+SIGN(1.,V*B(J+NK+N4))+
     +      SIGN(1.,V*B(NH+N4))+SIGN(1.,V*B(NH+NK))+SIGN(1.,V*B(NH))+
     +      SIGN(1.,V*B(J+NK))+SIGN(1.,V*B(NI+NK))+SIGN(1.,V*B(J+NL))+
     +      SIGN(1.,V*B(J+N4))+SIGN(1.,V*B(NI+NL))+SIGN(1.,V*B(NH+NL))+
     +      SIGN(1.,V*B(NI+N4))+SIGN(1.,V*B(NI+NK+N4)))
 101        CONTINUE
 102      CONTINUE
 103    CONTINUE
      US=REAL(N)/(13.*S)
      IF(IQ.EQ.3)GOTO 115
C
C Modify density
C
      J=N1*N2*N7
        DO 104 I=1,N5,2
        J=J+1
        U=B(I)+0.5
        P=AMAX1(0.0001,C(J))
        B(I)=U*SQRT(P**4/(RV**2+P**2))-(1.-U)*RG*C(J)
 104    CONTINUE
C
C Inverse FFT
C
 105  N=N4*(N3-1)
        DO 106 I=1,N4,2
        B(I)=0.5*B(I)
        B(I+N)=0.5*B(I+N)
 106    CONTINUE
        DO 107 I=2,N5,2
        B(I)=0.
 107    CONTINUE
      M=1
        DO 108 L=1,N3
        CALL SXFT(B(M),N1,N2,-1)
        M=M+N4
 108    CONTINUE
      NT=NR+N3
      NN=-9999
        DO 114 I=1,NR
        D(I)=0.
        E(I)=0.
        IF(IN(I).EQ.0)GOTO 114
        X=REAL(IH(I))
        Y=REAL(IK(I))
        Z=REAL(IL(I))
          DO 113 K=21,LY,12
          NH=NINT(X*A(K)+Y*A(K+3)+Z*A(K+6))
          NK=NINT(X*A(K+1)+Y*A(K+4)+Z*A(K+7))
          NL=NINT(X*A(K+2)+Y*A(K+5)+Z*A(K+8))
          S=1.
          L=NL
          IF(L.EQ.0)L=NK
          IF(L.EQ.0)L=NH
          IF(L.GE.0)GOTO 109
          S=-S
          NH=-NH
          NK=-NK
          NL=-NL
 109      IF(NL.EQ.NN)GOTO 111
          P=6.283185*REAL(NL)
          T=P*AZ
          U=COS(T)
          V=SIN(T)
          T=P*DZ
          P=COS(T)
          Q=SIN(T)
            DO 110 N=NS,NT
            WA(N)=U
            WB(N)=V
            T=U*Q
            U=U*P-V*Q
            V=T+V*P
 110        CONTINUE
          NN=NL
 111      U=0.
          V=0.
          IF(NH.LT.0)NH=NH+N1
          IF(NK.LT.0)NK=NK+N2
          J=2*(NH+NK*N1)+1
            DO 112 N=NS,NT
            U=U+WA(N)*B(J)+WB(N)*B(J+1)
            V=V-WA(N)*B(J+1)+WB(N)*B(J)
            J=J+N4
 112        CONTINUE
          V=V*S
          T=6.283185*(X*A(K+9)+Y*A(K+10)+Z*A(K+11))
          P=COS(T)
          Q=SIN(T)
          D(I)=D(I)+U*P-V*Q
          E(I)=E(I)+V*P+U*Q
 113      CONTINUE
 114    CONTINUE
 115  RETURN
      END
