/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "util.h"

void *emalloc(char *pos, size_t size)
{
  void *p;

  if (size == 0)
    return NULL;

  if ((p=malloc(size)) == NULL) {
    malloc_error_exit(pos, size);
  }
  return p;
}

void malloc_error_exit(char *pos, size_t size)
{
  fprintf(stderr, "ERROR: Can't allocate memory(size=%ld) in %s. \n", (unsigned long) size,pos);
  exit(1);
}

char *cfg_fgets(char *buf, int size, FILE *fp)
{
  int ok, i, len;
  char *ret;

  do {
    ret = fgets(buf, size, fp);
    if (ret == NULL) return NULL;
    ok = 1;
    for (i=0;i<size;i++)
      if (buf[i] != ' ' && buf[i] != '\t') break;
    if (buf[i]=='*') ok = 0;
    if (buf[i]=='\n') ok = 0; 
    if (buf[i]=='!') ok = 0;
    if (buf[i]=='#') ok = 0;
  } while (!ok);

  ok = 0;
  do {
    len = strlen(buf);
    if (len>=2 && buf[len-1]=='\n' && buf[len-2]=='\\') {
      ret = fgets(&buf[len-2], size-(len-2), fp);
      if (ret == NULL) return NULL;
    } else {
      ok=1;
    }
  } while (!ok);

  return ret;
}

char *par_fgets(char *buf, int size, FILE *fp)
{
  int ok, i,len;
  char *ret;
  do {
    ret = fgets(buf, size, fp);
    if (ret == NULL) return NULL;
    ok = 1;
    for (i=0;i<size;i++)
      if (buf[i] != ' ' && buf[i] != '\t') break;
    if (buf[i]=='*') ok = 0;
    if (buf[i]=='\n') ok = 0;
    if (buf[i]=='!') ok = 0;
  } while (!ok);

  ok = 0;
  do {
    len = strlen(buf);
    if (len>=2 && buf[len-1]=='\n' && buf[len-2]=='-') {
      ret = fgets(&buf[len-2], size-(len-2), fp);
      if (ret == NULL) return NULL;
    } else {
      ok=1;
    }
  } while (!ok);

  for (i=0;i<size;i++) {
    if (buf[i] >= 'a' && buf[i] <= 'z') {
      buf[i] += 'A'-'a';
    }
    if (buf[i]=='\0') break;
    if (buf[i]=='!') {
      buf[i]='\0';
      break;
    }
  }
  /* printf(buf); */

  return buf;
}

char *top_fgets(char *buf, int size, FILE *fp)
{
  int ok, i;
  char *ret;
  do {
    ret = fgets(buf, size, fp);
    if (ret == NULL) return NULL;
    ok = 1;
    for (i=0;i<size;i++)
      if (buf[i] != ' ' && buf[i] != '\t') break;
    if (buf[i]=='*') ok = 0;
    if (buf[i]=='\n') ok = 0;
    if (buf[i]=='!') ok = 0;
  } while (!ok);

  for (i=0;i<size;i++) {
    if (buf[i] >= 'a' && buf[i] <= 'z') {
      buf[i] += 'A'-'a';
    }
    if (buf[i]=='\0') break;
    if (buf[i]=='!') {
      buf[i]='\0';
      break;
    }
  }


  return ret;
}

/* vector calculation routines */
/* c = a + b; */
void v_add(vec_t *c, vec_t *a, vec_t *b)
{
  c->x = a->x + b->x;
  c->y = a->y + b->y;
  c->z = a->z + b->z;
}


/* c = a - b; */
void v_sub(vec_t *c, vec_t *a, vec_t *b)
{
  c->x = a->x - b->x;
  c->y = a->y - b->y;
  c->z = a->z - b->z;
}

/* c = a * b; */
void v_mul(vec_t *c, double a, vec_t *b)
{
  c->x = a * b->x;
  c->y = a * b->y;
  c->z = a * b->z;
}

double v_inner_pro(vec_t *a, vec_t *b)
{
  return (a->x*b->x + a->y*b->y + a->z*b->z);
}

void v_outer_pro(vec_t *c, vec_t *a, vec_t *b)
{
  c->x = a->y*b->z - a->z*b->y;
  c->y = a->z*b->x - a->x*b->z;
  c->z = a->x*b->y - a->y*b->x;
}

double v_len(vec_t *a)
{
  return sqrt(a->x*a->x + a->y*a->y + a->z*a->z);
}

double v_ang(vec_t *a, vec_t *b)
{
  double la,lb, ab;
  
  la = sqrt(a->x*a->x + a->y*a->y + a->z*a->z);
  lb = sqrt(b->x*b->x + b->y*b->y + b->z*b->z);
  ab = a->x*b->x + a->y*b->y + a->z*b->z;

  return acos(ab/(la*lb));
}

void v_norm(vec_t *a)
{
  double l;
  l = sqrt(a->x*a->x + a->y*a->y + a->z*a->z);

  a->x /= l; a->y /= l; a->z /= l;
}

/* b = mat * a; */
void v_mat_mul(vec_t *b, double mat[3][3], vec_t *a)
{
  vec_t tmp;
  tmp.x = mat[0][0]*a->x + mat[0][1]*a->y + mat[0][2]*a->z;
  tmp.y = mat[1][0]*a->x + mat[1][1]*a->y + mat[1][2]*a->z;
  tmp.z = mat[2][0]*a->x + mat[2][1]*a->y + mat[2][2]*a->z;

  *b=tmp;
}

/* b = a * mat; */
void v_mul_mat(vec_t *b, vec_t *a, double mat[3][3])
{
  vec_t tmp;
  tmp.x = a->x*mat[0][0] + a->y*mat[1][0] + a->z*mat[2][0];
  tmp.y = a->x*mat[0][1] + a->y*mat[1][1] + a->z*mat[2][1];
  tmp.z = a->x*mat[0][2] + a->y*mat[1][2] + a->z*mat[2][2];

  *b=tmp;
}

void v_rot(vec_t *b, vec_t *axis, double theta, vec_t *a)
{
  vec_t  nax;
  double mtx[3][3];

  nax = *axis;
  v_norm(&nax);
  
  mtx[0][0] = nax.x*nax.x + (1-nax.x*nax.x)*cos(theta);
  mtx[0][1] = (nax.x)*(nax.y)*(1-cos(theta))-(nax.z)*sin(theta);
  mtx[0][2] = (nax.x)*(nax.z)*(1-cos(theta))+(nax.y)*sin(theta);
  
  mtx[1][0] = (nax.x)*(nax.y)*(1-cos(theta))+(nax.z)*sin(theta);
  mtx[1][1] = nax.y*nax.y + (1-nax.y*nax.y)*cos(theta);
  mtx[1][2] = (nax.y)*(nax.z)*(1-cos(theta))-(nax.x)*sin(theta);
  
  mtx[2][0] = (nax.x)*(nax.z)*(1-cos(theta))-(nax.y)*sin(theta);
  mtx[2][1] = (nax.y)*(nax.z)*(1-cos(theta))+(nax.x)*sin(theta);
  mtx[2][2] = nax.z*nax.z + (1-nax.z*nax.z)*cos(theta);

  v_mat_mul(b, mtx, a);
}

