/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#define SOLVATE_C

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>

#include "util.h"
#include "charmm_par.h"
#include "pdb.h"
#include "charmm_top.h"
#include "mdat.h"
#include "patch.h"
#include "config.h"
#include "solvate.h"
#include "ion.h"

void solvate(mdat_t *mdat)
{
  printf("Setup of Solvation.\n");

  if (_cfg.solvent_pdb[0] == '\0' && _cfg.ion_cation[0] != '\0') {
    fprintf(stderr, "ERROR: When ions are added, solvent file must be inputed.\n");
    exit(1);
  }

  pdb_init(&_solvent_input_pdb);
  pdb_init(&_solvent_pdb);
  mdat_init(&_solvent_mdat);
  
  /* set crystal water and ion 
  sol_set_crystal_solvent(mdat);
  */
  
  /* get min and max of coordinate */
  sol_setup_box_size(mdat);

  /* read solvent pdb file. */
  printf("Solvent PDB file:\n");
  pdb_read_file(&_solvent_input_pdb, _cfg.solvent_pdb);

  /* fill solvent in box */
  sol_fill_solvent(mdat, &_solvent_pdb, &_solvent_input_pdb);

  /* exclude overlapped solvent */
  sol_exclude_solvent(mdat, &_solvent_pdb);

  /* ion_setup */
  if (_cfg.ion_cation[0])
    ion_setup(mdat);
  
  pdb_remove_residues(&_solvent_pdb);
  pdb_renumber(&_solvent_pdb, 1, _cfg.solvent_chain);
  pdb_count_data(&_solvent_pdb);

  pdb_print_info(&_solvent_pdb, "configured_solvent"); 

  /* convert to mdat */
  mdat_make_data_for_charmm(&_solvent_mdat, &_solvent_pdb);
  mdat_add_mass(&_solvent_mdat, &_top);
  mdat_make_coordinates(&_solvent_mdat);

  sol_set_added_solvent(&_solvent_mdat);

  /* merge solvent */
  mdat_merge(mdat, &_solvent_mdat);

  if (_cfg.solvent_renumbering == 1)
    sol_solvent_renumbering(mdat);
  else if (_cfg.solvent_renumbering == 2)
    sol_solvent_renumbering_chain(mdat);
}

void sol_set_crystal_solvent(mdat_t *mdat)
{
  mdat_res_t *r;

  mdat->n_c_water = mdat->n_c_cation = mdat->n_c_anion = 0;
  
  for (r=mdat->res;r!=NULL;r=r->next) {
    if (strcmp(r->name, "TIP3") == 0 ||
	strcmp(r->name, "HOH") == 0  ||
	strcmp(r->name, "WAT") == 0) {
      r->flag = MDAT_CRYSTAL_WATER;
      mdat->n_c_water++;
    } else if (strcmp(r->name, _cfg.ion_cation) == 0)  {
      r->flag = MDAT_CRYSTAL_CATION;
      mdat->n_c_cation++;
    } else if (strcmp(r->name, _cfg.ion_anion) == 0)  {
      r->flag = MDAT_CRYSTAL_ANION;
      mdat->n_c_anion++;
    } else {
      r->flag = MDAT_SOLUTE;
    }
  }
  printf("%d waters are found in input pdb file.\n", mdat->n_c_water);
  printf("%d cations are found in input pdb file.\n", mdat->n_c_cation);
  printf("%d anions are found in input pdb file.\n", mdat->n_c_anion);
}

void sol_set_added_solvent(mdat_t *mdat)
{
  mdat_res_t *r;

  mdat->n_c_water = mdat->n_c_cation = mdat->n_c_anion = 0;
  
  for (r=mdat->res;r!=NULL;r=r->next) {
    if (strcmp(r->name, "TIP3") == 0 ||
	strcmp(r->name, "HOH") == 0  ||
	strcmp(r->name, "WAT") == 0) {
      r->flag = MDAT_ADDED_WATER;
    } else if (strcmp(r->name, _cfg.ion_cation) == 0)  {
      r->flag = MDAT_ADDED_CATION;
    } else if (strcmp(r->name, _cfg.ion_anion) == 0)  {
      r->flag = MDAT_ADDED_ANION;
    } else {
      r->flag = MDAT_ADDED_OTHER;
    }
  }
  /*
  printf("%d waters are found in input pdb file.\n", mdat->n_c_water);
  printf("%d cations are found in input pdb file.\n", mdat->n_c_cation);
  printf("%d anions are found in input pdb file.\n", mdat->n_c_anion);
  */
}


void sol_setup_box_size(mdat_t *mdat)
{
  mdat_res_t *r;
  mdat_atom_t *a;
  double x[3], min[3], max[3], box[3], maxbox;
  int i,j;

  if (mdat->ifbox)
    return;

  min[0] = min[1] = min[2] =  1.0e10;
  max[0] = max[1] = max[2] = -1.0e10;
  
  for (r=mdat->res;r!=NULL;r=r->next) {
    if (r->flag != MDAT_SOLUTE) continue; 
    for (a=r->beg_atom;a!=NULL&&a->prev!=r->end_atom;a=a->next) {
      x[0] = a->x; x[1] = a->y; x[2] = a->z;
      for (i=0;i<3;i++) {
	if (min[i] > x[i]) min[i] = x[i];
	if (max[i] < x[i]) max[i] = x[i];
      }
    }
  }
  printf("Minimum and maximum coordinates of solute:\n");
  printf("  (%.2f,%.2f,%.2f)-(%.2f,%.2f,%.2f)\n",
	 min[0], min[1], min[2],
	 max[0], max[1], max[2]);
  printf("solvent_buffer %.2f Angstrom\n",_cfg.solvent_buffer);

  maxbox = 0.0;
  for (i=0;i<3;i++) {
    min[i] -= _cfg.solvent_buffer;
    max[i] += _cfg.solvent_buffer;
    mdat->boxv[i][i] = max[i] - min[i];
    if (maxbox < mdat->boxv[i][i]) maxbox = mdat->boxv[i][i];
  }
  if (_cfg.solvent_cube) {
    printf("solvent_cube option: on\n");
    for (i=0;i<3;i++) {
      min[i] -= (maxbox - mdat->boxv[i][i])*0.5;
      max[i] += (maxbox - mdat->boxv[i][i])*0.5;
      mdat->boxv[i][i] = max[i] - min[i];
    }
  }
  printf("All atoms are shifted: (%.2f,%.2f,%.2f)\n",
	 -min[0], -min[1], -min[2]);
  for (a=mdat->atom;a!=NULL;a=a->next) {
    a->x -= min[0];
    a->y -= min[1];
    a->z -= min[2];
  }
  for (i=0;i<3;i++)
    for (j=0;j<3;j++)
      if (i!=j)
	mdat->boxv[i][j]=0.0;
  mdat->ifbox = 1;
  printf("Simulation box is configured as (%.2f,%.2f,%.2f)\n",
	 mdat->boxv[0][0], mdat->boxv[1][1], mdat->boxv[2][2]);
}

void sol_fill_solvent(mdat_t *mdat, pdb_t *o_pdb, pdb_t *i_pdb)
{
  int dup[3], i, j, k;
  double shift[3];
  pdb_res_t *r;
  pdb_atom_t *a, *p;

  dup[0] = (int) (mdat->boxv[0][0] / i_pdb->a) + 1;
  dup[1] = (int) (mdat->boxv[1][1] / i_pdb->b) + 1;
  dup[2] = (int) (mdat->boxv[2][2] / i_pdb->c) + 1;

  if (mdat->boxv[0][0] == i_pdb->a)
    dup[0] = 1;
  if (mdat->boxv[1][1] == i_pdb->b)
    dup[1] = 1;
  if (mdat->boxv[2][2] == i_pdb->c)
    dup[2] = 1;

  printf("Box Size of Input Solvent PDB: (%.2f,%.2f,%.2f)\n",
	 i_pdb->a,i_pdb->b,i_pdb->c);
  printf("Duplicated: (%d,%d,%d)\n",
	 dup[0],dup[1],dup[2]);

  for (i=0;i<dup[0];i++) {
    shift[0] = i_pdb->a * i;
    for (j=0;j<dup[1];j++) {
      shift[1] = i_pdb->b * j;
      for (k=0;k<dup[2];k++) {
	shift[2] = i_pdb->c * k;
	pdb_duplicate_atoms(o_pdb, i_pdb, shift);
      }
    }
  }
  pdb_make_res_list(o_pdb);
  pdb_count_data(o_pdb);
  /* pdb_print_info(o_pdb, "filled_solvent"); */

  for (r=o_pdb->res;r!=NULL;r=r->next) {
    r->check = 0;
    for (p=a=r->beg_atom;p!=r->end_atom;p=a,a=a->next) {
      if (a->name[0] == 'H') continue;

      if (a->x > mdat->boxv[0][0] - _cfg.box_buffer ||
	  a->y > mdat->boxv[1][1] - _cfg.box_buffer ||
	  a->z > mdat->boxv[2][2] - _cfg.box_buffer ) {
	r->check = 1;
	break;
      }
    }
  }
}


void sol_exclude_solvent(mdat_t *mdat, pdb_t *sol)
{
  pdb_res_t *r;
  pdb_atom_t *p,*a;
  mdat_atom_t *ma;
  double cut, cut2, len2, x, y, z;
  double dx, dy, dz;
  int i,j,k, ii, jj, kk;
  int nex;

  /*
  cut = _cfg.solvent_closeness;
  printf("Excluding threshold distance: %.2f Angstrom\n",cut);
  */
  mdat_set_radii(mdat);
  cut = 0.0;

  if (_cfg.solvent_radius == 0.0) {
    printf("No Excluded Solvent.\n");
    return;
  }
  
  for (ma=mdat->atom;ma!=NULL;ma=ma->next) {
    ma->r += _cfg.solvent_radius + _cfg.solvent_exclusion_layer;
    if (cut < ma->r) cut = ma->r;
    ma->r *= ma->r;
  }
  printf("solvent_radius %.2f Angstrom\n", _cfg.solvent_radius);
  printf("solvent_exclusion_layer %.2f Angstrom\n", _cfg.solvent_exclusion_layer);
  
  sol_initialize_lc(mdat, cut);
  sol_assign_lc(mdat);

  for (r=sol->res;r!=NULL;r=r->next) {
    for (p=a=r->beg_atom;p!=r->end_atom;p=a,a=a->next) {
      if (a->name[0] == 'H') continue;
      i = (int) (a->x / _lc.dx);
      j = (int) (a->y / _lc.dy);
      k = (int) (a->z / _lc.dz);
      for (ii=i-1;ii<=i+1;ii++) {
	if (ii < 0 || ii >= _lc.nx) continue;
	for (jj=j-1;jj<=j+1;jj++) {
	  if (jj < 0 || jj >= _lc.ny) continue;
	  for (kk=k-1;kk<=k+1;kk++) { 
	    if (kk < 0 || kk >= _lc.nz) continue;
	    for (ma=LC(ii,jj,kk).atom;ma!=NULL;ma=ma->gnext) {
	      if (ma->name[0] == 'H') continue;
	      dx = a->x - ma->x;
	      dy = a->y - ma->y;
	      dz = a->z - ma->z;
	      len2 = dx * dx + dy * dy + dz * dz;
	      if (ma->r > len2) {
		r->check = 1;
		goto loopout;
	      }
	    }
	  }
	}
      }
    }
  loopout:;
  }

  nex=0;
  for (r=sol->res;r!=NULL;r=r->next) {
    if (r->check) {
      nex++;
    }
  }
}

void sol_initialize_lc(mdat_t *mdat, double cut)
{
  int i;
  /* initialize cell */
  _lc.nx = (int) (mdat->boxv[0][0] / cut) + 1;
  _lc.ny = (int) (mdat->boxv[1][1] / cut) + 1;
  _lc.nz = (int) (mdat->boxv[2][2] / cut) + 1;

  _lc.dx = mdat->boxv[0][0] / _lc.nx;
  _lc.dy = mdat->boxv[1][1] / _lc.ny;
  _lc.dz = mdat->boxv[2][2] / _lc.nz;

  _lc.cell = emalloc("sol_initialize_ic",sizeof(cell_t)*_lc.nx*_lc.ny*_lc.nz);
  for (i=0;i<_lc.nx*_lc.ny*_lc.nz;i++) {
    _lc.cell[i].atom = NULL;
  }
}

void sol_assign_lc(mdat_t *mdat)
{
  mdat_atom_t *a;
  int i,j,k;
  
  for (a=mdat->atom;a!=NULL;a=a->next) {
    i = (int) (a->x / _lc.dx);
    j = (int) (a->y / _lc.dy);
    k = (int) (a->z / _lc.dz);
    if (i >= _lc.nx || j >= _lc.ny || k >= _lc.nz) {
      fprintf(stderr, "ERROR: The following atom locates outside of the box:\n");
      mdat_output_pdb_atom(a, stderr);
      exit(1);
      
    }
    a->gnext = LC(i,j,k).atom;
    LC(i,j,k).atom = a;
  }
}

void sol_set_pos_crystal_water(mdat_t *mdat)
{
  mdat_res_t *r;
  mdat_atom_t *a,*p,*o,*h[2];
  int nh, n;
  double length, angle;
  length = 0.9572;
  angle = 104.52;

  n=0;
  for (r=mdat->res;r!=NULL;r=r->next) {
    if (strcmp(r->name,"TIP3") == 0) {
      nh=0;
      for (p=a=r->beg_atom;p!=r->end_atom;p=a,a=a->next) {
	if (a->name[0]=='O') o = a;
	if (a->name[0]=='H') {
	  h[nh] = a;
	  nh++;
	}
      }
      if (h[0]->coord == 0 && h[1]->coord == 0) {
	n++;
	h[0]->x = o->x + length;
	h[0]->y = o->y;
	h[0]->z = o->z;
	h[1]->x = o->x + cos(angle*M_PI/180.0) * length;
	h[1]->y = o->y + sin(angle*M_PI/180.0) * length;
	h[1]->z = o->z;
	h[0]->coord = h[1]->coord = 1;
      } else if (h[0]->coord == 0 || h[1]->coord == 0) {
	printf("Why are coodinates of only one of hydrogen atoms determined?\n");
	if (r->pdb_res) {
	  printf("res_no: %d\n", r->pdb_res->no);
	}
	exit(1);
      }
    }
  }
  if (n>0)
    printf("Coordinates of hydrogens of %d crystal waters are generated.\n", n);
}

void sol_solvent_renumbering(mdat_t *mdat)
{
  int i, max;
  mdat_res_t *r, *pr, *nr;

  max = 0;
  for (r=mdat->res,pr=NULL;r!=NULL;pr=r,r=r->next) {
    if (r->flag > MDAT_INPUT_PDB  /*_cfg.ion_order_flag*/)
      break;
    if (max < r->pdb_no) max = r->pdb_no;
  }
  if (pr) {
    nr = pr->next;
  } else {
    nr = mdat->res;
  }
  i = ((max-1)/1000+1)*1000+1;
  /*i = max + 1;*/
  for (r=nr;r!=NULL;r=r->next) {
    r->pdb_no = i++;
    r->pdb_chain = ' ';
  }
}

void sol_solvent_renumbering_chain(mdat_t *mdat)
{
  int i_ion, i_water, i_other, max;
  mdat_res_t *r, *pr, *nr;

  max = 0;
  for (r=mdat->res,pr=NULL;r!=NULL;pr=r,r=r->next) {
    if (r->flag > MDAT_INPUT_PDB     /*_cfg.ion_order_flag*/)
      break;
    if (max < r->pdb_no) max = r->pdb_no;
  }
  if (pr) {
    nr = pr->next;
  } else {
    nr = mdat->res;
  }

  i_ion = i_water = i_other = 1;

  for (r=nr;r!=NULL;r=r->next) {
    switch (r->flag) {
    case MDAT_CRYSTAL_CATION:
    case MDAT_CRYSTAL_ANION:
    case MDAT_ADDED_CATION:
    case MDAT_ADDED_ANION:
      r->pdb_no = i_ion++;
      r->pdb_chain = _cfg.ion_chain;
      break;
    case MDAT_CRYSTAL_WATER:
    case MDAT_ADDED_WATER:
      r->pdb_no = i_water++;
      r->pdb_chain = _cfg.solvent_chain;
      break;
    default:
      r->pdb_no = i_other++;
      r->pdb_chain = _cfg.other_chain;
      break;
    }
  }
}
