/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>

#include "util.h"
#include "pdb.h"
#include "charmm_par.h"
#include "charmm_top.h"
#include "config.h"

void pdb_init(pdb_t *pdb)
{
  pdb->atom = NULL;
  pdb->res = NULL;
  pdb->n_atom = pdb->n_hetatm = pdb->n_res = 0;
  pdb->a = pdb->b = pdb->c = pdb->alpha = pdb->beta = pdb->gamma = 0.0;
}

void pdb_read_file(pdb_t *pdb, char *fname)
{
  pdb_atom_t *p, *atom_prev;
  char buf[100], dst[10], alt_loc;
  FILE *fp;
  int len,hid;

  fp = env_fopen(fname, "r");
  if (fp == NULL) {
    fprintf(stderr,"ERROR: %s: No such pdb file\n",fname);
    exit(1);
  }

  pdb->atom = p = NULL;
  atom_prev = NULL;
  
  while (fgets(buf,100,fp) != NULL) {
    hid = pdb_parse_header(buf);
    
    switch (hid) {
    case PDB_ATOM:
    case PDB_HETATM:
      if (strlen(buf)<31) continue; 
      p = emalloc("read_pdb_file", sizeof(pdb_atom_t));
      p->header = hid;
      p->ter = 0;
      p->check = 0;
	  
      if (sscanf(cutstr(buf,dst, 7,11), "%d", &p->no) != 1)  continue;
      if (sscanf(cutstr(buf,dst,13,16), "%s", p->name)  != 1)  continue;
      if (sscanf(cutstr(buf,dst,17,17), "%c", &p->alt_loc)  != 1)  continue;
      if (sscanf(cutstr(buf,dst,18,21), "%s", p->res_name)  != 1)  continue;
      if (sscanf(cutstr(buf,dst,22,22), "%c", &p->chain)  != 1)  continue;
      if (sscanf(cutstr(buf,dst,23,27), "%d", &p->res_no)  != 1)  continue;
      /*
      if (sscanf(cutstr(buf,dst,27,27), "%c", &p->code_ins)  != 1)  continue;
      */
      if (sscanf(cutstr(buf,dst,31,38), "%lf", &p->x)  != 1)  continue;
      if (sscanf(cutstr(buf,dst,39,46), "%lf", &p->y)  != 1)  continue;
      if (sscanf(cutstr(buf,dst,47,54), "%lf", &p->z)  != 1)  continue;
      if (sscanf(cutstr(buf,dst,55,60), "%lf", &p->o)  != 1)  p->o = 1.0;
      if (sscanf(cutstr(buf,dst,61,66), "%lf", &p->t)  != 1)  p->t = 0.0;

      pdb_hydrogen_number(p->name);
      
      p->next = NULL;
      if (pdb->atom==NULL)
	pdb->atom=p;
      else
	atom_prev->next = p;

      if (atom_prev) {
	if (p->chain != atom_prev->chain) {
	  atom_prev->ter = 1;
	}
	if (p->header != atom_prev->header) {
	  atom_prev->ter = 1;
	}
      }
      atom_prev = p;
      break;
    case PDB_TER:
      if (p==NULL) break;
      p->ter = 1;
      break;
    case PDB_CRYST1:
      break;
    case PDB_MBLBOX:
      sscanf(buf,"%s %lf %lf %lf %lf %lf %lf",dst,
	     &pdb->a,&pdb->b,&pdb->c,
	     &pdb->alpha,&pdb->beta,&pdb->gamma);
      break;
    case PDB_CONNCT:
      printf("Warning: Ignoring CONNCT line in pdb file.\n");
      break;
    default :
      break;
    }
  }
  pdb_make_res_list(pdb);
  pdb_count_data(pdb);
  pdb_print_info(pdb, fname);
}

void pdb_hydrogen_number(char *buf)
{
  char new_name[10];
  int i;
  
  if (isdigit(buf[0])) {
    for (i=1;i<10;i++) {
      if (buf[i]=='\0') break;
      new_name[i-1]=buf[i];
    }
    new_name[i-1]=buf[0];
    new_name[i]='\0';
    strcpy(buf, new_name);
  }
}

int pdb_parse_header(char *buf)
{
  if (strncmp(buf,"ATOM",4) == 0) return PDB_ATOM;
  else if (strncmp(buf,"HETATM",6) == 0) return PDB_HETATM;
  else if (strncmp(buf,"TER",3) == 0) return PDB_TER;
  else if (strncmp(buf,"CRYST1",6) == 0) return PDB_CRYST1;
  else if (strncmp(buf,"MBLBOX",6) == 0) return PDB_MBLBOX;
  else if (strncmp(buf,"CONNCT",6) == 0) return PDB_CONNCT;
  else return PDB_UNKNOWN;
}

void pdb_make_res_list(pdb_t *pdb)
{
  pdb_atom_t *a;
  pdb_res_t  *r, *r_prev;
  int prev_res_no;

  if (pdb->atom == NULL) {
    pdb->res = NULL;
    return;
  }
  pdb->res = NULL;
  prev_res_no = pdb->atom->res_no-1;
  for (a = pdb->atom; a != NULL; a = a->next) {
    if (prev_res_no != a->res_no ||
	r->ter == 1 ||
	strcmp(r->name, a->res_name) != 0) {
      r = emalloc("make_pdb_res_list", sizeof(pdb_res_t));
      r -> next = NULL;
      if (pdb->res == NULL) {
	pdb->res = r;
      } else {
	r_prev->next = r;
      }
      
      r -> beg_atom = a;
      prev_res_no = r -> no = a->res_no;
      strcpy(r->name, a->res_name);
      strcpy(r->pdb_name, a->res_name);
      r_prev = r;
    }
    r->ter = a->ter;
    r->chain = a->chain;
    r->end_atom = a;
  }
}

void pdb_read_seq(pdb_t *pdb, char *fname)
{
  pdb_res_t  *r, *r_prev;
  FILE *fp;
  char buf[1000], chain, c;
  int no=1, d;

  fp = env_fopen(fname,"r");
  if (fp==NULL) {
    fprintf(stderr,"ERROR: %s: No such sequence file\n",fname);
    exit(1);
  }
  fgets(buf,1000,fp);
  printf("Title of sequence file: %s", buf);

  pdb->res = NULL;
  pdb->atom = NULL;
  r_prev=NULL;
  chain = ' ';
  while (fscanf(fp,"%s",buf)==1) {
    if (strcmp(buf,"TER") == 0) {
      if (r_prev)
	r_prev->ter = 1;
      no=1;
      continue;
    }
    if (sscanf(buf,"chain=%c", &c) == 1) {
      chain = c;
      if (r_prev)
	r_prev->ter = 1;
      no=1;
      continue;
    }
    if (sscanf(buf,"no=%d", &d) == 1) {
      no=d;
      continue;
    }
    r = emalloc("pdb_read_seq", sizeof(pdb_res_t));
    strcpy(r->name, buf);
    strcpy(r->pdb_name, buf);
    r->beg_atom = r->end_atom = NULL;
    r->chain = chain;
    r->ter = 0;
    r->no = no++;
    if (pdb->res == NULL) {
      pdb->res = r;
    } else {
      r_prev->next = r;
    }
    r_prev = r;
  }
}

void pdb_read_seq_buf(pdb_t *pdb, char *buf0)
{
  pdb_res_t  *r, *r_prev;
  char buf[1000], *bufp, chain, c;
  int no=1, n, d;

  pdb->res = NULL;
  pdb->atom = NULL;
  r_prev=NULL;
  bufp = buf0;
  chain = ' ';
  while (sscanf(bufp,"%s%n",buf,&n)==1) {
    bufp += n;
    if (strcmp(buf,"TER") == 0) {
      if (r_prev)
	r_prev->ter = 1;
      no=1;
      continue;
    }
    if (sscanf(buf,"chain=%c", &c) == 1) {
      chain = c;
      if (r_prev)
	r_prev->ter = 1;
      no=1;
      continue;
    }
    if (sscanf(buf,"no=%d", &d) == 1) {
      no=d;
      continue;
    }
    r = emalloc("pdb_read_seq_buf", sizeof(pdb_res_t));
    strcpy(r->name, buf);
    strcpy(r->pdb_name, buf);
    r->beg_atom = r->end_atom = NULL;
    r->chain = chain;
    r->ter = 0;
    r->no = no++;
    if (pdb->res == NULL) {
      pdb->res = r;
    } else {
      r_prev->next = r;
    }
    r_prev = r;
  }
}


char *cutstr(char *buf, char *dst, int from, int to)
{
  int len;

  len = strlen(buf);

  if (from > len || to > len) {
    dst[0] = '\0';
  } else {
    strncpy(dst, &buf[from-1], to - from + 1);
    dst[to-from+1] = '\0';
  }
  /* printf("%d %d %s\n",from, to, dst); */
  return dst;
}

void pdb_count_data(pdb_t *pdb)
{
  pdb_atom_t *a;
  pdb_res_t  *r;
  pdb->n_atom = 0;
  for (a=pdb->atom;a!=NULL;a=a->next) {
    if (a->header == PDB_ATOM)
      pdb->n_atom++;
    else
      pdb->n_hetatm++;
  }
  
  pdb->n_res = 0;
  for (r=pdb->res;r!=NULL;r=r->next) {
    pdb->n_res++;
  }
}

void pdb_print_info(pdb_t *pdb, char *fname)
{
  printf("PDB FILE:  %s\n", fname);
  printf("  Number of atoms : %d\n", pdb->n_atom);
  printf("  Number of hetero atoms :  %d\n", pdb->n_hetatm);
  printf("  Number of residues : %d\n", pdb->n_res);
  printf("\n");
}

pdb_atom_t *pdb_search_atom_in_res(char *name, pdb_res_t *res)
{
  pdb_atom_t *a,*prev;

  for (a=res->beg_atom,prev=NULL;prev!=res->end_atom;prev=a,a=a->next) {
    if (aliascmp(name,a->name) == 0) return a;
  }
  return NULL;
}

void pdb_print_unchecked(pdb_t *pdb)
{
  pdb_atom_t *atom;
  int ok;

  ok = 1;
  for (atom=pdb->atom;atom!=NULL;atom=atom->next) {
    if (!atom->check) {
      printf("Warning: Atom %-4s (%4d%c) in residue %-4s(%4d%c) in pdb file is unused.\n",
	     atom->name,atom->no, atom->alt_loc,atom->res_name,atom->res_no,atom->chain);
      ok = 0;
    }
  }
  if (!ok && !_no_abort) {
    printf("Exit because of unused atoms in pdb file.\n");
    exit(1);
  }
}

char *pdb_atom_name(char *name, char *buf)
{
  if (strlen(name)<=3)
    sprintf(buf," %s", name);
  else
    sprintf(buf,"%s", name);
  return buf;
}

void pdb_renumber(pdb_t *pdb, int start, char chain)
{
  pdb_res_t *r;
  pdb_atom_t *a,*prev;
  int no;

  no = start;
  for (r=pdb->res;r!=NULL;r=r->next) {
    r->no = no++;
    r->chain = chain;
    for (a=r->beg_atom,prev=NULL;prev!=r->end_atom;prev=a,a=a->next) {
      a->res_no = r->no;
      a->chain = chain;
    }
  }
}

/*
void pdb_make_array(pdb_t *pdb)
{
  pdb_atom_t *a;
  pdb_res_t  *r;
  int i;
  
  pdb->al = emalloc("pdb_make_array", sizeof(pdb_atom_t*)*pdb->n_atom);
  pdb->rl = emalloc("pdb_make_array", sizeof(pdb_res_t*)*pdb->n_res);
  
  for (a=pdb->atom,i=0;a!=NULL;a=a->next,i++) {
    pdb->al[i] = a;
  }
    
  for (r=pdb->res,i=0;r!=NULL;r=r->next,i++) {
    pdb->rl[i] = r;
  }
}
*/

void pdb_rename_res(pdb_t *pdb, pdb_rename_res_t *res_list)
{
  pdb_rename_res_t *rename_res;
  pdb_res_t *r;
  pdb_atom_t *a, *prev;
  
  for (rename_res = res_list; rename_res!=NULL; rename_res=rename_res->next) {
    for (r=pdb->res;r!=NULL;r=r->next) {
      if (rename_res->res_no == r->no &&
	  rename_res->chain  == r->chain) {
	printf("Renaming residue %4d%c %s -> %s\n", r->no, r->chain, r->name,
	       rename_res->res_name);
	
	strcpy(r->name, rename_res->res_name);
	for (a=r->beg_atom,prev=NULL;prev!=r->end_atom;prev=a,a=a->next) {
	  strcpy(a->res_name,r->name);
	}
      }
    }
  }
}

void pdb_duplicate_atoms(pdb_t *o_pdb, pdb_t *i_pdb, double shift[3])
{
  pdb_atom_t *oa,*prev,*ia;

  if (o_pdb->atom!=NULL) {
    for (oa=o_pdb->atom;oa->next!=NULL;oa=oa->next);
    prev = oa;
  }

  for (ia=i_pdb->atom;ia!=NULL;ia=ia->next) {
    oa = emalloc("pdb_duplicate_atoms", sizeof(pdb_atom_t));
    *oa = *ia;
    oa->x += shift[0];
    oa->y += shift[1];
    oa->z += shift[2];
    oa->next = NULL;
    if (o_pdb->atom==NULL)
      o_pdb->atom = oa;
    else
      prev->next = oa;
    prev=oa;
  }
}

void pdb_remove_residues(pdb_t *pdb)
{
  pdb_res_t *r, **prevr;
  pdb_atom_t *a, *p, **preva;
  
  for (a=pdb->atom;a!=NULL;a=a->next) {
    a->check = 0;
  }
  
  prevr = &(pdb->res);
  for (r=pdb->res;r!=NULL;r=*prevr) {
    if (r->check) {
      for (p=a=r->beg_atom;p!=r->end_atom;p=a,a=a->next) {
	a->check = 1;
      }
      *prevr = r->next;
      free(r);
    } else {
      prevr = &(r->next);
    }
  }

  preva = &(pdb->atom);
  for (a=pdb->atom;a!=NULL;a=*preva) {
    if (a->check) {
      *preva = a->next;
      free(a);
    } else {
      preva = &(a->next);
    }
  }
}
