/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>

#include "util.h"
#include "pdb.h"
#include "charmm_par.h"
#include "charmm_top.h"
#include "mdat.h"
#include "patch.h"
#include "config.h"

/* for debug */

void patch(mdat_t *mdat)
{
  patch_cfg(mdat, 0);
  patch_terminal_residue(mdat); 
}

void patch_no_auto(mdat_t *mdat)
{
  mdat_atom_t *atom;
  cfg_patch_t *cp;
  int no_patch = 1;

  for (cp = _cfg.patch; cp!=NULL; cp=cp->next) {
    if (cp->no_auto != 1) continue;
    no_patch = 0;
  }
  
  if (no_patch) return;

  printf("\nPatching after auto generating angles and dihedrals.\n");
  patch_cfg(mdat, 1);

  /* clear connections */
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    atom->n_b_atom=0;
    if (atom->b_atom)
      free(atom->b_atom);
  }
  mdat_make_connect(mdat);
  patch_make_nb14(mdat);
}

void patch_terminal_residue(mdat_t *mdat)
{
  mdat_res_t *mres, *mres2[3];
  top_res_t  *pres;

  mres2[1]=mres2[2]=NULL;
  for(mres=mdat->res; mres!=NULL; mres=mres->next) {
    if (mres->first && mres->top_res->patch[0][0]!='\0' && !(mres->patched & 2)) {
      pres = top_search_pres(mres->top_res->patch[0]);
      if (pres!=NULL) {
	mres2[0]=mres;
	patch_residue(mdat, mres2, pres, 1, 0);
      }
    }
    if (mres->last && mres->top_res->patch[1][0]!='\0' && !(mres->patched & 2)) {
      pres = top_search_pres(mres->top_res->patch[1]);
      if (pres!=NULL) {
	mres2[0]=mres;
	patch_residue(mdat, mres2, pres, 1, 0);
      }
    }
  }
}


void patch_cfg(mdat_t *mdat, int no_auto)
{
  cfg_patch_t *cp;
  top_res_t *pres;
  mdat_res_t *res[3];
  int i, ip;
  
  for (cp = _cfg.patch; cp!=NULL; cp=cp->next) {
    /*if (cp->no_auto != no_auto) continue; */
    pres = top_search_pres(cp->patch_name);
    if (pres==NULL) {
      printf("ERROR: No patch %s in top file\n",cp->patch_name);
      exit(1);
    }
    for (ip=0;ip<cp->n_patch;ip++) {
      res[0]=res[1]=res[2]=NULL;
      for (i=0;i<cp->n_pres;i++) {
	res[i] = mdat_search_res_from_pdbno(mdat, cp->res[i]+ip*cp->step,
					    cp->chain[i]);
	if (res[i]==NULL) {
	  printf("ERROR: No residue %d%c in pdb file for patching\n",
		 cp->res[i]+ip*cp->step, cp->chain[i]);
	  exit(1);
	}
      }
      if (no_auto == 0 && cp->no_auto == 0) {
	/* normal patch */
	patch_residue(mdat, res, pres, cp->ter, 0);
      } else if (no_auto == 0 && cp->no_auto == 1) {
	/* no_auto mode, but this is before auto generation of angles */
	/* only add and delete atoms */
	patch_residue(mdat, res, pres, cp->ter, 2);
      } else if (no_auto == 1 && cp->no_auto == 1) {
	/* no_auto mode, but this is after auto generation of angles */
	/* add bond and angles */
	patch_residue(mdat, res, pres, cp->ter, 1);
      }
    }
  }
}

void patch_residue(mdat_t *mdat, mdat_res_t *mres[3], top_res_t *pres, int ter, int no_auto)
{
  printf("Patching %-4s to %s(%d%c)",
	 pres->name,
	 mres[0]->name, mres[0]->pdb_res->no, mres[0]->pdb_res->chain);
  if (ter)
    mres[0]->patched |= 2;
  else
    mres[0]->patched |= 1;
  if (mres[1]) {
    printf(" and %s(%d%c)",
	   mres[1]->name, mres[1]->pdb_res->no, mres[1]->pdb_res->chain);
    if (ter)
      mres[1]->patched |= 2;
    else
      mres[1]->patched |= 1;
  }
  if (mres[2]) {
    printf(" and %s(%d%c)",
	   mres[2]->name, mres[2]->pdb_res->no, mres[2]->pdb_res->chain);
  }
  if (no_auto == 2) {
    printf(" no-auto mode");
  }

  printf("\n");
  if (no_auto == 0) {
    patch_delete_angle_ex(mdat, mres, pres);
    patch_delete_impr_ex(mdat, mres, pres);
    patch_delete_atom(mdat, mres, pres);
    patch_generate_atom(mdat, mres, pres);

    patch_generate_bond(mdat, mres, pres);
    patch_generate_impr(mdat, mres, pres);
    patch_generate_cmap(mdat, mres, pres);
    patch_generate_ic(mdat, mres, pres);
  } else if (no_auto == 2) {
    patch_delete_angle_ex(mdat, mres, pres);
    patch_delete_impr_ex(mdat, mres, pres);
    patch_delete_atom(mdat, mres, pres);
    patch_generate_atom(mdat, mres, pres);
  } else if (no_auto == 1) {
    patch_generate_bond(mdat, mres, pres);
    patch_generate_angle(mdat, mres, pres);
    patch_generate_dihedral(mdat, mres, pres);
    patch_generate_impr(mdat, mres, pres);
    patch_generate_cmap(mdat, mres, pres);
    patch_generate_ic(mdat, mres, pres);
  }
}

void patch_delete_atom(mdat_t *mdat, mdat_res_t *mres[3], top_res_t *pres)
{
  top_d_atom_t *d_atom;
  top_res_t *n_res;
  mdat_atom_t *d_matom;
  int i;
      
  /*delete atom*/
  for(d_atom=pres->d_atom;d_atom!=NULL;d_atom=d_atom->next) {
    d_matom = mdat_search_atom_in_res2(d_atom->name, mres, &i);

    if (d_matom==NULL) {
      printf("ERROR: No atom %s in residue %s(%d%c) for patch %s.\n",
	     d_atom->name, mres[i]->name, mres[i]->pdb_res->no, mres[i]->pdb_res->chain,
	     pres->name);
      exit(1);
    }
    if (d_matom->prev!=NULL)
      d_matom->prev->next = d_matom->next;
    else
      mdat->atom = d_matom->next;
    if (d_matom->next!=NULL)
      d_matom->next->prev = d_matom->prev;
    if (mres[i]->beg_atom==d_matom)
      mres[i]->beg_atom = d_matom->next;
    if (mres[i]->end_atom==d_matom)
      mres[i]->end_atom = d_matom->prev;

    if (_debug>=1)
      printf(" deleting atom %s(%s%d%c)\n", d_matom->name,
	     mres[i]->name, mres[i]->pdb_res->no, mres[i]->pdb_res->chain);

    patch_delete_bond(mdat,d_matom);
    patch_delete_angle(mdat,d_matom);
    patch_delete_dihedral(mdat,d_matom);
    patch_delete_impr(mdat,d_matom);
    patch_delete_cmap(mdat,d_matom);
    patch_delete_ic(mdat,d_matom);
  }
}
  

void patch_delete_bond(mdat_t *mdat, mdat_atom_t *d_atom)
{
  mdat_bond_t *bond;

  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    if (bond->atom[0]==d_atom || bond->atom[1]==d_atom) {
      if (bond->prev!=NULL)
	bond->prev->next = bond->next;
      else
	mdat->bond = bond->next;
      if (bond->next!=NULL)
	bond->next->prev = bond->prev;

      if (bond->res->beg_bond == bond && bond->res->end_bond == bond) {
	bond->res->beg_bond = bond->res->end_bond = NULL;
      } else if (bond->res->beg_bond == bond){
	bond->res->beg_bond = bond->next;
      } else if (bond->res->end_bond == bond){
	bond->res->end_bond = bond->prev;
      }
      
      if (_debug>=1)
	printf(" deleting bond %s - %s\n", bond->atom[0]->name, bond->atom[1]->name);
    }
  }
}

void patch_delete_angle(mdat_t *mdat, mdat_atom_t *d_atom)
{
  mdat_angle_t *angle;

  for (angle=mdat->angle;angle!=NULL;angle=angle->next) {
    if (angle->atom[0]==d_atom || angle->atom[1]==d_atom || angle->atom[2]==d_atom) {
      if (angle->prev!=NULL)
	angle->prev->next = angle->next;
      else
	mdat->angle = angle->next;
      if (angle->next!=NULL)
	angle->next->prev = angle->prev;

      if (_debug>=1)
	printf(" deleting angle %s - %s - %s\n", 
	       angle->atom[0]->name, angle->atom[1]->name, angle->atom[2]->name);
    }
  }
}

void patch_delete_angle_ex(mdat_t *mdat, mdat_res_t *mres[3], top_res_t *pres)
{
  top_angle_t  *angle;
  mdat_atom_t *atom[3];
  mdat_angle_t *mangle, *last;
  int i,exist,ires[3];

  if (pres->d_angle == NULL) return;
  
  /*delete angle*/
  for(angle=pres->d_angle;angle!=NULL;angle=angle->next) {

    for (i=0;i<3;i++) {
      atom[i] = mdat_search_atom_in_res2(angle->atom[i], mres, &ires[i]);
      if (atom[i]==NULL) {
	printf("Error: Can't find atom %s in patching %s\n", angle->atom[i], pres->name);
	exit(1);
      }
    }

    for (mangle = mdat->angle; mangle!=NULL;mangle=mangle->next) {
      if (mangle->atom[1] == atom[1] && 
	  (mangle->atom[0] == atom[0] &&
	   mangle->atom[2] == atom[2]) ||
	  (mangle->atom[0] == atom[2] &&
	   mangle->atom[2] == atom[0])) {

	if (mangle->prev!=NULL)
	  mangle->prev->next = mangle->next;
	else
	  mdat->angle = mangle->next;
	if (mangle->next!=NULL)
	  mangle->next->prev = mangle->prev;

	if (_debug>=1)
	  printf(" deleting angle %s - %s - %s\n", 
		 mangle->atom[0]->name, mangle->atom[1]->name, mangle->atom[2]->name);
      }
    }
  }
}



void patch_delete_dihedral(mdat_t *mdat, mdat_atom_t *d_atom)
{
  mdat_dihedral_t *dihedral;

  for (dihedral=mdat->dihedral;dihedral!=NULL;dihedral=dihedral->next) {
    if (dihedral->atom[0]==d_atom || dihedral->atom[1]==d_atom || 
	dihedral->atom[2]==d_atom || dihedral->atom[3]==d_atom) {
      if (dihedral->prev!=NULL)
	dihedral->prev->next = dihedral->next;
      else
	mdat->dihedral = dihedral->next;
      if (dihedral->next!=NULL)
	dihedral->next->prev = dihedral->prev;

      if (_debug>=1)
	printf(" deleting dihedral %s - %s - %s - %s\n", 
	       dihedral->atom[0]->name, dihedral->atom[1]->name, dihedral->atom[2]->name,
	       dihedral->atom[2]->name);
    }
  }
}

void patch_delete_impr(mdat_t *mdat, mdat_atom_t *d_atom)
{
  mdat_impr_t *impr;
  
  for (impr=mdat->impr;impr!=NULL;impr=impr->next) {
    if (impr->atom[0]==d_atom || impr->atom[1]==d_atom ||
	impr->atom[2]==d_atom || impr->atom[3]==d_atom) {
      if (impr->prev!=NULL)
	impr->prev->next = impr->next;
      else
	mdat->impr = impr->next;
      if (impr->next!=NULL)
	impr->next->prev = impr->prev;

      if (impr->res->beg_impr == impr && impr->res->end_impr == impr) {
	impr->res->beg_impr = impr->res->end_impr = NULL;
      } else if (impr->res->beg_impr == impr){
	impr->res->beg_impr = impr->next;
      } else if (impr->res->end_impr == impr){
	impr->res->end_impr = impr->prev;
      }
      
      if (_debug>=1)
	printf(" deleting impr %s - %s - %s - %s\n", impr->atom[0]->name, impr->atom[1]->name,
	       impr->atom[2]->name, impr->atom[3]->name);
    }
  }
}

void patch_delete_impr_ex(mdat_t *mdat, mdat_res_t *mres[3], top_res_t *pres)
{
  top_impr_t  *impr;
  mdat_atom_t *atom[4];
  mdat_impr_t *mimpr, *last;
  int i,exist,ires[4];
  int ok;

  if (pres->d_impr == NULL) return;
  
  /*delete impr*/
  for(impr=pres->d_impr;impr!=NULL;impr=impr->next) {

    for (i=0;i<4;i++) {
      atom[i] = mdat_search_atom_in_res2(impr->atom[i], mres, &ires[i]);
      if (atom[i]==NULL) {
	printf("Error: Can't find atom %s in patching %s\n", impr->atom[i], pres->name);
	exit(1);
      }
    }

    ok = 0;
    for (mimpr = mdat->impr; mimpr!=NULL;mimpr=mimpr->next) {
      if (mimpr->atom[0] == atom[0] && 
	  mimpr->atom[1] == atom[1] &&
	  mimpr->atom[2] == atom[2] &&
	  mimpr->atom[3] == atom[3]) {

	ok = 1;
	if (mimpr->prev!=NULL)
	  mimpr->prev->next = mimpr->next;
	else
	  mdat->impr = mimpr->next;
	if (mimpr->next!=NULL)
	  mimpr->next->prev = mimpr->prev;

	if (_debug>=1)
	  printf(" deleting impr %s - %s - %s - %s\n", 
		 mimpr->atom[0]->name, mimpr->atom[1]->name, mimpr->atom[2]->name, mimpr->atom[3]->name);
      }
    }
    if (!ok) {
      printf("WARNING: DELETE IMPR: unable to find impr %s - %s - %s - %s\n", 
	     atom[0]->name, atom[1]->name, atom[2]->name, atom[3]->name);
    }
  }
}


void patch_delete_cmap(mdat_t *mdat, mdat_atom_t *d_atom)
{
  mdat_cmap_t *cmap;
  int i, ok;
  
  for (cmap=mdat->cmap;cmap!=NULL;cmap=cmap->next) {
    ok = 1;
    for (i=0;i<8;i++) {
      if (cmap->atom[i]==d_atom) {
	ok = 0;
	break;
      }
    }
    if (!ok) {
      if (cmap->prev!=NULL)
	cmap->prev->next = cmap->next;
      else
	mdat->cmap = cmap->next;
      if (cmap->next!=NULL)
	cmap->next->prev = cmap->prev;
      
      if (cmap->res->beg_cmap == cmap && cmap->res->end_cmap == cmap) {
	cmap->res->beg_cmap = cmap->res->end_cmap = NULL;
      } else if (cmap->res->beg_cmap == cmap){
	cmap->res->beg_cmap = cmap->next;
      } else if (cmap->res->end_cmap == cmap){
	cmap->res->end_cmap = cmap->prev;
      }
      
      if (_debug>=1) {
	for (i=0;i<8;i++) {
	  if (i==0)
	    printf(" deleting cmap %s", cmap->atom[i]->name);
	  else if (i==4)
	    printf(" , %s", cmap->atom[i]->name);
	  else 
	    printf(" -%s", cmap->atom[i]->name);
	}
	printf("\n");
      }
    }
  }
}


void patch_delete_ic(mdat_t *mdat, mdat_atom_t *d_atom)
{
  mdat_ic_t *ic;
  
  for (ic=mdat->ic;ic!=NULL;ic=ic->next) {
    if (ic->atom[0]==d_atom || ic->atom[1]==d_atom ||
	ic->atom[2]==d_atom || ic->atom[3]==d_atom) {
      if (ic->prev!=NULL)
	ic->prev->next = ic->next;
      else
	mdat->ic = ic->next;
      if (ic->next!=NULL)
	ic->next->prev = ic->prev;

      if (ic->res->beg_ic == ic && ic->res->end_ic == ic) {
	ic->res->beg_ic = ic->res->end_ic = NULL;
      } else if (ic->res->beg_ic == ic){
	ic->res->beg_ic = ic->next;
      } else if (ic->res->end_ic == ic){
	ic->res->end_ic = ic->prev;
      }
      
      if (_debug>=1)
	printf(" deleting ic %s - %s - %s - %s\n", ic->atom[0]->name, ic->atom[1]->name,
	       ic->atom[2]->name, ic->atom[3]->name);
    }
  }
}

void patch_generate_atom(mdat_t *mdat, mdat_res_t *mres[3], top_res_t *pres)
{
  top_atom_t *patom, *patom2;
  mdat_atom_t *prev_atom[2], *matom, *matom2;
  int i, i2, prev_group;
  
  /*generate atom*/
  /* old version
  for (i=0;i<2 && mres[i]!=NULL;i++) {
    prev_atom[i] = mres[i]->end_atom;
    if (prev_atom[i]==NULL) {
      printf("ERROR: No atom in residue %s(%d%c) for patch %s.\n",
	     mres[i]->name, mres[i]->pdb_res->no, mres[i]->pdb_res->chain,
	     pres->name);
      exit(1);
    }
  }
  */
  prev_atom[0] = prev_atom[1] = NULL;
  prev_group = -1000;
  
  for(patom=pres->atom;patom!=NULL;prev_group=patom->group,patom=patom->next) {

    matom = mdat_search_atom_in_res2(patom->name, mres, &i);
    if (matom!=NULL) {
      matom->top_atom = patom;
      matom->charge = patom->charge;
      if (_debug>=1)
	printf(" modifying ");

    } else {
      
      matom=emalloc("patch_generate_atom",sizeof(mdat_atom_t));
      matom->top_atom = patom;
      matom->charge = patom->charge;
      matom->coord = 0;
      matom->res = mres[i];

      if (prev_atom[i]!=NULL /*&& patom->group == prev_group*/) {
	/* if an atom is already added or modified */
	matom->next = prev_atom[i]->next;
	matom->prev = prev_atom[i];
	if (matom->next!=NULL)
	  matom->next->prev = matom;
	prev_atom[i]->next = matom;
	if (mres[i]->end_atom==prev_atom[i])
	  mres[i]->end_atom = matom;
      } else {
	matom2=NULL;
	for(patom2=patom->next;
	    patom2!=NULL && patom2->group == patom->group;
	    patom2=patom2->next) {
	  matom2 = mdat_search_atom_in_res2(patom2->name, mres, &i2);
	  if (matom2!=NULL && i==i2) break;
	}
	if (matom2!=NULL) {
	  prev_atom[i] = matom2;
	} else {
	  /* rare case */
	  prev_atom[i] = mres[i]->beg_atom;
	}
	matom->prev = prev_atom[i]->prev;
	matom->next = prev_atom[i];
	if (matom->prev!=NULL)
	  matom->prev->next = matom;
	prev_atom[i]->prev = matom;
	if (mres[i]->beg_atom==prev_atom[i])
	  mres[i]->beg_atom = matom;
	if (mdat->atom == prev_atom[i])
	  mdat->atom = matom;
      }
      
      /*
	old version
      matom->next = prev_atom[i]->next;
      matom->prev = prev_atom[i];
      if (matom->next!=NULL)
	matom->next->prev = matom;
      prev_atom[i]->next = matom;
      if (mres[i]->end_atom==prev_atom[i])
	mres[i]->end_atom = matom;
      */
      
      if (_debug>=1)
	printf(" generating ");
    }
    prev_atom[i]=matom;
    
    if (patom->name[0]=='1' || patom->name[0]=='2' || patom->name[0]=='3')
      strcpy(matom->name, &patom->name[1]);
    else
      strcpy(matom->name,  patom->name);

    if (_debug>=1)
      printf("atom %s(%s%d%c)\n", matom->name,
	     mres[i]->name, mres[i]->pdb_res->no, mres[i]->pdb_res->chain);
  }
}
      

void patch_generate_bond(mdat_t *mdat, mdat_res_t *mres[3], top_res_t *pres)
{
  top_bond_t  *bond;
  mdat_atom_t *atom0, *atom1;
  mdat_bond_t *mbond, *last;
  int i,exist,ires[2];
  
  for (last = mdat->bond; last!=NULL&&last->next!=NULL;last=last->next);
  
  /*generate bond*/
  for(bond=pres->bond;bond!=NULL;bond=bond->next) {
    /* check whether same bond exist or not */
    exist=0;
    for (i=0;i<3&&mres[i]!=NULL;i++) {
      for (mbond=mres[i]->beg_bond;mbond!=NULL&&mbond->prev==mres[i]->end_bond;mbond=mbond->next) {
	if ((strcmp(mbond->atom[0]->name, bond->atom[0]) == 0 &&
	     strcmp(mbond->atom[1]->name, bond->atom[1]) == 0) ||
	    (strcmp(mbond->atom[1]->name, bond->atom[0]) == 0 &&
	     strcmp(mbond->atom[0]->name, bond->atom[1]) == 0)) {
	  exist=1;
	  break;
	}
      }
    }
    if (exist) continue;
    /* end of check */
      
    atom0 = mdat_search_atom_in_res2(bond->atom[0], mres, &ires[0]);
    if (atom0==NULL) {
      printf("Warning: Can't find atom %s in patching %s\n", bond->atom[0], pres->name);
      continue;
    }
    atom1 = mdat_search_atom_in_res2(bond->atom[1], mres, &ires[1]);
    if (atom1==NULL) {
      printf("Warning: Can't find atom %s in patching %s\n", bond->atom[1], pres->name);
      continue;
    }
    mbond = emalloc("patch_generate_bond", sizeof(mdat_bond_t));
    mbond->atom[0] = atom0;
    mbond->atom[1] = atom1;
    mbond->res = mres[ires[0]];

    if (mres[ires[0]]->end_bond != NULL) {
      mbond->next = mres[ires[0]]->end_bond->next;
      mbond->prev = mres[ires[0]]->end_bond;
      mres[ires[0]]->end_bond->next = mbond;
      if (last == mbond->prev) {
	last = mbond;
      }
    } else {
      if (mdat->bond == NULL) {
	mdat->bond = mbond;
	mbond->next = NULL;
	mbond->prev = NULL;
	last = mbond;
      } else {
	last->next = mbond;
	mbond->next = NULL;
	mbond->prev = last;
	last = mbond;
      }
      mres[ires[0]]->beg_bond = mbond;
    }
      
    if (mbond->next) {
      mbond->next->prev = mbond;
    }
    mres[ires[0]]->end_bond = mbond;
    
    if (_debug>=1)
      printf(" generating bond %s - %s\n", mbond->atom[0]->name, mbond->atom[1]->name);
    
  }
}

void patch_generate_angle(mdat_t *mdat, mdat_res_t *mres[3], top_res_t *pres)
{
  top_angle_t  *angle;
  mdat_atom_t *atom[3];
  mdat_angle_t *mangle, *last;
  int i,exist,ires[3];

  if (pres->angle == NULL) return;
  
  for (last = mdat->angle; last!=NULL&&last->next!=NULL;last=last->next);
  
  /*generate angle*/
  for(angle=pres->angle;angle!=NULL;angle=angle->next) {
      
    for (i=0;i<3;i++) {
      atom[i] = mdat_search_atom_in_res2(angle->atom[i], mres, &ires[i]);
      if (atom[i]==NULL) {
	printf("Error: Can't find atom %s in patching %s\n", angle->atom[i], pres->name);
	exit(1);
      }
    }

    /* check whether same angle exist or not */
    exist = 0;
    for (mangle = mdat->angle; mangle!=NULL;mangle=mangle->next) {
      if (mangle->atom[1] == atom[1] && 
	  (mangle->atom[0] == atom[0] &&
	   mangle->atom[2] == atom[2]) ||
	  (mangle->atom[0] == atom[2] &&
	   mangle->atom[2] == atom[0])) {
	exist = 1;
	break;
      }
    }
    if (exist) continue;
    /* end of check */

    mangle = emalloc("patch_generate_angle", sizeof(mdat_angle_t));
    for (i=0;i<3;i++) {
      mangle->atom[i] = atom[i];
    }
    /* mangle->res = mres[ires[0]]; */

    if (mdat->angle == NULL) {
      mdat->angle = mangle;
      mangle->next = NULL;
      mangle->prev = NULL;
      last = mangle;
    } else {
      last->next = mangle;
      mangle->next = NULL;
      mangle->prev = last;
      last = mangle;
    }
      
    if (mangle->next) {
      mangle->next->prev = mangle;
    }
    
    if (_debug>=1)
      printf(" generating angle %s - %s - %s\n", 
	     mangle->atom[0]->name, mangle->atom[1]->name, mangle->atom[2]->name);
  }
}

void patch_generate_dihedral(mdat_t *mdat, mdat_res_t *mres[3], top_res_t *pres)
{
  top_dihedral_t  *dihedral;
  mdat_atom_t *atom[4];
  mdat_dihedral_t *mdihedral, *last;
  int i,exist,ires[4];

  if (pres->dihedral == NULL) return;
  
  for (last = mdat->dihedral; last!=NULL&&last->next!=NULL;last=last->next);
  
  /*generate dihedral*/
  for(dihedral=pres->dihedral;dihedral!=NULL;dihedral=dihedral->next) {
      
    for (i=0;i<4;i++) {
      atom[i] = mdat_search_atom_in_res2(dihedral->atom[i], mres, &ires[i]);
      if (atom[i]==NULL) {
	printf("Error: Can't find atom %s in patching %s\n", dihedral->atom[i], pres->name);
	exit(1);
      }
    }

    /* check whether same dihedral exist or not */
    exist = 0;
    for (mdihedral = mdat->dihedral; mdihedral!=NULL;mdihedral=mdihedral->next) {
      if ((mdihedral->atom[0] == atom[0] && 
	   mdihedral->atom[1] == atom[1] &&
	   mdihedral->atom[2] == atom[2] &&
	   mdihedral->atom[3] == atom[3]) ||
	  (mdihedral->atom[0] == atom[3] && 
	   mdihedral->atom[1] == atom[2] &&
	   mdihedral->atom[2] == atom[1] &&
	   mdihedral->atom[3] == atom[0])) {
	exist = 1;
	break;
      }
    }
    if (exist) continue;
    /* end of check */

    mdihedral = emalloc("patch_generate_dihedral", sizeof(mdat_dihedral_t));
    mdihedral->only14 = 0;
    for (i=0;i<4;i++) {
      mdihedral->atom[i] = atom[i];
    }

    if (mdat->dihedral == NULL) {
      mdat->dihedral = mdihedral;
      mdihedral->next = NULL;
      mdihedral->prev = NULL;
      last = mdihedral;
    } else {
      last->next = mdihedral;
      mdihedral->next = NULL;
      mdihedral->prev = last;
      last = mdihedral;
    }
      
    if (mdihedral->next) {
      mdihedral->next->prev = mdihedral;
    }
    
    if (_debug>=1)
      printf(" generating dihedral %s - %s - %s - %s\n", 
	     mdihedral->atom[0]->name, mdihedral->atom[1]->name, 
	     mdihedral->atom[2]->name, mdihedral->atom[3]->name);
  }
}


void patch_generate_impr(mdat_t *mdat, mdat_res_t *mres[3], top_res_t *pres)
{
  top_impr_t *impr;
  mdat_atom_t *atom[4];
  mdat_impr_t *mimpr, *last;
  int exist,i, ires[4];

  for (last = mdat->impr; last!=NULL&&last->next!=NULL;last=last->next);
  
  /*generate impr*/
  for(impr=pres->impr;impr!=NULL;impr=impr->next) {
    /* check whether same impr exist or not */
    exist=0;
    for (i=0;i<3&&mres[i]!=NULL;i++) {
      for (mimpr=mres[i]->beg_impr;mimpr!=NULL&&mimpr->prev==mres[i]->end_impr;mimpr=mimpr->next) {
	if (strcmp(mimpr->atom[0]->name, impr->atom[0]) == 0 &&
	    strcmp(mimpr->atom[1]->name, impr->atom[1]) == 0 &&
	    strcmp(mimpr->atom[2]->name, impr->atom[2]) == 0 &&
	    strcmp(mimpr->atom[3]->name, impr->atom[3]) == 0) {
	  exist=1;
	  break;
	}
      }
    }
    if (exist) continue;
    /* end of check */

    exist=1;
    for (i=0;i<4;i++) {
      atom[i] = mdat_search_atom_in_res2(impr->atom[i], mres, &ires[i]);
      if (atom[i]==NULL) {
	printf("Warning: Can't find atom %s in patching %s\n", impr->atom[i], pres->name);
	exist=0;
      }
    }
    if (exist==0) continue;
    
    mimpr = emalloc("patch_generate_impr", sizeof(mdat_impr_t));
    for (i=0;i<4;i++)
      mimpr->atom[i] = atom[i];
    mimpr->res  = mres[ires[0]];
    if (mres[ires[0]]->end_impr != NULL) {
      mimpr->next = mres[ires[0]]->end_impr->next;
      mimpr->prev = mres[ires[0]]->end_impr;
      mres[ires[0]]->end_impr->next = mimpr;
      if (last == mimpr->prev) {
	last = mimpr;
      }
    } else {
      if (mdat->impr == NULL) {
	mdat->impr = mimpr;
	mimpr->next = NULL;
	mimpr->prev = NULL;
	last = mimpr;
      } else {
	last->next = mimpr;
	mimpr->next = NULL;
	mimpr->prev = last;
	last = mimpr;
      }
      mres[ires[0]]->beg_impr = mimpr;
    }
    if (mimpr->next) {
      mimpr->next->prev = mimpr;
    }
    mres[ires[0]]->end_impr = mimpr;
    
    if (_debug>=1)
      printf(" generating impr %s - %s - %s - %s\n", mimpr->atom[0]->name, mimpr->atom[1]->name,
	   mimpr->atom[2]->name, mimpr->atom[3]->name);
  }
}

void patch_generate_cmap(mdat_t *mdat, mdat_res_t *mres[3], top_res_t *pres)
{
  top_cmap_t *cmap;
  mdat_atom_t *atom[8];
  mdat_cmap_t *mcmap, *last;
  int exist,i, ires[8], j;

  for (last = mdat->cmap; last!=NULL&&last->next!=NULL;last=last->next);
  
  /*generate cmap*/
  for(cmap=pres->cmap;cmap!=NULL;cmap=cmap->next) {
    /* check whether same cmap exist or not */
    exist=0;
    for (i=0;i<3&&mres[i]!=NULL;i++) {
      for (mcmap=mres[i]->beg_cmap;mcmap!=NULL&&mcmap->prev==mres[i]->end_cmap;mcmap=mcmap->next) {
	exist = 1;
	for (j=0;j<8;j++) {
	  if (strcmp(mcmap->atom[j]->name, cmap->atom[j]) != 0) {
	    exist = 0;
	  }
	}
	if (exist)
	  break;
      }
    }
    if (exist) continue;
    /* end of check */

    exist=1;
    for (i=0;i<8;i++) {
      atom[i] = mdat_search_atom_in_res2(cmap->atom[i], mres, &ires[i]);
      if (atom[i]==NULL) {
	printf("Warning: Can't find atom %s in patching %s\n", cmap->atom[i], pres->name);
	exist=0;
      }
    }
    if (exist==0) continue;
    
    mcmap = emalloc("patch_generate_cmap", sizeof(mdat_cmap_t));
    for (i=0;i<8;i++)
      mcmap->atom[i] = atom[i];
    mcmap->res  = mres[ires[0]];
    if (mres[ires[0]]->end_cmap != NULL) {
      mcmap->next = mres[ires[0]]->end_cmap->next;
      mcmap->prev = mres[ires[0]]->end_cmap;
      mres[ires[0]]->end_cmap->next = mcmap;
      if (last == mcmap->prev) {
	last = mcmap;
      }
    } else {
      if (mdat->cmap == NULL) {
	mdat->cmap = mcmap;
	mcmap->next = NULL;
	mcmap->prev = NULL;
	last = mcmap;
      } else {
	last->next = mcmap;
	mcmap->next = NULL;
	mcmap->prev = last;
	last = mcmap;
      }
      mres[ires[0]]->beg_cmap = mcmap;
    }
    if (mcmap->next) {
      mcmap->next->prev = mcmap;
    }
    mres[ires[0]]->end_cmap = mcmap;
    
    if (_debug>=1) {
      for (i=0;i<8;i++) {
	if (i==0)
	  printf(" generating cmap %s", mcmap->atom[i]->name);
	else if (i==4)
	  printf(" , %s", mcmap->atom[i]->name);
	else 
	  printf(" -%s", mcmap->atom[i]->name);
      }
      printf("\n");
    }
  }
}


void patch_generate_ic(mdat_t *mdat, mdat_res_t *mres[3], top_res_t *pres)
{
  top_ic_t *ic;
  mdat_atom_t *atom[4];
  mdat_ic_t *mic, *last;
  int exist,i, ires[4];

  for (last = mdat->ic; last!=NULL&&last->next!=NULL;last=last->next);
  
  /*generate ic*/
  for(ic=pres->ic;ic!=NULL;ic=ic->next) {
    /* check whether same ic exist or not */
    exist=0;
    for (i=0;i<3&&mres[i]!=NULL;i++) {
      for (mic=mres[i]->beg_ic;mic!=NULL&&mic->prev==mres[i]->end_ic;mic=mic->next) {
	if (strcmp(mic->atom[0]->name, ic->atom[0]) == 0 &&
	    strcmp(mic->atom[1]->name, ic->atom[1]) == 0 &&
	    strcmp(mic->atom[2]->name, ic->atom[2]) == 0 &&
	    strcmp(mic->atom[3]->name, ic->atom[3]) == 0) {
	  mic->length[0] = ic->length[0];
	  mic->length[1] = ic->length[1];
	  mic->angle[0]  = ic->angle[0];
	  mic->angle[1]  = ic->angle[1];
	  mic->dihedral  = ic->dihedral;
	  exist=1;
	  break;
	}
      }
    }
    if (exist) continue;
    /* end of check */

    exist=1;
    for (i=0;i<4;i++) {
      atom[i] = mdat_search_atom_in_res2(ic->atom[i], mres, &ires[i]);
      if (atom[i]==NULL) {
	printf("Warning: Can't find atom %s in patching %s\n", ic->atom[i], pres->name);
	exist=0;
      }
    }
    if (exist==0) continue;
    
    mic = emalloc("patch_generate_ic", sizeof(mdat_ic_t));
    for (i=0;i<4;i++)
      mic->atom[i] = atom[i];
    mic->res  = mres[ires[0]];
    if (ic->atom[2][0] == '*')
      mic->impr = 1;
    else
      mic->impr = 0;
    mic->length[0] = ic->length[0];
    mic->length[1] = ic->length[1];
    mic->angle[0]  = ic->angle[0];
    mic->angle[1]  = ic->angle[1];
    mic->dihedral  = ic->dihedral;

    if (mres[ires[0]]->end_ic != NULL) {
      mic->next = mres[ires[0]]->end_ic->next;
      mic->prev = mres[ires[0]]->end_ic;
      mres[ires[0]]->end_ic->next = mic;
      if (last == mic->prev) {
	last = mic;
      }
    } else {
      if (mdat->ic == NULL) {
	mdat->ic = mic;
	mic->next = NULL;
	mic->prev = NULL;
	last = mic;
      } else {
	last->next = mic;
	mic->next = NULL;
	mic->prev = last;
	last = mic;
      }
      mres[ires[0]]->beg_ic = mic;
    }
   if (mic->next) {
     mic->next->prev = mic;
   }
    mres[ires[0]]->end_ic = mic;
    
    if (_debug>=1)
      printf(" generating ic %s - %s - %s - %s\n", mic->atom[0]->name, mic->atom[1]->name,
	   mic->atom[2]->name, mic->atom[3]->name);
  }
}

void patch_make_nb14(mdat_t *mdat)
{
  mdat_dihedral_t *mdihedral, *last;
  mdat_bond_t *bond;
  mdat_atom_t *atom0, *atom1, *atom2, *atom3;
  int i,j;
  int exist;

  for (last = mdat->dihedral; last!=NULL&&last->next!=NULL;last=last->next);

  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    if (bond->res->nodihedral) continue;
    atom1=bond->atom[0];
    atom2=bond->atom[1];
    for (i=0;i<atom1->n_b_atom;i++) {
      atom0=atom1->b_atom[i];
      if (atom0==atom2) continue;
      for (j=0;j<atom2->n_b_atom;j++) {
	atom3=atom2->b_atom[j];
	if (atom1==atom3) continue;

	if (atom0==atom3) {
	  printf("Warning: First and Fourth of atoms is same in dihedral Omitted.\n");
	  continue;
	}
	if (atom1->top_atom->sym[0] == 'H' ||
	    atom2->top_atom->sym[0] == 'H') {
	  printf("Warning: Second or Third  atom is Hydrogen. Omitted.\n");
	  continue;
	}
	
	exist = 0;
	for (mdihedral = mdat->dihedral; mdihedral!=NULL;mdihedral=mdihedral->next) {
	  if ((mdihedral->atom[0] == atom0 && 
	       mdihedral->atom[1] == atom1 &&
	       mdihedral->atom[2] == atom2 &&
	       mdihedral->atom[3] == atom3) ||
	      (mdihedral->atom[0] == atom3 && 
	       mdihedral->atom[1] == atom2 &&
	       mdihedral->atom[2] == atom1 &&
	       mdihedral->atom[3] == atom0)) {
	    exist = 1;
	    break;
	  }
	}
	if (exist) continue;
	/* end of check */

	mdihedral = emalloc("patch_make_nb14", sizeof(mdat_dihedral_t));
	mdihedral->atom[0] = atom0;
	mdihedral->atom[1] = atom1;
	mdihedral->atom[2] = atom2;
	mdihedral->atom[3] = atom3;
	mdihedral->only14 = 1;

	if (mdat->dihedral == NULL) {
	  mdat->dihedral = mdihedral;
	  mdihedral->next = NULL;
	  mdihedral->prev = NULL;
	  last = mdihedral;
	} else {
	  last->next = mdihedral;
	  mdihedral->next = NULL;
	  mdihedral->prev = last;
	  last = mdihedral;
	}
      
	if (mdihedral->next) {
	  mdihedral->next->prev = mdihedral;
	}
	
	if (_debug>=1)
	  printf(" generating nonbond14 %s - %s - %s - %s\n", 
		 mdihedral->atom[0]->name, mdihedral->atom[1]->name, 
		 mdihedral->atom[2]->name, mdihedral->atom[3]->name);
      }
    }
  }
}
