/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>

#include "util.h"
#include "charmm_par.h"
#include "pdb.h"
#include "charmm_top.h"
#include "config.h"
#include "mdat.h"

#define MDAT_MAJOR_VERSION 0
#define MDAT_MINOR_VERSION 6

/* DEBUG */
void mdat_output(mdat_t *mdat, par_t *par)
{
  FILE *fp;
  if ((fp = fopen(_cfg.output_mdat, "w")) == NULL) {
    printf("ERROR: Can't open output mdat file %s\n", _cfg.output_mdat);
    exit(1);
  }

  if (_cfg.mdat_amber) {
    /* output header */
    fprintf(fp,"mdat %d.%d format 0 molx %d.%d.%d\n", 0, 5, MAJOR_VERSION, MINOR_VERSION, SUBMINOR_VERSION);
    /* minor version == 2 means pdb_chain OK, and cmap not. */
    fprintf(fp,"%-80s\n", _cfg.title);

    mdat->scnb=2.0;
    mdat->scee=1.2;
    mdat_output_atom(mdat, par, fp, 1);
    mdat_output_bond(mdat, par, fp);
    mdat_output_angle(mdat, par, fp, 1);
    mdat_output_dihedral_amber(mdat, par, fp);
    mdat_output_rigid_body(mdat,par,fp);
    mdat_output_boundary(mdat, par, fp);
    
  } else {
  
    /* output header */
    fprintf(fp,"mdat %d.%d format 1 molx %d.%d.%d\n", MDAT_MAJOR_VERSION, MDAT_MINOR_VERSION, MAJOR_VERSION, MINOR_VERSION, SUBMINOR_VERSION);
    fprintf(fp,"%-80s\n", _cfg.title);
    
    mdat_output_atom(mdat, par, fp, 0);
    mdat_output_bond(mdat, par, fp);
    mdat_output_angle(mdat, par, fp, 0);
    mdat_output_dihedral(mdat, par, fp);
    mdat_output_cmap(mdat, par, fp);
    mdat_output_rigid_body(mdat,par,fp);
    mdat_output_boundary(mdat, par, fp);
  }
  fclose(fp);
}
  
void mdat_output_atom(mdat_t *mdat, par_t *par, FILE *fp, int amber)
{
  int i,j, count, n_type, n_hbpair;
  mdat_atom_t *atom;
  mdat_res_t  *res;
  mdat_mol_t  *mol;

  /* atom information */
  fprintf(fp,"%d\n",mdat->n_atom);
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    if (amber)
      fprintf(fp,"%4d %-4s %-6s %4d%16.8e%16.8e\n",atom->no,atom->name,
	      atom->top_atom->sym,
	      atom->par_nonbonded->no, atom->charge*18.2223, atom->mass->weight);
    else
      fprintf(fp,"%4d %-4s %-6s %4d%16.8e%16.8e\n",atom->no,atom->name,
	      atom->top_atom->sym,
	      atom->par_nonbonded->no, atom->charge, atom->mass->weight);
  }

  /* residue information */
  fprintf(fp,"%d\n",mdat->n_res);
  for (res=mdat->res;res!=NULL;res=res->next) {
    /*
    if (amber)
      fprintf(fp,"%d %-4s %d %d %d\n",res->no,res->name,
	      res->beg_atom->no, res->n_atom, 0);
	      else*/
      fprintf(fp,"%d %-4s %d %d %d \'%c\'\n",res->pdb_no,res->name,
	      res->beg_atom->no, res->n_atom, 0, res->pdb_chain);
  }
  
  fprintf(fp,"%d %d\n",mdat->n_mol, mdat->n_solute_mol);
  for (mol=mdat->mol;mol!=NULL;mol=mol->next) {
    fprintf(fp,"%d %d%c\n",mol->n_atom,0,mol->chain);
  }
  
  /* atom other information */
  n_type = par->n_used_nonbonded;
  n_hbpair = 0;

  fprintf(fp,"%d %d\n", n_type, n_hbpair);
  fprintf(fp,"%f %f\n",mdat->scnb, mdat->scee);  /* scnb scee */
  count=0;
  for (i=0; i<n_type; i++) {
    for (j=0; j<n_type; j++) {
      fprintf(fp,"%4d ",par->nbpair[i][j].no);
      if (++count % 15 == 0)
	fprintf(fp,"\n");
    }
  }
  fprintf(fp,"\n");

  if (amber) {
    char *format = "%.8e ";
    /*char *format = "%.15e ";*/
    
    count=0;
    for (i=0; i<n_type; i++) {
      for (j=i; j<n_type; j++) {
	fprintf(fp,format, par->nbpair[i][j].eps*pow(par->nbpair[i][j].rmin,12.0));
	if (++count % 5 == 0)
	  fprintf(fp,"\n");
      }
    }
    fprintf(fp,"\n");
    count=0;
    for (i=0; i<n_type; i++) {
      for (j=i; j<n_type; j++) {
	fprintf(fp,format,2*par->nbpair[i][j].eps*pow(par->nbpair[i][j].rmin,6.0));
	if (++count % 5 == 0)
	  fprintf(fp,"\n");
      }
    }
    fprintf(fp,"\n");
    
  } else {
    count=0;
    for (i=0; i<n_type; i++) {
      for (j=i; j<n_type; j++) {
	fprintf(fp,"%.15e ", par->nbpair[i][j].eps);
	if (++count % 5 == 0)
	  fprintf(fp,"\n");
      }
    }
    fprintf(fp,"\n");
    count=0;
    for (i=0; i<n_type; i++) {
      for (j=i; j<n_type; j++) {
	fprintf(fp,"%.15e ",par->nbpair[i][j].rmin);
	if (++count % 5 == 0)
	  fprintf(fp,"\n");
      }
    }
    fprintf(fp,"\n");

    count=0;
    for (i=0; i<n_type; i++) {
      for (j=i; j<n_type; j++) {
	fprintf(fp,"%.15e ", par->nbpair[i][j].eps14);
	if (++count % 5 == 0)
	  fprintf(fp,"\n");
      }
    }
    fprintf(fp,"\n");
    count=0;
    for (i=0; i<n_type; i++) {
      for (j=i; j<n_type; j++) {
	fprintf(fp,"%.15e ",par->nbpair[i][j].rmin14);
	if (++count % 5 == 0)
	  fprintf(fp,"\n");
      }
    }
    fprintf(fp,"\n");
  }
  
  /* exatom information */
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    fprintf(fp,"%d %d",atom->no, atom->n_ex_atom);
    for (j=0; j<atom->n_ex_atom;j++) {
      fprintf(fp," %d",atom->ex_atom[j]->no);
    }
    fprintf(fp,"\n");
  }
}



void mdat_output_bond(mdat_t *mdat, par_t *par, FILE *fp)
{
  /* bond information */
  mdat_bond_t *bond;
  par_bond_t *pbond;
  int atom0, atom1;

  fprintf(fp,"%d %d\n",mdat->n_bond, 0);

  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    if (bond->atom[0]->no<bond->atom[1]->no) {
      atom0 = bond->atom[0]->no;
      atom1 = bond->atom[1]->no;
    } else {
      atom1 = bond->atom[0]->no;
      atom0 = bond->atom[1]->no;
    }
    fprintf(fp,"%d %d %d\n",atom0, atom1, bond->par_bond->no);
  }
  
  fprintf(fp,"%d\n",par->n_used_bond);
  
  for (pbond=par->bond;pbond!=NULL;pbond=pbond->next) {
    if (pbond->check)
      fprintf(fp,"%lf %lf\n",pbond->kb, pbond->b0);
  }
}

#define AMBER_PI 3.141594
  
void mdat_output_angle(mdat_t *mdat, par_t *par, FILE *fp, int amber)
{
  mdat_angle_t *angle;
  par_angle_t *pangle;
  int atom0, atom1, atom2;
  
  /* angle information */
  fprintf(fp,"%d %d\n",mdat->n_angle, 0);

  for (angle=mdat->angle;angle!=NULL;angle=angle->next) {
    if (angle->atom[0]->no<angle->atom[2]->no) {
      atom0 = angle->atom[0]->no;
      atom2 = angle->atom[2]->no;
    } else {
      atom2 = angle->atom[0]->no;
      atom0 = angle->atom[2]->no;
    }
    atom1 = angle->atom[1]->no;
    fprintf(fp,"%d %d %d %d\n",atom0, atom1, atom2, angle->par_angle->no);
  }
  
  fprintf(fp,"%d\n",par->n_used_angle);
  for (pangle=par->angle;pangle!=NULL;pangle=pangle->next) {
    if (pangle->check)
      if (amber)
	fprintf(fp,"%.8e %.8e %.8e %.8e\n",pangle->Ktheta,
		pangle->Theta0 * AMBER_PI/180.0 ,
		pangle->Kub,pangle->S0);
      else
	fprintf(fp,"%.13e %.13e %.13e %.13e\n",pangle->Ktheta,
		pangle->Theta0 * M_PI/180.0 ,
		pangle->Kub,pangle->S0);

  }
}

void mdat_output_dihedral(mdat_t *mdat, par_t *par, FILE *fp)
{
  mdat_dihedral_t *dihedral;
  par_dihedral_t *pdihedral;
  int pdihedral_no;
  mdat_impr_t *impr;
  par_impr_t *pimpr;
  int i, flag;
  
  /* dihedral information */
  fprintf(fp,"%d %d\n",mdat->n_dihedral_term+mdat->n_impr,
	  mdat->n_impr);

  for (dihedral=mdat->dihedral;dihedral!=NULL;dihedral=dihedral->next) {
    if (dihedral->omit14)
      flag = DIHED_RING;
    else
      flag = 0;
    if (dihedral->only14) {
      flag |= DIHED_ONLY14;
    }
    for (i=0;i<dihedral->n_par_dihedral;i++) {
      if (dihedral->only14) {
	pdihedral_no = -1;
      } else {
	pdihedral_no = dihedral->par_dihedral[i]->no;
      }
      fprintf(fp,"%d %d %d %d %d %d\n",
	      dihedral->atom[0]->no,dihedral->atom[1]->no,
	      dihedral->atom[2]->no,dihedral->atom[3]->no,
	      pdihedral_no,
	      (i<dihedral->n_par_dihedral-1) ? flag|DIHED_MULTI : flag);
    }
    
  }
  
  for (impr=mdat->impr;impr!=NULL;impr=impr->next) {
    fprintf(fp,"%d %d %d %d %d %d\n",
	    impr->atom[0]->no,impr->atom[1]->no,
	    impr->atom[2]->no,impr->atom[3]->no,
	    impr->par_impr->no, DIHED_CHARMM_IMPR);
  }
  
  fprintf(fp,"%d\n",par->n_used_dihedral);
  for (pdihedral=par->dihedral;pdihedral!=NULL;pdihedral=pdihedral->next) {
    if (pdihedral->check)
      fprintf(fp,"%.13e %.13e %.13e\n",pdihedral->Kchi,(double)(pdihedral->n),
	      pdihedral->delta*M_PI/180.0);
  }
  
  fprintf(fp,"%d\n",par->n_used_impr);
  for (pimpr=par->impr;pimpr!=NULL;pimpr=pimpr->next) {
    if (pimpr->check)
      fprintf(fp,"%.13e %.13e\n",pimpr->Kpsi,pimpr->psi0*M_PI/180.0);
  }
}

void mdat_output_dihedral_amber(mdat_t *mdat, par_t *par, FILE *fp)
{
  mdat_dihedral_t *dihedral;
  par_dihedral_t *pdihedral;
  int pdihedral_no;
  mdat_impr_t *impr;
  par_impr_t *pimpr;
  int i, flag;
  
  /* dihedral information */
  fprintf(fp,"%d %d\n",mdat->n_dihedral_term+mdat->n_impr, 0);

  for (dihedral=mdat->dihedral;dihedral!=NULL;dihedral=dihedral->next) {
    if (dihedral->omit14)
      flag = DIHED_RING;
    else
      flag = 0;
    if (dihedral->only14) {
      flag |= DIHED_ONLY14;
    }
    for (i=0;i<dihedral->n_par_dihedral;i++) {
      if (dihedral->only14) {
	pdihedral_no = -1;
      } else {
	pdihedral_no = dihedral->par_dihedral[i]->no;
      }
      fprintf(fp,"%d %d %d %d %d %d\n",
	      dihedral->atom[0]->no,dihedral->atom[1]->no,
	      dihedral->atom[2]->no,dihedral->atom[3]->no,
	      pdihedral_no,
	      (i<dihedral->n_par_dihedral-1) ? flag|DIHED_MULTI : flag);
    }
    
  }
  
  for (impr=mdat->impr;impr!=NULL;impr=impr->next) {
    fprintf(fp,"%d %d %d %d %d %d\n",
	    impr->atom[0]->no,impr->atom[1]->no,
	    impr->atom[2]->no,impr->atom[3]->no,
	    impr->par_impr->no+par->n_used_dihedral, DIHED_AMBER_IMPR);
  }
  
  fprintf(fp,"%d\n",par->n_used_dihedral+par->n_used_impr);
  for (pdihedral=par->dihedral;pdihedral!=NULL;pdihedral=pdihedral->next) {
    if (pdihedral->check)
      fprintf(fp,"%.13e %.13e %.13e\n",pdihedral->Kchi,(double)(pdihedral->n),
	      pdihedral->delta*M_PI/180.0);
      /*fprintf(fp,"%.8e %.8e %.8e\n",pdihedral->Kchi,(double)(pdihedral->n),
	      pdihedral->delta*AMBER_PI/180.0);
      */
  }
  
  /*fprintf(fp,"%d\n",par->n_used_impr);*/
  for (pimpr=par->impr;pimpr!=NULL;pimpr=pimpr->next) {
    if (pimpr->check)
      fprintf(fp,"%.13e %.13e %.13e\n",pimpr->Kpsi,pimpr->psi,
	      pimpr->psi0*M_PI/180.0);
      /*fprintf(fp,"%.8e %.8e %.8e\n",pimpr->Kpsi,pimpr->psi,
	      pimpr->psi0*AMBER_PI/180.0);*/
  }
  fprintf(fp,"%d\n",0);  /* for impr */
  
}


void mdat_output_cmap(mdat_t *mdat, par_t *par, FILE *fp)
{
  mdat_cmap_t *cmap;
  par_cmap_t *pcmap;
  int i;

  fprintf(fp,"%d\n",mdat->n_cmap);
  for (cmap=mdat->cmap;cmap!=NULL;cmap=cmap->next) {
    for (i=0;i<8;i++) {
      fprintf(fp,"%d ",cmap->atom[i]->no);
    }
    fprintf(fp,"%d\n", cmap->par_cmap->no);
  }

  fprintf(fp,"%d\n",par->n_used_cmap);
  for (pcmap=par->cmap;pcmap!=NULL;pcmap=pcmap->next) {
    if (pcmap->check) {
      fprintf(fp,"%d\n",pcmap->ndiv);
      for (i=0;i<pcmap->ndiv*pcmap->ndiv;i++) {
	fprintf(fp,"%.6f",pcmap->map[i]);
	if (((i+1)%pcmap->ndiv)%5==0) {
	  fprintf(fp,"\n");
	} else {
	  fprintf(fp," ");
	}
      }
    }
  }
}


void mdat_output_rigid_body(mdat_t *mdat, par_t *par, FILE *fp)
{
  fprintf(fp, "0\n");
}

void mdat_output_boundary(mdat_t *mdat, par_t *par, FILE *fp)
{
  int i,j;
  
  fprintf(fp, "%d %f %f %f %f %f\n", mdat->ifcap, mdat->radius, mdat->kcap,
	  mdat->center[0],mdat->center[1],mdat->center[2]);
  fprintf(fp, "%d\n",mdat->ifbox);
  for (i=0;i<3;i++) {
    fprintf(fp,"%f %f %f\n",
	    mdat->boxv[i][0],mdat->boxv[i][1],mdat->boxv[i][2]);
  }

}

void mdat_output_pdb_atom(mdat_atom_t *a, FILE *fp)
{
  mdat_res_t *r;
  char buf[5];
  char atom_no[10], res_no[10];
  int orig_res_no;
  char orig_chain;
  char *header;

  r = a->res;
  
  if (a->no+1 < 100000) {
    sprintf(atom_no, "%5d ", a->no+1);
  } else {
    sprintf(atom_no, "%6d" , a->no+1);
  }
  if (r->pdb_no < 10000) {
    sprintf(res_no, "%4d ", r->pdb_no);
  } else {
    sprintf(res_no, "%5d" , r->pdb_no);
  }
  if (r->pdb_res) {
    orig_res_no = r->pdb_res->no;
    orig_chain = r->pdb_res->chain;
  } else {
    orig_res_no = r->no+1;
    orig_chain = ' ';
  }
  header = "ATOM  ";
  fprintf(fp,"%-s%-6s%-4s %-4s%c%-5s   %8.3f%8.3f%8.3f %5.2f %5.2f\n",
	  header,
	  atom_no,pdb_atom_name(a->name,buf),r->name,r->pdb_chain,res_no,
	  a->x,a->y,a->z, 1.0, 1.0);
}

void mdat_output_pdb(mdat_t *mdat, char *fname)
{
  mdat_atom_t *a;
  mdat_res_t *r;
  int i=1;
  FILE *fp;
  char buf[5];
  char atom_no[10], res_no[10];
  int orig_res_no;
  char orig_chain;

  if ((fp=fopen(fname,"w"))==NULL) {
    printf("ERROR: Can't open pdb file %s.\n", fname);
    return;
  }

  for (r=mdat->res;r!=NULL;r=r->next) {
    for (a=r->beg_atom;a!=NULL&&a->prev!=r->end_atom;a=a->next) {
      if (a->no+1 < 100000) {
	sprintf(atom_no, " %5d ", a->no+1);
      } else if (a->no+1 < 1000000) {
	sprintf(atom_no, " %6d" , a->no+1);
      } else {
	sprintf(atom_no, "%7d" , a->no+1);
      }
      /*
      if (r->no+1 < 10000) {
	sprintf(res_no, "%4d ", r->no+1);
      } else {
	sprintf(res_no, "%5d" , r->no+1);
      }
      if (r->pdb_res) {
	orig_res_no = r->pdb_res->no;
	orig_chain = r->pdb_res->chain;
      } else {
	orig_res_no = r->no+1;
	orig_chain = ' ';
      }
      */
      if (r->pdb_no < 10000) {
	sprintf(res_no, "%4d  ", r->pdb_no);
      } else if (r->pdb_no < 100000) {
	sprintf(res_no, "%5d " , r->pdb_no);
      } else {
	sprintf(res_no, "%6d" , r->pdb_no);
      }
      if (r->pdb_res) {
	orig_res_no = r->pdb_res->no;
	orig_chain = r->pdb_res->chain;
      } else {
	orig_res_no = r->no+1;
	orig_chain = ' ';
      }
      /*
      fprintf(fp,"ATOM  %-6s%-4s %-4s%c%-5s   %8.3f%8.3f%8.3f %5.2f %5.2f %5d%c\n",
	      atom_no,pdb_atom_name(a->name,buf),r->name,r->pdb_chain,res_no,
	      a->x,a->y,a->z, 1.0, 1.0, orig_res_no, orig_chain);
      */
      fprintf(fp,"ATOM %-7s%-4s %-4s%c%-6s  %8.3f%8.3f%8.3f %5.2f %5.2f",
	      atom_no,pdb_atom_name(a->name,buf),r->name,r->pdb_chain,res_no,
	      a->x,a->y,a->z, 1.0, 1.0);
      if (mdat->output_pdb_lj_q) {
	fprintf(fp," %9f %9f %9f\n", a->charge, a->par_nonbonded->eps, 
		a->par_nonbonded->rmin2);
      } else {
	fprintf(fp," %5d%c\n", orig_res_no, orig_chain);
      }
    }
    if (r->last) {
      fprintf(fp,"TER\n");
    }
  }
  fprintf(fp,"END\n");
  fclose(fp);
}

void mdat_output_pdb_original_res(mdat_t *mdat, char *fname)
{
  mdat_atom_t *a;
  mdat_res_t *r;
  int i=1;
  FILE *fp;
  char buf[5];
  char atom_no[10], res_no[10];

  if ((fp=fopen(fname,"w"))==NULL) {
    printf("ERROR: Can't open pdb file %s.\n", fname);
    return;
  }

  for (r=mdat->res;r!=NULL;r=r->next) {
    for (a=r->beg_atom;a!=NULL&&a->prev!=r->end_atom;a=a->next) {
      if (a->no+1 < 100000) {
	sprintf(atom_no, "%5d ", a->no+1);
      } else {
	sprintf(atom_no, "%6d" , a->no+1);
      }
      if (r->pdb_res) {
	if (r->pdb_res->no < 10000) {
	  sprintf(res_no, "%4d  ", r->pdb_res->no);
	} else if (r->pdb_res->no < 100000) {
	  sprintf(res_no, "%5d ", r->pdb_res->no);
	} else {
	  sprintf(res_no, "%6d" , r->pdb_res->no);
	}
	fprintf(fp,"ATOM  %-6s%-4s %-4s%c%-6s  %8.3f%8.3f%8.3f %5.2f %5.2f %6d\n",
		atom_no, pdb_atom_name(a->name,buf),
		r->pdb_res->pdb_name,
		r->pdb_res->chain, res_no,
		a->x,a->y,a->z, 1.0, 1.0, r->no+1);
	/*
	else
	  fprintf(fp,"ATOM%7d %-4s %-4s %5d    %8.3f%8.3f%8.3f %5.2f %5.2f\n",
		  a->no+1,pdb_atom_name(a->name,buf),
		  r->pdb_res->pdb_name, res_no,
		  a->x,a->y,a->z, 1.0, 1.0);
	*/
      } else {
	if (r->no+1 < 10000) {
	  sprintf(res_no, "%4d  ", r->no+1);
	} else if (r->no+1 < 100000) {
	  sprintf(res_no, "%5d ", r->no+1);
	} else {
	  sprintf(res_no, "%6d" , r->no+1);
	}
	fprintf(fp,"ATOM  %-6s%-4s %-4s %-6s  %8.3f%8.3f%8.3f %5.2f %5.2f %6d\n",
		atom_no,pdb_atom_name(a->name,buf),r->name, res_no,
		a->x,a->y,a->z, 1.0, 1.0, r->no+1);
      }
    }
    if (r->last) {
      fprintf(fp,"TER\n");
    }
  }
  fprintf(fp,"END\n");
  fclose(fp);
}


void mdat_output_crd(mdat_t *mdat, char *fname)
{
  mdat_atom_t *a;
  mdat_res_t *r;
  int i=1;
  FILE *fp;

  if ((fp=fopen(fname,"w"))==NULL) {
    printf("ERROR: Can't open crd file %s.\n", fname);
    return;
  }

  fprintf(fp,"MARBLE CRD FILE (X) Ver. %d.%d %f\n",1, 2, 0.0);
  fprintf(fp,"%d\n", mdat->n_atom);

  for (a=mdat->atom;a!=NULL;a=a->next) {
    fprintf(fp," %13e %13e %13e", a->x,a->y,a->z);
    if (i++ % 2 == 0) fprintf(fp,"\n");
  }
  fprintf(fp,"\n");
  fclose(fp);
}

void mdat_output_bond_file(mdat_t *mdat, char *fname)
{
  mdat_bond_t *bond;
  par_bond_t *pbond;
  mdat_atom_t *atom0, *atom1;
  int n;
  FILE *fp;
  vec_t v;

  if ((fp=fopen(fname,"w"))==NULL) {
    printf("ERROR: Can't open file %s.\n", fname);
    return;
  }

  fprintf(fp,"# number of bonds %d\n",mdat->n_bond);

  n = 1;
  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    if (bond->atom[0]->no<bond->atom[1]->no) {
      atom0 = bond->atom[0];
      atom1 = bond->atom[1];
    } else {
      atom1 = bond->atom[0];
      atom0 = bond->atom[1];
    }
    v.x = atom0->x - atom1->x;
    v.y = atom0->y - atom1->y;
    v.z = atom0->z - atom1->z;

    fprintf(fp,"%d %d %d %lf %lf\n",n, atom0->no+1, atom1->no+1, 
	    v_len(&v), bond->par_bond->b0);
    n++;
  }
  fclose(fp);
}

void mdat_output_angle_file(mdat_t *mdat, char *fname)
{
  mdat_angle_t *angle;
  par_angle_t *pangle;
  mdat_atom_t *atom0, *atom1, *atom2;
  vec_t v0, v2;
  int n;
  FILE *fp;

  if ((fp=fopen(fname,"w"))==NULL) {
    printf("ERROR: Can't open file %s.\n", fname);
    return;
  }

  fprintf(fp,"# number of angles %d\n",mdat->n_angle);

  n = 1;
  for (angle=mdat->angle;angle!=NULL;angle=angle->next) {
    if (angle->atom[0]->no < angle->atom[2]->no) {
      atom0 = angle->atom[0];
      atom2 = angle->atom[2];
    } else {
      atom2 = angle->atom[0];
      atom0 = angle->atom[2];
    }
    atom1 = angle->atom[1];
    v0.x = atom0->x - atom1->x;
    v0.y = atom0->y - atom1->y;
    v0.z = atom0->z - atom1->z;
    v2.x = atom2->x - atom1->x;
    v2.y = atom2->y - atom1->y;
    v2.z = atom2->z - atom1->z;

    fprintf(fp,"%d %d %d %lf %lf\n",n, atom0->no+1, atom1->no+1, 
	    v_ang(&v0, &v2)*180/M_PI, angle->par_angle->Theta0);
    n++;
  }
  fclose(fp);
}

void mdat_input_crd(mdat_t *mdat, char *fname)
{
  mdat_atom_t *a;
  mdat_res_t *r;
  int i=1;
  FILE *fp;
  char buf[1000], type_str[10];
  int major, minor;
  double current_time;
  int n_atom;

  if ((fp=fopen(fname,"r"))==NULL) {
    printf("ERROR: Can't open crd file %s.\n", fname);
    return;
  }

  printf("Input crd file: %s\n", fname);

  fgets(buf, 1000, fp);   
  if (sscanf(buf, "MARBLE CRD FILE (%[A-Z]) Ver. %d.%d %lf",
	     type_str, &major, &minor, &current_time) != 4) {
    printf("ERROR: unknown file header: %s\n", fname);
    fclose(fp);
    return;
  }

  fscanf(fp,"%d", &n_atom);
  
  if (mdat->n_atom > n_atom) {
    printf("ERROR: the number of atoms in crd file (%d) is smaller than that of mdat file (%d)\n", n_atom, mdat->n_atom);
    fclose(fp);
    return;
  } else if (mdat->n_atom < n_atom) {
    printf("ERROR: the number of atoms in crd file (%d) is larger than that of mdat file (%d)\n", n_atom, mdat->n_atom);
  }

  for (a=mdat->atom;a!=NULL;a=a->next) {
    fscanf(fp,"%lf%lf%lf", &a->x, &a->y, &a->z);
  }
  fclose(fp);
  printf("\n");
}

void mdat_output_xyzs(mdat_t *mdat, char *fname)
{
  mdat_atom_t *a;
  FILE *fp;
  double c;
  char buf[5];

  if ((fp=fopen(fname,"w"))==NULL) {
    printf("ERROR: Can't open file %s.\n", fname);
    return;
  }

  /* atom information */
  c = pow(2.0,-1.0/6.0);
  fprintf(fp,"%d\n",mdat->n_atom);
  for (a=mdat->atom;a!=NULL;a=a->next) {
    fprintf(fp,"%10.3f %10.3f %10.3f %10.7f %-4s %-4s%c%6d\n",
	    a->x, a->y, a->z, a->par_nonbonded->rmin2*c,
	    pdb_atom_name(a->name,buf),
	    a->res->pdb_res->pdb_name,
	    a->res->pdb_res->chain, 
	    a->res->pdb_res->no);
  }
  fclose(fp);
}
