/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifndef _MDAT_H_INCLUDED
#define _MDAT_H_INCLUDED

typedef struct mdat {
  int n_atom, n_res, n_mol, n_bond, n_angle, n_dihedral, n_impr, n_cmap;
  int n_dihedral_term;
  int n_solute_mol;
  double charge, scnb, scee;
  
  struct mdat_atom *atom;
  struct mdat_res  *res;
  struct mdat_mol  *mol;
  struct mdat_bond  *bond;
  struct mdat_angle *angle;
  struct mdat_dihedral *dihedral;
  struct mdat_impr *impr;
  struct mdat_cmap *cmap;
  struct mdat_ic *ic;

  int ifcap, ifbox;
  double boxv[3][3], center[3], kcap, radius;
  double recip[3][3], reclen[3];

  int n_c_water, n_c_cation, n_c_anion;

  int output_pdb_lj_q;
} mdat_t;


typedef struct mdat_atom {
  int no, coord;
  char name[MAXANUM];
  double x, y, z, r, charge;
  
  pdb_atom_t *pdb_atom;
  top_atom_t *top_atom;
  top_mass_t *mass;
  par_nonbonded_t *par_nonbonded;
  struct mdat_res *res;

  int n_b_atom, n_ex_atom;
  struct mdat_atom *prev, *next;
  struct mdat_atom **b_atom, **ex_atom;
  struct mdat_atom *gnext;
} mdat_atom_t;

#define MDAT_SOLUTE         1
#define MDAT_CRYSTAL_CATION 2
#define MDAT_CRYSTAL_ANION  3
#define MDAT_CRYSTAL_WATER  4
#define MDAT_INPUT_PDB      4        
#define MDAT_ADDED_CATION   5
#define MDAT_ADDED_ANION    6
#define MDAT_ADDED_WATER    7
#define MDAT_ADDED_OTHER    8

typedef struct mdat_res {
  int no, n_atom, mol_no;
  char name[MAXRNUM];
  struct pdb_res *pdb_res;
  struct top_res *top_res;

  struct mdat_atom     *beg_atom,*end_atom;
  struct mdat_bond     *beg_bond,*end_bond;
  struct mdat_impr     *beg_impr,*end_impr;
  struct mdat_cmap     *beg_cmap,*end_cmap;
  struct mdat_ic       *beg_ic,  *end_ic;

  struct mdat_mol      *mol;
  /*
  struct mdat_angle    *beg_angle,*end_angle;
  struct mdat_dihedral *beg_dihedral,*end_dihedral; 
  */
  
  struct mdat_res *prev, *next;

  int first, last, patched, noangle, nodihedral, flag;

  int  pdb_no;
  char pdb_chain;
} mdat_res_t;

typedef struct mdat_mol {
  int no, n_atom;
  char chain;
  struct mdat_res *beg_res, *end_res;
  struct mdat_mol *prev, *next;
} mdat_mol_t;

typedef struct mdat_bond{
  int no;
  int type;
  struct mdat_atom *atom[2];
  struct mdat_res  *res;
  struct par_bond *par_bond;
  struct mdat_bond *prev,*next;
} mdat_bond_t;

typedef struct mdat_angle {
  int no;
  struct mdat_atom *atom[3];
  struct par_angle *par_angle;
  struct mdat_angle *prev,*next;
} mdat_angle_t;

/* import from marble */
#define DIHED_OTHER_NODE   1
#define DIHED_AMBER_IMPR   2
#define DIHED_CHARMM_IMPR  4
#define DIHED_RING         8
#define DIHED_MULTI       16
#define DIHED_OMIT14      30  /* 2+4+8+16 */
#define DIHED_ONLY14      32  

typedef struct mdat_dihedral {
  int no;
  int omit14, only14;
  int n_par_dihedral;
  struct mdat_atom *atom[4];
  struct par_dihedral **par_dihedral;
  struct mdat_dihedral *prev,*next;
} mdat_dihedral_t;

typedef struct mdat_impr {
  int no;
  struct mdat_atom *atom[4];
  struct mdat_res  *res;
  struct par_impr  *par_impr;
  struct mdat_impr *prev,*next;
} mdat_impr_t;

typedef struct mdat_cmap {
  int no;
  struct mdat_atom *atom[8];
  struct mdat_res  *res;
  struct par_cmap  *par_cmap;
  struct mdat_cmap *prev,*next;
} mdat_cmap_t;

typedef struct mdat_ic {
  int no, impr;
  struct mdat_atom *atom[4];
  double            length[2];
  double            angle[2];
  double            dihedral;
  struct mdat_res  *res;
  struct mdat_ic   *prev,*next;
} mdat_ic_t;

void mdat_init(mdat_t *mdat);
void mdat_make_data_for_charmm(mdat_t *mdat, pdb_t *pdb);
void mdat_make_res_from_pdb_res(mdat_t *mdat, pdb_t *pdb);
void mdat_make_atom(mdat_t *mdat);
void mdat_insert_residues(mdat_t *mdat, mdat_res_t *pmr, top_res_t *tr, int num);
void mdat_make_mol(mdat_t *mdat);
void mdat_make_bond(mdat_t *mdat);
void mdat_make_impr(mdat_t *mdat);
void mdat_make_cmap(mdat_t *mdat);
void mdat_make_ic(mdat_t *mdat);
void mdat_make_connect(mdat_t *mdat);
void mdat_make_ex_atom(mdat_t *mdat);
int  mdat_atom_cmp(const mdat_atom_t **a1, const mdat_atom_t **a2);
void mdat_make_angle(mdat_t *mdat);
void mdat_make_dihedral(mdat_t *mdat);
void mdat_make_coordinates(mdat_t *mdat);
void mdat_make_coord_from_ic(mdat_t *mdat, par_t *par);
void mdat_ic_add(mdat_t *mdat);
void mdat_make_ic_param(mdat_t *mdat, par_t *par);
void mdat_apply_ic_seed(mdat_t *mdat);
double mdat_search_length_from_ic(mdat_t *mdat, mdat_atom_t *atom[2]);
double mdat_search_angle_from_ic(mdat_t *mdat, mdat_atom_t *atom[3]);
void mdat_internal_to_xyz(mdat_atom_t *atom,  mdat_atom_t *atom0,
			  mdat_atom_t *atom1, mdat_atom_t *atom2,
			  double length, double angle, double dihedral);


mdat_atom_t *mdat_search_atom_in_res(char *name, mdat_res_t *mres);
mdat_atom_t *mdat_search_atom_in_res2(char *name, mdat_res_t *mres2[2], int *i);
mdat_res_t *mdat_search_res_from_pdbno(mdat_t *mdat, int resno, char chain);

void mdat_add_mass(mdat_t *mdat, top_t *top);
void mdat_add_par(mdat_t *mdat, par_t *par);
void mdat_add_par_nonbonded(mdat_t *mdat, par_t *par);

void mdat_count_all(mdat_t *mdat);
void mdat_count_atom(mdat_t *mdat);
void mdat_count_bond(mdat_t *mdat);
void mdat_count_angle(mdat_t *mdat);
void mdat_count_dihedral(mdat_t *mdat);
void mdat_count_impr(mdat_t *mdat);
void mdat_count_cmap(mdat_t *mdat);
void mdat_set_solute(mdat_t *mdat);

void mdat_print_info(mdat_t *mdat, par_t *par);

void mdat_print_info(mdat_t *mdat, par_t *par);
void mdat_set_boxv(mdat_t *mdat, double f1, double f2, double f3,
                   double alpha, double beta, double gamma);
void mdat_merge(mdat_t *dest, mdat_t *src);
void mdat_set_radii(mdat_t *mdat);
mdat_atom_t *mdat_search_atom(mdat_t *pdb, char *atom, char *res, int res_no, char chain);
void mdat_renumber_residue(mdat_t *mdat);
void mdat_check_res_no(mdat_t *mdat);
void mdat_wrap_molecules(mdat_t *mdat);
void mdat_make_recip(mdat_t *mdat);


/* mdat_output.c  */

void mdat_output(mdat_t *mdat, par_t *par);
void mdat_output_atom(mdat_t *mdat, par_t *par, FILE *fp, int amber);
void mdat_output_bond(mdat_t *mdat, par_t *par, FILE *fp);
void mdat_output_angle(mdat_t *mdat, par_t *par, FILE *fp, int amber);
void mdat_output_dihedral(mdat_t *mdat, par_t *par, FILE *fp);
void mdat_output_dihedral_amber(mdat_t *mdat, par_t *par, FILE *fp);
void mdat_output_impr(mdat_t *mdat, par_t *par, FILE *fp);
void mdat_output_cmap(mdat_t *mdat, par_t *par, FILE *fp);
void mdat_output_rigid_body(mdat_t *mdat, par_t *par, FILE *fp);
void mdat_output_boundary(mdat_t *mdat, par_t *par, FILE *fp);

void mdat_output_pdb_atom(mdat_atom_t *a, FILE *fp);

void mdat_output_pdb(mdat_t *mdat, char *fname);
void mdat_output_pdb_original_res(mdat_t *mdat, char *fname);
void mdat_output_crd(mdat_t *mdat, char *fname);

void mdat_set_boxv(mdat_t *mdat, double f1, double f2, double f3,
		   double alpha, double beta, double gamma);

void mdat_merge(mdat_t *dest, mdat_t *src);
void mdat_set_radii(mdat_t *mdat);
void mdat_set_radii_orig(mdat_t *mdat);
mdat_atom_t *mdat_search_atom(mdat_t *pdb, char *atom, char *res, int res_no, char chain);
void mdat_renumber_residue(mdat_t *mdat);
void mdat_check_res_no(mdat_t *mdat);
void mdat_output_bond_file(mdat_t *mdat, char *fname);
void mdat_output_angle_file(mdat_t *mdat, char *fname);
void mdat_input_crd(mdat_t *mdat, char *fname);
void mdat_output_xyzs(mdat_t *mdat, char *fname);

extern mdat_t _mdat;

#endif /* _MDAT_H_INCLUDED */
