/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>

#include "util.h"
#include "charmm_par.h"
#include "pdb.h"
#include "charmm_top.h"
#include "config.h"
#include "mdat.h"
#include "solvate.h"
#include "align_axis.h"

void mdat_init(mdat_t *mdat)
{
  int i,j;
  
  mdat->ifcap = mdat->ifbox = 0;
  for (i=0;i<3;i++) {
    mdat->center[i]=0.0;
    for (j=0;j<3;j++)
      mdat->boxv[i][j]=0.0;
  }
  mdat->kcap = mdat->radius = 0.0;
  mdat->scnb = mdat->scee = 1.0;
  mdat->output_pdb_lj_q = 0;
}  

void mdat_make_data_for_charmm(mdat_t *mdat, pdb_t *pdb)
{
  /* make mdat_res and mdat_atom from pdb_res */
  mdat_make_res_from_pdb_res(mdat, pdb);
  mdat_make_atom(mdat);
  
  /* make mdat_bond */
  mdat_make_bond(mdat);
  mdat_make_impr(mdat);
  mdat_make_cmap(mdat);
  mdat_make_ic(mdat);
}

/* make mdat from pdb residue list */
void mdat_make_res_from_pdb_res(mdat_t *mdat, pdb_t *pdb)
{
  mdat_res_t  *mr,*mr_prev;
  mdat_atom_t *ma,*ma_prev;
  pdb_res_t *pr;
  top_res_t  *tr;
  top_atom_t *ta;
  int ok = 1, no = 1;

  /* allocation and make list for mdat_res and mdat_atom */
  mdat->res  = NULL;
  for (pr = pdb->res,mr_prev=NULL;pr!=NULL;pr=pr->next,mr_prev=mr) {
    mr = emalloc("mdat_make_res_from_pdb_res", sizeof(mdat_res_t));
    mr->next = NULL;
    if (mdat->res == NULL) {
      mdat->res = mr;
      mr->prev = NULL;
    } else {
      mr_prev->next = mr;
      mr->prev = mr_prev;
    }

    mr->no = no++;
    mr->pdb_res = pr;
    mr->pdb_no = pr->no;
    mr->pdb_chain = pr->chain;
    
    mr->top_res = tr = top_search_res(pr->name);
    if (mr->top_res==NULL) {
      printf("ERROR: Unable to assign residue %s (%d%c) in pdb file to residue in top file\n",
	     pr->name,pr->no, pr->chain);
      ok = 0;
    }
    if (tr!=NULL)
      strcpy(mr->name,tr->name);
    /*
    strcpy(mr->name,pr->name);
    */
    
    mr->noangle = mr->nodihedral = 0;
    if (tr!=NULL && strcmp(tr->name,"TIP3") == 0) {
      mr->nodihedral = 1;
    }

    mr->patched = 0;
    mr->first = mr->last = 0;
    if (mr->prev==NULL || mr->prev->last==1) mr->first = 1;
    if (pr->next==NULL || pr->ter) mr->last = 1;
  }
  if (!ok)
    exit(1);

  if (!mdat->ifbox && pdb->alpha!=0.0) {
    mdat_set_boxv(mdat, pdb->a,pdb->b,pdb->c,
		  pdb->alpha,pdb->beta,pdb->gamma);
  }
}



void mdat_make_atom(mdat_t *mdat)
{
  mdat_res_t  *mres;
  mdat_atom_t *matom,*matom_prev;
  top_atom_t  *ta;
  int no = 1;

  mdat->atom = NULL;
  for (mres = mdat->res;mres!=NULL;mres=mres->next) {
    for (ta=mres->top_res->atom;ta!=NULL;ta=ta->next,matom_prev=matom) {
      matom = emalloc("mdat_make_atom", sizeof(mdat_atom_t));
      matom->top_atom=ta;
      matom->res = mres;
      if (ta==mres->top_res->atom)
	mres->beg_atom = matom;
      matom->next = NULL;
      if (mdat->atom == NULL) {
	mdat->atom = matom;
	matom->prev = NULL;
      } else {
	matom_prev->next = matom;
	matom->prev = matom_prev;
      }
      strcpy(matom->name,ta->name);
      matom->no = no++;
      matom->coord = 0;
      mres->end_atom=matom;
    }
  }
}

void mdat_insert_residues(mdat_t *mdat, mdat_res_t *pmr, top_res_t *tr, int num)
{
  mdat_res_t *mr;
  mdat_atom_t *matom, *matom_prev, *matom_next;
  top_atom_t *ta;
  char *fname = "mdat_insert_residues";
  
  int i;

  for (i=0;i<num;i++) {
    mr = emalloc(fname, sizeof(mdat_res_t));
    if (pmr) {
      mr->next = pmr->next;
      if (pmr->next) {
	pmr->next->prev = mr;
	matom_next = pmr->next->beg_atom;
      } else {
	matom_next = NULL;
      }
      pmr->next = mr;
      matom_prev = pmr->end_atom;
    } else {
      mdat->res = mr;
      matom_prev = NULL;
      matom_next = mdat->atom;
    }
    mr->prev = pmr;
    pmr=mr;

    mr->no = 0;
    mr->pdb_res = NULL;
    mr->pdb_no     = 0;
    mr->pdb_chain  = ' ';
    
    mr->top_res = tr;
    mr->noangle=mr->nodihedral=1;
    mr->patched = 0;
    mr->first = mr->last = 0;
    strcpy(mr->name, tr->name);

    for (ta=tr->atom;ta!=NULL;ta=ta->next) {
      matom = emalloc(fname, sizeof(mdat_atom_t));
      matom->top_atom=ta;
      matom->charge = ta->charge;
      matom->res = mr;
      if (ta==mr->top_res->atom)
	mr->beg_atom = matom;
      
      strcpy(matom->name,ta->name);
      matom->no = 0;
      matom->coord = 0;
      mr->end_atom=matom;

      matom->next = matom_next;
      if (matom_next)
	matom_next->prev = matom;
      
      matom_prev->next = matom;
      matom->prev = matom_prev;
      matom_prev = matom;
    }
  }
}

void mdat_set_mol_no_to_res(mdat_res_t *res, int mol_no)
{
  mdat_atom_t *atom;
  int i;

  res->mol_no = mol_no;
  for (atom=res->beg_atom;atom!=NULL;atom=atom->next) {
    for (i=0;i<atom->n_b_atom;i++) {
      if (atom->b_atom[i]->res->mol_no < 0)
	mdat_set_mol_no_to_res(atom->b_atom[i]->res,mol_no);
      else if (atom->b_atom[i]->res->mol_no != mol_no) {
	printf("ERROR: Internal error. why mol_no is different?: %d(%d)-%d(%d)\n",
	       atom->b_atom[i]->res->mol_no, 
	       atom->b_atom[i]->res->pdb_res->no,
	       mol_no, res->pdb_res->no);
	exit(1);
      }
    }
    if (atom==res->end_atom) break;
  }
  /*
  for (bond=res->beg_bond;bond!=NULL;bond=bond->next) {
    if (bond->atom[0]->res->mol_no != mol_no)
      mdat_set_mol_no_to_res(bond->atom[0]->res,mol_no);
    if (bond->atom[1]->res->mol_no != mol_no)
      mdat_set_mol_no_to_res(bond->atom[1]->res,mol_no);
    if (bond==res->end_bond) break;
  }
  */
}

/* require mdat_make_bond and patch */
void mdat_make_mol(mdat_t *mdat)
{
  mdat_res_t *res;
  mdat_mol_t *mol, *mol_prev, *mol1, *mol2;
  int mol_no = 0;

  /* initialization */
  for (res=mdat->res;res!=NULL;res=res->next) {
    res->mol_no = -1;
  }
  
  for (res=mdat->res;res!=NULL;res=res->next) {
    if (res->mol_no < 0) {
      mdat_set_mol_no_to_res(res, mol_no);
      mol_no++;
    }
  }


  mol_no = -1;
  mol=NULL;
  for (res=mdat->res;res!=NULL;res=res->next) {
    if (mol_no != res->mol_no) {
      mol_no = res->mol_no;
      /*
      for (mol=mdat->mol;mol!=NULL;mol=mol->next) {
	if (mol->no == mol_no) {
	  printf("ERROR: Connected atoms in a molecule must be placed continously. (residue %d, %d\n", res->pdb_res->no, mol->beg_res->pdb_res->no);
	  exit(1);
	}
      }
      */
      if (mol) {
	if (mol->no > mol_no) {
	  if (res->pdb_res)
	    printf("ERROR: Connected atoms in a molecule must be placed continously. (residue %d, %d)\n", res->pdb_res->no, mol->beg_res->pdb_res->no);
	  else
	    printf("ERROR: Connected atoms in a molecule must be placed continously. (residue %d, %d)\n", res->no, mol->beg_res->no);
	    
	  exit(1);
	}
      }
      
      mol = emalloc("mdat_make_mol", sizeof(mdat_mol_t));
      mol->next = NULL;
      if (mdat->mol==NULL) {
	mdat->mol = mol;
	mol->prev = NULL;
      } else {
	mol_prev->next = mol;
	mol->prev = mol;
      }
      mol->no = res->mol_no;
      mol->beg_res = res;
      if (res->pdb_res)
	mol->chain = res->pdb_res->chain;
      else
	mol->chain = ' ';
      mol_prev = mol;
    }
    res->mol = mol;
    mol->end_res = res;
  }
}
    
void mdat_make_bond(mdat_t *mdat)
{
  mdat_res_t  *mres;
  mdat_bond_t *mbond,*mbond_prev;
  top_bond_t  *tbond;

  mdat->bond = NULL;
  for (mres = mdat->res;mres!=NULL;mres=mres->next) {
    for (tbond=mres->top_res->bond; tbond!=NULL; tbond=tbond->next) {
      mbond = emalloc("mdat_make_bond", sizeof(mdat_bond_t));
      mbond->atom[0] = mdat_search_atom_in_res(tbond->atom[0], mres);
      mbond->atom[1] = mdat_search_atom_in_res(tbond->atom[1], mres);
      
      if (mbond->atom[0]==NULL|| mbond->atom[1]==NULL) {
	free(mbond);
	continue;
      }
      
      if (tbond == mres->top_res->bond)
	mres->beg_bond = mbond;
      mres->end_bond = mbond;
      mbond->res = mres;
      
      mbond->next = NULL;
      if (mdat->bond == NULL) {
	mdat->bond = mbond;
	mbond->prev = NULL;
      } else {
	mbond_prev->next = mbond;
	mbond->prev = mbond_prev;
      }
      
      mbond_prev=mbond;
    }
  }
}

void mdat_make_impr(mdat_t *mdat)
{
  mdat_res_t  *mres;
  mdat_impr_t *mimpr,*mimpr_prev;
  top_impr_t  *timpr;

  mdat->impr = NULL;
  for (mres = mdat->res;mres!=NULL;mres=mres->next) {
    mres->beg_impr = mres->end_impr = NULL;
    for (timpr=mres->top_res->impr; timpr!=NULL; timpr=timpr->next) {
      mimpr = emalloc("mdat_make_impr", sizeof(mdat_impr_t));
      mimpr->atom[0] = mdat_search_atom_in_res(timpr->atom[0], mres);
      mimpr->atom[1] = mdat_search_atom_in_res(timpr->atom[1], mres);
      mimpr->atom[2] = mdat_search_atom_in_res(timpr->atom[2], mres);
      mimpr->atom[3] = mdat_search_atom_in_res(timpr->atom[3], mres);
      if (mimpr->atom[0]==NULL|| mimpr->atom[1]==NULL ||
	  mimpr->atom[2]==NULL|| mimpr->atom[3]==NULL) {
	free(mimpr);
	continue;
      }
      if (timpr == mres->top_res->impr)
	mres->beg_impr = mimpr;
      mres->end_impr = mimpr;
      mimpr->res = mres;

      mimpr->next = NULL;
      if (mdat->impr == NULL) {
	mdat->impr = mimpr;
	mimpr->prev = NULL;
      } else {
	mimpr_prev->next = mimpr;
	mimpr->prev = mimpr_prev;
      }
      
      mimpr_prev=mimpr;
    }
  }
}

void mdat_make_cmap(mdat_t *mdat)
{
  mdat_res_t  *mres;
  mdat_cmap_t *mcmap,*mcmap_prev;
  top_cmap_t  *tcmap;
  int j;

  mdat->cmap = NULL;
  for (mres = mdat->res;mres!=NULL;mres=mres->next) {
    mres->beg_cmap = mres->end_cmap = NULL;
    for (tcmap=mres->top_res->cmap; tcmap!=NULL; tcmap=tcmap->next) {
      mcmap = emalloc("mdat_make_cmap", sizeof(mdat_cmap_t));
      for (j=0;j<8;j++) {
	mcmap->atom[j] = mdat_search_atom_in_res(tcmap->atom[j], mres);
	if (mcmap->atom[j]==NULL) {
	  free(mcmap);
	  j=-1;
	  break;
	}
      }
      if (j==-1) {
	printf("Warning: CMAP term not assigned: ");
	for (j=0;j<8;j++) {
	  printf(" %s", tcmap->atom[j]);
	}
	printf(" in residue %d%c\n", mres->pdb_no, mres->pdb_chain);
	continue;
      }

      if (tcmap == mres->top_res->cmap)
	mres->beg_cmap = mcmap;
      mres->end_cmap = mcmap;
      mcmap->res = mres;

      mcmap->next = NULL;
      if (mdat->cmap == NULL) {
	mdat->cmap = mcmap;
	mcmap->prev = NULL;
      } else {
	mcmap_prev->next = mcmap;
	mcmap->prev = mcmap_prev;
      }
      
      mcmap_prev=mcmap;
    }
  }
}


void mdat_make_ic(mdat_t *mdat)
{
  mdat_res_t *mres;
  mdat_ic_t  *mic,*mic_prev;
  top_ic_t   *tic;
  int i;

  mdat->ic = NULL;
  for (mres = mdat->res;mres!=NULL;mres=mres->next) {
    for (tic=mres->top_res->ic; tic!=NULL; tic=tic->next) {
      mic = emalloc("mdat_make_ic", sizeof(mdat_ic_t));
      mic->atom[0] = mdat_search_atom_in_res(tic->atom[0], mres);
      mic->atom[1] = mdat_search_atom_in_res(tic->atom[1], mres);
      mic->atom[2] = mdat_search_atom_in_res(tic->atom[2], mres);
      mic->atom[3] = mdat_search_atom_in_res(tic->atom[3], mres);
      if (mic->atom[0]==NULL|| mic->atom[1]==NULL ||
	  mic->atom[2]==NULL|| mic->atom[3]==NULL) {
	free(mic);
	continue;
      }
      for (i=0;i<2;i++) {
	mic->length[i]=tic->length[i];
	mic->angle[i] =tic->angle[i];
      }
      mic->dihedral = tic->dihedral;
      if (tic->atom[2][0] == '*')
	mic->impr = 1;
      else
	mic->impr = 0;
      
      if (tic == mres->top_res->ic)
	mres->beg_ic = mic;
      mres->end_ic = mic;
      mic->res = mres; 

      mic->next = NULL;
      if (mdat->ic == NULL) {
	mdat->ic = mic;
	mic->prev = NULL;
      } else {
	mic_prev->next = mic;
	mic->prev = mic_prev;
      }
      
      mic_prev=mic;
    }
  }
}

void mdat_make_connect(mdat_t *mdat)
{
  mdat_atom_t *atom;
  mdat_bond_t *bond;
  double dx, dy, dz, len2, limit2;
  int check;

  /* count connections */
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    atom->n_b_atom=0;
  }
  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    bond->atom[0]->n_b_atom++;
    bond->atom[1]->n_b_atom++;
  }
  /* allocation */
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    atom->b_atom=emalloc("mdat_make_connect",sizeof(mdat_atom_t*)*atom->n_b_atom);
    atom->n_b_atom=0;
  }
  /* making data */
  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    bond->atom[0]->b_atom[bond->atom[0]->n_b_atom++] = bond->atom[1];
    bond->atom[1]->b_atom[bond->atom[1]->n_b_atom++] = bond->atom[0];
  }
  /* check length of bonds */
  check = 0;

  limit2 = _cfg.bond_length_limit*_cfg.bond_length_limit;
  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    dx = bond->atom[0]->x - bond->atom[1]->x;
    dy = bond->atom[0]->y - bond->atom[1]->y;
    dz = bond->atom[0]->z - bond->atom[1]->z;
    len2 = dx*dx+dy*dy+dz*dz;
    if (len2 >= limit2) {
      check = 1;
      printf("Warning: Bond length %.2f between %s (%s %d%c) and %s (%s %d%c) is larger than bond_length_limit %.2f\n",
	     sqrt(len2),
	     bond->atom[0]->name, bond->atom[0]->res->name, bond->atom[0]->res->pdb_res->no,bond->atom[0]->res->pdb_res->chain,
	     bond->atom[1]->name, bond->atom[1]->res->name, bond->atom[1]->res->pdb_res->no,bond->atom[1]->res->pdb_res->chain,
	     _cfg.bond_length_limit);
      /*printf("%d %d\n", bond->atom[0]->no, bond->atom[1]->no);*/
    }
  }
  if (check & !_no_abort) {
    printf("Exit due to large distances of bonds\n");
    printf("Check TER in PDB file, or\n");
    printf("Set the value of bond_length_limit larger, if you want to avoid exitting.\n");
    exit(1);
  }
}

void mdat_make_ex_atom(mdat_t *mdat)
{
  mdat_atom_t *atom, *atom0, *atom2;
  mdat_bond_t *bond;
  mdat_angle_t *angle;
  mdat_dihedral_t *dihedral;
  int i, j, k, ok;

  /* count connections */
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    atom->n_ex_atom=0;
  }
  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    bond->atom[0]->n_ex_atom++;
    bond->atom[1]->n_ex_atom++;
  }

  /*
  for (angle=mdat->angle;angle!=NULL;angle=angle->next) {
    angle->atom[0]->n_ex_atom++;
    angle->atom[2]->n_ex_atom++;
  }
  */

  /* angle */
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    if (atom->res->noangle) continue;
    /* for TIP3P */
    if (strcmp(atom->top_atom->sym,"HT") == 0) continue;
    /* for H */
    if (atom->top_atom->sym[0] == 'H') continue;
    
    for (i=0;i<atom->n_b_atom;i++) {
      for (j=i+1;j<atom->n_b_atom;j++) {
	atom->b_atom[i]->n_ex_atom++;
	atom->b_atom[j]->n_ex_atom++;
      }
    }
  }

  for (dihedral=mdat->dihedral;dihedral!=NULL;dihedral=dihedral->next) {
    dihedral->atom[0]->n_ex_atom++;
    dihedral->atom[3]->n_ex_atom++;
  }

  /* allocation */
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    if (atom->n_ex_atom!=0)
      atom->ex_atom=emalloc("mdat_make_ex_atom",sizeof(mdat_atom_t*)*atom->n_ex_atom);
    else
      atom->ex_atom=NULL;
    atom->n_ex_atom=0;
  }
  /* making data */
  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    bond->atom[0]->ex_atom[bond->atom[0]->n_ex_atom++] = bond->atom[1];
    bond->atom[1]->ex_atom[bond->atom[1]->n_ex_atom++] = bond->atom[0];
  }

  /*
  for (angle=mdat->angle;angle!=NULL;angle=angle->next) {
    ok = 1;
    for (i=0;i<angle->atom[0]->n_ex_atom;i++) {
      if (angle->atom[0]->ex_atom[i]==angle->atom[2]) {
	ok = 0; break;
      }
    }
    if (ok) {
      angle->atom[0]->ex_atom[angle->atom[0]->n_ex_atom++] = angle->atom[2];
      angle->atom[2]->ex_atom[angle->atom[2]->n_ex_atom++] = angle->atom[0];
    }
  }
  */
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    if (atom->res->noangle) continue;
    /* for TIP3P */
    if (strcmp(atom->top_atom->sym,"HT") == 0) continue;
    /* for H */
    if (atom->top_atom->sym[0] == 'H') continue;
    
    for (i=0;i<atom->n_b_atom;i++) {
      for (j=i+1;j<atom->n_b_atom;j++) {
	ok = 1;
	atom0 = atom->b_atom[i];
	atom2 = atom->b_atom[j];
	for (k=0;k<atom0->n_ex_atom;k++) {
	  if (atom0->ex_atom[k]==atom2) {
	    ok = 0; break;
	  }
	}
	if (ok) {
	  atom0->ex_atom[atom0->n_ex_atom++] = atom2;
	  atom2->ex_atom[atom2->n_ex_atom++] = atom0;
	}
      }
    }
  }


  for (dihedral=mdat->dihedral;dihedral!=NULL;dihedral=dihedral->next) {
    ok = 1;
    for (i=0;i<dihedral->atom[0]->n_ex_atom;i++) {
      if (dihedral->atom[0]->ex_atom[i]==dihedral->atom[3]) {
	ok = 0; break;
      }
    }
    if (ok) {
      dihedral->atom[0]->ex_atom[dihedral->atom[0]->n_ex_atom++] = dihedral->atom[3];
      dihedral->atom[3]->ex_atom[dihedral->atom[3]->n_ex_atom++] = dihedral->atom[0];
      dihedral->omit14 = 0;
    } else {
      dihedral->omit14 = 1;
    }
  }
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    qsort(atom->ex_atom, atom->n_ex_atom, sizeof(mdat_atom_t *),
	  (int (*) (const void *, const void *)) mdat_atom_cmp);
  }
}

int mdat_atom_cmp(const mdat_atom_t **a1, const mdat_atom_t **a2)
{
  if ((*a1)->no > (*a2)->no) return 1;
  if ((*a1)->no < (*a2)->no) return -1;
  if ((*a1)->no == (*a2)->no) {
    printf("ERROR: Internal error: Why equal in atom cmp?? %d\n", (*a1)->no);
    return 0;
  }
  return 0;
}

void mdat_make_angle(mdat_t *mdat)
{
  mdat_angle_t *angle, *angle_prev;
  mdat_atom_t  *atom;
  int i,j;

  mdat->angle = NULL;
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    if (atom->res->noangle) continue;
    
    /* for TIP3P */
    if (strcmp(atom->top_atom->sym,"HT") == 0) continue;
    if (strcmp(atom->top_atom->sym,"HW") == 0) continue;
    /* for H */
    if (atom->top_atom->sym[0] == 'H') {
      if (atom->n_b_atom > 1)
	printf("Warning: Hydrogen %s in %s is the center atom of angle. Omitted.\n",
	       atom->name,atom->res->name
	       );
      continue;
    }
    
    for (i=0;i<atom->n_b_atom;i++) {
      for (j=i+1;j<atom->n_b_atom;j++) {
	angle = emalloc("mdat_make_angle", sizeof(mdat_angle_t));
	angle->atom[0] = atom->b_atom[i];
	angle->atom[1] = atom;
	angle->atom[2] = atom->b_atom[j];
	angle->next = NULL;
	if (mdat->angle == NULL) {
	  mdat->angle = angle;
	  angle->prev = NULL;
	} else {
	  angle_prev->next = angle;
	  angle->prev = angle_prev;
	}
	angle_prev = angle;
      }
    }
  }
}

void mdat_make_dihedral(mdat_t *mdat)
{
  mdat_dihedral_t *dihedral, *dihedral_prev;
  mdat_bond_t *bond;
  mdat_atom_t *atom0,*atom1,*atom2,*atom3;
  int i,j;

  mdat->dihedral = NULL;
  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    if (bond->res->nodihedral) continue;
    atom1=bond->atom[0];
    atom2=bond->atom[1];
    /*
    if (atom1->no > atom2->no) {
      atom0=atom1;
      atom1=atom2;
      atom2=atom0;
    }
    */
    for (i=0;i<atom1->n_b_atom;i++) {
      atom0=atom1->b_atom[i];
      if (atom0==atom2) continue;
      for (j=0;j<atom2->n_b_atom;j++) {
	atom3=atom2->b_atom[j];
	if (atom1==atom3) continue;

	if (atom0==atom3) {
	  printf("Warning: First and Fourth of atoms is same in dihedral Omitted.\n");
	  continue;
	}
	if (atom1->top_atom->sym[0] == 'H' ||
	    atom2->top_atom->sym[0] == 'H') {
	  printf("Warning: Second or Third  atom is Hydrogen. Omitted.\n");
	  continue;
	}
	
	dihedral = emalloc("mdat_make_dihedral", sizeof(mdat_dihedral_t));
	dihedral->atom[0] = atom0;
	dihedral->atom[1] = atom1;
	dihedral->atom[2] = atom2;
	dihedral->atom[3] = atom3;
	dihedral->only14 = 0;
	dihedral->next = NULL;
	if (mdat->dihedral == NULL) {
	  mdat->dihedral = dihedral;
	  dihedral->prev = NULL;
	} else {
	  dihedral_prev->next = dihedral;
	  dihedral->prev = dihedral_prev;
	}
	dihedral_prev = dihedral;
      }
    }
  }
}


void mdat_make_coordinates(mdat_t *mdat)
{
  mdat_atom_t *atom;
  mdat_res_t  *res;
  int miss=0;
  
  for (res=mdat->res;res!=NULL;res=res->next) {
    for (atom=res->beg_atom;atom!=NULL&&atom->prev!=res->end_atom;atom=atom->next) {
      atom->pdb_atom = pdb_search_atom_in_res(atom->name,res->pdb_res);
      atom->charge = atom->top_atom->charge;
      if (atom->pdb_atom) {
	atom->coord = 1;
	atom->pdb_atom->check = 1;
	atom->x = atom->pdb_atom->x;
	atom->y = atom->pdb_atom->y;
	atom->z = atom->pdb_atom->z;
      } else {
	/* if there is no information in pdb ... */
	miss++;
	if (_warning == 1 || atom->name[0] != 'H') 
	  printf("Warning: Atom %-4s in residue %-4s(%4d%c) is missing in pdb file.\n",
		 atom->name,res->name,res->pdb_res->no,res->pdb_res->chain);
      }
    }
  }
  printf("%d atoms are missing in pdb file.\n\n", miss);
}

void mdat_make_coord_from_ic(mdat_t *mdat, par_t *par)
{
  int applied=1, miss;
  mdat_ic_t *ic;
  mdat_atom_t *atom;

  mdat_ic_add(mdat);
  mdat_make_ic_param(mdat, par);
  mdat_apply_ic_seed(mdat);

  while (applied) {
    applied=0;
    for (ic=mdat->ic; ic!=NULL; ic=ic->next) {
      if (ic->atom[0]->coord==0 &&
	  ic->atom[1]->coord==1 &&
	  ic->atom[2]->coord==1 &&
	  ic->atom[3]->coord==1) {
	if (ic->impr == 0)
	  mdat_internal_to_xyz(ic->atom[0], ic->atom[1], ic->atom[2], ic->atom[3],
			       ic->length[0], ic->angle[0], ic->dihedral);
	else
	  mdat_internal_to_xyz(ic->atom[0], ic->atom[2], ic->atom[1], ic->atom[3],
			       ic->length[0], ic->angle[0], -ic->dihedral);
	ic->atom[0]->coord = 1;
	applied=1;
      }
      if (ic->atom[0]->coord==1 &&
	  ic->atom[1]->coord==1 &&
	  ic->atom[2]->coord==1 &&
	  ic->atom[3]->coord==0) {
	if (ic->impr == 0)
	  mdat_internal_to_xyz(ic->atom[3], ic->atom[2], ic->atom[1], ic->atom[0],
			       ic->length[1], ic->angle[1], ic->dihedral);
	else
	  mdat_internal_to_xyz(ic->atom[3], ic->atom[2], ic->atom[1], ic->atom[0],
			       ic->length[1], ic->angle[1], ic->dihedral);

	ic->atom[3]->coord = 1;
	applied=1;
      }
    }
  }

  sol_set_pos_crystal_water(mdat);

  miss=0;
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    if (atom->coord==0) {
      miss++;
      if (_debug)
	printf("Coordinate of atom %s in resiude %s %d%c is not determined.\n",
	       atom->name, atom->res->pdb_res->name,
	       atom->res->pdb_res->no, atom->res->pdb_res->chain);
      
    }
  }
  if (miss) {
    printf("Coordinates of %d atoms are not determined.\n\n", miss);
    exit(1);
  }
  printf("All Coordinates are determined.\n\n");
}

void mdat_ic_add(mdat_t *mdat)
{
  mdat_ic_t *mic;
  cfg_ic_add_t *ic_add;
  mdat_res_t *res;
  mdat_atom_t *atom[4];
  int i;

  for (ic_add=_cfg.ic_add;ic_add!=NULL;ic_add=ic_add->next) {

    for (i=0;i<4;i++) {
      res = mdat_search_res_from_pdbno(mdat, ic_add->resno[i],ic_add->chain[i]);
      if (res==NULL) {
	printf("ERROR: No residue %d%c for ic.\n",
	       ic_add->resno[i], ic_add->chain[i]);
	exit(1);
      }
      atom[i] = mdat_search_atom_in_res(ic_add->atom[i], res);
      if (atom[i]==NULL) {
	printf("ERROR: No atom %s in residue %d%c for ic.\n",
	       ic_add->atom[i], ic_add->resno[i], ic_add->chain[i]);
	exit(1);
      }
    }
    for (mic = mdat->ic; mic!=NULL; mic=mic->next) {
      if ((atom[0] == mic->atom[0] &&
	   atom[1] == mic->atom[1] &&
	   atom[2] == mic->atom[2] &&
	   atom[3] == mic->atom[3]) ||
	  (mic->impr == 0          &&
	   atom[0] == mic->atom[3] &&
	   atom[1] == mic->atom[2] &&
	   atom[2] == mic->atom[1] &&
	   atom[3] == mic->atom[0])) {
	for (i=0;i<2;i++) {
	  mic->length[i] = ic_add->length[i];
	  mic->angle[i]  = ic_add->angle[i];
	}
	mic->dihedral = ic_add->dihedral;
	mic->impr = ic_add->impr;
	break;
      }
    }
    if (mic==NULL) {
      mic = emalloc("mdat_add_ic", sizeof(mdat_ic_t));
      for (i=0;i<4;i++)
	mic->atom[i] = atom[i];
      for (i=0;i<2;i++) {
	mic->length[i] = ic_add->length[i];
	mic->angle[i]  = ic_add->angle[i];
      }
      mic->dihedral = ic_add->dihedral;
      mic->impr = ic_add->impr;
      mic->next = mdat->ic;
      mdat->ic = mic;
    }
  }
}

void mdat_make_ic_param(mdat_t *mdat, par_t *par)
{
  mdat_ic_t *ic;
  par_bond_t *pbond;
  par_angle_t *pangle;
  int aid[2][3],i;

  for (ic=mdat->ic; ic!=NULL; ic=ic->next) {
    if (ic->impr==0) {
      aid[0][0] = 0; aid[0][1] = 1; aid[0][2] = 2;
      aid[1][0] = 3; aid[1][1] = 2; aid[1][2] = 1;
    } else {
      aid[0][0] = 0; aid[0][1] = 2; aid[0][2] = 1;
      aid[1][0] = 3; aid[1][1] = 2; aid[1][2] = 1;
    }
      
    for (i=0;i<2;i++) {
      if (ic->length[i]==0.0) {
	pbond=par_search_bond(par,
			      ic->atom[aid[i][0]]->top_atom->sym,
			      ic->atom[aid[i][1]]->top_atom->sym);
	if (pbond==NULL) {
	  /*
	  printf("ERROR: Could not find bond parameters %s-%s for generation of optimal internal coordinates\n",
		 ic->atom[aid[i][0]]->top_atom->sym,ic->atom[aid[i][1]]->top_atom->sym);
	  */
	  printf("ERROR: No bond parameter %s-%s for internal coordinate %s-%s\n\n",
		 ic->atom[aid[i][0]]->top_atom->sym,
		 ic->atom[aid[i][1]]->top_atom->sym,
		 ic->atom[aid[i][0]]->top_atom->name,
		 ic->atom[aid[i][1]]->top_atom->name);
	  exit(1);
	}
	ic->length[i]=pbond->b0;
      }
      
      if (ic->angle[i]==0.0) {
	pangle=par_search_angle(par,
				ic->atom[aid[i][0]]->top_atom->sym,
				ic->atom[aid[i][1]]->top_atom->sym,
				ic->atom[aid[i][2]]->top_atom->sym);
	if (pangle==NULL) {
	  /*
	  printf("ERROR: Could not find angle parameters %s-%s-%s for generation of optimal internal coordinates\n\n",
		 ic->atom[aid[i][0]]->top_atom->sym,
		 ic->atom[aid[i][1]]->top_atom->sym,
		 ic->atom[aid[i][2]]->top_atom->sym);
	  */
	  printf("ERROR: No angle parameter %s-%s-%s for internal coordinate %s-%s-%s\n\n",
		 ic->atom[aid[i][0]]->top_atom->sym,
		 ic->atom[aid[i][1]]->top_atom->sym,
		 ic->atom[aid[i][2]]->top_atom->sym,
		 ic->atom[aid[i][0]]->top_atom->name,
		 ic->atom[aid[i][1]]->top_atom->name,
		 ic->atom[aid[i][2]]->top_atom->name);
	  exit(1);
	}
	ic->angle[i]=pangle->Theta0;
      }
    }
  }

  if (_debug)
    for (ic=mdat->ic; ic!=NULL; ic=ic->next) {
      printf("IC %d %-4s %d %-4s %d %-4s %d %-4s %.4f %.3f %8.3f %.3f %.4f\n",
	     ic->atom[0]->res->no,ic->atom[0]->name,
	     ic->atom[1]->res->no,ic->atom[1]->name,
	     ic->atom[2]->res->no,ic->atom[2]->name,
	     ic->atom[3]->res->no,ic->atom[3]->name,
	     ic->length[0],ic->angle[0],ic->dihedral,ic->angle[1],ic->length[1]);
    }
}

void mdat_apply_ic_seed(mdat_t *mdat)
{
  cfg_ic_seed_t *seed;
  mdat_res_t *res;
  mdat_atom_t *atom[3];
  vec_t p, p0, p1, v, v1, axis;
  double length, angle;
  int i;

  for (seed = _cfg.ic_seed; seed!=NULL; seed=seed->next) {
    for (i=0;i<3;i++) {
      res = mdat_search_res_from_pdbno(mdat, seed->resno[i],seed->chain[i]);
      if (res==NULL) {
	printf("ERROR: No residue %d%c for ic seed.\n",
	       seed->resno[i], seed->chain[i]);
	exit(1);
      }
      atom[i] = mdat_search_atom_in_res(seed->atom[i], res);
      if (atom[i]==NULL) {
	printf("ERROR: No atom %s in residue %d%c for ic seed.\n",
	       seed->atom[i], seed->resno[i], seed->chain[i]);
	exit(1);
      }
    }
    /* first atom */
    atom[0]->x = atom[0]->y = atom[0]->z = 0.0;
    atom[0]->coord = 1;
    
    /* second atom */
    length = mdat_search_length_from_ic(mdat, atom);
    if (length == 0.0) {
      printf("ERROR: No length value between atoms %s(%d%c)-%s(%d%c).\n",
	     atom[0]->name, atom[0]->res->pdb_res->no,atom[0]->res->pdb_res->chain,
	     atom[1]->name, atom[1]->res->pdb_res->no,atom[1]->res->pdb_res->chain);
      exit(1);
    }
    atom[1]->x = length;
    atom[1]->y = atom[1]->z = 0.0;
    atom[1]->coord = 1;

    /* third atom */
    length = mdat_search_length_from_ic(mdat, &atom[1]);
    if (length == 0.0) {
      printf("ERROR: No length value between atoms %s(%d%c)-%s(%d%c).\n",
	     atom[1]->name, atom[1]->res->pdb_res->no,atom[1]->res->pdb_res->chain,
	     atom[2]->name, atom[2]->res->pdb_res->no,atom[2]->res->pdb_res->chain);
      exit(1);
    }
    
    angle = mdat_search_angle_from_ic(mdat, atom);
    if (angle == 0.0) {
      printf("ERROR: No angle value between atoms %s(%d%c)-%s(%d%c)-%s(%d%c).\n",
	     atom[0]->name, atom[0]->res->pdb_res->no,atom[0]->res->pdb_res->chain,
	     atom[1]->name, atom[1]->res->pdb_res->no,atom[1]->res->pdb_res->chain,
	     atom[2]->name, atom[2]->res->pdb_res->no,atom[2]->res->pdb_res->chain);
      exit(1);
    }
    
    p0.x = atom[1]->x; p0.y = atom[1]->y; p0.z = atom[1]->z;
    p1.x = atom[0]->x; p1.y = atom[0]->y; p1.z = atom[0]->z;
    v_sub(&v1, &p1, &p0);
    v_norm(&v1);
    v_mul(&v, length, &v1);
    axis.x = 0.0; axis.y = 0.0; axis.z = -1.0;
    angle *= M_PI / 180.0;
    v_rot(&v, &axis, angle, &v);
    v_add(&p, &p0, &v);
    atom[2]->x = p.x; atom[2]->y = p.y; atom[2]->z = p.z;
    atom[2]->coord = 1;
  }
}

double mdat_search_length_from_ic(mdat_t *mdat, mdat_atom_t *atom[2])
{
  mdat_ic_t *ic;
  for (ic=mdat->ic;ic!=NULL;ic=ic->next) {
    if (ic->impr==0) {
      if (ic->atom[0]==atom[0] && ic->atom[1]==atom[1] ||
	  ic->atom[1]==atom[0] && ic->atom[0]==atom[1])
	return ic->length[0];
    } else {
      if (ic->atom[0]==atom[0] && ic->atom[2]==atom[1] ||
	  ic->atom[2]==atom[0] && ic->atom[0]==atom[1])
	return ic->length[0];
    }
    if (ic->atom[2]==atom[0] && ic->atom[3]==atom[1] ||
	ic->atom[3]==atom[0] && ic->atom[2]==atom[1])
      return ic->length[1];
  }
  return 0.0;
}

double mdat_search_angle_from_ic(mdat_t *mdat, mdat_atom_t *atom[3])
{
  mdat_ic_t *ic;
  for (ic=mdat->ic;ic!=NULL;ic=ic->next) {
    if (ic->impr==0) {
      if (ic->atom[0]==atom[0] && ic->atom[1]==atom[1] && ic->atom[2]==atom[2] ||
	  ic->atom[2]==atom[0] && ic->atom[1]==atom[1] && ic->atom[0]==atom[2])
	return ic->angle[0];
    } else {
      if (ic->atom[0]==atom[0] && ic->atom[2]==atom[1] && ic->atom[1]==atom[2] ||
	  ic->atom[1]==atom[0] && ic->atom[2]==atom[1] && ic->atom[0]==atom[2])
	return ic->angle[0];
    }
    if (ic->atom[3]==atom[0] && ic->atom[2]==atom[1] && ic->atom[1]==atom[2] ||
	ic->atom[1]==atom[0] && ic->atom[2]==atom[1] && ic->atom[3]==atom[2])
      return ic->angle[1];
  }
  return 0.0;
}

void mdat_internal_to_xyz(mdat_atom_t *atom,  mdat_atom_t *atom0,
			  mdat_atom_t *atom1, mdat_atom_t *atom2,
			  double length, double angle, double dihedral)
{
  vec_t p, p0, p1, p2, v, v1, v2, axis;

  angle    *= M_PI / 180.0;
  dihedral *= M_PI / 180.0;

  p0.x= atom0->x; p0.y= atom0->y; p0.z= atom0->z;
  p1.x= atom1->x; p1.y= atom1->y; p1.z= atom1->z;
  p2.x= atom2->x; p2.y= atom2->y; p2.z= atom2->z;

  v_sub(&v1, &p1, &p0);
  v_sub(&v2, &p2, &p0);
  v_norm(&v1);
  v_mul(&v, length, &v1);
  v_outer_pro(&axis, &v1, &v2);
  v_rot(&v, &axis, angle, &v);
  v_rot(&v, &v1,  -dihedral, &v);
  v_add(&p, &p0, &v);
  
  atom->x = p.x; atom->y = p.y; atom->z = p.z;
}
  
  
mdat_atom_t *mdat_search_atom_in_res(char *name, mdat_res_t *mres)
{
  mdat_atom_t *matom;
  
  if (name[0]=='+') {
    if (mres->last) return NULL;
    name++;
    mres=mres->next;
  } else if (name[0]=='-') {
    if (mres->first) return NULL;
    name++;
    mres=mres->prev;
  } else if (name[0]=='*') {
    name++;
  }

  if (mres==NULL) return NULL;
  
  for (matom=mres->beg_atom;matom!=NULL&&matom->prev!=mres->end_atom;matom=matom->next) {
    if (strcmp(name, matom->name) == 0) {
      return matom;
    }
  }

  return NULL;
}

mdat_atom_t *mdat_search_atom_in_res2(char *name, mdat_res_t *mres2[3], int *i)
{
  mdat_atom_t *matom;
  mdat_res_t *mres;
  
  if (name[0]=='*') {
    name++;
  }

  if (name[0]=='1') {
    *i = 0;
    name++;
  } else if (name[0]=='2') {
    *i = 1;
    if (mres2[1] == NULL) {
      printf("ERROR: Searching atom %s but no second residue is specified.\n", name);
      exit(1);
    }
    name++;
  } else if (name[0]=='3') {
    *i = 2;
    if (mres2[2] == NULL) {
      printf("ERROR: Searching atom %s but no third residue is specified.\n", name);
      exit(1);
    }
    name++;
  } else {
    *i = 0;
  }
  mres = mres2[*i];

  if (name[0]=='+') {
    if (mres->last) return NULL;
    name++;
    mres=mres->next;
  } else if (name[0]=='-') {
    if (mres->first) return NULL;
    name++;
    mres=mres->prev;
  } else if (name[0]=='*') {
    name++;
  }

  if (mres==NULL) return NULL;
  
  for (matom=mres->beg_atom;matom!=NULL&&matom->prev!=mres->end_atom;matom=matom->next) {
    if (strcmp(name, matom->name) == 0) {
      return matom;
    }
  }

  return NULL;
}


mdat_res_t *mdat_search_res_from_pdbno(mdat_t *mdat, int resno, char chain)
{
  mdat_res_t *res;

  for (res=mdat->res;res!=NULL;res=res->next) {
    if (res->pdb_res->no==resno && res->pdb_res->chain==chain)
      return res;
  }
  return NULL;
}

void mdat_add_mass(mdat_t *mdat, top_t *top)
{
  mdat_atom_t *atom;
  top_mass_t *mass;

  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    mass = top_search_mass(top, atom->top_atom->sym);
    if (mass==NULL) {
      printf("ERROR: No atom mass %s\n", atom->top_atom->sym);
      exit(1);
    }
    atom->mass = mass;
    mass->check = 1;
  }
}

void mdat_add_par(mdat_t *mdat, par_t *par)
{
  mdat_bond_t *bond;
  mdat_angle_t *angle;
  mdat_dihedral_t *dihedral;
  mdat_impr_t *impr;
  mdat_cmap_t *cmap;
  mdat_atom_t *atom;
  par_dihedral_t *pd;
  int ok = 1;
  int n_term;

  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    bond->par_bond = par_search_bond(par, bond->atom[0]->top_atom->sym,
				     bond->atom[1]->top_atom->sym);
    if (bond->par_bond==NULL) {
      printf("ERROR: Missing bond parameter %s-%s for %s.%s-%s.%s\n",
	     bond->atom[0]->top_atom->sym,
	     bond->atom[1]->top_atom->sym,
	     bond->atom[0]->name, bond->atom[0]->res->name,
	     bond->atom[1]->name, bond->atom[1]->res->name);
      ok=0;
    } else
      bond->par_bond->check = 1;
  }
  
  for (angle=mdat->angle;angle!=NULL;angle=angle->next) {
    angle->par_angle = 
      par_search_angle(par,
		       angle->atom[0]->top_atom->sym,
		       angle->atom[1]->top_atom->sym,
		       angle->atom[2]->top_atom->sym);
    if (angle->par_angle==NULL) {
      printf("ERROR: Missing angle parameter %s-%s-%s for %s.%s-%s.%s-%s.%s\n",
	     angle->atom[0]->top_atom->sym,
	     angle->atom[1]->top_atom->sym,
	     angle->atom[2]->top_atom->sym, 
	     angle->atom[0]->name, angle->atom[0]->res->name,
	     angle->atom[1]->name, angle->atom[1]->res->name,
	     angle->atom[2]->name, angle->atom[2]->res->name);
      ok=0;
    } else
      angle->par_angle->check = 1;
  }
  
  for (dihedral=mdat->dihedral;dihedral!=NULL;dihedral=dihedral->next) {
    n_term = 0;
    if (dihedral->only14) {
      dihedral->n_par_dihedral = 1;
      dihedral->par_dihedral = NULL;
      continue;
    }
    for (pd=par->dihedral;pd!=NULL;pd=pd->next) {
      if (par_match_dihedral(pd,
			     dihedral->atom[0]->top_atom->sym,
			     dihedral->atom[1]->top_atom->sym,
			     dihedral->atom[2]->top_atom->sym,
			     dihedral->atom[3]->top_atom->sym) == 0) {
	n_term++;
      }
    }
    if (n_term > 0) {
      dihedral->par_dihedral = emalloc("mdat_add_par", sizeof(struct par_dihedral *)*n_term);
      dihedral->n_par_dihedral = n_term;
      n_term = 0;
      for (pd=par->dihedral;pd!=NULL;pd=pd->next) {
	if (par_match_dihedral(pd,
			       dihedral->atom[0]->top_atom->sym,
			       dihedral->atom[1]->top_atom->sym,
			       dihedral->atom[2]->top_atom->sym,
			       dihedral->atom[3]->top_atom->sym) == 0) {
	  dihedral->par_dihedral[n_term] = pd;
	  pd->check = 1;
	  n_term++;
	}
      }
      continue;
    }
    /* case of n_term == 0  test wild card*/
    n_term = 0;
    for (pd=par->dihedral;pd!=NULL;pd=pd->next) {
      if (par_match_dihedral_wild_card(pd,
		             dihedral->atom[0]->top_atom->sym,
			     dihedral->atom[1]->top_atom->sym,
			     dihedral->atom[2]->top_atom->sym,
			     dihedral->atom[3]->top_atom->sym) == 0) {
	n_term++;
      }
    }
    if (n_term > 0) {
      dihedral->par_dihedral = emalloc("mdat_add_par", sizeof(struct par_dihedral *)*n_term);
      dihedral->n_par_dihedral = n_term;
      n_term = 0;
      for (pd=par->dihedral;pd!=NULL;pd=pd->next) {
	if (par_match_dihedral_wild_card(pd,
			       dihedral->atom[0]->top_atom->sym,
			       dihedral->atom[1]->top_atom->sym,
			       dihedral->atom[2]->top_atom->sym,
			       dihedral->atom[3]->top_atom->sym) == 0) {
	  dihedral->par_dihedral[n_term] = pd;
	  pd->check = 1;
	  n_term++;
	}
      }
      continue;
    }
    printf("ERROR: Missing dihedral parameter %s-%s-%s-%s\n",
	   dihedral->atom[0]->top_atom->sym,
	   dihedral->atom[1]->top_atom->sym,
	   dihedral->atom[2]->top_atom->sym,
	   dihedral->atom[3]->top_atom->sym);
    ok=0;
  }  /* for dihedral */
  /* end of dihedral */

  for (impr=mdat->impr;impr!=NULL;impr=impr->next) {
    impr->par_impr = 
      par_search_impr(par,
		      impr->atom[0]->top_atom->sym,
		      impr->atom[1]->top_atom->sym,
		      impr->atom[2]->top_atom->sym,
		      impr->atom[3]->top_atom->sym);
    if (impr->par_impr == NULL) {
      impr->par_impr = 
	par_search_impr_wild_card(par,
				  impr->atom[0]->top_atom->sym,
				  impr->atom[1]->top_atom->sym,
				  impr->atom[2]->top_atom->sym,
				  impr->atom[3]->top_atom->sym);
    }
    if (impr->par_impr==NULL) {
      printf("ERROR: Missing impr parameter %s-%s-%s-%s\n",
	     impr->atom[0]->top_atom->sym,
	     impr->atom[1]->top_atom->sym,
	     impr->atom[2]->top_atom->sym,
	     impr->atom[3]->top_atom->sym);
      ok=0;
    } else
      impr->par_impr->check = 1;
  }

  /* for cmap */
  for (cmap=mdat->cmap;cmap!=NULL;cmap=cmap->next) {
    cmap->par_cmap = 
      par_search_cmap(par,
		      cmap->atom[0]->top_atom->sym,
		      cmap->atom[1]->top_atom->sym,
		      cmap->atom[2]->top_atom->sym,
		      cmap->atom[3]->top_atom->sym,
		      cmap->atom[4]->top_atom->sym,
		      cmap->atom[5]->top_atom->sym,
		      cmap->atom[6]->top_atom->sym,
		      cmap->atom[7]->top_atom->sym);
    if (cmap->par_cmap==NULL) {
      printf("ERROR: Missing cmap parameter %s-%s-%s-%s %s-%s-%s-%s\n",
	     cmap->atom[0]->top_atom->sym,
	     cmap->atom[1]->top_atom->sym,
	     cmap->atom[2]->top_atom->sym,
	     cmap->atom[3]->top_atom->sym,
	     cmap->atom[4]->top_atom->sym,
	     cmap->atom[5]->top_atom->sym,
	     cmap->atom[6]->top_atom->sym,
	     cmap->atom[7]->top_atom->sym);
      ok=0;
    } else
      cmap->par_cmap->check = 1;
  }
  
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    atom->par_nonbonded = 
      par_search_nonbonded(par,atom->top_atom->sym);

    if (atom->par_nonbonded==NULL) {
      printf("ERROR: Missing atom nonbonded parameter %s\n",
	     atom->top_atom->sym);
      ok=0;
    } else
      atom->par_nonbonded->check = 1;
  }
  
  if (!ok)
    exit(1);
}

void mdat_add_par_nonbonded(mdat_t *mdat, par_t *par)
{
  mdat_atom_t *atom;
  int ok = 1;
  
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    atom->par_nonbonded = 
      par_search_nonbonded(par,atom->top_atom->sym);

    if (atom->par_nonbonded==NULL) {
      printf("ERROR: Missing atom nonbonded parameter %s\n",
	     atom->top_atom->sym);
      ok=0;
    } else
      atom->par_nonbonded->check = 1;
  }

  if (!ok)
    exit(1);
}

void mdat_count_all(mdat_t *mdat)
{
  mdat_count_atom(mdat);
  mdat_count_bond(mdat);
  mdat_count_angle(mdat);
  mdat_count_dihedral(mdat);
  mdat_count_impr(mdat);
  mdat_count_cmap(mdat);
}

void mdat_count_atom(mdat_t *mdat)
{
  int n;
  mdat_atom_t *atom;
  mdat_res_t *res;
  mdat_mol_t *mol;
  double charge = 0.0;

  n=0;
  for (mol=mdat->mol;mol!=NULL;mol=mol->next) {
    mol->no     = n++;
    mol->n_atom = 0;
  }
  mdat->n_mol = n;
  
  n=0;
  for (res=mdat->res;res!=NULL;res=res->next) {
    res->no     = n++;
    res->n_atom = 0;
  }
  mdat->n_res = n;

  n=0;
  for (atom=mdat->atom;atom!=NULL;atom=atom->next) {
    atom->no = n++;
    atom->res->n_atom++;
    atom->res->mol->n_atom++;
    charge += atom->charge;
  }
  mdat->n_atom = n;
  mdat->charge = charge;
}

void mdat_count_bond(mdat_t *mdat)
{
  int n=0;
  mdat_bond_t *bond;

  for (bond=mdat->bond;bond!=NULL;bond=bond->next) {
    bond->no = n++;
  }
  mdat->n_bond = n;
}

void mdat_count_angle(mdat_t *mdat)
{
  int n=0;
  mdat_angle_t *angle;

  for (angle=mdat->angle;angle!=NULL;angle=angle->next) {
    angle->no = n++;
  }
  mdat->n_angle = n;
}

void mdat_count_dihedral(mdat_t *mdat)
{
  int n=0, n_term=0;
  mdat_dihedral_t *dihedral;
  par_dihedral_t *pdihedral;

  for (dihedral=mdat->dihedral;dihedral!=NULL;dihedral=dihedral->next) {
    if (dihedral->only14) {
      dihedral->no = -1;
      n_term++;
    } else {
      dihedral->no = n++;
      n_term += dihedral->n_par_dihedral;
    }
    /* change by ike on 2002/10/26 */
    /*
    pdihedral = dihedral->par_dihedral;
    while (pdihedral) {
      n_term++;
      if (pdihedral->multi==0) break;
      pdihedral = pdihedral->next;
    }
    */
  }
  mdat->n_dihedral_term = n_term;
  mdat->n_dihedral      = n;
}

void mdat_count_impr(mdat_t *mdat)
{
  int n=0;
  mdat_impr_t *impr;

  for (impr=mdat->impr;impr!=NULL;impr=impr->next) {
    impr->no = n++;
  }
  mdat->n_impr = n;
}

void mdat_count_cmap(mdat_t *mdat)
{
  int n=0;
  mdat_cmap_t *cmap;

  for (cmap=mdat->cmap;cmap!=NULL;cmap=cmap->next) {
    cmap->no = n++;
  }
  mdat->n_cmap = n;
}

void mdat_set_solute(mdat_t *mdat)
{
  int i,j;
  mdat_mol_t *mol, *last_solute;
  mdat_res_t *res;

  last_solute = mdat->mol;
  for (res=mdat->res;res!=NULL;res=res->next) {
    if (res->flag == MDAT_SOLUTE) {
      /* printf("sol:%d\n",res->mol->no); */
      last_solute = res->mol;
    }
  }
  if (last_solute)
    mdat->n_solute_mol = last_solute->no + 1;
  else
    mdat->n_solute_mol = 0;
}

void mdat_print_info(mdat_t *mdat, par_t *par)
{
  printf("\n");
  printf("Molecular Data (mdat) Information:\n");
  printf("  Number of atoms: %d\n", mdat->n_atom);
  printf("  Number of atom types: %d\n", par->n_used_nonbonded);
  printf("  Number of residues: %d\n", mdat->n_res);
  printf("  Number of molecules: %d\n", mdat->n_mol);
  printf("  Number of bonds: %d\n", mdat->n_bond);
  printf("  Number of bond types: %d\n", par->n_used_bond);
  printf("  Number of angles: %d\n", mdat->n_angle);
  printf("  Number of angle types: %d\n", par->n_used_angle);
  printf("  Number of dihedrals: %d (term: %d)\n", mdat->n_dihedral,
	 mdat->n_dihedral_term);
  printf("  Number of dihedral types: %d\n", par->n_used_dihedral);
  printf("  Number of impropers: %d\n", mdat->n_impr);
  printf("  Number of improper types: %d\n", par->n_used_impr);
  printf("  Number of cmap terms: %d\n", mdat->n_cmap);
  printf("  Number of cmap types: %d\n", par->n_used_cmap);
  printf("  Number of solute molecules: %d\n", mdat->n_solute_mol);
  printf("  Total charge: %f\n", mdat->charge);
  
  if (mdat->ifbox) {
    int i;
    printf("  Periodic Boundary Box:\n");
    for (i=0;i<3;i++) {
      printf("      %6.2f %6.2f %6.2f\n",
	     mdat->boxv[i][0],mdat->boxv[i][1],mdat->boxv[i][2]);
    }
  }
}

void mdat_set_boxv(mdat_t *mdat, double f1, double f2, double f3,
		   double alpha, double beta, double gamma)
{
  int i,j;
  
  for (i=0;i<3;i++)
    for (j=0;j<3;j++)
      mdat->boxv[i][j] = 0.0;

  if (alpha == 90.0 && beta == 90.0 && gamma == 90.0) {
    mdat->boxv[0][0] = f1;
    mdat->boxv[1][1] = f2;
    mdat->boxv[2][2] = f3;
    
  } else {
    mdat->boxv[0][0] = f1;
    mdat->boxv[1][0] = f2*cos(gamma*M_PI/180.0);
    mdat->boxv[1][1] = f2*sin(gamma*M_PI/180.0);
    mdat->boxv[2][0] = f3*cos(beta*M_PI/180.0);
    mdat->boxv[2][1] = (f2*f3*cos(alpha*M_PI/180.0)
			-mdat->boxv[2][0]*mdat->boxv[1][0])/mdat->boxv[1][1];
    mdat->boxv[2][2] = sqrt(f3*f3-SQR(mdat->boxv[2][0])-SQR(mdat->boxv[2][1]));
  }

  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      if (fabs(mdat->boxv[i][j]) < 1.0e-12)
	mdat->boxv[i][j] = 0.0;
    }
  }
  mdat->ifbox = 1;
}


void mdat_merge(mdat_t *dest, mdat_t *src)
{
  mdat_atom_t *atom;
  mdat_res_t  *res;
  mdat_mol_t  *mol;
  mdat_bond_t *bond;
  mdat_angle_t *angle;
  mdat_dihedral_t *dihedral;
  mdat_impr_t *impr;
  mdat_ic_t *ic;

  if (dest->atom==NULL) {
    dest->atom = src->atom;
  } else {
    for (atom=dest->atom;atom->next!=NULL;atom=atom->next);
    atom->next = src->atom;
    if (src->atom)
      src->atom->prev  = atom;
  }

  if (dest->res==NULL) {
    dest->res = src->res;
  } else {
    for (res=dest->res;res->next!=NULL;res=res->next);
    res->next = src->res;
    if (src->res)
      src->res->prev  = res;
  }

  if (dest->mol==NULL) {
    dest->mol = src->mol;
  } else {
    for (mol=dest->mol;mol->next!=NULL;mol=mol->next);
    mol->next = src->mol;
    if (src->mol)
      src->mol->prev  = mol;
  }

  if (dest->bond==NULL) {
    dest->bond = src->bond;
  } else {
    for (bond=dest->bond;bond->next!=NULL;bond=bond->next);
    bond->next = src->bond;
    if (src->bond)
      src->bond->prev  = bond;
  }

  if (dest->angle==NULL) {
    dest->angle = src->angle;
  } else {
    for (angle=dest->angle;angle->next!=NULL;angle=angle->next);
    angle->next = src->angle;
    if (src->angle)
      src->angle->prev  = angle;
  }

  if (dest->dihedral==NULL) {
    dest->dihedral = src->dihedral;
  } else {
    for (dihedral=dest->dihedral;dihedral->next!=NULL;dihedral=dihedral->next);
    dihedral->next = src->dihedral;
    if (src->dihedral)
      src->dihedral->prev  = dihedral;
  }

  if (dest->impr==NULL) {
    dest->impr = src->impr;
  } else {
    for (impr=dest->impr;impr->next!=NULL;impr=impr->next);
    impr->next = src->impr;
    if (src->impr)
      src->impr->prev  = impr;
  }

  if (dest->ic==NULL) {
    dest->ic = src->ic;
  } else {
    for (ic=dest->ic;ic->next!=NULL;ic=ic->next);
    ic->next = src->ic;
    if (src->ic)
      src->ic->prev  = ic;
  }
}

void mdat_set_radii(mdat_t *mdat)
{
  mdat_atom_t *a;
  double coef;

  if (_cfg.use_vdw_radii) {
    mdat_set_radii_orig(mdat);
    return;
  }

  coef = pow(2.0, -1.0/6.0);

  mdat_add_par_nonbonded(mdat, &_par);

  for (a=mdat->atom;a!=NULL;a=a->next) {
    if (a->par_nonbonded == NULL) {
      printf("ERROR: A nonbonded parameter is not assigned for %s\n",
	     a->top_atom->sym);
    }
    a->r = a->par_nonbonded->rmin2 * coef;  /* lj 0 point */
  }
}


void mdat_set_radii_orig(mdat_t *mdat)
{
  mdat_atom_t *a;

  /* Bondi Radii */
  for (a=mdat->atom;a!=NULL;a=a->next) {
    if (strcmp(a->mass->type,"C") == 0) {
      a->r = 1.7;
    } else if (strcmp(a->mass->type,"O") == 0) {
      a->r = 1.52;
    } else if (strcmp(a->mass->type,"N") == 0) {
      a->r = 1.55;
    } else if (strcmp(a->mass->type,"H") == 0) {
      a->r = 1.2;
    } else if (strcmp(a->mass->type,"S") == 0) {
      a->r = 1.8;
    } else if (strcmp(a->mass->type,"P") == 0) {
      a->r = 1.8;
    } else if (strcmp(a->mass->type,"NA") == 0) {
      a->r = 0.95;  /* pauling ion radii */
    } else if (strcmp(a->mass->type,"K") == 0) {
      a->r = 1.33;  /* pauling ion radii */
    } else if (strcmp(a->mass->type,"CL") == 0) {
      a->r = 1.81;  /* pauling ion radii */
    } else if (strcmp(a->mass->type,"MG") == 0) {
      a->r = 0.65;  /* pauling ion radii */
    } else if (strcmp(a->mass->type,"CA") == 0) {
      a->r = 0.99;  /* pauling ion radii */
    } else {
      printf("ERROR: Van der Waals radius of %s not assigned\n", a->mass->type);
      exit(1);
    }
  }
}

mdat_atom_t *mdat_search_atom(mdat_t *pdb, char *atom, char *res, int res_no, char chain)
{
  mdat_atom_t *a, *prev;
  mdat_res_t  *r;
  
  for (r=pdb->res;r!=NULL;r=r->next) {
    if (r->pdb_res &&
	r->pdb_res->no == res_no &&
	r->pdb_res->chain == chain) {
      for (a=r->beg_atom,prev=NULL;prev!=r->end_atom;prev=a,a=a->next) {
	if (strcmp(atom,a->name) == 0) return a;
      }
    }
  }
  return NULL;
}


void mdat_renumber_residue(mdat_t *mdat)
{
  mdat_res_t *r;
  int no;

  no = 1;
  for (r=mdat->res;r!=NULL;r=r->next) {
    r->pdb_no = no++;
    r->pdb_chain = ' ';
  }
}

void mdat_check_res_no(mdat_t *mdat)
{
  mdat_res_t *r1, *r2;
  /*

  for (r1=mdat->res;r1!=NULL;r1=r1->next) {
    for (r2=r1->next;r2!=NULL;r2=r2->next) {
      if (r1->pdb_no == r2->pdb_no &&
	  r1->pdb_chain == r2->pdb_chain) {
	printf("ERROR: The same pdb residue number and chain id at serapated positions: %d (chain:%c)\n", r1->pdb_no, r1->pdb_chain);
	exit(1);
      }
    }
  }
  */

  for (r1=mdat->res;r1!=NULL;r1=r1->next) {
    r2 = r1->next;
    if (r2!=NULL) {
      if ((r1->pdb_chain >  r2->pdb_chain && r2->pdb_chain != ' ') ||
	  (r1->pdb_chain == ' '           && r2->pdb_chain != ' ') ||
	  (r1->pdb_chain == r2->pdb_chain && r1->pdb_no >= r2->pdb_no)) {
	printf("ERROR: Residue number and chain id must be in descending order:\n");
	printf("       %d (chain:%c) -> %d (chain:%c)\n", 
	       r1->pdb_no, r1->pdb_chain,
	       r2->pdb_no, r2->pdb_chain);
	if (r1->pdb_chain == ' ' && r2->pdb_chain != ' ') {
	  printf("       Chain ' ' must be placed last.\n");
	}
	exit(1);
      }
    }
  }    
}


#define VEC_MUL_MAT_X(a,m)  (a.x*m[0][0]+a.y*m[1][0]+a.z*m[2][0])
#define VEC_MUL_MAT_Y(a,m)  (a.x*m[0][1]+a.y*m[1][1]+a.z*m[2][1])
#define VEC_MUL_MAT_Z(a,m)  (a.x*m[0][2]+a.y*m[1][2]+a.z*m[2][2])
#define MAT_MUL_VEC_X(m,a)  (m[0][0]*a.x+m[0][1]*a.y+m[0][2]*a.z)
#define MAT_MUL_VEC_Y(m,a)  (m[1][0]*a.x+m[1][1]*a.y+m[1][2]*a.z)
#define MAT_MUL_VEC_Z(m,a)  (m[2][0]*a.x+m[2][1]*a.y+m[2][2]*a.z)

void mdat_wrap_molecules(mdat_t *mdat)
{
  mdat_mol_t *mol;
  mdat_res_t *res;
  mdat_atom_t *atom;
  vec_t center, frac, offset;
  double w;

  mdat_make_recip(mdat);

  for (mol=mdat->mol;mol!=NULL;mol=mol->next) {
    center.x = center.y = center.z = w = 0.0;
    for (res=mol->beg_res;res!=NULL;res=res->next) {
      for (atom=res->beg_atom;atom!=NULL;atom=atom->next) {
	center.x += atom->x;
	center.y += atom->y;
	center.z += atom->z;
	w += atom->mass->weight;
	if (atom==res->end_atom) break;
      }
      if (res==mol->end_res) break;
    }
    if (w==0.0) {
      printf("ERROR: no atom in a molecule\n");
      continue;
    }
    center.x /= w;
    center.y /= w;
    center.z /= w;

    /*
    printf("%f %f %f\n", center.x, center.y, center.z);
    */

    frac.x = VEC_MUL_MAT_X(center,mdat->recip);
    frac.y = VEC_MUL_MAT_Y(center,mdat->recip);
    frac.z = VEC_MUL_MAT_Z(center,mdat->recip);

    /*
    if (mdat->origin_flag == 1) {
      frac.x += 0.5;
      frac.y += 0.5;
      frac.z += 0.5;
    }
    */
    /*
    printf("%f %f %f\n", frac.x, frac.y, frac.z);
    */

    frac.x = floor(frac.x);
    frac.y = floor(frac.y);
    frac.z = floor(frac.z);

    
    offset.x = VEC_MUL_MAT_X(frac,mdat->boxv);
    offset.y = VEC_MUL_MAT_Y(frac,mdat->boxv);
    offset.z = VEC_MUL_MAT_Z(frac,mdat->boxv);

    
    if (offset.x == 0.0 && offset.y == 0.0 && offset.z == 0.0)
      continue;

    /*
    printf("offset %f %f %f\n", offset.x, offset.y, offset.z);
    */
    
    for (res=mol->beg_res;res!=NULL;res=res->next) {
      for (atom=res->beg_atom;atom!=NULL;atom=atom->next) {
	atom->x -= offset.x;
	atom->y -= offset.y;
	atom->z -= offset.z;
	if (atom==res->end_atom) break;
      }
      if (res==mol->end_res) break;
    }
  }
}

void mdat_make_recip(mdat_t *mdat)
{
  int i, j;
  double v01[3], v12[3], v20[3], tmp, V;

  cross3(v12, mdat->boxv[1],mdat->boxv[2]);
  cross3(v20, mdat->boxv[2],mdat->boxv[0]);
  cross3(v01, mdat->boxv[0],mdat->boxv[1]);
  V = dot3(mdat->boxv[0],v12);
  /* mdat->A = mdat->boxv[0][0]*mdat->boxv[1][1]; */

  /*
  printf("boxv\n");
  print_mat33(mdat->boxv);
  printf("end of boxv\n");
  */

  for (j=0;j<3;j++) {
    mdat->recip[j][0] = v12[j] / V;
    mdat->recip[j][1] = v20[j] / V;
    mdat->recip[j][2] = v01[j] / V;
  }
  for (i=0;i<3;i++) {
    tmp=0.0;
    for (j=0;j<3;j++)
      tmp+=mdat->recip[j][i]*mdat->recip[j][i];
    mdat->reclen[i]=1.0/sqrt(tmp);
  }
}

