/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <sys/time.h>

#include <unistd.h>
#include <time.h>

#include "util.h"
#include "charmm_par.h"
#include "pdb.h"
#include "charmm_top.h"
#include "mdat.h"
#include "patch.h"
#include "config.h"
#include "align_axis.h"
#include "solvate.h"
#include "ion.h"

/* global variables */ 
cfg_t _cfg;   /* extern defined in config.h */
par_t _par;   /* extern defined in charmm_par.h */
top_t _top;   /* extern defined in charmm_top.h */
pdb_t _pdb;   /* extern defined in pdb.h */
mdat_t _mdat; /* extern defined in mdat.h */
linked_cell_t _lc;        /* extern defined in solvate.h */
pdb_t _solvent_input_pdb; /* extern defined in solvate.h */
pdb_t _solvent_pdb;       /* extern defined in solvate.h */
mdat_t _solvent_mdat;     /* extern defined in solvate.h */
ion_t _ion;   /* extern defined in ion.h */

int _warning;  /* extern defined in util.h */
int _debug;    /* extern defined in util.h */
int _no_abort; /* extern defined in util.h */

/* functions */
void print_usage();
void opening(char *ctrl_file);
char *strtime(void);

/* main function */
int main(int argc, char *argv[])
{
  int next_flag;
  char c;
  
  _warning = 0;
  _debug = 0;
  _no_abort = 0;
  if (argc == 1)
    print_usage();

  while (--argc > 0 && (*++argv)[0] == '-') {
    next_flag = 0;
    while (!next_flag && (c = *++(argv[0]))) {
      switch (c) {
      case 'w' :
	_warning = atoi(argv[0]+1);
	break;
      case 'd' :
	_debug++;
	break;
      case 'k' :
	_no_abort++;
	break;
      case 'v' :
	printf("molx: Version %d.%d.%d%s\n", MAJOR_VERSION, MINOR_VERSION, SUBMINOR_VERSION, PATCH_LEVEL);
	break;
      case 'h' :
      case '?' :
	print_usage();
      default:
	printf("molx: illegal option %c\n",c);
      }
    }
  }
  
  opening(argv[0]);
  mdat_init(&_mdat);
  par_init();
  top_init();
  
  cfg_read_file(argv[0]);

  top_read_file(_cfg.top_file, _cfg.n_top_file);
  par_read_file(_cfg.par_file, _cfg.n_par_file);
  
  toppar_read_file(_cfg.toppar_file, _cfg.n_toppar_file);

  pdb_init(&_pdb);
  if (_cfg.input_seq[0])
    pdb_read_seq(&_pdb, _cfg.input_seq);
  else if (_cfg.input_seq_buf != NULL)
    pdb_read_seq_buf(&_pdb, _cfg.input_seq_buf);
  
  if (_cfg.input_pdb[0])
    pdb_read_file(&_pdb, _cfg.input_pdb);

  if (_cfg.rename_res)
    pdb_rename_res(&_pdb, _cfg.rename_res);

  mdat_make_data_for_charmm(&_mdat, &_pdb);

  patch(&_mdat);
  
  mdat_add_mass(&_mdat, &_top);

  if (_cfg.input_crd[0])
    mdat_input_crd(&_mdat, _cfg.input_crd);

  mdat_make_coordinates(&_mdat);
  
  pdb_print_unchecked(&_pdb); 
  mdat_make_coord_from_ic(&_mdat, &_par);

  sol_set_crystal_solvent(&_mdat);
  
  if (_cfg.align_axis)
    align_axis(&_mdat);

  if (_cfg.solvent_pdb[0] || _cfg.ion_cation[0])
    solvate(&_mdat);


  mdat_make_connect(&_mdat);   /* */
  mdat_make_angle(&_mdat);     /* require mdat_make_connect */
  mdat_make_dihedral(&_mdat);  /* require mdat_make_connect */

  patch_no_auto(&_mdat);

  if (_cfg.renumber_pdb) {
    mdat_renumber_residue(&_mdat);
    printf("\nRenumbered residue No. and chain.\n");
  }
  /*
  if (!_cfg.renumber_pdb) {
    mdat_check_res_no(&_mdat); 
  }
  */

  mdat_add_par(&_mdat, &_par);
  mdat_make_mol(&_mdat);       /* require mdat_make_connect */


  mdat_count_all(&_mdat);      /* require making all data (atom,bond,angle,dihedral,
				  improper, mol */
  mdat_set_solute(&_mdat);
  mdat_make_ex_atom(&_mdat);   /* require mdat_count_all */
  par_count_used(&_par);
  par_make_nonbonded_pair(&_par);

  if (_cfg.wrap_molecules)
    mdat_wrap_molecules(&_mdat);

  mdat_print_info(&_mdat, &_par);

  if (_cfg.output_mdat[0])
    mdat_output(&_mdat, &_par);
  
  if (_cfg.output_crd[0])
    mdat_output_crd(&_mdat, _cfg.output_crd);

  if (_cfg.output_pdb[0]) {
    mdat_output_pdb(&_mdat,_cfg.output_pdb);
  }

  if (_cfg.output_pdb_orig_res_no[0]) {
    mdat_output_pdb_original_res(&_mdat,_cfg.output_pdb_orig_res_no);
  }

  if (_cfg.output_bond_file[0]) {
    mdat_output_bond_file(&_mdat,_cfg.output_bond_file);
  }

  if (_cfg.output_angle_file[0]) {
    mdat_output_angle_file(&_mdat,_cfg.output_angle_file);
  }

  if (_cfg.output_xyzs[0]) {
    mdat_output_xyzs(&_mdat,_cfg.output_xyzs);
  }
 
  return 0;
}

void print_usage()
{
  fprintf(stderr, "molx [-wlevel] control_file\n");
  exit(1);
}

void opening(char *ctrl_file)
{
  char hostname[100];

  printf("****************************************\n");
  printf("     Molx (Version %d.%d.%d%s)\n", MAJOR_VERSION, MINOR_VERSION, SUBMINOR_VERSION, PATCH_LEVEL);
  if (gethostname(hostname,100) == 0) {
    printf("     Host: %s\n",hostname);
  }

  printf("     Date: %s\n", strtime());
  
  printf("     Control File: %s\n", ctrl_file);
  printf("****************************************\n");
  printf("\n");
}

char *strtime()
{
  time_t t;
  char *p,*bp;
  t = time(0);
  bp = p = ctime(&t);
  while (*p != '\0') {
    if (*p == '\n')
      *p = '\0';
    p++;
  }
  return bp;
}

static char *_env_marblehome = NULL;
#ifndef DATDIR
#define DATDIR "/usr/local/marble/dat"
#endif

/*
  Caution:
    The memory reagion "fname" is copyed after the directory string is added!
 */
FILE *env_fopen(char fname[FNAME_LEN], char *mode)
{
  FILE *fp;
  char dirfname[FNAME_LEN];

  if ((fp = fopen(fname, mode)) != NULL)
    return fp;

  if (_env_marblehome == NULL)
    _env_marblehome = getenv("MARBLEHOME");

  if (_env_marblehome) {
    sprintf(dirfname,"%s/dat/%s", _env_marblehome, fname);
    if ((fp = fopen(dirfname, mode)) != NULL) {
      
      strcpy(fname, dirfname);

      return fp;
    }
  }

  sprintf(dirfname,"%s/%s", DATDIR, fname);
  fp = fopen(dirfname, mode);
  
  if (fp != NULL)
    strcpy(fname, dirfname);
  
  return fp;
}
