/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#define _ION_C

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>

#include "util.h"
#include "charmm_par.h"
#include "pdb.h"
#include "charmm_top.h"
#include "mdat.h"
#include "patch.h"
#include "config.h"
#include "solvate.h"
#include "ion.h"

void ion_initialize_grid(ion_t *ion, mdat_t *mdat, double delta);
void ion_setup_grid(ion_t *ion, mdat_t *mdat);

void ion_setup(mdat_t *mdat)
{
  ion_initialize(&_ion, mdat, &_solvent_pdb);
  ion_initialize_grid(&_ion, mdat, _cfg.ion_grid_spacing);
  ion_setup_grid(&_ion, mdat);
  /*
  ion_setup_potential(&_ion, mdat);
  */
  ion_setup_number(&_ion, mdat);
  ion_placement(&_ion, mdat);
  ion_remove_overlap_water(&_ion, &_solvent_pdb);
  ion_finalize(&_ion);
}

void ion_initialize(ion_t *ion, mdat_t *mdat, pdb_t *sol)
{
  int i,n;
  pdb_res_t *r;

  ion->n_water = 0;
  for (r=sol->res;r!=NULL;r=r->next) {
    if (!r->check)
      ion->n_water++;
  }

  ion->water = emalloc("ion_initialize",
		       sizeof(bulk_water_t)*ion->n_water);
  n=0;
  for (r=sol->res;r!=NULL;r=r->next) {
    if (!r->check) {
      ion->water[n].r = r;
      ion->water[n].x = r->beg_atom->x;
      ion->water[n].y = r->beg_atom->y;
      ion->water[n].z = r->beg_atom->z;
      ion->water[n].flag = 0;
      ion->water[n].epot = 0.0;
      ion->water[n].ion_density = 0.0;
      n++;
    }
  }
}

void ion_finalize(ion_t *ion)
{
  if (ion->water)
    free(ion->water);
  if (ion->grid)
    free(ion->grid);
}

void ion_initialize_grid(ion_t *ion, mdat_t *mdat, double delta)
{
  int i;
  int ii,jj,kk;
  bulk_water_t *site;
  pdb_atom_t *a;
  
  /* initialize cell */
  ion->nx = (int) (mdat->boxv[0][0] / delta) + 1;
  ion->ny = (int) (mdat->boxv[1][1] / delta) + 1;
  ion->nz = (int) (mdat->boxv[2][2] / delta) + 1;

  ion->dx = mdat->boxv[0][0] / ion->nx;
  ion->dy = mdat->boxv[1][1] / ion->ny;
  ion->dz = mdat->boxv[2][2] / ion->nz;

  ion->min[0] = ion->min[1] = ion->min[2] = 0.0;

  ion->grid = emalloc("ion_initialize_grid",
		     sizeof(ion_grid_t)*ion->nx*ion->ny*ion->nz);
  for (i=0;i<ion->nx*ion->ny*ion->nz;i++) {
    ion->grid[i].flag = 0;
  }

  printf("ION: Grid Spacing   (%.2f,%.2f,%.2f)\n", ion->dx,ion->dy,ion->dz);
  printf("ION: Number of Grid (%d,%d,%d)\n", ion->nx,ion->ny,ion->nz);

  for (i=0;i<ion->n_water;i++) {
    site = &ion->water[i];
    
    ii = (int) ((site->x - ion->min[0]) / ion->dx);
    jj = (int) ((site->y - ion->min[1]) / ion->dy);
    kk = (int) ((site->z - ion->min[2]) / ion->dz);

    if (ION_GRID(ii,jj,kk).flag) {
      printf("ION: two water molecules occupied same grid.\n");
      exit(1);
    }
    ION_GRID(ii,jj,kk).flag = 1;
    ION_GRID(ii,jj,kk).water = i;
  }
}

void ion_setup_grid(ion_t *ion, mdat_t *mdat)
{
  mdat_atom_t *a;
  int nb[3];
  int i,j,k;
  int ii,jj,kk;
  double dx,dy,dz, len2;
  double cutoffs2, cutoffm2, cutoffl2;
  bulk_water_t *site;

  mdat_set_radii(mdat);

  if (_cfg.ion_placement == ION_RANDOM) {
    _cfg.ion_cutoff = 2.0 + _cfg.solvent_radius + _cfg.ion_exclusion_layer;
  }

  printf("ION: placement %s\n", (_cfg.ion_placement == ION_ENERGY) ? "energy" : "random");
  printf("ION: ion_cutoff %.2f Angstrom\n", _cfg.ion_cutoff);
  printf("ION: solvent_radius %.2f Angstrom\n", _cfg.solvent_radius);
  printf("ION: ion_exclusion_layer %.2f Angstrom\n", _cfg.ion_exclusion_layer);
  printf("ION: ion2_exclusion_layer %.2f Angstrom\n", _cfg.ion2_exclusion_layer);
  
  nb[0] = (int) (_cfg.ion_cutoff / ion->dx) + 1;
  nb[1] = (int) (_cfg.ion_cutoff / ion->dy) + 1;
  nb[2] = (int) (_cfg.ion_cutoff / ion->dz) + 1;

  printf("ION: Starting to Charge Grid ");
  fflush(stdout);
  for (a=mdat->atom;a!=NULL;a=a->next) {
    i = (a->x - ion->min[0]) / ion->dx;
    j = (a->y - ion->min[1]) / ion->dy;
    k = (a->z - ion->min[2]) / ion->dz;
    cutoffs2 = a->r + _cfg.solvent_radius;
    cutoffm2 = cutoffs2 + _cfg.ion_exclusion_layer;
    cutoffl2 = _cfg.ion_cutoff;
    cutoffs2 *= cutoffs2;
    cutoffm2 *= cutoffm2;
    cutoffl2 *= cutoffl2;
    for (ii=i-nb[0];ii<=i+nb[0];ii++) {
      if (ii<0 || ii >= ion->nx) continue;
      for (jj=j-nb[1];jj<=j+nb[1];jj++) {
	if (jj<0 || jj >= ion->ny) continue;
	for (kk=k-nb[2];kk<=k+nb[2];kk++) {
	  if (kk<0 || kk >= ion->nz) continue;
	  if (ION_GRID(ii,jj,kk).flag != 1) continue;

	  site = &ion->water[ION_GRID(ii,jj,kk).water];

	  dx = site->x - a->x;
	  dy = site->y - a->y;
	  dz = site->z - a->z;
	  len2 = dx*dx + dy*dy + dz*dz;
	  if (len2 <= cutoffm2) {
	    ION_GRID(ii,jj,kk).flag = 3;
	    site->flag = 3;
	  } else if (len2 <= cutoffl2) {
	    if (ION_GRID(ii,jj,kk).flag <= 1) {
	      ION_GRID(ii,jj,kk).flag = 1; /* calculate elec-potential */
	      site->flag = 1;
	      site->epot += a->charge/len2;
	    }
	  }
	}
      }
    }
  }
  printf(" Done\n");
}


void ion_setup_potential(ion_t *ion, mdat_t *mdat)
{
  double dx,dy,dz, len2;
  double cutoff2, cutoffs2;
  bulk_water_t *site;
  pdb_atom_t *a;
  mdat_atom_t *ma;
  int i,j,k, ii, jj, kk;
  int nb[3];

  mdat_set_radii(mdat);

  cutoff2  = _cfg.ion_cutoff;
  cutoffs2 = _cfg.ion_exclusion_layer;

  /*
  if (_cfg.ion_placement == ION_RANDOM) {
    cutoff2 = cutoffs2;
  }
  printf("ION: placement %s\n", (_cfg.ion_placement == ION_ENERGY) ? "energy" : "random");
  */
  
  nb[0] = (int) (cutoff2 / _lc.dx) + 1;
  nb[1] = (int) (cutoff2 / _lc.dy) + 1;
  nb[2] = (int) (cutoff2 / _lc.dz) + 1;
  
  cutoff2  *= cutoff2;
  cutoffs2 *= cutoffs2;

  printf("ION: Calculating the electrostatic field ");
  fflush(stdout);
  
  for (i=0;i<ion->n_water;i++) {
    site = &ion->water[i];
    site->epot = 0.0;
    i = (int) (site->x / _lc.dx);
    j = (int) (site->y / _lc.dy);
    k = (int) (site->z / _lc.dz);
    for (ii=i-nb[0];ii<=i+nb[0];ii++) {
      if (ii < 0 || ii >= _lc.nx) continue;
      for (jj=j-nb[1];jj<=j+nb[1];jj++) {
	if (jj < 0 || jj >= _lc.ny) continue;
	for (kk=k-nb[2];kk<=k+nb[2];kk++) { 
	  if (kk < 0 || kk >= _lc.nz) continue;
	  for (ma=LC(ii,jj,kk).atom;ma!=NULL;ma=ma->gnext) {
	    dx = site->x - ma->x;
	    dy = site->y - ma->y;
	    dz = site->z - ma->z;
	    len2 = dx * dx + dy * dy + dz * dz;
	    if (len2 <= cutoffs2) {
	      site->flag = 3;                     /* calculate elec-potential */
	    } else if (len2 <= cutoff2) {
	      site->flag = 1;                     /* calculate elec-potential */
	      site->epot += ma->charge/len2;
	    }
	  }
	}
      }
    }
  }
  printf(" Done\n");
}

void ion_setup_number(ion_t *ion, mdat_t *mdat)
{
  int total_charge, h_total_charge;
  int i, nw, n_c_ion, n_ion;
  double volume, f_total_charge, tmp;
  mdat_atom_t *a;
  pdb_res_t *r;
  
  ion->cation_top  = top_search_res(_cfg.ion_cation);
  if (ion->cation_top==NULL) {
    printf("ERROR: No cation %s in top file\n", _cfg.ion_cation);
    exit(1);
  }
  ion->anion_top = top_search_res(_cfg.ion_anion);
  if (ion->anion_top==NULL) {
    printf("ERROR: No anion %s in top file\n", _cfg.ion_anion);
    exit(1);
  }

  f_total_charge = 0.0;
  for (a=mdat->atom;a!=NULL;a=a->next) {
    f_total_charge += a->charge;
  }

  /* This floor reterns the nearest integer of f_total_charge. */
  total_charge = (int) floor(f_total_charge+0.5);

  if (ion->n_cation == 0 && ion->n_anion == 0) {
    if (_cfg.ion_density > 0.0) {

      nw = mdat->n_c_water;
      for (r=_solvent_pdb.res;r!=NULL;r=r->next)
	if (r->check == 0)
	  nw++;

      /*
      volume = n / (0.997 / (15.9994+2*1.008) * MOL / 1.0e24);
      average = nint(volume * _cfg.ion_density * 1.0e-3 * MOL / 1.0e27 * 2);
      */
      /*
	n_ion = (n_water - n_ion) * volume * ion_density * 1.0e-3 *
	         MOL / 1.0e27 * 2;
      */
      tmp = (15.9994+2*1.008) / 0.997 * _cfg.ion_density * 1.0e-6 * 2;
      n_c_ion = mdat->n_c_cation + mdat->n_c_anion;
      /* This floor reterns the nearest integer of (nw+n_c_ion)*tmp. */
      n_ion = floor((nw + n_c_ion) * tmp + 0.5) - n_c_ion;
      n_ion = (n_ion/2) * 2;

      printf("ION: total charge is %d\n", total_charge);
      printf("ION: %d waters are solvated.\n", nw);
      printf("ION: %d ions are in input pdb file\n", n_c_ion);
      printf("ION: Number of ions is estimated to be %d.\n", n_ion + n_c_ion);
      printf("ION: %d ions are added.\n", n_ion);
      
      h_total_charge = total_charge / 2;
      ion->n_cation = n_ion/2 - (total_charge - h_total_charge);
      ion->n_anion  = n_ion/2 + h_total_charge;
      if (ion->n_anion < 0) {
	ion->n_cation -= ion->n_anion;
	ion->n_anion   = 0;
      } else if (ion->n_cation < 0) {
	ion->n_anion -= ion->n_cation;
	ion->n_cation  = 0;
      }
    } else {
      if (total_charge > 0) {
	ion->n_anion  = total_charge;
	ion->n_cation = 0;
      } else {
	ion->n_cation = -total_charge;
	ion->n_anion  = 0;
      }
    }
  }
  printf("ION: %s %d, %s %d\n", _cfg.ion_cation, ion->n_cation,
	 _cfg.ion_anion, ion->n_anion);

  if (ion->n_cation)
    ion->cation = emalloc("ion_setup_number", sizeof(ion_site_t) * ion->n_cation);
  
  if (ion->n_anion)
    ion->anion  = emalloc("ion_setup_number", sizeof(ion_site_t) * ion->n_anion);
  
  ion_insert_residues(ion, mdat);
  mdat_add_mass(mdat, &_top);
  mdat_set_radii(mdat);
}

void ion_insert_residues(ion_t *ion, mdat_t *mdat)
{
  int i;
  mdat_res_t *r, *pr, *nr;
  
  if (ion->n_anion == 0 && ion->n_cation == 0) 
    return;

  for (r=mdat->res,pr=NULL;r!=NULL;pr=r,r=r->next) {
    if (r->flag > _cfg.ion_order_flag)
      break;
  }

  if (pr) {
    nr = pr->next;
  } else {
    nr = mdat->res;
  }

  if (ion->n_anion > 0) {
    mdat_insert_residues(mdat, pr, ion->anion_top, ion->n_anion);
    for (i=0,r=pr->next;i<ion->n_anion;i++,r=r->next) {
      ion->anion[i].a = r->beg_atom;
      r->flag = MDAT_ADDED_ANION;
    }
  }

  if (ion->n_cation > 0) {
    mdat_insert_residues(mdat, pr, ion->cation_top, ion->n_cation);
    for (i=0,r=pr->next;i<ion->n_cation;i++,r=r->next) {
      ion->cation[i].a = r->beg_atom;
      r->flag = MDAT_ADDED_CATION;
    }
  }
  
  for (r=pr->next,i=1;r!=NULL;r=r->next) {
    r->pdb_no = i;
    r->pdb_chain = _cfg.ion_chain;
    if (++i > (ion->n_cation+ion->n_anion)) break;
  }

  if (pr) {
    if (pr->next)
      pr->next->first = 1;
  } else {
    mdat->res->first = 1;
  }
  if (nr) {
    if (nr->prev)
      nr->prev->last = 1;
  } else {
    for (nr=pr;nr!=NULL&&nr->next!=NULL;nr=nr->next);
    if (nr)
      nr->last = 1;
  }
}

void ion_placement(ion_t *ion, mdat_t *mdat)
{
  int n_cation, n_anion, put_cation;
  int i_cation, i_anion;
  int i,j,k;
  int ii,jj,kk;
  int iemin, iemax;
  int put_site;
  double emin, emax;
  double x,y,z;
  mdat_atom_t *a;
  char buf[100];
  double cutoffs2, cutoffm2, cutoffl2;
  double dx, dy, dz, len2;
  int nb[3];
  bulk_water_t *site;
  

  n_cation = ion->n_cation;
  n_anion  = ion->n_anion;

  i_cation = i_anion = 0;

  while (n_cation > 0 || n_anion > 0) {
    if (n_cation >= n_anion) {
      put_cation = 1;
      n_cation--;
    } else {
      put_cation = 0;
      n_anion--;
    }
    
    if (_cfg.ion_placement == ION_ENERGY) {
      emin = 1.0e10;
      emax = -1.0e10;
      iemin = -1;
      iemax = -1;
      for (i=0;i<ion->n_water;i++) {
	if (ion->water[i].flag <= 1) {
	  if (emin > ion->water[i].epot) {
	    emin = ion->water[i].epot;
	    iemin = i;
	  }
	  if (emax < ion->water[i].epot) {
	    emax = ion->water[i].epot;
	    iemax = i;
	  }
	}
      }
      if (put_cation) {
	put_site = iemin;
      } else {
	put_site = iemax;
      }
      if (put_site < 0) {
	printf("ERROR: No ion site\n");
	exit(1);
      }
      sprintf(buf,"by energy");
    } else if (_cfg.ion_placement == ION_RANDOM) {
      put_site = -1;
      while (put_site == -1) {
	put_site = drand48() * ion->n_water;
	if (ion->water[put_site].flag > 1) put_site = -1;
      }
      sprintf(buf,"by random");
    }
    ion->water[put_site].flag = 2;
    x = ion->water[put_site].x;
    y = ion->water[put_site].y;
    z = ion->water[put_site].z;
    if (put_cation) {
      ion->cation[i_cation].site = put_site;
      a = ion->cation[i_cation].a;
      a->x = x;
      a->y = y;
      a->z = z;
      a->coord = 1;
      i_cation++;
      printf("ION: cation %s ", _cfg.ion_cation);
    } else {
      ion->anion[i_anion].site = put_site;
      a  = ion->anion[i_anion].a;
      a->x = x;
      a->y = y;
      a->z = z;
      a->coord = 1;
      i_anion++;
      printf("ION: anion  %s ", _cfg.ion_anion);
    }
    printf("(%.2f,%.2f,%.2f) %s\n", x, y, z, buf);

  /* set potential */
    if (_cfg.ion_placement == ION_RANDOM) continue; 
    /*
    { double min,len;
    min = 1000.0;
    for (i=0;i<ion->n_cation-n_cation;i++) {
      if (ion->cation[i].a == a) continue; 
      dx=ion->cation[i].a->x - a->x;
      dy=ion->cation[i].a->y - a->y;
      dz=ion->cation[i].a->z - a->z;
      len=sqrt(dx*dx+dy*dy+dz*dz);
      printf("ca %d len = %f\n", i, len);
    
      if (min>len)
	min = len;
    }
    for (i=0;i<ion->n_anion-n_anion;i++) {
      if (ion->anion[i].a == a) continue; 
      dx=ion->anion[i].a->x - a->x;
      dy=ion->anion[i].a->y - a->y;
      dz=ion->anion[i].a->z - a->z;
      len=sqrt(dx*dx+dy*dy+dz*dz);
      printf("an %d len = %f\n", i, len);
      if (min>len)
	min = len;
    }
    printf("min = %f\n", min);
    }
    */

    nb[0] = (int) (_cfg.ion_cutoff / ion->dx) + 1;
    nb[1] = (int) (_cfg.ion_cutoff / ion->dy) + 1;
    nb[2] = (int) (_cfg.ion_cutoff / ion->dz) + 1;
    i = (x - ion->min[0]) / ion->dx;
    j = (y - ion->min[1]) / ion->dy;
    k = (z - ion->min[2]) / ion->dz;

    /*
    printf("gogo %d %d %d \n", i,j,k);
    printf("gogx %d %d \n", i-nb[0],i+nb[0]);
    printf("gogy %d %d \n", j-nb[1],j+nb[1]);
    printf("gogz %d %d \n", k-nb[2],k+nb[2]);
    */
    
    cutoffs2 = a->r + _cfg.solvent_radius;
    cutoffm2 = cutoffs2 + _cfg.ion2_exclusion_layer;
    cutoffl2 = _cfg.ion_cutoff;
    cutoffs2 *= cutoffs2;
    cutoffm2 *= cutoffm2;
    cutoffl2 *= cutoffl2;
    for (ii=i-nb[0];ii<=i+nb[0];ii++) {
      if (ii<0 || ii >= ion->nx) continue;
      for (jj=j-nb[1];jj<=j+nb[1];jj++) {
	if (jj<0 || jj >= ion->ny) continue;
	for (kk=k-nb[2];kk<=k+nb[2];kk++) {
	  if (kk<0 || kk >= ion->nz) continue;

	  if (ION_GRID(ii,jj,kk).flag != 1) continue; 

	  site = &ion->water[ION_GRID(ii,jj,kk).water];
	  if (site->flag == 2) continue; 

	  dx = site->x - a->x;
	  dy = site->y - a->y;
	  dz = site->z - a->z;
	  len2 = dx*dx + dy*dy + dz*dz;

	  if (len2 <= cutoffm2) {
	    site->flag = 3;
	    ION_GRID(ii,jj,kk).flag = 3;
	  } else if (len2 <= cutoffl2) {
	    if (site->flag <= 1) {
	      site->flag = 1;
	      ION_GRID(ii,jj,kk).flag = 1;  /* calculate elec-potential */
	      site->epot += a->charge/len2;
	    }
	  }
	}
      }
    }
  }
    
  printf("ION: %s %d, %s %d\n", _cfg.ion_cation, ion->n_cation,
	 _cfg.ion_anion, ion->n_anion);

  /*
  if (_cfg.monte_carlo > 0) {
    ion_setup_potential(ion, mdat);
    ion_monte_carlo(ion, mdat);
  }
  */
}

/*
void ion_monte_carlo(ion_t *ion, mdat_t *mdat)
{
  
}
*/

void ion_remove_overlap_water(ion_t *ion, pdb_t *sol)
{
  int i;
  pdb_atom_t *aw;
  pdb_res_t *r;
  mdat_atom_t *ai;

  for (i=0;i<ion->n_water;i++) {
    if (ion->water[i].flag == 2) {
      ion->water[i].r->check = 1;
    }
  }
}
