/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <ctype.h>

#include "util.h"
#include "charmm_par.h"
#include "pdb.h"
#include "charmm_top.h"
#include "config.h"
#include "mdat.h"


void cfg_init()
{
  strcpy(_cfg.input_pdb, "");
  strcpy(_cfg.input_seq, "");
  strcpy(_cfg.input_crd, "");
  _cfg.input_seq_buf = NULL;
  strcpy(_cfg.output_pdb, "");
  strcpy(_cfg.output_pdb_orig_res_no, "");
  strcpy(_cfg.output_mdat, "");
  strcpy(_cfg.output_crd, "");
  strcpy(_cfg.output_xyzs, "");
  strcpy(_cfg.title, "");

  _cfg.mdat_amber = 0;

  _cfg.patch = NULL;
  _cfg.alias = NULL;
  _cfg.ic_seed  = NULL;
  _cfg.ic_add = NULL;
  _cfg.rename_res = NULL;
  _cfg.bond_length_limit = 3.0;

  _cfg.renumber_pdb = 0;

  _cfg.align_axis = 0;

  _cfg.solvent_buffer = 10.0;
  _cfg.solvent_cube = 0;
  _cfg.solvent_exclusion_layer = 0.0;
  _cfg.box_buffer = 0.0;
  _cfg.solvent_chain = 'W';
  _cfg.solvent_renumbering = 1;
  _cfg.wrap_molecules = 0;

  _cfg.ion_chain = 'V';
  _cfg.other_chain = 'Z';
  _cfg.ion_cutoff = 10.0;
  _cfg.solvent_radius = 1.4;
  _cfg.ion_order_flag = MDAT_CRYSTAL_ANION;
  _cfg.ion_placement = ION_RANDOM;
  _cfg.ion_exclusion_layer = 4.0;
  _cfg.ion2_exclusion_layer = 2.0;
  _cfg.ion_density = 0.0;
  
  _cfg.ion_cation[0] = '\0';
  _cfg.ion_anion[0] = '\0';
  _cfg.n_cation = 0;
  _cfg.n_anion = 0;
  _cfg.ion_grid_spacing = 1.0;
  _cfg.pos_site = _cfg.neg_site = NULL;

  _cfg.n_top_file = _cfg.n_par_file = _cfg.n_toppar_file = 0;

  _cfg.use_vdw_radii = 0;

  strcpy(_cfg.output_bond_file, "");
  strcpy(_cfg.output_angle_file, "");
}

void cfg_read_file(char *fname)
{
  FILE *fp;
  char buf[CFG_BUFLEN];

  cfg_init();
  fp = fopen(fname, "r");
  if (fp == NULL) {
    fprintf(stderr, "ERROR: %s: No such file\n", fname);
    exit(1);
  }
  while(cfg_fgets(buf, CFG_BUFLEN, fp)){
    if (cfg_read_filenames(buf)) continue;
    if (cfg_read_patch(buf)) continue;
    if (cfg_read_alias(buf)) continue;
    if (cfg_read_ic(buf)) continue;
    if (cfg_read_misc(buf)) continue;
    if (cfg_read_boundary(buf)) continue;
    if (cfg_read_ion(buf)) continue;
    if (cfg_read_solvate(buf)) continue;
    printf("ERROR: Unrecognized line in control file: %s", buf);
    exit(1);
  }

  /* DEBUG
  printf("top file is %s\n",_cfg.top_file);
  printf("par file is %s\n",_cfg.par_file);
  printf("input pdb file is %s\n",_cfg.input_pdb);
  printf("output pdb file is %s\n",_cfg.output_pdb);
  printf("mdat file is %s\n",_cfg.output_mdat);
  
  print_patch();
  print_alias();
  */
  fclose(fp);

}

int cfg_read_filenames(char *buf)
{
  char filename[FNAME_LEN], buf2[CFG_BUFLEN];

  if (sscanf(buf, " charmm_top_file %s", filename) == 1){
    strcpy(_cfg.top_file[_cfg.n_top_file],filename);
    if (++(_cfg.n_top_file) >= MAX_N_TOP_FILE) {
      printf("ERROR: Number of top_file exceeded (MAX: %d)\n",
	     MAX_N_TOP_FILE);
      exit(1);
    }
    return 1;
  }
  if (sscanf(buf, " charmm_par_file %s", filename) == 1){
    strcpy(_cfg.par_file[_cfg.n_par_file],filename);
    if (++(_cfg.n_par_file) >= MAX_N_PAR_FILE) {
      printf("ERROR: Number of par_file exceeded (MAX: %d)\n",
	     MAX_N_PAR_FILE);
      exit(1);
    }
    return 1;
  }
  if (sscanf(buf, " charmm_toppar_file %s", filename) == 1){
    strcpy(_cfg.toppar_file[_cfg.n_toppar_file],filename);
    if (++(_cfg.n_toppar_file) >= MAX_N_TOPPAR_FILE) {
      printf("ERROR: Number of toppar_file exceeded (MAX: %d)\n",
	     MAX_N_TOPPAR_FILE);
      exit(1);
    }
    return 1;
  }

  if (sscanf(buf, " input_pdb_file %s", filename) == 1){
    strcpy(_cfg.input_pdb,filename);
    return 1;
  }
  if (sscanf(buf, " input_seq_file %s", filename) == 1){
    strcpy(_cfg.input_seq,filename);
    return 1;
  }
  if (sscanf(buf, " input_sequence %[^\n]", buf2) == 1){
    _cfg.input_seq_buf = emalloc("input_sequence", sizeof(char)*CFG_BUFLEN);
    strcpy(_cfg.input_seq_buf,buf2);
    /* printf("seq: %s\n", buf2); */
    return 1;
  }
  if (sscanf(buf, " output_pdb_file_orig_res_no %s", filename) == 1){
    strcpy(_cfg.output_pdb_orig_res_no,filename);
    return 1;
  }
  if (sscanf(buf, " output_pdb_file %s", filename) == 1){
    strcpy(_cfg.output_pdb,filename);
    return 1;
  }
  if (sscanf(buf, " output_mdat_file %s", filename) == 1){
    strcpy(_cfg.output_mdat,filename);
    return 1;
  }
  
  if (sscanf(buf, " output_crd_file %s", filename) == 1){
    strcpy(_cfg.output_crd,filename);
    return 1;
  }

  if (sscanf(buf, " output_xyzs_file %s", filename) == 1){
    strcpy(_cfg.output_xyzs,filename);
    return 1;
  }

  if (sscanf(buf, " output_pdb_lj_q %s", buf2) == 1){
    if (strcmp(buf2,"on") == 0)
      _mdat.output_pdb_lj_q = 1;
    else
      _mdat.output_pdb_lj_q = 0;
    return 1;
  }

  return 0;
}

int cfg_read_misc(char *buf)
{
  char str[1000], str2[100], dummy;
  long ld;
  double f;
  pdb_rename_res_t *rename_res;
  static pdb_rename_res_t *rename_res_prev;

  if (sscanf(buf, " title %s", str) == 1) {
    strcpy(_cfg.title, str); 
    return 1;
  } else if (sscanf(buf, " seed %ld", &ld) == 1) {
    srand48(ld);
    return 1;
  } else if (sscanf(buf, " renumber_residue %s", str) == 1) {
    if (strcmp(str,"off") == 0)
      _cfg.renumber_pdb = 0;
    else 
      _cfg.renumber_pdb = 1;
    return 1;
  } else if (sscanf(buf, " align_axis %s", str) == 1) {
    if (strcmp(str,"on") == 0)
      _cfg.align_axis = ALIGN_AXIS;
    else if (strcmp(str,"normal") == 0)
      _cfg.align_axis = ALIGN_AXIS;
    else if (strcmp(str,"z") == 0)
      _cfg.align_axis = ALIGN_AXIS_Z;
    else if (strcmp(str,"diagonal") == 0)
      _cfg.align_axis = ALIGN_DIAGONAL;;
    return 1;
  } else if (sscanf(buf, " rename_residue %s %s", str2, str) == 2) {
    rename_res = emalloc("rename_residue",sizeof(pdb_rename_res_t));
    
    cfg_str_to_res_no(str2, &(rename_res->res_no), &(rename_res->chain));

    strcpy(rename_res->res_name,str);
    rename_res->next = NULL;
    if (_cfg.rename_res == NULL) {
      _cfg.rename_res = rename_res;
    } else {
      rename_res_prev->next = rename_res;
    }
    rename_res_prev = rename_res;
    
    return 1;
    
  } else if (sscanf(buf, " bond_length_limit %lf", &f) == 1) {
    _cfg.bond_length_limit = f;
    return 1;

  } else if (sscanf(buf, " use_vdw_radii %s", str) == 1) {
    if (strcmp(str,"on") == 0)
      _cfg.use_vdw_radii = 1;
    return 1;

  } else if (sscanf(buf, " mdat_amber %s", str) == 1) {
    if (strcmp(str,"on") == 0)
      _cfg.mdat_amber = 1;
    return 1;
  }
  
  return 0;
}

int cfg_read_patch(char *buf)
{
  char chain[3];
  char res_str1[10], res_str2[10], res_str3[10];
  int res[3], n_pres;
  char patch_name[5];
  cfg_patch_t *patch;
  int n_patch, step;
  int ter;
  int no_auto = 0;
  static cfg_patch_t *patch_prev = NULL ;
  
  if (sscanf(buf, " patch_multi %s%s%s%d%d", patch_name, res_str1, res_str2, &n_patch, &step) == 5) {
    n_pres = 2;
    ter = 0;
  } else if (sscanf(buf, " patch_multi %s%s%d%d", patch_name, res_str1, &n_patch, &step) == 4) {
    n_pres = 1;
    ter = 0;
  } else if (sscanf(buf, " patch_ter %s%s%s", patch_name, res_str1, res_str2) == 3 ) {
    n_pres = 2;
    n_patch = step = 1;
    ter = 1;
  } else if (sscanf(buf, " patch_ter %s%s", patch_name, res_str1 ) == 2) {
    n_pres = 1;
    n_patch = step = 1;
    ter = 1;
  } else if (sscanf(buf, " patch_no_auto %s%s%s%s", patch_name, res_str1, res_str2, res_str3) == 4 ) {
    n_pres = 3;
    n_patch = step = 1;
    ter = 0;
    no_auto = 1;
  } else if (sscanf(buf, " patch_no_auto %s%s%s", patch_name, res_str1, res_str2) == 3 ) {
    n_pres = 2;
    n_patch = step = 1;
    ter = 0;
    no_auto = 1;
  } else if (sscanf(buf, " patch_no_auto %s%s", patch_name, res_str1 ) == 2) {
    n_pres = 1;
    n_patch = step = 1;
    ter = 0;
    no_auto = 1;
  } else if (sscanf(buf, " patch %s%s%s", patch_name, res_str1, res_str2) == 3 ) {
    n_pres = 2;
    n_patch = step = 1;
    ter = 0;
  } else if (sscanf(buf, " patch %s%s", patch_name, res_str1 ) == 2) {
    n_pres = 1;
    n_patch = step = 1;
    ter = 0;
  } else {
    return 0;
  }

  if (n_pres == 1) {
    cfg_str_to_res_no(res_str1, &res[0], &chain[0]);
    res[1] = 0;
    chain[1] = ' ' ;
    res[2] = 0;
    chain[2] = ' ' ;
  } else if (n_pres == 2) {
    cfg_str_to_res_no(res_str1, &res[0], &chain[0]);
    cfg_str_to_res_no(res_str2, &res[1], &chain[1]);
    res[2] = 0;
    chain[2] = ' ' ;
  } else if (n_pres == 3) {
    cfg_str_to_res_no(res_str1, &res[0], &chain[0]);
    cfg_str_to_res_no(res_str2, &res[1], &chain[1]);
    cfg_str_to_res_no(res_str3, &res[2], &chain[2]);
  }
  
  patch = emalloc("patch",sizeof(cfg_patch_t));
  strcpy(patch->patch_name,patch_name);
  patch->chain[0]=chain[0];
  patch->chain[1]=chain[1];
  patch->chain[2]=chain[2];
  patch->res[0]=res[0];
  patch->res[1]=res[1];
  patch->res[2]=res[2];
  patch->n_pres = n_pres;
  patch->n_patch = n_patch;
  patch->step = step;
  patch->ter  = ter;
  patch->no_auto = no_auto;
  patch->next = NULL;
  if (patch_prev == NULL) {
    _cfg.patch = patch;
  }else{
    patch_prev->next = patch ;
  }
  patch_prev = patch;
  return 1;
}

int cfg_read_alias(char *buf)
{
  char str1[10], str2[10];
  cfg_alias_t *alias; 
  static cfg_alias_t *alias_prev = NULL ;
  
  if (sscanf(buf, " alias %s%s", str1, str2) == 2) {
  } else {
    return 0;
  }
  
  alias = emalloc("alias",sizeof(cfg_alias_t));
  strcpy(alias->str[0],str1);
  strcpy(alias->str[1],str2);
  alias->next = NULL;
  if (alias_prev == NULL) {
    _cfg.alias = alias;
  }else{
    alias_prev->next = alias ;
  }
  alias_prev = alias;
  return 1;
}

int cfg_read_ic(char *buf)
{
  char atom[4][20], atom_str[4][20];
  cfg_ic_seed_t *ic_seed;
  cfg_ic_add_t  *ic_add;
  int i;
  double d[6];

  /*
  if (sscanf(buf, " ic_seed %s%s%s%s%s%s",
	     res_str[0], atom[0],
	     res_str[1], atom[1],
	     res_str[2], atom[2]) == 6) {
  */
  if (sscanf(buf, " ic_seed %s%s%s",
	     atom_str[0],  atom_str[1], atom_str[2]) == 3) {

    ic_seed = emalloc("ic",sizeof(cfg_ic_seed_t));
    
    for (i=0;i<3;i++) {
      cfg_str_to_atom_res_no(atom_str[i], ic_seed->atom[i], 
			     &ic_seed->resno[i],&ic_seed->chain[i]);
    }

    ic_seed->next = _cfg.ic_seed;
    _cfg.ic_seed = ic_seed;
    return 1;
  }

  /*
  if (sscanf(buf, " IC %s%s%s%s%s%s%s%s%lf%lf%lf%lf%lf",
	     res_str[0], atom[0], 
	     res_str[1], atom[1], 
	     res_str[2], atom[2], 
	     res_str[3], atom[3], 
	     &d[0],&d[1],&d[2],&d[3],&d[4]) == 13 ||
      sscanf(buf," ic %s%s%s%s%s%s%s%s%lf%lf%lf%lf%lf",
	     res_str[0], atom[0], 
	     res_str[1], atom[1], 
	     res_str[2], atom[2], 
	     res_str[3], atom[3], 
	     &d[0],&d[1],&d[2],&d[3],&d[4]) == 13) {
  */
  if (sscanf(buf, " IC %s%s%s%s%lf%lf%lf%lf%lf",
	     atom_str[0],
	     atom_str[1],
	     atom_str[2],
	     atom_str[3],
	     &d[0],&d[1],&d[2],&d[3],&d[4]) == 9 ||
      sscanf(buf, " ic %s%s%s%s%lf%lf%lf%lf%lf",
	     atom_str[0],
	     atom_str[1],
	     atom_str[2],
	     atom_str[3],
	     &d[0],&d[1],&d[2],&d[3],&d[4]) == 9) {

    ic_add = emalloc("ic", sizeof(cfg_ic_add_t));

    for (i=0;i<4;i++) {
      cfg_str_to_atom_res_no(atom_str[i], atom[i], 
			     &ic_add->resno[i],&ic_add->chain[i]);
      strcpy(ic_add->atom[i], atom[i]);
    }
    
    if (atom[2][0] == '*') {
      ic_add->impr = 1;
      strcpy(ic_add->atom[2], &atom[2][1]);
    } else {
      ic_add->impr = 0;
    }
    ic_add->length[0] = d[0];
    ic_add->angle[0]  = d[1];
    ic_add->dihedral  = d[2];
    ic_add->angle[1]  = d[3];
    ic_add->length[1] = d[4];
    ic_add->next = _cfg.ic_add;
    _cfg.ic_add = ic_add;
    return 1;
  }

  return 0;
}

int cfg_read_ion(char *buf)
{
  char str1[10],str2[10],str3[10];
  double f;
  double xx, yy, zz;
  int d;
  cfg_ionic_site_t *pos_site, *neg_site;
  static cfg_ionic_site_t *pr_pos_site, *pr_neg_site;
  
  if (sscanf(buf, " ion_cutoff %lf", &f) == 1){
    _cfg.ion_cutoff = f;
    return 1;
  } else if (sscanf(buf, " ion_chain %c", str1) == 1){
    _cfg.ion_chain = str1[0];
    return 1;
  } else if (sscanf(buf, " ion_order %s", str1) == 1){
    if (strcmp(str1,"after_solute") == 0) {
      _cfg.ion_order_flag = MDAT_SOLUTE;
    } else if (strcmp(str1,"after_crystal_cation") == 0) {
      _cfg.ion_order_flag = MDAT_CRYSTAL_CATION;
    } else if (strcmp(str1,"after_crystal_anion") == 0) {
      _cfg.ion_order_flag = MDAT_CRYSTAL_ANION;
    } else if (strcmp(str1,"after_input_pdb") == 0) {
      _cfg.ion_order_flag = MDAT_INPUT_PDB;
    } else {
      _cfg.ion_order_flag = MDAT_INPUT_PDB;
    }
    return 1;
  } else if (sscanf(buf, " ion_grid_spacing %lf", &f) == 1){
    _cfg.ion_grid_spacing = f;
    return 1;
  } else if (sscanf(buf, " solvent_radius %lf", &f) == 1){
    _cfg.solvent_radius = f;
    return 1;
  } else if (sscanf(buf, " ion_excluded_layer %lf", &f) == 1 ||
	     sscanf(buf, " ion_exclusion_layer %lf", &f) == 1) {
    _cfg.ion_exclusion_layer = f;
    return 1;
  } else if (sscanf(buf, " ion2_excluded_layer %lf", &f) == 1 ||
	     sscanf(buf, " ion_exclusion_layer %lf", &f) == 1) {
    _cfg.ion2_exclusion_layer = f;
    return 1;
  } else if (sscanf(buf, " ion_density %lf", &f) == 1) {
    _cfg.ion_density = f;
    return 1;
  } else if (sscanf(buf, " ion_placement %s", str1) == 1) {
    if (strcmp(str1,"energy") == 0) {
      _cfg.ion_placement = ION_ENERGY;
    } else if (strcmp(str1,"random") == 0) {
      _cfg.ion_placement = ION_RANDOM;
    } else {
      printf("ERROR: Ion_placement: Minimum or random must be specified.\n");
      exit(1);
    }
    return 1;
  } else if (sscanf(buf, " n_cation %d", &d) == 1) {
    _cfg.n_cation = d;
    return 1;
  } else if (sscanf(buf, " n_anion %d", &d) == 1) {
    _cfg.n_anion  = d;
    return 1;
  } else if (sscanf(buf, " ion %s %s", str1, str2) == 2) {
    strcpy(_cfg.ion_cation, str1);
    strcpy(_cfg.ion_anion,  str2);
    return 1;
  } else if (sscanf(buf, " cation_position %lf %lf %lf", &xx, &yy, &zz) ==3 ) {
    pos_site = emalloc("cfg_read_ion",sizeof(cfg_ionic_site_t));
    pos_site->next = NULL;
    if (_cfg.pos_site == NULL) {
      _cfg.pos_site = pos_site;
    } else {
      pr_pos_site->next = pos_site;
    }
    pr_pos_site = pos_site;

    pos_site->x = xx;
    pos_site->y = yy;
    pos_site->z = zz;
    return 1;
  } else if (sscanf(buf, " anion_position %lf %lf %lf", &xx, &yy, &zz) ==3 ) {
    neg_site = emalloc("cfg_read_ion",sizeof(cfg_ionic_site_t));
    neg_site->next = NULL;
    if (_cfg.neg_site == NULL) {
      _cfg.neg_site = neg_site;
    } else {
      pr_neg_site->next = neg_site;
    }
    pr_neg_site = neg_site;

    neg_site->x = xx;
    neg_site->y = yy;
    neg_site->z = zz;
    
    return 1;
  }
  
  return 0;
}

int cfg_read_boundary(char *buf)
{
  double f1, f2, f3, alpha, beta, gamma;
  char buf2[100];

  if (sscanf(buf, " box %lf %lf %lf %lf %lf %lf",
	     &f1,&f2,&f3,&alpha,&beta,&gamma) == 6) {
    mdat_set_boxv(&_mdat,f1,f2,f3,alpha,beta,gamma);
    return 1;
  } else if (sscanf(buf, " box %lf %lf %lf",
	     &f1,&f2,&f3) == 3) {
    mdat_set_boxv(&_mdat, f1,f2,f3,90.0,90.0,90.0);
    return 1;
  } else if (sscanf(buf, " wrap_molecules %s",buf2) == 1) {
    if (strcmp(buf2,"on") == 0) {
      _cfg.wrap_molecules = 1;
    } else {
      _cfg.wrap_molecules = 0;
    }
    return 1;
  }
  
  return 0;
}

int cfg_read_solvate(char *buf)
{
  char buf2[100];
  double d;
  
  if (sscanf(buf, " solvent_pdb_file %s", buf2) == 1) {
    strcpy(_cfg.solvent_pdb, buf2);
    return 1;
  } else if(sscanf(buf, " solvent_chain %c", buf2) == 1){
    _cfg.solvent_chain = buf2[0];
    return 1;
  } else if(sscanf(buf, " solvent_renumbering %s", buf2) == 1){
    if (strcmp(buf2,"on") == 0) {
      _cfg.solvent_renumbering = 1;
    } else if (strcmp(buf2,"chain") == 0) {
      _cfg.solvent_renumbering = 2;
    } else {
      _cfg.solvent_renumbering = 0;
    }
    return 1;
  } else if(sscanf(buf, " solvent_buffer %lf", &d) == 1){
    _cfg.solvent_buffer = d;
    return 1;
  } else if(sscanf(buf, " box_buffer %lf", &d) == 1){
    _cfg.box_buffer = d;
    return 1;
  } else if(sscanf(buf, " solvent_excluded_layer %lf", &d) == 1 ||
	    sscanf(buf, " solvent_exclusion_layer %lf", &d) == 1 ){
    _cfg.solvent_exclusion_layer = d;
    return 1;
  } else if(sscanf(buf, " solvent_cube %s", buf2) == 1){
    if (strcmp(buf2,"on") == 0) {
      _cfg.solvent_cube = 1;
    }
    return 1;
  }
  return 0;
}



void cfg_print_patch()
{
  cfg_patch_t *patch;
  
  for (patch=_cfg.patch;patch!=NULL;patch=patch->next){
    printf("patch %s %d%c %d%c\n",patch->patch_name, patch->res[0],
	   patch->chain[0], patch->res[1], patch->chain[1]);
  }
}

void cfg_print_alias()
{
  cfg_alias_t *alias;
  
  for (alias=_cfg.alias;alias!=NULL;alias=alias->next){
    printf("alias %s %s\n",alias->str[0], alias->str[1]);
  }
}

int aliascmp(char *target1,char *target2)
{
  cfg_alias_t *alias1, *alias2;
  int i;

  /*
  if (strcmp(target1,target2) == 0)
    return 0;
  */

  for (i=0;i<100;i++) {
    if (target1[i] != target2[i]) {
      /* for nucleic acid atoms */
      if (target1[i] != '\'' || target2[i] != '*') 
	break;
    }
    if (target1[i] == '\0' || target2[i] == '\0')
      return 0;
  }

  for(alias1 = _cfg.alias;alias1 != NULL;alias1 = alias1->next) {
    if(strcmp(target1,alias1->str[0]) == 0) {
      if (strcmp(alias1->str[1], target2) == 0) return 0;
      /*
      for(alias2 = _cfg.alias;alias2 != NULL;alias2 = alias2->next) {
	if (strcmp(target2, alias2->str[0]) == 0) {
	  if (strcmp(alias1->str[1], alias2->str[1]) == 0) return 0;
	}
      }
      */
    }
    if (strcmp(target2, alias1->str[0]) == 0) {
      if (strcmp(target1, alias1->str[1]) == 0) return 0;
    }
  }
      
  return 1;
}

void cfg_str_to_atom_res_no(char *atom_str, char *atom, int *resno, char *chain)
{
  char res_str[100];

  if (sscanf(atom_str, "%[^. \t].%s", atom, res_str) != 2) {
    printf("ERROR: Atom Name Format Error \"%s\"\n",atom_str);
    exit(1);
  }
  cfg_str_to_res_no(res_str,resno,chain);
}

void cfg_str_to_res_no(char *name, int *resno, char *chain)
{
  int err = 0;
  
  if (isdigit(name[0])) {
    if (sscanf(name, "%d%c", resno, chain) != 2) {
      if (sscanf(name, "%d", resno) != 1) {
	err=1;
      }
      *chain = ' ';
    }
  } else {
    if (sscanf(name, "%c%d", chain, resno) != 2) {
      err=1;
    }
  }

  if (err) {
    printf("ERROR: Residue Name Format Error \"%s\"\n",name);
    exit(1);
  }
}
