/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>

#include "util.h"
#include "charmm_par.h"
#include "charmm_top.h" 
#include "pdb.h" 
#include "config.h" 

#define TOP_BUFLEN 1000

void top_init()
{
  _top.mass = NULL;
  _top.res = NULL;
  _top.pres = NULL;
}


void top_read_file(char fname[][FNAME_LEN], int n_fname)
{
  FILE *fp;
  char fname2[FNAME_LEN];
  int i;
  
  for (i=0;i<n_fname;i++) {
    fp = env_fopen(fname[i], "r");
    if (fp==NULL) {
      fprintf(stderr,"ERROR: %s: No such file\n", fname[i]);
      exit(1);
    }
    top_read(fp);
    top_count();
    if (i==0) {
      strcpy(fname2,fname[i]);
    } else {
      sprintf(fname2,"+ %s", fname[i]);
    }
    top_print_info(fname2);
    top_check_dup();
    fclose(fp);
  }
}


void top_read(FILE *fp)
{
  char buf[TOP_BUFLEN],buf1[TOP_BUFLEN];
  char header[10], headercp[10],residue[MAXRNUM],name[MAXANUM],sym[MAXSYM],type[MAXSYM];
  char arg[MAXNARG][MAXANUM],first[MAXRNUM],last[MAXRNUM];
  int i,j,resnum,presnum,group,a_num,d_num,success;
  int hid, hid2, resi_flag, pres_flag;
  double charge;
  top_mass_t *mass, *mass_prev;
  top_res_t *res, *res_prev;
  top_res_t *pres, *pres_prev;
  top_atom_t *atom,*atom_prev;
  top_d_atom_t *d_atom,*d_atom_prev;
  top_bond_t *bond,*bond_prev;
  top_angle_t *angle,*angle_prev;
  top_angle_t *d_angle,*d_angle_prev;
  top_dihedral_t *dihedral,*dihedral_prev;
  top_impr_t *impr,*impr_prev;
  top_impr_t *d_impr,*d_impr_prev;
  top_cmap_t *cmap,*cmap_prev;
  top_donor_t *donor,*donor_prev;
  top_donor_t *d_donor,*d_donor_prev;
  top_acceptor_t *accept,*accept_prev;
  top_acceptor_t *d_accept,*d_accept_prev;
  top_ic_t *ic,*ic_prev;
  top_res_t *pointer;

  /*initialize*/
  resi_flag = pres_flag = 0;

  /* MASS */
  if (_top.mass == NULL) {
    mass_prev = NULL;
  } else {
    for (mass = _top.mass; mass!=NULL; mass=mass->next) {
      mass_prev = mass;
    }
  }

  /* RESIDUE */
  if (_top.res == NULL) {
    res_prev = NULL;
    resnum = 0;
  } else {
    for (res = _top.res; res!=NULL; res=res->next) {
      res_prev = res;
      resnum = res_prev->num;
    }
  }

  /* RESIDUE PATCH */
  if (_top.pres == NULL) {
    pres_prev = NULL;
    presnum = 0;
  } else {
    for (pres = _top.pres; pres!=NULL; pres=pres->next) {
      pres_prev = pres;
      presnum = pres_prev->num;
    }
  }

  top_fgets(buf, TOP_BUFLEN, fp);
  sscanf(buf, "%d%d",&_top.major_version,&_top.minor_version);

  /*printf(buf);*/

  while (top_fgets(buf, TOP_BUFLEN, fp)) {
    /*printf(buf);*/

    strcpy(header,"");
    strcpy(headercp,"");
    sscanf(buf,"%s",headercp);

    for(i=0;i<MAXNARG;i++){
      strcpy(arg[i],"");
    }

    hid = top_resi_header_cmp(headercp);

    switch (hid) {
    case TOP_MASS:
      mass = emalloc("top_mass", sizeof(top_mass_t));
      sscanf(buf, "%s%d%s%lf%s",header,&(mass->no),
	     mass->sym, &(mass->weight), mass->type);
      mass->check = 0;

      mass->next = NULL;
      if (mass_prev == NULL) {
	_top.mass = mass;
      } else {
	mass_prev->next = mass;
      }
      mass_prev = mass;
      break;
      
    case TOP_DEFA:      /*HEADER is DEFA*/
      sscanf(buf,"%s%s%s%s%s",headercp,arg[0],first,arg[1],last);
      break;
    case TOP_RESI:     /* HEADER is RESI */
    case TOP_PRES:     /* HEADER is PRES */
    case TOP_END:      /* HEADER is END */
      if (resi_flag) {
	/*END of struct res*/ 
	res->next = NULL;
	if (res_prev == NULL) {
	  _top.res = res;
	} 
	else{
	  res_prev->next = res;
	}
	res_prev = res;
	resi_flag = 0;
      } else if (pres_flag) {
	/*END of struct pres*/

	pres->next = NULL;
	if (pres_prev == NULL) {
	  _top.pres = pres;
	} 
	else{
	  pres_prev->next = pres;
	}
	pres_prev = pres;
	pres_flag = 0;
      }

      if(hid == TOP_RESI) {
	/*Start of struct res*/
	res = emalloc("top_res", sizeof(top_res_t));
	resi_flag = 1;
	resnum++;
	pointer = res;
      } else if(hid == TOP_PRES) {
	/*Start of struct pres*/
	pres = emalloc("top_pres", sizeof(top_res_t));
	pres_flag = 1;
	presnum++;
	pointer = pres;
      } else if (hid == TOP_END) {
	/*END of loop*/
	return;
      }
      
      /*initialize*/
      a_num = 0;
      d_num = 0;
      group = 0;
      pointer->atom = atom_prev = NULL;
      pointer->d_atom = d_atom_prev = NULL;
      pointer->bond = bond_prev = NULL;
      pointer->impr = impr_prev = NULL;
      pointer->cmap = cmap_prev = NULL;
      pointer->donor = donor_prev = NULL;
      pointer->accept = accept_prev = NULL;
      pointer->angle = angle_prev = NULL;
      pointer->dihedral = dihedral_prev = NULL;
      pointer->d_angle = d_angle_prev = NULL;
      pointer->d_donor = d_donor_prev = NULL;
      pointer->d_accept = d_accept_prev = NULL;
      pointer->d_impr = d_impr_prev = NULL;
      pointer->ic = ic_prev = NULL;
      strcpy(pointer->patch[0],first);
      strcpy(pointer->patch[1],last);

      
      /* read */ 
      sscanf(buf,"%s%s%lf",header,residue,&charge);
      pointer->charge = charge;
      if (hid==TOP_RESI)
	pointer->num = resnum;
      else 
	pointer->num = presnum;
      strcpy(pointer->name,residue);
      break;
    
    case TOP_GROUP:  /*HEADER is GROUP*/
      group++;
      break;
    
    case TOP_ATOM:   /*HEADER is ATOM*/
      a_num++;
      atom = emalloc("top_atom",sizeof(top_atom_t));
      sscanf(buf,"%s%s%s%lf",header,name,sym,&charge);
      
      atom->group = group;
      atom->charge = charge;
      atom->no = a_num;
      strcpy(atom->name,name);
      strcpy(atom->sym,sym);    

      atom->next = NULL;
      if (atom_prev == NULL) {
	pointer->atom = atom;
      } 
      else{
	atom_prev->next = atom;
      }
      atom_prev = atom;
      break;
    case TOP_BOND:
    case TOP_DOUBLE:   
    case TOP_TRIPLE:   /*HEADER is BOND or DOUBLE or TRIPLE */
      success = top_scan_string_list(buf,arg);
      i = 1; 
      while(i<=success-2) {
        bond = emalloc("top_bond",sizeof(top_bond_t));
        strcpy(bond->atom[0],arg[i]);
	strcpy(bond->atom[1],arg[i+1]);
	if (hid == TOP_BOND)
	  bond->type = 1;
	else if (hid == TOP_DOUBLE)
	  bond->type = 2;
	else if (hid == TOP_TRIPLE)
	  bond->type = 3;
	i+=2;
	bond->next = NULL;
	if (bond_prev == NULL) {
	  pointer->bond = bond;
	} else {
	  bond_prev->next = bond;
	}
	bond_prev = bond;
      }
      break;
      
    case TOP_IMPR:   /*HEADER is IMPR*/
      success = top_scan_string_list(buf,arg);
      i = 1;
      while(i<=success-4) {

        impr = emalloc("top_impr",sizeof(top_impr_t));
        strcpy(impr->atom[0],arg[i]);
        strcpy(impr->atom[1],arg[i+1]);
        strcpy(impr->atom[2],arg[i+2]);
        strcpy(impr->atom[3],arg[i+3]);
        i += 4;
      
        impr->next = NULL;
        if (impr_prev == NULL) {
  	  pointer->impr = impr;
        } 
        else{
	  impr_prev->next = impr;
        }
        impr_prev = impr;
      }
      break;
      
    case TOP_CMAP:   /* HEADER is CMAP */
      success = top_scan_string_list(buf,arg);
      i = 1;
      while(i<=success-8) {
	cmap = emalloc("top_cmap",sizeof(top_cmap_t));
	for (j=0;j<8;j++)
	  strcpy(cmap->atom[j],arg[i+j]);
	i+=8;

        cmap->next = NULL;
        if (cmap_prev == NULL) {
  	  pointer->cmap = cmap;
        } 
        else{
	  cmap_prev->next = cmap;
        }
        cmap_prev = cmap;
      }
      break;
      
    case TOP_DONOR:  /* HEADER is DONOR */
      donor = emalloc("top_donor",sizeof(top_donor_t));
      
      success = sscanf(buf,"%s%s%s",header,arg[0],arg[1]);
      i=0;
      
      while(i<success-1){
	if(arg[i][0]=='!')
	  break;
	strcpy(donor->atom[i],arg[i]);
	i++;
      }

      donor->next = NULL;
      if (donor_prev == NULL) {
	pointer->donor = donor;
      } 
      else{
	donor_prev->next = donor;
      }
      donor_prev = donor;
      break;
    
    case TOP_ACCEPTOR:  /*HEADER is ACCEPTOR*/
      accept = emalloc("top_acceptor", sizeof(top_acceptor_t));
      
      success = sscanf(buf,"%s%s%s",header,arg[0],arg[1]);
      
      i=0;
      while(i<success-1){
	if(arg[i][0]=='!')
	  break;
	strcpy(accept->atom[i],arg[i]);
	i++;
      }

      accept->next = NULL;
      if (accept_prev == NULL){
	pointer->accept = accept;
      } 
      else{
	accept_prev->next = accept;
      }
      accept_prev = accept;
      break;

    case TOP_ANGLE:  /*HEADER is ANGLE*/
      success = top_scan_string_list(buf,arg);
      i = 1;
       while(i <= success-3){
	 angle = emalloc("top_angle",sizeof(top_angle_t));
	 strcpy(angle->atom[0],arg[i]);
	 strcpy(angle->atom[1],arg[i+1]);
	 strcpy(angle->atom[2],arg[i+2]);
	 i += 3;
	 pointer->c_angle++;

	 angle->next = NULL;
	 if (angle_prev == NULL) {
	   pointer->angle = angle;
	 } else {
	   angle_prev->next = angle;
	 }
	 angle_prev = angle;
       }
       break;
       
    case TOP_DIHE:  /*HEADER is DIHE*/
      success = top_scan_string_list(buf,arg);
      i = 1;
      while(i <= success - 4){
        dihedral = emalloc("top_dihedral",sizeof(top_dihedral_t));
        strcpy(dihedral->atom[0],arg[i]);
        strcpy(dihedral->atom[1],arg[i+1]);
        strcpy(dihedral->atom[2],arg[i+2]);
        strcpy(dihedral->atom[3],arg[i+3]);
        i += 4;
	
	pointer->c_dihedral++;
     
        dihedral->next = NULL;
        if (dihedral_prev == NULL) {
  	  pointer->dihedral = dihedral;
        } else{
	  dihedral_prev->next = dihedral;
        }
        dihedral_prev = dihedral;
      }
      break;
      
    case TOP_PATCHING:    /*HEADER is PATCHING*/
      success = sscanf(buf,"%s%s%s%s%s",header,arg[0],arg[1],arg[2],arg[3]);
      for (i=0;i<success-1;i+=2) {
	
	if (strncmp(arg[i+1],"NONE",4)==0)
	  arg[i+1][0]='\0';
	
	if (strncmp(arg[i],"FIRS",4) == 0) {
	  strcpy(pointer->patch[0],arg[i+1]);
	} else if (strncmp(arg[i],"LAST",4) == 0) {
	  strcpy(pointer->patch[1],arg[i+1]);
	}
      }
      break;
       
    case TOP_DELETE:    /*HEADER is DELETE*/
      success = sscanf(buf,"%s%s",header,headercp);
      hid2 = top_resi_header_cmp(headercp);
      
      switch (hid2) {
	
      case TOP_ATOM: /*2nd header is ATOM*/
	success = sscanf(buf,"%s%s%s",header,headercp,arg[0]);
	d_atom = emalloc("top_d_atom", sizeof(top_d_atom_t));
        strcpy(d_atom->name,arg[0]);
	d_atom->next = NULL;
        if (d_atom_prev == NULL) {
  	  pointer->d_atom = d_atom;
        } else{
	  d_atom_prev->next = d_atom;
        }
        d_atom_prev = d_atom;
	break;
	
      case TOP_DONOR:  /*2nd header is DONOR */
	success = sscanf(buf,"%s%s%s%s",header,headercp,arg[0],arg[1]);
	d_donor = emalloc("top_d_donor",sizeof(top_donor_t));
        strcpy(d_donor->atom[0],arg[0]);
	if(success>3)
	  strcpy(d_donor->atom[1],arg[1]);
     
        d_donor->next = NULL;
        if (d_donor_prev == NULL) {
  	  pointer->d_donor = d_donor;
        } 
        else{
	  d_donor_prev->next = d_donor;
        }
        d_donor_prev = d_donor;
	break;
	
      case TOP_ACCEPTOR:  /*2nd header is ACCE*/
	success = sscanf(buf,"%s%s%s%s",header,headercp,arg[0],arg[1]);
   
	d_accept = emalloc("top_d_acceptor", sizeof(top_acceptor_t));
        strcpy(d_accept->atom[0],arg[0]);
	if(success>3)
	  strcpy(d_accept->atom[1],arg[1]);
     
        d_accept->next = NULL;
        if (d_accept_prev == NULL) {
  	  pointer->d_accept = d_accept;
        } 
        else{
	  d_accept_prev->next = d_accept;
        }
        d_accept_prev = d_accept;
	break;

      case TOP_ANGLE:
	/*2nd header is ANGLE*/
	success = top_scan_string_list(buf,arg);
	i = 2;
	while (i <= success - 3){
	  d_angle = emalloc("top_d_angle", sizeof(top_angle_t));
	  strcpy(d_angle->atom[0],arg[i]);
	  strcpy(d_angle->atom[1],arg[i+1]);
	  strcpy(d_angle->atom[2],arg[i+2]);
	  i += 3;
	  pointer->c_angle++;

	  d_angle->next = NULL;
	  if (d_angle_prev == NULL) {
	    pointer->d_angle = d_angle;
	  } else {
	    d_angle_prev->next = d_angle;
	  }
	  d_angle_prev = d_angle;
       	}
	break;

      case TOP_IMPR:
	/*2nd header is IMPR*/
	success = top_scan_string_list(buf,arg);
	i = 2;
	while (i <= success - 4){
	  d_impr = emalloc("top_d_impr", sizeof(top_impr_t));
	  strcpy(d_impr->atom[0],arg[i]);
	  strcpy(d_impr->atom[1],arg[i+1]);
	  strcpy(d_impr->atom[2],arg[i+2]);
	  strcpy(d_impr->atom[3],arg[i+3]);
	  i += 4;
	  /*pointer->c_angle++;*/

	  d_impr->next = NULL;
	  if (d_impr_prev == NULL) {
	    pointer->d_impr = d_impr;
	  } else {
	    d_impr_prev->next = d_impr;
	  }
	  d_impr_prev = d_impr;
       	}
	break;

      default:
	printf("ERROR: Invalid DELETE command: %s\n", headercp);
	exit(1);
      } /* switch (hid2) */
      break;
    case TOP_IC:
      ic = emalloc("top_ic",sizeof(top_ic_t));
      success = sscanf(buf,"%s%s%s%s%s%lf%lf%lf%lf%lf",header,
		       ic->atom[0],ic->atom[1],ic->atom[2],ic->atom[3],
		       &ic->length[0],&ic->angle[0],&ic->dihedral,
		       &ic->angle[1],&ic->length[1]);
      if (success != 10) {
	printf("ERROR: IC: Invalid file format\n");
	exit(1);
      }
      ic->next = NULL;
      if (ic_prev == NULL)
	pointer->ic = ic;
      else
	ic_prev->next = ic;
      ic_prev = ic;
      break;
    case TOP_DECL:
    case TOP_AUTO:
      break;
    default :
      printf("ERROR: Invalid Header: %s\n",headercp);
      exit(1);
    } /* switch (hid) */
  } /* top_fgets */
}

void top_count()
{
  top_res_t  *r;
  top_mass_t *m;

  _top.n_mass = 0;
  for (m=_top.mass;m!=NULL;m=m->next) {
    _top.n_mass++;
  }

  _top.n_res = 0;
  for (r=_top.res;r!=NULL;r=r->next) {
    _top.n_res++;
  }
  _top.n_pres = 0;
  for (r=_top.pres;r!=NULL;r=r->next) {
    _top.n_pres++;
  }
}

void top_print_info(char *fname)
{
  printf("CHARMM TOP FILE: %s\n", fname);
  printf("  Version %d.%d\n", _top.major_version, _top.minor_version);
  printf("  Number of types of atomic mass : %d\n", _top.n_mass);
  printf("  Number of residues : %d\n", _top.n_res);
  printf("  Number of residues for patching : %d\n", _top.n_pres);
  
}

void top_print_res(top_res_t *pointer){

  top_atom_t *atom;
  top_d_atom_t *d_atom;
  top_bond_t *bond, *d_bond;
  top_angle_t *angle, *d_angle;
  top_dihedral_t *dihedral;
  top_impr_t *impr;
  top_donor_t *donor, *d_donor;
  top_acceptor_t *accept, *d_accept;
  top_ic_t *ic;

  for (atom=pointer->atom;atom!=NULL;atom=atom->next) {
      printf(" ATOM %d %d %.2f %s %s\n", atom->no, atom->group, atom->charge,atom->name,atom->sym);
    }
    for(d_atom=pointer->d_atom;d_atom!=NULL;d_atom=d_atom->next){
      printf(" DEL ATOM %s\n", d_atom->name);
    }
    for(bond=pointer->bond;bond!=NULL;bond=bond->next){
      printf(" BOND %s %s %d\n", bond->atom[0],bond->atom[1],bond->type);
    }
    for(bond=pointer->d_bond;bond!=NULL;bond=bond->next){
      printf(" DEL BOND %s %s %d\n", bond->atom[0],bond->atom[1],bond->type);
    }
    for(angle=pointer->angle;angle!=NULL;angle=angle->next){
      printf(" ANGL %s %s %s\n", angle->atom[0],angle->atom[1],angle->atom[2]);
    }
    for(d_angle=pointer->d_angle;d_angle!=NULL;d_angle=d_angle->next){
      printf(" DEL ANGLE %s %s %s\n", d_angle->atom[0],d_angle->atom[1],d_angle->atom[2]);
    }
    for(dihedral=pointer->dihedral;dihedral!=NULL;dihedral=dihedral->next){
      printf(" DIHE %s %s %s %s\n", dihedral->atom[0],dihedral->atom[1],dihedral->atom[2],dihedral->atom[3]);
    }
    for(impr=pointer->impr;impr!=NULL;impr=impr->next){
      printf(" IMPR %s %s %s %s\n", impr->atom[0],impr->atom[1],impr->atom[2],impr->atom[3]);
    }
    for(donor=pointer->donor;donor!=NULL;donor=donor->next){
      printf(" DONOR %s %s\n", donor->atom[0],donor->atom[1]);
    }
    for(d_donor=pointer->d_donor;d_donor!=NULL;d_donor=d_donor->next){
      printf(" DEL DONOR %s %s\n", d_donor->atom[0],d_donor->atom[1]);
    }
    for(accept=pointer->accept;accept!=NULL;accept=accept->next){
      printf(" ACCEPT %s %s\n", accept->atom[0],accept->atom[1]);
    }
    for(d_accept=pointer->d_accept;d_accept!=NULL;d_accept=d_accept->next){
      printf(" DEL ACCEPT %s %s\n", d_accept->atom[0],d_accept->atom[1]);
    }
    for(ic=pointer->ic;ic!=NULL;ic=ic->next){
      printf(" IC %s %s %s %s %f %f %f %f %f\n",
	     ic->atom[0],ic->atom[1],ic->atom[2],ic->atom[3],
	     ic->length[0],ic->angle[0],ic->dihedral,
	     ic->angle[1],ic->length[1]);
    }
    printf(" PATCHING FIRS %s LAST %s\n",
	   pointer->patch[0], pointer->patch[1]);
    
    printf("\n\n");
}

int top_resi_header_cmp(char *header)
{ 
  if (strncmp(header,"DEFA",4)==0)
    return(TOP_DEFA);
  else if (strncmp(header,"RESI",4)==0)
    return(TOP_RESI);
  else if (strncmp(header,"PRES",4)==0)
    return(TOP_PRES);
  else if (strncmp(header,"END",4)==0)
    return(TOP_END);
  else if (strncmp(header,"GROUP",4)==0)
    return(TOP_GROUP);
  else if (strncmp(header,"ATOM",4)==0)
    return(TOP_ATOM);
  else if (strncmp(header,"BOND",4)==0)
    return(TOP_BOND);
  else if (strncmp(header,"DOUBLE",4)==0)
    return(TOP_DOUBLE);
  else if (strncmp(header,"TRIPLE",4)==0)
    return(TOP_TRIPLE);
  else if (strncmp(header,"IMPR",4)==0)
    return(TOP_IMPR);
  else if (strncmp(header,"IMPH",4)==0)
    return(TOP_IMPR);
  else if (strncmp(header,"CMAP",4)==0)
    return(TOP_CMAP);
  else if (strncmp(header,"DONOR",4)==0)
    return(TOP_DONOR);
  else if (strncmp(header,"ACCEPTOR",4)==0)
    return(TOP_ACCEPTOR);
  else if (strncmp(header,"ANGLE",4)==0)
    return(TOP_ANGLE);
  else if (strncmp(header,"THET",4)==0)
    return(TOP_ANGLE);
  else if (strncmp(header,"DIHE",4)==0)
    return(TOP_DIHE);
  else if (strncmp(header,"PATCHING",4)==0)
    return(TOP_PATCHING);
  else if (strncmp(header,"DELETE",4)==0)
    return(TOP_DELETE);
  else if (strncmp(header,"IC",2)==0)
    return(TOP_IC);
  else if (strncmp(header,"BILD",4)==0)
    return(TOP_IC);
  else if (strncmp(header,"DECL",4)==0)
   return(TOP_DECL);
  else if (strncmp(header,"AUTO",4)==0)
   return(TOP_AUTO);
  else if (strncmp(header,"MASS",4)==0)
   return(TOP_MASS);
  else return(TOP_UNKNOWN);
}


top_res_t *top_search_res(char *name)
{
  top_res_t *r;

  for (r=_top.res;r!=NULL;r=r->next) {
    if (aliascmp(name, r->name) == 0) {
      return r;
    }
  }
  return NULL;
}

top_res_t *top_search_pres(char *name)
{
  top_res_t *r;

  for (r=_top.pres;r!=NULL;r=r->next) {
    if (aliascmp(name, r->name) == 0) {
      return r;
    }
  }
  return NULL;
}

top_mass_t *top_search_mass(top_t *top, char *name)
{
  top_mass_t *mass;

  for (mass=top->mass;mass!=NULL;mass=mass->next) {
    if (strcmp(name, mass->sym) == 0) {
      return mass;
    }
  }
  return NULL;
}

int top_scan_string_list(char *buf1, char arg[MAXNARG][MAXANUM])
{
  char *pbuf;
  int n, pp;
  
  pbuf=buf1; n = 0;
      
  while (1) {
    if (sscanf(pbuf,"%s %n",arg[n],&pp) == 1) {
      n++;
      if (n>=MAXNARG) {
	printf("ERROR: Number of top data exceeds %d\n", MAXNARG);
	printf("%s\n", buf1);
	exit(1);
      }
      pbuf += pp;
    } else break;
  }
  return n;
}

void top_check_dup()
{
  top_check_dup_mass();
  top_check_dup_res();
  top_check_dup_pres();
  printf("\n");
}

void top_check_dup_mass()
{
  top_mass_t *mass1, *mass2;
  for (mass1 = _top.mass; mass1!=NULL; mass1=mass1->next) {
    for (mass2 = mass1->next; mass2!=NULL; mass2=mass2->next) {
      if (strcmp(mass1->sym, mass2->sym) == 0) {
	if (mass1->weight == mass2->weight) {
	  printf("  Warning: Duplicated Mass %s with same paramters\n", mass1->sym);
	} else {
	  printf("  ERROR:   Mulple Mass %s with different parameters\n",mass1->sym);
	  printf("           w=%f,%f\n", mass1->weight, mass2->weight);
	  exit(1);
	}
      }
    }
  }
}

void top_check_atom(top_atom_t *res1_atom, top_atom_t *res2_atom)
{
  top_atom_t *atom1, *atom2;

  for (atom1=res1_atom,atom2=res2_atom;
       atom1!=NULL && atom2!=NULL;
       atom1=atom1->next,  atom2=atom2->next) {
    if (strcmp(atom1->name, atom2->name) != 0 ||
	strcmp(atom1->sym,  atom2->sym)  != 0 ||
	atom1->charge != atom2->charge) {
      printf("   ERROR: Inconsistent ATOM (%s, %s, %f),(%s, %s, %f)\n",
	     atom1->name, atom1->sym, atom1->charge, 
	     atom2->name, atom2->sym, atom2->charge);
      exit(1);
    }
  }
}

void top_check_d_atom(top_d_atom_t *res1_atom, top_d_atom_t *res2_atom)
{
  top_d_atom_t *atom1, *atom2;
  int ok;

  for (atom1=res1_atom; atom1!=NULL; atom1=atom1->next) {
    ok = 0;
    for (atom2=res2_atom; atom2!=NULL; atom2=atom2->next) {
      if (strcmp(atom1->name, atom2->name) == 0) {
	ok = 1;
	break;
      }
    }
    if (!ok) {
      printf("   ERROR: Inconsistent DELETE ATOM %s\n",
	     atom1->name);
      exit(1);
    }
  }
}


void top_check_bond(top_bond_t *res1_bond, top_bond_t *res2_bond)
{
  top_bond_t *bond1, *bond2;
  int ok;
  
  for (bond1=res1_bond; bond1!=NULL; bond1=bond1->next) {
    ok = 0;
    for (bond2=res2_bond; bond2!=NULL; bond2=bond2->next) {
      if ((strcmp(bond1->atom[0], bond2->atom[0]) == 0 &&
	   strcmp(bond1->atom[1], bond2->atom[1]) == 0) ||
	  (strcmp(bond1->atom[0], bond2->atom[1]) == 0 &&
	   strcmp(bond1->atom[1], bond2->atom[0]) == 0)) {
	ok = 1;
	break;
      }
    }
    if (!ok) {
      printf("   ERROR: Inconsistent BOND %s-%s\n",
	     bond1->atom[0], bond1->atom[1]);
      exit(1);
    }
  }
}

void top_check_angle(top_angle_t *res1_angle, top_angle_t *res2_angle)
{
  top_angle_t *angle1, *angle2;
  int ok;

  for (angle1=res1_angle; angle1!=NULL; angle1=angle1->next) {
    ok = 0;
    for (angle2=res2_angle; angle2!=NULL; angle2=angle2->next) {
      if ((strcmp(angle1->atom[0], angle2->atom[0]) == 0 &&
	   strcmp(angle1->atom[1], angle2->atom[1]) == 0 &&
	   strcmp(angle1->atom[2], angle2->atom[2]) == 0) ||
	  (strcmp(angle1->atom[0], angle2->atom[2]) == 0 &&
	   strcmp(angle1->atom[1], angle2->atom[1]) == 0 &&
	   strcmp(angle1->atom[2], angle2->atom[0]) == 0)) {
	ok = 1;
	break;
      }
    }
    if (!ok) {
      printf("   ERROR: Inconsistent ANGLE %s-%s-%s\n",
	     angle1->atom[0], angle1->atom[1], angle1->atom[2]);
      exit(1);
    }
  }
}

void top_check_dihedral(top_dihedral_t *res1_dihedral, top_dihedral_t *res2_dihedral)
{
  top_dihedral_t *dihedral1, *dihedral2;
  int ok;

  for (dihedral1=res1_dihedral; dihedral1!=NULL; dihedral1=dihedral1->next) {
    ok = 0;
    for (dihedral2=res2_dihedral; dihedral2!=NULL; dihedral2=dihedral2->next) {
      if ((strcmp(dihedral1->atom[0], dihedral2->atom[0]) == 0 &&
	   strcmp(dihedral1->atom[1], dihedral2->atom[1]) == 0 &&
	   strcmp(dihedral1->atom[2], dihedral2->atom[2]) == 0 &&
	   strcmp(dihedral1->atom[3], dihedral2->atom[3]) == 0) ||
	  (strcmp(dihedral1->atom[0], dihedral2->atom[3]) == 0 &&
	   strcmp(dihedral1->atom[1], dihedral2->atom[2]) == 0 &&
	   strcmp(dihedral1->atom[2], dihedral2->atom[1]) == 0 &&
	   strcmp(dihedral1->atom[3], dihedral2->atom[0]) == 0)) {
	ok = 1;
	break;
      }
    }
    if (!ok) {
      printf("   ERROR: Inconsistent DIHEDRAL %s-%s-%s-%s\n",
	     dihedral1->atom[0], dihedral1->atom[1], dihedral1->atom[2], dihedral1->atom[3]);
      exit(1);
    }
  }
}

void top_check_impr(top_impr_t *res1_impr, top_impr_t *res2_impr)
{
  top_impr_t *impr1, *impr2;
  int ok;

  for (impr1=res1_impr; impr1!=NULL; impr1=impr1->next) {
    ok = 0;
    for (impr2=res2_impr; impr2!=NULL; impr2=impr2->next) {
      if (strcmp(impr1->atom[0], impr2->atom[0]) == 0 &&
	  strcmp(impr1->atom[1], impr2->atom[1]) == 0 &&
	  strcmp(impr1->atom[2], impr2->atom[2]) == 0 &&
	  strcmp(impr1->atom[3], impr2->atom[3]) == 0) {
	ok = 1;
	break;
      }
    }
    if (!ok) {
      printf("   ERROR: Inconsistent IMPR %s-%s-%s-%s\n",
	     impr1->atom[0], impr1->atom[1], impr1->atom[2], impr1->atom[3]);
      exit(1);
    }
  }
}

void top_check_cmap(top_cmap_t *res1_cmap, top_cmap_t *res2_cmap)
{
  top_cmap_t *cmap1, *cmap2;
  int i,ok, ok_i;

  for (cmap1=res1_cmap; cmap1!=NULL; cmap1=cmap1->next) {
    ok = 0;
    for (cmap2=res2_cmap; cmap2!=NULL; cmap2=cmap2->next) {
      ok_i = 1;
      for (i=0;i<8;i++) {
	if (strcmp(cmap1->atom[i], cmap2->atom[i]) != 0) {
	  ok_i = 0;
	}
      }
      if (ok_i) {
	ok = 1;
	break;
      }
    }
    if (!ok) {
      printf("   ERROR: Inconsistent CMAP %s-%s-%s-%s, %s-%s-%s-%s\n",
	     cmap1->atom[0], cmap1->atom[1], cmap1->atom[2], cmap1->atom[3],
	     cmap1->atom[4], cmap1->atom[5], cmap1->atom[6], cmap1->atom[7]);
      exit(1);
    }
  }
}


void top_check_dup_res()
{
  top_res_t *res1, *res2;
  top_atom_t *atom1, *atom2;
  int i;

  for (res1 = _top.res; res1!=NULL; res1=res1->next) {
    for (res2 = res1->next; res2!=NULL; res2=res2->next) {
      if (strcmp(res1->name, res2->name) == 0) {
	printf("  Warning: Residue %s duplicated\n", res1->name);

	/* atom */
	top_check_atom(res1->atom, res2->atom);

	/* bond */
	top_check_bond(res1->bond, res2->bond);
	top_check_bond(res2->bond, res1->bond);

	/* angle */
	top_check_angle(res1->angle, res2->angle);
	top_check_angle(res2->angle, res1->angle);

	/* dihedral */
	top_check_dihedral(res1->dihedral, res2->dihedral);
	top_check_dihedral(res2->dihedral, res1->dihedral);

	/* impr */
	top_check_impr(res1->impr, res2->impr);
	top_check_impr(res2->impr, res1->impr);

	/* cmap */
	top_check_cmap(res1->cmap, res2->cmap);
	top_check_cmap(res2->cmap, res1->cmap);

#if 0
	/* donor */
	top_check_donor(res1->donor, res2->donor);
	top_check_donor(res2->donor, res1->donor);

	/* acceptor */
	top_check_acceptor(res1->acceptor, res2->acceptor);
	top_check_acceptor(res2->acceptor, res1->acceptor);
#endif

      }
    }
  }
}

void top_check_dup_pres()
{
  top_res_t *pres1, *pres2;
  for (pres1 = _top.pres; pres1!=NULL; pres1=pres1->next) {
    for (pres2 = pres1->next; pres2!=NULL; pres2=pres2->next) {
      if (strcmp(pres1->name, pres2->name) == 0) {
	printf("  Warning: PRES %s duplicated\n", pres1->name);

	/* atom */
	top_check_atom(pres1->atom, pres2->atom);
	top_check_d_atom(pres1->d_atom, pres2->d_atom);
	top_check_d_atom(pres2->d_atom, pres1->d_atom);

	/* bond */
	top_check_bond(pres1->bond, pres2->bond);
	top_check_bond(pres2->bond, pres1->bond);
	top_check_bond(pres1->d_bond, pres2->d_bond);
	top_check_bond(pres2->d_bond, pres1->d_bond);

	/* angle */
	top_check_angle(pres1->angle, pres2->angle);
	top_check_angle(pres2->angle, pres1->angle);
	top_check_angle(pres1->d_angle, pres2->d_angle);
	top_check_angle(pres2->d_angle, pres1->d_angle);

	/* dihedral */
	top_check_dihedral(pres1->dihedral, pres2->dihedral);
	top_check_dihedral(pres2->dihedral, pres1->dihedral);
	top_check_dihedral(pres1->d_dihedral, pres2->d_dihedral);
	top_check_dihedral(pres2->d_dihedral, pres1->d_dihedral);

	/* impr */
	top_check_impr(pres1->impr, pres2->impr);
	top_check_impr(pres2->impr, pres1->impr);
	top_check_impr(pres1->d_impr, pres2->d_impr);
	top_check_impr(pres2->d_impr, pres1->d_impr);

	/* cmap */
	top_check_cmap(pres1->cmap, pres2->cmap);
	top_check_cmap(pres2->cmap, pres1->cmap);

#if 0
	/* donor */
	top_check_donor(pres1->donor, pres2->donor);
	top_check_donor(pres2->donor, pres1->donor);

	/* acceptor */
	top_check_acceptor(pres1->acceptor, pres2->acceptor);
	top_check_acceptor(pres2->acceptor, pres1->acceptor);
#endif
	
      }
    }
  }
}
