/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifndef _CHARMM_PAR_H_INCLUDED
#define _CHARMM_PAR_H_INCLUDED

#define PAR_UNKNOWN   0
#define PAR_BOND      1
#define PAR_ANGLE     2
#define PAR_DIHEDRAL  3
#define PAR_IMPROPER  4
#define PAR_NONBONDED 5
#define PAR_NBFIX     6
#define PAR_HBOND     7
#define PAR_END       8
#define PAR_CMAP      9
#define PAR_EOF      10

#define MAXSYM 7

typedef struct par_bond {
  int no;
  char sym1[MAXSYM];
  char sym2[MAXSYM];
  double kb;
  double b0;
  int check;
  struct par_bond *next;
} par_bond_t;

typedef struct par_angle {
  int no;
  char sym1[MAXSYM];
  char sym2[MAXSYM];
  char sym3[MAXSYM];
  double Ktheta;
  double Theta0;
  double Kub;
  double S0;
  int check;
  struct par_angle *next;
} par_angle_t;

typedef struct par_dihedral {
  int no, n;
  char sym1[MAXSYM];
  char sym2[MAXSYM];
  char sym3[MAXSYM];
  char sym4[MAXSYM];
  double Kchi;
  double delta;
  int check;
  /* int multi; */
  int wild_card;
  struct par_dihedral *next;
  struct par_dihedral *multi;  /* for dup check */
  int first_term;              /* for dup check */
} par_dihedral_t;

typedef struct par_impr {
  int no;
  char sym1[MAXSYM];
  char sym2[MAXSYM];
  char sym3[MAXSYM];
  char sym4[MAXSYM];
  double Kpsi, psi, psi0 ;
  int check;
  int wild_card;
  struct par_impr *next ;
} par_impr_t;

typedef struct par_cmap {
  int no;
  char sym[8][MAXSYM];
  double *map;
  int ndiv;
  int check;
  struct par_cmap *next ;
} par_cmap_t;

typedef struct par_nonbonded {
  int no;
  char sym1[MAXSYM];
  double ignored;
  double eps;
  double rmin2;
  double ignored2;
  double eps14;
  double rmin214;
  int check;
  struct par_nonbonded *next;
} par_nonbonded_t;

typedef struct par_nbfix {
  int no;
  char sym1[MAXSYM];
  char sym2[MAXSYM];
  double eps;
  double rmin;
  double eps14;
  double rmin14;
  struct par_nbfix *next;
} par_nbfix_t;

typedef struct par_nbpair {
  int no;
  double eps;
  double rmin;
  double eps14;
  double rmin14;
  par_nonbonded_t *nb1, *nb2;
  par_nbfix_t *nbfix;
} par_nbpair_t;

typedef struct par {
  int major_version, minor_version;

  int n_bond, n_angle, n_dihedral, n_impr, n_cmap, n_nonbonded, n_nbfix;
  
  par_bond_t *bond;
  par_angle_t *angle;
  par_dihedral_t *dihedral, *dihedral_new;
  par_impr_t *impr;
  par_cmap_t *cmap;
  par_nonbonded_t *nonbonded;
  par_nbfix_t *nbfix;

  int n_used_bond, n_used_angle, n_used_dihedral, n_used_impr, n_used_cmap, n_used_nonbonded;
  par_nbpair_t **nbpair;

  double cuthb;
  
} par_t;


void par_init();
void par_read_file(char fname[][FNAME_LEN], int n_fname);
void par_read(FILE *fp);
int par_header_cmp(char *buf);
char *par_header_name(int id);
void par_read_error(char *buf, int header);

int par_read_bond(FILE *fp, char *buf);
int par_read_angle(FILE *fp, char *buf);
int par_read_dihedral(FILE *fp, char *buf);
int par_read_impr(FILE *fp, char *buf);
int par_read_cmap(FILE *fp, char *buf);
int par_read_nonbonded(FILE *fp, char *buf);
int par_read_nbfix(FILE *fp, char *buf);
int par_read_hbond(FILE *fp, char *buf);
void par_print_info(char *fname);

par_bond_t      *par_search_bond(par_t *par, char *sym1, char *sym2);
par_angle_t     *par_search_angle(par_t *par, char *sym1, char *sym2, char *sym3);
int par_match_dihedral(par_dihedral_t *dihedral, char *sym1, char *sym2, char *sym3, char *sym4);
int par_match_dihedral_wild_card(par_dihedral_t *dihedral, char *sym1, char *sym2, char *sym3, char *sym4);
par_impr_t      *par_search_impr(par_t *par, char *sym1, char *sym2, char *sym3, char *sym4);
par_impr_t      *par_search_impr_wild_card(par_t *par, char *sym1, char *sym2, char *sym3, char *sym4);
par_cmap_t *par_search_cmap(par_t *par, 
			    char *sym1, char *sym2, char *sym3, char *sym4,
			    char *sym5, char *sym6, char *sym7, char *sym8);
par_nonbonded_t *par_search_nonbonded(par_t *par, char *sym1);
par_nbfix_t     *par_search_nbfix(par_t *par, char *sym1, char *sym2);

void par_count_used(par_t *par);
void par_make_nonbonded_pair(par_t *par);

void par_check_dup();
void par_check_dup_bond();
void par_check_dup_angle();
void par_check_dup_dihedral();
void par_check_dup_impr();
void par_check_dup_cmap();
void par_check_dup_nonbonded();
void par_check_dup_nbfix();

extern par_t _par;

#endif /* _CHARMM_PAR_H_INCLUDED */
