/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "util.h"
#include "charmm_par.h"

void par_init()
{
  /* BOND */
  _par.bond = NULL;
  _par.n_bond = 0;

  /* ANGLE */
  _par.angle = NULL;
  _par.n_angle = 0;

  /* DIHEDRAL */
  _par.dihedral = NULL;
  _par.n_dihedral = 0;
  
  /* IMPROPER */
  _par.impr = NULL;
  _par.n_impr = 0;

  /* CMAP */
  _par.cmap = NULL;
  _par.n_cmap = 0;

  /* NONBONDED */
  _par.nonbonded = NULL;
  _par.n_nonbonded = 0;

  /* NBFIX */
  _par.nbfix = NULL ;
  _par.n_nbfix = 0;

  /* HBOND */
  _par.cuthb = 0.0;
}

void par_read_file(char fname[][FNAME_LEN], int n_fname)
{
  FILE *fp;
  char fname2[FNAME_LEN];
  int i;
  
  for (i=0;i<n_fname;i++) {
    fp = env_fopen(fname[i], "r");
    if (fp==NULL) {
      fprintf(stderr,"ERROR: %s: No such file\n", fname[i]);
      exit(1);
    }
    par_read(fp);
    if (i==0) {
      strcpy(fname2,fname[i]);
    } else {
      sprintf(fname2,"+ %s", fname[i]);
    }
    par_print_info(fname2);
    par_check_dup();
    fclose(fp);
  }
}

#define PAR_BUFLEN 1000

void par_read(FILE *fp)
{
  char buf[PAR_BUFLEN];
  int header;
    
  while (par_fgets(buf, PAR_BUFLEN, fp)) {
    if ((header = par_header_cmp(buf))!=PAR_UNKNOWN) {
      break;
    }
  }

  while (header != PAR_END && header != PAR_EOF) {
    switch (header) {
    case PAR_BOND:
      header = par_read_bond(fp, buf);
      break;
    case PAR_ANGLE:
      header = par_read_angle(fp, buf);
      break;
    case PAR_DIHEDRAL:
      header = par_read_dihedral(fp, buf);
      break;
    case PAR_IMPROPER:
      header = par_read_impr(fp, buf);
      break;
    case PAR_CMAP:
      header = par_read_cmap(fp, buf);
      break;
    case PAR_NONBONDED:
      header = par_read_nonbonded(fp, buf);
      break;
    case PAR_NBFIX:
      header = par_read_nbfix(fp, buf);
      break;
    case PAR_HBOND:
      header = par_read_hbond(fp, buf);
      break;
    case PAR_END:
      break;
    }
  }
}

struct _s_par_header {
  char *name;
  int n, id;
};

static struct _s_par_header _par_header[] = {
  "BOND"     , 4, PAR_BOND,
  "ANGLE"    , 4, PAR_ANGLE,
  "THETAS"   , 4, PAR_ANGLE,
  "DIHEDRAL" , 4, PAR_DIHEDRAL,
  "PHI"      , 3, PAR_DIHEDRAL,
  "IMPROPER" , 4, PAR_IMPROPER,
  "IMPHI"    , 4, PAR_IMPROPER,
  "CMAP"     , 4, PAR_CMAP,
  "NONBONDED", 4, PAR_NONBONDED,
  "NBFIX"    , 4, PAR_NBFIX,
  "HBOND"    , 4, PAR_HBOND,
  "END"      , 3, PAR_END,
  NULL       , 0, PAR_UNKNOWN 
};

int par_header_cmp(char *buf)
{
  int i;
  for (i=0;_par_header[i].name!=NULL;i++) {
    if (strncmp(buf,_par_header[i].name, _par_header[i].n) == 0) {
      return _par_header[i].id;
    }
  }
  return PAR_UNKNOWN;
}

char *par_header_name(int id)
{
  int i;
  for (i=0;_par_header[i].name!=NULL;i++) {
    if (_par_header[i].id == id) {
      return _par_header[i].name;
    }
  }
  return "";
}

void par_read_error(char *buf, int header)
{
  printf("ERROR: Invalid line in %s of charmm par file\n%s\n",
	 par_header_name(header), buf);
  exit(1);
}

int par_read_bond(FILE *fp, char *buf)
{
  char sym1[MAXSYM], sym2[MAXSYM];
  int header;
  int no;
  double kb, b0;
  par_bond_t *bond, *bond_prev;

  /*
  do{
    par_fgets(buf, PAR_BUFLEN, fp);
  }while(strncmp(buf, "BOND", 4) != 0);
  */
  if (_par.bond == NULL) {
    bond_prev = NULL;
  } else {
    for (bond = _par.bond; bond!=NULL; bond=bond->next) {
      bond_prev = bond;
    }
  }
  
  while (par_fgets(buf, PAR_BUFLEN, fp)) {
    
    if(header = par_header_cmp(buf)) return header;

    if(sscanf(buf, "%s%s%lf%lf",sym1,sym2, &kb, &b0) != 4) {
      par_read_error(buf, PAR_BOND);
    }

    /* This line is BOND. */
    bond = emalloc("par_bond", sizeof(par_bond_t));
    
    strcpy(bond->sym1, sym1);
    strcpy(bond->sym2, sym2);
    bond->kb=kb;
    bond->b0=b0;
    bond->check=0;
    bond->next = NULL;
    if (bond_prev == NULL) {
      _par.bond = bond;
    } else {
      bond_prev->next = bond;
    }
    bond_prev = bond;
    _par.n_bond++;
  }
  /* DEBUG  
  for (bond=_par.bond;bond!=NULL;bond=bond->next) {
    printf("%s %s %lf %lf\n", bond->sym1, bond->sym2, bond->kb, bond->b0);
  }
  */
  return PAR_EOF;
}

int par_read_angle(FILE *fp, char *buf)
{
  char sym1[MAXSYM], sym2[MAXSYM], sym3[MAXSYM];
  int header;
  int no;
  double Ktheta, Theta0, Kub, S0;
  par_angle_t *angle, *angle_prev;

  if (_par.angle == NULL) {
    angle_prev = NULL;
  } else {
    for (angle = _par.angle; angle!=NULL; angle=angle->next) {
      angle_prev = angle;
    }
  }
  
  while(par_fgets(buf, PAR_BUFLEN, fp)){
    
    if(header = par_header_cmp(buf)) return header;
    
    if(sscanf(buf, "%s%s%s%lf%lf%lf%lf",sym1,sym2, sym3, &Ktheta, &Theta0, &Kub, &S0 ) != 7) {
      if(sscanf(buf, "%s%s%s%lf%lf%lf%lf",sym1,sym2, sym3, &Ktheta, &Theta0, &Kub, &S0 ) != 5) {
	par_read_error(buf, PAR_ANGLE);
      } else {
        Kub = 0.0;
        S0 = 0.0;
      }
    }
    /* This line is ANGLE. */
    
    angle = emalloc("read_angle", sizeof(par_angle_t));
    strcpy(angle->sym1, sym1);
    strcpy(angle->sym2, sym2);
    strcpy(angle->sym3, sym3);
    angle->Ktheta=Ktheta;
    angle->Theta0=Theta0;
    angle->Kub=Kub;
    angle->S0=S0;
    angle->check=0;
    angle->next = NULL;
    if (angle_prev == NULL) {
      _par.angle = angle;
    } else {
      angle_prev->next = angle;
    }
    angle_prev = angle;
    _par.n_angle++;
  }
  /* DEBUG  
  for (angle=_par.angle;angle!=NULL;angle=angle->next) {
    printf("%s %s %s %lf %lf %lf %lf \n", angle->sym1, angle->sym2, angle->sym3, angle->Ktheta, angle->Theta0, angle->Kub, angle->S0 );
  }
  */
  return PAR_EOF;
}



int par_read_dihedral(FILE *fp, char *buf)
{
  char sym1[MAXSYM], sym2[MAXSYM], sym3[MAXSYM], sym4[MAXSYM];
  int header;
  int no, n, first;
  double Kchi, delta;
  par_dihedral_t *dihedral, *dihedral_prev;
  
  if (_par.dihedral == NULL) {
    dihedral_prev = NULL;
  } else {
    for (dihedral = _par.dihedral; dihedral!=NULL; dihedral=dihedral->next) {
      dihedral_prev = dihedral;
    }
  }
  first = 1;
  while(par_fgets(buf, PAR_BUFLEN, fp)){

    if(header = par_header_cmp(buf)) return header;
    
    if(sscanf(buf, "%s%s%s%s%lf%d%lf",sym1, sym2, sym3, sym4, &Kchi, &n, &delta ) != 7) {
      par_read_error(buf, PAR_DIHEDRAL);
    }

    /* This line is DIHEDRAL. */
    dihedral = emalloc("read_dihedral",sizeof(par_dihedral_t));
    
    strcpy(dihedral->sym1, sym1);
    strcpy(dihedral->sym2, sym2);
    strcpy(dihedral->sym3, sym3);
    strcpy(dihedral->sym4, sym4);
    dihedral->Kchi=Kchi;
    dihedral->n=n;
    dihedral->delta=delta;
    dihedral->check=0;
    /* dihedral->multi = 0; */
    if (strcmp(sym1,"X") == 0 ||
	strcmp(sym2,"X") == 0 ||
	strcmp(sym3,"X") == 0 ||
	strcmp(sym4,"X") == 0)
      dihedral->wild_card = 1;
    else
      dihedral->wild_card = 0;

    if (first) {
      _par.dihedral_new = dihedral;
      first = 0;
    }

    dihedral->next = NULL;
    if (dihedral_prev == NULL) {
      _par.dihedral = dihedral;
    } else {
      dihedral_prev->next = dihedral;
      /*
      if ((strcmp(dihedral_prev->sym1, sym1) == 0 &&
	   strcmp(dihedral_prev->sym2, sym2) == 0 &&
	   strcmp(dihedral_prev->sym3, sym3) == 0 &&
	   strcmp(dihedral_prev->sym4, sym4) == 0) ||
	  (strcmp(dihedral_prev->sym1, sym4) == 0 &&
	   strcmp(dihedral_prev->sym2, sym3) == 0 &&
	   strcmp(dihedral_prev->sym3, sym2) == 0 &&
	   strcmp(dihedral_prev->sym4, sym1) == 0)) {
	dihedral_prev->multi = 1;
      }
      */
    }
    dihedral_prev = dihedral;
    _par.n_dihedral++;
  }

#if 0  
  /* DEBUG */
  for (dihedral=_par.dihedral;dihedral!=NULL;dihedral=dihedral->next) {
    if (dihedral->multi) {
      printf("dihedral: multiple term: %-3s-%-3s-%-3s-%-3s\n",
	     dihedral->sym1, dihedral->sym2, dihedral->sym3, dihedral->sym4);
    }
    /*
    printf("%s %s %s %s %lf %d %lf \n", dihedral->sym1, dihedral->sym2, dihedral->sym3, dihedral->sym4, dihedral->Kchi, dihedral->n, dihedral->delta );
    */
  }
#endif

  return PAR_EOF;
}


int par_read_impr(FILE *fp, char *buf)
{
  char sym1[MAXSYM], sym2[MAXSYM], sym3[MAXSYM], sym4[MAXSYM];
  int header;
  int no;
  double Kpsi, psi0, psi;
  par_impr_t *impr, *impr_prev;

  if (_par.impr == NULL) {
    impr_prev = NULL;
  } else {
    for (impr = _par.impr; impr!=NULL; impr=impr->next) {
      impr_prev = impr;
    }
  }
  
  while(par_fgets(buf, PAR_BUFLEN, fp)){

    if(header = par_header_cmp(buf)) return header;

    if(sscanf(buf, "%s%s%s%s%lf%lf%lf",sym1, sym2, sym3, sym4, &Kpsi, &psi, &psi0 ) != 7)  {
      par_read_error(buf, PAR_IMPROPER);
    }
     
    /* This line is IMPR. */
    impr = emalloc("read_impr",sizeof(par_impr_t));
    strcpy(impr->sym1, sym1);
    strcpy(impr->sym2, sym2);
    strcpy(impr->sym3, sym3);
    strcpy(impr->sym4, sym4);
    impr->Kpsi=Kpsi;
    impr->psi=psi;
    impr->psi0=psi0;
    impr->check=0;
    if (strcmp(sym1,"X") == 0 ||
	strcmp(sym2,"X") == 0 ||
	strcmp(sym3,"X") == 0 ||
	strcmp(sym4,"X") == 0)
      impr->wild_card = 1;
    else
      impr->wild_card = 0;
    
    impr->next = NULL;
    if (impr_prev == NULL) {
      _par.impr = impr;
    } else {
      impr_prev->next = impr;
    }
    impr_prev = impr;
    _par.n_impr++;
  }
  /* DEBUG  
  for (impr=_par.impr;impr!=NULL;impr=impr->next) {
    printf("%s %s %s %s %lf %lf %lf \n", impr->sym1, impr->sym2, impr->sym3, impr->sym4, impr->Kpsi, impr->psi, impr->psi0 );
  }
  */
  return PAR_EOF;
}

int par_read_cmap(FILE *fp, char *buf)
{
  char sym[8][MAXSYM], *pbuf;
  int header;
  int no;
  int i,j,pp,ndiv;
  par_cmap_t *cmap, *cmap_prev;

  if (_par.cmap == NULL) {
    cmap_prev = NULL;
  } else {
    for (cmap = _par.cmap; cmap!=NULL; cmap=cmap->next) {
      cmap_prev = cmap;
    }
  }
  
  while(par_fgets(buf, PAR_BUFLEN, fp)){

    if(header = par_header_cmp(buf)) return header;
    /*if(header = par_header_cmp(buf)) break;*/

    if(sscanf(buf, "%s%s%s%s%s%s%s%s%d",
	      sym[0], sym[1], sym[2], sym[3], 
	      sym[4], sym[5], sym[6], sym[7], 
	      &ndiv ) != 9)  {
      par_read_error(buf, PAR_CMAP);
    }
     
    /* This line is CMAP. */
    cmap = emalloc("read_cmap",sizeof(par_cmap_t));
    for (j=0;j<8;j++)
      strcpy(cmap->sym[j], sym[j]);
    cmap->ndiv = ndiv;

    cmap->map = emalloc("read_cmap",sizeof(double)*ndiv*ndiv);

    par_fgets(buf, PAR_BUFLEN, fp);
    pbuf = buf;
    for (j=0;j<ndiv*ndiv;j++) {
      if (sscanf(pbuf,"%lf%n",&cmap->map[j],&pp) != 1) {
	par_fgets(buf, PAR_BUFLEN, fp);
	pbuf = buf;
	if (sscanf(pbuf,"%lf%n",&cmap->map[j],&pp) != 1) {
	  printf("ERROR: invalid cmap format\n");
	  exit(1);
	}
      }
      pbuf += pp;
    }

    cmap->check=0;
    
    cmap->next = NULL;
    if (cmap_prev == NULL) {
      _par.cmap = cmap;
    } else {
      cmap_prev->next = cmap;
    }
    cmap_prev = cmap;
    _par.n_cmap++;
  }

  for (cmap=_par.cmap;cmap!=NULL;cmap=cmap->next) {
    for(j=0;j<8;j++)
      printf("%s ", cmap->sym[j]);
    printf("%d\n", cmap->ndiv);
    for(i=0;i<ndiv;i++) {
      for(j=0;j<ndiv;j++) {
	printf("%f ",cmap->map[i*ndiv+j]);
	if ((j+1)%5==0)
	  printf("\n");
      }
      printf("\n\n");
    }
  }

  return PAR_EOF;
}

int par_read_nonbonded(FILE *fp, char *buf)
{
  char sym1[MAXSYM];
  int header;
  int no;
  double ignored, epsilon, rmin2, ignored2, eps14, rmin214;
  par_nonbonded_t *nonbonded, *nonbonded_prev;
  
  if (_par.nonbonded == NULL) {
    nonbonded_prev = NULL;
  } else {
    for (nonbonded = _par.nonbonded; nonbonded!=NULL; nonbonded=nonbonded->next) {
      nonbonded_prev = nonbonded;
    }
  }
  
  while(par_fgets(buf, PAR_BUFLEN, fp)){
    
    if(header = par_header_cmp(buf)) return header;
    
    if(sscanf(buf, "%s%lf%lf%lf%lf%lf%lf",sym1, &ignored, &epsilon, &rmin2, &ignored2, &eps14, &rmin214 ) != 7) {
      if (sscanf(buf, "%s%lf%lf%lf%lf%lf%lf",sym1, &ignored, &epsilon, &rmin2, &ignored2, &eps14, &rmin214 ) >= 4) {
	ignored2 = 0.0;
	eps14 = epsilon;
	rmin214 = rmin2;
      } else {
	par_read_error(buf, PAR_NONBONDED);
      }
    }
    /* This line is NONBONDED. */
    nonbonded = emalloc("read_nonbonded", sizeof(par_nonbonded_t));
    strcpy(nonbonded->sym1, sym1);
    nonbonded->ignored=ignored;
    nonbonded->eps=epsilon;
    nonbonded->rmin2=rmin2;
    nonbonded->ignored2=ignored2;
    nonbonded->eps14=eps14;
    nonbonded->rmin214=rmin214;
    nonbonded->check=0;
    nonbonded->next = NULL;
    if (nonbonded_prev == NULL) {
      _par.nonbonded = nonbonded;
    } else {
      nonbonded_prev->next = nonbonded;
    }
    nonbonded_prev = nonbonded;
    _par.n_nonbonded++;
  }
  /* DEBUG  
  for (nonbonded=_par.nonbonded;nonbonded!=NULL;nonbonded=nonbonded->next) {
    printf("%s %lf %lf %lf %lf %lf %lf \n", nonbonded->sym1, nonbonded->ignored, nonbonded->epsilon, nonbonded->rmin2, nonbonded->ignored2, nonbonded->eps14, nonbonded->rmin214 );
  }
  */
  return PAR_EOF;
}

int par_read_nbfix(FILE *fp, char *buf)
{
  char sym1[MAXSYM], sym2[MAXSYM] ;
  int header;
  int no;
  double emin, rmin;
  double emin14, rmin14;
  par_nbfix_t *nbfix, *nbfix_prev;

  if (_par.nbfix == NULL) {
    nbfix_prev = NULL;
  } else {
    for (nbfix = _par.nbfix; nbfix!=NULL; nbfix=nbfix->next) {
      nbfix_prev = nbfix;
    }
  }
  
  while(par_fgets(buf, PAR_BUFLEN, fp)){
    if(header = par_header_cmp(buf)) return header;

    if(sscanf(buf, "%s%s%lf%lf%lf%lf", sym1, sym2, &emin, &rmin, &emin14, &rmin14) != 6) {
      if(sscanf(buf, "%s%s%lf%lf", sym1, sym2, &emin, &rmin) != 4 ) {
	par_read_error(buf, PAR_NBFIX);
      } else {
	emin14 = emin;
	rmin14 = rmin;
      }
    }

    /* This line is NBFIX */
    
    nbfix = emalloc("read_nbfix",sizeof(par_nbfix_t));
    strcpy(nbfix->sym1, sym1);
    strcpy(nbfix->sym2, sym2);
    nbfix->eps=emin;
    nbfix->rmin=rmin;
    nbfix->eps14=emin14;
    nbfix->rmin14=rmin14;
    nbfix->next = NULL;
    if (nbfix_prev == NULL) {
      _par.nbfix = nbfix;
    } else {
      nbfix_prev->next = nbfix;
    }
    nbfix_prev = nbfix;
    _par.n_nbfix++;
  }

  /* DEBUG
  for (nbfix=_par.nbfix;nbfix!=NULL;nbfix=nbfix->next) {
    printf("%s %s %lf %lf \n", nbfix->sym1, nbfix->sym2, nbfix->emin, nbfix->rmin );
  }
  */
  return PAR_EOF;
}

int par_read_hbond(FILE *fp, char *buf)
{
  char buf2[PAR_BUFLEN];
  double f;
  int header;
  int n;

  if (sscanf(buf,"%*s %s%n",  buf2, &n) == 1) {
    if (strncmp(buf2,"CUTHB", 4) == 0) {
      if (sscanf(&buf[n],"%lf", &f) == 1) {
	_par.cuthb = f;
      }
    }
  }

  /*printf("cuthb=%f\n", _par.cuthb);*/
  
  while(par_fgets(buf, PAR_BUFLEN, fp)) {
    if(header = par_header_cmp(buf)) return header;
  }
  return PAR_EOF;
}

void par_print_info(char *fname)
{
  printf("CHARMM PAR FILE: %s\n", fname);
  printf("  Number of bond types:  %d\n", _par.n_bond);
  printf("  Number of angle types: %d\n", _par.n_angle);
  printf("  Number of dihedral types: %d\n", _par.n_dihedral);
  printf("  Number of improper dihedral types: %d\n", _par.n_impr);
  printf("  Number of cmap dihedral types: %d\n", _par.n_cmap);
  printf("  Number of nonbonded atom types: %d\n", _par.n_nonbonded);
  printf("  Number of modified nonbonded atom pairs: %d\n", _par.n_nbfix);
}

int parcmp(char *sym1, char *sym2)
{
  return strcmp(sym1, sym2);
}

int parcmpstar(char *sym1, char *sym2)
{
  char *star;
  int n;

  if ((star=strchr(sym2, '*'))!=NULL) {
    n = (int) (star-sym2);
    return strncmp(sym1,sym2,n);
  }
  return strcmp(sym1, sym2);
}

int parcmp_X(char *sym1, char *sym2)
{
  if (sym2[0]=='X') return 0;
  else return strcmp(sym1, sym2);
}

par_bond_t *par_search_bond(par_t *par, char *sym1, char *sym2)
{
  par_bond_t *bond;

  for (bond=par->bond;bond!=NULL;bond=bond->next) {
    if ((parcmp(sym1, bond->sym1) == 0 &&
	 parcmp(sym2, bond->sym2) == 0) ||
	(parcmp(sym1, bond->sym2) == 0 &&
	 parcmp(sym2, bond->sym1) == 0))
      return bond;
  }
  return NULL;
}

par_angle_t *par_search_angle(par_t *par, char *sym1, char *sym2, char *sym3)
{
  par_angle_t *angle;

  for (angle=par->angle;angle!=NULL;angle=angle->next) {
    if (parcmp(sym2, angle->sym2) == 0) {
      if ((parcmp(sym1, angle->sym1) == 0 &&
	   parcmp(sym3, angle->sym3) == 0) ||
	  (parcmp(sym1, angle->sym3) == 0 &&
	   parcmp(sym3, angle->sym1) == 0))
	return angle;
    }
  }
  return NULL;
}


/*
par_dihedral_t *par_search_dihedral(par_t *par, char *sym1, char *sym2, char *sym3, char *sym4)
{
  par_dihedral_t *dihedral;

  for (dihedral=par->dihedral;dihedral!=NULL;dihedral=dihedral->next) {
    if ((parcmp(sym1, dihedral->sym1) == 0 &&
	 parcmp(sym2, dihedral->sym2) == 0 &&
	 parcmp(sym3, dihedral->sym3) == 0 &&
	 parcmp(sym4, dihedral->sym4) == 0) ||
	(parcmp(sym1, dihedral->sym4) == 0 &&
	 parcmp(sym2, dihedral->sym3) == 0 &&
	 parcmp(sym3, dihedral->sym2) == 0 &&
	 parcmp(sym4, dihedral->sym1) == 0))
      return dihedral;
  }
  return NULL;
}
*/

int par_match_dihedral(par_dihedral_t *dihedral, char *sym1, char *sym2, char *sym3, char *sym4)
{
  if (dihedral->wild_card) return 1;
  if ((parcmp(sym1, dihedral->sym1) == 0 &&
       parcmp(sym2, dihedral->sym2) == 0 &&
       parcmp(sym3, dihedral->sym3) == 0 &&
       parcmp(sym4, dihedral->sym4) == 0) ||
      (parcmp(sym1, dihedral->sym4) == 0 &&
       parcmp(sym2, dihedral->sym3) == 0 &&
       parcmp(sym3, dihedral->sym2) == 0 &&
       parcmp(sym4, dihedral->sym1) == 0))
    return 0;
  else
    return 1;
}

int par_match_dihedral_wild_card(par_dihedral_t *dihedral, char *sym1, char *sym2, char *sym3, char *sym4)
{
  if (!dihedral->wild_card) return 1;
  if ((parcmp_X(sym1, dihedral->sym1) == 0 &&
       parcmp_X(sym2, dihedral->sym2) == 0 &&
       parcmp_X(sym3, dihedral->sym3) == 0 &&
       parcmp_X(sym4, dihedral->sym4) == 0) ||
      (parcmp_X(sym1, dihedral->sym4) == 0 &&
       parcmp_X(sym2, dihedral->sym3) == 0 &&
       parcmp_X(sym3, dihedral->sym2) == 0 &&
       parcmp_X(sym4, dihedral->sym1) == 0))
    return 0;
  else
    return 1;
}


par_impr_t *par_search_impr(par_t *par, char *sym1, char *sym2, char *sym3, char *sym4)
{
  par_impr_t *impr, *ret_impr;

  ret_impr = NULL;
  for (impr=par->impr;impr!=NULL;impr=impr->next) {
    if (impr->wild_card) continue;
    if ((parcmp(sym1, impr->sym1) == 0 &&
	 parcmp(sym2, impr->sym2) == 0 &&
	 parcmp(sym3, impr->sym3) == 0 &&
	 parcmp(sym4, impr->sym4) == 0) ||
	(parcmp(sym1, impr->sym4) == 0 &&
	 parcmp(sym2, impr->sym3) == 0 &&
	 parcmp(sym3, impr->sym2) == 0 &&
	 parcmp(sym4, impr->sym1) == 0)) {
      if (ret_impr!=NULL) {
	printf("ERROR: match improper %s-%s-%s-%s to more than one:\n",
		sym1,sym2,sym3,sym4);
	printf("       %s-%s-%s-%s, %s-%s-%s-%s\n",
	       impr->sym1,impr->sym2,impr->sym3,impr->sym4,
	       ret_impr->sym1,ret_impr->sym2,ret_impr->sym3,ret_impr->sym4);
	printf("       multiple terms of improper are not supported.\n");
	exit(1);
      }
      ret_impr = impr;
    }
  }
  return ret_impr;
}

par_impr_t *par_search_impr_wild_card(par_t *par, char *sym1, char *sym2, char *sym3, char *sym4)
{
  par_impr_t *impr, *ret_impr;

  ret_impr = NULL;
  for (impr=par->impr;impr!=NULL;impr=impr->next) {
    if (!impr->wild_card) continue;
    if ((parcmp_X(sym1, impr->sym1) == 0 &&
	 parcmp_X(sym2, impr->sym2) == 0 &&
	 parcmp_X(sym3, impr->sym3) == 0 &&
	 parcmp_X(sym4, impr->sym4) == 0) ||
	(parcmp_X(sym1, impr->sym4) == 0 &&
	 parcmp_X(sym2, impr->sym3) == 0 &&
	 parcmp_X(sym3, impr->sym2) == 0 &&
	 parcmp_X(sym4, impr->sym1) == 0)) {
      if (ret_impr!=NULL) {
	printf("ERROR: match improper %s-%s-%s-%s to more than one:\n",
		sym1,sym2,sym3,sym4);
	printf("       %s-%s-%s-%s, %s-%s-%s-%s\n",
	       impr->sym1,impr->sym2,impr->sym3,impr->sym4,
	       ret_impr->sym1,ret_impr->sym2,ret_impr->sym3,ret_impr->sym4);
	printf("       multiple terms of improper are not supported.\n");
	exit(1);
      }
      ret_impr = impr;
    }
  }
  return ret_impr;
}


par_cmap_t *par_search_cmap(par_t *par, 
			    char *sym1, char *sym2, char *sym3, char *sym4,
			    char *sym5, char *sym6, char *sym7, char *sym8)
{
  par_cmap_t *cmap, *ret_cmap;

  ret_cmap = NULL;
  for (cmap=par->cmap;cmap!=NULL;cmap=cmap->next) {
    if ((parcmp(sym1, cmap->sym[0]) == 0 &&
	 parcmp(sym2, cmap->sym[1]) == 0 &&
	 parcmp(sym3, cmap->sym[2]) == 0 &&
	 parcmp(sym4, cmap->sym[3]) == 0 &&
	 parcmp(sym5, cmap->sym[4]) == 0 &&
	 parcmp(sym6, cmap->sym[5]) == 0 &&
	 parcmp(sym7, cmap->sym[6]) == 0 &&
	 parcmp(sym8, cmap->sym[7]) == 0)) {
      if (ret_cmap!=NULL) {
	printf("ERROR: match cmap %s-%s-%s-%s %s-%s-%s-%s to more than one:\n",
		sym1,sym2,sym3,sym4,
		sym5,sym6,sym7,sym8);
	printf("       multiple terms of cmap not supported.\n");
	exit(1);
      }
      ret_cmap = cmap;
    }
  }
  return ret_cmap;
}




par_nonbonded_t *par_search_nonbonded(par_t *par, char *sym1)
{
  par_nonbonded_t *nonbonded;

  for (nonbonded=par->nonbonded;nonbonded!=NULL;nonbonded=nonbonded->next) {
    if (parcmpstar(sym1, nonbonded->sym1) == 0)
      return nonbonded;
  }
  return NULL;
}

par_nbfix_t *par_search_nbfix(par_t *par, char *sym1, char *sym2)
{
  par_nbfix_t *nbfix;

  for (nbfix=par->nbfix;nbfix!=NULL;nbfix=nbfix->next) {
    if ((parcmp(sym1, nbfix->sym1) == 0 &&
	 parcmp(sym2, nbfix->sym2) == 0) ||
	(parcmp(sym1, nbfix->sym2) == 0 &&
	 parcmp(sym2, nbfix->sym1) == 0))
      return nbfix;
  }
  return NULL;
}


void par_count_used(par_t *par)
{
  par_bond_t *bond;
  par_angle_t *angle;
  par_dihedral_t *dihedral;
  par_impr_t *impr;
  par_cmap_t *cmap;
  par_nonbonded_t *nonbonded;

  par->n_used_bond = 0;
  for (bond=par->bond; bond!=NULL; bond=bond->next) {
    if (bond->check) {
      bond->no = par->n_used_bond++;
    }
  }
  par->n_used_angle = 0;
  for (angle=par->angle; angle!=NULL; angle=angle->next) {
    if (angle->check) {
      angle->no = par->n_used_angle++;
    }
  }
  par->n_used_dihedral = 0;
  for (dihedral=par->dihedral; dihedral!=NULL; dihedral=dihedral->next) {
    if (dihedral->check) {
      dihedral->no = par->n_used_dihedral++;
    }
  }
  par->n_used_impr = 0;
  for (impr=par->impr; impr!=NULL; impr=impr->next) {
    if (impr->check) {
      impr->no = par->n_used_impr++;
    }
  }
  par->n_used_cmap = 0;
  for (cmap=par->cmap; cmap!=NULL; cmap=cmap->next) {
    if (cmap->check) {
      cmap->no = par->n_used_cmap++;
    }
  }
  par->n_used_nonbonded = 0;
  for (nonbonded=par->nonbonded; nonbonded!=NULL; nonbonded=nonbonded->next) {
    if (nonbonded->check) {
      nonbonded->no = par->n_used_nonbonded++;
    }
  }
}

void par_make_nonbonded_pair(par_t *par)
{
  int i, j, no;
  par_nonbonded_t *nb1, *nb2;
  par_nbpair_t *nbp;
  
  par->nbpair=emalloc("make_nonbonded_pair",sizeof(par_nbpair_t*)*par->n_used_nonbonded);
  for (i=0;i<par->n_used_nonbonded;i++) {
    par->nbpair[i]=emalloc("make_nonbonded_pair",sizeof(par_nbpair_t)*par->n_used_nonbonded);
  }
  for (nb1=par->nonbonded; nb1!=NULL; nb1=nb1->next) {
    if (nb1->check) {
      for (nb2=par->nonbonded; nb2!=NULL; nb2=nb2->next) {
	if (nb2->check) {
	  nbp=&par->nbpair[nb1->no][nb2->no];
	  nbp->nb1 = nb1;
	  nbp->nb2 = nb2;
	  nbp->nbfix=par_search_nbfix(par,nb1->sym1,nb2->sym1);
	  if (nbp->nbfix) {
	    nbp->eps = fabs(nbp->nbfix->eps);
	    nbp->rmin = nbp->nbfix->rmin;
	    nbp->eps14 = fabs(nbp->nbfix->eps14);
	    nbp->rmin14 = nbp->nbfix->rmin14;
	  } else {
	    nbp->eps    = sqrt(nb1->eps*nb2->eps);
	    nbp->rmin   = nb1->rmin2+nb2->rmin2;
	    nbp->eps14  = sqrt(nb1->eps14*nb2->eps14);
	    nbp->rmin14 = nb1->rmin214+nb2->rmin214;
	  }
	}
      }
    }
  }
  no = 0;
  for (i=0;i<par->n_used_nonbonded;i++) {
    for (j=i;j<par->n_used_nonbonded;j++) {
      par->nbpair[i][j].no = no;
      par->nbpair[j][i].no = no;
      no++;
    }
  }
}

void par_check_dup()
{
  par_check_dup_bond();
  par_check_dup_angle();
  par_check_dup_dihedral();
  par_check_dup_impr();
  par_check_dup_cmap();
  par_check_dup_nonbonded();
  par_check_dup_nbfix();
  printf("\n");
}

void par_check_dup_bond()
{
  par_bond_t *bond1, *bond2;

  for (bond1 = _par.bond; bond1!=NULL; bond1=bond1->next) {
    for (bond2 = bond1->next; bond2!=NULL; bond2=bond2->next) {
      if ((strcmp(bond1->sym1, bond2->sym1) == 0 &&
	   strcmp(bond1->sym2, bond2->sym2) == 0) ||
	  (strcmp(bond1->sym1, bond2->sym2) == 0 &&
	   strcmp(bond1->sym2, bond2->sym1) == 0)) {
	if (bond1->kb == bond2->kb &&
	    bond1->b0 == bond2->b0) {
	  printf("  Warning: Duplicated BOND %s-%s with same parameters\n", 
		 bond1->sym1,bond1->sym2);
	} else {
	  printf("  ERROR:   Multiple BOND %s-%s with different parameters\n",
		 bond1->sym1,bond1->sym2);
	  printf("           k=%f,%f\n",  bond1->kb,bond2->kb);
	  printf("           r0=%f,%f\n", bond1->b0,bond2->b0);
	}
      }
    }
  }
}

void par_check_dup_angle()
{
  par_angle_t *angle1, *angle2;

  for (angle1 = _par.angle; angle1!=NULL; angle1=angle1->next) {
    for (angle2 = angle1->next; angle2!=NULL; angle2=angle2->next) {
      if ((strcmp(angle1->sym1, angle2->sym1) == 0 &&
	   strcmp(angle1->sym2, angle2->sym2) == 0 &&
	   strcmp(angle1->sym3, angle2->sym3) == 0) ||
	  (strcmp(angle1->sym1, angle2->sym3) == 0 &&
	   strcmp(angle1->sym2, angle2->sym2) == 0 &&
	   strcmp(angle1->sym3, angle2->sym1) == 0)) {
	if (angle1->Ktheta == angle2->Ktheta &&
	    angle1->Theta0 == angle2->Theta0 &&
	    angle1->Kub == angle2->Kub &&
	    angle1->S0  == angle2->S0) { 
	  printf("  Warning: Duplicated ANGLE %s-%s-%s with same parameters\n", 
		 angle1->sym1,angle1->sym2,angle1->sym3);
	} else {
	  printf("  ERROR:   Multiple ANGLE %s-%s-%s with different parameters\n", 
		 angle1->sym1,angle1->sym2,angle1->sym3);
	  printf("           k=%f,%f\n", angle1->Ktheta, angle2->Ktheta);
	  printf("           theta0=%f,%f\n", angle1->Theta0, angle2->Theta0);
	  printf("           Kub=%f,%f\n", angle1->Kub, angle2->Kub);
	  printf("           KubR0=%f,%f\n", angle1->S0, angle2->S0);
	  exit(1);
	}
      }
    }
  }
}

int par_check_dihedral_cmp(par_dihedral_t *dihedral1, par_dihedral_t *dihedral2)
{
 return ((strcmp(dihedral1->sym1, dihedral2->sym1) == 0 &&
	  strcmp(dihedral1->sym2, dihedral2->sym2) == 0 &&
	  strcmp(dihedral1->sym3, dihedral2->sym3) == 0 &&
	  strcmp(dihedral1->sym4, dihedral2->sym4) == 0) ||
	 (strcmp(dihedral1->sym1, dihedral2->sym4) == 0 &&
	  strcmp(dihedral1->sym2, dihedral2->sym3) == 0 &&
	  strcmp(dihedral1->sym3, dihedral2->sym2) == 0 &&
	  strcmp(dihedral1->sym4, dihedral2->sym1) == 0));
}

void par_check_dup_dihedral()
{
  par_dihedral_t *dihedral1, *dihedral2, *dd1, *dd2;
  int ok, all_ok;

  for (dihedral1 = _par.dihedral_new; dihedral1!=NULL; dihedral1=dihedral1->next) {
    dihedral1->multi = NULL;
    dihedral1->first_term = 1;
  }
  for (dihedral1 = _par.dihedral_new; dihedral1!=NULL; dihedral1=dihedral1->next) {
    if (dihedral1->multi != NULL || dihedral1->wild_card) continue;

    for (dihedral2 = dihedral1->next; dihedral2!=NULL; dihedral2=dihedral2->next) {
      if (dihedral2->multi != NULL || dihedral2->wild_card) continue;

      if (par_check_dihedral_cmp(dihedral1,dihedral2)) {
	dihedral2->multi = dihedral1->multi;
	dihedral1->multi = dihedral2;
	dihedral2->first_term = 0;
      }
    }
  }
      
      
  for (dihedral1 = _par.dihedral; dihedral1!=_par.dihedral_new; dihedral1=dihedral1->next) {
    if (!dihedral1->first_term) continue;
    
    for (dihedral2 = _par.dihedral_new; dihedral2!=NULL; dihedral2=dihedral2->next) {
      if (!dihedral2->first_term) continue;
   
      if (par_check_dihedral_cmp(dihedral1,dihedral2)) {
      
	all_ok = 1;
	for (dd1=dihedral1;dd1!=NULL;dd1=dd1->multi) {
	  ok = 0;
	  for (dd2=dihedral1;dd2!=NULL;dd2=dd2->multi) {
	    if (dd1->Kchi == dd2->Kchi &&
		dd1->delta == dd2->delta &&
		dd1->n == dd2->n) {
	      ok = 1;
	      break;
	    }
	  }
	  if (ok==0) {
	    all_ok = 0;
	    break;
	  }
	}
	
	if (all_ok) {
	  printf("  Warning: Duplicated DIHEDRAL %s-%s-%s-%s with same parameters\n", 
		 dihedral1->sym1,dihedral1->sym2,dihedral1->sym3,dihedral1->sym4);
	} else {
	  printf("  ERROR:   Multiple DIHEDRAL %s-%s-%s-%s with different parameters\n", 
		 dihedral1->sym1,dihedral1->sym2,dihedral1->sym3,dihedral1->sym4);
	  printf("    In previous file\n");
	  for (dd1=dihedral1;dd1!=NULL;dd1=dd1->multi) {
	    printf("           k=%f, delta=%f, n=%d\n", 
		   dd1->Kchi, dd1->delta, dd1->n);
	  }
	  printf("    In appended file\n");
	  for (dd2=dihedral2;dd2!=NULL;dd2=dd2->multi) {
	    printf("           k=%f, delta=%f, n=%d\n", 
		   dd2->Kchi, dd2->delta, dd2->n);
	  }
	  exit(1);
	}
      }
    }
  }
}


void par_check_dup_impr()
{
  par_impr_t *impr1, *impr2;

  for (impr1 = _par.impr; impr1!=NULL; impr1=impr1->next) {
    for (impr2 = impr1->next; impr2!=NULL; impr2=impr2->next) {
      if (strcmp(impr1->sym1, impr2->sym1) == 0 &&
	  strcmp(impr1->sym2, impr2->sym2) == 0 &&
	  strcmp(impr1->sym3, impr2->sym3) == 0 &&
	  strcmp(impr1->sym4, impr2->sym4) == 0) {
	if (impr1->Kpsi == impr2->Kpsi &&
	    impr1->psi0 == impr2->psi0) {
	  printf("  Warning: Duplicated IMPR %s-%s-%s-%s with same parameters\n", 
		 impr1->sym1,impr1->sym2,impr1->sym3,impr1->sym4);
	} else {
	  printf("  ERROR:   Multiple IMPR %s-%s-%s-%s with same parameters\n", 
		 impr1->sym1,impr1->sym2,impr1->sym3,impr1->sym4);
	  printf("           k=%f,%f\n",impr1->Kpsi, impr2->Kpsi);
	  printf("           psi0=%f,%f\n",impr1->psi0, impr2->psi0);
	  exit(1);
	}
      }
    }
  }
}

void par_check_dup_cmap()
{
  par_cmap_t *cmap1, *cmap2;
  int i, ok, ok2;

  for (cmap1 = _par.cmap; cmap1!=NULL; cmap1=cmap1->next) {
    for (cmap2 = cmap1->next; cmap2!=NULL; cmap2=cmap2->next) {
      ok = 0;
      for (i=0;i<8;i++) {
	if (strcmp(cmap1->sym[i], cmap2->sym[i]) != 0) {
	  ok = 1;
	  break;
	}
      }
      if (!ok) {
	if (cmap1->ndiv == cmap2->ndiv) {
	  ok2 = 1;
	  for (i=0;i<cmap1->ndiv*cmap1->ndiv;i++) {
	    if (cmap1->map[i] != cmap2->map[i]) {
	      ok2 = 0;
	    }
	  }
	} else {
	  ok2 = 0;
	}
	if (ok2) {
	  printf("  Warning: Duplicated CMAP %s-%s-%s-%s, %s-%s-%s-%s with same parameters\n", 
		 cmap1->sym[0],cmap1->sym[1],cmap1->sym[2],cmap1->sym[3],
		 cmap1->sym[4],cmap1->sym[5],cmap1->sym[6],cmap1->sym[7]);
	} else {
	  printf("  ERROR:   Multiple CMAP %s-%s-%s-%s, %s-%s-%s-%s with different parameters\n", 
		 cmap1->sym[0],cmap1->sym[1],cmap1->sym[2],cmap1->sym[3],
		 cmap1->sym[4],cmap1->sym[5],cmap1->sym[6],cmap1->sym[7]);
	  exit(1);
	}
      }
    }
  }
}

void par_check_dup_nonbonded()
{
  par_nonbonded_t *nonbonded1, *nonbonded2;

  for (nonbonded1 = _par.nonbonded; nonbonded1!=NULL; nonbonded1=nonbonded1->next) {
    for (nonbonded2 = nonbonded1->next; nonbonded2!=NULL; nonbonded2=nonbonded2->next) {
      if (strcmp(nonbonded1->sym1, nonbonded2->sym1) == 0) {
	if (nonbonded1->eps == nonbonded2->eps &&
	    nonbonded1->rmin2 == nonbonded2->rmin2 &&
	    nonbonded1->eps14 == nonbonded2->eps14 &&
	    nonbonded1->rmin214 == nonbonded2->rmin214) {
	  printf("  Warning: Duplicated NONBONDED %s with same parameters\n", 
		 nonbonded1->sym1);
	} else {
	  printf("  ERROR:   Multiple NONBONDED %s with different parameters\n", 
		 nonbonded1->sym1);
	  printf("           eps= %f, %f\n", nonbonded1->eps, nonbonded2->eps);
	  printf("           rmin2= %f, %f\n", nonbonded1->rmin2, nonbonded2->rmin2);
	  printf("           eps14= %f, %f\n", nonbonded1->eps14, nonbonded2->eps14);
	  printf("           rmin214= %f, %f\n", nonbonded1->rmin214, nonbonded2->rmin214);
	  exit(1);
	}
      }
    }
  }
}

void par_check_dup_nbfix()
{
  par_nbfix_t *nbfix1, *nbfix2;

  for (nbfix1 = _par.nbfix; nbfix1!=NULL; nbfix1=nbfix1->next) {
    for (nbfix2 = nbfix1->next; nbfix2!=NULL; nbfix2=nbfix2->next) {
      if ((strcmp(nbfix1->sym1, nbfix2->sym1) == 0 &&
	   strcmp(nbfix1->sym2, nbfix2->sym2) == 0) ||
	  (strcmp(nbfix1->sym1, nbfix2->sym2) == 0 &&
	   strcmp(nbfix1->sym2, nbfix2->sym1) == 0)) {
	if (nbfix1->eps == nbfix2->eps &&
	    nbfix1->rmin == nbfix2->rmin &&
	    nbfix1->eps14 == nbfix2->eps14 &&
	    nbfix1->rmin14 == nbfix2->rmin14) {
	  printf("  Warning: Duplicated NBFIX %s-%s with same parameters\n", 
		 nbfix1->sym1, nbfix1->sym2);
	} else {
	  printf("  ERROR:   Multiple NBFIX %s-%s with different parameters\n", 
		 nbfix1->sym1, nbfix1->sym2);
	  printf("           eps=%f,%f\n", nbfix1->eps, nbfix2->eps);
	  printf("           rmin=%f,%f\n", nbfix1->rmin, nbfix2->rmin);
	  printf("           eps14=%f,%f\n", nbfix1->eps14, nbfix2->eps14);
	  printf("           rmin14=%f,%f\n", nbfix1->rmin14, nbfix2->rmin14);
	  exit(1);
	}
      }
    }
  }
}
