/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>

#include "util.h"
#include "charmm_par.h"
#include "pdb.h"
#include "charmm_top.h"
#include "mdat.h"
#include "patch.h"
#include "config.h"
#include "align_axis.h"
#include "matrix.h"

void align_axis(mdat_t *mdat)
{
  double m[3][3], u[3][3], ip[3];

  printf("Align principal axes of molecules.\n");

  aln_centering(mdat);
  aln_moment_of_inertia(mdat, m);
  
  /* debug
     print_mat33(m); */
  
  diag33(m, ip, u);
  if (_cfg.align_axis == ALIGN_AXIS_Z) {
    sort_eigen(3, ip, &u[0][0], 1);
  } else {
    sort_eigen(3, ip, &u[0][0], 0);
  }
  
  /* fprintf(stderr, "x = %f, y = %f, z = %f\n",ip[0],ip[1],ip[2]); */
  aln_correct_mirror(u);
  aln_rotate(mdat, u);
  
  aln_moment_of_inertia(mdat, m);
  
  /* debug
  print_mat33(m);
  */
  
  if (_cfg.align_axis == ALIGN_DIAGONAL) {
    printf("Align the longest axis to diagonal of a cube\n");
    aln_diagonal_rotate(mdat);
  }
}

void aln_centering(mdat_t *mdat)
{
  mdat_atom_t *a;
  double cx,cy,cz,tw;

  cx = cy = cz = 0.0;
  tw = 0.0;
  for (a=mdat->atom;a!=NULL;a=a->next) {
    cx += a->mass->weight * a->x;
    cy += a->mass->weight * a->y;
    cz += a->mass->weight * a->z;
    tw += a->mass->weight;
  }
  cx /= tw;
  cy /= tw;
  cz /= tw;
  for (a=mdat->atom;a!=NULL;a=a->next) {
    a->x -= cx;
    a->y -= cy;
    a->z -= cz;
  }
}

void aln_moment_of_inertia(mdat_t *mdat, double m[3][3])
{
  mdat_atom_t *a;
  int j,k;
  double mm, x[3];
  
  for (j=0;j<3;j++)
    for (k=0;k<3;k++)
      m[j][k] = 0.0;
  
  for (a=mdat->atom;a!=NULL;a=a->next) {
    x[0] = a->x;
    x[1] = a->y;
    x[2] = a->z;
    for (j=0;j<3;j++)
      for (k=0;k<3;k++)
	m[j][k] -= a->mass->weight * x[j]* x[k];
  }
  mm = -m[0][0] - m[1][1] - m[2][2];
  for(j=0;j<3;j++)
    m[j][j] += mm;
}

void aln_correct_mirror(double u[3][3])
{
  double v0[3], v1[3], v2[3], cross[3], dot;
  int i;

  for (i=0;i<3;i++) {
    v0[i] = u[i][0];
    v1[i] = u[i][1];
    v2[i] = u[i][2];
  }
  cross3(cross, v0, v1);
  dot = dot3(v2, cross);
  /* fprintf(stderr,"%f\n",dot); */
  if (dot < 0.0) {
    u[0][2] *= -1;
    u[1][2] *= -1;
    u[2][2] *= -1;
  }
}

void aln_rotate(mdat_t *mdat, double u[3][3])
{
  mdat_atom_t *a;
  double new_x[3], x[3];
  int i,j,k;

  for (a=mdat->atom;a!=NULL;a=a->next) {
    x[0] = a->x;
    x[1] = a->y;
    x[2] = a->z;

    for (j=0;j<3;j++) {
      new_x[j] = 0.0;
      for (k=0;k<3;k++) {
	new_x[j] += u[k][j] * x[k];
      }
    }
    a->x = new_x[0];
    a->y = new_x[1];
    a->z = new_x[2];
  }
}

void cross3(double a[3],double b[3],double c[3])
{
  a[0] = b[1]*c[2] - b[2]*c[1];
  a[1] = b[2]*c[0] - b[0]*c[2];
  a[2] = b[0]*c[1] - b[1]*c[0];
}

double dot3(double a[3], double b[3])
{
  return a[0]*b[0]+a[1]*b[1]+a[2]*b[2];
}

void normalize3(double a[3])
{
  double l;
  l = sqrt(a[0]*a[0]+a[1]*a[1]+a[2]*a[2]);
  a[0]/=l; a[1]/=l; a[2]/=l;
}

void aln_diagonal_rotate(mdat_t *mdat)
{
  mdat_atom_t *ma;
  double a[3],b[3],c[3], m[3][3], x_new[3], x[3], len;
  int i,j,k;

  for (j=0;j<3;j++) a[j]= 1.0/sqrt(3.0);
  c[0] = a[0];  c[1] = -a[1];  c[2] = -a[2];
  
  cross3(b,c,a);
  normalize3(b);
  cross3(c,a,b);
  normalize3(c);
  for (j=0;j<3;j++) {
    m[j][0] = a[j];
    m[j][1] = b[j];
    m[j][2] = c[j];
  }
  for (ma=mdat->atom;ma!=NULL;ma=ma->next) {
    x[0] = ma->x;
    x[1] = ma->y;
    x[2] = ma->z;
    
    for (j=0;j<3;j++) {
      x_new[j] = 0.0;
      for (k=0;k<3;k++) {
	x_new[j] += m[j][k] * x[k];
      }
    }
    
    ma->x = x_new[0];
    ma->y = x_new[1];
    ma->z = x_new[2];
  }
}

