/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifndef _RIGID_MOL_H_INCLUDED
#define _RIGID_MOL_H_INCLUDED

#ifndef _MISC_H_INCLUDED
#include "misc.h"
#endif

#ifndef _ATOM_H_INCLUDED
#include "atom.h"
#endif

typedef struct _s_quan {
  double qw, qx, qy, qz;
  
  /* double vqw, vqx, vqy, vqz; */
} QUAN;

typedef struct _s_rmol_type_atom {
  char name[5], sym[5];
  int    na,nb,nc;
  double length, angle, dihedral;
  double w;
  VEC x;
} RMOL_TYPE_ATOM;

typedef struct _s_mol_type {
  double weight;             /* all weight */
  VEC ip;                    /* moment of inertia  */
  double Izy, Izx;

  int n_atom;                /* number of atoms */
  
  RMOL_TYPE_ATOM *atom;      /* atom coordinates */

  char (*name)[5];           /* name */
  int n_name;

  int flag, n_mol;           /* flag and number of molecules */

} RMOL_TYPE;

#define RMOL_TYPE_AMINO  1

typedef struct _s_rigid_mol {
  RMOL_TYPE *type;             /* molecular type */
  int parent_atom;          /* number of starting atom */
  int n_atom;                 /* number of atom */

  VEC rg;                     /* center of gravity */
  VEC vg;                     /*    derivative */

  QUAN q;                     /* quaternion parameters */
  VEC  l;                     /* angular momentum in the room coordinate */

  VEC wp;                     /* angular velocity */
  VEC wp_half;                /* angular velocity at dt + dt/2 */
  
  VEC fg;                     /* force */
  VEC torq;		      /* torque */

  int iex;                    /* number of ex_system */
  int cell;                   /* cell id */

  double p[4];                /* for no squish */
  
#ifdef MPI_SDMD
  int node_rmol_n;
#endif  

} RIGID_MOL;

typedef struct {
  VEC rg;                     /* center of gravity */
  VEC vg;                     /*    derivative */
  QUAN q;                     /* quaternion parameters */
  VEC  l;                     /* angular momentum in the room coordinate*/
} RMOL_CRD;

typedef struct _s_molcular_data {
  int n_mol, n_mol6, n_mol5;
  RIGID_MOL *mol;
  
  RMOL_CRD *crd;
  
  int n_type, n_alloc_type;
  RMOL_TYPE *mol_type;

#ifdef MPI_SDMD
  int node_rmol_h;
#endif  
  
} RMOL_DATA;

/*************************************************************/
/*  for initilization                                        */
/*************************************************************/
#define TIP3P_INDEX   0
#define TIP3P_NAME    "IP3"
#define TIP3P_BOND    0.9572
#define TIP3P_ANGLE   104.52
#define TIP3P_CHARGE  0.417
#define TIP3P_LJ_A    582.0e3
#define TIP3P_LJ_C    595.0

#define TIP4P_INDEX   1
#define TIP4P_NAME    "IP4 "
#define TIP4P_BOND    0.9572
#define TIP4P_ANGLE   104.52
#define TIP4P_SHIFT   0.15
#define TIP4P_CHARGE  0.52
#define TIP4P_LJ_A    600.0e3
#define TIP4P_LJ_C    610.0

#define SPC_INDEX     2
#define SPC_NAME     "SPC "
#define SPC_BOND      1.0
#define SPC_ANGLE     109.47
#define SPC_CHARGE    0.41
#define SPC_LJ_A      629.4e3
#define SPC_LJ_C      625.5

/*****  QUAN  ******/
void QUAN_room_to_mol(QUAN *q, VEC *from, VEC *to);
void QUAN_mol_to_room(QUAN *q, VEC *from, VEC *to);

/*****  RIGID_MOL  ******/
void RIGID_MOL_leapfrog(RIGID_MOL *mol, double dt);
void RIGID_MOL_force_and_torque(RIGID_MOL *mol, ATOM_DATA *ad);
void RIGID_MOL_time_integration_v1(RIGID_MOL *mol, double dt);
void RIGID_MOL_time_integration_p_rg(RIGID_MOL *mol, double dt);
void RIGID_MOL_time_integration_p_rg_NPT(RIGID_MOL *mol, double AA2, double BB);
void RIGID_MOL_time_integration_p_rg_NPT_full(RIGID_MOL *mol, double Vg_vec[3][3],
					      double AA2[3], double BB[3]);
void RIGID_MOL_time_integration_p_rot(RIGID_MOL *mol, double dt);
void RIGID_MOL_time_integration_v2(RIGID_MOL *mol, double dt);
void RIGID_MOL_set_direction(RIGID_MOL *mol, double c);

/* nosquish routines */
void RIGID_MOL_time_integration_v1_no_squish(RIGID_MOL *mol, double dt);
void RIGID_MOL_time_integration_p_rot_no_squish(RIGID_MOL *mol, double dt, int Mrot);
void RIGID_MOL_time_integration_v2_no_squish(RIGID_MOL *mol, double dt);

/*****  RMOL_TYPE  ******/
void RMOL_TYPE_internal_to_xyz(RMOL_TYPE *mt);
void RMOL_TYPE_normalize_eigv(double eig[3], double eigv[3][3]);
void RMOL_TYPE_moment_of_inertia(RMOL_TYPE *mt);
void RMOL_TYPE_init_wat_3P(RMOL_TYPE *mt, char *name,
			   char *o_name, char *h1_name, char *h2_name,
			   double bond, double angle,
			   double wo, double wh);
void RMOL_TYPE_init_wat_4P(RMOL_TYPE *mt, char *name,
			   char *o_name, char *h1_name, char *h2_name,
			   char *m_name,
			   double bond, double angle, double shift,
			   double wo, double wh);

/*****  RMOL_DATA  ******/
void RMOL_DATA_init(RMOL_DATA *);
void RMOL_DATA_finalize(RMOL_DATA *);
int  RMOL_DATA_n_mol(RMOL_DATA *md);
int  RMOL_DATA_search_mol_type_name(RMOL_DATA *md, char *name);
void RMOL_DATA_set_rigid_mol_type(RMOL_DATA *md, char *name, char *fname);
void RMOL_DATA_crd_to_rg_quan(RMOL_DATA *md, ATOM_DATA *ad);
void RMOL_DATA_set_velocity_zero(RMOL_DATA *md);
void RMOL_DATA_Maxwell_velocity(RMOL_DATA *md, double temperature);
void RMOL_DATA_scale_velocity(RMOL_DATA *md, double *scale_tr, double *scale_rot);
void RMOL_DATA_scale_velocity_full(RMOL_DATA *md, double scale_tr[MAX_EX_SYSTEM][3][3],
				   double scale_rot[MAX_EX_SYSTEM]);
int  RMOL_DATA_read_data(RMOL_DATA *md, ATOM_DATA *, FILE *fp);

void RMOL_DATA_setup(RMOL_DATA *md, ATOM_DATA *ad, BOND_DATA *bd,
		     ANGLE_DATA *ag, char *fname);
void RMOL_DATA_read_library(RMOL_DATA *md, ATOM_DATA *ad, char *fname);
int  RMOL_DATA_read_library_fp(RMOL_DATA *md, ATOM_DATA *ad, FILE *fp);
void RMOL_DATA_make_rmol(RMOL_DATA *md, ATOM_DATA *ad);
void RMOL_DATA_initialize_types(RMOL_DATA *md, ATOM_DATA *ad);
void RMOL_DATA_setup_water(RMOL_DATA *md, ATOM_DATA *ad);
void RMOL_DATA_setup_hydrogen(RMOL_DATA *md, ATOM_DATA *ad, BOND_DATA *bd, ANGLE_DATA *ag);
int  RMOL_DATA_read_crd(RMOL_DATA *md, FILE *fp, int skip);
void RMOL_DATA_write_crd(RMOL_DATA *md, FILE *fp);
void RMOL_DATA_print(RMOL_DATA *md, FILE *fp);
void RMOL_DATA_mol_to_room(RMOL_DATA *md, ATOM_DATA *ad);
void RMOL_DATA_mol_to_room_all(RMOL_DATA *md, ATOM_DATA *ad);
void RMOL_DATA_mol_to_room_with_check(RMOL_DATA *md, ATOM_DATA *ad);
void RMOL_DATA_kene(RMOL_DATA *md, double *kene_t, double *kene_r, double *kene_arr);
void RMOL_DATA_kene_full(RMOL_DATA *md, double *kene_t, double *kene_r, int n_ex_system,
			 double kene_tr_arr[MAX_EX_SYSTEM][3][3], double kene_rot_arr[MAX_EX_SYSTEM]);
void RMOL_DATA_leapfrog(RMOL_DATA *md, ATOM_DATA *ad, double dt);
void RMOL_DATA_init_time0(RMOL_DATA *md, ATOM_DATA *ad);

void RMOL_DATA_time_integration_v1(RMOL_DATA *md, ATOM_DATA *ad, double dt);
void RMOL_DATA_time_integration_p(RMOL_DATA *md, ATOM_DATA *ad, double dt);
void RMOL_DATA_time_integration_p_NPT(RMOL_DATA *md, ATOM_DATA *ad, double dt,
				      double AA2, double BB);
void RMOL_DATA_time_integration_p_NPT_full(RMOL_DATA *md, ATOM_DATA *ad, double dt,
					   double Vg_vec[3][3], double AA2[3], double BB[3]);
void RMOL_DATA_time_integration_v2(RMOL_DATA *md, ATOM_DATA *ad, double dt);
void RMOL_DATA_set_direction(RMOL_DATA *md, ATOM_DATA *ad, double c);

void RMOL_DATA_set_atom_velocity(RMOL_DATA *md, ATOM_DATA *ad);
void RMOL_DATA_set_mol_velocity(RMOL_DATA *md, ATOM_DATA *ad);
void RMOL_DATA_set_atom_velocity_all(RMOL_DATA *md, ATOM_DATA *ad);
void RMOL_DATA_set_mol_velocity_all(RMOL_DATA *md, ATOM_DATA *ad);

void RMOL_DATA_n_rigid_mol(RMOL_DATA *md, ATOM_DATA *ad,
			   int *n_rigid_mol6_ex,int *n_rigid_mol5_ex);
void RMOL_DATA_correct_virial(RMOL_DATA *md, ATOM_DATA *ad);

void RMOL_DATA_check_rigid_body(RMOL_DATA *md);

void RMOL_DATA_gather(RMOL_DATA *md, LINKED_CELL *lc, ATOM_DATA *ad);

void RMOL_DATA_copy_rmolcrd_to_buf(RMOL_DATA *md);
void RMOL_DATA_copy_rmolcrd_from_buf(RMOL_DATA *md);
void Vec_cmp(char *name, VEC *v1, VEC *v2, int rank);
void Quan_cmp(char *name, QUAN *q1, QUAN *q2, int rank);
int RMOL_DATA_test(RMOL_DATA *md);

#endif     /* RIGID_MOL_H_INCLUDED */
