/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

static char *rcsid = "$Header: /home/share/cvs/marble-0.6/src/marble/rigid_mol.c,v 1.2 2012/09/26 11:54:26 ike Exp $";

#include "misc.h"
#include "atom.h"
#include "bond.h"
#include "angle.h"
#include "boundary.h"
#include "linked_cell.h"
#include "rigid_mol.h"

/* #define OLDQUAN */
#define NO_SQUISH  /* */

inline void QUAN_room_to_mol(QUAN *q, VEC *from, VEC *to)
{
#ifdef OLDQUAN  
  to->x =
    (-(q->qw*q->qw)+(q->qx*q->qx)-(q->qy*q->qy)+(q->qz*q->qz)) * from->x
    + 2.0*((q->qy*q->qz)-(q->qw*q->qx))                        * from->y
    + 2.0*((q->qx*q->qy)+(q->qw*q->qz))                        * from->z;
  
  to->y =
    - 2.0*((q->qy*q->qz)+(q->qw*q->qx))                        * from->x
    + ((q->qw*q->qw)-(q->qx*q->qx)-(q->qy*q->qy)+(q->qz*q->qz))* from->y
    + 2.0*((q->qx*q->qz)-(q->qw*q->qy))                        * from->z;
	     
  to->z =
    2.0*((q->qx*q->qy)-(q->qw*q->qz))                           * from->x
    - 2.0*((q->qx*q->qz)+(q->qw*q->qy))                         * from->y
    + (-(q->qw*q->qw)-(q->qx*q->qx)+(q->qy*q->qy)+(q->qz*q->qz))* from->z;
#else
  to->x =
      ((q->qw*q->qw)+(q->qx*q->qx)-(q->qy*q->qy)-(q->qz*q->qz)) * from->x
    + 2.0*((q->qx*q->qy)+(q->qw*q->qz))                         * from->y
    + 2.0*((q->qx*q->qz)-(q->qw*q->qy))                         * from->z;
  
  to->y =
      2.0*((q->qx*q->qy)-(q->qw*q->qz))                         * from->x
    + ((q->qw*q->qw)-(q->qx*q->qx)+(q->qy*q->qy)-(q->qz*q->qz)) * from->y
    + 2.0*((q->qy*q->qz)+(q->qw*q->qx))                         * from->z;
	     
  to->z =
      2.0*((q->qx*q->qz)+(q->qw*q->qy))                         * from->x
    + 2.0*((q->qy*q->qz)-(q->qw*q->qx))                         * from->y
    + ((q->qw*q->qw)-(q->qx*q->qx)-(q->qy*q->qy)+(q->qz*q->qz)) * from->z;
#endif
}

inline void QUAN_mol_to_room(QUAN *q, VEC *from, VEC *to)
{
#ifdef OLDQUAN  
  to->x =
    (-(q->qw*q->qw)+(q->qx*q->qx)-(q->qy*q->qy)+(q->qz*q->qz)) * from->x
    - 2.0*((q->qy*q->qz)+(q->qw*q->qx))                        * from->y
    + 2.0*((q->qx*q->qy)-(q->qw*q->qz))                        * from->z;

  to->y =
      2.0*((q->qy*q->qz)-(q->qw*q->qx))                         * from->x
    + ((q->qw*q->qw)-(q->qx*q->qx)-(q->qy*q->qy)+(q->qz*q->qz)) * from->y
    - 2.0*((q->qx*q->qz)+(q->qw*q->qy))                         * from->z;
	     
  to->z =
      2.0*((q->qx*q->qy)+(q->qw*q->qz))                         * from->x
    + 2.0*((q->qx*q->qz)-(q->qw*q->qy))                         * from->y
    + (-(q->qw*q->qw)-(q->qx*q->qx)+(q->qy*q->qy)+(q->qz*q->qz))* from->z;
#else
  to->x =
     ((q->qw*q->qw)+(q->qx*q->qx)-(q->qy*q->qy)-(q->qz*q->qz)) * from->x
    + 2.0*((q->qx*q->qy)-(q->qw*q->qz))                        * from->y
    + 2.0*((q->qx*q->qz)+(q->qw*q->qy))                        * from->z;
  
  to->y =
      2.0*((q->qx*q->qy)+(q->qw*q->qz))                        * from->x
    + ((q->qw*q->qw)-(q->qx*q->qx)+(q->qy*q->qy)-(q->qz*q->qz))* from->y
    + 2.0*((q->qy*q->qz)-(q->qw*q->qx))                        * from->z;
	     
  to->z =
      2.0*((q->qx*q->qz)-(q->qw*q->qy))                         * from->x
    + 2.0*((q->qy*q->qz)+(q->qw*q->qx))                         * from->y
    + ((q->qw*q->qw)-(q->qx*q->qx)-(q->qy*q->qy)+(q->qz*q->qz)) * from->z;
#endif
}

inline void QUAN_normalize(QUAN *q)
{
  double tmp;

  tmp = sqrt(SQR(q->qw) + SQR(q->qx) + SQR(q->qy) + SQR(q->qz));

  q->qw /= tmp;
  q->qx /= tmp;
  q->qy /= tmp;
  q->qz /= tmp;
}

inline void RIGID_MOL_mol_to_room(RIGID_MOL *mol, ATOM_DATA *ad)
{
  int i,j;
  
  for (i=0,j=mol->parent_atom;i<mol->n_atom;i++,j=ad->ex[j].child_list) {
    
    QUAN_mol_to_room(&(mol->q), &(mol->type->atom[i].x),
		     &(ad->x[j]));
    
    ad->x[j].x += mol->rg.x;
    ad->x[j].y += mol->rg.y;
    ad->x[j].z += mol->rg.z;
  }
}

inline void RIGID_MOL_leapfrog(RIGID_MOL *mol, double dt)
{
#ifdef OLDQUAN  
  VEC ln;    /*  angular momemtum at time n */
  VEC lp;    /*  angular momemtum at time n */
  VEC wp;    /*  angular velocity */
  QUAN q_n_half;  /* quaternion parameter for time n+1/2 */
  RMOL_TYPE *mol_t;

  mol_t = mol->type;
  /**************************************************************/
  /*  leap frog for gravity center */ 
  /**************************************************************/

  /* proceed velocity from time n-1/2 to n+1/2 */
  mol->vg.x += mol->fg.x * dt / mol_t->weight;
  mol->vg.y += mol->fg.y * dt / mol_t->weight;
  mol->vg.z += mol->fg.z * dt / mol_t->weight;
  
  /* proceed gravity center from time n to n+1 */
  mol->rg.x += mol->vg.x * dt;
  mol->rg.y += mol->vg.y * dt;
  mol->rg.z += mol->vg.z * dt;

  /* output:  mol->rg and mol->vg */

  /**************************************************************/
  /* leap frog for molecular angle  */ 
  /**************************************************************/

  /* proceed angular momemtum from time n-1/2 to n using torque */
  /* mol->l: angular momemtum at time n-1/2 */
  /* ln:     angular momemtum at time n     */
  ln.x = mol->l.x + 0.5* dt * mol->torq.x;
  ln.y = mol->l.y + 0.5* dt * mol->torq.y;
  ln.z = mol->l.z + 0.5* dt * mol->torq.z;

  /* translate angular momemtum from the room coord. to molecular coord. */
  QUAN_room_to_mol(&(mol->q), &(ln), &(lp));

  /* calculate angular velocity from angular momentum */
  wp.x = lp.x / mol_t->ip.x;
  wp.y = lp.y / mol_t->ip.y;
  wp.z = lp.z / mol_t->ip.z;

  /* keep angular velocity */
  mol->wp = wp;

  /* proceed quaternion parameters by 0.5 dt */
  q_n_half.qw = mol->q.qw +
    0.25*dt*(-(mol->q.qy*wp.x) - (mol->q.qz*wp.y) + (mol->q.qx*wp.z));
  q_n_half.qx = mol->q.qx +
    0.25*dt*( (mol->q.qz*wp.x) - (mol->q.qy*wp.y) - (mol->q.qw*wp.z));
  q_n_half.qy = mol->q.qy +
    0.25*dt*( (mol->q.qw*wp.x) + (mol->q.qx*wp.y) + (mol->q.qz*wp.z));
  q_n_half.qz = mol->q.qz +
    0.25*dt*(-(mol->q.qx*wp.x) + (mol->q.qw*wp.y) - (mol->q.qy*wp.z));

  /* proceed angular momemtum from time n-1/2 to n+1/2 using torque */
  /* mol->l: angular momemtum at time n-1/2 ===> n+1/2 */
  mol->l.x += dt * mol->torq.x;
  mol->l.y += dt * mol->torq.y;
  mol->l.z += dt * mol->torq.z;

  /* translate angular momemtum from the room coord. to molecular coord. */
  QUAN_room_to_mol(&q_n_half, &(mol->l), &lp);

  /* calculate angular velocity from angular momentum */
  /* wp:     angular velocity at time n+1/2 */
  wp.x = lp.x / mol_t->ip.x;
  wp.y = lp.y / mol_t->ip.y;
  wp.z = lp.z / mol_t->ip.z;

  /* keep angular velocity */
  mol->wp_half = wp;

  /* proceed quaternion parameters by 1 dt */
  mol->q.qw +=
    0.5*dt*(-(q_n_half.qy*wp.x) - (q_n_half.qz*wp.y) + (q_n_half.qx*wp.z));
  mol->q.qx +=
    0.5*dt*( (q_n_half.qz*wp.x) - (q_n_half.qy*wp.y) - (q_n_half.qw*wp.z));
  mol->q.qy +=
    0.5*dt*( (q_n_half.qw*wp.x) + (q_n_half.qx*wp.y) + (q_n_half.qz*wp.z));
  mol->q.qz +=
    0.5*dt*(-(q_n_half.qx*wp.x) + (q_n_half.qw*wp.y) - (q_n_half.qy*wp.z));

  QUAN_normalize(&(mol->q));

  /* output:  mol->q and mol->l */

#else /* NEW QUAN */
  
  VEC ln;    /*  angular momemtum at time n */
  VEC lp;    /*  angular momemtum at time n */
  VEC wp;    /*  angular velocity */
  QUAN q_n_half;  /* quaternion parameter for time n+1/2 */
  RMOL_TYPE *mol_t;

  mol_t = mol->type;
  /**************************************************************/
  /*  leap frog for gravity center */ 
  /**************************************************************/

  /* proceed velocity from time n-1/2 to n+1/2 */
  mol->vg.x += mol->fg.x * dt / mol_t->weight;
  mol->vg.y += mol->fg.y * dt / mol_t->weight;
  mol->vg.z += mol->fg.z * dt / mol_t->weight;
  
  /* proceed gravity center from time n to n+1 */
  mol->rg.x += mol->vg.x * dt;
  mol->rg.y += mol->vg.y * dt;
  mol->rg.z += mol->vg.z * dt;

  /* output:  mol->rg and mol->vg */

  /**************************************************************/
  /* leap frog for molecular angle  */ 
  /**************************************************************/

  /* proceed angular momemtum from time n-1/2 to n using torque */
  /* mol->l: angular momemtum at time n-1/2 */
  /* ln:     angular momemtum at time n     */
  ln.x = mol->l.x + 0.5* dt * mol->torq.x;
  ln.y = mol->l.y + 0.5* dt * mol->torq.y;
  ln.z = mol->l.z + 0.5* dt * mol->torq.z;

  /* translate angular momemtum from the room coord. to molecular coord. */
  QUAN_room_to_mol(&(mol->q), &(ln), &(lp));

  /* calculate angular velocity from angular momentum */
  wp.x = lp.x / mol_t->ip.x;
  wp.y = lp.y / mol_t->ip.y;
  wp.z = lp.z / mol_t->ip.z;

  /* keep angular velocity */
  mol->wp = wp;

  /* proceed quaternion parameters by 0.5 dt */
  q_n_half.qw = mol->q.qw +
    0.25*dt*(-(mol->q.qx*wp.x) - (mol->q.qy*wp.y) - (mol->q.qz*wp.z));
  q_n_half.qx = mol->q.qx +
    0.25*dt*( (mol->q.qw*wp.x) - (mol->q.qz*wp.y) + (mol->q.qy*wp.z));
  q_n_half.qy = mol->q.qy +
    0.25*dt*( (mol->q.qz*wp.x) + (mol->q.qw*wp.y) - (mol->q.qx*wp.z));
  q_n_half.qz = mol->q.qz +
    0.25*dt*(-(mol->q.qy*wp.x) + (mol->q.qx*wp.y) + (mol->q.qw*wp.z));

  /* proceed angular momemtum from time n-1/2 to n+1/2 using torque */
  /* mol->l: angular momemtum at time n-1/2 ===> n+1/2 */
  mol->l.x += dt * mol->torq.x;
  mol->l.y += dt * mol->torq.y;
  mol->l.z += dt * mol->torq.z;

  /* translate angular momemtum from the room coord. to molecular coord. */
  QUAN_room_to_mol(&q_n_half, &(mol->l), &lp);

  /* calculate angular velocity from angular momentum */
  /* wp:     angular velocity at time n+1/2 */
  wp.x = lp.x / mol_t->ip.x;
  wp.y = lp.y / mol_t->ip.y;
  wp.z = lp.z / mol_t->ip.z;

  /* keep angular velocity */
  mol->wp_half = wp;

  /* proceed quaternion parameters by 1 dt */
  mol->q.qw +=
    0.5*dt*(-(q_n_half.qx*wp.x) - (q_n_half.qy*wp.y) - (q_n_half.qz*wp.z));
  mol->q.qx +=
    0.5*dt*( (q_n_half.qw*wp.x) - (q_n_half.qz*wp.y) + (q_n_half.qy*wp.z));
  mol->q.qy +=
    0.5*dt*( (q_n_half.qz*wp.x) + (q_n_half.qw*wp.y) - (q_n_half.qx*wp.z));
  mol->q.qz +=
    0.5*dt*(-(q_n_half.qy*wp.x) + (q_n_half.qx*wp.y) + (q_n_half.qw*wp.z));

  QUAN_normalize(&(mol->q));

  /* output:  mol->q and mol->l */

#endif  /* #ifdef OLD_QUAN */  
}

inline void RIGID_MOL_time_integration_v1(RIGID_MOL *mol, double dt)
{
  static double hdt;
  static double IzyLy, cos_IzyLy, sin_IzyLy;
  static double IzxLx, cos_IzxLx, sin_IzxLx;
  static double Lz1;
  
  RMOL_TYPE *mt;

  mt = mol->type;

  hdt = 0.5 * dt;
  /**************************************************************/
  /*  integrate equation of motion of  gravity center           */ 
  /**************************************************************/
  /* proceed velocity from time n to n+1/2 */
  mol->vg.x += hdt * mol->fg.x / mt->weight;
  mol->vg.y += hdt * mol->fg.y / mt->weight;
  mol->vg.z += hdt * mol->fg.z / mt->weight;
  

  /**************************************************************/
  /*  integrate equation of motion of angular momemtum          */ 
  /**************************************************************/

  /* Step 1.  the angular momemtum is advanced just by the torque term */

  mol->l.x += hdt * mol->torq.x;
  mol->l.y += hdt * mol->torq.y;

  if (mt->ip.z == 0.0) return;   /* in the case of two-atom molecules */
  
  mol->l.z += hdt * mol->torq.z;

  /* Step 2.  the angular momemtum is advanced by free motion */
  IzyLy = hdt * mt->Izy * mol->l.y;
  cos_IzyLy = cos(IzyLy);
  sin_IzyLy = sin(IzyLy);

  Lz1      = -sin_IzyLy * mol->l.x + cos_IzyLy * mol->l.z;
  mol->l.x =  cos_IzyLy * mol->l.x + sin_IzyLy * mol->l.z;

  IzxLx = hdt * mt->Izx * mol->l.x;
  cos_IzxLx = cos(IzxLx);
  sin_IzxLx = sin(IzxLx);
  
  mol->l.z =  sin_IzxLx * mol->l.y + cos_IzxLx * Lz1;
  mol->l.y =  cos_IzxLx * mol->l.y - sin_IzxLx * Lz1;

}

inline void RIGID_MOL_time_integration_p_rg(RIGID_MOL *mol, double dt)
{
  mol->rg.x += mol->vg.x * dt;
  mol->rg.y += mol->vg.y * dt;
  mol->rg.z += mol->vg.z * dt;
}

inline void RIGID_MOL_time_integration_p_rg_NPT(RIGID_MOL *mol, double AA2, double BB)
{
  mol->rg.x = mol->rg.x * AA2 + mol->vg.x * BB;
  mol->rg.y = mol->rg.y * AA2 + mol->vg.y * BB;
  mol->rg.z = mol->rg.z * AA2 + mol->vg.z * BB;
}


inline void RIGID_MOL_time_integration_p_rg_NPT_full(RIGID_MOL *mol, double Vg_vec[3][3],
					      double AA2[3], double BB[3])
{
  VEC ux, uv;
  
  ux.x = VEC_MUL_MAT_X(mol->rg,Vg_vec);
  ux.y = VEC_MUL_MAT_Y(mol->rg,Vg_vec);
  ux.z = VEC_MUL_MAT_Z(mol->rg,Vg_vec);
  uv.x = VEC_MUL_MAT_X(mol->vg,Vg_vec);
  uv.y = VEC_MUL_MAT_Y(mol->vg,Vg_vec);
  uv.z = VEC_MUL_MAT_Z(mol->vg,Vg_vec);
    
  ux.x = ux.x * AA2[0] + uv.x * BB[0];
  ux.y = ux.y * AA2[1] + uv.y * BB[1];
  ux.z = ux.z * AA2[2] + uv.z * BB[2];

  mol->rg.x = MAT_MUL_VEC_X(Vg_vec,ux);
  mol->rg.y = MAT_MUL_VEC_Y(Vg_vec,ux);
  mol->rg.z = MAT_MUL_VEC_Z(Vg_vec,ux);
}

inline void RIGID_MOL_time_integration_p_rot(RIGID_MOL *mol, double dt)
{
  static double hdt;
  static double A[4][4];
  static double len_wp, cos_len_wp, sin_len_wp;
  static VEC wp;
  static QUAN tmp_q;
  RMOL_TYPE *mt;

  mt = mol->type;
  hdt = 0.5 * dt;
  
  /* proceed gravity center from time n to n+1 
  mol->rg.x += mol->vg.x * dt;
  mol->rg.y += mol->vg.y * dt;
  mol->rg.z += mol->vg.z * dt;
  */
  
  /*  integration of motion of quarterniion */
  
  wp.x = mol->l.x / mt->ip.x;
  wp.y = mol->l.y / mt->ip.y;

  if (mt->ip.z != 0.0)
    wp.z = mol->l.z / mt->ip.z;
  else
    wp.z = 0.0;     /* in the case of two-atom molecules */

  len_wp = sqrt(wp.x*wp.x+wp.y*wp.y+wp.z*wp.z);
  cos_len_wp = cos(hdt * len_wp);
  sin_len_wp = sin(hdt * len_wp) / len_wp;

  A[0][0] = cos_len_wp;
  A[1][1] = cos_len_wp;
  A[2][2] = cos_len_wp;
  A[3][3] = cos_len_wp;

  A[1][0] = A[2][3] = sin_len_wp * wp.x;
  A[2][0] = A[3][1] = sin_len_wp * wp.y;
  A[3][0] = A[1][2] = sin_len_wp * wp.z;
  
  A[0][1] = A[3][2] = -sin_len_wp * wp.x;
  A[0][2] = A[1][3] = -sin_len_wp * wp.y;
  A[0][3] = A[2][1] = -sin_len_wp * wp.z;

  tmp_q.qw = A[0][0]*mol->q.qw + A[0][1]*mol->q.qx + A[0][2]*mol->q.qy + A[0][3]*mol->q.qz;
  tmp_q.qx = A[1][0]*mol->q.qw + A[1][1]*mol->q.qx + A[1][2]*mol->q.qy + A[1][3]*mol->q.qz;
  tmp_q.qy = A[2][0]*mol->q.qw + A[2][1]*mol->q.qx + A[2][2]*mol->q.qy + A[2][3]*mol->q.qz;
  tmp_q.qz = A[3][0]*mol->q.qw + A[3][1]*mol->q.qx + A[3][2]*mol->q.qy + A[3][3]*mol->q.qz;

  mol->q = tmp_q;
}

inline void RIGID_MOL_time_integration_v2(RIGID_MOL *mol, double dt)
{
  static double hdt;
  static double IzyLy, cos_IzyLy, sin_IzyLy;
  static double IzxLx, cos_IzxLx, sin_IzxLx;
  static double A[4][4];
  static double Lz1, Lz2;
  static double len_wp, cos_len_wp, sin_len_wp;
  static VEC wp;
  static QUAN tmp_q;
  
  RMOL_TYPE *mt;

  mt = mol->type;

  hdt = 0.5 * dt;
  /**************************************************************/
  /*  integrate equation of motion of  gravity center           */ 
  /**************************************************************/

  /* proceed velocity from time n+1/2 to n+1 */
  mol->vg.x += hdt * mol->fg.x / mt->weight;
  mol->vg.y += hdt * mol->fg.y / mt->weight;
  mol->vg.z += hdt * mol->fg.z / mt->weight;
  
  /**************************************************************/
  /*  integrate equation of motion of angular momemtum          */ 
  /**************************************************************/
  /* Step 3.  the angular momemtum is advanced by free motion */

  if (mt->ip.z != 0.0) {
  
    IzxLx = hdt * mt->Izx * mol->l.x;
    cos_IzxLx = cos(IzxLx);
    sin_IzxLx = sin(IzxLx);

    Lz2      =  sin_IzxLx * mol->l.y + cos_IzxLx * mol->l.z;
    mol->l.y =  cos_IzxLx * mol->l.y - sin_IzxLx * mol->l.z;
  
    IzyLy = hdt * mt->Izy * mol->l.y;
    cos_IzyLy = cos(IzyLy);
    sin_IzyLy = sin(IzyLy);

    mol->l.z = -sin_IzyLy * mol->l.x + cos_IzyLy * Lz2;
    mol->l.x =  cos_IzyLy * mol->l.x + sin_IzyLy * Lz2;

  /* Step 4.  the angular momemtum is advanced just by the torque term */
    mol->l.z += hdt * mol->torq.z;
  }

  mol->l.x += hdt * mol->torq.x;
  mol->l.y += hdt * mol->torq.y;
}

/* These are NO_SQUISH routines */
inline void make_S_no_squish(double S[4][4], QUAN *q)
{
  S[0][0] = q->qw;  S[0][1] = -q->qx;  S[0][2] = -q->qy;  S[0][3] = -q->qz;
  S[1][0] = q->qx;  S[1][1] =  q->qw;  S[1][2] = -q->qz;  S[1][3] =  q->qy;
  S[2][0] = q->qy;  S[2][1] =  q->qz;  S[2][2] =  q->qw;  S[2][3] = -q->qx;
  S[3][0] = q->qz;  S[3][1] = -q->qy;  S[3][2] =  q->qx;  S[3][3] =  q->qw;
}

inline void RIGID_MOL_time_integration_v1_no_squish(RIGID_MOL *mol, double dt)
{
  int i,j;
  static double hdt;
  RMOL_TYPE *mt;

  mt = mol->type;
  hdt = 0.5 * dt;
  
  /* integrate equation of motion of  gravity center  */ 
  mol->vg.x += hdt * mol->fg.x / mt->weight;
  mol->vg.y += hdt * mol->fg.y / mt->weight;
  mol->vg.z += hdt * mol->fg.z / mt->weight;

  /* integrate rotation L4 */
  mol->l.x += hdt * mol->torq.x;
  mol->l.y += hdt * mol->torq.y;
  if (mt->ip.z != 0.0)
    mol->l.z += hdt * mol->torq.z;
  else
    mol->l.z = 0.0;
}

inline void make_Pk_q(double Pk_q[4], double q[4], int k)
{
  switch (k) {
  case 1:
    Pk_q[0] = -q[1]; Pk_q[1] =  q[0];  Pk_q[2] =  q[3];  Pk_q[3] = -q[2];
    break;
  case 2:
    Pk_q[0] = -q[2]; Pk_q[1] = -q[3];  Pk_q[2] =  q[0];  Pk_q[3] =  q[1];
    break;
  case 3:
    Pk_q[0] = -q[3]; Pk_q[1] =  q[2];  Pk_q[2] = -q[1];  Pk_q[3] =  q[0];
    break;
  }
}

/*inline*/ void no_squish_rotate(RIGID_MOL *mol, int k, double dt, double ip)
{
  double Pk_q[4], q[4], Pk_p[4], zeta_dt, cosz, sinz;
  int i;

  q[0] = mol->q.qw;  q[1] = mol->q.qx;  q[2] = mol->q.qy;  q[3] = mol->q.qz;

  make_Pk_q(Pk_q, q, k);
  zeta_dt = 0.0;
  for (i=0;i<4;i++) {
    zeta_dt += mol->p[i]*Pk_q[i];
  }
  zeta_dt *= dt / (ip*4.0);
  cosz = cos(zeta_dt);  sinz = sin(zeta_dt);
  
  make_Pk_q(Pk_p, mol->p, k);

  for (i=0;i<4;i++) {
    q[i]      = cosz *      q[i] + sinz * Pk_q[i];
    mol->p[i] = cosz * mol->p[i] + sinz * Pk_p[i];
  }

  mol->q.qw = q[0];  mol->q.qx = q[1];  mol->q.qy = q[2];  mol->q.qz = q[3];
}

inline void RIGID_MOL_time_integration_p_rot_no_squish(RIGID_MOL *mol, double dt, int Mrot)
{
  int i, j, m;
  double hdt, S[4][4], p0[4];
  RMOL_TYPE *mt;
  mt = mol->type;

  p0[0] = 0.0;
  p0[1] = mol->l.x * 2.0;
  p0[2] = mol->l.y * 2.0;
  p0[3] = mol->l.z * 2.0;
  
  make_S_no_squish(S, &(mol->q));
  for (i=0;i<4;i++) {
    mol->p[i] = 0.0;
    for (j=1;j<4;j++) {
      mol->p[i] += S[i][j]*p0[j];
    }
  }
  
  hdt = dt / (Mrot*2.0);
  for (m=0;m<Mrot;m++) {
    if (mt->ip.z != 0.0)
      no_squish_rotate(mol, 3, hdt, mt->ip.z);
    no_squish_rotate(mol, 2, hdt,     mt->ip.y);
    no_squish_rotate(mol, 1, hdt*2.0, mt->ip.x);
    no_squish_rotate(mol, 2, hdt,     mt->ip.y);
    if (mt->ip.z != 0.0)
      no_squish_rotate(mol, 3, hdt, mt->ip.z);
  }
  
  make_S_no_squish(S, &(mol->q));
  for (i=1;i<4;i++) {
    p0[i] = 0.0;
    for (j=0;j<4;j++) {
      p0[i] += S[j][i]*mol->p[j];
    }
  }
  
  mol->l.x = p0[1] * 0.5;
  mol->l.y = p0[2] * 0.5;
  mol->l.z = p0[3] * 0.5;
}

inline void RIGID_MOL_time_integration_v2_no_squish(RIGID_MOL *mol, double dt)
{
  int i,j;
  static double hdt, S[4][4], p0[4];
  RMOL_TYPE *mt;

  mt = mol->type;
  hdt = 0.5 * dt;
  
  /* integrate equation of motion of  gravity center  */ 
  mol->vg.x += hdt * mol->fg.x / mt->weight;
  mol->vg.y += hdt * mol->fg.y / mt->weight;
  mol->vg.z += hdt * mol->fg.z / mt->weight;

  /* integrate rotation L4 */
  mol->l.x += hdt * mol->torq.x;
  mol->l.y += hdt * mol->torq.y;
  if (mt->ip.z != 0.0)
    mol->l.z += hdt * mol->torq.z;
  else
    mol->l.z = 0.0;
}
/* End of NO_SQUISH routines */


/* This routine is for minimization */
void RIGID_MOL_set_direction(RIGID_MOL *mol, double c)
{
  mol->vg.x = mol->vg.x * c + mol->fg.x;
  mol->vg.y = mol->vg.y * c + mol->fg.y;
  mol->vg.z = mol->vg.z * c + mol->fg.z;
  
  mol->l.x = mol->l.x * c + mol->torq.x * mol->type->ip.x;
  mol->l.y = mol->l.y * c + mol->torq.y * mol->type->ip.y;
  mol->l.z = mol->l.z * c + mol->torq.z * mol->type->ip.z;
}

#if 1
void RIGID_MOL_torq_room_to_mol(RIGID_MOL *mol)
{
  VEC to;

  QUAN_room_to_mol(&mol->q, &mol->torq, &to);
  mol->torq = to;
}

/* calculate force and torque on gravity center */
void RIGID_MOL_force_and_torque(RIGID_MOL *mol, ATOM_DATA *ad)
{
  int j;
  double sx, sy, sz;

  mol->fg.x = mol->fg.y = mol->fg.z = 0.0;
  mol->torq.x = mol->torq.y = mol->torq.z = 0.0;
  
  for (j=mol->parent_atom;j>=0;j=ad->ex[j].child_list) {
    mol->fg.x += ad->f[j].x;
    mol->fg.y += ad->f[j].y;
    mol->fg.z += ad->f[j].z;

    sx = ad->x[j].x - mol->rg.x;
    sy = ad->x[j].y - mol->rg.y;
    sz = ad->x[j].z - mol->rg.z;

    mol->torq.x += sy * ad->f[j].z - sz * ad->f[j].y;
    mol->torq.y += sz * ad->f[j].x - sx * ad->f[j].z;
    mol->torq.z += sx * ad->f[j].y - sy * ad->f[j].x;
  }
}
#else
void RIGID_MOL_torq_room_to_mol(RIGID_MOL *mol)
{
}

/* calculate force and torque on gravity center */
void RIGID_MOL_force_and_torque(RIGID_MOL *mol, ATOM_DATA *ad)
{
  int i,j;
  VEC s, sm, fr, fm;
  RMOL_TYPE *mt;

  mt = mol->type;
  mol->fg.x = mol->fg.y = mol->fg.z = 0.0;
  mol->torq.x = mol->torq.y = mol->torq.z = 0.0;
  
  for (j=mol->parent_atom;j>=0;j=ad->ex[j].child_list) {
    mol->fg.x += ad->f[j].x;
    mol->fg.y += ad->f[j].y;
    mol->fg.z += ad->f[j].z;
  }

  for (i=0,j=mol->parent_atom;j>=0;j=ad->ex[j].child_list,i++) {
    s.x = ad->x[j].x - mol->rg.x;
    s.y = ad->x[j].y - mol->rg.y;
    s.z = ad->x[j].z - mol->rg.z;

    fr.x = ad->f[j].x - mol->fg.x * ad->w[j] / mt->weight;
    fr.y = ad->f[j].y - mol->fg.y * ad->w[j] / mt->weight;
    fr.z = ad->f[j].z - mol->fg.z * ad->w[j] / mt->weight;
    QUAN_room_to_mol(&mol->q, &fr, &fm);

    sm = mt->atom[i].x;

    mol->torq.x += sm.y * fm.z - sm.z * fm.y;
    mol->torq.y += sm.z * fm.x - sm.x * fm.z;
    mol->torq.z += sm.x * fm.y - sm.y * fm.x;
  }
}
#endif


void RIGID_MOL_set_atom_velocity(RIGID_MOL *mol, ATOM_DATA *ad)
{
  VEC wp_m, wp_r, r;
  RMOL_TYPE *mt;
  int i, n;
  
  mt = mol->type;
  wp_m.x = mol->l.x / mt->ip.x;
  wp_m.y = mol->l.y / mt->ip.y;
  if (mt->ip.z != 0.0)
    wp_m.z = mol->l.z / mt->ip.z;
  else
    wp_m.z = 0.0;

  QUAN_mol_to_room(&mol->q, &wp_m, &wp_r);
  
  for (n=mol->parent_atom;n>=0;n=ad->ex[n].child_list) {
    r.x = ad->x[n].x - mol->rg.x;
    r.y = ad->x[n].y - mol->rg.y;
    r.z = ad->x[n].z - mol->rg.z;
    ad->v[n].x = mol->vg.x + wp_r.y*r.z - wp_r.z*r.y;
    ad->v[n].y = mol->vg.y + wp_r.z*r.x - wp_r.x*r.z;
    ad->v[n].z = mol->vg.z + wp_r.x*r.y - wp_r.y*r.x;
  }
}

void RIGID_MOL_set_mol_velocity(RIGID_MOL *mol, ATOM_DATA *ad)
{
  VEC r, v, l;
  double tg;
  int n;
  
  mol->vg.x =  mol->vg.y  =   mol->vg.z = tg = 0.0;
  for (n=mol->parent_atom;n>=0;n=ad->ex[n].child_list) {
    mol->vg.x += ad->v[n].x * ad->w[n];
    mol->vg.y += ad->v[n].y * ad->w[n];
    mol->vg.z += ad->v[n].z * ad->w[n];
    tg += ad->w[n];
  }
  mol->vg.x /= tg;
  mol->vg.y /= tg;
  mol->vg.z /= tg;

  l.x = l.y = l.z = 0.0;
  for (n=mol->parent_atom;n>=0;n=ad->ex[n].child_list) {
    
    r.x = ad->x[n].x - mol->rg.x;
    r.y = ad->x[n].y - mol->rg.y;
    r.z = ad->x[n].z - mol->rg.z;
      
    v.x = ad->v[n].x - mol->vg.x;
    v.y = ad->v[n].y - mol->vg.y;
    v.z = ad->v[n].z - mol->vg.z;

    l.x += (r.y*v.z-r.z*v.y)*ad->w[n];
    l.y += (r.z*v.x-r.x*v.z)*ad->w[n];
    l.z += (r.x*v.y-r.y*v.x)*ad->w[n];
  }
  QUAN_room_to_mol(&mol->q, &l, &mol->l);
}

/** for initialization **/
void RMOL_TYPE_internal_to_xyz(RMOL_TYPE *mt)
{
  int i;
  double angle, dihedral;
  VEC *p0, *p1, *p2;
  VEC v1, v2, v, axis;

  /* The first atom is placed at origin. */

  mt->atom[0].x.x = mt->atom[0].x.y = mt->atom[0].x.z = 0.0;

  /* The second atom is placed at z axis. */
  mt->atom[1].x.x = mt->atom[1].x.y = 0.0;
  mt->atom[1].x.z = mt->atom[1].length;

  if (mt->n_atom == 2) return;

  /* The third atom is placed at yz place. */
  p0 = &mt->atom[mt->atom[2].na].x;
  p1 = &mt->atom[mt->atom[2].nb].x;
  v_sub(&v1, p1, p0);
  v_norm(&v1);
  v_mul(&v, mt->atom[2].length, &v1);
  axis.x = -1.0; axis.y = 0.0; axis.z = 0.0;
  angle = mt->atom[2].angle * M_PI / 180.0;
  v_rot(&v, &axis, angle, &v);
  v_add(&mt->atom[2].x, p0, &v);

  /* Atoms after the third atom are placed according to internal coordinates */
  for (i=3;i<mt->n_atom;i++) {
    
    angle    = mt->atom[i].angle    * M_PI / 180.0;
    dihedral = mt->atom[i].dihedral * M_PI / 180.0;
    
    p0 = &mt->atom[mt->atom[i].na].x;
    p1 = &mt->atom[mt->atom[i].nb].x;
    p2 = &mt->atom[mt->atom[i].nc].x;

    v_sub(&v1, p1, p0);
    v_sub(&v2, p2, p0);
    v_norm(&v1);
    v_mul(&v, mt->atom[i].length, &v1);
    v_outer_pro(&axis, &v1, &v2);
    v_rot(&v, &axis, angle, &v);
    v_rot(&v, &v1,   -dihedral, &v);

    v_add(&mt->atom[i].x, p0, &v);
  }
}

#define EPSEG 1.0e-6
void RMOL_TYPE_normalize_eigv(double eig[3], double eigv[3][3])
{
  int i,j,k;
  int mid, sid1, sid2;
  double v0[3], v1[3], v2[3];
  double len;

  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      if (fabs(eigv[j][i]) >= EPSEG) {
	if (eigv[j][i] < 0.0) {
	  for (k=0;k<3;k++) {
	    eigv[k][i] *= -1.0;
	  }
	}
	break;
      }
    }
  }

  if (fabs(eig[0]-eig[1]) < EPSEG && fabs(eig[1]-eig[2]) < EPSEG) {
    mid = 0; sid1 = 1; sid2 = 2;
    eigv[0][0] = 1.0; eigv[1][0] = 0.0; eigv[2][0] = 0.0;
  } else if (fabs(eig[0]-eig[1]) < EPSEG) {
    mid = 2; sid1 = 0; sid2 = 1;
  } else if (fabs(eig[1]-eig[2]) < EPSEG) {
    mid = 0; sid1 = 1; sid2 = 2;
  } else {
    mid = -1;
  }

  if (mid>=0) {
    if (fabs(eigv[0][mid]) > EPSEG || fabs(eigv[1][mid]) > EPSEG) {
      eigv[2][sid1] = 0.0;
      eigv[1][sid1] =  eigv[0][mid];
      eigv[0][sid1] = -eigv[1][mid];
    } else {
      eigv[0][sid1] = 0.0;
      eigv[1][sid1] =  eigv[2][mid];
      eigv[2][sid1] = -eigv[1][mid];
    }
    len=0.0;
    for (i=0;i<3;i++) {
      len+=eigv[i][sid1]*eigv[i][sid1];
    }
    len = 1.0/sqrt(len);
    for (i=0;i<3;i++) {
      eigv[i][sid1]*=len;
    }
    for (i=0;i<3;i++) {
      v0[i] = eigv[i][mid];
      v1[i] = eigv[i][sid1];
    }
    cross3(v0,v1,v2);
    for (i=0;i<3;i++) {
      eigv[i][sid2] = v2[i];
    }
  }
}


void RMOL_TYPE_moment_of_inertia(RMOL_TYPE *mt)
{
  int i,j;
  VEC center;
  double x, y, z, w;
  double imat[3][3], eig[3], eigv[3][3];
  double v0[3], v1[3], v2[3], dot, cross[3];
  
  center.x = center.y = center.z = 0.0;
  mt->weight = 0.0;
  
  for (i = 0; i < mt->n_atom; i++) {
    center.x += mt->atom[i].w*mt->atom[i].x.x;
    center.y += mt->atom[i].w*mt->atom[i].x.y;
    center.z += mt->atom[i].w*mt->atom[i].x.z;
    mt->weight += mt->atom[i].w;
  }
  
  center.x /= mt->weight;
  center.y /= mt->weight;
  center.z /= mt->weight;

  for (i = 0; i < mt->n_atom; i++) {
    mt->atom[i].x.x -= center.x;
    mt->atom[i].x.y -= center.y;
    mt->atom[i].x.z -= center.z;
  }

  if (mt->n_atom == 2) {
    mt->ip.x = mt->ip.y = mt->ip.z = 0.0;
    for (i=0;i<2;i++) {
      x=mt->atom[i].x.x;
      y=mt->atom[i].x.y;
      z=mt->atom[i].x.z;
      w=mt->atom[i].w;
      mt->ip.x += w*(y*y+z*z);
      mt->ip.y += w*(z*z+x*x);
      /* mt->ip.z += w*(x*x+y*y); */
    }
    mt->Izx = mt->Izy = 0.0;
    /* In this case, mt->ip.z == 0 */
    return;
  }
  
  for (i=0;i<3;i++)
    for (j=0;j<3;j++)
      imat[i][j] = 0.0;
  
  for (i = 0; i < mt->n_atom; i++) {
    x=mt->atom[i].x.x;
    y=mt->atom[i].x.y;
    z=mt->atom[i].x.z;
    w=mt->atom[i].w;
    
    imat[0][0] += w*(y*y+z*z);
    imat[0][1] -= w*x*y;
    imat[0][2] -= w*x*z;
    imat[1][1] += w*(z*z+x*x);
    imat[1][2] -= w*y*z;
    imat[2][2] += w*(x*x+y*y);
  }
  imat[1][0] = imat[0][1];
  imat[2][0] = imat[0][2];
  imat[2][1] = imat[1][2];

  /*
  lprintf("n_atom = %d\n",mt->n_atom);
  print_mat33(imat);
  */
  
  diag33(imat, eig, eigv);
  sort_eigen(3,eig,&eigv[0][0],0);
  RMOL_TYPE_normalize_eigv(eig,eigv); 

  /* correction of mirror image */
  for (i=0;i<3;i++) {
    v0[i] = eigv[i][0];
    v1[i] = eigv[i][1];
    v2[i] = eigv[i][2];
  }
  cross3(v0, v1, cross);
  dot = dot3(v2, cross);
  if (dot < 0.0) {
    eigv[0][2] *= -1;
    eigv[1][2] *= -1;
    eigv[2][2] *= -1;
  }
  /* end of correction */

  for (i = 0; i < mt->n_atom; i++)
    v_mul_mat(&mt->atom[i].x, &mt->atom[i].x, eigv);
  
  /* check */
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      imat[i][j] = 0.0;
    }
  }
  
  for (i = 0; i < mt->n_atom; i++) {
    x=mt->atom[i].x.x;
    y=mt->atom[i].x.y;
    z=mt->atom[i].x.z;
    w=mt->atom[i].w;
    
    imat[0][0] += w*(y*y+z*z);
    imat[0][1] -= w*x*y;
    imat[0][2] -= w*x*z;
    imat[1][1] += w*(z*z+x*x);
    imat[1][2] -= w*y*z;
    imat[2][2] += w*(x*x+y*y);
  }

  for (i=0;i<3;i++) {
    if (fabs(imat[i][i] - eig[i]) > EPSEG) {
      lprintf("ERROR: internal error; calculation of moment of inertia failed.\n");
      for (j=0;j<3;j++) {
	lprintf("eig(%d) = %e\n", j,eig[j]);
      }
      imat[1][0] = imat[0][1];
      imat[2][0] = imat[0][2];
      imat[2][1] = imat[1][2];
      print_mat33(imat);
      marble_exit(1);
    }
  }
  
  for (i=0;i<3;i++) {
    for (j=i+1;j<3;j++) {
      if (fabs(imat[i][j]) > EPSEG) {
	lprintf("ERROR: internal error; calculation of moment of inertia failed.\n");
	imat[1][0] = imat[0][1];
	imat[2][0] = imat[0][2];
	imat[2][1] = imat[1][2];
	print_mat33(imat);
	marble_exit(1);
      }
    }
  }

  /*
  mt->ip.x = eig[0];
  mt->ip.y = eig[1];
  mt->ip.z = eig[2];
  */
  mt->ip.x = imat[0][0];
  mt->ip.y = imat[1][1];
  mt->ip.z = imat[2][2];
  mt->Izy = 1.0/mt->ip.z - 1.0/mt->ip.y;
  mt->Izx = 1.0/mt->ip.z - 1.0/mt->ip.x;

  /* debug 
  for (i=0;i<3;i++)
    for (j=i;j<3;j++)
      lprintf("%d %d = %e\n", i,j,imat[i][j]);
  
  lprintf("%e %e %e\n", mt->ip.x, mt->ip.y, mt->ip.z);
  */
}

int RMOL_TYPE_eq_residue(RMOL_TYPE *mt, RESIDUE *r)
{
  int i;
  if (strcmp(mt->name[0],"*AMN") == 0) {
    if (r->flag & RES_FLAG_AMINO)
      return 1;
    else
      return 0;
  }
  for (i=0;i<mt->n_name;i++) {
    if (strcmp(mt->name[i],r->name)==0)
      return 1;
  }
  return 0;
}

void RMOL_TYPE_init_wat_3P(RMOL_TYPE *mt, char *name,
			   char *o_name, char *h1_name, char *h2_name,
			   double length, double angle,
			   double wo, double wh)
{
  char *fname = "RMOL_TYPE_init_wat_3P";
  char *dum = "";

  mt->n_name = 1;
  mt->n_mol = 0;
  mt->name = emalloc(fname,sizeof(char)*5*mt->n_name);
  strcpy(mt->name[0], name);
  
  mt->n_atom = 3;
  mt->atom = emalloc(fname,sizeof(RMOL_TYPE_ATOM)*mt->n_atom);
  
  strcpy(mt->atom[0].name, o_name);
  mt->atom[0].na = mt->atom[0].nb = mt->atom[0].nc = -1;
  mt->atom[0].length = mt->atom[0].angle = mt->atom[0].dihedral = 0.0;
  mt->atom[0].w = wo;
  strcpy(mt->atom[0].sym,dum);

  strcpy(mt->atom[1].name, h1_name);
  mt->atom[1].na = 0;
  mt->atom[1].nb = mt->atom[1].nc = -1;
  mt->atom[1].length = length;
  mt->atom[1].angle = mt->atom[1].dihedral = 0.0;
  mt->atom[1].w = wh;
  strcpy(mt->atom[1].sym,dum);
  
  strcpy(mt->atom[2].name, h2_name);
  mt->atom[2].na = 0;
  mt->atom[2].nb = 1;
  mt->atom[2].nc = -1;
  mt->atom[2].length = length;
  mt->atom[2].angle = angle;
  mt->atom[2].dihedral = 0.0;
  mt->atom[2].w = wh;
  strcpy(mt->atom[2].sym,dum);
}

void RMOL_TYPE_init_wat_4P(RMOL_TYPE *mt, char *name,
			   char *o_name, char *h1_name, char *h2_name,
			   char *m_name,
			   double length, double angle, double shift,
			   double wo, double wh)
{
  char *fname = "RMOL_TYPE_init_wat_4P";
  char *dum = "";
  
  mt->n_name = 1;
  mt->n_mol = 0;
  mt->name = emalloc(fname,sizeof(char)*5*mt->n_name);
  strcpy(mt->name[0], name);
  
  mt->n_atom = 4;
  mt->atom = emalloc(fname,sizeof(RMOL_TYPE_ATOM)*mt->n_atom);
  
  strcpy(mt->atom[0].name, o_name);
  mt->atom[0].na = mt->atom[0].nb = mt->atom[0].nc = -1;
  mt->atom[0].length = mt->atom[0].angle = mt->atom[0].dihedral = 0.0;
  mt->atom[0].w = wo;
  strcpy(mt->atom[0].sym,dum);

  strcpy(mt->atom[1].name, h1_name);
  mt->atom[1].na = 0;
  mt->atom[1].nb = mt->atom[1].nc = -1;
  mt->atom[1].length = length;
  mt->atom[1].angle = mt->atom[1].dihedral = 0.0;
  mt->atom[1].w = wh;
  strcpy(mt->atom[1].sym,dum);

  
  strcpy(mt->atom[2].name, h2_name);
  mt->atom[2].na = 0;
  mt->atom[2].nb = 1;
  mt->atom[2].nc = -1;
  mt->atom[2].length = length;
  mt->atom[2].angle = angle;
  mt->atom[2].dihedral = 0.0;
  mt->atom[2].w = wh;
  strcpy(mt->atom[2].sym,dum);

  strcpy(mt->atom[3].name, m_name);
  mt->atom[3].na = 0;
  mt->atom[3].nb = 1;
  mt->atom[3].nc = 2;
  mt->atom[3].length = length;
  mt->atom[3].angle = angle*0.5;
  mt->atom[3].dihedral = 0.0;
  mt->atom[3].w = 0.0;
  strcpy(mt->atom[3].sym,dum);
}

/*****************************************************************/
/*    RMOL_DATA                                                   */
/*****************************************************************/

void RMOL_DATA_init(RMOL_DATA *md)
{
  md->n_mol = md->n_type = 0;
  md->mol = NULL;
#ifdef MPI_SDMD  
  md->node_rmol_h = -1;
#endif  
}

void RMOL_DATA_finalize(RMOL_DATA *md)
{
  if (md->mol) free(md->mol);
  if (md->mol_type) free(md->mol_type);
}

int RMOL_DATA_n_mol(RMOL_DATA *md)
{
  return md->n_mol;
}

int RMOL_DATA_search_mol_type_name(RMOL_DATA *md, char *name)
{
  int i;
  for (i=0;i<md->n_type;i++) {
    if (strcmp(name, md->mol_type[i].name[0]) == 0) {
      return i;
    }
  }
  return -1;
}

void RMOL_DATA_set_rigid_mol_type(RMOL_DATA *md, char *name, char *fname)
{
  int i, itype, end;
  char res_name[5];

  end = 0;
  for (i=0;i<4;i++) {
    if (name[i] == '\0') end = 1;
    if (end) res_name[i] = ' ';
    else     res_name[i] = name[i];
  }
  res_name[4] = '\0';

  itype = RMOL_DATA_search_mol_type_name(md, res_name);
  if (itype < 0) {
    if (fname[0] == 0) {
      lprintf("  ERROR: Unknown rigid molecule type \"%s\"\n",res_name);
      marble_exit(1);
    } else {
      lprintf("  ERROR: Unknown rigid molecule type \"%s\" and a residue type file is not currently supported.\n",res_name);
      marble_exit(1);
    }
  }
  md->mol_type[itype].flag = 1;
}


void RMOL_DATA_crd_to_rg_quan(RMOL_DATA *md, ATOM_DATA *ad)
{
  int i,ia,j,k;
  int im,jm,km;
  VEC a1, a2;
  double len, alpha, beta, gamma;
  double m[3][3],mol_m[3][3],room_m[3][3];

  
  for (i=0;i<md->n_mol;i++) {
    /* printf("ok? %d %d\n",i, md->n_mol); */
    md->mol[i].rg.x = 0.0;
    md->mol[i].rg.y = 0.0;
    md->mol[i].rg.z = 0.0;
    for (j=0,ia=md->mol[i].parent_atom; j<md->mol[i].n_atom;
	 j++,ia=ad->ex[ia].child_list) {
      md->mol[i].rg.x += ad->x[ia].x * md->mol[i].type->atom[j].w;
      md->mol[i].rg.y += ad->x[ia].y * md->mol[i].type->atom[j].w;
      md->mol[i].rg.z += ad->x[ia].z * md->mol[i].type->atom[j].w;
    }
    md->mol[i].rg.x /= md->mol[i].type->weight;
    md->mol[i].rg.y /= md->mol[i].type->weight;
    md->mol[i].rg.z /= md->mol[i].type->weight;
    
    /*  mol(vector) = m(matrix) * room(vector) */
    /*  room_m : transpose of room */
    ia = md->mol[i].parent_atom;
    a1.x = ad->x[ia].x - md->mol[i].rg.x;
    a1.y = ad->x[ia].y - md->mol[i].rg.y;
    a1.z = ad->x[ia].z - md->mol[i].rg.z;
    len = sqrt(SQR(a1.x)+SQR(a1.y)+SQR(a1.z));
    
    room_m[0][0] = a1.x / len;
    room_m[0][1] = a1.y / len;
    room_m[0][2] = a1.z / len;

    if (md->mol[i].type->n_atom == 2) {
      if (room_m[0][0]!=0.0||room_m[0][1]!=0.0) {
	room_m[2][0] = -room_m[0][1];
	room_m[2][1] =  room_m[0][0];
	room_m[2][2] = 0.0;
      } else {
	room_m[2][0] = 0.0;
	room_m[2][1] = -room_m[0][2];
	room_m[2][2] =  room_m[0][1];
      }
    } else {
      ia=ad->ex[ia].child_list;
      a2.x = ad->x[ia].x - md->mol[i].rg.x;
      a2.y = ad->x[ia].y - md->mol[i].rg.y;
      a2.z = ad->x[ia].z - md->mol[i].rg.z;

      room_m[2][0] = a1.y * a2.z - a1.z * a2.y;
      room_m[2][1] = a1.z * a2.x - a1.x * a2.z;
      room_m[2][2] = a1.x * a2.y - a1.y * a2.x;
    }

    len = sqrt(SQR(room_m[2][0])+SQR(room_m[2][1])+SQR(room_m[2][2]));
    room_m[2][0] /= len;
    room_m[2][1] /= len;
    room_m[2][2] /= len;

    room_m[1][0] = room_m[2][1] * room_m[0][2] - room_m[2][2] * room_m[0][1];
    room_m[1][1] = room_m[2][2] * room_m[0][0] - room_m[2][0] * room_m[0][2];
    room_m[1][2] = room_m[2][0] * room_m[0][1] - room_m[2][1] * room_m[0][0];

    /***** mol_m: mol matrix  *****/

    a1.x = md->mol[i].type->atom[0].x.x;
    a1.y = md->mol[i].type->atom[0].x.y;
    a1.z = md->mol[i].type->atom[0].x.z;
    len = sqrt(SQR(a1.x)+SQR(a1.y)+SQR(a1.z));
    
    mol_m[0][0] = a1.x / len;
    mol_m[0][1] = a1.y / len;
    mol_m[0][2] = a1.z / len;

    if (md->mol[i].type->n_atom == 2) {
      if (mol_m[0][0]!=0.0||mol_m[0][1]!=0.0) {
	mol_m[2][0] = -mol_m[0][1];
	mol_m[2][1] =  mol_m[0][0];
	mol_m[2][2] = 0.0;
      } else {
	mol_m[2][0] = 0.0;
	mol_m[2][1] = -mol_m[0][2];
	mol_m[2][2] =  mol_m[0][1];
      }
    } else {
      a2.x = md->mol[i].type->atom[1].x.x;
      a2.y = md->mol[i].type->atom[1].x.y;
      a2.z = md->mol[i].type->atom[1].x.z;

      mol_m[2][0] = a1.y * a2.z - a1.z * a2.y;
      mol_m[2][1] = a1.z * a2.x - a1.x * a2.z;
      mol_m[2][2] = a1.x * a2.y - a1.y * a2.x;
    }

    len = sqrt(SQR(mol_m[2][0])+SQR(mol_m[2][1])+SQR(mol_m[2][2]));
    mol_m[2][0] /= len;
    mol_m[2][1] /= len;
    mol_m[2][2] /= len;

    mol_m[1][0] = mol_m[2][1] * mol_m[0][2] - mol_m[2][2] * mol_m[0][1];
    mol_m[1][1] = mol_m[2][2] * mol_m[0][0] - mol_m[2][0] * mol_m[0][2];
    mol_m[1][2] = mol_m[2][0] * mol_m[0][1] - mol_m[2][1] * mol_m[0][0];
    
    for (im=0;im<3;im++) {
      for (jm=0;jm<3;jm++) {
	m[im][jm] = 0.0;
	for (km=0;km<3;km++) {
	  m[im][jm] += mol_m[km][im] * room_m[km][jm];
	}
      }
    }
    /*
    { double v[3];
    for (jm=0;jm<3;jm++) {
      v[jm] = m[jm][0] * md->mol[i].type->atom[1].x +
	      m[jm][1] * md->mol[i].type->atom[1].y +
	      m[jm][2] * md->mol[i].type->atom[1].z;
    }
    v[0] += md->mol[i].rg.x;
    v[1] += md->mol[i].rg.y;
    v[2] += md->mol[i].rg.z;
      printf("%f %f %f\n",v[0],v[1],v[2]);
    }
    */
    if (fabs(m[0][2]) < EPS && fabs(m[1][2]) < EPS) {
      /* beta == 0 or M_PI */
      if (m[2][2] > 0.0) {
	beta = 0.0;
	alpha = atan2(m[0][0], m[0][1]) / 2.0;
	gamma = alpha;
      } else {
	beta = 1.0;
	alpha = atan2(m[0][0], m[0][1]) / 2.0;
	gamma = -alpha;
      }
    } else {
      /* normal case */
      gamma = atan2(m[0][2], m[1][2]);
      alpha = atan2(m[2][0],-m[2][1]);
    
      if (fabs(cos(gamma)) > fabs(sin(gamma))) {
	beta  = atan2(m[1][2]/cos(gamma),m[2][2]);
      } else {
	beta  = atan2(m[0][2]/sin(gamma),m[2][2]);
      }
    }

#ifdef OLDQUAN
    md->mol[i].q.qw = sin(0.5*beta)*sin((gamma-alpha)*0.5);
    md->mol[i].q.qx = sin(0.5*beta)*cos((gamma-alpha)*0.5);
    md->mol[i].q.qy = cos(0.5*beta)*sin((gamma+alpha)*0.5);
    md->mol[i].q.qz = cos(0.5*beta)*cos((gamma+alpha)*0.5);
#else
    md->mol[i].q.qw = cos(0.5*beta)*cos((alpha+gamma)*0.5);
    md->mol[i].q.qx = sin(0.5*beta)*cos((alpha-gamma)*0.5);
    md->mol[i].q.qy = sin(0.5*beta)*sin((alpha-gamma)*0.5);
    md->mol[i].q.qz = cos(0.5*beta)*sin((alpha+gamma)*0.5);
#endif
  }
}

void RMOL_DATA_set_velocity_zero(RMOL_DATA *md)
{
  int i;

  for (i=0;i<md->n_mol;i++) {
    md->mol[i].vg.x = 0.0;
    md->mol[i].vg.y = 0.0;
    md->mol[i].vg.z = 0.0;
    md->mol[i].l.x = 0.0;
    md->mol[i].l.y = 0.0;
    md->mol[i].l.z = 0.0;
    md->mol[i].wp.x = 0.0;
    md->mol[i].wp.y = 0.0;
    md->mol[i].wp.z = 0.0;
    md->mol[i].wp_half.x = 0.0;
    md->mol[i].wp_half.y = 0.0;
    md->mol[i].wp_half.z = 0.0;
  }
}

void RMOL_DATA_Maxwell_velocity(RMOL_DATA *md, double temperature)
{
  double rtemp;
  int i;
  RIGID_MOL *rm;

  for (i=0;i<md->n_mol;i++) {
    rm = &md->mol[i];
    rtemp = sqrt(KCAL*K*temperature/rm->type->weight);
    rm->vg.x = rtemp * gauss_rand();
    rm->vg.y = rtemp * gauss_rand();
    rm->vg.z = rtemp * gauss_rand();
    rtemp = sqrt(KCAL*K*temperature/rm->type->ip.x);
    rm->wp.x = rtemp * gauss_rand();
    rtemp = sqrt(KCAL*K*temperature/rm->type->ip.y);
    rm->wp.y = rtemp * gauss_rand();
    if (rm->type->ip.z != 0.0) {
      rtemp = sqrt(KCAL*K*temperature/rm->type->ip.z);
      rm->wp.z = rtemp * gauss_rand();
    } else {
      rm->wp.z = 0.0;
    }
    rm->l.x = rm->wp.x * rm->type->ip.x;
    rm->l.y = rm->wp.y * rm->type->ip.y;
    rm->l.z = rm->wp.z * rm->type->ip.z;
  }
}

void RMOL_DATA_scale_velocity(RMOL_DATA *md, double *scale_tr, double *scale_rot)
{
  int i,iex;

#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif
    iex = md->mol[i].iex;
    md->mol[i].vg.x *= scale_tr[iex];
    md->mol[i].vg.y *= scale_tr[iex];
    md->mol[i].vg.z *= scale_tr[iex];
    md->mol[i].l.x  *= scale_rot[iex];
    md->mol[i].l.y  *= scale_rot[iex];
    md->mol[i].l.z  *= scale_rot[iex];
  }
}

void RMOL_DATA_scale_velocity_full(RMOL_DATA *md, double scale_tr[MAX_EX_SYSTEM][3][3],
				   double scale_rot[MAX_EX_SYSTEM])
{
  int i,iex;
  double vx, vy, vz;

#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif
    iex = md->mol[i].iex;
    
    vx = VEC_MUL_MAT_X(md->mol[i].vg,scale_tr[iex]);
    vy = VEC_MUL_MAT_Y(md->mol[i].vg,scale_tr[iex]);
    vz = VEC_MUL_MAT_Z(md->mol[i].vg,scale_tr[iex]);

    md->mol[i].vg.x = vx;
    md->mol[i].vg.y = vy;
    md->mol[i].vg.z = vz;

    md->mol[i].l.x  *= scale_rot[iex];
    md->mol[i].l.y  *= scale_rot[iex];
    md->mol[i].l.z  *= scale_rot[iex];
  }
}

void RMOL_DATA_setup(RMOL_DATA *md, ATOM_DATA *ad, BOND_DATA *bd,
		     ANGLE_DATA *ag, char *fname)
{
  if (strcmp(fname, "hydrogen") == 0)
    RMOL_DATA_setup_hydrogen(md, ad, bd, ag);
  else if (strcmp(fname, "water") == 0)
    RMOL_DATA_setup_water(md, ad);
  else
    RMOL_DATA_read_library(md, ad, fname);
}

void RMOL_DATA_read_library(RMOL_DATA *md, ATOM_DATA *ad, char *fname)
{
  FILE *fp;

  if ((fp = fopen(fname,"r")) == NULL) {
    lprintf("ERROR: %s: No such rigid group library file\n", fname);
    marble_exit(1);
  }
  if (RMOL_DATA_read_library_fp(md, ad, fp))
    marble_exit(1);
  fclose(fp);
}

int RMOL_DATA_read_library_fp(RMOL_DATA *md, ATOM_DATA *ad, FILE *fp)
{
  int i, j, pp, atom_no;
  char buf[1000], buf2[1000], *pbuf;
  RMOL_TYPE *mt;
  char *fname = "RMOL_DATA_read_library";

  com_fgets(buf,1000,fp);
  
  if (sscanf(buf,"%d",&(md->n_type)) != 1) {
    lprintf("ERROR: %s: invalid n_type\n", fname);
    return 1;
  }

  md->n_alloc_type = md->n_type;
  if (md->n_type == 0) {
    md->n_mol = 0;
    return 0;
  }
  md->mol_type = emalloc(fname,sizeof(RMOL_TYPE)*md->n_type);

  for (i=0;i<md->n_type;i++) {
    mt=&(md->mol_type[i]);
    mt->n_mol = 0;
    mt->flag = 0;
    if (com_fgets(buf,1000,fp) == NULL) {
      md->n_type = i;
      break;
    }
    pbuf = buf;
    mt->n_name = 0;
    while (sscanf(pbuf,"%s %n",buf2,&pp) == 1) {
      mt->n_name++;
      pbuf += pp;
    }
    mt->name = emalloc(fname,sizeof(char)*5*mt->n_name);
    pbuf = buf;
    for (j=0;j<mt->n_name;j++) {
      sscanf(pbuf,"%s %n",mt->name[j],&pp);
      /* fill_space(mt->name[j],4); */
      pbuf+=pp;
    }
    com_fgets(buf,1000,fp);
    sscanf(buf,"%d", &(mt->n_atom));
    /* DEBUG  
    printf("[%s], %d, %d\n", name, mol_type, n_atom); */
    mt->atom = emalloc(fname,sizeof(RMOL_TYPE_ATOM)*mt->n_atom);
    for (j=0;j<mt->n_atom;j++) {
      com_fgets(buf,1000,fp);
      /* if (sscanf(buf, "%d %4[ A-Za-z0-9+'*\\-] %d%d%d%lf%lf%lf", */
      if (sscanf(buf, "%d %4s %d%d%d%lf%lf%lf",
		 &atom_no,
		 mt->atom[j].name, 
		 &(mt->atom[j].na),&(mt->atom[j].nb),&(mt->atom[j].nc),
		 &(mt->atom[j].length),&(mt->atom[j].angle),
		 &(mt->atom[j].dihedral)) != 8) {
	lprintf("ERROR: %s: Invalid format of atom (%d) in rigid_group (%d).\n",
		fname, j+1,i+1);
	return 1;
      }
      mt->atom[j].na--;
      mt->atom[j].nb--;
      mt->atom[j].nc--;

      if ((j==1&& mt->atom[j].na<0) ||
	  (j==2&&(mt->atom[j].na<0||mt->atom[j].nb<0)) ||
	  (j>=3&&(mt->atom[j].na<0||mt->atom[j].nb<0||mt->atom[j].nc<0)))  {
	lprintf("ERROR: wrong definition in internal coordinates:\n%s",buf);
	return 1;
      }
    }
    com_fgets(buf,1000,fp);
    if (strncmp(buf,"END",3) != 0) {
      lprintf("ERROR: Invalid format in rigid_group (%d): No END\n",i);
      lprintf(buf);
      return 1;
    }
  }

  lprintf("  Number of types of rigid body: %d\n", md->n_type);
  
  RMOL_DATA_make_rmol(md, ad);
  RMOL_DATA_initialize_types(md, ad);

  return 0;
}

void RMOL_DATA_make_rmol(RMOL_DATA *md, ATOM_DATA *ad)
{
  int i,j,k,l, parent_atom, imol, ok, prev;
  RMOL_TYPE *mt;
  char *fname = "RMOL_DATA_make_rmol";

  /* Count number of molecules */
  imol=0;
  for (i=0;i<ad->nres;i++) {
    for (j=0;j<md->n_type;j++) {
      mt = &md->mol_type[j];
      if (RMOL_TYPE_eq_residue(mt,&ad->r[i])) {
	for (k=0;k<mt->n_atom;k++) {
	  ok=0;
	  for (l=ad->r[i].start_atom;l<=ad->r[i].end_atom;l++) {
	    if (ad->ex[l].flag & ATOM_FIXED) continue;

	    if (strcmp(mt->atom[k].name,ad->a[l].name)==0) {
	      if (ad->ex[l].flag & ATOM_RIGID) continue;
	      ok=1;
	      if (k==0) {
		parent_atom = l;
	      }
	      break;
	    }
	  }
	  if (!ok) break;
	}
	if (ok) {
	  if (ad->ex[parent_atom].flag & ATOM_RIGID) continue;
	  
	  imol++;
	  ad->ex[parent_atom].flag |= (ATOM_PARENT | ATOM_RIGID);
	  ad->ex[parent_atom].parent = j;
	  mt->n_mol++;
	  
	  for (k=0;k<mt->n_atom;k++) {
	    for (l=ad->r[i].start_atom;l<=ad->r[i].end_atom;l++) {
	      if (strcmp(mt->atom[k].name,ad->a[l].name)==0) 
		ad->ex[l].flag |= ATOM_RIGID;
	    }
	  }
	}
      }
    }
  }

  md->n_mol = imol;
  md->mol = emalloc(fname, sizeof(RIGID_MOL)*md->n_mol);
#if defined(MPI_RDMD) || defined(MPI_SDMD)
  md->crd = emalloc(fname, sizeof(RMOL_CRD)*md->n_mol);
#endif  

  imol=0;
  for (i=0;i<ad->nres;i++) {
    for (j=ad->r[i].start_atom;j<=ad->r[i].end_atom;j++) {
      if ((ad->ex[j].flag & ATOM_PARENT) && (ad->ex[j].flag & ATOM_RIGID)) {
	mt = &md->mol_type[ad->ex[j].parent];
	md->mol[imol].type = mt;
	md->mol[imol].parent_atom = j;
	md->mol[imol].n_atom = mt->n_atom;
	ad->ex[j].parent = imol;
	ad->ex[j].child_list = -1;
	mt->atom[0].w = ad->w[j];
	prev = j;
	imol++;
	for (k=1;k<mt->n_atom;k++) {
	  for (l=ad->r[i].start_atom;l<=ad->r[i].end_atom;l++) {
	    if (strcmp(mt->atom[k].name,ad->a[l].name)==0) {
	      /* because we already checked above 
	      if (ad->ex[l].flag & ATOM_RIGID) {
		lprintf("ERROR: Atom %d is doubly counted in rigid body.\n",
			l+1);
		exit(1);
	      }
	      */
	      ad->ex[l].flag |= ATOM_RIGID;
	      ad->ex[l].flag |= ATOM_CHILD;
	      ad->ex[l].parent = j;
	      ad->ex[l].child_list = -1;
	      ad->ex[prev].child_list = l;
	      prev = l;
	      mt->atom[k].w = ad->w[l];
	    }
	  }
	}
      }
    }
  }
  
  md->n_mol6 = md->n_mol5 = 0;
  for (i=0;i<md->n_mol;i++) {
    if (md->mol[i].type->n_atom == 2)
      md->n_mol5++;
    else
      md->n_mol6++;
  }

  
#ifdef MPI_SDMD
  md->node_rmol_h = 0;
  for (i=0;i<md->n_mol-1;i++) {
    md->mol[i].node_rmol_n = i+1;
  }
  if (md->n_mol>0)
    md->mol[md->n_mol-1].node_rmol_n = -1;
  else
    md->node_rmol_h = -1;
  ATOM_DATA_set_node_fatom(ad);
#endif
  
  lprintf("  Number of rigid groups: %d (full:%d, line:%d)\n",md->n_mol,md->n_mol6,md->n_mol5);
#ifdef NO_SQUISH  
  lprintf("  Integrator: NO_SQUISH\n");
#else
  lprintf("  Integrator: Matubayasi\n");
#endif  
  /*
  for (imol=0;imol<md->n_mol;imol++) {
    mt = md->mol[imol].type;
    j=md->mol[imol].parent_atom;
    lprintf("%d[%s][%s]%d\n",imol,ad->a[j].name, ad->r[ad->a[j].resno].name,j);
  }
  */
}

void RMOL_DATA_initialize_types(RMOL_DATA *md, ATOM_DATA *ad)
{
  int i, j, k;
  RMOL_TYPE *mt;

  for (i=0;i<md->n_type;i++) {
    if (md->mol_type[i].n_mol > 0) {
      RMOL_TYPE_internal_to_xyz(&md->mol_type[i]);
      RMOL_TYPE_moment_of_inertia(&md->mol_type[i]); 
    }
  }
}

static double RMOL_DATA_standard_length(BOND_DATA *bd, int atom1, int atom2)
{
  double val;

  val = BOND_DATA_standard_length(bd, atom1, atom2);
  if (val == 0.0) {
    lprintf("ERROR: No standard length of atoms (%d-%d)\n", atom1+1, atom2+1);
    marble_exit(1);
  }
  return val;
}

static double RMOL_DATA_standard_angle(ANGLE_DATA *and, BOND_DATA *bd, int atom1, int atom2, int atom3)
{
  double val, val1, val2;

  val = ANGLE_DATA_standard_angle(and, atom1, atom2, atom3);
  if (val == 0.0) {
    val = BOND_DATA_standard_length(bd, atom1, atom3);
    if (val == 0.0) {
      lprintf("ERROR: No standard angles of atoms (%d-%d-%d)\n", atom1+1, atom2+1, atom3+1);
      marble_exit(1);
    }
    val1 = BOND_DATA_standard_length(bd, atom1, atom2);
    val2 = BOND_DATA_standard_length(bd, atom2, atom3);

    val = acos((val1*val1+val2*val2-val*val)/(2.0*val1*val2));
  }
  return val * 180.0 / M_PI;
}

void RMOL_DATA_setup_water(RMOL_DATA *md, ATOM_DATA *ad)
{
  char *fname = "RMOL_DATA_setup_water";
  int i;
  
  md->n_alloc_type = md->n_type = _n_water_data;
  md->mol_type = emalloc(fname,sizeof(RMOL_TYPE)*md->n_alloc_type);

  for (i=0;i<md->n_type;i++) {
    if (_water_data[i].n_atom == 3) {
      RMOL_TYPE_init_wat_3P(&md->mol_type[i], _water_data[i].name,
			    _water_data[i].o_name, _water_data[i].h1_name,
			    _water_data[i].h2_name,
			    _water_data[i].bond,   _water_data[i].angle,
			    _water_data[i].wo, _water_data[i].wh);
    } else if (_water_data[i].n_atom == 4) {
      RMOL_TYPE_init_wat_4P(&md->mol_type[i], _water_data[i].name,
			    _water_data[i].o_name, _water_data[i].h1_name,
			    _water_data[i].h2_name, _water_data[i].m_name,
			    _water_data[i].bond,   _water_data[i].angle,
			    _water_data[i].shift,
                            _water_data[i].wo, _water_data[i].wh);
    } else {
      lprintf("ERROR: Internal Error water atom # is not 3 or 4\n");
      marble_exit(1);
    }
    
  }

  RMOL_DATA_make_rmol(md, ad);
  RMOL_DATA_initialize_types(md, ad);
}

void RMOL_DATA_setup_hydrogen(RMOL_DATA *md, ATOM_DATA *ad, BOND_DATA *bd, ANGLE_DATA *and)
{
  int i, j, n, ok;
  int atom1,atom2, atom1_parent, atom2_parent, parent, child;
  int atom_no[10], ia;
  char *p1,*p2;
  RMOL_TYPE *mt;
  char *fname = "RMOL_DATA_setup_hydrogen";
  double ang, xy, hh2;
    
  for (i=0; i<ad->natom; i++) {
    if (ad->ex[i].flag & ATOM_RIGID) continue;
    ad->ex[i].child_list = -1;
    ad->ex[i].parent = i;
  }
  
  for (i=0;i<bd->n_bond;i++) {
    /* Consider that we merge two groups... */
    atom1 = bd->bonds[i].atom1;
    atom2 = bd->bonds[i].atom2;
    
    p1 = get_atom_sym(ad,atom1);
    p2 = get_atom_sym(ad,atom2);
    
    if (*p1 != 'H' && *p2 != 'H') continue;

    if (*p1 == 'H' && *p2 == 'H') continue;
    /* Both atoms are hydrogens -- omit */

    /* Here, Hydrogen bonds ... */
    
    if ((ad->ex[atom1].flag & ATOM_FIXED) &&
	(ad->ex[atom2].flag & ATOM_FIXED)) continue;
    /* Both atoms are fixed. */
    
    /* If either atom is fixed, error. */
    if ((ad->ex[atom1].flag & ATOM_FIXED) ||
	(ad->ex[atom2].flag & ATOM_FIXED)) {
      lprintf("ERROR: No fixed atom can join a rigid body.\nInconsistent atoms:\n");
      ATOM_DATA_print_atom(ad, atom1);
      ATOM_DATA_print_atom(ad, atom2);
      marble_exit(1);
    }
    
    atom1_parent = ad->ex[atom1].parent;
    atom2_parent = ad->ex[atom2].parent;

    /* if parents are common, do nothing. */
    if (atom1_parent == atom2_parent) continue;

    p1 = get_atom_sym(ad,atom1_parent);
    p2 = get_atom_sym(ad,atom2_parent);

    /* One is the heavy atom and the other is hydrogen */
    if (*p1 != 'H') {
      parent = atom1_parent;
      child  = atom2_parent;
    } else {
      parent = atom2_parent;
      child  = atom1_parent;
    }

    if (ad->ex[child].flag & ATOM_RIGID) {
      lprintf("ERROR: hydrogen atom %d already connects another heavy atom.\n", child+1);
      marble_exit(1);
    }
    
    ad->ex[atom1].flag |= ATOM_RIGID;
    ad->ex[atom2].flag |= ATOM_RIGID;
    ad->ex[parent].flag |= ATOM_PARENT;
    ad->ex[child].flag  &= ~ATOM_PARENT;
    ad->ex[child].flag  |= ATOM_CHILD;
    ad->ex[child].parent = parent;

    /* job for child_list */
    for (j=parent;j>=0;j=ad->ex[j].child_list) {
      if (child < ad->ex[j].child_list || ad->ex[j].child_list < 0) {
	ad->ex[child].child_list = ad->ex[j].child_list;
	ad->ex[j].child_list = child;
	break;
      }
    }
  }

  /* count group */
  md->n_mol = 0;
  for (i=0;i<ad->natom;i++)
    if (ad->ex[i].flag & ATOM_PARENT)
      md->n_mol++;

  md->mol = emalloc(fname, sizeof(RIGID_MOL)*md->n_mol);
#if defined(MPI_RDMD) || defined(MPI_SDMD)
  md->crd = emalloc(fname, sizeof(RMOL_CRD)*md->n_mol);
#endif  
  
  n = 0;
  md->n_mol6 = md->n_mol5 = 0;
  for (i=0;i<ad->natom;i++) {
    if (ad->ex[i].flag & ATOM_PARENT) {
      md->mol[n].type = NULL;
      md->mol[n].parent_atom = i;
#ifdef MPI_SDMD      
      md->mol[n].node_rmol_n = n + 1;
#endif      
      ad->ex[i].parent = n;
      md->mol[n].n_atom = 1;
      for (j=ad->ex[i].child_list;j>=0;j=ad->ex[j].child_list) {
	md->mol[n].n_atom++;
      }
      if (md->mol[n].n_atom == 2) {
	md->n_mol5++;
      } else {
	md->n_mol6++;
      }
      n++;
    }
  }

#ifdef MPI_SDMD  
  if (md->n_mol == 0)
    md->node_rmol_h = -1;
  else {
    md->node_rmol_h = 0;
    md->mol[n-1].node_rmol_n = -1;
  }
  ATOM_DATA_set_node_fatom(ad);
#endif  

  /* here we make mol_type */
  md->n_alloc_type = 100;
  md->mol_type = emalloc(fname,sizeof(RMOL_TYPE)*md->n_alloc_type);

  md->n_type = 0;
  for (n=0;n<md->n_mol;n++) {
    /* search mol type */
    ok = 0;
    for (j=0;j<md->n_type;j++) {
      if (md->mol[n].n_atom != md->mol_type[j].n_atom) continue;
      ok = 1;
      for (i=md->mol[n].parent_atom,ia=0;i>=0;i=ad->ex[i].child_list,ia++) {
	p1 = get_atom_sym(ad,i);
	if (strcmp(p1, md->mol_type[j].atom[ia].sym) != 0) {
	  ok = 0;
	  break;
	}
      }
      if (ok) break;
    }
    if (ok) {
      md->mol[n].type = &md->mol_type[j];
      md->mol_type[j].n_mol++;
      continue;
    }
    /* here we have to make mol_type */
    md->n_type++;
    if (md->n_type > md->n_alloc_type) {
      md->n_alloc_type += 100;
      md->mol_type = erealloc(fname, md->mol_type, sizeof(RMOL_TYPE)*md->n_alloc_type);
    }
    mt = &md->mol_type[md->n_type-1];
    md->mol[n].type = mt;
    mt->n_mol = 1;
    mt->n_atom = md->mol[n].n_atom;
    mt->atom = emalloc(fname, sizeof(RMOL_TYPE_ATOM)*mt->n_atom);
    mt->n_name = 1;
    mt->name = emalloc(fname,sizeof(char)*5*mt->n_name);
    strcpy(mt->name[0], "");
    for (i=md->mol[n].parent_atom,ia=0;i>=0;i=ad->ex[i].child_list,ia++) {
      strcpy(mt->atom[ia].sym, get_atom_sym(ad,i));
      strcpy(mt->atom[ia].name, "");
      mt->atom[ia].w = ad->w[i];
      atom_no[ia] = i;
      switch (ia) {
      case 0:
	mt->atom[ia].na = -1;
	mt->atom[ia].nb = -1;
	mt->atom[ia].nc = -1;
	mt->atom[ia].length   = 0.0;
	mt->atom[ia].angle    = 0.0;
	mt->atom[ia].dihedral = 0.0;
	break;
      case 1:
	mt->atom[ia].na = 0;
	mt->atom[ia].nb = -1;
	mt->atom[ia].nc = -1;
	mt->atom[ia].length = RMOL_DATA_standard_length(bd,atom_no[0],atom_no[1]);
	mt->atom[ia].angle  = 0.0;
	mt->atom[ia].dihedral = 0.0;
	break;
      case 2:
	if (strcmp(mt->atom[1].sym,mt->atom[2].sym)!=0) {
	  lprintf("ERROR: Connected hydrogen is not same symbol! (%d,%d)\n",
		  atom_no[1]+1,atom_no[2]+1);
	  marble_exit(1);
	}
	mt->atom[ia].na = 0;
	mt->atom[ia].nb = 1;
	mt->atom[ia].nc = -1;
	mt->atom[ia].length = RMOL_DATA_standard_length(bd,atom_no[0],atom_no[2]);
	mt->atom[ia].angle  = RMOL_DATA_standard_angle(and,bd,atom_no[1],atom_no[0],atom_no[2]);
	mt->atom[ia].dihedral = 0.0;
	break;
      case 3:
	if (strcmp(mt->atom[2].sym, mt->atom[3].sym)!=0) {
	  lprintf("ERROR: Hydrogens in the same group are not same symbol! (%d,%d)\n",
		  atom_no[2]+1,atom_no[3]+1);
	  marble_exit(1);
	}
	mt->atom[ia].na = 0;
	mt->atom[ia].nb = 1;
	mt->atom[ia].nc = 2;
	mt->atom[ia].length = RMOL_DATA_standard_length(bd,atom_no[0],atom_no[3]);
	mt->atom[ia].angle  = RMOL_DATA_standard_angle(and,bd,atom_no[1],atom_no[0],atom_no[3]);
	  
	ang = mt->atom[ia].angle * M_PI / 180.0;
	xy  = sin(ang);
	hh2 = 2.0*(1.0-cos(ang));
	mt->atom[ia].dihedral = acos((2.0*xy*xy-hh2)/(2*xy*xy)) * 180.0 / M_PI;
	
	break;
      default :
	lprintf("ERROR: Number of hydrogens in the same group (%d) must be less than 3.\n",
		atom_no[0]+1);
	marble_exit(1);
      }
    }
  }
  
  lprintf("  Number of rigid groups: %d (full:%d, line:%d)\n",md->n_mol,md->n_mol6,md->n_mol5);
  lprintf("  Number of types of rigid body: %d\n", md->n_type);
#ifdef NO_SQUISH  
  lprintf("  Integrator: NO_SQUISH\n");
#else
  lprintf("  Integrator: Matubayasi\n");
#endif  
  RMOL_DATA_initialize_types(md, ad);
}

int RMOL_DATA_read_crd(RMOL_DATA *md, FILE *fp, int skip)
{
  int i, n_mol;

  fscanf(fp,"%d",&n_mol);
  if (n_mol != md->n_mol || skip) {
    if (!skip)
      lprintf("Warning: Number of rigid groups in crd file (%d) is not equal to number of current data (%d)\n", n_mol, md->n_mol);
    for (i=0; i<n_mol; i++) {
      fscanf(fp,"%*lf%*lf%*lf%*lf%*lf%*lf%*lf");
      fscanf(fp,"%*lf%*lf%*lf%*lf%*lf%*lf");
    }
    return -1;
  }

  for (i=0; i<md->n_mol; i++) {
    if (fscanf(fp,"%lf%lf%lf%lf%lf%lf%lf",
	       &(md->mol[i].rg.x),&(md->mol[i].rg.y), &(md->mol[i].rg.z),
	       &(md->mol[i].q.qw),&(md->mol[i].q.qx),
	       &(md->mol[i].q.qy),&(md->mol[i].q.qz)) != 7) {
      return -1;
    }
    QUAN_normalize(&(md->mol[i].q));
    if (fscanf(fp,"%lf%lf%lf%lf%lf%lf",
	       &(md->mol[i].vg.x),&(md->mol[i].vg.y), &(md->mol[i].vg.z),
	       &(md->mol[i].l.x),&(md->mol[i].l.y), &(md->mol[i].l.z)) != 6) {
      return -1;
    }
  }
  return 0;
}

void RMOL_DATA_write_crd(RMOL_DATA *md, FILE *fp)
{
  int i;

  fprintf(fp,"%d\n",md->n_mol);

  for (i=0; i<md->n_mol; i++) {
    fprintf(fp,"%13e %13e %13e %13e %13e %13e %13e\n",
	   md->mol[i].rg.x, md->mol[i].rg.y, md->mol[i].rg.z,
	   md->mol[i].q.qw, md->mol[i].q.qx,
	   md->mol[i].q.qy, md->mol[i].q.qz);
    fprintf(fp,"%13e %13e %13e %13e %13e %13e\n",
	   md->mol[i].vg.x,md->mol[i].vg.y,md->mol[i].vg.z,
	   md->mol[i].l.x, md->mol[i].l.y, md->mol[i].l.z);
  }
}


void RMOL_DATA_print(RMOL_DATA *md, FILE *fp)
{
  int i;
  for (i=0; i<md->n_mol; i++) {
    fprintf(fp,"%f %f %f %f %f %f %f\n", (md->mol[i].rg.x),(md->mol[i].rg.y),
	   (md->mol[i].rg.z),   (md->mol[i].q.qw),(md->mol[i].q.qx),
	   (md->mol[i].q.qy),   (md->mol[i].q.qz));
  }
}

void RMOL_DATA_mol_to_room(RMOL_DATA *md, ATOM_DATA *ad)
{
  int i;

#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    RIGID_MOL_mol_to_room(&(md->mol[i]),ad);
  }
}

void RMOL_DATA_mol_to_room_all(RMOL_DATA *md, ATOM_DATA *ad)
{
  int i;

  for (i = 0; i < md->n_mol; i++) {
    RIGID_MOL_mol_to_room(&(md->mol[i]),ad);
  }
}

void RMOL_DATA_mol_to_room_with_check(RMOL_DATA *md, ATOM_DATA *ad)
{
  double eps=2.0,len2,eps2;
  int i;
  
  for (i=0;i<ad->natom;i++) ad->f[i]=ad->x[i];

  RMOL_DATA_mol_to_room(md, ad);

  eps2=eps*eps;

  for (i=0;i<ad->natom;i++) {
    len2=SQR(ad->f[i].x - ad->x[i].x)
      +SQR(ad->f[i].y - ad->x[i].y)+SQR(ad->f[i].z - ad->x[i].z);
    if (len2>eps2) {
      lprintf("%d %f=(%f %f %f)-(%f %f %f)\n",i+1,sqrt(len2),
	      ad->x[i].x,ad->x[i].y,ad->x[i].z,
	      ad->f[i].x,ad->f[i].y,ad->f[i].z);
    }
  }
}


/* caution: each kene value is twice as much as real value */ 
void RMOL_DATA_kene(RMOL_DATA *md, double *kene_t, double *kene_r, double *kene_arr)
{
  int i;
  RIGID_MOL *rm;
  double kt, kr, kt_all, kr_all, kene_arr_all[MAX_EX_SYSTEM];

#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    rm = &md->mol[i];
    kt = rm->type->weight * Length2(rm->vg.x,rm->vg.y,rm->vg.z);
    kr = rm->l.x * rm->l.x / rm->type->ip.x + rm->l.y * rm->l.y / rm->type->ip.y;
    if (rm->type->ip.z != 0.0)
      kr += rm->l.z * rm->l.z / rm->type->ip.z;

    *kene_t += kt;
    *kene_r += kr;
    kene_arr[rm->iex] += kt+kr;
  }
}

/* caution: each kene value is twice as much as real value */ 
void RMOL_DATA_kene_full(RMOL_DATA *md, double *kene_t, double *kene_r, int n_ex_system,
			 double kene_tr_arr[MAX_EX_SYSTEM][3][3], double kene_rot_arr[MAX_EX_SYSTEM])
{
  int i,iex,k,l;
  RIGID_MOL *rm;
  RMOL_TYPE *mt;
  double ktx, kty, ktz;
  
#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    rm = &md->mol[i];
    mt = rm->type;
    iex = rm->iex;

    ktx = mt->weight * rm->vg.x * rm->vg.x;
    kty = mt->weight * rm->vg.y * rm->vg.y;
    ktz = mt->weight * rm->vg.z * rm->vg.z;

    *kene_t += ktx + kty + ktz;
    
    kene_tr_arr[iex][0][0] += ktx;
    kene_tr_arr[iex][0][1] += mt->weight * rm->vg.x * rm->vg.y;
    kene_tr_arr[iex][0][2] += mt->weight * rm->vg.x * rm->vg.z;
    kene_tr_arr[iex][1][1] += kty;
    kene_tr_arr[iex][1][2] += mt->weight * rm->vg.y * rm->vg.z;
    kene_tr_arr[iex][2][2] += ktz;

    kene_rot_arr[iex] += rm->l.x * rm->l.x / mt->ip.x + rm->l.y * rm->l.y / mt->ip.y;
    if (mt->ip.z != 0.0)
      kene_rot_arr[iex] += rm->l.z * rm->l.z / mt->ip.z;
  }

  for (iex=0;iex<n_ex_system;iex++) {
    *kene_r += kene_rot_arr[iex];
  }
}

void RMOL_DATA_leapfrog(RMOL_DATA *md, ATOM_DATA *ad, double dt)
{
  int i;

#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    RIGID_MOL_force_and_torque(&(md->mol[i]), ad);
    RIGID_MOL_leapfrog(&(md->mol[i]),dt);
    RIGID_MOL_mol_to_room(&(md->mol[i]),ad);
  }
}

void RMOL_DATA_init_time0(RMOL_DATA *md, ATOM_DATA *ad)
{
  int i;

#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    RIGID_MOL_force_and_torque(&(md->mol[i]), ad);
    RIGID_MOL_torq_room_to_mol(&(md->mol[i]));
  }
}

void RMOL_DATA_time_integration_v1(RMOL_DATA *md, ATOM_DATA *ad, double dt)
{
  int i;
#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif
#ifdef NO_SQUISH    
    RIGID_MOL_time_integration_v1_no_squish(&(md->mol[i]),dt);
#else    
    RIGID_MOL_time_integration_v1(&(md->mol[i]),dt);
#endif    
  }
}

void RMOL_DATA_time_integration_p(RMOL_DATA *md, ATOM_DATA *ad, double dt)
{
  int i;
  
#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    RIGID_MOL_time_integration_p_rg(&(md->mol[i]),dt);
    
#ifdef NO_SQUISH    
    RIGID_MOL_time_integration_p_rot_no_squish(&(md->mol[i]),dt, 1);
#else    
    RIGID_MOL_time_integration_p_rot(&(md->mol[i]),dt);
#endif
    
    RIGID_MOL_mol_to_room(&(md->mol[i]),ad);
  }
}

void RMOL_DATA_time_integration_p_NPT(RMOL_DATA *md, ATOM_DATA *ad, double dt,
				      double AA2, double BB)
{
  int i;
  
#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    RIGID_MOL_time_integration_p_rg_NPT(&(md->mol[i]),AA2, BB);
    
#ifdef NO_SQUISH    
    RIGID_MOL_time_integration_p_rot_no_squish(&(md->mol[i]),dt, 1);
#else    
    RIGID_MOL_time_integration_p_rot(&(md->mol[i]),dt);
#endif
    
    RIGID_MOL_mol_to_room(&(md->mol[i]),ad);
  }
}


void RMOL_DATA_time_integration_p_NPT_full(RMOL_DATA *md, ATOM_DATA *ad, double dt,
					   double Vg_vec[3][3], double AA2[3], double BB[3])
{
  int i;
  
#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    RIGID_MOL_time_integration_p_rg_NPT_full(&(md->mol[i]),Vg_vec, AA2, BB);
    
#ifdef NO_SQUISH    
    RIGID_MOL_time_integration_p_rot_no_squish(&(md->mol[i]),dt, 1);
#else    
    RIGID_MOL_time_integration_p_rot(&(md->mol[i]),dt);
#endif
    
    RIGID_MOL_mol_to_room(&(md->mol[i]),ad);
  }
}

void RMOL_DATA_time_integration_v2(RMOL_DATA *md, ATOM_DATA *ad, double dt)
{
  int i;
  
#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    RIGID_MOL_force_and_torque(&(md->mol[i]), ad);
    RIGID_MOL_torq_room_to_mol(&(md->mol[i]));
#ifdef NO_SQUISH    
    RIGID_MOL_time_integration_v2_no_squish(&(md->mol[i]),dt);
#else    
    RIGID_MOL_time_integration_v2(&(md->mol[i]),dt);
#endif    
  }
}

void RMOL_DATA_set_direction(RMOL_DATA *md, ATOM_DATA *ad, double c)
{
  int i;
  
#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    RIGID_MOL_force_and_torque(&(md->mol[i]), ad);
    RIGID_MOL_torq_room_to_mol(&(md->mol[i]));
    RIGID_MOL_set_direction(&(md->mol[i]),c);
  }
}
  

void RMOL_DATA_set_atom_velocity(RMOL_DATA *md, ATOM_DATA *ad)
{
  int i;
#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    RIGID_MOL_set_atom_velocity(&md->mol[i], ad);
  }
}

void RMOL_DATA_set_mol_velocity(RMOL_DATA *md, ATOM_DATA *ad)
{
  int i;
#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif    
    RIGID_MOL_set_mol_velocity(&md->mol[i], ad);
  }
}

void RMOL_DATA_set_atom_velocity_all(RMOL_DATA *md, ATOM_DATA *ad)
{
  int i;
  for (i = 0; i < md->n_mol; i++) {
    RIGID_MOL_set_atom_velocity(&md->mol[i], ad);
  }
}

void RMOL_DATA_set_mol_velocity_all(RMOL_DATA *md, ATOM_DATA *ad)
{
  int i;
  for (i = 0; i < md->n_mol; i++) {
    RIGID_MOL_set_mol_velocity(&md->mol[i], ad);
  }
}

void RMOL_DATA_n_rigid_mol(RMOL_DATA *md, ATOM_DATA *ad,
			   int *n_rigid_mol6_ex,int *n_rigid_mol5_ex)
{
  int i, iex;
  
  for (iex=0;iex<MAX_EX_SYSTEM;iex++) {
    n_rigid_mol6_ex[iex] = n_rigid_mol5_ex[iex]=0;
  }
  
  for (i=0;i<md->n_mol;i++) {
    iex = ATOM_FLAG_EX_SYSTEM(ad->ex[md->mol[i].parent_atom].flag);
    md->mol[i].iex = iex;
    if (md->mol[i].type->n_atom > 2)
      n_rigid_mol6_ex[iex]++;
    else
      n_rigid_mol5_ex[iex]++;
  }
}

void RMOL_DATA_correct_virial(RMOL_DATA *md, ATOM_DATA *ad)
{
  int i,j;
  double dx,dy,dz;
  RIGID_MOL *mol;
  
#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i = 0; i < md->n_mol; i++) {
#endif
    mol=&(md->mol[i]);
    for (j=mol->parent_atom;j>=0;j=ad->ex[j].child_list) {
      dx = ad->x[j].x - mol->rg.x;
      dy = ad->x[j].y - mol->rg.y;
      dz = ad->x[j].z - mol->rg.z;
      
      ad->virial[0] -= ad->f[j].x * dx;
      ad->virial[1] -= ad->f[j].y * dy;
      ad->virial[2] -= ad->f[j].z * dz;
      
      ad->virial[3] -= ad->f[j].x * dy;
      ad->virial[4] -= ad->f[j].x * dz;
      ad->virial[5] -= ad->f[j].y * dz;
      
      ad->virial[6] -= ad->f[j].y * dx;
      ad->virial[7] -= ad->f[j].z * dx;
      ad->virial[8] -= ad->f[j].z * dy;
    }
  }  
}

void RMOL_DATA_check_rigid_body(RMOL_DATA *md)
{
  int i, j;
  RMOL_TYPE *mt;

  for (i=0;i<md->n_type;i++) {
    mt=&md->mol_type[i];
    if (mt->n_mol == 0) continue;
    lprintf("[%s] %d %f %f %f\n",mt->name[0],mt->n_atom, mt->ip.x, mt->ip.y, mt->ip.z);
    for (j=0;j<mt->n_atom;j++) {
      /*
	lprintf("%d [%s] [%s] %2d %2d %2d %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f\n",
      */
      lprintf("%d [%s] [%s] %2d %2d %2d %8.3f %12.7f %12.7f %12.7f %8.3f %8.3f %8.3f\n",
	     j+1,mt->atom[j].name,mt->atom[j].sym,
	     mt->atom[j].na,mt->atom[j].nb,mt->atom[j].nc,
	     mt->atom[j].w,
	     mt->atom[j].length,mt->atom[j].angle,mt->atom[j].dihedral,
	     mt->atom[j].x.x,mt->atom[j].x.y,mt->atom[j].x.z);
    }
  }

}

#ifdef MPI
void RMOL_DATA_gather(RMOL_DATA *md, LINKED_CELL *lc, ATOM_DATA *ad)
{
  int i, n, pe, mol_pe;
  MPI_Status stat;
  
  if (mpi.master) {
    for (pe=0;pe<mpi.n_pe;pe++) {
      if (pe==mpi.rank) continue;
      MPI_Recv(md->crd,md->n_mol*13, MPI_DOUBLE, pe, 6, mpi.comm, &stat);
      n=0;
      for (i=0;i<md->n_mol;i++) {
	mol_pe = ATOM_CPU(lc,ad,md->mol[i].parent_atom);
	if (mol_pe == pe) {
	  md->mol[i].rg = md->crd[n].rg;
	  md->mol[i].vg = md->crd[n].vg;
	  md->mol[i].q  = md->crd[n].q;
	  md->mol[i].l  = md->crd[n].l;
	  n++;
	}
      }
    }
  } else {
    /* other node */
    n=0;
    for (i=0;i<md->n_mol;i++) {
      mol_pe = ATOM_CPU(lc,ad,md->mol[i].parent_atom);
      if (mol_pe == mpi.rank) {
	md->crd[n].rg = md->mol[i].rg;
	md->crd[n].vg = md->mol[i].vg;
	md->crd[n].q  = md->mol[i].q;
	md->crd[n].l  = md->mol[i].l;
	n++;
      }
    }
    MPI_Send(md->crd,n*13, MPI_DOUBLE, mpi.master_pe, 6, mpi.comm);
  }
}
  
void RMOL_DATA_copy_rmolcrd_to_buf(RMOL_DATA *md)
{
  int i;
  for (i = 0; i < md->n_mol; i++) {
    md->crd[i].rg = md->mol[i].rg;
    md->crd[i].vg = md->mol[i].vg;
    md->crd[i].q  = md->mol[i].q;
    md->crd[i].l  = md->mol[i].l;
  }
}

void RMOL_DATA_copy_rmolcrd_from_buf(RMOL_DATA *md)
{
  int i;
  for (i = 0; i < md->n_mol; i++) {
    md->mol[i].rg = md->crd[i].rg;
    md->mol[i].vg = md->crd[i].vg;
    md->mol[i].q  = md->crd[i].q;
    md->mol[i].l  = md->crd[i].l;
  }
}


void Vec_cmp(char *name, VEC *v1, VEC *v2, int rank)
{
  if (v1->x != v2->x ||
      v1->y != v2->y ||
      v1->z != v2->z) {
    printf("%3s: %2d(%.16e,%.16e,%.16e)\n  != %2d(%.16e,%.16e,%.16e)\n",
	   name, rank,   v1->x, v1->y, v1->z,
	         rank+1, v2->x, v2->y, v2->z);
  }
}

void Quan_cmp(char *name, QUAN *q1, QUAN *q2, int rank)
{
  if (q1->qw != q2->qw ||
      q1->qx != q2->qx ||
      q1->qy != q2->qy ||
      q1->qz != q2->qz) {
    printf("%3s: %2d(%.16e,%.16e,%.16e,%.16e)\n  != %2d(%.16e,%.16e,%.16e,%.16e)\n",
	   name, rank,   q1->qw, q1->qx, q1->qy, q1->qz,
	         rank+1, q2->qw, q2->qx, q2->qy, q2->qz);
  }
}

int RMOL_DATA_test(RMOL_DATA *md)
{
  MPI_Status stat;
  int i;
  
  if (mpi.rank % 2 == 1) {
    RMOL_DATA_copy_rmolcrd_to_buf(md);
    MPI_Send(md->crd, md->n_mol*13, MPI_DOUBLE, mpi.rank-1, 10, mpi.comm);
  } else {
    MPI_Recv(md->crd, md->n_mol*13, MPI_DOUBLE, mpi.rank+1, 10, mpi.comm,
	     &stat);
    for (i=0;i<md->n_mol;i++) {
      Vec_cmp("rg", &md->mol[i].rg, &md->crd[i].rg, mpi.rank);
      Vec_cmp("vg", &md->mol[i].vg, &md->crd[i].vg, mpi.rank);
      Quan_cmp("q", &md->mol[i].q,  &md->crd[i].q, mpi.rank);
      Vec_cmp("l",  &md->mol[i].l,  &md->crd[i].l, mpi.rank);
    }
  }
  return 1;
}

#endif  /* MPI */
