/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "md_system.h"
#ifdef MPI_RDMD
#include "parallel.h"
#endif

/* main update routines */
void PTC_update1(MD_SYSTEM *sys, double dt)
{
  if (sys->Ex_System_T_flag)
    if (sys->Ex_System_P_flag)
      PTC_Ex_System_PT_update(sys, dt*0.5);
    else
      PTC_Ex_System_T_update(sys, dt*0.5);
}

void PTC_update2(MD_SYSTEM *sys, double dt)
{
  if (sys->Ex_System_T_flag)
    if (sys->Ex_System_P_flag)
      PTC_Ex_System_PT_update(sys, dt*0.5);
    else
      PTC_Ex_System_T_update(sys, dt*0.5);
  
  if (sys->rescaling_T_flag)
    PTC_rescaling_T2(sys);

  /*
    if (sys->weak_coupling_T_flag)
    PTC_weak_coupling_T(sys, sys->dt_ps);
    
    if (sys->weak_coupling_P_flag)
    PTC_weak_coupling_P(sys, sys->dt_ps);
  */
}

void PTC_set_gradual_change_T(MD_SYSTEM *sys, int gradual_change_step,
			      double target_T1)
{
  sys->gradual_change_T_step = gradual_change_step;
  
  sys->target_temperature0 = sys->target_temperature;
  sys->target_temperature1 = target_T1;

  lprintf("  Gradual Change of Temperature: %f -> %f in %d steps\n",
	  sys->target_temperature0, sys->target_temperature1,
	  sys->gradual_change_T_step);
}

void PTC_gradual_change(MD_SYSTEM *sys, int step)
{
  double lambda;

  if (sys->gradual_change_T_step == 0) return;

  if (step < sys->gradual_change_T_step) {
    lambda = (double)step/sys->gradual_change_T_step;
    sys->target_temperature = sys->target_temperature0*(1.0-lambda) 
      + sys->target_temperature1*lambda;
  } else {
    sys->target_temperature = sys->target_temperature1;
  }
}

/*************************************************************************/
/***** Weak Coupling Methods: Berendesen Methods..                   *****/
/*************************************************************************/

void PTC_set_weak_coupling_T(MD_SYSTEM *sys, double T, double tau_t)
{
  int i, ex_system_div_flag = 0;
  ATOM_DATA *ad;

  ad = &sys->atom;
  
  lprintf("Weak Coupling Temperature Control:");
  if (T < 0.0) {
    sys->weak_coupling_T_flag = 0;
    lprintf(" OFF\n");
  } else {
    sys->weak_coupling_T_flag = 1;
    sys->target_temperature = T;
    sys->tau_t = tau_t;
    lprintf(" ON (target: %.2f K, tau: %.2f ps)\n", T, tau_t);

    /* clear ATOM_EX_SYSTEM */
    for (i=0;i<ad->natom;i++) {
      ad->ex[i].flag &= ~ATOM_EX_SYSTEM;
    }

    if (ex_system_div_flag==0) {
      /* The whole system is coupled with one thermostat. */
      sys->n_ex_system = 1;
      sys->atom_ex_system[0][0]=0;
      sys->atom_ex_system[0][1]=sys->atom.natom-1;
    } else if (ex_system_div_flag==1) {
      /* The solute/solvent are coupled with each thermostat. */
      sys->n_ex_system = 2;
      sys->atom_ex_system[0][0]=0;
      sys->atom_ex_system[0][1]=ad->n_solute_atom-1;
      sys->atom_ex_system[1][0]=ad->n_solute_atom;
      sys->atom_ex_system[1][1]=ad->natom-1;
      for (i=ad->n_solute_atom;i<ad->natom;i++) {
	ad->ex[i].flag |= ATOM_EX_SYSTEM1;
      }
    }
    MD_SYSTEM_degree_of_freedom(sys);
  }
  if (sys->Ex_System_T_flag)
    lprintf("Extended System Temperature Control: TURNED OFF\n");
  sys->Ex_System_T_flag = 0;
}

void PTC_weak_coupling_T(MD_SYSTEM *sys, double dt_ps)
{
  double scale[MAX_EX_SYSTEM];
  int i,j,k;

  for (i=0;i<sys->n_ex_system;i++) {
    if (sys->temperature_arr[i] > 0.0) {
      scale[i] = sqrt(1.0 + dt_ps/sys->tau_t*
		      (sys->target_temperature/sys->temperature_arr[i] - 1.0));

      sys->kene_arr[i] *= scale[i] * scale[i];
      for (j=0;j<3;j++) {
	for (k=0;k<3;k++) {
	  sys->kene_tr_arr[i][j][k] *= scale[i] * scale[i];
	}
      }
      sys->kene_rot_arr[i] *= scale[i] * scale[i];

    } else {
      scale[i] = 1.0;
    }
  }
  PTC_scale_velocity(sys, scale, scale);
}

void PTC_set_weak_coupling_P(MD_SYSTEM *sys, double P, double tau_p,
			     double comp)
{
  lprintf("Weak Coupling Pressure Control:");
  if (P < 0.0) {
    sys->weak_coupling_P_flag = 0;
    sys->ewald.pressure_flag = 0;
    lprintf(" OFF\n");
  } else {
    sys->weak_coupling_P_flag = 1;
    sys->Pex = P * ATM_TO_KCALMOLANG;
    sys->comp = comp / ATM_TO_KCALMOLANG;
    sys->tau_p = tau_p;
    sys->ewald.pressure_flag = 1;
    lprintf(" ON (target: %.2f atm, compressibility: %.2f atm^(-1), tau: %.2f ps)\n",
	    P, comp, tau_p);
  }
  if (sys->Ex_System_P_flag)
    lprintf("Extended System Pressure Control: TURNED OFF\n");
  sys->Ex_System_P_flag = 0;
}

void PTC_weak_coupling_P(MD_SYSTEM *sys, double dt_ps)
{
  double kai;

  MD_SYSTEM_calc_Pint(sys);
  kai = pow(1.0-sys->comp*dt_ps/sys->tau_p*(sys->Pex-sys->Pint), 1.0/3.0);
  PTC_scale_system(sys, kai);
}

/*************************************************************************/
/***** End of Weak Coupling Methods                                  *****/
/*************************************************************************/

/*************************************************************************/
/*****    Constraint T: not yet                                      *****/
/*************************************************************************/
void PTC_set_constraint_T(MD_SYSTEM *sys, double T)
{
  lprintf("Constraint Temperature:");
  if (T < 0.0) {
    sys->constraint_T_flag = 0;
    lprintf(" OFF\n");
  } else {
    sys->constraint_T_flag = 1;
    sys->target_temperature = T;
    lprintf(" ON (target: %.2f K)\n", T);
  }
  
  if (sys->Ex_System_T_flag)
    lprintf("Extended System Temperature Control: TURNED OFF\n");
  sys->Ex_System_T_flag = 0;
}

void PTC_constraint_T(MD_SYSTEM *sys, double T)
{
  /* dummy */
}

void PTC_constraint_T1(MD_SYSTEM *sys, double *chi)
{
  /*
  int i;
  double scale_v;

  scale_v = 2.0 - 1.0 / (*chi);
  MD_SYSTEM_scale_velocity(sys, scale_v);
  */
}

void PTC_constraint_T2(MD_SYSTEM *sys, double *chi)
{
  /*
  int i;
  ATOM_DATA *ad;
  double scale_v;

  MD_SYSTEM_calc_kene(sys);
  scale_v = sqrt(sys->target_temperature/sys->temperature);
  *chi = scale_v;
  MD_SYSTEM_scale_velocity(sys, scale_v);
  */
}

/*************************************************************************/
/*****    Rescaling T:                                               *****/
/*************************************************************************/
void PTC_set_rescaling_T(MD_SYSTEM *sys, double T, double delta_T)
{
  lprintf("Rescaling Velocities for T Control:");
  if (T < 0.0) {
    sys->rescaling_T_flag = 0;
    lprintf(" OFF\n");
  } else {
    lprintf(" ON (target: %.2f K)\n", T);
    if (delta_T > 0.0) {
      lprintf("Allowed temperature: %.2f K +- %.2f K\n", T, delta_T);
    }

    sys->rescaling_T_flag = 1;
    sys->target_temperature = T;
    sys->delta_T = delta_T;

    sys->n_ex_system = 1;
    sys->atom_ex_system[0][0]=0;
    sys->atom_ex_system[0][1]=sys->atom.natom-1;
    MD_SYSTEM_degree_of_freedom(sys);
  }
  
  if (sys->Ex_System_T_flag)
    lprintf("Extended System Temperature Control: TURNED OFF\n");
  sys->Ex_System_T_flag = 0;
}

/*
void PTC_rescaling_T(MD_SYSTEM *sys, double T)
{
}

void PTC_rescaling_T1(MD_SYSTEM *sys, double *chi)
{

}
*/

void PTC_rescaling_T2(MD_SYSTEM *sys)
{
  double scale[MAX_EX_SYSTEM];
  int i,j,k,scale_flag;

  /*
  lprintf("%d %f\n", sys->n_ex_system, sys->temperature_arr[0]);
  */
  scale_flag = 0;
  for (i=0;i<sys->n_ex_system;i++) {
    if (sys->temperature_arr[i] > 0.0 &&
	(sys->delta_T == 0.0 ||
	 fabs(sys->temperature_arr[i]-sys->target_temperature) > sys->delta_T)) {
      scale[i] = sqrt(sys->target_temperature/sys->temperature_arr[i]);
      scale_flag = 1;

      sys->kene_arr[i] *= scale[i] * scale[i];
      for (j=0;j<3;j++) {
	for (k=0;k<3;k++) {
	  sys->kene_tr_arr[i][j][k] *= scale[i] * scale[i];
	}
      }
      sys->kene_rot_arr[i] *= scale[i] * scale[i];

    } else {
      scale[i] = 1.0;
    }
  }

  if (scale_flag)
    PTC_scale_velocity(sys, scale, scale);
}


/*************************************************************************/
/***** Extended System Methods: Nose-Hoover                          *****/
/*************************************************************************/
/* period: charasteristic period in ps */
void PTC_set_Ex_System_T_cnst(MD_SYSTEM *sys, double target_T,
			      double period, int n_chain_T, int ex_system_div_flag)
{
  int i, j, dof;
  double omega;
  ATOM_DATA *ad;
  
  if (n_chain_T <= 0 || n_chain_T >= MAX_CHAIN_T) {
    lprintf("  ERROR: n_chain_T must be greater than 0 and less than %d.\n",
	    MAX_CHAIN_T);
    marble_exit(1);
  }
  ad = &sys->atom;
  sys->Ex_System_T_flag = 1;
  sys->weak_coupling_T_flag = 0;

  /* clear ATOM_EX_SYSTEM */
  for (i=0;i<ad->natom;i++) {
    ad->ex[i].flag &= ~ATOM_EX_SYSTEM;
  }

  if (ex_system_div_flag==0) {
    /* The whole system is coupled with one thermostat. */
    sys->n_ex_system = 1;
    sys->atom_ex_system[0][0]=0;
    sys->atom_ex_system[0][1]=sys->atom.natom-1;
  } else if (ex_system_div_flag==1) {
    /* The solute/solvent are coupled with each thermostat. */
    sys->n_ex_system = 2;
    sys->atom_ex_system[0][0]=0;
    sys->atom_ex_system[0][1]=ad->n_solute_atom-1;
    sys->atom_ex_system[1][0]=ad->n_solute_atom;
    sys->atom_ex_system[1][1]=ad->natom-1;
    for (i=ad->n_solute_atom;i<ad->natom;i++) {
      ad->ex[i].flag |= ATOM_EX_SYSTEM1;
    }
  }

  sys->target_temperature = target_T;
  sys->n_chain_T = n_chain_T;
  omega = 2.0 * M_PI / (period * 10.0 * sqrt(4.184));

  /*
  for (i=0;i<ad->natom;i++) {
    lprintf("%d %d %d\n",i,ATOM_FLAG_EX_SYSTEM(ad->ex[i].flag),
    ad->ex[i].flag&ATOM_EX_SYSTEM1);
  }
  */

  MD_SYSTEM_degree_of_freedom(sys);

  for (i=0;i<sys->n_ex_system;i++) {
    sys->Q[i][0] = sys->degree_of_freedom_arr[i] * K * KCAL * target_T / (omega*omega);
    sys->eta[i][0] = 0.0;
    sys->eta_v[i][0] = 0.0;

    for (j = 1; j < sys->n_chain_T; j++) {
      sys->Q[i][j] = K * KCAL * target_T / (omega*omega);
      sys->eta[i][j] = 0.0;
      sys->eta_v[i][j] = 0.0;
    }
  }
}

void PTC_setup_P_flag(MD_SYSTEM *sys)
{
     /* 1 for isotropic cell fluctuation,
	2 for full flexible cell fluctuation
        3 for xyz independently flucutuated 
        4 for only z  flucutuated
        5 for xy isotoropically flucutuated
     */
  if (sys->Ex_System_P_flag==1) {
    sys->Ex_System_n_P_var = 1;
  } else if (sys->Ex_System_P_flag==2) {
    sys->Ex_System_n_P_var = 9;
  } else if (sys->Ex_System_P_flag==3){
    sys->Ex_System_n_P_var = 3;
  } else if (sys->Ex_System_P_flag==4){
    sys->Ex_System_n_P_var = 1;
  } else if (sys->Ex_System_P_flag==5){
    sys->Ex_System_n_P_var = 2;
  }
    
  if (sys->Ex_System_P_flag)
    sys->ewald.pressure_flag = 1;
  else
    sys->ewald.pressure_flag = 0;
}

void PTC_set_Ex_System_PT_cnst(MD_SYSTEM *sys,
			       double target_T,double period_T,
			       int    n_chain_T,
			       int    ex_system_div_flag,
			       double target_P,double period_P,
			       int    P_flag)
{
  int i,j,dof;
  double omega, modulus;
  ATOM_DATA *ad;

  if (n_chain_T <= 0 || n_chain_T >= MAX_CHAIN_T) {
    lprintf("ERROR: n_chain_T must be greater than 0 and less than %d.\n",
	    MAX_CHAIN_T);
    marble_exit(1);
  }

  ad = &sys->atom;
  sys->Ex_System_T_flag = 1;
  sys->weak_coupling_T_flag = 0;
  
  sys->Ex_System_P_flag = P_flag;
  sys->weak_coupling_P_flag = 0;
  PTC_setup_P_flag(sys);
  
  sys->logv_v = 0.0;
  for (i=0;i<3;i++)
    for (j=0;j<3;j++)
      sys->Vg[i][j]=0.0;

  /* clear ATOM_FLAG_EX_SYS */
  for (i=0;i<ad->natom;i++) {
    ad->ex[i].flag &= ~ATOM_EX_SYSTEM;
  }

  if (ex_system_div_flag==0) {
    /* The whole system + pressure are coupled with one thermostat. */
    sys->n_ex_system = 1;
    sys->atom_ex_system[0][0]=0;
    sys->atom_ex_system[0][1]=sys->atom.natom-1;
    sys->pc_ex_system = 0;
  } else if (ex_system_div_flag==2) {
    /* The solute+pressure/solvent are coupled with three thermostats. */
    sys->n_ex_system = 2;
    sys->atom_ex_system[0][0]=0;
    sys->atom_ex_system[0][1]=ad->n_solute_atom-1;
    sys->atom_ex_system[1][0]=ad->n_solute_atom;
    sys->atom_ex_system[1][1]=ad->natom-1;
    sys->atom_ex_system[2][0]=1;
    sys->atom_ex_system[2][1]=0;
    sys->pc_ex_system = 0;
    for (i=ad->n_solute_atom;i<ad->natom;i++) {
      ad->ex[i].flag |= ATOM_EX_SYSTEM1;
    }
  } else if (ex_system_div_flag==1) {
    /* The solute/solvent/pressure are coupled with three thermostats. */
    sys->n_ex_system = 3;
    sys->atom_ex_system[0][0]=0;
    sys->atom_ex_system[0][1]=ad->n_solute_atom-1;
    sys->atom_ex_system[1][0]=ad->n_solute_atom;
    sys->atom_ex_system[1][1]=ad->natom-1;
    sys->atom_ex_system[2][0]=1;
    sys->atom_ex_system[2][1]=0;
    sys->pc_ex_system = 2;
    for (i=ad->n_solute_atom;i<ad->natom;i++) {
      ad->ex[i].flag |= ATOM_EX_SYSTEM1;
    }
  } else if (ex_system_div_flag==3) {
    /* The whole system + pressure are coupled with two thermostats. */
    sys->n_ex_system = 2;
    sys->atom_ex_system[0][0]=0;
    sys->atom_ex_system[0][1]=sys->atom.natom-1;
    sys->atom_ex_system[1][0]=1;
    sys->atom_ex_system[1][1]=0;
    sys->pc_ex_system = 1;
  }

  sys->target_temperature = target_T;
  sys->Pex = target_P * ATM_TO_KCALMOLANG;
  sys->n_chain_T = n_chain_T;
  
  MD_SYSTEM_degree_of_freedom(sys);
  
  omega = 2.0 * M_PI / (period_P * 10.0 * sqrt(4.184));
  if (sys->Ex_System_P_flag==1) dof = sys->degree_of_freedom + 3;
  else if (sys->Ex_System_P_flag>=2) dof = (sys->degree_of_freedom + 3)/3;

  sys->W = dof * K * KCAL * target_T / (omega*omega);
  /*
  modulus = ATM_TO_KCALMOLANG/compress;
  sys->W = 9 * sys->boundary.V * modulus/(omega*omega);
  */

  omega = 2.0 * M_PI / (period_T * 10.0 * sqrt(4.184));
  for (i=0;i<sys->n_ex_system;i++) {
    dof = sys->degree_of_freedom_arr[i];
    if (sys->pc_ex_system==i) {
      if (sys->Ex_System_P_flag) dof+=sys->Ex_System_n_P_var;
    }
    sys->Q[i][0] = dof * K * KCAL * target_T / (omega*omega);
    sys->eta[i][0] = 0.0;
    sys->eta_v[i][0] = 0.0;

    for (j = 1; j < sys->n_chain_T; j++) {
      /* sys->Q[i][j] = K * KCAL * target_T / (omega*omega); */
      sys->Q[i][j] = K * KCAL * target_T / (omega*omega);
      sys->eta[i][j] = 0.0;
      sys->eta_v[i][j] = 0.0;
    }
  }
}

void PTC_set_Ex_System_off(MD_SYSTEM *sys)
{
  sys->Ex_System_T_flag = sys->Ex_System_P_flag = 0;
  sys->n_ex_system = 1;
  sys->atom_ex_system[0][0] = 0;
  sys->atom_ex_system[0][1] = sys->atom.natom-1;
  sys->ewald.pressure_flag = 0;
  
  MD_SYSTEM_degree_of_freedom(sys);
}

void PTC_set_gamma_ex(MD_SYSTEM *sys, double f)
{
  sys->gamma_ex = f * DYNCM_TO_KCALMOLANG;
}

void PTC_Ex_System_print(MD_SYSTEM *sys)
{
  int i,j;
  if (sys->Ex_System_T_flag) {
    lprintf("EXTENDED SYSTEM T-CONSTANT SIMULATION: T=%.2f [K]\n", sys->target_temperature);
    lprintf("  Atoms of extended system:");
    for (i=0;i<sys->n_ex_system;i++) {
      if (sys->atom_ex_system[i][1]>sys->atom_ex_system[i][0])
	lprintf(" %d->%d",sys->atom_ex_system[i][0]+1,
		          sys->atom_ex_system[i][1]+1);
      else
	lprintf(" 0");
    }
    lprintf("\n");
    lprintf("  Degree of freedom:");
    for (i=0;i<sys->n_ex_system;i++) {
      lprintf(" %d",sys->degree_of_freedom_arr[i]);
      if (i==sys->pc_ex_system) {
	if (sys->Ex_System_P_flag)
	  lprintf("+%d(barostat)",sys->Ex_System_n_P_var);
      }
    }
    lprintf("\n");
    for (i=0;i<sys->n_ex_system;i++) {
      for (j=0;j<sys->n_chain_T;j++) {
	lprintf("  %d,%d: Q = %e, eta =%13e, eta_v =%13e\n",
		i, j, sys->Q[i][j], sys->eta[i][j], sys->eta_v[i][j]);
      }
    }
  }
  if (sys->Ex_System_P_flag) {
    lprintf("EXTENDED SYSTEM P-CONSTANT SIMULATION: P=%.2f [atm]\n",
	    sys->Pex / ATM_TO_KCALMOLANG);
    if (sys->Ex_System_P_flag==1) {
      lprintf("       W = %e, V =%13e, logv_v =%13e\n",
	      sys->W, sys->boundary.V, sys->logv_v);
    } else if (sys->Ex_System_P_flag>=2) {
      lprintf("       W = %e, V =%13e\n",
	      sys->W, sys->boundary.V);
      for (i=0;i<3;i++) {
	for (j=i;j<3;j++) {
	  lprintf("       Vg[%d][%d] =%13e\n",i,j,sys->Vg[i][j]);
	}
      }
    }
    if (sys->gamma_ex != 0.0) {
      lprintf("  EXTERNAL SURFACE TENSION: %f [dyn/cm]\n", sys->gamma_ex/DYNCM_TO_KCALMOLANG);
    }
    /*
    lprintf("  METHOD TO SCALE SYSTEM: ");
    switch (sys->scale_system_method) {
    case ATOM_BASED_SCALE:
      lprintf("ATOM-BASED SCALE\n"); break;
    case MOL_BASED_SCALE:
      lprintf("MOLECULE-BASED SCALE\n"); break;
    }
    */
  }
}

void PTC_Ex_System_read_data(MD_SYSTEM *sys, FILE *fp)
{
  int i, j, ret;
  char *err_str = "ERROR: reading extended system data in crd file.\n";

  /* lprintf("Reading data of extended system.\n"); */
  
  ret = fscanf(fp, "%d %d\n", &sys->Ex_System_T_flag, &sys->Ex_System_P_flag);
  if (ret != 2) {
    lprintf(err_str);
    marble_exit(1);
  }
  
  if (sys->Ex_System_T_flag) {
    ret = fscanf(fp, "%d %d %le\n", &sys->n_ex_system, &sys->n_chain_T, &sys->target_temperature);
    if (ret != 3) {
      lprintf(err_str); marble_exit(1);
    }
    for (i=0;i<sys->n_ex_system;i++) {
      ret = fscanf(fp, "%d %d\n", &sys->atom_ex_system[i][0],&sys->atom_ex_system[i][1]);
      if (ret != 2) {
	lprintf(err_str); marble_exit(1);
      }
      
      for (j=0;j<sys->n_chain_T;j++) {
	ret = fscanf(fp, "%le %le %le\n", &sys->eta[i][j], &sys->eta_v[i][j], &sys->Q[i][j]);
	if (ret != 3) {
	  lprintf(err_str);  marble_exit(1);
	}
      }
    }
    MD_SYSTEM_degree_of_freedom(sys);    
  }
  
  if (sys->Ex_System_P_flag == 1) {
    ret = fscanf(fp, "%le %le %le %d\n", &sys->logv_v, &sys->W, &sys->Pex, &sys->pc_ex_system);
    if (ret != 4) {
      lprintf(err_str); marble_exit(1);
    }
  } else if (sys->Ex_System_P_flag >= 2) {
    ret = fscanf(fp, "%le %le %d\n", &sys->W, &sys->Pex, &sys->pc_ex_system);
    if (ret != 3) {
      lprintf(err_str); marble_exit(1);
    }
    for (i=0;i<3;i++) {
      ret = fscanf(fp, "%le %le %le\n", &sys->Vg[i][0],&sys->Vg[i][1],&sys->Vg[i][2]);
      if (ret != 3) {
	lprintf(err_str); marble_exit(1);
      }
    }
  }
  
  PTC_setup_P_flag(sys);
}

#ifdef MPI
void PTC_Ex_System_comm_data(MD_SYSTEM *sys)
{
  int i;

  MPI_Bcast(&sys->Ex_System_T_flag, 1, MPI_INT, mpi.master_pe, mpi.comm);
  MPI_Bcast(&sys->Ex_System_P_flag, 1, MPI_INT, mpi.master_pe, mpi.comm);

  if (sys->Ex_System_T_flag) {    
    MPI_Bcast(&sys->n_ex_system, 1, MPI_INT, mpi.master_pe, mpi.comm);
    MPI_Bcast(&sys->n_chain_T, 1, MPI_INT, mpi.master_pe, mpi.comm);
    MPI_Bcast(&sys->target_temperature, 1, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    
    MPI_Bcast(sys->atom_ex_system, MAX_EX_SYSTEM*2, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    MPI_Bcast(sys->eta, MAX_EX_SYSTEM*MAX_CHAIN_T, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    MPI_Bcast(sys->eta_v, MAX_EX_SYSTEM*MAX_CHAIN_T, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    MPI_Bcast(sys->Q, MAX_EX_SYSTEM*MAX_CHAIN_T, MPI_DOUBLE, mpi.master_pe, mpi.comm);

    MD_SYSTEM_degree_of_freedom(sys);
  }
    
  if (sys->Ex_System_P_flag == 1) {
    MPI_Bcast(&sys->logv_v, 1, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    MPI_Bcast(&sys->W, 1, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    MPI_Bcast(&sys->Pex, 1, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    MPI_Bcast(&sys->pc_ex_system, 1, MPI_INT, mpi.master_pe, mpi.comm);

  } else if (sys->Ex_System_P_flag >= 2) {
    MPI_Bcast(&sys->W, 1, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    MPI_Bcast(&sys->Pex, 1, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    MPI_Bcast(&sys->pc_ex_system, 1, MPI_INT, mpi.master_pe, mpi.comm);
    
    MPI_Bcast(sys->Vg, 9, MPI_DOUBLE, mpi.master_pe, mpi.comm);
  }
    
  PTC_setup_P_flag(sys);
}
#endif

void PTC_Ex_System_write_data(MD_SYSTEM *sys, FILE *fp)
{
  int i, j;

  fprintf(fp,"%d %d\n", sys->Ex_System_T_flag, sys->Ex_System_P_flag);
  
  if (sys->Ex_System_T_flag) {
    fprintf(fp, "%d %d %.15e\n", sys->n_ex_system, sys->n_chain_T, sys->target_temperature);
    for (i=0;i<sys->n_ex_system;i++) {
      fprintf(fp, "%d %d\n", sys->atom_ex_system[i][0],sys->atom_ex_system[i][1]);
      for (j=0;j<sys->n_chain_T;j++) {
	fprintf(fp, "%.15e %.15e %.15e\n", sys->eta[i][j], sys->eta_v[i][j], sys->Q[i][j]);
      }
    }
  }
  
  if (sys->Ex_System_P_flag == 1) {
    fprintf(fp, "%.15e %.15e %.15e %d\n", sys->logv_v, sys->W, sys->Pex, sys->pc_ex_system);
  } else if (sys->Ex_System_P_flag >= 2) {
    fprintf(fp, "%.15e %.15e %d\n", sys->W, sys->Pex, sys->pc_ex_system);
    for (i=0;i<3;i++) {
      fprintf(fp, "%.15e %.15e %.15e\n", sys->Vg[i][0], sys->Vg[i][1],sys->Vg[i][2]);
    }
  }
}

#if 0
void PTC_scale_velocity(MD_SYSTEM *sys, double *scale_tr, double *scale_rot)
{
  int i, iex;
  ATOM_DATA *ad;

  ad = &sys->atom;
#ifdef MPI_SDMD
  MPI_Bcast(scale_tr, sys->n_ex_system, MPI_DOUBLE, mpi.master_pe, mpi.comm);
  if (scale_tr!=scale_rot)
    MPI_Bcast(scale_rot, sys->n_ex_system, MPI_DOUBLE, mpi.master_pe, mpi.comm);
#endif  

  /* for (i=0;i<sys->n_flex_atom;i++) */
  for (i=ad->node_fatom_h;i>=0;i=ad->node_fatom_n[i]) {
    iex = ATOM_FLAG_EX_SYSTEM(ad->ex[i].flag);
    
    ad->v[i].x *= scale_tr[iex];
    ad->v[i].y *= scale_tr[iex];
    ad->v[i].z *= scale_tr[iex];
  }
  
  if (sys->rigid_mol_flag)
    RMOL_DATA_scale_velocity(&sys->rigid_mol, scale_tr, scale_rot);
  if (sys->rattle.flag)
    RATTLE_scale_velocity(&sys->rattle, &sys->atom, scale_tr, scale_rot);

#ifdef MPI_SDMD    
  if (!mpi.master) return;
#endif

  sys->kene = 0.0;
  for (i=0;i<sys->n_ex_system;i++) {
    if (sys->degree_of_freedom_arr[i]>0)
      sys->temperature_arr[i] = 2.0*sys->kene_arr[i]/(sys->degree_of_freedom_arr[i]*K*KCAL);
    else
      sys->temperature_arr[i] = 0.0;
    sys->kene += sys->kene_arr[i];
  }

  sys->kene_tr = sys->kene_rot = 0.0;
  for (iex=0;iex<sys->n_ex_system;iex++) {
    sys->kene_rot += sys->kene_rot_arr[iex];
    for (i=0;i<3;i++)
      sys->kene_tr += sys->kene_tr_arr[iex][i][i];
  }

  sys->temperature = 2.0*sys->kene/(sys->degree_of_freedom*K*KCAL);
}

#else  /* if 0 */

void PTC_scale_velocity(MD_SYSTEM *sys, double *scale_tr, double *scale_rot)
{
  int i,j,k;
  double scale_tr_full[MAX_EX_SYSTEM][3][3];

  /* Although sys->kene_tr is updated when calling PTC_scale_velocity,
     sys->kene_tr_arr and sys->kene_rot_arr are not updated. 
     So, they are updated here.
     
     I changed sys->kene_tr_arr and sys->kene_rot_arr updated before calling
     
  for (i=0;i<sys->n_ex_system;i++) {
    for (j=0;j<3;j++) {
      for (k=0;k<3;k++) {
	sys->kene_tr_arr[i][j][k] *= scale_tr[i] * scale_tr[i];
      }
    }
    sys->kene_rot_arr[i] *= scale_rot[i] * scale_rot[i];
  }
  */
  
  for (i=0;i<sys->n_ex_system;i++) {
    for (j=0;j<3;j++) {
      for (k=0;k<3;k++) {
	if (j==k)
	  scale_tr_full[i][j][k] = scale_tr[i];
	else
	  scale_tr_full[i][j][k] = 0.0;
      }
    }
  }
  PTC_scale_velocity_full(sys, scale_tr_full, scale_rot);
}
#endif

void PTC_scale_velocity_full(MD_SYSTEM *sys, double scale_tr[MAX_EX_SYSTEM][3][3],
			     double scale_rot[MAX_EX_SYSTEM])
{
  int i,iex;
  double vx,vy,vz;
  ATOM_DATA *ad;

  ad = &sys->atom;
#ifdef MPI_SDMD
  MPI_Bcast(scale_tr, sys->n_ex_system*3*3, MPI_DOUBLE, mpi.master_pe, mpi.comm);
  MPI_Bcast(scale_rot, sys->n_ex_system,    MPI_DOUBLE, mpi.master_pe, mpi.comm);
#endif
  
  /* for (i=0;i<sys->n_flex_atom;i++) */
  for (i=ad->node_fatom_h;i>=0;i=ad->node_fatom_n[i]) {
    iex = ATOM_FLAG_EX_SYSTEM(ad->ex[i].flag);
    
    vx = VEC_MUL_MAT_X(ad->v[i],scale_tr[iex]);
    vy = VEC_MUL_MAT_Y(ad->v[i],scale_tr[iex]);
    vz = VEC_MUL_MAT_Z(ad->v[i],scale_tr[iex]);
    
    ad->v[i].x = vx;
    ad->v[i].y = vy;
    ad->v[i].z = vz;
  }
  
  if (sys->rigid_mol_flag)
    RMOL_DATA_scale_velocity_full(&sys->rigid_mol, scale_tr, scale_rot);
  if (sys->rattle.flag)
    RATTLE_scale_velocity_full(&sys->rattle, &sys->atom, scale_tr, scale_rot);
  
  sys->kene = 0.0;
  for (i=0;i<sys->n_ex_system;i++) {
    if (sys->degree_of_freedom_arr[i]>0)
      sys->temperature_arr[i] = 2.0*sys->kene_arr[i]/(sys->degree_of_freedom_arr[i]*K*KCAL);
    else
      sys->temperature_arr[i] = 0.0;
    sys->kene += sys->kene_arr[i];
  }
  
  sys->kene_tr = sys->kene_rot = 0.0;
  for (iex=0;iex<sys->n_ex_system;iex++) {
    sys->kene_rot += sys->kene_rot_arr[iex];
    for (i=0;i<3;i++)
      sys->kene_tr += sys->kene_tr_arr[iex][i][i];
  }

  sys->temperature = 2.0*sys->kene/(sys->degree_of_freedom*K*KCAL);
}

void PTC_Ex_System_T_update(MD_SYSTEM *sys, double hdt)
{
  double kT, G[MAX_CHAIN_T], scale_v[MAX_EX_SYSTEM];
  double dt2n, dt4n, dt8n;
  double AA;
  int i, j, k, n, n_div;
  ATOM_DATA *ad;

#ifdef MPI_SDMD
  if (!mpi.master) {
    PTC_scale_velocity(sys, scale_v, scale_v);
    return;
  }
#endif  
  
  ad = &sys->atom;
  kT = sys->target_temperature * KCAL * K;
#ifndef MPI_SDMD  
  MD_SYSTEM_calc_kene(sys);
#endif  
  
  n_div = 1;
  dt2n = hdt / n_div;
  dt4n = dt2n * 0.5;
  dt8n = dt4n * 0.5;
  
  for (i = 0;i < sys->n_ex_system; i++) {
    scale_v[i] = 1.0;
    for (n = 0;n < n_div; n++) {
      G[0] = 2.0*sys->kene_arr[i] - sys->degree_of_freedom_arr[i] * kT;
      for (j=1;j<sys->n_chain_T;j++) {
	G[j] = sys->eta_v[i][j-1]*sys->eta_v[i][j-1]*sys->Q[i][j-1] - kT;
      }
      /* make up eta_v[n_chain_T-1] : dt/4n */
      sys->eta_v[i][sys->n_chain_T-1] += G[sys->n_chain_T-1]/sys->Q[i][sys->n_chain_T-1]*dt4n;

      /* make up eta_v[n_chain_T-2] ---> eta_v[0] : dt/4n */
      for (j = sys->n_chain_T-2; j>=0; j--) {
	AA = exp(-dt8n*sys->eta_v[i][j+1]);
	sys->eta_v[i][j] *= AA;
	sys->eta_v[i][j] += G[j]/sys->Q[i][j]*dt4n;
	sys->eta_v[i][j] *= AA;
      }

      /* make up pi : dt/2n */
      scale_v[i] *= exp(-dt2n*sys->eta_v[i][0]);

      sys->kene_arr[i] *= scale_v[i] * scale_v[i];
      for (j=0;j<3;j++) {
	for (k=0;k<3;k++) {
	  sys->kene_tr_arr[i][j][k] *= scale_v[i] * scale_v[i];
	}
      }
      sys->kene_rot_arr[i] *= scale_v[i] * scale_v[i];
    
      /* make up pi : dt/2n */
      for (j=0;j<sys->n_chain_T;j++) {
	sys->eta[i][j] += sys->eta_v[i][j] * dt2n;
      }

      /* reverse dt/4n */
      G[0] = 2.0*sys->kene_arr[i] - sys->degree_of_freedom_arr[i] * kT;
      /* make up eta_v[0] ---> eta_v[n_chain_T-2] : dt/4n */
      for (j = 0; j<=sys->n_chain_T-2; j++) {
	AA = exp(-dt8n*sys->eta_v[i][j+1]);
	sys->eta_v[i][j] *= AA;
	sys->eta_v[i][j] += G[j]/sys->Q[i][j]*dt4n;
	sys->eta_v[i][j] *= AA;
      
	G[j+1] = sys->eta_v[i][j]*sys->eta_v[i][j]*sys->Q[i][j] - kT;
      }
      /* make up eta_v[n_chain_T-1] : dt/4n */
      sys->eta_v[i][sys->n_chain_T-1] += G[sys->n_chain_T-1]/sys->Q[i][sys->n_chain_T-1]*dt4n;
    }
  }
  /* at last, scale all velocities */
  PTC_scale_velocity(sys, scale_v, scale_v);
}


void PTC_Ex_System_PT_update(MD_SYSTEM *sys, double hdt)
{
  if (sys->Ex_System_P_flag == 1)
    PTC_Ex_System_PT_update_iso(sys, hdt);
  else if (sys->Ex_System_P_flag >= 2)
    PTC_Ex_System_PT_update_full(sys, hdt);
}

void PTC_Ex_System_PT_update_iso(MD_SYSTEM *sys, double hdt)
{
  double kT, G[MAX_CHAIN_T], scale_tr[MAX_EX_SYSTEM], scale_rot[MAX_EX_SYSTEM];
  double kene_tr_arr[MAX_EX_SYSTEM];
  double kene_tr_all;
  double G_logv, Nf_coef, PintPextV, AA, BB;
  double dt2n, dt4n, dt8n;
  int i, j, k, n, n_div;
  ATOM_DATA *ad;

#ifdef MPI_SDMD
  if (!mpi.master) {
    MPI_Bcast(&sys->logv_v, 1, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    PTC_scale_velocity(sys, scale_tr, scale_rot);
    return;
  }
#endif  
  
  ad = &sys->atom;
  kT = sys->target_temperature * KCAL * K;
  
#ifndef MPI_SDMD
  MD_SYSTEM_calc_kene_full(sys);
#endif  
  
  n_div = 1;
  dt2n = hdt / n_div;
  dt4n = dt2n * 0.5;
  dt8n = dt4n * 0.5;
  
  Nf_coef = 1.0+3.0/sys->degree_of_freedom_tr;
  PintPextV = ad->virial[0]+ad->virial[1]+ad->virial[2]-3.0*sys->Pex*sys->boundary.V;

  kene_tr_all=0.0;
  for (i = 0;i < sys->n_ex_system; i++) {
    scale_tr[i] = scale_rot[i] = 1.0;
    kene_tr_arr[i] = 0.0;
    for (j=0;j<3;j++) {
      kene_tr_arr[i] += sys->kene_tr_arr[i][j][j];
    }
    kene_tr_all+=kene_tr_arr[i];
  }
  
  for (n = 0;n < n_div; n++) {
    /* eta_v update dt/4n */
    for (i = 0;i < sys->n_ex_system; i++) {
      G[0] = 2.0*sys->kene_arr[i] - sys->degree_of_freedom_arr[i] * kT;
      
      if (i == sys->pc_ex_system) {
	G[0] += sys->W * sys->logv_v * sys->logv_v - kT;
      }
      
      for (j=1;j<sys->n_chain_T;j++) {
	G[j] = sys->eta_v[i][j-1]*sys->eta_v[i][j-1]*sys->Q[i][j-1] - kT;
      }
      
      /* make up eta_v[n_eta-1] : dt/4n */
      j = sys->n_chain_T-1;
      sys->eta_v[i][j] += G[j]/sys->Q[i][j]*dt4n;

      /* make up eta_v[n_eta-2] ---> eta_v[0] : dt/4n */
      for (j = sys->n_chain_T-2; j>=0; j--) {
	AA = exp(-dt8n*sys->eta_v[i][j+1]);
	sys->eta_v[i][j] *= AA;
	sys->eta_v[i][j] += G[j]/sys->Q[i][j]*dt4n;
	sys->eta_v[i][j] *= AA;
      }
      /* end of eta_v update */
    } /* i */
    /* update logv_v: dt/4n */
    G_logv = kene_tr_all*2.0*Nf_coef + PintPextV;
    AA = exp(-dt8n*sys->eta_v[sys->pc_ex_system][0]);
    sys->logv_v *= AA;
    sys->logv_v += G_logv*dt4n/sys->W;
    sys->logv_v *= AA;
    
    /* make up velocity : dt/2n */
    kene_tr_all = 0.0;
    for (i = 0;i < sys->n_ex_system; i++) {
      AA = exp(-dt2n*(sys->eta_v[i][0]));
      BB = exp(-dt2n*(Nf_coef*sys->logv_v))*AA;
      /*lprintf("AA = %e eta_v %e, logv %e\n",AA,sys->eta_v[i][0],Nf_coef*sys->logv_v);*/
      scale_tr[i] *= BB;
      kene_tr_arr[i] *= BB * BB;
      kene_tr_all += kene_tr_arr[i];
      for (j=0;j<3;j++)
	for (k=0;k<3;k++)
	  sys->kene_tr_arr[i][j][k] *= BB * BB;
      
      scale_rot[i] *= AA;
      sys->kene_rot_arr[i] *= AA * AA;
      sys->kene_arr[i] = kene_tr_arr[i] + sys->kene_rot_arr[i];
      
      for (j=0;j<sys->n_chain_T;j++) {
	sys->eta[i][j] += sys->eta_v[i][j] * dt2n;
      }
    } /* end of vel update */

    /* update logv_v: dt/4n */
    G_logv = kene_tr_all*2.0*Nf_coef + PintPextV;
    AA = exp(-dt8n*sys->eta_v[sys->pc_ex_system][0]);
    sys->logv_v *= AA;
    sys->logv_v += G_logv*dt4n/sys->W;
    sys->logv_v *= AA;

    /* eta_v update dt/4n */
    for (i = 0;i < sys->n_ex_system; i++) {
      G[0] = 2.0*sys->kene_arr[i] - sys->degree_of_freedom_arr[i] * kT;
      
      if (i == sys->pc_ex_system) {
	G[0] += sys->W * sys->logv_v * sys->logv_v - kT;
      }

      /* make up eta_v[0] -> eta_v[n_chainT-2] */
      for (j = 0; j <= sys->n_chain_T-2; j++) {
	AA = exp(-dt8n*sys->eta_v[i][j+1]);
	sys->eta_v[i][j] *= AA;
	sys->eta_v[i][j] += G[j]/sys->Q[i][j]*dt4n;
	sys->eta_v[i][j] *= AA;

	G[j+1] = sys->eta_v[i][j]*sys->eta_v[i][j]*sys->Q[i][j] - kT;
      }
      /* end of eta_v update */
      
      /* make up eta_v[n_eta-1] : dt/4n */
      j = sys->n_chain_T-1;
      sys->eta_v[i][j] += G[j]/sys->Q[i][j]*dt4n;
    } /* i */
  }

#ifdef MPI_SDMD
  MPI_Bcast(&sys->logv_v, 1, MPI_DOUBLE, mpi.master_pe, mpi.comm);
#endif  
  PTC_scale_velocity(sys, scale_tr, scale_rot);

  /*
  sys->kene_tr  = kene_tr_all;
  sys->kene_rot = 0.0;
  for (i=0;i<sys->n_ex_system;i++)
    sys->kene_rot += sys->kene_rot_arr[i];
  */
}

void PTC_Ex_System_PT_update_full(MD_SYSTEM *sys, double hdt)
{
  double kT, G[MAX_CHAIN_T],Gg[3][3],virialM[3][3];
  double Vtemp[3][3], Veig[3], Vvec[3][3];
  double scale_tr[MAX_EX_SYSTEM][3][3], scale_rot[MAX_EX_SYSTEM];
  double kene_tr_all, kene_tr[3][3], sc[3], scm[3][3], tmp_m[3][3], tmp_m2[3][3];
  double AA, TrVg, kene_vg;
  double dt2n, dt4n, dt8n;
  int i, j, k, l, n, m, n_div;
  ATOM_DATA *ad;
  
  n_div = 1;
  ad = &sys->atom;
  kT = sys->target_temperature * KCAL * K;
  dt2n = hdt / n_div;
  dt4n = dt2n * 0.5;
  dt8n = dt4n * 0.5;
  
#ifdef MPI_SDMD
  if (!mpi.master) {
    MPI_Bcast(sys->Vg, 9, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    PTC_scale_velocity_full(sys, scale_tr, scale_rot);
    return;
  }
#endif

  for (k=0;k<3;k++)
    virialM[k][k]=ad->virial[k];
  virialM[0][1]=virialM[1][0]=(ad->virial[3]+ad->virial[6])*0.5;
  virialM[0][2]=virialM[2][0]=(ad->virial[4]+ad->virial[7])*0.5;
  virialM[1][2]=virialM[2][1]=(ad->virial[5]+ad->virial[8])*0.5;
  
#ifndef MPI_SDMD  
  MD_SYSTEM_calc_kene_full(sys);
#endif

  /* caluclation of kene_vg */
  kene_vg = 0.0;
  for (k=0;k<3;k++) {
    for (l=0;l<3;l++) {
      kene_vg += sys->Vg[k][l]*sys->Vg[k][l];
    }
  }
  kene_vg *= sys->W;

  /* caluclation of kene_tr and kene_tr_all */
  kene_tr_all=0.0;
  for (k=0;k<3;k++) {
    for (l=0;l<3;l++) {
      kene_tr[k][l]=0.0;
      for (i = 0;i < sys->n_ex_system; i++) {
	kene_tr[k][l]+=sys->kene_tr_arr[i][k][l];
      }
    }
    kene_tr_all+=kene_tr[k][k];
  }

  /* initialization of scale_tr and scale_rot */
  for (i = 0;i < sys->n_ex_system; i++) {
    scale_rot[i]=1.0;
    for (k=0;k<3;k++) {
      for (l=0;l<3;l++) {
	if (k==l)
	  scale_tr[i][k][l]=1.0;
	else 
	  scale_tr[i][k][l]=0.0;
      }
    }
  }

  for (n = 0;n < n_div; n++) {
    /* eta_v update dt/4n */
    for (i = 0;i < sys->n_ex_system; i++) {
      G[0] = 2.0*sys->kene_arr[i] - sys->degree_of_freedom_arr[i] * kT;
      
      if (i == sys->pc_ex_system) {
	G[0] += kene_vg - sys->Ex_System_n_P_var*kT;
      }
      
      for (j=1;j<sys->n_chain_T;j++) {
	G[j] = sys->eta_v[i][j-1]*sys->eta_v[i][j-1]*sys->Q[i][j-1] - kT;
      }
      
      /* make up eta_v[n_chain_T-1] : dt/4n */
      j = sys->n_chain_T-1;
      sys->eta_v[i][j] += G[j]/sys->Q[i][j]*dt4n;

      /* make up eta_v[n_eta-2] ---> eta_v[0] : dt/4n */
      for (j = sys->n_chain_T-2; j>=0; j--) {
	AA = exp(-dt8n*sys->eta_v[i][j+1]);
	sys->eta_v[i][j] *= AA;
	sys->eta_v[i][j] += G[j]/sys->Q[i][j]*dt4n;
	sys->eta_v[i][j] *= AA;
      }
      /* end of eta_v update */
    } /* i */
    
    /* update Vg: dt/4n */
    for (k=0;k<3;k++) {
      for (l=0;l<3;l++) {
	Gg[k][l] = kene_tr[k][l]*2.0 + virialM[k][l];
	if (k==l)
	  Gg[k][l] += kene_tr_all*2.0/sys->degree_of_freedom_tr
  	           -  sys->Pex * sys->boundary.V;
	if (sys->gamma_ex != 0.0 && k==l && k<2) 
	  Gg[k][l] += sys->gamma_ex * sys->boundary.A;
      }
    }
    AA = exp(-dt8n*sys->eta_v[sys->pc_ex_system][0]);
    for (k=0;k<3;k++) {
      for (l=0;l<3;l++) {
	sys->Vg[k][l] *= AA;
	sys->Vg[k][l] += Gg[k][l]*dt4n/sys->W;
	sys->Vg[k][l] *= AA;
      }
    }
    if (sys->Ex_System_P_flag >= 3) {
      for (k=0;k<3;k++)
	for (l=0;l<3;l++)
	  if (k!=l)
	    sys->Vg[k][l] = 0.0;
      if (sys->Ex_System_P_flag == 4) {
	sys->Vg[0][0] = sys->Vg[1][1] = 0.0;
      } else if (sys->Ex_System_P_flag == 5) {
	sys->Vg[0][0] = sys->Vg[1][1] = 0.5 * (sys->Vg[0][0]+sys->Vg[1][1]);
      }
    }
    
    /* make up velocity : dt/2n */
    TrVg = (sys->Vg[0][0]+sys->Vg[1][1]+sys->Vg[2][2])/sys->degree_of_freedom_tr;
    for (i = 0;i < sys->n_ex_system; i++) {
      for (k=0;k<3;k++) {
	for (l=0;l<3;l++) {
	  Vtemp[k][l] = sys->Vg[k][l];
	  if (k==l) {
	    Vtemp[k][l] += TrVg + sys->eta_v[i][0];
	  }
	}
      }
      diag33(Vtemp, Veig, Vvec);

      for (k=0;k<3;k++) {
	sc[k] = exp(-dt2n*Veig[k]);
      }
      for (k=0;k<3;k++) {
	for (l=0;l<3;l++) {
	  scm[k][l]=0.0;
	  for (m=0;m<3;m++) {
	    scm[k][l]+= Vvec[k][m]*sc[m]*Vvec[l][m];
	  }
	}
      }
      for (k=0;k<3;k++) {
	for (l=0;l<3;l++) {
	  tmp_m[k][l]=0.0;
	  for (m=0;m<3;m++) {
	    tmp_m[k][l] += scale_tr[i][k][m]*scm[m][l];
	  }
	}
      }
      for (k=0;k<3;k++) {
	for (l=0;l<3;l++) {
	  scale_tr[i][k][l] = tmp_m[k][l];
	}
      }
      for (k=0;k<3;k++) {
	for (l=0;l<3;l++) {
	  tmp_m[k][l]=0.0;
	  for (m=0;m<3;m++) {
	    tmp_m[k][l] += sys->kene_tr_arr[i][k][m]*scm[m][l];
	  }
	}
      }
      for (k=0;k<3;k++) {
	for (l=0;l<3;l++) {
	  sys->kene_tr_arr[i][k][l]=0.0;
	  for (m=0;m<3;m++) {
	    sys->kene_tr_arr[i][k][l]+=scm[m][k]*tmp_m[m][l];
	  }
	}
      }
      AA=exp(-dt2n*sys->eta_v[i][0]);
      scale_rot[i] *= AA;
      sys->kene_rot_arr[i] *= AA*AA;
      sys->kene_arr[i]=sys->kene_rot_arr[i];
      for (k=0;k<3;k++) {
	sys->kene_arr[i]+=sys->kene_tr_arr[i][k][k];
      }
    }

    kene_tr_all=0.0;
    for (k=0;k<3;k++) {
      for (l=0;l<3;l++) {
	kene_tr[k][l]=0.0;
	for (i = 0;i < sys->n_ex_system; i++) {
	  kene_tr[k][l]+=sys->kene_tr_arr[i][k][l];
	}
      }
      kene_tr_all+=kene_tr[k][k];
    }

    /*
    printf("%e %e %e\n",Veig[0][3][0],Veig[0][3][1],Veig[0][3][2]);
    print_mat33(Veig[0]);
    */

    /*****
    PTC_scale_velocity_full(sys, Veig, dt2n);
    MD_SYSTEM_calc_kene_full(sys);
    ****/

    /* make up eta : 0 -> dt/2 */
    for (i = 0;i < sys->n_ex_system; i++) {
      for (j=0;j<sys->n_chain_T;j++) {
	sys->eta[i][j] += sys->eta_v[i][j] * dt2n;
      }
    }
    
    /* make up Vg dt/4 -> dt/2 */
    for (k=0;k<3;k++) {
      for (l=0;l<3;l++) {
	Gg[k][l] = kene_tr[k][l]*2.0 + virialM[k][l];
	if (k==l)
	  Gg[k][l] += kene_tr_all*2.0/sys->degree_of_freedom_tr
	            - sys->Pex * sys->boundary.V;
	if (sys->gamma_ex != 0.0 && k==l && k<2) 
	  Gg[k][l] += sys->gamma_ex * sys->boundary.A;
      }
    }
    AA = exp(-dt8n*sys->eta_v[sys->pc_ex_system][0]);
    for (k=0;k<3;k++) {
      for (l=0;l<3;l++) {
	sys->Vg[k][l] *= AA;
	sys->Vg[k][l] += Gg[k][l]*dt4n/sys->W;
	sys->Vg[k][l] *= AA;
      }
    }
    if (sys->Ex_System_P_flag >= 3) {
      for (k=0;k<3;k++)
	for (l=0;l<3;l++)
	  if (k!=l)
	    sys->Vg[k][l] = 0.0;
      if (sys->Ex_System_P_flag == 4) {
	sys->Vg[0][0] = sys->Vg[1][1] = 0.0;
      } else if (sys->Ex_System_P_flag == 5) {
	sys->Vg[0][0] = sys->Vg[1][1] = 0.5 * (sys->Vg[0][0]+sys->Vg[1][1]);
      }
    }

    kene_vg = 0.0;
    for (k=0;k<3;k++) {
      for (l=0;l<3;l++) {
	kene_vg += sys->Vg[k][l]*sys->Vg[k][l];
      }
    }
    kene_vg *= sys->W;
    
    for (i = 0;i < sys->n_ex_system; i++) {
      G[0] = 2.0*sys->kene_arr[i] - sys->degree_of_freedom_arr[i] * kT;
      
      if (i == sys->pc_ex_system) {
	G[0] += kene_vg - sys->Ex_System_n_P_var*kT;
      }
      
      /* make up eta_v[0] -> eta_v[n_chainT-2] */
      for (j = 0; j <= sys->n_chain_T-2; j++) {
	AA = exp(-dt8n*sys->eta_v[i][j+1]);
	sys->eta_v[i][j] *= AA;
	sys->eta_v[i][j] += G[j]/sys->Q[i][j]*dt4n;
	sys->eta_v[i][j] *= AA;

	G[j+1] = sys->eta_v[i][j]*sys->eta_v[i][j]*sys->Q[i][j] - kT;
      }
      /* end of eta_v update */
      
      /* make up eta_v[n_eta-1] : dt/4n */
      j = sys->n_chain_T-1;
      sys->eta_v[i][j] += G[j]/sys->Q[i][j]*dt4n;
    } /* i */
  } /* n */


#ifdef MPI_SDMD
  MPI_Bcast(sys->Vg, 9, MPI_DOUBLE, mpi.master_pe, mpi.comm);
#endif  
  PTC_scale_velocity_full(sys, scale_tr, scale_rot);

  /*
  lprintf("kene = %f\n", sys->kene);
  for (i=0;i<sys->n_ex_system;i++) {
    lprintf("kene_arr[%d] = %f\n", i, sys->kene_arr[i]);
    lprintf("kene_rot_arr[%d] = %f\n", i, sys->kene_rot_arr[i]);
    for (k=0;k<3;k++) {
      for (l=0;l<3;l++) {
	lprintf("kene_tr_arr[%d][%d][%d] = %f\n", i, k, l, sys->kene_tr_arr[i][k][l]);
      }
    }
  }
  MD_SYSTEM_calc_kene_full(sys);
  lprintf("kene = %f\n", sys->kene);
  for (i=0;i<sys->n_ex_system;i++) {
    lprintf("kene_arr[%d] = %f\n", i, sys->kene_arr[i]);
    lprintf("kene_rot_arr[%d] = %f\n", i, sys->kene_rot_arr[i]);
    for (k=0;k<3;k++) {
      for (l=0;l<3;l++) {
	lprintf("kene_tr_arr[%d][%d][%d] = %f\n", i, k, l, sys->kene_tr_arr[i][k][l]);
      }
    }
  }
  marble_exit(1);
  */
}


void PTC_Ex_System_PT_update_x(MD_SYSTEM *sys, double dt)
{
  if (sys->Ex_System_P_flag == 1)
    PTC_Ex_System_PT_update_x_iso(sys, dt);
  else if (sys->Ex_System_P_flag >= 2)
    PTC_Ex_System_PT_update_x_full(sys, dt);
}

void PTC_Ex_System_PT_update_x_iso(MD_SYSTEM *sys, double dt)
{
  static double E2 = 1.0/6.0, E4 = 1.0/120.0, E6=1.0/5040.0, E8=1.0/362880.0;
  double AA, AA2, arg2, BB;
  int i;
  ATOM_DATA *ad;

  ad = &sys->atom;
  arg2 = 0.5*dt*sys->logv_v;
  AA = exp(arg2);
  arg2 = arg2*arg2;
  AA2 = AA * AA;
  BB = (((E8*arg2+E6)*arg2+E4)*arg2+E2)*arg2+1.0;
  BB *= AA*dt;

  /* for (i=0;i<sys->n_flex_atom;i++) { */
  for (i = ad->node_fatom_h; i>=0; i=ad->node_fatom_n[i]) {
    ad->x[i].x = ad->x[i].x * AA2 + ad->v[i].x * BB;
    ad->x[i].y = ad->x[i].y * AA2 + ad->v[i].y * BB;
    ad->x[i].z = ad->x[i].z * AA2 + ad->v[i].z * BB;
  }
  
  if (sys->rigid_mol_flag) {
    RMOL_DATA_time_integration_p_NPT(&sys->rigid_mol, ad, dt, AA2, BB);
  }
  if (sys->rattle.flag) {
    RATTLE_time_integration_p_NPT(&sys->rattle, &sys->bond, ad, dt, AA2, BB);
  }
  
  BOUNDARY_scale_boxv(&sys->boundary,AA2);
}

void PTC_Ex_System_PT_update_x_full(MD_SYSTEM *sys, double dt)
{
  static double E2 = 1.0/6.0, E4 = 1.0/120.0, E6=1.0/5040.0, E8=1.0/362880.0;
  double AA, AA2[3], arg2, BB[3];
  double Vg_eig[3], Vg_vec[3][3];
  double ubox[3][3];
  int i,j,k;
  VEC ux, uv;
  ATOM_DATA *ad;
  BOUNDARY *bc;

  ad = &sys->atom;
  bc = &sys->boundary;

  diag33(sys->Vg, Vg_eig, Vg_vec);

  for (i=0;i<3;i++) {
    AA = exp(0.5*dt*Vg_eig[i]);
    AA2[i] = AA * AA;
    arg2 = (dt*Vg_eig[i])*(dt*Vg_eig[i]);
    BB[i] = (((E8*arg2+E6)*arg2+E4)*arg2+E2)*arg2+1.0;
    BB[i] *= AA*dt;
  }

  /*
  print_mat33(sys->Vg);
  {
    double tmp[3][3];
    for (i=0;i<3;i++) {
      for (j=0;j<3;j++) {
	ubox[i][j] = AA2[i]*Vg_vec[j][i];
      }
    }
    
    for (i=0;i<3;i++) {
      for (j=0;j<3;j++) {
	tmp[i][j]=0.0;
	for (k=0;k<3;k++) {
	  tmp[i][j] += Vg_vec[i][k]*ubox[k][j];
	}
      }
    }
    print_mat33(tmp);
  }
  */

  /* for (i=0;i<sys->n_flex_atom;i++) { */
  for (i = ad->node_fatom_h; i>=0; i=ad->node_fatom_n[i]) {
    ux.x = VEC_MUL_MAT_X(ad->x[i],Vg_vec);
    ux.y = VEC_MUL_MAT_Y(ad->x[i],Vg_vec);
    ux.z = VEC_MUL_MAT_Z(ad->x[i],Vg_vec);
    uv.x = VEC_MUL_MAT_X(ad->v[i],Vg_vec);
    uv.y = VEC_MUL_MAT_Y(ad->v[i],Vg_vec);
    uv.z = VEC_MUL_MAT_Z(ad->v[i],Vg_vec);
    
    ux.x = ux.x * AA2[0] + uv.x * BB[0];
    ux.y = ux.y * AA2[1] + uv.y * BB[1];
    ux.z = ux.z * AA2[2] + uv.z * BB[2];

    ad->x[i].x = MAT_MUL_VEC_X(Vg_vec,ux);
    ad->x[i].y = MAT_MUL_VEC_Y(Vg_vec,ux);
    ad->x[i].z = MAT_MUL_VEC_Z(Vg_vec,ux);
  }
  
  if (sys->rigid_mol_flag) {
    RMOL_DATA_time_integration_p_NPT_full(&sys->rigid_mol, ad, dt, Vg_vec, AA2, BB);
  }
  if (sys->rattle.flag) {
    RATTLE_time_integration_p_NPT_full(&sys->rattle, &sys->bond, ad, dt, Vg_vec, AA2, BB);
  }

  /* update box */
  /* caution : order of boxv */
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      ubox[i][j] = 0.0;
      for (k=0;k<3;k++) {
	ubox[i][j] += Vg_vec[k][i]*bc->boxv[j][k];
      }
    }
  }
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      ubox[i][j] = AA2[i] * ubox[i][j];
    }
  }
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      bc->boxv[j][i] = 0.0;
      for (k=0;k<3;k++) {
	bc->boxv[j][i] += Vg_vec[i][k]*ubox[k][j];
      }
    }
  }
  BOUNDARY_make_recip(bc);
  BOUNDARY_set_offset_v(bc);
}


/*************************************************************************/
/***** End of Extended System Methods: Nose-Hoover                   *****/
/*************************************************************************/

/*************************************************************************/
/***** Scaling for Pressure Contorol                                 *****/
/*************************************************************************/

void PTC_scale_system(MD_SYSTEM *sys, double scale)
{
  int i;
  double dx, scale3[3];

  scale3[0] = scale3[1] = scale3[2] = scale;
  PTC_scale_system_anisotropic(sys, scale3);
}

void PTC_scale_system_anisotropic(MD_SYSTEM *sys, double scale[3])
{
  int i, j, a_no;
  double dx;
  
  for (i=0;i<3;i++) {
    sys->boundary.min[i] *= scale[i];
    sys->boundary.box[i] *= scale[i];
    sys->boundary.boxh[i] = sys->boundary.box[i]*0.5;
    /*
    dx = sys->boundary.box[i]/sys->linked_cell.n_grid[i];
    sys->linked_cell.neighbor[i] = (sys->cutoff_list / dx) + 1;
    */
  }
  sys->boundary.V = sys->boundary.box[0]*sys->boundary.box[1]*sys->boundary.box[2];

  if (sys->scale_system_method == ATOM_BASED_SCALE) {
    /* wrong; use node_fatom */
    for (i=0;i<sys->n_flex_atom;i++) {
      sys->atom.x[i].x *= scale[0];
      sys->atom.x[i].y *= scale[1];
      sys->atom.x[i].z *= scale[2];
    }
  } else if (sys->scale_system_method == MOL_BASED_SCALE) {
    ATOM_DATA_mol_gravity_center(&sys->atom);
    for (i=0;i<sys->atom.nmol;i++) {
      for (j=0;j<sys->atom.mol[i].natom;j++) {
	a_no = sys->atom.mol[i].start_atom + j;
	sys->atom.x[a_no].x = (sys->atom.x[a_no].x - sys->atom.mol[i].center.x)
	  + sys->atom.mol[i].center.x * scale[0];
	sys->atom.x[a_no].y = (sys->atom.x[a_no].y - sys->atom.mol[i].center.y)
	  + sys->atom.mol[i].center.y * scale[1];
	sys->atom.x[a_no].z = (sys->atom.x[a_no].z - sys->atom.mol[i].center.z)
	  + sys->atom.mol[i].center.z * scale[2];
      }
    }
    /* After scaling, the gravity center is not correct. Thus,
       the value must be reccalculated befere using it. */
  }
}

/*************************************************************************/
/***** END of Scaling for Pressure Contorol                          *****/
/*************************************************************************/

