/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifndef _PDB_H_INCLUDED
#define _PDB_H_INCLUDED

#define PDB_H_ATOM    1
#define PDB_H_HETATM  2
#define PDB_H_TER     3
#define PDB_H_CRYST1  4
#define PDB_H_CONNCT  5
#define PDB_H_UNKNOWN 0

#define GM_UNKNOWN -1
#define GM_OR       0
#define GM_AND      1
#define GM_OR_NOT   2
#define GM_AND_NOT  3

/* flag for origin definition */
#define O_MINIMUM  0  
#define O_CENTER   1

typedef struct pdb_s {
  struct pdb_atom_s *atom, **al;
  struct pdb_res_s  *res,  **rl;
  int n_atom, n_hetatm, n_res;
  double a,b,c,alpha,beta,gamma;
} PDB;

typedef struct pdb_atom_s {
  int  header;
  int  no;
  int  pdb_no;
  char name[5];
  char alt_loc;
  char res_name[20];
  char chain;
  int  res_no;
  char code_ins;
  double x, y, z, o, t, w;
  int  ter, check;
  unsigned int  group;
  struct pdb_res_s  *res;
  struct pdb_atom_s *next;
} PDB_ATOM; 

typedef struct pdb_res_s {
  int  no;
  int  pdb_no;
  char name[20];
  char chain;
  int  ter, first;
  double x, y, z;
  struct pdb_atom_s *beg_atom,*end_atom;
  int iba, iea;

  struct pdb_res_s *next;
} PDB_RES;

void PDB_init(PDB *pdb);
void PDB_free(PDB *pdb);
int  PDB_read_config(PDB *pdb, char *buf);
void PDB_print_format();
void PDB_read_file(PDB *pdb, char *fname);
int  PDB_parse_header(char *buf);
void PDB_make_res_list(PDB *pdb);
char *PDB_cutstr(char *buf, char *dst, int from, int to);
void PDB_count_data(PDB *pdb);
void PDB_print_info(PDB *pdb, char *fname);

void PDB_write_file(PDB *pdb, char *fname);
void PDB_write_file_group_fp(PDB *pdb, FILE *fp, int group_no);
void PDB_write_file_group(PDB *pdb, char *fname, int group_no);
void PDB_print_atom(PDB *pdb, int no);
void PDB_copy_coor_from_vec(PDB *pdb, VEC *x, int n_atom);
void PDB_copy_group_coor_from_vec(PDB *pdb, VEC *x, int n_atom, int group_no);
PDB_ATOM *PDB_search_atom_in_res(char *name, PDB_RES *res);
void PDB_print_unchecked(PDB *pdb);
char *PDB_atom_name(char *name, char *buf);

void PDB_renumber(PDB *pdb);

void PDB_make_array(PDB *pdb);
void PDB_calc_res_center(PDB *pdb);

void PDB_set_weight(PDB *pdb);
void PDB_get_weight(PDB *pdb, double *w, int n_atom, int *list);
void PDB_set_elec_weight(PDB *pdb);
void PDB_get_size(PDB *pdb, double min[3], double max[3]);

void PDB_set_group_str_list(PDB *pdb, STR_LIST *str_list, int g_no, char *header);
void PDB_set_group_buf(PDB *pdb, char *buf, int g_no, char *header);

int  PDB_count_group_atoms(PDB *pdb, int g_no);
void PDB_list_group_atoms(PDB *pdb, int g_no, int *list);
int  PDB_get_mode_id(char *mode);
void PDB_set_g_bit(PDB *pdb, int atom_no, int g_bit, int ok, int mode);
void PDB_clear_group(PDB *pdb, int g_no);
void PDB_group_atom_no(PDB *pdb, int g_no, int no1, int no2, int mode);
void PDB_group_res_no(PDB *pdb, int g_no, int no1, int no2, int mode);
void PDB_group_atom_name(PDB *pdb, int g_no, char *name, int mode);
void PDB_group_atom_name_res_no(PDB *pdb, int g_no, char *name, char *no1, char *no2, int mode);
void PDB_group_from_another_group(PDB *pdb, int g_no, int src_g_no, int mode);
int PDB_str_to_atom_no(PDB *pdb, char *name);
void PDB_str_to_res_no(PDB *pdb, char *name, int *id, int *resno, char *chain, int flag);
void PDB_group_atom_name_res_name(PDB *pdb, int g_no, char *name, char *res_name, int mode);
void PDB_group_res_name(PDB *pdb, int g_no, char *name, int mode);
void PDB_group_atom_type(PDB *pdb, int g_no, int mode, int flag);
void PDB_check_from_group(PDB *pdb, int g_no, int check);
void PDB_get_atoms_in_group(PDB *pdb, int g_no, int **parray, int *pn);
void PDB_get_coord_in_group(PDB *pdb, int g_no, VEC *x0, int n_atom);
void PDB_make_recip(double boxv[3][3], double recip[3][3]);

void PDB_wrap_atoms_vec(PDB *pdb, VEC *x, double boxv[3][3], int origin);
void PDB_wrap_atoms_vec_res(PDB *pdb, VEC *x, double boxv[3][3], int origin);
PDB_ATOM *PDB_search_atom_by_name(PDB *pdb, char *str);

#endif /* _PDB_H_INCLUDED */
