/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>

#include "misc.h"
#include "pdb.h"
#include "atom.h"

#ifdef MPI
FILE *par_fopen(char *fname, char *mode);
char *par_fgets(char *buf, int size, FILE *fp);
int   par_fclose(FILE *fp);
#endif

void PDB_init(PDB *pdb)
{
  pdb->atom = NULL;
  pdb->res = NULL;
  pdb->n_atom = pdb->n_hetatm = pdb->n_res = 0;
  pdb->a = pdb->b = pdb->c = pdb->alpha = pdb->beta = pdb->gamma = 0.0;
}

void PDB_free(PDB *pdb)
{
  PDB_ATOM *a,*anext;
  PDB_RES  *r,*rnext;

  for (a=pdb->atom;a!=NULL;a=anext) {
    anext = a->next;
    free(a);
  }
  for (r=pdb->res;r!=NULL;r=rnext) {
    rnext = r->next;
    free(r);
  }
  free(pdb->al);
  free(pdb->rl);
  free(pdb);
}

void PDB_read_file(PDB *pdb, char *fname)
{
  PDB_ATOM *p, *atom_prev;
  char buf[100], dst[10], alt_loc;
  FILE *fp;
  int len,hid;

#if 0 /*def MPI*/
  fp = par_fopen(fname, "r");
#else
  fp = fopen(fname, "r");
#endif

  if (fp == NULL) {
    lprintf("ERROR: No such file: %s\n",fname);
    marble_exit(1);
  }

  pdb->atom = p = NULL;
  
#if 0 /* def MPI */
  while (par_fgets(buf,100,fp) != NULL) {
#else
  while (fgets(buf,100,fp) != NULL) {
#endif

    hid = PDB_parse_header(buf);
    
    switch (hid) {
    case PDB_H_ATOM:
    case PDB_H_HETATM:
      if (strlen(buf)<31) continue; 
      p = emalloc("read_pdb_file", sizeof(PDB_ATOM));
      p->header = hid;
      p->ter = 0;
      p->check = 0;
      p->group = 0;
	  
      if (sscanf(PDB_cutstr(buf,dst, 7,11), "%d", &p->pdb_no) != 1)  continue;
      if (sscanf(PDB_cutstr(buf,dst,13,16), "%s", p->name)  != 1)  continue;
      if (sscanf(PDB_cutstr(buf,dst,17,17), "%c", &p->alt_loc)  != 1)  continue;
      if (sscanf(PDB_cutstr(buf,dst,18,21), "%s", p->res_name)  != 1)  continue;
      if (sscanf(PDB_cutstr(buf,dst,22,22), "%c", &p->chain)  != 1)  continue;
      if (sscanf(PDB_cutstr(buf,dst,23,27), "%d", &p->res_no)  != 1)  continue;
      /*
      if (sscanf(PDB_cutstr(buf,dst,27,27), "%c", &p->code_ins)  != 1)  continue;
      */
      if (sscanf(PDB_cutstr(buf,dst,31,38), "%lf", &p->x)  != 1)  continue;
      if (sscanf(PDB_cutstr(buf,dst,39,46), "%lf", &p->y)  != 1)  continue;
      if (sscanf(PDB_cutstr(buf,dst,47,54), "%lf", &p->z)  != 1)  continue;
      if (sscanf(PDB_cutstr(buf,dst,55,60), "%lf", &p->o)  != 1)  p->o = 1.0;
      if (sscanf(PDB_cutstr(buf,dst,61,66), "%lf", &p->t)  != 1)  p->t = 0.0;
      
      p->next = NULL;
      if (pdb->atom==NULL)
	pdb->atom=p;
      else
	atom_prev->next = p;
      atom_prev = p;
      break;
    case PDB_H_TER:
      if (p==NULL) break;
      p->ter = 1;
      break;
    case PDB_H_CRYST1:
      /*
      sscanf(buf,"%s %lf %lf %lf %lf %lf %lf",dst,
	     &pdb->a,&pdb->b,&pdb->c,
	     &pdb->alpha,&pdb->beta,&pdb->gamma);
      */
      break;
    case PDB_H_CONNCT:
      break;
    default :
      break;
    }
  }
  PDB_make_res_list(pdb);
  PDB_count_data(pdb);
  PDB_make_array(pdb);
  /* PDB_print_info(pdb, fname); */

#if 0 /* def MPI */
  par_fclose(fp);
#else
  fclose(fp);
#endif
}

int PDB_parse_header(char *buf)
{
  if (strncmp(buf,"ATOM",4) == 0) return PDB_H_ATOM;
  else if (strncmp(buf,"HETATM",6) == 0) return PDB_H_HETATM;
  else if (strncmp(buf,"TER",3) == 0) return PDB_H_TER;
  else if (strncmp(buf,"CRYST1",6) == 0) return PDB_H_CRYST1;
  else if (strncmp(buf,"CONNCT",6) == 0) return PDB_H_CONNCT;
  else return PDB_H_UNKNOWN;
}

void PDB_make_res_list(PDB *pdb)
{
  PDB_ATOM *a;
  PDB_RES  *r, *r_prev;
  int prev_res_no, id;
  char prev_chain;

  if (pdb->atom == NULL) {
    pdb->res = NULL;
    return;
  }
  pdb->res = NULL;
  prev_res_no = pdb->atom->res_no-1;
  prev_chain = '\0';
  r = NULL;
  for (a = pdb->atom; a != NULL; a = a->next) {
    if (prev_chain != a->chain ||
	prev_res_no != a->res_no ||
	r->ter == 1 ||
	strcmp(r->name, a->res_name) != 0) {
      r_prev = r;
      r = emalloc("make_pdb_res_list", sizeof(PDB_RES));
      r->ter   = 0;
      if (r_prev && r_prev->ter)
	r->first = 1;
      else
	r->first = 0;
      
      r -> next = NULL;
      if (pdb->res == NULL) {
	pdb->res = r;
	r->first = 1;
      } else {
	r_prev->next = r;
      }
      
      r -> beg_atom = a;
      prev_res_no = r -> pdb_no = a->res_no;
      prev_chain = r->chain = a->chain;
      strcpy(r->name, a->res_name);
    }
    a->res = r;
    r->ter = a->ter;
    r->chain = a->chain;
    r->end_atom = a;
  }
}

void PDB_read_seq(PDB *pdb, char *fname)
{
  PDB_RES  *r, *r_prev;
  FILE *fp;
  char buf[1000];
  int no=1;

  fp = fopen(fname,"r");
  if (fp==NULL) {
    lprintf("ERROR: Can't open sequence file: %s\n",fname);
    marble_exit(1);
  }
  fgets(buf,1000,fp);
  lprintf("Title of sequence file: %s", buf);

  pdb->res = NULL;
  pdb->atom = NULL;
  r_prev=NULL;
  while (fscanf(fp,"%s",buf)==1) {
    if (strcmp(buf,"TER") == 0) {
      if (r_prev)
	r_prev->ter = 1;
      continue;
    }
    r = emalloc("pdb_read_seq", sizeof(PDB_RES));
    strcpy(r->name, buf);
    r->beg_atom = r->end_atom = NULL;
    r->chain = ' ';
    r->ter = 0;
    r->pdb_no = no++;
    if (pdb->res == NULL) {
      pdb->res = r;
    } else {
      r_prev->next = r;
    }
    r_prev = r;
  }
}

char *PDB_cutstr(char *buf, char *dst, int from, int to)
{
  int len;

  len = strlen(buf);

  if (from > len || to > len) {
    dst[0] = '\0';
  } else {
    strncpy(dst, &buf[from-1], to - from + 1);
    dst[to-from+1] = '\0';
  }
  /* lprintf("%d %d %s\n",from, to, dst); */
  return dst;
}

void PDB_count_data(PDB *pdb)
{
  PDB_ATOM *a;
  PDB_RES  *r;
  pdb->n_atom = 0;
  pdb->n_hetatm = 0;
  for (a=pdb->atom;a!=NULL;a=a->next) {
    a->no = pdb->n_atom;
    pdb->n_atom++;
    if (a->header == PDB_H_HETATM)
      pdb->n_hetatm++;
  }
  
  pdb->n_res = 0;
  for (r=pdb->res;r!=NULL;r=r->next) {
    r->no = pdb->n_res;
    r->iba = r->beg_atom->no;
    r->iea = r->end_atom->no;
    pdb->n_res++;
  }
}

void PDB_print_info(PDB *pdb, char *fname)
{
  lprintf("PDB FILE:  %s\n", fname);
  lprintf("  Number of atoms : %d\n", pdb->n_atom);
  lprintf("  Number of hetero atoms :  %d\n", pdb->n_hetatm);
  lprintf("  Number of residues : %d\n", pdb->n_res);
  lprintf("\n");
}

char* PDB_make_atom_name(char *pdb_name, char *name)
{
  if (strlen(name) <= 3)
    sprintf(pdb_name," %s",name);
  else
    sprintf(pdb_name,"%s",name);
  
  return pdb_name;
}

void PDB_write_file(PDB *pdb, char *fname)
{
  PDB_write_file_group(pdb, fname, -1);
}

void PDB_write_file_group_fp(PDB *pdb, FILE *fp, int group_no)
{
  int i, ia, n;
  unsigned int g_id;
  char aname[10], res_no[10];
  
  if (group_no < 0) {
    g_id = 0;
  } else {
    g_id = 1 << group_no;
  }

  n = 1;
  for (i=0;i<pdb->n_atom;i++) {
    if (g_id == 0 || (pdb->al[i]->group & g_id)) {
      
      if (pdb->al[i]->res_no < 10000) {
	sprintf(res_no, "%4d ", pdb->al[i]->res_no);
      } else {
	sprintf(res_no, "%5d" , pdb->al[i]->res_no);
      }
      
      fprintf(fp,"ATOM%7d %-4s %-4s%c%-5s   %8.3f%8.3f%8.3f%6.2f%6.2f",
	      n, PDB_make_atom_name(aname,pdb->al[i]->name),
	      pdb->al[i]->res_name,
	      pdb->al[i]->chain,
	      res_no,
	      pdb->al[i]->x, pdb->al[i]->y, pdb->al[i]->z,
	      pdb->al[i]->o, pdb->al[i]->t);
      fprintf(fp,"\n");
      n++;
    }
  }
}

void PDB_write_file_group(PDB *pdb, char *fname, int group_no)
{
  FILE *fp;

  if ((fp = fopen(fname,"w")) == NULL) {
    lprintf("  ERROR: Can't open file %s\n", fname);
    marble_exit(1);
  }

  PDB_write_file_group_fp(pdb, fp, group_no);
  
  fclose(fp);
}

void PDB_print_atom(PDB *pdb, int no)
{
  char aname[10];
  
  lprintf("ATOM%7d %-4s %-4s%c%4d    %8.3f%8.3f%8.3f%6.2f%6.2f\n",
	  no+1, PDB_make_atom_name(aname,pdb->al[no]->name),
	  pdb->al[no]->res_name,
	  pdb->al[no]->chain,
	  pdb->al[no]->res_no,
	  pdb->al[no]->x, pdb->al[no]->y, pdb->al[no]->z,
	  pdb->al[no]->o, pdb->al[no]->t);
}


void PDB_copy_coor_from_vec(PDB *pdb, VEC *x, int n_atom)
{
  int i;

  for (i=0;i<n_atom;i++) {
    pdb->al[i]->x = x[i].x;
    pdb->al[i]->y = x[i].y;
    pdb->al[i]->z = x[i].z;
  }
}

void PDB_copy_group_coor_from_vec(PDB *pdb, VEC *x, int n_atom, int group_no)
{
  int i, j, g_bit;

  if (group_no < 0) {
    g_bit = 0;
  } else {
    g_bit = 1 << group_no;
  }

  j=0;
  for (i=0;i<pdb->n_atom;i++) {
    if (g_bit == 0 || pdb->al[i]->group & g_bit) {
      if (j >= n_atom) {
	lprintf("Internal ERROR: Exceed limit of coordinate array!\n");
	marble_exit(1);
      }
      pdb->al[i]->x = x[j].x;
      pdb->al[i]->y = x[j].y;
      pdb->al[i]->z = x[j].z;
      ++j;
    }
  }
}


PDB_ATOM *PDB_search_atom_in_res(char *name, PDB_RES *res)
{
  PDB_ATOM *a,*prev;

  for (a=res->beg_atom,prev=NULL;prev!=res->end_atom;prev=a,a=a->next) {
    if (strcmp(name,a->name) == 0) return a;
  }
  return NULL;
}

void PDB_print_unchecked(PDB *pdb)
{
  PDB_ATOM *atom;

  for (atom=pdb->atom;atom!=NULL;atom=atom->next) {
    if (!atom->check) {
      lprintf("Warning: Atom %-4s (%4d%c) in residue %-4s(%4d%c) in pdb file is unused.\n",
	      atom->name,atom->pdb_no, atom->alt_loc,
	      atom->res_name,atom->res_no,atom->chain);
    }
  }
}

char *PDB_atom_name(char *name, char *buf)
{
  if (strlen(name)<=3)
    sprintf(buf," %s", name);
  else
    sprintf(buf,"%s", name);
  return buf;
}

void PDB_renumber(PDB *pdb)
{
  PDB_RES *r;
  PDB_ATOM *a,*prev;
  int no;

  no = 1;
  for (r=pdb->res;r!=NULL;r=r->next) {
    r->pdb_no = no++;
    r->chain = ' ';
    for (a=r->beg_atom,prev=NULL;prev!=r->end_atom;prev=a,a=a->next) {
      a->res_no = r->pdb_no;
      a->chain = ' ';
    }
  }
}

void PDB_make_array(PDB *pdb)
{
  PDB_ATOM *a;
  PDB_RES  *r;
  int i;
  char *fname = "PDB_make_array";
  
  pdb->al = emalloc(fname, sizeof(PDB_ATOM*)*pdb->n_atom);
  pdb->rl = emalloc(fname, sizeof(PDB_RES*)*pdb->n_res);
  
  for (a=pdb->atom,i=0;a!=NULL;a=a->next,i++) {
    pdb->al[i] = a;
  }
    
  for (r=pdb->res,i=0;r!=NULL;r=r->next,i++) {
    pdb->rl[i] = r;
  }
}

void PDB_calc_res_center(PDB *pdb)
{
  int i, ia, n;

  for (i=0;i<pdb->n_res;i++) {
    pdb->rl[i]->x = pdb->rl[i]->y = pdb->rl[i]->z = 0.0;
    n=0;
    for (ia=pdb->rl[i]->iba;ia<=pdb->rl[i]->iea;ia++) {
      pdb->rl[i]->x += pdb->al[ia]->x;
      pdb->rl[i]->y += pdb->al[ia]->y;
      pdb->rl[i]->z += pdb->al[ia]->z;
      n++;
    }
    pdb->rl[i]->x /= n;
    pdb->rl[i]->y /= n;
    pdb->rl[i]->z /= n;
  }
}

void PDB_set_weight(PDB *pdb)
{
  PDB_ATOM *a;
  int i;
  static struct {
    char *name;
    double w;
  } wlist[] = {
    "SOD", 22.989770,
    "MG",  24.305000,
    "POT", 39.102000,
    "CAL", 40.080000,
    "CLA", 35.450000,
    "ZN",  65.370000,
    "",    0.0
  };

  for (a=pdb->atom;a!=NULL;a=a->next) {
    a->w = 0.0;
    for (i=0;wlist[i].name[0] != '\0';i++) {
      if (strcmp(a->name, wlist[i].name) == 0) {
	a->w = wlist[i].w;
      }
    }
    if (a->w != 0.0) continue;
    switch (a->name[0]) {
    case 'H' :
      a->w = 1.008;
      break;
    case 'C' :
      a->w = 12.011;
      break;
    case 'N' :
      a->w = 14.007;
      break;
    case 'O' :
      a->w = 15.9994;
      break;
    case 'P' :
      a->w = 30.974;
      break;
    case 'S' :
      a->w = 32.06;
      break;
    default:
      lprintf("Unknown atom %s for weight.\n", a->name);
      marble_exit(1);
    }
  }

  /*
  for (a=pdb->atom;a!=NULL;a=a->next) {
    double round();
    a->w = round(a->w);
  }
  */
	 
}

void PDB_get_weight(PDB *pdb, double *w, int n_atom, int *list)
{
  int i;
  for (i=0;i<n_atom;i++) {
    w[i] = pdb->al[list[i]]->w;
  }
}


void PDB_set_elec_weight(PDB *pdb)
{
  PDB_ATOM *a;

  for (a=pdb->atom;a!=NULL;a=a->next) {
    switch (a->name[0]) {
    case 'H' :
      a->w = 1.0;
      break;
    case 'C' :
      a->w = 6.0;
      break;
    case 'N' :
      a->w = 7.0;
      break;
    case 'O' :
      a->w = 8.0;
      break;
    case 'P' :
      a->w = 15.0;
      break;
    case 'S' :
      a->w = 16.0;
      break;
    default:
      lprintf("Unknown atom for weight.\n");
    }
  }
	 
}

void PDB_get_size(PDB *pdb, double min[3], double max[3])
{
  PDB_ATOM *a;
  double x[3];
  int i;
  
  min[0] = min[1] = min[2] =  1.0e10;
  max[0] = max[1] = max[2] = -1.0e10;
  
  for (a=pdb->atom;a!=NULL;a=a->next) {
    x[0] = a->x; x[1] = a->y; x[2] = a->z;
    for (i=0;i<3;i++) {
      if (min[i] > x[i]) min[i] = x[i];
      if (max[i] < x[i]) max[i] = x[i];
    }
  }
}


/* pdb group routines */
#define PDB_GROUP_BUF_ATOM_NO          1
#define PDB_GROUP_BUF_RES_NO           2
#define PDB_GROUP_BUF_ATOM_NAME        3
#define PDB_GROUP_BUF_ATOM_NAME_RES_NO 4
#define PDB_GROUP_BUF_RES_NAME         5
#define PDB_GROUP_BUF_ATOM_TYPE        6
#define PDB_GROUP_BUF_GROUP_NO         7

void PDB_set_group_str_list(PDB *pdb, STR_LIST *str_list, int g_no, char *header)
{
  STR_LIST *s;

  PDB_clear_group(pdb, g_no);

  for (s=str_list;s != NULL; s=s->next) {
    PDB_set_group_buf(pdb, s->buf, g_no, header);
  }
}

void PDB_set_group_buf(PDB *pdb, char *buf, int g_no, char *header)
{
  char atom1[100], atom2[100];
  char res1[100],  res2[100];
  char mode_str[100];
  int  mode_id, type, ret;
  int id1, id2;
  int no1, no2;
  int d;
  int src_g_no;
  char chain1, chain2;

  if ((ret = sscanf(buf, " atom_no %99s %99s %99s", atom1, atom2, mode_str)) == 3) {
    type = PDB_GROUP_BUF_ATOM_NO;
  } else if (ret == 2) {
    type = PDB_GROUP_BUF_ATOM_NO;
    strcpy(mode_str,"add");

  } else if ((ret = sscanf(buf, " res_no %99s %99s %99s", res1, res2, mode_str)) == 3) {
    type = PDB_GROUP_BUF_RES_NO;
  } else if (ret == 2) {
    type = PDB_GROUP_BUF_RES_NO;
    strcpy(mode_str,"add");

  } else if ((ret = sscanf(buf, " atom_name %99s %99s %99s %99s", atom1, res1, res2, mode_str)) == 4) {
    type = PDB_GROUP_BUF_ATOM_NAME_RES_NO;
  } else if (ret == 3) {
    type = PDB_GROUP_BUF_ATOM_NAME_RES_NO;
    strcpy(mode_str,"add");
  } else if (ret == 2) {
    type = PDB_GROUP_BUF_ATOM_NAME;
    strcpy(mode_str,res1);
  } else if (ret == 1) {
    type = PDB_GROUP_BUF_ATOM_NAME;
    strcpy(mode_str,"add");

  } else if ((ret = sscanf(buf, " atom %99s %99s %99s %99s", atom1, res1, res2, mode_str)) == 4) {
    type = PDB_GROUP_BUF_ATOM_NAME_RES_NO;
  } else if (ret == 3) {
    type = PDB_GROUP_BUF_ATOM_NAME_RES_NO;
    strcpy(mode_str,"add");
  } else if (ret == 2) {
    type = PDB_GROUP_BUF_ATOM_NAME;
    strcpy(mode_str,res1);
  } else if (ret == 1) {
    type = PDB_GROUP_BUF_ATOM_NAME;
    strcpy(mode_str,"add");
    
  } else if ((ret = sscanf(buf, " res_name %99s %99s", res1, mode_str)) == 2) {
    type = PDB_GROUP_BUF_RES_NAME;
  } else if (ret == 1) {
    type = PDB_GROUP_BUF_RES_NAME;
    strcpy(mode_str,"add");

  } else if ((ret = sscanf(buf, " group_no %d %99s", &d, mode_str)) == 2) {
    type = PDB_GROUP_BUF_GROUP_NO;    
    src_g_no = d;
  } else if (ret == 1) {
    type = PDB_GROUP_BUF_GROUP_NO;    
    src_g_no = d;
    strcpy(mode_str,"add");

  } else if ((ret = sscanf(buf, " %d %99s", &d, mode_str)) == 2) {
    type = PDB_GROUP_BUF_GROUP_NO;    
    src_g_no = d;
  } else if (ret == 1) {
    type = PDB_GROUP_BUF_GROUP_NO;    
    src_g_no = d;
    strcpy(mode_str,"add");

  } else {
    lprintf("%s ERROR: unrecognized group definition: %s\n", header, buf);
    marble_exit(1);
  }

  mode_id = PDB_get_mode_id(mode_str);
  
  switch (type) {
  case PDB_GROUP_BUF_ATOM_NO:
    lprintf("%s atoms from %s to %s, Operator = %s\n", header, atom1, atom2, mode_str);
    id1 = PDB_str_to_atom_no(pdb, atom1);
    id2 = PDB_str_to_atom_no(pdb, atom2);
    PDB_group_atom_no(pdb, g_no, id1, id2, mode_id);
    break;
  case PDB_GROUP_BUF_RES_NO:
    lprintf("%s residues from %s to %s, Operator = %s\n", header, res1, res2, mode_str);
    PDB_str_to_res_no(pdb, res1, &id1, &no1, &chain1, 1);
    PDB_str_to_res_no(pdb, res2, &id2, &no2, &chain2, -1);
    PDB_group_res_no(pdb, g_no, id1, id2, mode_id);
    break;
  case PDB_GROUP_BUF_ATOM_NAME:
    lprintf("%s atom name \"%s\", Operator = %s\n", header, atom1, mode_str);
    PDB_group_atom_name(pdb, g_no, atom1, mode_id);
    break;
  case PDB_GROUP_BUF_ATOM_NAME_RES_NO:
    lprintf("%s atom name \"%s\" in residues from %s to %s, Operator = %s\n", header, atom1, res1, res2, mode_str);
    PDB_group_atom_name_res_no(pdb, g_no, atom1, res1, res2, mode_id);
    break;
  case PDB_GROUP_BUF_RES_NAME:
    lprintf("%s residue name \"%s\", Operator = %s\n", header, res1, mode_str);
    PDB_group_res_name(pdb, g_no, res1, mode_id);
    break;
  case PDB_GROUP_BUF_GROUP_NO:
    lprintf("%s group no. %d, Operator = %s\n", header, src_g_no, mode_str);
    PDB_group_from_another_group(pdb, g_no, src_g_no, mode_id);
    break;
  }    
  lprintf("%s          %d atoms selected in total\n", header,PDB_count_group_atoms(pdb, g_no));
}

int PDB_count_group_atoms(PDB *pdb, int g_no)
{
  unsigned int g_bit, i, count;

  g_bit = 1 << g_no;
  count = 0;

  for (i=0;i<pdb->n_atom;i++) {
    if (pdb->al[i]->group & g_bit) {
      count++;
    }
  }
  return count;
}

void PDB_list_group_atoms(PDB *pdb, int g_no, int *list)
{
  unsigned int g_bit, i, count;

  g_bit = 1 << g_no;
  count = 0;

  for (i=0;i<pdb->n_atom;i++) {
    if (pdb->al[i]->group & g_bit) {
      list[count++] = i;
    }
  }
}

int PDB_get_mode_id(char *mode)
{
  if      (strcmp(mode,"add") == 0)     return GM_OR;
  else if (strcmp(mode,"exclude") == 0) return GM_AND_NOT;
  else if (strcmp(mode,"and") == 0)     return GM_AND;
  else if (strcmp(mode,"or") == 0)      return GM_OR;
  else if (strcmp(mode,"and_not") == 0) return GM_AND_NOT;
  else if (strcmp(mode,"or_not") == 0)  return GM_OR_NOT;

  lprintf("Unknown operator %s\n", mode);
  marble_exit(1);
  return 1;
}

void PDB_set_g_bit(PDB *pdb, int atom_no, int g_bit, int ok, int mode)
{
  switch (mode) {
  case GM_OR:
    if (ok)
      pdb->al[atom_no]->group |= g_bit;
    break;
  case GM_AND:
    if (!ok)
      pdb->al[atom_no]->group &= ~g_bit;
    break;
  case GM_OR_NOT:
    if (!ok)
      pdb->al[atom_no]->group |= g_bit;
    break;
  case GM_AND_NOT:
    if (ok)
      pdb->al[atom_no]->group &= ~g_bit;
    break;
  }
}

void PDB_clear_group(PDB *pdb, int g_no)
{
  int i;
  unsigned int g_bit;

  g_bit = 1 << g_no;

  for (i=0;i<pdb->n_atom;i++) {
    PDB_set_g_bit(pdb, i, g_bit, 1, GM_AND_NOT);
  }
}

void PDB_group_atom_no(PDB *pdb, int g_no, int no1, int no2, int mode)
{
  int i, ok;
  unsigned int g_bit;
  
  /* no1--;  no2--; */
  g_bit = 1 << g_no;
  
  for (i=0;i<pdb->n_atom;i++) {
    if (i >= no1 && i <= no2) ok = 1;
    else ok = 0;

    PDB_set_g_bit(pdb, i, g_bit, ok, mode);
  }
}

void PDB_group_res_no(PDB *pdb, int g_no, int no1, int no2, int mode)
{
  int i, j, ok;
  unsigned int g_bit;
  
  /* no1--;  no2--; */
  g_bit = 1 << g_no;

  for (i=0;i<pdb->n_atom;i++) {
    if (pdb->al[i]->res_no >= no1 && pdb->al[i]->res_no <= no2) ok = 1;
    else ok = 0;
    
    PDB_set_g_bit(pdb, i, g_bit, ok, mode);
  }
}

void PDB_group_atom_name(PDB *pdb, int g_no, char *name, int mode)
{
  int i, ok, flag;
  unsigned int g_bit;
  
  g_bit = 1 << g_no;

  flag = ATOM_DATA_atom_name_to_flag(name);

  for (i=0;i<pdb->n_atom;i++) {
    ok = 0;

    if (flag==0) {
      if (strmatch(pdb->al[i]->name, name) == 0) ok = 1;
    } else if (flag==1) {
      /* heavy_atom or non_hydrogen */
      if (ATOM_DATA_match_non_hydrogen(pdb->al[i]->name)) ok = 1;
    } else if (flag==2) {
      /* backbone */
      if (ATOM_DATA_match_backbone(pdb->al[i]->name)) ok = 1;
    } else if (flag==3) {
	/* all */
      ok = 1;
    }

    PDB_set_g_bit(pdb, i, g_bit, ok, mode);
  }
}

void PDB_group_atom_name_res_no(PDB *pdb, int g_no, char *name, char *resno1, char *resno2, int mode)
{
  int i, ok, flag;
  unsigned int g_bit;
  int id1, id2, no1, no2;
  char chain1, chain2;

  g_bit = 1 << g_no;

  PDB_str_to_res_no(pdb, resno1, &id1, &no1, &chain1, 1);
  PDB_str_to_res_no(pdb, resno2, &id2, &no2, &chain2, -1);
  
  flag = ATOM_DATA_atom_name_to_flag(name);
  
  for (i=0;i<pdb->n_atom;i++) {
    ok = 0;

    if (pdb->al[i]->res->no >= id1 && 
	pdb->al[i]->res->no <= id2) {

      if (flag==0) {
	if (strmatch(pdb->al[i]->name, name) == 0) ok = 1;
      } else if (flag==1) {
	/* heavy_atom or non_hydrogen */
	if (ATOM_DATA_match_non_hydrogen(pdb->al[i]->name)) ok = 1;
      } else if (flag==2) {
	/* backbone */
	if (ATOM_DATA_match_backbone(pdb->al[i]->name)) ok = 1;
      } else if (flag==3) {
	/* all */
	ok = 1;
      }
      
    }
    PDB_set_g_bit(pdb, i, g_bit, ok, mode);
  }
}

void PDB_group_from_another_group(PDB *pdb, int g_no, int src_g_no, int mode)
{
  int i, ok;
  unsigned int g_bit, src_g_bit;
  
  g_bit = 1 << g_no;
  src_g_bit = 1 << src_g_no;

  for (i=0;i<pdb->n_atom;i++) {
    if (pdb->al[i]->group & src_g_bit) ok = 1;
    else ok = 0;

    PDB_set_g_bit(pdb, i, g_bit, ok, mode);
  }
}


int PDB_str_to_atom_no(PDB *pdb, char *name)
{
  char aname[10], rname[10], chain;
  int  i, j, res_no;

  if (isdigit(name[0])) {
    i = atoi(name);
    return (--i);
  }
  
  if (sscanf(name, "%[^ \t\n.].%s", aname, rname) != 2) {
    lprintf("  ERROR: %s: Invalid atom name format\n",name);
    marble_exit(1);
  }
  PDB_str_to_res_no(pdb, rname, &j, &res_no, &chain, 0);

  for (i=pdb->rl[j]->iba;i<=pdb->rl[j]->iea;i++) {
    if (strcmp(aname,pdb->al[i]->name) == 0) 
      break;
  }
  if (i>pdb->rl[j]->iea) {
    lprintf("  ERROR: %s: No atom %s in Residue %s\n",name, aname, rname);
    marble_exit(1);
  }
  return i;
}


void PDB_str_to_res_no(PDB *pdb, char *name, int *id, int *resno, char *chain, int flag)
{
  int  i, j, err=0;
  int nearest_j, diff;

  if (isdigit(name[0])) {
    if (sscanf(name, "%d%c", resno, chain) != 2) {
      if (sscanf(name, "%d", resno) != 1) {
	err=1;
      }
      *chain = ' ';
    }
  } else {
    if (sscanf(name, "%c%d", chain, resno) != 2) {
      err=1;
    }
  }

  if (err) {
    lprintf("ERROR: Residue Name Format Error \"%s\"\n",name);
    marble_exit(1);
  }

  nearest_j = -1;
  err = 1;
  for (j=0;j<pdb->n_res;j++) {
    if (pdb->rl[j]->chain == *chain) {
      if (pdb->rl[j]->pdb_no == *resno) {
	*id = j;
	return;
      }
      diff = pdb->rl[j]->pdb_no - *resno;
      if (flag == -1 && diff < 0 && (err || (nearest_j < j))) {
	nearest_j = j; 
	err = 0;
      }
      if (flag == 1 && diff > 0 && (err || (nearest_j > j))) {
	nearest_j = j; 
	err = 0;
      }
    }
  }

  /*
  if (err) {
    if (flag) {
      lprintf("ERROR: Can't find chain \'%c\'",*chain);
    } else {
      lprintf("ERROR: Can't find residue %d in chain \'%c\'",*resno,*chain);
    }
    marble_exit(1);
  }
  */
  *id = nearest_j;
  return;
}

void PDB_group_atom_name_res_name(PDB *pdb, int g_no, char *name, char *res_name, int mode)
{
  int i, ok, flag;
  unsigned int g_bit;

  /* no1--; no2--; */
  g_bit = 1 << g_no;
  
  flag = ATOM_DATA_atom_name_to_flag(name);

  for (i=0;i<pdb->n_atom;i++) {
    ok = 0;

    if (strmatch(pdb->al[i]->res_name, res_name) == 0) {

      if (flag==0) {
	if (strmatch(pdb->al[i]->name, name) == 0) ok = 1;
      } else if (flag==1) {
	/* heavy_atom or non_hydrogen */
	if (ATOM_DATA_match_non_hydrogen(pdb->al[i]->name)) ok = 1;
      } else if (flag==2) {
	/* backbone */
	if (ATOM_DATA_match_backbone(pdb->al[i]->name)) ok = 1;
      } else if (flag==3) {
	/* all */
	ok = 1;
      }
      
    }

    PDB_set_g_bit(pdb, i, g_bit, ok, mode);
  }
}

void PDB_group_res_name(PDB *pdb, int g_no, char *name, int mode)
{
  int i, j, ok;
  unsigned int g_bit;
  
  g_bit = 1 << g_no;
  
  for (i=0;i<pdb->n_atom;i++) {
    if (strmatch(pdb->al[i]->res_name, name) == 0) ok = 1;
    else ok = 0;

    PDB_set_g_bit(pdb, i, g_bit, ok, mode);
  }
}

void PDB_group_atom_type(PDB *pdb, int g_no, int mode, int flag)
{
  int i, ok;
  unsigned int g_bit;
  
  g_bit = 1 << g_no;

  for (i=0;i<pdb->n_atom;i++) {

    ok = 0;
    if (flag == 1) {
      /* backbone */
      if (ATOM_DATA_match_backbone(pdb->al[i]->name)) ok = 1;
    } else if (flag == 2) {
      /* heavy_atom or non_hydrogen */
      if (ATOM_DATA_match_non_hydrogen(pdb->al[i]->name)) ok = 1;
#if 0      
    } else if (flag == 3) {
      /* solute_heavy_atom */
      if (i < pdb->n_solute_atom &&
	  ATOM_DATA_match_non_hydrogen(pdb->al[i]->name)) ok = 1;
#endif      
    }
    
    PDB_set_g_bit(pdb, i, g_bit, ok, mode);
  }
}

void PDB_check_from_group(PDB *pdb, int g_no, int check)
{
  int i;
  unsigned int g_bit;
  
  g_bit = 1 << g_no;
  for (i=0;i<pdb->n_atom;i++) {
    if (pdb->al[i]->group & g_bit) {
      if (pdb->al[i]->check != 0 && pdb->al[i]->check != check) {
	lprintf("Warning: Atom %d is overlapped between groups %d and %d. Overwited.\n",
		i+1, pdb->al[i]->check, check);
      }
      pdb->al[i]->check = check;
    }
  }
}

void PDB_get_atoms_in_group(PDB *pdb, int g_no, int **parray, int *pn)
{
  int i,n,*array;
  unsigned int g_bit;
  
  g_bit = 1 << g_no;
  
  n = 0;
  for (i=0;i<pdb->n_atom;i++) {
    if (pdb->al[i]->group & g_bit) {
      n++;
    }
  }
  array = emalloc("pdb_get_array_of_group", sizeof(int)*n);
  n = 0;
  for (i=0;i<pdb->n_atom;i++) {
    if (pdb->al[i]->group & g_bit) {
      array[n] = i;
      n++;
    }
  }
  *parray = array;
  *pn = n;
}

void PDB_get_coord_in_group(PDB *pdb, int g_no, VEC *x0, int n_atom)
{
  int i,n;
  unsigned int g_bit;
  
  g_bit = 1 << g_no;
  
  n = 0;
  for (i=0;i<pdb->n_atom;i++) {
    if (pdb->al[i]->group & g_bit) {
      x0[n].x = pdb->al[i]->x;
      x0[n].y = pdb->al[i]->y;
      x0[n].z = pdb->al[i]->z;
      n++;
      if (n > n_atom) {
	lprintf("ERROR: exceed number of selected atoms in pdb file\n");
	marble_exit(1);
      }
    }
  }
}

void PDB_make_recip(double boxv[3][3], double recip[3][3])
{
  int i, j;
  double v01[3], v12[3], v20[3], tmp, V;

  cross3(boxv[1],boxv[2],v12);
  cross3(boxv[2],boxv[0],v20);
  cross3(boxv[0],boxv[1],v01);
  V = dot3(boxv[0],v12);
  
  for (j=0;j<3;j++) {
    recip[j][0] = v12[j] / V;
    recip[j][1] = v20[j] / V;
    recip[j][2] = v01[j] / V;
  }
}


void PDB_wrap_atoms_vec(PDB *pdb, VEC *x, double boxv[3][3], int origin)
{
  double recip[3][3];
  VEC frac;
  int i;

  PDB_make_recip(boxv, recip);
  
  for (i=0;i<pdb->n_atom;i++) {
    frac.x = VEC_MUL_MAT_X(x[i],recip);
    frac.y = VEC_MUL_MAT_Y(x[i],recip);
    frac.z = VEC_MUL_MAT_Z(x[i],recip);
    
    if (origin == O_CENTER) {
      frac.x += 0.5;
      frac.y += 0.5;
      frac.z += 0.5;
    }
    frac.x -= floor(frac.x);
    frac.y -= floor(frac.y);
    frac.z -= floor(frac.z);
    
    if (origin == O_CENTER) {
      frac.x -= 0.5;
      frac.y -= 0.5;
      frac.z -= 0.5;
    }
    
    x[i].x = VEC_MUL_MAT_X(frac,boxv);
    x[i].y = VEC_MUL_MAT_Y(frac,boxv);
    x[i].z = VEC_MUL_MAT_Z(frac,boxv);
  }  
}

void PDB_wrap_atoms_vec_res(PDB *pdb, VEC *x, double boxv[3][3], int origin)
{
  double recip[3][3];
  VEC frac, fold, offset;
  int i, j;

  PDB_make_recip(boxv, recip);
  
  for (j=0;j<pdb->n_res;j++) {
    i = pdb->rl[j]->iba;
    
    frac.x = VEC_MUL_MAT_X(x[i],recip);
    frac.y = VEC_MUL_MAT_Y(x[i],recip);
    frac.z = VEC_MUL_MAT_Z(x[i],recip);
    
    if (origin == O_CENTER) {
      frac.x += 0.5;
      frac.y += 0.5;
      frac.z += 0.5;
    }

    frac.x = floor(frac.x);
    frac.y = floor(frac.y);
    frac.z = floor(frac.z);
    
    offset.x = VEC_MUL_MAT_X(frac,boxv);
    offset.y = VEC_MUL_MAT_Y(frac,boxv);
    offset.z = VEC_MUL_MAT_Z(frac,boxv);
    
    for (;i<=pdb->rl[j]->iea;i++) {
      x[i].x -= offset.x;
      x[i].y -= offset.y;
      x[i].z -= offset.z;
    }
  }  
}

PDB_ATOM *PDB_search_atom_by_name(PDB *pdb, char *str)
{
  char name[100], chain;
  int  res_no;
  int  ret;
  PDB_RES *r;
  PDB_ATOM *a;

  ret = sscanf(str,"%[A-Za-z0-9*'_].%d%c", name, &res_no, &chain);
  if (ret == 2)
    chain = ' ';
  else if (ret != 3) {
    lprintf("ERROR: wrong atom string %s\n",str);
    lprintf("       name.res_no[chain] (ex. N.10A)\n");
    lprintf("%s.%d%c**\n", name,res_no,chain);
    marble_exit(1);
  }

  for (r=pdb->res;r!=NULL;r=r->next) {
    if (r->pdb_no == res_no && r->chain == chain) {
      for (a=r->beg_atom;a!=NULL;a=a->next) {
	if (strcmp(a->name, name) == 0)
	  return a;
	if (a == r->end_atom)
	  break;
      }
    }
  }
  
  lprintf("ERROR: Can't find an atom %s in pdb file\n", str);
  /*lprintf("%s%d%c**\n", name,res_no,chain);*/
  marble_exit(1);
  return NULL;
}
