/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifndef _NONBOND_H_INCLUDED
#define _NONBOND_H_INCLUDED

#define NL_UNKNOWN       0
#define NL_ATOM_BASED    1
#define NL_RESIDUE_BASED 2
#define NL_LINKED_CELL   3
#define NL_FMM           4

#ifdef NONBOND_C
SEL_DATA _sel_NONBOND_LIST_METHOD[] = {
  "atom_based",    NL_ATOM_BASED,
  "residue_based", NL_RESIDUE_BASED,
  "linked_cell",   NL_LINKED_CELL,
  "FMM",           NL_FMM,
  NULL,            NL_UNKNOWN
};
#else
extern SEL_DATA _sel_NONBOND_LIST_METHOD[];
#endif

#define NE_UNKNOWN   0
#define NE_PSW       1
#define NE_FSH       2
#define NE_RPSW      3
#define NE_EWALD     4
#define NE_FMM       5

#ifdef NONBOND_C
SEL_DATA _sel_NONBOND_ELEC_METHOD[] = {
  "potential_switch",           NE_PSW,
  "shifted_force",              NE_FSH,
  "residue_potential_switch",   NE_RPSW,
  "ewald",                      NE_EWALD,
  "FMM",                        NE_FMM,
  NULL,                         NE_UNKNOWN
};
#else
extern SEL_DATA _sel_NONBOND_ELEC_METHOD[];
#endif

#define NV_UNKNOWN 0
#define NV_PSW     1
#define NV_FSW     2
#define NV_TRN     3

#ifdef NONBOND_C
SEL_DATA _sel_NONBOND_VDW_METHOD[] = {
  "potential_switch", NV_PSW,
  "force_switch",     NV_FSW,
  "truncate",         NV_TRN,
  NULL,               NV_UNKNOWN
};
#else
extern SEL_DATA _sel_NONBOND_VDW_METHOD[];
#endif

#define J_LIST

typedef struct _s_NONBOND_LIST {
  int n_alloc;
  int n_list;
#ifdef J_LIST
  int n_i_list;
  int *j_list;
#endif  
  int (*ij_list)[2];

  double alloc_ratio;   /* additional allocation ratio */
  int alloc_unit;
  int n_alloc_atom;
  VEC *x_at_update;
  
  int nl_method;     /* method to make nonbond-pair list*/
  int vdw_method;    /* method to calculate vdw energy and force */
  int elec_method;  /* method to calculate electrostatic energy and force */
  
  int update_step;        /* update steps */
  double rl_list;         /* outside cutoff length for nonbond list */
  double rl_on;           /* outside length to turn on  smoothing */
  double rl_off, rl_off2; /* outside length to turn off smoothing */
  double rs_list;         /* inside cutoff length for nonbond list */
  double rs_on, rs_on2;   /* inside length to turn on  smoothing */
  double rs_off;          /* inside length to turn off smoothing */
  
  double rl_list0, rl_off0;  /* user specified rl_list, rl_off
				this is not affected from box size */

  double group_margin, cell_margin, cell_div;
  int group_flag;
				
  int outside_flag;       /* whether this nonbond list is outside or not */

  double time_list, time_fmm_setup;

  LINKED_CELL *lc;
  
#ifdef MPI_RDMD
  int start_task, end_task;
#endif  
} NONBOND_LIST;


/*
#define NBLIST_ALLOC_UNIT  10000
#define NBLIST_FIRST_ALLOC 100
*/

void NONBOND_LIST_init(NONBOND_LIST *, BOUNDARY *);
void NONBOND_LIST_set_group_flag(NONBOND_LIST *nl, ATOM_DATA *ad, int flag);
void NONBOND_LIST_set_param(NONBOND_LIST *nl, BOUNDARY *bc, ATOM_DATA *ad,
			    int nl_method, int vdw_method, int elec_method,
			    double cutoff, double smooth_margin, double list_margin, 
			    double hydr_margin, double cell_margin, int update_step, int vir_cor);
void NONBOND_LIST_set_cutoff(NONBOND_LIST *nl, BOUNDARY *,
			     double rl, double rl_sm, double rl_list_buf,
			     double rs, double rs_sm, double rs_list_buf);
void NONBOND_LIST_reset_cutoff_from_boundary(NONBOND_LIST *,BOUNDARY *);
void NONBOND_LIST_get_rl(NONBOND_LIST *nl,
			 double *rl, double *rl_sm, double *rl_list);
int NONBOND_LIST_check_update(NONBOND_LIST *nl, ATOM_DATA *ad, BOUNDARY *, int step);
void NONBOND_LIST_print_property(NONBOND_LIST *nl);
void NONBOND_LIST_setup(NONBOND_LIST *nl, ATOM_DATA *ad,
			LINKED_CELL *lc,  BOUNDARY *bc);
void make_nonbond_list(NONBOND_LIST *nl, ATOM_DATA *ad, LINKED_CELL *lc,
		       BOUNDARY *bc);
void make_nonbond_list_atom_based(NONBOND_LIST *nl, ATOM_DATA *ad, 
				  BOUNDARY *bc);
void make_nonbond_list_residue_based(NONBOND_LIST *nl, ATOM_DATA *ad, 
				     BOUNDARY *bc, int only_count);
void alloc_nonbond_list(NONBOND_LIST *nl);
void nonbond_energy_force(NONBOND_LIST *nl, ATOM_DATA *ad,
			  BOUNDARY *bc,
                          double *vdw, double *elec, double *hbond);
void nonbond_energy_force_smooth(NONBOND_LIST *nl, ATOM_DATA *ad,
				 BOUNDARY *bc,
				 double *vdw, double *elec, double *hbond);
void nonbond_energy_force_all(NONBOND_LIST *nl, ATOM_DATA *ad,
			      BOUNDARY *bc,
			      double *vdw, double *elec, double *hbond);
void nonbond_energy_force_residue_based(NONBOND_LIST *nl, ATOM_DATA *ad,
					BOUNDARY *bc,
					double *vdw, double *elec, double *hbond);

/* from linked_cell.h */
void LINKED_CELL_setup(LINKED_CELL *lc,NONBOND_LIST *nl,
                       ATOM_DATA *ad, BOUNDARY *bc);
void LINKED_CELL_set_neighbor(LINKED_CELL *lc, BOUNDARY *bc, NONBOND_LIST *nl);
void LINKED_CELL_alloc_nonbond_list(LINKED_CELL *lc, NONBOND_LIST *nl,
                                    ATOM_DATA *ad, BOUNDARY *bc);
int LINKED_CELL_make_nonbond_list(LINKED_CELL *lc,NONBOND_LIST *nl,
                                  ATOM_DATA *ad, BOUNDARY *bc, int only_count);


#endif  /* _NONBOND_H_INCLUDED */
