/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "md_system.h"

#ifdef MPI_SDMD
#include "parallel.h"
#include "sdmd.h"
#endif

void EP_NMR_DIS_setup(EXTRA_POT *ep,
		      ATOM_DATA *ad,
		      LINKED_CELL *lc,
		      char *file_name, double scl_dis, double scl_dis1,
		      double Ceil,
		      double sqof, double sqcs, double sqex,
		      double soex, double asym, double rswi,
		      double viol_noe, int n_output_viol,
		      int gradual_change_step, int virial_flag)
{
  EP_NMRDIS *ndis;
  FILE *fp;
  char buf[1000];
  int nnoedis, num_lh, num_rh, i, k, kp, j, h, hp, g_id, ss;
  int count, stl, str;
  char *fname = "NMR_DIS_setup";

  ndis = emalloc(fname, sizeof(EP_NMRDIS));
  EP_add_item(ep, (EP_ITEM*)ndis);
  ndis->type =EP_T_NMR_DIS;
  strcpy(ndis->fname, file_name);
  ndis->virial_flag = virial_flag;
  ndis->gradual_change_step = gradual_change_step;
  ndis->scl_dis = ndis->scl_dis0 = scl_dis;
  ndis->scl_dis1 = scl_dis1;
  ndis->Ceil = Ceil;
  ndis->sqof = sqof;
  ndis->sqcs = sqcs;
  ndis->sqex = sqex;
  ndis->soex = soex;
  ndis->asym = asym;
  ndis->rswi = rswi;
  ndis->viol_noe = viol_noe;
  ndis->n_output_viol = n_output_viol;

  lprintf("NMR NOE DISTANCE RESTRAINT (SOFT-SQUARE)\n");

#ifdef MPI
  if((fp = par_fopen(file_name, "r")) == NULL) {
#else
  if((fp = fopen(file_name, "r")) == NULL) {
#endif
    lprintf("ERROR: %s: No such file\n", file_name);
    marble_exit(1);
  }
#ifdef MPI
  par_fgets(buf,100,fp);
#else
  fgets(buf,100,fp);
#endif

  if (sscanf(buf,"# distance_restraint ver. %d.%d",
	     &ndis->major_version,
	     &ndis->minor_version) != 2) {
    lprintf("ERROR: Invalid distance restraint format in %s\n", file_name);
    marble_exit(1);
  }

  if (ndis->major_version < 1 ||
      ndis->minor_version < 0) {
    lprintf("ERROR: Unsupported version of the file (%d.%d)\n",
	    ndis->major_version, ndis->minor_version);
    marble_exit(1);
  }

#ifdef MPI
  par_fgets(buf,1000,fp);
#else
  fgets(buf,1000,fp);
#endif
  if (sscanf(buf,"%d%d%d", &(nnoedis),&(num_lh),&(num_rh)) != 3) {
    lprintf("ERROR: Invalid file format in %s\n",  file_name);
    marble_exit(1);
  }
  ndis->nnoedis = nnoedis;
  ndis->num_lh = num_lh;
  ndis->num_rh = num_rh;

  lprintf("  REFERENCE RESTRAINT FILE: %s (Ver. %d.%d)\n", ndis->fname,
	  ndis->major_version, ndis->minor_version);
  lprintf("  RESTRAINT PARAMETER\n");
  lprintf("    NUMBER OF NOE RESTRAINT: %d\n",ndis->nnoedis);
  lprintf("    ENERGY SCALE FACTOR S: %f\n", ndis->scl_dis);
  lprintf("    ENERGY SCALE FACTOR C: %f\n", ndis->sqcs);
  lprintf("    CEILING FACTOR: %f\n", ndis->Ceil);
  lprintf("    SQ-EXPONENT: %f\n", ndis->sqex);
  lprintf("    SOFT EXPONENT: %f\n", ndis->soex);
  lprintf("    SQ-OFFSET: %f [Angstrom]\n", ndis->sqof);
  lprintf("    R-SWITCH: %f [Angstrom]\n", ndis->rswi);
  lprintf("    ASYMPTOTE: %f\n", ndis->asym);
  lprintf("  VIRIAL FLAG: %s\n", (ndis->virial_flag) ? "ON" : "OFF");
  if(ndis->gradual_change_step) {
    lprintf("  GRADUAL CHANGE OF FORCE CONSTANT: %f -> %f in %d steps\n",
	    ndis->scl_dis0, ndis->scl_dis1, ndis->gradual_change_step);
  }

  ndis->lrnoe = emalloc(fname,sizeof(int) * (num_lh+num_rh));
  ndis->pn = emalloc(fname,sizeof(BOTH) * nnoedis);
  ndis->rst = emalloc(fname,sizeof(RESTS) * nnoedis);
  for(i = 0;i < nnoedis;i++){
#ifdef MPI
    par_fgets(buf,1000,fp);
#else
    fgets(buf,1000,fp);
#endif
    if(sscanf(buf, "%d%d%d%lf%lf%lf", &(ndis->pn[i].noe_num),
	      &(ndis->pn[i].lh), &(ndis->pn[i].rh),
	      &(ndis->rst[i].ds), &(ndis->rst[i].dm),
	      &(ndis->rst[i].dp)) != 6) {
      lprintf("ERROR: Invalid NOE restraint number in noe list file %s\n",
	      file_name);
      marble_exit(1);
    }
  }
  k = 0;
  h = num_lh;
  kp = 0;
  hp = num_lh;
  stl = 0;
  str = 0;
  count = 0;
  for(i = 0,j = 0;i < nnoedis;i++){
    ndis->pn[i].start_lh = j;
    for(k = 0; k < ndis->pn[i].lh;k++){ 
      fscanf(fp,"%d",&(ndis->lrnoe[j++]));
    }
    ndis->pn[i].start_rh = j;
    for(k = 0; k < ndis->pn[i].rh;k++){ 
      fscanf(fp,"%d",&(ndis->lrnoe[j++]));
    }
  }

  /* lprintf("check: %d %d %d.\n", j, num_lh, num_rh); */
  if(j != num_lh + num_rh) {
    lprintf("ERROR: Invalid total noe restraint number in noe list file %s\n",
    file_name);
    marble_exit(1);
  }

#ifdef MPI
  par_fclose(fp);
#else
  fclose(fp);
#endif
  lprintf("\n");

  for(ss = 0;ss < num_lh + num_rh;ss++){
    ndis->lrnoe[ss] -= 1;
  }
#ifdef MPI_SDMD
  SDMD_setup_ex_tr_atom(lc, ndis->num_lh+ndis->num_rh, ndis->lrnoe);
#endif
}

void EP_NMR_DIS_energy_force(EP_NMRDIS *ndis,ATOM_DATA *ad, LINKED_CELL *lc)
{
  int h,i,j,k;
  int tlh, trh, kl, kr;
  int llh, lrh;
  unsigned int gid;
  double dx, dy, dz;
  double fx, fy, fz;
  double lx, ly, lz;
  double rx, ry, rz;
  double xx, yy, zz, dist, zero;
  double  scal, Ceil, sqof;
  double sqcs, sqex, soex;
  double asym, rswi, scl2;
  double cc, d_upp, d_low, delta;
  double df, dff, a, b, ep;
  double cllh, clrh;
  double viol_chk;

#ifdef MPI_SDMD
  int pe;

  pe = ATOM_CPU(lc, ad, ndis->lrnoe[0]);
  if (pe != mpi.rank) {
    ndis->energy = 0.0;
    return;
  }
#endif

  scl2 = ndis->scl_dis*2.0; 
  ndis->energy = 0.0;
  zero = 0.0;
  ndis->num_viol_noe = 0;

  if (((ndis->sqcs)*(ndis->scl_dis)) < ndis->Ceil) {
    cc = ((ndis->sqcs)*(ndis->scl_dis));
  } else {cc = (ndis->Ceil);}

  cllh = clrh = 0.0;
  for(i=0;i < ndis->nnoedis;i++){
    viol_chk = 0;
    ep=0;
    df=0;
    a=b=delta=d_upp=d_low=dist=0;
    /* calculate the center of atoms (in the case of pseudo atom)*/
    /* left hand atoms*/
    llh = ndis->pn[i].start_lh;
    tlh = llh + ndis->pn[i].lh;
    lrh = ndis->pn[i].start_rh;
    trh = lrh + ndis->pn[i].rh;
    rx = ry = rz = lx = ly = lz = 0.0;
    cllh = 1.0/(double)ndis->pn[i].lh;
    for(j = llh;j < tlh;j++){
      k = ndis->lrnoe[j];
      lx = lx + ad->x[k].x*cllh;
      ly = ly + ad->x[k].y*cllh;
      lz = lz + ad->x[k].z*cllh;
    }
    /* right hand atoms*/
    clrh = 1.0/(double)ndis->pn[i].rh;
    for(j = lrh;j < trh;j++){
      k = ndis->lrnoe[j];
      rx = rx + ad->x[k].x*clrh;
      ry = ry + ad->x[k].y*clrh;
      rz = rz + ad->x[k].z*clrh;
    }
    xx = yy = zz = 0;
    xx = (lx - rx);
    yy = (ly - ry);
    zz = (lz - rz);
    dist = sqrt(xx*xx + yy*yy + zz*zz);

    /*
      delta = dist - (ds + dp)  (ds + dp  < R)
              0                 (ds - dm  < R < ds + dp)
              d - dm - dist     (R < ds - dm)
  
      E = min(Ceil,SC) * (a + b/delta**soex + c*delta)   
                                  (for ds + dp + dsw < dist)
                       * (delta**sqex)
                                  (for ds + dp + dsw > dist)

      continuous at (ds + dp + dsw)
      a = (exp/soex + 1) * dsw**sqex - (1/soex+1) * c * dsw
      b = -(exp/soex) * dsw**(sqex+soex) + c * dsw / soex
    */

    /*deside the restraint condistion*/
    /* deside the delta value */	

    d_upp = ((ndis->rst[i].ds) + (ndis->rst[i].dp) - (ndis->sqof));
    d_low = ((ndis->rst[i].ds) - (ndis->rst[i].dm));
    /* check violations*/
    if(d_upp - dist > 0) {
      if(d_low - dist > 0) {
	viol_chk = fabs(d_low - dist);
      }  
      if(d_low - dist <= 0) {
	viol_chk = 0.0;
      }
    } else {
      viol_chk = fabs(d_upp - dist);
    }
    if(viol_chk > ndis->viol_noe){
      ndis->num_viol_noe += 1;
    }
    if(dist - d_upp > 0){  
      delta = dist - d_upp;
    }
    if(((dist - d_upp) <= 0) && ((dist - d_low) > 0)){
      delta = 0;
    }
    if(dist - d_low <= 0){
      delta = dist - d_low;
    }
    if(delta < ndis->rswi) {
      ep = cc*pow(fabs(delta),(ndis->sqex));
      df = (ndis->sqex)*cc*pow(delta,((ndis->sqex)-1));
    } else {
      a = ((ndis->sqex)/(ndis->soex)+1)*pow((ndis->rswi),(ndis->sqex))
	-(ndis->asym)*(1+1/(ndis->soex))*(ndis->rswi);
      b = -1*((ndis->sqex)/(ndis->soex))*
	pow((ndis->rswi),((ndis->soex)+(ndis->sqex)))+
	(ndis->asym)/(ndis->soex)*pow((ndis->rswi),((ndis->soex)+1));
      ep = cc*(a+b/pow(delta,(ndis->soex))+(ndis->asym)*delta);
      df = cc*((-1*(ndis->soex))*b/pow(delta,((ndis->soex)+1))+(ndis->asym));
    }

    ndis->energy += ep;

    /*force & virial*/
    dff = df/dist;
    fx = -dff * xx;
    fy = -dff * yy;
    fz = -dff * zz;
    for(j = llh;j < tlh;j++){
      kl=ndis->lrnoe[j];
      ad->f[kl].x += fx*cllh;
      ad->f[kl].y += fy*cllh;
      ad->f[kl].z += fz*cllh;
    }
    for(j = lrh;j < trh;j++){
      kr=ndis->lrnoe[j];
      ad->f[kr].x -= fx*clrh;
      ad->f[kr].y -= fy*clrh;
      ad->f[kr].z -= fz*clrh;
    }
    if(ndis->virial_flag){
      ad->virial[0] += -dff * xx * xx;
      ad->virial[1] += -dff * yy * yy;
      ad->virial[2] += -dff * zz * zz;
      ad->virial[3] += -dff * xx * yy;
      ad->virial[4] += -dff * xx * zz;
      ad->virial[5] += -dff * yy * zz;
    }
  }
}

void EP_NMR_DIS_prop_header(EP_NMRDIS *ndis,
			     FILE *fp, int *id)
{
  fprintf(fp, "%2dN_Viol(dis) ", (*id)++);
  fprintf(fp, "%2dENE(dis)    ", (*id)++);
}
/****/
void EP_NMR_DIS_prop_out(EP_NMRDIS *ndis, MD_SYSTEM *sys, FILE *fp)
{
#ifdef MPI_SDMD
  int pe;
  LINKED_CELL *lc;
  ATOM_DATA *ad;
  MPI_Status stat;
  int buf[1];
  double dbuf[1];

  lc = &sys->linked_cell;
  ad = &sys->atom;
  pe = ATOM_CPU(lc, ad, ndis->lrnoe[0]);
  
  if (mpi.master) {
    if (pe != mpi.rank) {
      MPI_Recv(buf,1,MPI_INT, pe, SDMD_TAG_PROP, mpi.comm, &stat);
      ndis->num_viol_noe   = buf[0];
      MPI_Recv(dbuf,1,MPI_DOUBLE, pe, SDMD_TAG_PROP, mpi.comm, &stat);
      ndis->energy   = dbuf[0];
    }
  } else {
    if (pe == mpi.rank) {
      buf[0] = ndis->num_viol_noe;
      MPI_Send(buf,1,MPI_INT, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
      dbuf[0] = ndis->energy;
      MPI_Send(dbuf,1,MPI_DOUBLE, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
    }
    return;
  }
#endif  /*SDMD*/ 
  
  fprintf(fp, "  %8d    ", ndis->num_viol_noe);
  fprintf(fp, " %13.6e", ndis->energy);
}

/****/
void EP_NMR_DIS_gradual_change(EP_NMRDIS *ndis, int step)
{
  double lambda;
  if (ndis->gradual_change_step) {
    if (step < ndis->gradual_change_step) {
      lambda = (double)step/ndis->gradual_change_step;
      ndis->scl_dis = ndis->scl_dis0*(1.0-lambda) + ndis->scl_dis1*lambda;
    } else {
      ndis->scl_dis = ndis->scl_dis1;
    }
  }
}

void EP_NMR_DIS_output(EP_NMRDIS *ndis, MD_SYSTEM *sys)
{
  ATOM_DATA *ad;
  int h,i,j,k;
  int tlh, trh, kl, kr;
  int llh, lrh;
  unsigned int gid;
  double dx, dy, dz;
  double lx, ly, lz;
  double rx, ry, rz;
  double xx, yy, zz, dist;
  double d_upp, d_low;
  double cllh, clrh;
  double viol_chk, rmsd_viol;
  int num_viol_noe;
  struct {
    double viol, dist, d_upp, d_low;
    int n_left_atoms, n_right_atoms, left_atoms[10], right_atoms[10];
  } *viol_list;

  ad = &sys->atom;
  num_viol_noe = 0;
  rmsd_viol = 0.0;

  viol_list = emalloc("EP_NMR_DIS_output", sizeof(*viol_list)*ndis->n_output_viol);
  
  for(i=0;i < ndis->nnoedis;i++){
    viol_chk = 0.0;
    d_upp=d_low=dist=0.0;
    /* calculate the center of atoms (in the case of pseudo atom)*/
    /* left hand atoms*/
    llh = ndis->pn[i].start_lh;
    tlh = llh + ndis->pn[i].lh;
    lrh = ndis->pn[i].start_rh;
    trh = lrh + ndis->pn[i].rh;
    rx = ry = rz = lx = ly = lz = 0.0;
    cllh = 1.0/(double)ndis->pn[i].lh;
    for(j = llh;j < tlh;j++){
      k = ndis->lrnoe[j];
      lx = lx + ad->x[k].x*cllh;
      ly = ly + ad->x[k].y*cllh;
      lz = lz + ad->x[k].z*cllh;
    }
    /* right hand atoms*/
    clrh = 1.0/(double)ndis->pn[i].rh;
    for(j = lrh;j < trh;j++){
      k = ndis->lrnoe[j];
      rx = rx + ad->x[k].x*clrh;
      ry = ry + ad->x[k].y*clrh;
      rz = rz + ad->x[k].z*clrh;
    }
    xx = yy = zz = 0;
    xx = (lx - rx);
    yy = (ly - ry);
    zz = (lz - rz);
    dist = sqrt(xx*xx + yy*yy + zz*zz);
    d_upp = ((ndis->rst[i].ds) + (ndis->rst[i].dp) - (ndis->sqof));
    d_low = ((ndis->rst[i].ds) - (ndis->rst[i].dm));
    /* check violations */
    if(d_upp - dist > 0) {
      if(d_low - dist > 0) {
	viol_chk = fabs(d_low - dist);
      }  
      if(d_low - dist <= 0) {
	viol_chk = 0.0;
      }
    } else {
      viol_chk = fabs(d_upp - dist);
    }
    if(viol_chk > ndis->viol_noe){
      int iset, jset;

      if (num_viol_noe >= ndis->n_output_viol) {
	for (iset = 0; iset < ndis->n_output_viol; iset++) {
	  if (viol_list[iset].viol < viol_chk) {
	    /* shift */
	    for (jset=ndis->n_output_viol-2; jset >= iset; jset--) {
	      viol_list[jset+1] = viol_list[jset];
	    }
	    break;
	  }
	}
      } else {
	for (iset = 0; iset < num_viol_noe; iset++) {
	  if (viol_list[iset].viol < viol_chk) {
	    for (jset=num_viol_noe-1; jset >= iset; jset--) {
	      viol_list[jset+1] = viol_list[jset];
	    }
	    break;
	  }
	}
      }
	
      if (iset < ndis->n_output_viol) {
	viol_list[iset].n_left_atoms = tlh-llh;
	if (viol_list[iset].n_left_atoms >= 10) {
	  viol_list[iset].n_left_atoms = 0;
	} else {
	  for(j = llh;j < tlh;j++){
	    viol_list[iset].left_atoms[j-llh] = (ndis->lrnoe[j])+1;
	  }
	}
	viol_list[iset].n_right_atoms = trh-lrh;
	if (viol_list[iset].n_right_atoms >= 10) {
	  viol_list[iset].n_right_atoms = 0;
	} else {
	  for(j = lrh;j < trh;j++){
	    viol_list[iset].right_atoms[j-lrh] = (ndis->lrnoe[j])+1;
	  }
	}
	viol_list[iset].viol = viol_chk;
	viol_list[iset].dist = dist;
	viol_list[iset].d_upp = d_upp;
	viol_list[iset].d_low = d_low;
      }
      num_viol_noe++;
    }
    rmsd_viol += viol_chk*viol_chk;
  }
  if (ndis->nnoedis > 0) {
    rmsd_viol = sqrt(rmsd_viol / ndis->nnoedis);
  }

  lprintf("\n");
  lprintf("NMR distance restraint:\n");
  lprintf("  File: %s\n", ndis->fname);
  lprintf("  Number of restraints: %d\n", ndis->nnoedis);
  lprintf("  Number of violations (> %.2f A): %d\n", ndis->viol_noe, num_viol_noe);
  lprintf("  RMSD of violations : %lf Angstrom\n", rmsd_viol);
  if (num_viol_noe > 0) {
    int max, n, nn;

    if (ndis->n_output_viol > num_viol_noe) {
      max = num_viol_noe;
    } else {
      max = ndis->n_output_viol;
    }
    lprintf("  (atoms)-(atoms)                          dist   upper   lower    viol\n");
    for (i=0;i<max;i++) {
      lprintf("  ("); nn = 3;
      for (j=0;j<viol_list[i].n_left_atoms;j++) {
	if (j>0) {
	  lprintf(","); nn++;
	}
	lprintf("%d%n",viol_list[i].left_atoms[j],&n);
	nn += n;
      }
      lprintf(")-("); nn += 3;
      for (j=0;j<viol_list[i].n_right_atoms;j++) {
	if (j>0) {
	  lprintf(","); nn++;
	}
	lprintf("%d%n",viol_list[i].right_atoms[j], &n);
	nn += n;
      }      
      lprintf(")"); nn += 1;
      for (n=nn; n<40;n++) {
	lprintf(" ");
      }
      lprintf("%7.2f %7.2f %7.2f %7.2f\n",
	      viol_list[i].dist,   viol_list[i].d_upp,
	      viol_list[i].d_low,  viol_list[i].viol);
    }
  }
}


void EP_NMR_DIH_setup(EXTRA_POT *ep,
		      ATOM_DATA *ad,
		      LINKED_CELL *lc,
		      char *file_name, double scal, double scal1,
		      double viol_dih, int n_output_viol,
		      int gradual_change_step,
                      int virial_flag)
{
  EP_NMRDIH *ndih;
  FILE *fp;
  char buf[100];
  int i; 
  int dh1, dh2, dh3, dh4;
  int ccnt, chck;
  char *fname = "NMR_DIH_setup";

 ndih = emalloc(fname,sizeof(EP_NMRDIH));
 EP_add_item(ep, (EP_ITEM*)ndih);
 ndih->type = EP_T_NMR_DIH;
 strcpy(ndih->fname, file_name);
 ndih->virial_flag = virial_flag;
 ndih->gradual_change_step = gradual_change_step;
 ndih->scl_dih = ndih->scl_dih0 = scal;
 ndih->scl_dih1 = scal1;
 ndih->viol_dih = viol_dih * M_PI/180.0;
 ndih->n_output_viol = n_output_viol;

#ifdef MPI
  if((fp = par_fopen(file_name, "r")) == NULL) {
#else
  if((fp = fopen(file_name, "r")) == NULL) {
#endif
   lprintf("ERROR: %s: No such file\n",file_name);
   marble_exit(1);
 }

#ifdef MPI
  par_fgets(buf,100,fp);
#else
  fgets(buf,100,fp);
#endif

  if (sscanf(buf,"# dihedral_restraint ver. %d.%d",
	     &ndih->major_version,
	     &ndih->minor_version) != 2) {
    lprintf("ERROR: Invalid distance restraint format in %s\n", file_name);
    marble_exit(1);
  }

  if (ndih->major_version < 1 ||
      ndih->minor_version < 0) {
    lprintf("ERROR: Unsupported version of the file (%d.%d)\n",
	    ndih->major_version, ndih->minor_version);
    marble_exit(1);
  }

#ifdef MPI
  par_fgets(buf,100,fp);
#else
  fgets(buf,100,fp);
#endif
  if (sscanf(buf,"%d", &(ndih->nnoedih)) != 1) {
    lprintf("ERROR: Invalid file format in %s.\n",  file_name);
    marble_exit(1);
  }

 lprintf("NMR NOE DIHEDRAL ANGLE RESTRAINT\n");
 lprintf("  REFERENCE RESTRAINT FILE: %s (Ver. %d.%d)\n", ndih->fname,
	 ndih->major_version, ndih->minor_version);
 lprintf("  RESTRAINT PARAMETER\n");
 lprintf("    NUMBER OF DIHEDRAL RESTRAINT %d\n",ndih->nnoedih);
 lprintf("    ENERGY SCALE FACTOR C: %f\n",ndih->scl_dih);
 lprintf("    VIOLATION THRESHOLD %.2f [degree]\n",viol_dih);
  if(ndih->gradual_change_step){
   lprintf("  GRADUAL CHANGE OF FORCE CONSTANT %f -> %f in %d steps\n",
	   ndih->scl_dih, ndih->scl_dih1, ndih->gradual_change_step);
 }

 ndih->pks = emalloc(fname,sizeof(int) * 4 * ndih->nnoedih);
 ndih->f_cons = emalloc(fname,sizeof(double) * ndih->nnoedih);
 ndih->eqbl = emalloc(fname,sizeof(double) * ndih->nnoedih);
 ndih->rng = emalloc(fname,sizeof(double) * ndih->nnoedih);
 ndih->expo = emalloc(fname,sizeof(double) * ndih->nnoedih);
 ndih->numdi = emalloc(fname,sizeof(int) * ndih->nnoedih);
 ccnt = 0;
#ifdef MPI
 {
   int err = 0;
   if (mpi.master) {
     for(i = 0;i < ndih->nnoedih;i++){
       chck = 0;
       ndih->numdi[i] = ccnt;
       chck += fscanf(fp,"%d", &(ndih->pks[ccnt]));
       chck += fscanf(fp,"%d", &(ndih->pks[ccnt+1]));
       chck += fscanf(fp,"%d", &(ndih->pks[ccnt+2]));
       chck += fscanf(fp,"%d", &(ndih->pks[ccnt+3]));
       ccnt += 4;
       chck += fscanf(fp,"%lf", &(ndih->f_cons[i]));
       chck += fscanf(fp,"%lf", &(ndih->eqbl[i]));
       chck += fscanf(fp,"%lf", &(ndih->rng[i]));
       chck += fscanf(fp,"%lf", &(ndih->expo[i]));
       if(chck != 8) {
	 lprintf("ERROR: Invalid dihedral restraint number in list file %s\n",
		 file_name);
	 err = 1;
       }
     }
   }
   MPI_Bcast(&err, 1, MPI_INT, mpi.master_pe, mpi.comm);
   if (err)
     marble_exit(1);
   MPI_Bcast(ndih->pks, 4*ndih->nnoedih, MPI_INT, mpi.master_pe, mpi.comm);
   MPI_Bcast(ndih->f_cons, ndih->nnoedih, MPI_DOUBLE, mpi.master_pe, mpi.comm);
   MPI_Bcast(ndih->eqbl,   ndih->nnoedih, MPI_DOUBLE, mpi.master_pe, mpi.comm);
   MPI_Bcast(ndih->rng,    ndih->nnoedih, MPI_DOUBLE, mpi.master_pe, mpi.comm);
   MPI_Bcast(ndih->expo,   ndih->nnoedih, MPI_DOUBLE, mpi.master_pe, mpi.comm);
   MPI_Bcast(ndih->numdi,  ndih->nnoedih, MPI_DOUBLE, mpi.master_pe, mpi.comm);
 }
#else
 for(i = 0;i < ndih->nnoedih;i++){
   chck = 0;
   ndih->numdi[i] = ccnt;
     chck += fscanf(fp,"%d", &(ndih->pks[ccnt]));
     chck += fscanf(fp,"%d", &(ndih->pks[ccnt+1]));
     chck += fscanf(fp,"%d", &(ndih->pks[ccnt+2]));
     chck += fscanf(fp,"%d", &(ndih->pks[ccnt+3]));
     ccnt += 4;
     chck += fscanf(fp,"%lf", &(ndih->f_cons[i]));
     chck += fscanf(fp,"%lf", &(ndih->eqbl[i]));
     chck += fscanf(fp,"%lf", &(ndih->rng[i]));
     chck += fscanf(fp,"%lf", &(ndih->expo[i]));
     if(chck != 8) {
       lprintf("ERROR: Invalid dihedral restraint number in list file %s\n",
	       file_name);
       marble_exit(1);
     }
 }
#endif

#ifdef MPI
 par_fclose(fp);
#else
 fclose(fp);
#endif

 lprintf("\n");
 for(i=0;i<4*ndih->nnoedih;i++){
   ndih->pks[i] -= 1;
 }
#ifdef MPI_SDMD
 SDMD_setup_ex_tr_atom(lc, 4*ndih->nnoedih, ndih->pks);
#endif
}

void EP_NMR_DIH_energy_force(EP_NMRDIH *ndih, ATOM_DATA *ad, LINKED_CELL *lc)
{
  int i, ii;
  int atom1, atom2, atom3, atom4;
  double phi, phieq, diff, lo, hi, well, range, viol_chk, ee;
  double x12,y12,z12,x23,y23,z23,x34,y34,z34,nx23,ny23,nz23,len_23;
  double dx,dy,dz,gx,gy,gz,dd,gg,dg,len_dg;
  double ddx,ddy,ddz,dgx,dgy,dgz;
  double F, Fx1,Fy1,Fz1,Fx4,Fy4,Fz4,Fx2_d,Fy2_d,Fz2_d,Fx2_g,Fy2_g,Fz2_g;
  
#ifdef MPI_SDMD
  int pe;

  pe = ATOM_CPU(lc, ad, ndih->pks[0]);
  if (pe != mpi.rank) {
    ndih->energy = 0.0;
    return;
  }
#endif

  ndih->num_viol_dih = 0;
  ndih->energy = 0.0;
  for(i = 0;i< (ndih->nnoedih);i++){
    viol_chk=0.0;
    /* calculate the dihedral angle */
    ii=ndih->numdi[i];
    atom1 = (ndih->pks[ii]);
    atom2 = (ndih->pks[ii+1]);
    atom3 = (ndih->pks[ii+2]);
    atom4 = (ndih->pks[ii+3]);

    /* dihedral angle */
    x12 = ad->x[atom2].x - ad->x[atom1].x;
    y12 = ad->x[atom2].y - ad->x[atom1].y;
    z12 = ad->x[atom2].z - ad->x[atom1].z;
    x23 = ad->x[atom3].x - ad->x[atom2].x;
    y23 = ad->x[atom3].y - ad->x[atom2].y;
    z23 = ad->x[atom3].z - ad->x[atom2].z;
    x34 = ad->x[atom4].x - ad->x[atom3].x;
    y34 = ad->x[atom4].y - ad->x[atom3].y;
    z34 = ad->x[atom4].z - ad->x[atom3].z;
    dx  = y12*z23 - z12*y23;
    dy  = z12*x23 - x12*z23;
    dz  = x12*y23 - y12*x23;
    gx  = y23*z34 - z23*y34;
    gy  = z23*x34 - x23*z34;
    gz  = x23*y34 - y23*x34;

    len_23 = sqrt(x23*x23 + y23*y23 + z23*z23);
    nx23 = x23 / len_23;
    ny23 = y23 / len_23;
    nz23 = z23 / len_23;
    dd  = dx*dx + dy*dy + dz*dz;
    gg  = gx*gx + gy*gy + gz*gz;
    dg  = dx*gx + dy*gy + dz*gz;
    len_dg = sqrt(dd) * sqrt(gg);

    phi = atan2((nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)
		 +nz23*(dx*gy-dy*gx))/len_dg,  dg/len_dg);
    /*
      E = const * (Well(phi-phieq,range)**expo)
      Well(a,b) = a-b (a>b), 0 (-b<a<b), a+b (a<-b)
    */
    phieq = ndih->eqbl[i]*(M_PI/180.0);

    if (phi - phieq < -M_PI) phi += 2.0*M_PI;
    else if (phi - phieq > M_PI) phi -= 2.0*M_PI;

    diff  = phi - phieq;
    range = ndih->rng[i]*M_PI/180.0;

    lo = diff - range;
    hi = diff + range;
    /* calculate the difference between phi and phi0 */
    well = (0.0 > lo ? 0.0:lo) + (0.0 < hi ? 0.0:hi);
    /* calculate energy and first derivative */
    F = ndih->scl_dih * ndih->f_cons[i] * pow(well,ndih->expo[i]-1.0);
    ee = F*well;
    /* if well == 0, F = 0.0 */
    F *= - ndih->expo[i];
    /* accumulate energy */
    ndih->energy += ee;
    /* check the violations */
    viol_chk = fabs(well);
    if(viol_chk > ndih-> viol_dih) {
      ndih->num_viol_dih += 1;
    }

    /* debug
    if (viol_chk > 0.0) {
      lprintf("phi=%f, diff=%f,well=%f\n",
	      phi*180.0/M_PI,diff*180.0/M_PI,
	      well*180.0/M_PI);
    } 
    */
    
    /* force */  
    ddx = (dy*nz23 - dz*ny23)/dd;
    ddy = (dz*nx23 - dx*nz23)/dd;
    ddz = (dx*ny23 - dy*nx23)/dd;
    
    dgx = -(gy*nz23 - gz*ny23)/gg;
    dgy = -(gz*nx23 - gx*nz23)/gg;
    dgz = -(gx*ny23 - gy*nx23)/gg;

    Fx1 = F*(ddy * z23 - ddz * y23);
    Fy1 = F*(ddz * x23 - ddx * z23);
    Fz1 = F*(ddx * y23 - ddy * x23);
    Fx4 = F*(dgy * z23 - dgz * y23);
    Fy4 = F*(dgz * x23 - dgx * z23);
    Fz4 = F*(dgx * y23 - dgy * x23);
    Fx2_d = F*(ddy * z12 - ddz * y12);
    Fy2_d = F*(ddz * x12 - ddx * z12);
    Fz2_d = F*(ddx * y12 - ddy * x12);
    Fx2_g = F*(dgy * z34 - dgz * y34);
    Fy2_g = F*(dgz * x34 - dgx * z34);
    Fz2_g = F*(dgx * y34 - dgy * x34);
  
    ad->f[atom1].x += Fx1;
    ad->f[atom1].y += Fy1;
    ad->f[atom1].z += Fz1;
    ad->f[atom2].x += -Fx2_d + Fx2_g - Fx1;
    ad->f[atom2].y += -Fy2_d + Fy2_g - Fy1;
    ad->f[atom2].z += -Fz2_d + Fz2_g - Fz1;
    ad->f[atom3].x +=  Fx2_d - Fx2_g - Fx4;
    ad->f[atom3].y +=  Fy2_d - Fy2_g - Fy4;
    ad->f[atom3].z +=  Fz2_d - Fz2_g - Fz4;
    ad->f[atom4].x += Fx4;
    ad->f[atom4].y += Fy4;
    ad->f[atom4].z += Fz4;

    if (ndih->virial_flag) {
      ad->virial[0] += ( Fx1                   * ad->x[atom1].x + 
		        (-Fx2_d + Fx2_g - Fx1) * ad->x[atom2].x +
			( Fx2_d - Fx2_g - Fx4) * ad->x[atom3].x +
	                 Fx4                   * ad->x[atom4].x );
      ad->virial[1] += ( Fy1                   * ad->x[atom1].y + 
	                (-Fy2_d + Fy2_g - Fy1) * ad->x[atom2].y +
	                ( Fy2_d - Fy2_g - Fy4) * ad->x[atom3].y +
	                 Fy4                   * ad->x[atom4].y );
      ad->virial[2] += ( Fz1                   * ad->x[atom1].z + 
	                (-Fz2_d + Fz2_g - Fz1) * ad->x[atom2].z +
	                ( Fz2_d - Fz2_g - Fz4) * ad->x[atom3].z +
  	                 Fz4                   * ad->x[atom4].z );
      ad->virial[3] += ( Fx1                   * ad->x[atom1].y +
			(-Fx2_d + Fx2_g - Fx1) * ad->x[atom2].y +
			( Fx2_d - Fx2_g - Fx4) * ad->x[atom3].y +
			 Fx4                   * ad->x[atom4].y );
      ad->virial[4] += ( Fx1                   * ad->x[atom1].z +
			(-Fx2_d + Fx2_g - Fx1) * ad->x[atom2].z +
			( Fx2_d - Fx2_g - Fx4) * ad->x[atom3].z +
			 Fx4                   * ad->x[atom4].z );
      ad->virial[5] += ( Fy1                   * ad->x[atom1].z +
			(-Fy2_d + Fy2_g - Fy1) * ad->x[atom2].z +
			( Fy2_d - Fy2_g - Fy4) * ad->x[atom3].z +
			 Fy4                   * ad->x[atom4].z );
    }
  }
}

/*****/
void EP_NMR_DIH_prop_header(EP_NMRDIH *ndih,
			     FILE *fp, int *id)
{
  fprintf(fp, "%2dN_Viol(dih) ", (*id)++);
  fprintf(fp, "%2dENE(dih)    ", (*id)++);
}
/****/
void EP_NMR_DIH_prop_out(EP_NMRDIH *ndih, MD_SYSTEM *sys, FILE *fp)
{
#ifdef MPI_SDMD
  int pe;
  LINKED_CELL *lc;
  ATOM_DATA *ad;
  MPI_Status stat;
  int buf[1];
  double dbuf[1];

  lc = &sys->linked_cell;
  ad = &sys->atom;
  pe = ATOM_CPU(lc, ad, ndih->pks[0]);
  
  if (mpi.master) {
    if (pe != mpi.rank) {
      MPI_Recv(buf,1,MPI_INT, pe, SDMD_TAG_PROP, mpi.comm, &stat);
      ndih->num_viol_dih   = buf[0];
      MPI_Recv(dbuf,1,MPI_DOUBLE, pe, SDMD_TAG_PROP, mpi.comm, &stat);
      ndih->energy   = dbuf[0];
    }
  } else {
    if (pe == mpi.rank) {
      buf[0] = ndih->num_viol_dih;
      MPI_Send(buf,1,MPI_INT, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
      dbuf[0] = ndih->energy;
      MPI_Send(dbuf,1,MPI_DOUBLE, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
    }
    return;
  }
#endif /* SDMD*/
  
  fprintf(fp, "  %8d    ", ndih->num_viol_dih);
  fprintf(fp, " %13.6e",   ndih->energy);
}
/****/


void EP_NMR_DIH_gradual_change(EP_NMRDIH *ndih, int step)
{
  double lambda;
  if(ndih->gradual_change_step) {
    if(step < ndih->gradual_change_step) {
      lambda = (double)step/ndih->gradual_change_step;
      ndih->scl_dih = ndih->scl_dih0*(1.0-lambda) + ndih->scl_dih1*lambda;
    } else {
      ndih->scl_dih = ndih->scl_dih1;
    }
  }
}

void EP_NMR_DIH_output(EP_NMRDIH *ndih, MD_SYSTEM *sys)
{
  ATOM_DATA *ad;
  int i, ii;
  int atom1, atom2, atom3, atom4;
  int num_viol_dih;
  double rmsd_viol;
  double phi, phieq, diff, lo, hi, well, range, viol_chk, ee;
  double x12,y12,z12,x23,y23,z23,x34,y34,z34,nx23,ny23,nz23,len_23;
  double dx,dy,dz,gx,gy,gz,dd,gg,dg,len_dg;
  double ddx,ddy,ddz,dgx,dgy,dgz;
  struct {
    double viol, phi, phi_eq, range;
    int atom1, atom2, atom3, atom4;
  } *viol_list;

  ad = &sys->atom;

  num_viol_dih = 0;
  rmsd_viol = 0.0;

  viol_list = emalloc("EP_NMR_DIH_output", sizeof(*viol_list)*ndih->n_output_viol);

  for(i = 0;i < ndih->nnoedih;i++){
    viol_chk=0.0;
    /* calculate the dihedral angle */
    ii=ndih->numdi[i];
    atom1 = (ndih->pks[ii]);
    atom2 = (ndih->pks[ii+1]);
    atom3 = (ndih->pks[ii+2]);
    atom4 = (ndih->pks[ii+3]);

    /* dihedral angle */
    x12 = ad->x[atom2].x - ad->x[atom1].x;
    y12 = ad->x[atom2].y - ad->x[atom1].y;
    z12 = ad->x[atom2].z - ad->x[atom1].z;
    x23 = ad->x[atom3].x - ad->x[atom2].x;
    y23 = ad->x[atom3].y - ad->x[atom2].y;
    z23 = ad->x[atom3].z - ad->x[atom2].z;
    x34 = ad->x[atom4].x - ad->x[atom3].x;
    y34 = ad->x[atom4].y - ad->x[atom3].y;
    z34 = ad->x[atom4].z - ad->x[atom3].z;
    dx  = y12*z23 - z12*y23;
    dy  = z12*x23 - x12*z23;
    dz  = x12*y23 - y12*x23;
    gx  = y23*z34 - z23*y34;
    gy  = z23*x34 - x23*z34;
    gz  = x23*y34 - y23*x34;

    len_23 = sqrt(x23*x23 + y23*y23 + z23*z23);
    nx23 = x23 / len_23;
    ny23 = y23 / len_23;
    nz23 = z23 / len_23;
    dd  = dx*dx + dy*dy + dz*dz;
    gg  = gx*gx + gy*gy + gz*gz;
    dg  = dx*gx + dy*gy + dz*gz;
    len_dg = sqrt(dd) * sqrt(gg);

    phi = atan2((nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)
		 +nz23*(dx*gy-dy*gx))/len_dg,  dg/len_dg);
    /*
      E = const * (Well(phi-phieq,range)**expo)
      Well(a,b) = a-b (a>b), 0 (-b<a<b), a+b (a<-b)
    */
    phieq = ndih->eqbl[i]*(M_PI/180.0);

    if (phi - phieq < -M_PI) phi += 2.0*M_PI;
    else if (phi - phieq > M_PI) phi -= 2.0*M_PI;

    diff  = phi - phieq;
    range = ndih->rng[i]*M_PI/180.0;

    lo = diff - range;
    hi = diff + range;
    /* calculate the difference between phi and phi0 */
    well = (0.0 > lo ? 0.0:lo) + (0.0 < hi ? 0.0:hi);
    /* calculate energy and first derivative */
    /* check the violations */
    viol_chk = fabs(well);

    /*
    lprintf("%f %f %f\n", diff*180.0/M_PI, range*180.0/M_PI, well*180.0/M_PI);
    */

    if(viol_chk > ndih-> viol_dih) {
      int iset, jset;

      if (num_viol_dih >= ndih->n_output_viol) {
	for (iset = 0; iset < ndih->n_output_viol; iset++) {
	  if (viol_list[iset].viol < viol_chk) {
	    /* shift */
	    for (jset=ndih->n_output_viol-2; jset >= iset; jset--) {
	      viol_list[jset+1] = viol_list[jset];
	    }
	    break;
	  }
	}
      } else {
	for (iset = 0; iset < num_viol_dih; iset++) {
	  if (viol_list[iset].viol < viol_chk) {
	    for (jset=num_viol_dih-1; jset >= iset; jset--) {
	      viol_list[jset+1] = viol_list[jset];
	    }
	    break;
	  }
	}
      }
      if (iset < ndih->n_output_viol) {
	viol_list[iset].atom1 = atom1+1;
	viol_list[iset].atom2 = atom2+1;
	viol_list[iset].atom3 = atom3+1;
	viol_list[iset].atom4 = atom4+1;
	viol_list[iset].viol = viol_chk;
	viol_list[iset].phi = phi * 180.0 / M_PI;
	viol_list[iset].phi_eq = phieq * 180.0 / M_PI;
	viol_list[iset].range = range * 180.0 / M_PI;
      }
      num_viol_dih++;
    }
    rmsd_viol += viol_chk*viol_chk;
  }
  if (ndih->nnoedih > 0) {
    rmsd_viol = sqrt(rmsd_viol / ndih->nnoedih);
  }

  lprintf("\n");
  lprintf("NMR dihedral restraint:\n");
  lprintf("  File: %s\n", ndih->fname);
  lprintf("  Number of restraints: %d\n", ndih->nnoedih);
  lprintf("  Number of violations (> %.2f degree): %d\n", ndih->viol_dih*180.0/M_PI, num_viol_dih);
  lprintf("  RMSD of violations : %lf degree\n", rmsd_viol*180.0/M_PI);
  if (num_viol_dih > 0) {
    int max;
    if (ndih->n_output_viol > num_viol_dih) {
      max = num_viol_dih;
    } else {
      max = ndih->n_output_viol;
    }
    lprintf("  atom1 atom2 atom3 atom4     phi  phi_eq    diff   range    viol\n");
    for (i=0;i<max;i++) {
    lprintf("  %5d %5d %5d %5d %7.2f %7.2f %7.2f %7.2f %7.2f\n",
	    viol_list[i].atom1,  viol_list[i].atom2,
	    viol_list[i].atom3,  viol_list[i].atom4, 
	    viol_list[i].phi,    viol_list[i].phi_eq,
	    viol_list[i].phi-viol_list[i].phi_eq,
	    viol_list[i].range,  viol_list[i].viol*180.0/M_PI);
    }
  }
}


void EP_NMR_PLN_setup(EXTRA_POT *ep,
		      ATOM_DATA *ad,
		      LINKED_CELL *lc,
		      char *file_name,
		      double scal, double scal1, 
		      int gradual_change_step, 
		      int virial_flag)
{
  EP_NMRPLN *npln;
  FILE *fp;
  char buf[100];
  int h, i, j, k;
  int count, stl, str;
  double nnpln;
  int sss, ppp, s;
  char *fname = "NMR_PLN_setup";
  int t_num;


  npln = emalloc(fname, sizeof(EP_NMRPLN));
  EP_add_item(ep, (EP_ITEM*)npln);
  npln->type = EP_T_NMR_PLN;
  strcpy(npln->fname, file_name);
  npln->virial_flag = virial_flag;
  npln->gradual_change_step = gradual_change_step;
  npln->scl_pln  = npln->scl_pln0 = scal;
  npln->scl_pln1 = scal1;

  if((fp = fopen(file_name, "r")) == NULL) {
    lprintf("ERROR: %s: No such file\n", file_name);
    marble_exit(1);
  }
  fgets(buf,100,fp);
  if (sscanf(buf,"# plane_restraint ver. %d.%d",
	     &npln->major_version,
	     &npln->minor_version) != 2) {
    lprintf("ERROR: Invalid distance restraint format in %s\n", file_name);
    marble_exit(1);
  }

  if (npln->major_version < 1 ||
      npln->minor_version < 0) {
    lprintf("ERROR: Unsupported version of the file (%d.%d)\n",
	    npln->major_version, npln->minor_version);
    marble_exit(1);
  }

  if (fscanf(fp,"%d%d", &(npln->nnpln),&(npln->n_total_atom))!=2) {
    lprintf("ERROR: Invalid file format in %s\n",  file_name);
    marble_exit(1);
  }
  lprintf("PLANERITY RESTRAINT (SOFT-SQUARE)\n");
  lprintf("  REFERENCE RESTRAINT FILE: %s\n", npln->fname);
  lprintf("  RESTRAINT PARAMETER\n");
  lprintf("    NUMBER OF RESTRAINT: %d\n",npln->nnpln);
  lprintf("    SCALE : %f\n", npln->scl_pln);
  lprintf("  VIRIAL FLAG: %s\n", (npln->virial_flag) ? "ON" : "OFF");
  if(npln->gradual_change_step) {
    lprintf("  GRADUAL CHANGE OF FORCE CONSTANT: %f -> %f in %d steps\n",
	    npln->scl_pln0, npln->scl_pln1, npln->gradual_change_step);
  }

  npln->pln     = emalloc(fname,sizeof(struct _s_NMR_PLN) * npln->nnpln);
  npln->pl_atom = emalloc(fname,sizeof(int) * npln->n_total_atom);
  ppp = 0;
  for(i = 0, s = 0;i < npln->nnpln;i++){ 
    npln->pln[i].start_atom = s;
    if(fscanf(fp, "%d", &(npln->pln[i].n_atom)) != 1) {
      lprintf("ERROR: Invalid file format (No.2) in %s\n",  file_name);
      marble_exit(1);
    }
    for (j=0;j<npln->pln[i].n_atom;j++) {
      if(fscanf(fp, "%d", &(npln->pl_atom[s++])) != 1) {
	lprintf("ERROR: Invalid file format (No.3) in %s\n",  file_name);
	marble_exit(1);
      }
    }
    if(fscanf(fp, "%lf", &(npln->pln[i].weight)) != 1) {
      lprintf("ERROR: Invalid file format (No.4) in %s\n",  file_name);
      marble_exit(1);
    }
  } 
  fclose(fp);
  lprintf("\n");

  /* debug
  for (i=0;i<npln->nnpln;i++) {
    lprintf("%d", npln->pln[i].n_atom);
    for (j=0;j<npln->pln[i].n_atom;j++) {
      lprintf(" %3d", npln->pl_atom[npln->pln[i].start_atom+j]);
    }
    lprintf(" %f\n",npln->pln[i].weight);
  }
  */

  for(j = 0; j < npln->n_total_atom;j++){
    npln->pl_atom[j] -= 1;
  }
#ifdef MPI_SDMD
  SDMD_setup_ex_tr_atom(lc, npln->n_total_atom, npln->pl_atom);
#endif
}

void EP_NMR_PLN_energy_force(EP_NMRPLN *npln,ATOM_DATA *ad, LINKED_CELL *lc)
{
  int h,i,j,k;
  unsigned int gid;
  double xmass, ymass, zmass;
  double xdif, ydif, zdif;
  double xdif2, ydif2, zdif2;
  double T[9], w[3], work[9];
  double eigv[3][3];
  double const0;
  double dx, dy, dz;
  double dist, ep;
  double all_weight;
  char jobz='V', uplo='U';
  long int n=3, lda=3;
  long int lwork=9, info;
  int s, mineig;
  int itype;


#ifdef MPI_SDMD
  int pe;

  pe = ATOM_CPU(lc, ad, npln->pl_atom[0]);
  if (pe != mpi.rank) {
    npln->energy = 0.0;
    return;
  }
#endif

  npln->energy = 0.0;

  for(i=0;i < npln->nnpln;i++){
    s = npln->pln[i].start_atom;
    /* Calculate geometric center of mass */
    xmass = ymass = zmass = 0.0; 

    for(j=0;j < npln->pln[i].n_atom;j++){ 

      /*        printf("x = %lf y = %lf z = %lf\n",
                ad->x[npln->pl_atom[s+j]].x, 
                ad->x[npln->pl_atom[s+j]].y, 
                ad->x[npln->pl_atom[s+j]].z); */

      xmass += ad->x[npln->pl_atom[s+j]].x; 
      ymass += ad->x[npln->pl_atom[s+j]].y; 
      zmass += ad->x[npln->pl_atom[s+j]].z; 
    }
    xmass /= npln->pln[i].n_atom;
    ymass /= npln->pln[i].n_atom;
    zmass /= npln->pln[i].n_atom;
    /*        printf("xmass = %lf ymass = %lf zmass = %lf\n",
	      xmass, ymass, zmass); */

    /* Set up the moment of inertia tensor */

    T[0] = 0; 
    T[1] = 999; 
    T[2] = 999; 
    T[3] = 0; 
    T[4] = 0; 
    T[5] = 999; 
    T[6] = 0; 
    T[7] = 0; 
    T[8] = 0; 
 
    for(k=0;k < npln->pln[i].n_atom;k++){
      xdif = ydif = zdif = 0.0;
      xdif2 = ydif2 = zdif2 = 0.0;
      xdif = ad->x[npln->pl_atom[s+k]].x - xmass; 
      ydif = ad->x[npln->pl_atom[s+k]].y - ymass; 
      zdif = ad->x[npln->pl_atom[s+k]].z - zmass;
      /*        printf("xdif = %lf ydif = %lf zdif = %lf\n",
                xdif, 
                ydif, 
                zdif); */
      xdif2 = xdif*xdif;
      ydif2 = ydif*ydif;
      zdif2 = zdif*zdif;

      /*          printf("%lf %lf %lf\n", xdif,ydif,zdif); */

      T[0] += ad->x[npln->pl_atom[s+k]].x * xdif;
      T[3] += ad->x[npln->pl_atom[s+k]].x * ydif;
      T[4] += ad->x[npln->pl_atom[s+k]].y * ydif;
      T[6] += ad->x[npln->pl_atom[s+k]].x * zdif;
      T[7] += ad->x[npln->pl_atom[s+k]].y * zdif;
      T[8] += ad->x[npln->pl_atom[s+k]].z * zdif;

      /*          T[0] += (xdif2 - ad->x[npln->pl_atom[s+k]].x*
		  ad->x[npln->pl_atom[s+k]].x);
		  T[3] += ad->x[npln->pl_atom[s+k]].x * ydif;
		  T[4] += (ydif2 - ad->x[npln->pl_atom[s+k]].y*
		  ad->x[npln->pl_atom[s+k]].y);
		  T[6] += ad->x[npln->pl_atom[s+k]].x * zdif;
		  T[7] += ad->x[npln->pl_atom[s+k]].y * zdif;
		  T[8] += (zdif2 - ad->x[npln->pl_atom[s+k]].z*
		  ad->x[npln->pl_atom[s+k]].z);*/
    }

    /* printf("%lf %lf %lf\n%lf %lf %lf\n%lf %lf %lf\n",
       T[0],T[1],T[2],T[3],T[4],T[5],T[6],T[7],T[8]);*/
    /* use for lapack routine */
    /*
      dsyev_(&jobz, &uplo, &n, T, &lda, w, work, &lwork, &info);  
    */
    /* use myown routine instead of lapack */
    T[1] = T[3];  T[2] = T[6];  T[5] = T[7];
    diag33((double (*)[3])T, w, eigv);
    sort_eigen(3, w, &eigv[0][0], 0);
    T[0]=eigv[0][0];
    T[1]=eigv[1][0];
    T[2]=eigv[2][0];

    /*printf("info = %d\n",info); */
    /*lprintf("w[0]=%lf w[1]=%lf w[2]=%lf\n", w[0],w[1],w[2]);*/

    all_weight = npln->scl_pln * npln->pln[i].weight;
    ep =  all_weight * w[0]; 
    /*printf("pass03\n");*/
    /*printf("ep = %lf\n",ep);*/
    npln->energy += ep;
    const0  = T[0]*xmass + T[1]*ymass + T[2]*zmass;

    dx=dy=dz=0;
    for(j=0;j<npln->pln[i].n_atom;j++){
      dist = T[0]*(ad->x[npln->pl_atom[s+j]].x) +
	T[1]*(ad->x[npln->pl_atom[s+j]].y) +
	T[2]*(ad->x[npln->pl_atom[s+j]].z) -
	const0; 
    
      dist = dist*2.0 * all_weight;

      dx = -T[0]*dist;
      dy = -T[1]*dist;
      dz = -T[2]*dist;
      ad->f[npln->pl_atom[s+j]].x += dx;
      ad->f[npln->pl_atom[s+j]].y += dy;
      ad->f[npln->pl_atom[s+j]].z += dz;

      if(npln->virial_flag){
	ad->virial[0] +=  dx * ad->x[npln->pl_atom[s+j]].x;
	ad->virial[1] +=  dy * ad->x[npln->pl_atom[s+j]].y;
	ad->virial[2] +=  dz * ad->x[npln->pl_atom[s+j]].z;
	ad->virial[3] += (dx * ad->x[npln->pl_atom[s+j]].y +
			  dy * ad->x[npln->pl_atom[s+j]].x) * 0.5;
	ad->virial[4] += (dx * ad->x[npln->pl_atom[s+j]].z +
			  dz * ad->x[npln->pl_atom[s+j]].x) * 0.5;
	ad->virial[5] += (dy * ad->x[npln->pl_atom[s+j]].z +
			  dz * ad->x[npln->pl_atom[s+j]].y) * 0.5;
      }
    }
  }
}

void EP_NMR_PLN_gradual_change(EP_NMRPLN *npln, int step)
{
  double lambda;
  if (npln->gradual_change_step) {
    if (step < npln->gradual_change_step) {
      lambda = (double)step/npln->gradual_change_step;
      npln->scl_pln = npln->scl_pln0*(1.0-lambda) + npln->scl_pln1*lambda;
    } else {
      npln->scl_pln = npln->scl_pln1;
    }
  }
}

