/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifdef MPI

#define _MPI_SETUP_C_

#include <stdio.h>
#include <stdlib.h>

#include "misc.h"
#include "mpi_global.h"

#ifdef _OPENMP
#include "omp.h"
#endif

void mpi_setup()
{
  MPI_Comm_dup(MPI_COMM_WORLD, &mpi.comm);
  MPI_Comm_dup(MPI_COMM_WORLD, &mpi.all_comm);
  
  MPI_Comm_rank(mpi.comm, &mpi.rank);
  MPI_Comm_size(mpi.comm, &mpi.n_pe);
  
  if (mpi.rank == 0) {
    mpi.master = 1;
  } else {
    mpi.master = 0;
  }
  mpi.master_pe = 0;

#ifdef _OPENMP
  mpi.n_threads = omp_get_max_threads();
#else
  mpi.n_threads = 1;
#endif

}

void mpi_cart3D_setup(int npx[3])
{
  int i;

  if (mpi.n_pe != npx[0]*npx[1]*npx[2]) {
    if (mpi.master)
      printf("ERROR: mpi.n_pe != npx[0]*npx[1]*npx[2]\n");
    marble_exit(1);
  }
  for (i=0;i<3;i++) {
    mpi.npx[i] = npx[i];
  }

  mpi.cart_comm = mpi.comm;
  
  MPI_Comm_rank(mpi.cart_comm, &mpi.cart_rank);

  mpi_rank2xyz(mpi.cart_rank, &mpi.px[0], &mpi.px[1], &mpi.px[2]);

  /*
  mpi.px[2] = mpi.cart_rank / (npx[0]*npx[1]);
  mpi.px[0] = mpi.cart_rank % (npx[0]*npx[1]);
  mpi.px[1] = mpi.px[0] / npx[0];
  mpi.px[0] = mpi.px[0] % npx[0];
  */
}

void mpi_rank2xyz(int rank, int *px, int *py, int *pz)
{
#ifdef KCOMP
  int rank_in_node, nx, ny, nz, ppx, ppy, ppz;

  if (mpi.dim == 3) {
    if (mpi.np_in_node == 1) {
      FJMPI_Topology_rank2xyz(rank, px , py,  pz);
    } else {
      FJMPI_Topology_rank2xyz(rank, &nx , &ny,  &nz);
      rank_in_node = rank % mpi.np_in_node;
    
      ppz = rank_in_node / (mpi.npx_in_node[0] * mpi.npx_in_node[1]);
      ppy = rank_in_node % (mpi.npx_in_node[0] * mpi.npx_in_node[1]);
      ppx = ppy % mpi.npx_in_node[0];
      ppy /= mpi.npx_in_node[0];
      *px = nx * mpi.npx_in_node[0] + ppx;
      *py = ny * mpi.npx_in_node[1] + ppy;
      *pz = nz * mpi.npx_in_node[2] + ppz;
    }
  } else {
    /* x-first */
    *pz = rank / (mpi.npx[0] * mpi.npx[1]);
    *py = rank % (mpi.npx[0] * mpi.npx[1]);
    *px = *py  % mpi.npx[0];
    *py /= mpi.npx[0];
  }
#else  
  /* x-first */
  *pz = rank / (mpi.npx[0] * mpi.npx[1]);
  *py = rank % (mpi.npx[0] * mpi.npx[1]);
  *px = *py  % mpi.npx[0];
  *py /= mpi.npx[0];
#endif
}

void mpi_xyz2rank(int ix, int iy, int iz, int *prank)
{
#ifdef KCOMP
  int nx, ny, nz, inx, iny, inz, iix,iiy, iiz, node;

  if (mpi.dim == 3) {
    if (mpi.np_in_node == 1) {
      FJMPI_Topology_xyz2rank(ix, iy, iz, prank);
    } else {
      /* x-first */
      nx = mpi.npx[0] / mpi.npx_in_node[0];
      ny = mpi.npx[1] / mpi.npx_in_node[1];
      nz = mpi.npx[2] / mpi.npx_in_node[2];
      inx = ix / mpi.npx_in_node[0];
      iny = iy / mpi.npx_in_node[1];
      inz = iz / mpi.npx_in_node[2];
      iix = ix % mpi.npx_in_node[0];
      iiy = iy % mpi.npx_in_node[1];
      iiz = iz % mpi.npx_in_node[2];
      node = inx + nx * (iny + ny * inz);
      *prank = iix + mpi.npx_in_node[0] * (iiy + mpi.npx_in_node[1] * iiz) 
	+ node * mpi.np_in_node;
    }
  } else {
    /* x-first */
    *prank = ix + mpi.npx[0] * (iy + mpi.npx[1] * iz);
  }
#else  
  /* x-first */
  *prank = ix + mpi.npx[0] * (iy + mpi.npx[1] * iz);
#endif
}

#else /* MPI */
void mpi_dummy()
{
}
#endif /* MPI */
