/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifndef _MISC_H_INCLUDED
#define _MISC_H_INCLUDED

#ifdef MPI
#include "mpi_global.h"
#endif

#define Length(x,y,z)  sqrt((x)*(x)+(y)*(y)+(z)*(z))
#define Length2(x,y,z) ((x)*(x)+(y)*(y)+(z)*(z))
#ifndef M_PI
#define M_PI           3.14159265358979323846
#endif
#ifndef M_TWOPI
#define M_TWOPI        (2*M_PI)
#endif
#define SQR(x)         ((x)*(x))

#define DEGTORAD       (M_PI/180.0)
#define RADTODEG       (180.0/M_PI)

#define MOL     6.022045e23
#define ELEC    4.803242e-10
#define K       1.380662e-16           /* erg / K */

/* #define KCAL    1.43860e13           1.43e13 kcal/mol = 1 erg */
#define KCAL    1.4393252151051625e+13

#define ANG     1e-8
#define EKCAL   (ELEC * ELEC * KCAL / ANG)

/*    1 atm = 101325 Pa
      1 kcal/mol = 4.184e3/MOL J
      1 kcal/mol/ANG^3 (our pressure unit) = 4.184e3/MOL*1.0e30 Pa
      1 atm = (101325/(4.184e3/MOL*1.0e30)) kcal/mol/ANG^3  */
/*    0.0000145837   */
#define ATM_TO_KCALMOLANG  (101325/(4.184e3/MOL*1.0e30))

/*    1 dyn/cm = 1.0e-3 N/m
      1 kcal/mol/ANG^2 (our tension unit) = 4.184e3/MOL*1.0e20 N/m
      1 dyn/cm = (1.0e-3/(4.184e3/MOL*1.0e20)) kcal/mol/ANG^2 */
/*    0.0014393032982792  */
#define DYNCM_TO_KCALMOLANG  (1.0e-3/(4.184e3/MOL*1.0e20))

#define EPS     1.0e-10

#define MAX_FILE_NAME_LEN  300

typedef struct _s_vec {
  double x, y, z;
} VEC;

typedef struct _s_ivec {
  int x, y, z;
} IVEC;

typedef struct{
  double r, i;
} MD_COMPLEX;

typedef struct {
  char *name;
  int id;
} SEL_DATA;

#define MUL_COMPLEX(a, b, c)   { (a).r = (b).r * (c).r - (b).i * (c).i;   \
				 (a).i = (b).r * (c).i + (b).i * (c).r; }

/*  log file handing routines */
void lprintf(char *fmt, ...);
void lputc(int c);
void lprintf_bar();
void log_only_printf(char *fmt, ...);
int log_file_open(char *fname);
void log_file_close(void);
void set_log_file_flag(int flag);
void set_log_stdout_flag(int flag);
void lflush(void);
char *get_log_file_name();

/* memory allocation routines */
void malloc_error_exit(char *routine, size_t);
void *emalloc(char *routine, size_t);
void *erealloc(char *routine, void*, size_t);
void *alloc_3d_index(void *buf, size_t size, int n[3]);
void free_3d_index(void *buf, int n[3]);

/* obtaining CPU time */
void date_print(FILE *out);
void time_print(FILE *out);
void timer_init(void);
char *strtime(void);
void convert_times_d(double sec,int *h, int *m, float *s);
void user_system_times(int *user_h, int *user_m, float *user_s,
		       int *sys_h,  int *sys_m,  float *sys_s);
double get_user_time(void);
#ifdef TIME_FORCE
double dtime(void);
void add_dtime(double *);
#else   /* TIME_FORCE */
#define add_dtime(x)
#define dtime()
#endif  /* TIME_FORCE */
/*#define MPI_Wtime() 0*/

/* string utilities */
void padding(char *buf, int pad_len);
int strmatch(char *str, char *expr);
void change_extension(char *src, char *dest, char *ext);

/*** VEC utility routines */
double s_prod3d(int n, VEC *a, VEC *b);
void mat_inv33(double a[3][3], double b[3][3]);
void print_mat33(double mat[3][3]);
void diag33(double m[3][3], double eig[3], double eigv[3][3]);
void jacobi(int n, double *a, double *d, double *u);
void sort_eigen(int n, double *d, double *u, int dir);
double determinant33(double A[3][3]);
void transpose33(double A[3][3],double T[3][3]);  /* T = tA */
void mul_mat33(double A[3][3],double B[3][3], double C[3][3]);
void cross3(double a[3], double b[3], double c[3]);
double dot3(double a[3], double b[3]);
#define VEC_MUL_MAT_X(a,m)  (a.x*m[0][0]+a.y*m[1][0]+a.z*m[2][0])
#define VEC_MUL_MAT_Y(a,m)  (a.x*m[0][1]+a.y*m[1][1]+a.z*m[2][1])
#define VEC_MUL_MAT_Z(a,m)  (a.x*m[0][2]+a.y*m[1][2]+a.z*m[2][2])
#define MAT_MUL_VEC_X(m,a)  (m[0][0]*a.x+m[0][1]*a.y+m[0][2]*a.z)
#define MAT_MUL_VEC_Y(m,a)  (m[1][0]*a.x+m[1][1]*a.y+m[1][2]*a.z)
#define MAT_MUL_VEC_Z(m,a)  (m[2][0]*a.x+m[2][1]*a.y+m[2][2]*a.z)

#define V3_MUL_MAT_X(a,m)  (a[0]*m[0][0]+a[1]*m[1][0]+a[2]*m[2][0])
#define V3_MUL_MAT_Y(a,m)  (a[0]*m[0][1]+a[1]*m[1][1]+a[2]*m[2][1])
#define V3_MUL_MAT_Z(a,m)  (a[0]*m[0][2]+a[1]*m[1][2]+a[2]*m[2][2])


/* file utilities */
FILE *safe_fopen(char *fname, char *mode);

/* mathematics routines */
double gauss_rand(void);
double drand48(void);
double erfc(double);

/* sel_data routines */
int get_id_sel_data(SEL_DATA *sel_data, char *name);
char *get_name_sel_data(SEL_DATA *sel_data, int id);

char *com_fgets(char *buf, int size, FILE *fp);
void fill_space(char *buf, int n);

typedef struct s_str_list {
  char *buf;
  struct s_str_list *next;
} STR_LIST;

void append_str_list(STR_LIST **headp, char *buf);

/* vector calculation */
void v_add(VEC *c, VEC *a, VEC *b);
void v_sub(VEC *c, VEC *a, VEC *b);
void v_mul(VEC *c, double a, VEC *b);
double v_inner_pro(VEC *a, VEC *b);
void v_outer_pro(VEC *c, VEC *a, VEC *b);
void v_norm(VEC *a);
void v_mat_mul(VEC *b, double mat[3][3], VEC *a);
void v_mul_mat(VEC *b, VEC *a, double mat[3][3]);
void v_rot(VEC *b, VEC *axis, double theta, VEC *a);

/* exit */

void marble_exit_abort_flag(int);
void marble_exit(int);
void marble_abort(int);

void plprintf(char *fmt, ...);

typedef struct _s_resizable_int_array {
  int *data;
  int n_alloc;
  int n_unit;
} RESIZABLE_INT_ARRAY;

/********************************************/

typedef struct _s_resizable_vec_array {
  VEC *data;
  int n_alloc;
  int n_unit;
} RESIZABLE_VEC_ARRAY;

void RIA_init(RESIZABLE_INT_ARRAY *arr, int n_unit);
void RIA_alloc(RESIZABLE_INT_ARRAY *arr, int req);
void RIA_free(RESIZABLE_INT_ARRAY *arr, int req);
/*#define RIA_set(arr, id, val) if ((id) >= (arr)->n_alloc) RIA_alloc(arr,(id)+1); (arr)->data[id]=val */
#ifndef RIA_set
void RIA_set(RESIZABLE_INT_ARRAY *arr, int id, int val);
#endif

void RIA_set_all(RESIZABLE_INT_ARRAY *arr, int val);
void RIA_print(RESIZABLE_INT_ARRAY *arr, char *header, int n);
void RVA_init(RESIZABLE_VEC_ARRAY *arr, int n_unit);
void RVA_alloc(RESIZABLE_VEC_ARRAY *arr, int req);
void RVA_free(RESIZABLE_VEC_ARRAY *arr, int req);
void RVA_set(RESIZABLE_VEC_ARRAY *arr, int id, double x, double y, double z);
void RVA_set_all(RESIZABLE_VEC_ARRAY *arr, double x, double y, double z);
void RVA_clear(RESIZABLE_VEC_ARRAY *arr, int n);
void RVA_print(RESIZABLE_VEC_ARRAY *arr, char *header, int n);


#endif  /* _MISC_H_INCLUDED */
