/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#define _INCLUDE_POSIX_SOURCE

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/times.h>
#include <time.h>

#include "misc.h"
void convert_times(clock_t tms,int *h,int *m,float *s);

#define FBUF_SIZE   1000000

/*  log file handing routines */

FILE *_log_fp = NULL;
int _log_stdout_flag = 1;
int _log_file_flag   = 0;
char _log_file_name[MAX_FILE_NAME_LEN];
char *_safe_fopen_fname = NULL;

void lprintf(char *fmt, ...)
{
  va_list args;

  if (_log_stdout_flag) {
    va_start(args, fmt);
    vprintf(fmt, args);
    va_end(args);
  }
  if (_log_file_flag) {
    va_start(args, fmt);
    vfprintf(_log_fp, fmt, args);
    va_end(args);
  }
}

void lputc(int c)
{
  if (_log_stdout_flag)
    putchar(c);
  
  if (_log_file_flag)
    fputc(c, _log_fp);
}

void lprintf_bar()
{
  lprintf("-----------------------------------------------------------------------------\n");
}

void log_only_printf(char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);

  if (_log_file_flag) {
    vfprintf(_log_fp, fmt, args);
  }
  va_end(args);
}

int log_file_open(char *fname)
{
  log_file_close();
  if ((_log_fp = safe_fopen(fname,"a")) == NULL) {
    printf("ERROR: %s: Error in opening log file\n",fname);
    return 1;
  }
  _log_file_flag = 1;
  strcpy(_log_file_name,_safe_fopen_fname);
  return 0;
}

void log_file_close(void)
{
  if (_log_fp)  {
    fclose(_log_fp);
    _log_file_flag = 0;
  }
}
     
void set_log_file_flag(int flag)
{
  _log_file_flag = flag;
}

void set_log_stdout_flag(int flag)
{
  _log_stdout_flag = flag;
}

void lflush(void)
{
  if (_log_stdout_flag) {
    fflush(stdout);
  }
  if (_log_file_flag && _log_fp != NULL) {
    fflush(_log_fp);
  }
}

char *get_log_file_name()
{
  if (_log_file_flag) {
    return _log_file_name;
  } else {
    return NULL;
  }
}

/* memory allocation routines */

void malloc_error_exit(char *routine, size_t size)
{
  lprintf("%s: memory allocation error (size %d)\n", routine, size);
  fprintf(stderr,"%s: memory allocation error (size %d)\n", routine, (int) size);
  exit(1);
}

void *emalloc(char *routine, size_t size)
{
  void *ret;

  if (size == 0) return NULL;

  if ((ret = malloc(size)) == NULL) {
    malloc_error_exit(routine, size);
  }
  return ret;
}

void *erealloc(char *routine, void* dest, size_t size)
{
  void *ret;

  if (dest == NULL) {
    return emalloc(routine, size);
  }
  
  if ((ret = realloc(dest, size)) == NULL) {
    malloc_error_exit(routine, size);
  }
  return ret;
}

void *alloc_2d(size_t size, int n0, int n1)
{
  int i;
  char **cindex, *buf;
  char *f="alloc_2d";
  buf = emalloc(f,size*n0*n1);
  cindex = emalloc("alloc_2d",sizeof(char*)*n0);
  for (i=0;i<n0;i++)
    cindex[i]=&buf[size*n1*i];
  return cindex;
}

void *alloc_3d_index(void *buf, size_t size, int n[3])
{
  int i,j;
  char ***cindex;
  char *cbuf;
  
  cbuf = buf;
  cindex = emalloc("alloc_3d_index", sizeof(char**)*n[0]);
  for (i=0;i<n[0];i++) {
    cindex[i] = emalloc("alloc_3d_index", sizeof(char*)*n[1]);
    for (j=0;j<n[1];j++) {
      cindex[i][j] = &cbuf[size*(n[2]*(j+n[1]*i))];
    }
  }
  return cindex;
}

void free_3d_index(void *buf, int n[3])
{
  char ***cindex;
  int i;
  
  cindex = (char ***) buf;
  
  for (i=0;i<n[0];i++) {
    free(cindex[i]);
  }
  free(cindex);
}

/* obtaining CPU time and date */

void date_print(FILE *out)
{
  fprintf(out,"     Date: %s\n",strtime());
}

void time_print(FILE *out)
{
  struct tms buf;
  int utime_h,utime_m,stime_h,stime_m;
  float utime_s,stime_s;

  times(&buf);
  convert_times(buf.tms_utime,&utime_h,&utime_m,&utime_s);
  convert_times(buf.tms_stime,&stime_h,&stime_m,&stime_s);

  fprintf(out,"***************************************************\n");
  fprintf(out,"     Date: %s\n",strtime());
  fprintf(out,"     User Time:   %3dh %2dm %5.1fs\n",
	utime_h,utime_m,utime_s);
  fprintf(out,"     System Time: %3dh %2dm %5.1fs\n",
	stime_h,stime_m,stime_s);
  fprintf(out,"***************************************************\n\n");
  fflush(out);
}

long _timer_CLK_TCK;

void timer_init()
{
  _timer_CLK_TCK = sysconf(_SC_CLK_TCK);
}

char *strtime()
{
  time_t t;
  char *p,*bp;
  t = time(0);
  bp = p = ctime(&t);
  while (*p != '\0') {
    if (*p == '\n')
      *p = '\0';
    p++;
  }
  return bp;
}
  
void convert_times(clock_t tms,int *h,int *m,float *s)
{
  *s = (float )tms / _timer_CLK_TCK;
  *m = (int)*s / 60;
  *s -= *m * 60;
  *h = *m / 60;
  *m -= *h * 60;
}

void convert_times_d(double sec,int *h, int *m, float *s)
{
  *m = (int)sec / 60;
  *s = sec - *m * 60;
  *h = *m / 60;
  *m -= *h * 60;
}


void user_system_times(int *user_h, int *user_m, float *user_s,
		       int *sys_h,  int *sys_m,  float *sys_s)
{
  struct tms buf;
  
  times(&buf);
  convert_times(buf.tms_utime, user_h, user_m, user_s);
  convert_times(buf.tms_stime, sys_h,  sys_m,  sys_s);
}

double get_user_time()
{
#ifdef MPI
  return MPI_Wtime();
#else
  struct timeval tv;
  gettimeofday(&tv, NULL);
  return tv.tv_sec + (double) tv.tv_usec * 1.0e-6;
#endif
}

#ifdef TIME_FORCE
double dtime()
{
  static double previous = 0.0;
  double current, delta;

  current = get_user_time();
  delta = current - previous;
  previous = current;
  return delta;
}

void add_dtime(double *added_time)
{
  *added_time += dtime();
}
#endif

/* string utilities */
void padding(char *buf, int pad_len)
{
  int i,len;

  len=strlen(buf);
  for (i=len;i<pad_len;i++)
    buf[i] = ' ';
  buf[pad_len] = '\0';
}

int strmatch(char *str, char *expr)
{
  char *ps, *pe;

  ps = str;
  pe = expr;
  while (*ps == ' ') ps++;
  
  while (*ps || *pe) {
    if (*pe == '*') {
      pe++;
      while (*ps) {
	if (*ps == *pe) break;
	ps++;
      }
    }
    if (*ps != *pe) break;
    if (*ps) ps++;
    if (*pe) pe++;
  }
  while (*ps == ' ') ps++;
  while (*pe == ' ') pe++;

  if (*ps == '\0' && *pe == '\0') return 0;

  return 1;
}

void change_extension(char *src, char *dest, char *ext)
{
  int i, n;
  strcpy(dest, src);
  n = strlen(dest);
  for (i=n-1;i>=0;i--) {
    if (dest[i] == '.') {
      break;
    }
  }
  i++;
  strcpy(&dest[i], ext);
}

/*** VEC utility routines */

double s_prod3d(int n, VEC *a, VEC *b)
{
  double tmp = 0.0;
  int i;
  for (i=0;i<n;i++) {
    tmp += a[i].x * b[i].x + a[i].y * b[i].y + a[i].z * b[i].z;
  }
  return tmp;
}

void mat_inv33(double a[3][3], double b[3][3])
{
  int i, j;
  double det;
  
  b[0][0] = a[1][1]*a[2][2]-a[1][2]*a[2][1];
  b[0][1] = a[2][1]*a[0][2]-a[2][2]*a[0][1];
  b[0][2] = a[0][1]*a[1][2]-a[0][2]*a[1][1];
        
  b[1][0] = a[1][2]*a[2][0]-a[1][0]*a[2][2];
  b[1][1] = a[2][2]*a[0][0]-a[2][0]*a[0][2];
  b[1][2] = a[0][2]*a[1][0]-a[0][0]*a[1][2];
        
  b[2][0] = a[1][0]*a[2][1]-a[1][1]*a[2][0];
  b[2][1] = a[2][0]*a[0][1]-a[2][1]*a[0][0];
  b[2][2] = a[0][0]*a[1][1]-a[0][1]*a[1][0];

  det = 0.0;
  for (i=0;i<3;i++) 
    det += a[0][i]*b[i][0];

  /*
  if (fabs(det) < 1.0e-12) {
    return -1;
  }
  */
    
  for (i=0;i<3;i++)
    for (j=0;j<3;j++)
      b[i][j] /= det;

}

void print_mat33(double mat[3][3])
{
  int k,l;
  for (k=0;k<3;k++) {
    for (l=0;l<3;l++)
      lprintf("%e ",mat[k][l]);
    lprintf("\n");
  }
  lprintf("\n");
}

void diag33(double m[3][3], double eig[3], double eigv[3][3])
{
  double temp[3][3];
  int i,j;
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      temp[i][j]=m[i][j];
    }
  }
  jacobi(3, &temp[0][0], eig, &eigv[0][0]);

  /*
  { double tmp[3][3], tmp2[3][3]; int k;
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      tmp[i][j]=eig[i]*eigv[j][i];
    }
  }
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      tmp2[i][j]=0.0;
      for (k=0;k<3;k++) {
	tmp2[i][j]+=eigv[i][k]*tmp[k][j];
      }
    }
  }
  print_mat33(tmp2);
  print_mat33(temp);
  print_mat33(eigv);
  }*/
}


#define AA(i,j)   a[i*n+j]
#define UU(i,j)   u[i*n+j]
#define DD(i)     d[i]
void jacobi(int n, double *a, double *d, double *u)
{
  int i,j,k,count;
  double sum,th,theta,t,s,c,h,g,tau,aki,akj;
  double fabsii,fabsij,fabsjj;

  for (i=0;i<n;i++) {
    for (j=0;j<n;j++) {
      UU(i,j)=0.0;
    }
    UU(i,i)=1.0;
    DD(i)=AA(i,i);
  }

  for (count=1;count<=50;count++) {
    sum = 0.0;
    for (i=0;i<n-1;i++) {
      for(j=i+1;j<n;j++) {
	sum += fabs(AA(i,j));
      }
    }
    if (sum == 0.0) {
      return;
    }
    if (count < 4)
      th = 0.2*sum/(n*n);
    else
      th = 0.0;
    for (i=0;i<n-1;i++) {
      for(j=i+1;j<n;j++) {
	fabsij=fabs(AA(i,j));
	fabsii=fabs(DD(i));
	fabsjj=fabs(DD(j));
	g=100.0*fabsij;
	if (count > 4 && fabsii+g == fabsii
	              && fabsjj+g == fabsjj) {
	  AA(i,j) = 0.0;
	} else if (fabsij > th) {
	  h = DD(j)-DD(i);
	  if (fabs(h)+g==fabs(h)) {
	    t = AA(i,j) / h;
	  } else {
	    theta = 0.5 * h / AA(i,j);
	    t = 1.0/(fabs(theta)+sqrt(1.0+theta*theta));
	    if (theta<0.0) t = -t;
	  }
	  c=1.0/sqrt(1.0+t*t);
	  s=t*c;
	  tau=s/(1.0+c);
	  h=t*AA(i,j);
	  DD(i) -= h;
	  DD(j) += h;
	  AA(i,j) = 0.0;
	  for (k=0;k<i;k++) {
	    aki = AA(k,i);
	    akj = AA(k,j);
	    AA(k,i)=aki-s*(akj+tau*aki);
	    AA(k,j)=akj+s*(aki-tau*akj);
	  }
	  for (k=i+1;k<j;k++) {
	    aki = AA(i,k);
	    akj = AA(k,j);
	    AA(i,k)=aki-s*(akj+tau*aki);
	    AA(k,j)=akj+s*(aki-tau*akj);
	  }
	  for (k=j+1;k<n;k++) {
	    aki = AA(i,k);
	    akj = AA(j,k);
	    AA(i,k)=aki-s*(akj+tau*aki);
	    AA(j,k)=akj+s*(aki-tau*akj);
	  }
	  for (k=0;k<n;k++) {
	    aki = UU(k,i);
	    akj = UU(k,j);
	    UU(k,i)=aki-s*(akj+tau*aki);
	    UU(k,j)=akj+s*(aki-tau*akj);
	  }
	}
	/*
	printf("c=%d,i=%d,j=%d\n",count,i,j);
	print_mat(n,a);
	*/
      }
    }
  }
}

void sort_eigen(int n, double *d, double *u, int dir)
{
  int i,j,ip;
  double p;

  for (i=0;i<n;i++) {
    ip=i;
    p=d[i];
    for (j=i+1; j<n; j++) {
      if ((dir && d[j] >= p) || (!dir && d[j] <= p)) {
	p=d[j];
	ip=j;
      }
    }
    if (ip != i) {
      d[ip]=d[i];
      d[i]=p;
      for (j=0;j<n;j++) {
	p=UU(j,i);
	UU(j,i)=UU(j,ip);
	UU(j,ip)=p;
      }
    }
  }
}

double determinant33(double A[3][3])
{
  double ret;
  
  ret = (A[0][0]*A[1][1]*A[2][2] +
	 A[0][2]*A[1][0]*A[2][1] + 
	 A[0][1]*A[1][2]*A[2][0]) 
       -(A[0][2]*A[1][1]*A[2][0] +
         A[0][1]*A[1][0]*A[2][2] +
         A[0][0]*A[1][2]*A[2][1] );
  return(ret);
}

void transpose33(double A[3][3],double T[3][3])  /* T = tA */
{
  int i,j;

  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      T[i][j] = A[j][i];
    }
  }
}

void mul_mat33(double A[3][3],double B[3][3], double C[3][3])
     /* A * B = C */
{
  int i,j,k; 
  
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      C[i][j] = 0.0;
      for (k=0;k<3;k++)
	C[i][j] +=A[i][k]*B[k][j];
    }
  }
}


void cross3(double a[3], double b[3], double c[3])
{
  c[0]=a[1]*b[2]-a[2]*b[1];
  c[1]=a[2]*b[0]-a[0]*b[2];
  c[2]=a[0]*b[1]-a[1]*b[0];
}

double dot3(double a[3], double b[3])
{
  return a[0]*b[0]+a[1]*b[1]+a[2]*b[2];
}

/* file open utilities */
FILE *safe_fopen(char *fname, char *mode)
{
  FILE *fp;
  int id;
  char *func="safe_fopen";
  
  if (_safe_fopen_fname != NULL) {
    _safe_fopen_fname = erealloc(func, _safe_fopen_fname, strlen(fname)+3);
  } else {
    _safe_fopen_fname = emalloc(func, strlen(fname)+3);
  }
  
  if ((fp = fopen(fname, "r")) != NULL) {
    id = 0;
    do {
      fclose(fp);
      if (id >= 100) {
	lprintf("ERROR: \"%s.0-99\" already exists. Unable to create file!\n", fname);
	return NULL;
      }
      sprintf(_safe_fopen_fname, "%s.%d",fname,id++);
      fp = fopen(_safe_fopen_fname, "r");
    } while (fp != NULL);
    /*
    lprintf("WARNING: \"%s\" already exists. Open \"%s\" instead.\n",
	    fname, _safe_fopen_fname);
    */
    fp = fopen(_safe_fopen_fname, "w");
    /* free(_safe_fopen_fname); */
  } else {
    strcpy(_safe_fopen_fname, fname);
    fp = fopen(fname, "w");
  }

#ifdef LARGE_FBUF
  {
    char *buf;
    buf = emalloc("safe_open", FBUF_SIZE);
    setvbuf(fp, buf, _IOFBF, FBUF_SIZE);
  }
#endif /* LARGE_FBUF */
  
  return fp;
}

double gauss_rand()
{
  static int backup = 0;
  static double x_backup;
  double u1,u2;
  double v1,v2;
  double s, c;

  if (backup == 1) {
    backup = 0;
    return x_backup;
  }

  do {
    u1=drand48();
    u2=drand48();
    v1=2.0*u1-1.0;
    v2=2.0*u2-1.0;
    s=v1*v1+v2*v2;
  } while (s>=1.0);
  c=sqrt(-2.0*log(s)/s);
  x_backup = v1*c;
  return v2*c;
  
  /*
  int i;
  double sum = 0.0;

  for (i = 0; i < 12; i++) {
    sum += drand48();
  }
  return sum - 6.0;
  */
}

/* sel_data routines */
int get_id_sel_data(SEL_DATA *sel_data, char *name)
{
  int i;
  for (i=0;sel_data[i].name != NULL;i++) {
    if (strcmp(sel_data[i].name, name) == 0) {
      return sel_data[i].id;
    }
  }

  lprintf("ERROR: unrecognized term: %s\n", name);
  marble_exit(1);
  return 0;
}

char *get_name_sel_data(SEL_DATA *sel_data, int id)
{
  int i;
  for (i=0;sel_data[i].name != NULL;i++) {
    if (id == sel_data[i].id)
      break;
  }
  return sel_data[i].name;
}

char *com_fgets(char *buf, int size, FILE *fp)
{
  int ok, i;
  char *ret;
 
  do {
    ret = fgets(buf, size, fp);
    if (ret == NULL) return NULL;
    ok = 1;
    for (i=0;i<size;i++)
      if (buf[i] != ' ' && buf[i] != '\t') break;
    
    if (buf[i]=='#') ok = 0;
    if (buf[i]=='\n') ok = 0;
    if (buf[i]=='\0') ok = 0;
  } while (!ok);
  return ret;
}

void fill_space(char *buf, int n)
{
  int i;
  for (i=0;buf[i]!='\0';i++);
  for (;i<n;i++)
    buf[i]=' ';
  if (i==n) buf[i]='\0';
}

void append_str_list(STR_LIST **headp, char *buf)
{
  STR_LIST *p, *cur;
  
  cur = emalloc("append_str_list", sizeof(STR_LIST));
  cur->buf = emalloc("append_str_list", strlen(buf)+1);
  strcpy(cur->buf, buf);
  cur->next = NULL;
  
  if (*headp == NULL) {
    *headp = cur;
  } else {
    for (p=*headp;p->next!=NULL;p=p->next);
    p->next = cur;
  }
}


/* vector calculation routines */
/* c = a + b; */
void v_add(VEC *c, VEC *a, VEC *b)
{
  c->x = a->x + b->x;
  c->y = a->y + b->y;
  c->z = a->z + b->z;
}


/* c = a - b; */
void v_sub(VEC *c, VEC *a, VEC *b)
{
  c->x = a->x - b->x;
  c->y = a->y - b->y;
  c->z = a->z - b->z;
}

/* c = a * b; */
void v_mul(VEC *c, double a, VEC *b)
{
  c->x = a * b->x;
  c->y = a * b->y;
  c->z = a * b->z;
}

double v_inner_pro(VEC *a, VEC *b)
{
  return (a->x*b->x + a->y*b->y + a->z*b->z);
}

void v_outer_pro(VEC *c, VEC *a, VEC *b)
{
  c->x = a->y*b->z - a->z*b->y;
  c->y = a->z*b->x - a->x*b->z;
  c->z = a->x*b->y - a->y*b->x;
}

void v_norm(VEC *a)
{
  double l;
  l = sqrt(a->x*a->x + a->y*a->y + a->z*a->z);

  a->x /= l; a->y /= l; a->z /= l;
}

/* b = mat * a; */
void v_mat_mul(VEC *b, double mat[3][3], VEC *a)
{
  VEC tmp;
  tmp.x = mat[0][0]*a->x + mat[0][1]*a->y + mat[0][2]*a->z;
  tmp.y = mat[1][0]*a->x + mat[1][1]*a->y + mat[1][2]*a->z;
  tmp.z = mat[2][0]*a->x + mat[2][1]*a->y + mat[2][2]*a->z;

  *b=tmp;
}

/* b = a * mat; */
void v_mul_mat(VEC *b, VEC *a, double mat[3][3])
{
  VEC tmp;
  tmp.x = a->x*mat[0][0] + a->y*mat[1][0] + a->z*mat[2][0];
  tmp.y = a->x*mat[0][1] + a->y*mat[1][1] + a->z*mat[2][1];
  tmp.z = a->x*mat[0][2] + a->y*mat[1][2] + a->z*mat[2][2];

  *b=tmp;
}

void v_rot(VEC *b, VEC *axis, double theta, VEC *a)
{
  VEC nax;
  double mtx[3][3];

  nax = *axis;
  v_norm(&nax);
  
  mtx[0][0] = nax.x*nax.x + (1-nax.x*nax.x)*cos(theta);
  mtx[0][1] = (nax.x)*(nax.y)*(1-cos(theta))-(nax.z)*sin(theta);
  mtx[0][2] = (nax.x)*(nax.z)*(1-cos(theta))+(nax.y)*sin(theta);
  
  mtx[1][0] = (nax.x)*(nax.y)*(1-cos(theta))+(nax.z)*sin(theta);
  mtx[1][1] = nax.y*nax.y + (1-nax.y*nax.y)*cos(theta);
  mtx[1][2] = (nax.y)*(nax.z)*(1-cos(theta))-(nax.x)*sin(theta);
  
  mtx[2][0] = (nax.x)*(nax.z)*(1-cos(theta))-(nax.y)*sin(theta);
  mtx[2][1] = (nax.y)*(nax.z)*(1-cos(theta))+(nax.x)*sin(theta);
  mtx[2][2] = nax.z*nax.z + (1-nax.z*nax.z)*cos(theta);

  v_mat_mul(b, mtx, a);
}


void RIA_init(RESIZABLE_INT_ARRAY *arr, int n_unit)
{
  arr->data = NULL;
  arr->n_alloc = 0;
  arr->n_unit = n_unit;
}

void RIA_alloc(RESIZABLE_INT_ARRAY *arr, int req)
{
  char *func="RIA_alloc";

  if (arr->n_alloc < req) {
    arr->n_alloc = ((int) (req / arr->n_unit) + 1)* arr->n_unit;
    arr->data = erealloc(func, arr->data, sizeof(int)*arr->n_alloc);
  }
}

void RIA_free(RESIZABLE_INT_ARRAY *arr, int req)
{
  char *func="RIA_alloc";

  if (arr->data) {
    free(arr->data);
  }
  arr->data=NULL;
  arr->n_alloc = 0;
}

#ifndef RIA_set
void RIA_set(RESIZABLE_INT_ARRAY *arr, int id, int val)
{
  if (id >= arr->n_alloc) {
    RIA_alloc(arr, id+1);
  }
  arr->data[id] = val;
}
#endif

void RIA_set_all(RESIZABLE_INT_ARRAY *arr, int val)
{
  int i;
  for (i=0;i<arr->n_alloc;i++) {
    arr->data[i] = val;
  }
}

void RIA_print(RESIZABLE_INT_ARRAY *arr, char *header, int n)
{
  int i;
  for (i=0;i<n;i++) {
    printf("%s %d %d\n", header, i, arr->data[i]);
  }
}

/********************************************/

void RVA_init(RESIZABLE_VEC_ARRAY *arr, int n_unit)
{
  arr->data = NULL;
  arr->n_alloc = 0;
  arr->n_unit = n_unit;
}

void RVA_alloc(RESIZABLE_VEC_ARRAY *arr, int req)
{
  char *func="RVA_alloc";

  if (arr->n_alloc < req) {
    arr->n_alloc = ((int) (req / arr->n_unit) + 1)* arr->n_unit;
    arr->data = erealloc(func, arr->data, sizeof(VEC)*arr->n_alloc);
  }
}

void RVA_free(RESIZABLE_VEC_ARRAY *arr, int req)
{
  char *func="RVA_alloc";

  if (arr->data) {
    free(arr->data);
  }
  arr->data=NULL;
  arr->n_alloc = 0;
}


void RVA_set(RESIZABLE_VEC_ARRAY *arr, int id, double x, double y, double z)
{
  if (id >= arr->n_alloc) {
    RVA_alloc(arr, id+1);
  }
  arr->data[id].x = x;
  arr->data[id].y = y;
  arr->data[id].z = z;
}

void RVA_set_all(RESIZABLE_VEC_ARRAY *arr, double x, double y, double z)
{
  int i;
  for (i=0;i<arr->n_alloc;i++) {
    arr->data[i].x = x;
    arr->data[i].y = y;
    arr->data[i].z = z;
  }
}

void RVA_clear(RESIZABLE_VEC_ARRAY *arr, int n)
{
  memset(arr->data, 0, sizeof(VEC)*n);
}

void RVA_print(RESIZABLE_VEC_ARRAY *arr, char *header, int n)
{
  int i;
  for (i=0;i<n;i++) {
    printf("%s %d (%f %f %f)\n", header, i, arr->data[i].x, arr->data[i].y, arr->data[i].z);
  }
}
