/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "md_system.h"

#ifdef MPI_RDMD
#include "parallel.h"
#endif

void MDAT_read_file(MD_SYSTEM *sys, char *fname)
{
  int err;
  if (sys->read_mdat_flag) {
    MD_SYSTEM_finalize(sys);
  }

  strcpy(sys->mdat_fname, fname);

#ifdef MPI
  if (mpi.master) {
    err = MDAT_read_file_master(sys, fname);
  }
  MDAT_read_file_comm(sys, err);
#else
  err = MDAT_read_file_master(sys, fname);
  if (err) {
    marble_exit(1);
  }
#endif

  sys->rigid_mol_flag = RMOL_DATA_n_mol(&(sys->rigid_mol));

  ATOM_DATA_setup_solute(&(sys->atom));
  BOUNDARY_check_water(&(sys->boundary), &(sys->atom));

  MD_SYSTEM_degree_of_freedom(sys);
  MDAT_print_info(sys);

  sys->diel = 1.0;
  sys->read_mdat_flag = 1;

  BOND_DATA_set_hydrogen_group(&(sys->bond), &(sys->atom));

  /*
  {
    int i;
    double total_q = 0.0;
    ATOM_DATA *ad;
    ad = &sys->atom;
    for (i=0;i<ad->natom;i++) {
      total_q += ad->q[i];
    }
    lprintf("sum of charges : %f\n", total_q);
    total_q /= ad->natom;
    for (i=0;i<ad->natom;i++) {
      ad->q[i] -= total_q;
    }
  }
  */
}


int MDAT_read_file_master(MD_SYSTEM *sys, char *fname)
{
  FILE *fp;
  char buf[101];
  ATOM_DATA *ad;
  int i;
  
  if ((fp = fopen(fname,"r")) == NULL) {
    lprintf("ERROR: Can't open mdat file \"%s\"\n", fname);
    return 1;
  }
  fgets(buf, 100, fp);
  if (sscanf(buf,"mdat %d.%d format %d",
	     &sys->mdat_major_version,
	     &sys->mdat_minor_version, &sys->mdat_format) != 3) {
    lprintf("ERROR: The file \"%s\" is not mdat format.\n", fname);
    return 1;
  }
  if (sys->mdat_major_version == 0) {
    if (sys->mdat_minor_version <= 3) {
      lprintf("ERROR: Version of this mdat file (Ver %d.%d) is too old.\n",
	      sys->mdat_major_version, sys->mdat_minor_version);
      lprintf("       Current version is %d.%d\n", 0, 5);
      return 1;
    }
    if (sys->mdat_minor_version >= 5) {
      sys->atom.pdb_no_flag = 1;
    } else {
      sys->atom.pdb_no_flag = 0;
    }
    if (sys->mdat_minor_version >= 6) {
      sys->dihed.cmap_flag = 1;
    } else {
      sys->dihed.cmap_flag = 0;
    }
  }
  
  fgets(buf, 100, fp);
  sscanf(buf,"%80s", sys->title);

  if (MDAT_read_atom_data(&sys->atom, fp, sys->mdat_format)) return 1;

  if (MDAT_read_bond_data(&(sys->bond), &(sys->atom), fp)) return 1;

  if (MDAT_read_angle_data(&(sys->angle), fp)) return 1;

  if (MDAT_read_dihedral_data(&(sys->dihed), fp)) return 1;

  if (sys->dihed.cmap_flag)
    if (MDAT_read_cmap_data(&(sys->dihed), fp)) return 1;

  if (MDAT_read_rmol_data(&(sys->rigid_mol), &(sys->atom), fp)) return 1;

  if (MDAT_read_boundary_data(&(sys->boundary), fp)) return 1;

  fclose(fp);

  return 0;
}

int MDAT_read_atom_data(ATOM_DATA *ad, FILE *fp, int mdat_format)
{
  int i,j, atomno, resno, n_table;
  double rmin6;

  fscanf(fp, "%d", &(ad->natom));
  
  ATOM_DATA_allocate(ad);
  ad->total_w = 0.0;
  ad->mdat_format = mdat_format;
  for (i=0; i < ad->natom; i++) {
    /*
      if (fscanf(fp, "%d %4[ A-Za-z0-9+'*\\-] %4[ A-Za-z0-9+'*\\-] %d%lf%lf",
    */
    /*
    if (fscanf(fp, "%d %4[ A-Za-z0-9+'*\\-] %4s %d%lf%lf",
    */
    if (fscanf(fp, "%d %s %s %d%lf%lf",
	       &atomno,
	       ad->a[i].name, ad->a[i].sym,
	       &(ad->vdw_type[i]),
	       &(ad->q[i]),&(ad->w[i])) != 6) {
      lprintf("MDAT: ATOM DATA: read file error atomno %d\n", atomno);
      return 1;
    }
    if (atomno != i) {
      lprintf("MDAT: ATOM DATA: read file error atomno %d,%d\n", atomno,i);
      return 1;
    }
    ad->total_w += ad->w[i];
    ad->ex[i].group = 0;
    if (mdat_format == MDAT_CHARMM) {
      ad->q[i] *= sqrt(332.0716);
    }
  }
  fscanf(fp, "%d", &(ad->nres));
  ad->r = emalloc("read_atom_data",sizeof(RESIDUE)*ad->nres);
  if (ad->pdb_no_flag) {
    for (i=0; i<ad->nres; i++) {
      /* if (fscanf(fp, "%d %4[ A-Za-z0-9+'*\\-] %d%d%d", */
      if (fscanf(fp, "%d %s %d%d%d \'%c\'",
		 &(ad->r[i].pdb_no), ad->r[i].name,
		 &(ad->r[i].start_atom),&(ad->r[i].natom),&(ad->r[i].flag),
		 &(ad->r[i].pdb_chain)) != 6) {
	lprintf("Error: reading Residue No. %d in mdat file\n", ad->r[i].pdb_no);
	lprintf("       i=%d, nres=%d\n", i, ad->nres);
	lprintf("       %d %s %d %d %d [%c]\n", ad->r[i].pdb_no,
		ad->r[i].name,
		ad->r[i].start_atom, ad->r[i].natom, ad->r[i].flag,
		ad->r[i].pdb_chain);
	return 1;
      }
      ad->r[i].end_atom = ad->r[i].start_atom + ad->r[i].natom - 1;
      ad->r[i].center_atom = ad->r[i].start_atom;
    }
  } else {
    for (i=0; i<ad->nres; i++) {
    /* if (fscanf(fp, "%d %4[ A-Za-z0-9+'*\\-] %d%d%d", */
      if (fscanf(fp, "%d %s %d%d%d",
		 &resno,  ad->r[i].name,
		 &(ad->r[i].start_atom),&(ad->r[i].natom),&(ad->r[i].flag)) != 5) {
	lprintf("MDAT ATOM DATA: read file error resno %d\n", resno);
	return 1;
      }
      ad->r[i].end_atom = ad->r[i].start_atom + ad->r[i].natom - 1;
      ad->r[i].center_atom = ad->r[i].start_atom;
      ad->r[i].pdb_no = i + 1;
      ad->r[i].pdb_chain = ' ';
    }
  }
  for (i=0;i<ad->nres;i++) {
    for (j=0;j<ad->r[i].natom;j++) {
      ad->a[j+ad->r[i].start_atom].resno = i;
    }
  }
  
  fscanf(fp, "%d %d", &(ad->nmol), &(ad->n_solute_mol));
  ad->mol = emalloc("read_atom_data",sizeof(MOLECULE)*ad->nmol);
  atomno = 0;
  for (i=0;i<ad->nmol;i++) {
    fscanf(fp,"%d %d%c\n",&(ad->mol[i].natom), &(ad->mol[i].flag),&(ad->mol[i].chain));
    ad->mol[i].start_atom = atomno;
    atomno += ad->mol[i].natom;
  }

  /* calculate molecular weight */
  for (i=0;i<ad->nmol;i++) {
    ad->mol[i].w = 0.0;
    for (j=0;j<ad->mol[i].natom;j++) {
      atomno = ad->mol[i].start_atom + j;
      ad->mol[i].w += ad->w[atomno];
    }
  }

  /* read van der Waals and H bond parameters */
  fscanf(fp, "%d%d\n", &(ad->ntype), &(ad->n_hbpair));
  fscanf(fp, "%lf%lf\n", &(ad->scnb), &(ad->scee));
  
  ATOM_DATA_allocate_atom_type(ad);

  for (i=0; i < ad->ntype*ad->ntype; i++) {
    fscanf(fp, "%d", &(ad->index[i]));
  }

  n_table = ad->ntype*(ad->ntype+1)/2;

  if (mdat_format == MDAT_AMBER) {
  
    for (i=0; i < n_table; i++)
      fscanf(fp, "%lf", &(ad->vdw12[i]));
    for (i=0; i < n_table; i++)
      fscanf(fp, "%lf", &(ad->vdw6[i]));
      
    for (i=0; i < ad->n_hbpair; i++)
      fscanf(fp, "%lf", &(ad->hb12[i]));
    for (i=0; i < ad->n_hbpair; i++)
      fscanf(fp, "%lf", &(ad->hb10[i]));

    /* lprintf("ntype %d %d\n", ad->ntype, ad->n_hbpair); */
    
    for (i=0; i < n_table; i++) {
      ad->vdw12_14[i] = ad->vdw12[i] / ad->scnb;
      ad->vdw6_14[i]  = ad->vdw6[i]  / ad->scnb;
    }

    /* Convert vdw12,vdw6 to eps, rmin */
    for (i=0; i < n_table; i++) {
      if(ad->vdw6[i] != 0.0) {   /*if vdw6=0, rmin = eps = 0.0 */
	rmin6 = ad->vdw12[i] / ad->vdw6[i] * 2.0;
	ad->eps[i] = ad->vdw6[i] / rmin6 * 0.5;
	ad->rmin[i] = pow(rmin6 ,1.0/6.0);
      } else {
	ad->eps[i] = ad->rmin[i] = 0.0;
      }
    }
  } else if (mdat_format == MDAT_CHARMM) {
    for (i=0; i < n_table; i++)
      fscanf(fp, "%lf", &(ad->eps[i]));
    for (i=0; i < n_table; i++)
      fscanf(fp, "%lf", &(ad->rmin[i]));
    ad->n_hbpair = 0;
    for (i=0; i < n_table; i++)
      fscanf(fp, "%lf", &(ad->eps14[i]));
    for (i=0; i < n_table; i++)
      fscanf(fp, "%lf", &(ad->rmin14[i]));
    for (i=0; i < n_table; i++) {
      ad->vdw12[i] = ad->eps[i]*pow(ad->rmin[i],12.0);
      ad->vdw6[i]  = ad->eps[i]*pow(ad->rmin[i],6.0) * 2.0;
      ad->vdw12_14[i] = ad->eps14[i]*pow(ad->rmin14[i],12.0);
      ad->vdw6_14[i]  = ad->eps14[i]*pow(ad->rmin14[i],6.0) * 2.0;
    }
  }
  
  /* Generate of Symbols  */
  ad->sym = emalloc("read_atom_data",sizeof(char)*ad->ntype*5);
  for (i=0; i < ad->ntype; i++) {
    ad->sym[i][0] = '\0';
  }
  
  for (i=0;i<ad->natom;i++) {
    strcpy(ad->sym[ad->vdw_type[i]],ad->a[i].sym);
  }
  
  /* extra atom data */
  for (i=0;i < ad->natom; i++) {
    if (fscanf(fp, "%d%d",&atomno,&(ad->ex[i].n_exatom)) != 2) {
      lprintf("MDAT: ATOM DATA EXCLUSIVE 1: read file error %d\n", atomno);
      return 1;
    }
    if (atomno != i) {
      lprintf("MDAT: ATOM DATA EXCLUSIVE 2: read file error %d,%d\n",atomno,i);
      return 1;
    }
    ad->ex[i].exatom = emalloc("read_atom_data",
			       sizeof(int)*ad->ex[i].n_exatom);
    for (j=0; j<ad->ex[i].n_exatom;j++) {
      if (fscanf(fp, "%d",&(ad->ex[i].exatom[j])) != 1) {
	lprintf("MDAT: ATOM DATA EXCLUSIVE 2: read file error %d i=%d, n_exatom=%d, j=%d\n", atomno,i,ad->ex[i].n_exatom,j);
	return 1;
      }
    }
  }

  /*
  lprintf("atom: %d, type: %d, hb pair: %d\n",
         ad->natom, ad->ntype, ad->n_hbpair);
  */
  for (i=0;i<ad->natom;i++) {
    ad->ex[i].flag = 0;
  }

  ATOM_DATA_set_residue_type(ad);

  return 0;
}

int MDAT_read_bond_data(BOND_DATA* bd, ATOM_DATA *ad, FILE *fp)
{
  int i, atom1, atom2;

  fscanf(fp,"%d%d",&(bd->n_bond),&(bd->n_bond_h));

  bd->bonds = emalloc("read_bond_data",sizeof(BOND)*bd->n_bond);

  for (i=0; i < bd->n_bond; i++) {
    if (fscanf(fp, "%d %d %d\n", &(bd->bonds[i].atom1), &(bd->bonds[i].atom2),
	       &(bd->bonds[i].type)) != 3) {
      lprintf("MDAT: BOND DATA: file read error\n");
      return 1;
    }
  }

  fscanf(fp,"%d",&(bd->n_bond_type));
  bd->bond_type = emalloc("read_bond_data",sizeof(BOND_TYPE)*bd->n_bond_type);
  for (i=0; i < bd->n_bond_type; i++) {
    if (fscanf(fp, "%lf %lf\n", &(bd->bond_type[i].k), &(bd->bond_type[i].r0))
	!= 2) {
      lprintf("MDAT: BOND DATA: file read error\n");
      return 1;
    }
  }
  /*
  printf("bond: %d, bond_type:%d\n", bd->n_bond, bd->n_bond_type);
  */
  set_all_bonds_flag(bd, ad);

  return 0;
}

int MDAT_read_angle_data(ANGLE_DATA *ad, FILE *fp)
{
  int i;

  fscanf(fp,"%d%d",&(ad->n_angle),&(ad->n_angle_h));
  ad->angles = emalloc("read_angle_data",sizeof(ANGLE)*ad->n_angle);

  for (i=0; i < ad->n_angle; i++) {
    if (fscanf(fp,"%d %d %d %d\n",
	       &(ad->angles[i].atom1),
	       &(ad->angles[i].atom2),
	       &(ad->angles[i].atom3),
	       &(ad->angles[i].type)) != 4) {
      lprintf("MDAT: ANGLE DATA: file read error\n");
      return 1;
    }
    /* ad->angles[i].th0 *= M_PI / 180.0; */
  }
  
  fscanf(fp,"%d",&(ad->n_angle_type));
  ad->angle_type = emalloc("read_angle_data",
			   sizeof(ANGLE_TYPE)*ad->n_angle_type);
  
  for (i=0; i < ad->n_angle_type; i++) {
    if (fscanf(fp,"%lf %lf %lf %lf\n",
	       &(ad->angle_type[i].k),
	       &(ad->angle_type[i].th0),
	       &(ad->angle_type[i].kub),
	       &(ad->angle_type[i].s0)) != 4) {
      fprintf(stderr,"MDAT: ANGLE DATA: file read error\n");
      return 1;
    }
  }
  /*
  printf("angle:%d, angle_type:%d\n",ad->n_angle, ad->n_angle_type);
  */
  return 0;
}

int MDAT_read_dihedral_data(DIHEDRAL_DATA* dd, FILE *fp)
{
  int i, atom1, atom2, atom3, atom4;
  char *fname = "read_dihedral_data";

  fscanf(fp,"%d%d",&(dd->n_dihedral), &(dd->n_improper));
  dd->dihedrals = emalloc(fname,sizeof(DIHEDRAL)*dd->n_dihedral);
  for (i=0; i < dd->n_dihedral; i++) {
    if (fscanf(fp, "%d %d %d %d %d %d\n",
	       &(dd->dihedrals[i].atom1),
	       &(dd->dihedrals[i].atom2),
	       &(dd->dihedrals[i].atom3),
	       &(dd->dihedrals[i].atom4),
	       &(dd->dihedrals[i].type),
	       &(dd->dihedrals[i].flag)) != 6) {
      lprintf("MDAT: DIHEDRAL DATA: file read error\n");
      return 1;
    }
  }
  
  dd->n_multi = 0;
  dd->n_only14 = 0;
  for (i=0; i < dd->n_dihedral; i++) {
    if ((dd->dihedrals[i].flag & DIHED_MULTI) &&
	!(dd->dihedrals[i].flag & (DIHED_AMBER_IMPR | DIHED_CHARMM_IMPR)))
      dd->n_multi++;
    if (dd->dihedrals[i].flag & DIHED_ONLY14)
      dd->n_only14++;
  }

  fscanf(fp,"%d",&(dd->n_dihed_type));
  dd->dihed_type = emalloc(fname, sizeof(DIHEDRAL_TYPE)*dd->n_dihed_type);
  for (i=0; i < dd->n_dihed_type; i++) {
    if (fscanf(fp, "%lf %lf %lf\n",
	   &(dd->dihed_type[i].v),&(dd->dihed_type[i].eta),
	   &(dd->dihed_type[i].ganma)) != 3) {
      lprintf("MDAT: DIHEDRAL DATA: file read error\n");
      return 1;
    }
  }

  /* lprintf("Number of dihedral types: %d\n",dd->n_dihed_type); */
  

  fscanf(fp,"%d",&(dd->n_impr_type));
  if (dd->n_impr_type > 0)
    dd->impr_type = emalloc(fname, sizeof(IMPROPER_TYPE)*dd->n_impr_type);
  else 
    dd->impr_type = NULL;
  
  /* lprintf("Number of improper types: %d\n",dd->n_impr_type); */
  
  for (i=0; i < dd->n_impr_type; i++) {
    if (fscanf(fp, "%lf %lf\n",
	       &(dd->impr_type[i].kpsi),&(dd->impr_type[i].psi0)) != 2) {
      lprintf("MDAT: IMPROPER TYPE DATA: file read error\n");
      return 1;
    }
  }

  dd->n_nb14 = 0;
  for (i=0; i < dd->n_dihedral; i++) {
    if (dd->dihedrals[i].atom3 < 0 || dd->dihedrals[i].atom4 < 0)
      continue;
    if (dd->dihedrals[i].flag & DIHED_OMIT14) continue;
    dd->n_nb14++;
  }
  
  /*
  printf("dihedral %d: dihedral_type: %d\n", dd->n_dihedral, dd->n_dihed_type);
  */
  return 0;
}

int MDAT_read_cmap_data(DIHEDRAL_DATA* dd, FILE *fp)
{
  int i, j, k, n, n_alloc;
  double dx, *e, *dedx, *dedy, *d2edxdy, (*tmp)[4];
  char *fname = "MDAT_read_cmap_data";
  
  fscanf(fp,"%d",&(dd->n_cmap));
  
  if (dd->n_cmap > 0)
    dd->cmap = emalloc(fname,sizeof(CMAP)*dd->n_cmap);

  for (i=0; i < dd->n_cmap; i++) {
    if (fscanf(fp,"%d%d%d%d%d%d%d%d%d\n",
	       &dd->cmap[i].atom[0],
	       &dd->cmap[i].atom[1],
	       &dd->cmap[i].atom[2],
	       &dd->cmap[i].atom[3],
	       &dd->cmap[i].atom[4],
	       &dd->cmap[i].atom[5],
	       &dd->cmap[i].atom[6],
	       &dd->cmap[i].atom[7],
	       &dd->cmap[i].type) != 9) {
      lprintf("MDAT: CMAP: invalid file format\n");
      return 1;
    }
  }

  fscanf(fp,"%d",&(dd->n_cmap_type));
  
  if (dd->n_cmap_type > 0) 
    dd->cmap_type = emalloc(fname,sizeof(CMAP_TYPE)*dd->n_cmap_type);
  
  /* lprintf("%d %d\n", dd->n_cmap, dd->n_cmap_type); */

  n_alloc = 0;
  for (i=0;i<dd->n_cmap_type;i++) {
    fscanf(fp,"%d",&n);
    /* lprintf("%d\n", n); */
    dd->cmap_type[i].ndiv = n;
    dd->cmap_type[i].c = emalloc(fname, sizeof(double [4][4])*n*n);
    if (n_alloc < n) {
      if (n_alloc == 0) {
	e       = emalloc(fname, sizeof(double)*n*n);
	dedx    = emalloc(fname, sizeof(double)*n*n);
	dedy    = emalloc(fname, sizeof(double)*n*n);
	d2edxdy = emalloc(fname, sizeof(double)*n*n);
	tmp     = emalloc(fname, sizeof(double [4])*n);
      } else {
	e       = erealloc(fname,e      , sizeof(double)*n*n);
	dedx    = erealloc(fname,dedx   , sizeof(double)*n*n);
	dedy    = erealloc(fname,dedy   , sizeof(double)*n*n);
	d2edxdy = erealloc(fname,d2edxdy, sizeof(double)*n*n);
	tmp     = erealloc(fname,tmp    , sizeof(double [4])*n);
      }
      n_alloc = n;
    }
    for (j=0;j<n;j++) {
      for (k=0;k<n;k++) {
	fscanf(fp,"%lf", &e[j*n+k]);
      }
    }
    dx = 2.0*M_PI/n;

    DD_CMAP_make_derivatives(e, dedx, dedy, d2edxdy,
			     tmp, n, n, dx, dx);
    DD_CMAP_make_cij(e, dedx, dedy, d2edxdy,
		     n, n, dx, dx, dd->cmap_type[i].c);

  }
  /*lprintf("%d %d\n", dd->n_cmap, dd->n_cmap_type); */
  if (n_alloc > 0) {
    free(e);
    free(dedx);
    free(dedy);
    free(d2edxdy);
    free(tmp);
  }
  return 0;
}


int MDAT_read_rmol_data(RMOL_DATA *md, ATOM_DATA *ad, FILE *fp)
{
  char buf[1000];
  int n_rmol_type;
  char *func = "MDAT_read_rmol_data";
  
  /*
  return RMOL_DATA_read_library_fp(md, ad, fp);
  */
  com_fgets(buf,1000,fp);
  if (sscanf(buf,"%d",&n_rmol_type) != 1) {
    lprintf("ERROR: %s: invalid n_type\n", func);
    return 1;
  }

  /* printf("n_type = %d\n", n_rmol_type); */

  if (n_rmol_type != 0) {
    lprintf("ERROR: RMOLTYPE in mdat_file is not supported now.\n");
    return 1;
  }
  md->n_type = 0;
  md->n_mol = 0;

  return 0;
}

int MDAT_read_boundary_data(BOUNDARY *bc, FILE *fp)
{
  int flag_periodic, flag_spherical, i, j;

  if (fscanf(fp,"%d%lf%lf%lf%lf%lf\n",&(flag_spherical),
	     &(bc->radius),
	     &(bc->cap_fc),
	     &(bc->center[0]),
	     &(bc->center[1]),
	     &(bc->center[2])) != 6) {
    lprintf("MDAT: BOUNDARY DATA: can't read cap data\n");
    return 1;
  }

  if (flag_spherical)
    bc->type = SPHERICAL_CAP;
	 
  if (fscanf(fp,"%d",&(flag_periodic)) != 1) {
    lprintf("MDAT: BOUNDARY DATA: can't read periodic boundary data\n");
    return 1;
  }
  for (i=0;i<3;i++) {  
    for (j=0;j<3;j++) {
      if (fscanf(fp,"%lf",&(bc->boxv[i][j])) != 1) {
	lprintf("MDAT: BOUNDARY DATA: can't read periodic boundary data\n");
	return 1;
      }
    }
  }

  if (flag_periodic) {
    bc->type = PERIODIC_BOUNDARY;
    BOUNDARY_make_recip(bc);
  }
  return 0;
}

#ifdef MPI
void MDAT_read_file_comm(MD_SYSTEM *sys, int err)
{
  int dbuf[10];

  MPI_Bcast(&err, 1, MPI_INT, mpi.master_pe, mpi.all_comm);
  if (err) {
    marble_exit(1);
  }
  
  if (mpi.master) {
    dbuf[0]=sys->mdat_major_version;
    dbuf[1]=sys->mdat_minor_version;
    dbuf[2]=sys->mdat_format;
    MPI_Bcast(dbuf, 3, MPI_INT, mpi.master_pe, mpi.all_comm);

    MPI_Bcast(sys->title, 80, MPI_CHAR, mpi.master_pe, mpi.all_comm);

  } else {
    /* not master */
    MPI_Bcast(dbuf, 3, MPI_INT, mpi.master_pe, mpi.all_comm);
    sys->mdat_major_version = dbuf[0];
    sys->mdat_minor_version = dbuf[1];
    sys->mdat_format	    = dbuf[2];

    if (sys->mdat_minor_version >= 5) {
      sys->atom.pdb_no_flag = 1;
    } else {
      sys->atom.pdb_no_flag = 0;
    }
    if (sys->mdat_minor_version >= 6) {
      sys->dihed.cmap_flag = 1;
    } else {
      sys->dihed.cmap_flag = 0;
    }
    MPI_Bcast(sys->title, 80, MPI_CHAR, mpi.master_pe, mpi.all_comm);
  }

  MDAT_comm_atom_data(&sys->atom, sys->mdat_format);
  MDAT_comm_bond_data(&(sys->bond), &(sys->atom));
  MDAT_comm_angle_data(&(sys->angle));
  MDAT_comm_dihedral_data(&(sys->dihed));
  if (sys->dihed.cmap_flag)
    MDAT_comm_cmap_data(&(sys->dihed));
  MDAT_comm_rmol_data(&(sys->rigid_mol), &(sys->atom));
  MDAT_comm_boundary_data(&(sys->boundary));
}

void MDAT_comm_atom_data(ATOM_DATA *ad, int mdat_format)
{
  int dbuf[10];
  double fbuf[10];
  int i, n_table;
  char *func = "MDAT_comm_atom_data";
  
  if (mpi.master) {

    /* atoms */

    MPI_Bcast(&ad->natom, 1, MPI_INT, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->a, sizeof(ATOM_NAME)*ad->natom, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->vdw_type, ad->natom, MPI_INT, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->q, ad->natom, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->w, ad->natom, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(&ad->total_w, 1, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);

    /* residues */

    MPI_Bcast(&ad->nres, 1, MPI_INT, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->r, sizeof(RESIDUE)*ad->nres, MPI_BYTE, mpi.master_pe, mpi.all_comm);

    /* molecules */
    
    dbuf[0] = ad->nmol;
    dbuf[1] = ad->n_solute_mol;
    MPI_Bcast(dbuf, 2, MPI_INT, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->mol, sizeof(MOLECULE)*ad->nmol, MPI_BYTE, mpi.master_pe, mpi.all_comm);

    /* atom types */

    dbuf[0] = ad->ntype;
    dbuf[1] = ad->n_hbpair;
    MPI_Bcast(dbuf, 2, MPI_INT, mpi.master_pe, mpi.all_comm);

    fbuf[0] = ad->scnb;
    fbuf[1] = ad->scee;
    MPI_Bcast(fbuf, 2, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    
    MPI_Bcast(ad->index,ad->ntype*ad->ntype, MPI_INT, mpi.master_pe, mpi.all_comm);

    n_table = ad->ntype*(ad->ntype+1)/2;
    MPI_Bcast(ad->vdw12,    n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->vdw6,     n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->vdw12_14, n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->vdw6_14,  n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->eps,      n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->rmin,     n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->eps14,    n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->rmin14,   n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);

    if (ad->n_hbpair) {
      MPI_Bcast(ad->hb12, ad->n_hbpair, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
      MPI_Bcast(ad->hb10, ad->n_hbpair, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    }

    MPI_Bcast(ad->sym,   ad->ntype*5, MPI_CHAR, mpi.master_pe, mpi.all_comm);

    /* extra data for atoms */

    MPI_Bcast(ad->ex, sizeof(EXTRA_ATOM_DATA)*ad->natom, MPI_BYTE, mpi.master_pe, mpi.all_comm);

    for (i=0;i<ad->natom;i++) {
      MPI_Bcast(ad->ex[i].exatom, ad->ex[i].n_exatom, MPI_INT, mpi.master_pe, mpi.all_comm);
    }
    
  } else {
    /* not master */

    /* atoms */

    MPI_Bcast(&(ad->natom), 1, MPI_INT, mpi.master_pe, mpi.all_comm);
    ATOM_DATA_allocate(ad);
    ad->mdat_format = mdat_format;
    MPI_Bcast(ad->a, sizeof(ATOM_NAME)*ad->natom, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->vdw_type, ad->natom, MPI_INT, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->q, ad->natom, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->w, ad->natom, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(&ad->total_w, 1, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);

    /* residues */
    
    MPI_Bcast(&ad->nres, 1, MPI_INT, mpi.master_pe, mpi.all_comm);
    ad->r = emalloc(func,sizeof(RESIDUE)*ad->nres);
    MPI_Bcast(ad->r, sizeof(RESIDUE)*ad->nres, MPI_BYTE, mpi.master_pe, mpi.all_comm);

    /* molecules */
    
    MPI_Bcast(dbuf, 2, MPI_INT, mpi.master_pe, mpi.all_comm);
    ad->nmol = dbuf[0];
    ad->n_solute_mol = dbuf[1];
    ad->mol = emalloc(func,sizeof(MOLECULE)*ad->nmol);
    MPI_Bcast(ad->mol, sizeof(MOLECULE)*ad->nmol, MPI_BYTE, mpi.master_pe, mpi.all_comm);

    /* atom types */
    
    MPI_Bcast(dbuf, 2, MPI_INT, mpi.master_pe, mpi.all_comm);
    ad->ntype    = dbuf[0];
    ad->n_hbpair = dbuf[1];

    MPI_Bcast(fbuf, 2, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    ad->scnb = fbuf[0];
    ad->scee = fbuf[1];

    ATOM_DATA_allocate_atom_type(ad);

    MPI_Bcast(ad->index,ad->ntype*ad->ntype, MPI_INT, mpi.master_pe, mpi.all_comm);
    
    n_table = ad->ntype*(ad->ntype+1)/2;
    MPI_Bcast(ad->vdw12,    n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->vdw6,     n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->vdw12_14, n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->vdw6_14,  n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->eps,      n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->rmin,     n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->eps14,    n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(ad->rmin14,   n_table, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);

    if (ad->n_hbpair) {
      MPI_Bcast(ad->hb12, ad->n_hbpair, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
      MPI_Bcast(ad->hb10, ad->n_hbpair, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    }

    ad->sym = emalloc(func,sizeof(char)*ad->ntype*5);
    
    MPI_Bcast(ad->sym,   ad->ntype*5, MPI_CHAR, mpi.master_pe, mpi.all_comm);

    /* extra data for atoms */

    MPI_Bcast(ad->ex, sizeof(EXTRA_ATOM_DATA)*ad->natom, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    for (i=0;i<ad->natom;i++) {
      ad->ex[i].exatom = emalloc(func,sizeof(int)*ad->ex[i].n_exatom);
      MPI_Bcast(ad->ex[i].exatom, ad->ex[i].n_exatom, MPI_INT, mpi.master_pe, mpi.all_comm);
    }

    ATOM_DATA_set_residue_type(ad);
  }
}

void MDAT_comm_bond_data(BOND_DATA* bd, ATOM_DATA *ad)
{
  int dbuf[10];
  char *func = "MDAT_comm_bond_data";
  
  if (mpi.master) {
    dbuf[0] = bd->n_bond;
    dbuf[1] = bd->n_bond_h;
    dbuf[2] = bd->n_bond_type;
    MPI_Bcast(dbuf, 3, MPI_INT, mpi.master_pe, mpi.all_comm);
    
    MPI_Bcast(bd->bonds, sizeof(BOND)*bd->n_bond, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(bd->bond_type, sizeof(BOND_TYPE)*bd->n_bond_type, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    
  } else {
    /* not master */
    MPI_Bcast(dbuf, 3, MPI_INT, mpi.master_pe, mpi.all_comm);
    bd->n_bond      = dbuf[0];
    bd->n_bond_h    = dbuf[1];
    bd->n_bond_type = dbuf[2];

    bd->bonds = emalloc(func, sizeof(BOND)*bd->n_bond);
    MPI_Bcast(bd->bonds, sizeof(BOND)*bd->n_bond, MPI_BYTE, mpi.master_pe, mpi.all_comm);

    bd->bond_type = emalloc(func, sizeof(BOND_TYPE)*bd->n_bond_type);
    MPI_Bcast(bd->bond_type, sizeof(BOND_TYPE)*bd->n_bond_type, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    
  }
}

void MDAT_comm_angle_data(ANGLE_DATA *and)
{
  int dbuf[10];
  char *func = "MDAT_comm_angle_data";
  
  if (mpi.master) {
    dbuf[0] = and->n_angle;
    dbuf[1] = and->n_angle_h;
    dbuf[2] = and->n_angle_type;
    MPI_Bcast(dbuf, 3, MPI_INT, mpi.master_pe, mpi.all_comm);
    
    MPI_Bcast(and->angles, sizeof(ANGLE)*and->n_angle, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(and->angle_type, sizeof(ANGLE_TYPE)*and->n_angle_type, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    
  } else {
    /* not master */
    MPI_Bcast(dbuf, 3, MPI_INT, mpi.master_pe, mpi.all_comm);
    and->n_angle      = dbuf[0];
    and->n_angle_h    = dbuf[1];
    and->n_angle_type = dbuf[2];

    and->angles = emalloc(func, sizeof(ANGLE)*and->n_angle);
    MPI_Bcast(and->angles, sizeof(ANGLE)*and->n_angle, MPI_BYTE, mpi.master_pe, mpi.all_comm);

    and->angle_type = emalloc(func, sizeof(ANGLE_TYPE)*and->n_angle_type);
    MPI_Bcast(and->angle_type, sizeof(ANGLE_TYPE)*and->n_angle_type, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    
  }
}

void MDAT_comm_dihedral_data(DIHEDRAL_DATA* dd)
{
  int dbuf[10];
  char *func = "MDAT_comm_dihedral_data";
  
  if (mpi.master) {
    dbuf[0] = dd->n_dihedral;
    dbuf[1] = dd->n_improper;
    dbuf[2] = dd->n_multi;
    dbuf[3] = dd->n_nb14;
    dbuf[4] = dd->n_dihed_type;
    dbuf[5] = dd->n_impr_type;

    MPI_Bcast(dbuf, 6, MPI_INT, mpi.master_pe, mpi.all_comm);
    
    MPI_Bcast(dd->dihedrals, sizeof(DIHEDRAL)*dd->n_dihedral, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    MPI_Bcast(dd->dihed_type, sizeof(DIHEDRAL_TYPE)*dd->n_dihed_type, MPI_BYTE, mpi.master_pe, mpi.all_comm);

    MPI_Bcast(dd->impr_type, sizeof(IMPROPER_TYPE)*dd->n_impr_type, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    
  } else {
    /* not master */
    MPI_Bcast(dbuf, 6, MPI_INT, mpi.master_pe, mpi.all_comm);

    dd->n_dihedral   = dbuf[0];
    dd->n_improper   = dbuf[1];
    dd->n_multi      = dbuf[2];
    dd->n_nb14       = dbuf[3];
    dd->n_dihed_type = dbuf[4];
    dd->n_impr_type  = dbuf[5];

    dd->dihedrals = emalloc(func, sizeof(DIHEDRAL)*dd->n_dihedral);
    MPI_Bcast(dd->dihedrals, sizeof(DIHEDRAL)*dd->n_dihedral, MPI_BYTE, mpi.master_pe, mpi.all_comm);

    dd->dihed_type = emalloc(func, sizeof(DIHEDRAL_TYPE)*dd->n_dihed_type);
    MPI_Bcast(dd->dihed_type, sizeof(DIHEDRAL_TYPE)*dd->n_dihed_type, MPI_BYTE, mpi.master_pe, mpi.all_comm);

    dd->impr_type = emalloc(func, sizeof(IMPROPER_TYPE)*dd->n_impr_type);
    MPI_Bcast(dd->impr_type, sizeof(IMPROPER_TYPE)*dd->n_impr_type, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    
  }
}

void MDAT_comm_cmap_data(DIHEDRAL_DATA* dd)
{
  int dbuf[10], i;
  char *func = "MDAT_comm_cmap_data";
  
  if (mpi.master) {
    dbuf[0] = dd->n_cmap;
    dbuf[1] = dd->n_cmap_type;

    MPI_Bcast(dbuf, 2, MPI_INT, mpi.master_pe, mpi.all_comm);

    if (dd->n_cmap > 0) {
      MPI_Bcast(dd->cmap, sizeof(CMAP)*dd->n_cmap, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    }

    if (dd->n_cmap_type > 0) {
      MPI_Bcast(dd->cmap_type, sizeof(CMAP_TYPE)*dd->n_cmap_type, 
		MPI_BYTE, mpi.master_pe, mpi.all_comm);
      for (i=0;i<dd->n_cmap_type;i++) {
	MPI_Bcast(dd->cmap_type[i].c, 16*dd->cmap_type[i].ndiv*dd->cmap_type[i].ndiv, 
		  MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
	
      }
    }
    
  } else {
    /* not master */
    MPI_Bcast(dbuf, 2, MPI_INT, mpi.master_pe, mpi.all_comm);

    dd->n_cmap      = dbuf[0];
    dd->n_cmap_type = dbuf[1];

    if (dd->n_cmap > 0) {
      dd->cmap = emalloc(func,sizeof(CMAP)*dd->n_cmap);
      MPI_Bcast(dd->cmap, sizeof(CMAP)*dd->n_cmap, MPI_BYTE, mpi.master_pe, mpi.all_comm);
    } else {
      dd->cmap = NULL;
    }

    if (dd->n_cmap_type > 0) {
      dd->cmap_type = emalloc(func,sizeof(CMAP_TYPE)*dd->n_cmap_type);
      MPI_Bcast(dd->cmap_type, sizeof(CMAP_TYPE)*dd->n_cmap_type, 
		MPI_BYTE, mpi.master_pe, mpi.all_comm);
      
      for (i=0;i<dd->n_cmap_type;i++) {
	dd->cmap_type[i].c = emalloc(func,sizeof(double)*
				     16*dd->cmap_type[i].ndiv*dd->cmap_type[i].ndiv);
	MPI_Bcast(dd->cmap_type[i].c,16*dd->cmap_type[i].ndiv*dd->cmap_type[i].ndiv, 
		  MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
	
      }
    } else {
      dd->cmap_type = NULL;
    }
  }
}

void MDAT_comm_rmol_data(RMOL_DATA *md, ATOM_DATA *ad)
{
  md->n_type = 0;
  md->n_mol = 0;
}

void MDAT_comm_boundary_data(BOUNDARY *bc)
{
  int dbuf[10];
  double fbuf[10];
  char *func = "MDAT_comm_boundary_data";

  if (mpi.master) {
    MPI_Bcast(&bc->type, 1, MPI_INT, mpi.master_pe, mpi.all_comm);

    /* spherical cap */
    fbuf[0] = bc->radius;
    fbuf[1] = bc->cap_fc;
    fbuf[2] = bc->cap_fc;
    fbuf[3] = bc->center[0];
    fbuf[4] = bc->center[1];
    fbuf[5] = bc->center[2];

    MPI_Bcast(fbuf, 6, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    
    MPI_Bcast(bc->boxv, 9, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);

  } else {
    /* not master */
    MPI_Bcast(&bc->type, 1, MPI_INT, mpi.master_pe, mpi.all_comm);

    MPI_Bcast(fbuf, 6, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);

    bc->radius    = fbuf[0];
    bc->cap_fc    = fbuf[1];
    bc->cap_fc    = fbuf[2];
    bc->center[0] = fbuf[3];
    bc->center[1] = fbuf[4];
    bc->center[2] = fbuf[5];

    MPI_Bcast(bc->boxv, 9, MPI_DOUBLE, mpi.master_pe, mpi.all_comm);
    
    if (bc->type == PERIODIC_BOUNDARY) {
      BOUNDARY_make_recip(bc);
    }
  }
}


#endif  /* MPI */


/* print out routines */
void MDAT_print_info(MD_SYSTEM *sys)
{
  lprintf("MDAT FILE: %s\n",sys->mdat_fname);
  lprintf("  Title: %s\n",sys->title);
  lprintf("  Format: %s (Version %d.%d)\n",(sys->mdat_format == MDAT_AMBER) ? "AMBER" : "CHARMM", sys->mdat_major_version, sys->mdat_minor_version);
  lprintf("  Number of atoms:    %6d,  Number of atom types :   %4d\n",
	  sys->atom.natom, sys->atom.ntype);
  lprintf("  Number of bonds:    %6d,  Number of bond types :   %4d\n",
	  sys->bond.n_bond, sys->bond.n_bond_type);
  lprintf("  Number of angles:   %6d,  Number of angle types :  %4d\n",
	  sys->angle.n_angle, sys->angle.n_angle_type);
  lprintf("  Number of dihedrals:%6d,  Number of dihedral types:%4d\n",
	  sys->dihed.n_dihedral-sys->dihed.n_improper-sys->dihed.n_multi-sys->dihed.n_only14, 
	  sys->dihed.n_dihed_type);
  lprintf("  Number of terms of dihedrals: %5d\n",
	  sys->dihed.n_dihedral-sys->dihed.n_improper);
  lprintf("  Number of 14 interactions: %5d\n", sys->dihed.n_nb14);
  if (sys->mdat_format == MDAT_CHARMM)
    lprintf("  Number of impropers:%6d,  Number of improper types:%4d\n",
	    sys->dihed.n_improper, sys->dihed.n_impr_type);
  if (sys->dihed.cmap_flag)
    lprintf("  Number of cmap terms:%5d,  Number of cmap types:   %5d\n",
	    sys->dihed.n_cmap, sys->dihed.n_cmap_type);
  lprintf("  Number of residues: %6d,  Number of molecules:     %4d\n",
	  sys->atom.nres, sys->atom.nmol);
  lprintf("  Number of solute molecules and atoms: %d, %d\n",
	  sys->atom.n_solute_mol, sys->atom.n_solute_atom);
  /*
  lprintf("  Number of rigid molecules: %5d\n",
	  sys->rigid_mol.n_mol);
  */
  
  if (sys->boundary.wat_n_mol>0) {
    lprintf("  Water : name %s, start residue No. %d, number of molecules %d\n",
	    sys->boundary.wat_data->name, sys->boundary.wat_start_res+1,
	    sys->boundary.wat_n_mol);
  } else {
    lprintf("  No water molecule is detected.\n");
  }
  switch (sys->boundary.type) {
  case SPHERICAL_CAP :
    lprintf("  Spherical Boundary Condition: radius %.2f, center (%.2f,%.2f,%.2f)\n", sys->boundary.radius,sys->boundary.center[0],sys->boundary.center[1],sys->boundary.center[2]);
    break;
  case PERIODIC_BOUNDARY :
    lprintf("  Periodic Boundary Condition\n");
    /*
    lprintf("BOX (%.2f,%.2f,%.2f)\n",sys->boundary.box[0],sys->boundary.box[1],sys->boundary.box[2]);
    */
    break;
  }
  lprintf("\n");
}
