/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifndef _MD_SYSTEM_H_INCLUDED
#define _MD_SYSTEM_H_INCLUDED

/* #define RIGID_MD */

#ifndef _MISC_H_INCLUDED
#include "misc.h"
#endif

#ifndef _ATOM_H_INCLUDED
#include "atom.h"
#endif

#ifndef _BOND_H_INCLUDED
#include "bond.h"
#endif

#ifndef _ANGLE_H_INCLUDED
#include "angle.h"
#endif

#ifndef _BOUNDARY_H_INCLUDED
#include "boundary.h"
#endif

#ifndef _LINKED_CELL_H_INCLUDED
#include "linked_cell.h"
#endif

#ifndef _NONBOND_H_INCLUDED
#include "nonbond.h"
#endif

#ifndef _DIHEDRAL_H_INCLUDED
#include "dihedral.h"
#endif

#ifndef _RIGID_MOL_H_INCLUDED
#include "rigid_mol.h"
#endif

#ifndef _EWALD_H_INCLUDED
#include "ewald.h"
#endif

#ifndef _FIT_H_INCLUDED
#include "fit.h"
#endif

#ifndef _EXTRA_POT_H_INCLUDED
#include "extra_pot.h"
#endif

#ifndef _RATTLE_H_INCLUDED
#include "rattle.h"
#endif

#define TCT_X        (0x1)      /*  00000000 00000001 */
#define TCT_V        (0x2)      /*  00000000 00000010 */
#define TCT_BOUNDARY (0x4)      /*  00000000 00000100 */
#define TCT_EXT      (0x8)      /*  00000000 00001000 */
#define TCT_LAMBDA   (0x10)     /*  00000000 00010000 */
#define TCT_RMOL     (0x20)     /*  00000000 00100000 */
#define TCT_AMBER_BOUNDARY (0x40)     /*  00000000 01000000 */
#define TCT_ALL      (0xff)     /*  00000000 11111111 */

#define TRJ_X        (0x1)         /*  00000000 00000001 */
#define TRJ_V        (0x2)         /*  00000000 00000010 */
#define TRJ_BOUNDARY (0x4)         /*  00000000 00000100 */
#define TRJ_AMBER_BOUNDARY (0x40)  /*  00000000 01000000 */
#define TRJ_ALL      (0xff)        /*  00000000 11111111 */

#define CRD_MAJOR_VERSION  1
#define CRD_MINOR_VERSION  2

#define BOND_ENE       0
#define ANGLE_ENE      1
#define DIHED_ENE      2
#define VDW_ENE        3
#define ELEC_ENE       4
#define VDW14_ENE      5
#define ELEC14_ENE     6
#define HBOND_ENE      7
#define FMM_ENE        8
#define BOUNDARY_ENE   9
#define EXTRA_ENE     10
#define UB_ENE        11
#define IMPR_ENE      12
#define CMAP_ENE      13
#define EWREC_ENE     14
#define EWCOR_ENE     15
#define N_ENE         16


typedef enum {
  ATOM_BASED_SCALE, MOL_BASED_SCALE
} SCALE_SYSTEM_METHOD;

typedef struct _s_X_ENE_FORCE {
  double ene[N_ENE];
  double vdw1, elec1, hbond1, vdw2, elec2, hbond2, vdw3, elec3, hbond3;
  double potential, total_ene;

  int natom;
  VEC *x, *f, *f0, *f1, *f2, *f3;
  double box[3], min[3];
  double virial0[3], virial1[3], virial2[3], virial3[3];
} X_ENE_FORCE;

typedef struct _s_STATISTICS {
  int count;
  
  double potential;
  double kene;
  double total_ene;
  double Pint, V, HV, PintV[6];
  double time, time_total_ene;
  
} STATISTICS;

#define MDAT_AMBER  0
#define MDAT_CHARMM 1

typedef struct _s_MD_SYSTEM {
  char mdat_fname[MAX_FILE_NAME_LEN];
  char title[81];
  int mdat_major_version, mdat_minor_version, mdat_format;
  
  ATOM_DATA     atom;
  BOND_DATA     bond;
  ANGLE_DATA    angle;
  DIHEDRAL_DATA dihed;

  NONBOND_LIST  nonbond;

  /* NONBOND LIST FOR MULTI TIME STEP */
  NONBOND_LIST nonbond1, nonbond2, nonbond3;

  RATTLE rattle;

  /* file control */
  int read_mdat_flag;
  int read_crd_flag;

  int current_step;
  double temperature;
  double dt, dt_ps;
  double current_time;
  /* double scnb, scee; */
  double diel;
  /* double cutoff, cutoff_list, nb_smooth_len, nb_list_len; */
  VEC gc, gcv, wp;

  /** Weak Coupling **/
  int weak_coupling_T_flag;
  double target_temperature, tau_t, delta_T;
  int weak_coupling_P_flag;
  double comp, tau_p;

  /** Constraint Temperature Control **/
  int constraint_T_flag;

  int rescaling_T_flag;
  
  int gradual_change_T_step;
  double target_temperature0,target_temperature1;

  /** Hybrid MC **/
  int Hybrid_MC, Hybrid_MC_accept, Hybrid_MC_reject;
  int Hybrid_MC_P, Hybrid_MC_P_accept, Hybrid_MC_P_reject;
  double Hybrid_MC_dV;
  X_ENE_FORCE Hybrid_MC_Stored_Data;
  /** DEBUG **/
  double Pint_k, Pint_v;

  /* for Nose-Hoover PT constant integrator */
#define MAX_CHAIN_T   10
  int Ex_System_P_flag, Ex_System_T_flag;
  int Ex_System_n_P_var;
  int n_ex_system, n_chain_T;
  int degree_of_freedom, degree_of_freedom_tr;
  int degree_of_freedom_arr[MAX_EX_SYSTEM];
  int atom_ex_system[MAX_EX_SYSTEM][2];
  double kene_tr_arr[MAX_EX_SYSTEM][3][3], kene_rot_arr[MAX_EX_SYSTEM];
  double kene_arr[MAX_EX_SYSTEM], temperature_arr[MAX_EX_SYSTEM];
  double kene_tr, kene_rot;
  double eta[MAX_EX_SYSTEM][MAX_CHAIN_T], eta_v[MAX_EX_SYSTEM][MAX_CHAIN_T];
  double e_eta, e_eta_v;
  double Q[MAX_EX_SYSTEM][MAX_CHAIN_T];
  
  double logv_v, Vg[3][3], W, Pint, Pex, PV, PV_v, PintV[6], gamma_ex;
  int pc_ex_system;
  SCALE_SYSTEM_METHOD scale_system_method;

  /* FOR RESPA */
  int rRESPA_far_near_ewald;
  int max_step0, max_step1, max_step2;
  double dt1_ps, dt2_ps, dt3_ps;
  double dt1, dt2, dt3;
  double vdw1, elec1, hbond1, vdw2, elec2, hbond2, vdw3, elec3, hbond3;
  double cutoff_1, cutoff_2, cutoff_3;
  double cutoff2_1, cutoff2_2, cutoff2_3;

  /* CURRENT ENERGY VARIABLES */
  double ene[N_ENE];
  double potential;
  double kene;
  double total_ene;

  /* Check of conservation of total energy */
  double start_total_ene, total_ene_drift;
  double total_ene_fluct, total_ene_fluct_ave, total_ene_fluct_var;
  double start_total_ene_block;
  double total_ene_ave_block, total_ene_var_block;
  double kin_ene_ave_block, kin_ene_var_block;
  double total_ene_R_ave, total_ene_R_var;
  double total_ene_stime;
  int total_ene_n_sample, total_ene_n_ave;
  
  int sample_step;
  STATISTICS ave, ave2;

  /* CPU TIMES */
  double time_bond, time_angle, time_dihed, time_nonbond;
  double time_nonbond1, time_nonbond2, time_nonbond3;
  double time_ew_dir, time_ew_rec, time_ew_cor;
  double time_boundary;
  double time_fmm;
  double time_comm1, time_comm2, time_comm3, time_comm4;
  double time_rattle;

  /* print out routines */
  int    print_out_step;
  int    prop_out_step, trj_out_step, trj_n_atom, gzip_trj;
  int    prop_out_flag, trj_out_flag, trj_out_type, trj_wrap_mol;
  int    trj_xst_flag;
  FILE   *prop_fp, *trj_fp, *trj_xst_fp;
  char   prop_out_fname[MAX_FILE_NAME_LEN], trj_out_fname[MAX_FILE_NAME_LEN];
  char   trj_xst_fname[MAX_FILE_NAME_LEN];
  /* for DCD trajectory */
  float  *float_arr;
  int    n_float_arr;
  int  ene_long_format;
  int  pdb_wrap_mol_flag;

  /* rigid mol */
  int rigid_mol_flag;
  int n_flex_atom;
  int n_fixed_atom;
  RMOL_DATA  rigid_mol;

  int wat_start_atom, wat_start_res, wat_nmol;

  /* ewald */
  EWALD ewald;

  /* fmm */
  int fmm_flag;

  /* boundary condition */
  BOUNDARY boundary;

  LINKED_CELL linked_cell;
  
  /* extra_pot */
  EXTRA_POT extra_pot;

  int remove_momentum, remove_momentum_step, remove_momentum_atom;
  double remove_momentum_w;
  int remove_total_force;
  
#ifdef MPI_RDMD
  int rdmd_sync_step;
#endif
  
} MD_SYSTEM;

#define PROP_TIME          (0x1)
#define PROP_TEMPERATURE   (0x2)
#define PROP_TOTAL_ENE     (0x4)
#define PROP_POTENTIAL     (0x8)
#define PROP_KINETIC       (0x10)
#define PROP_ENE_DETAIL    (0x20)
#define PROP_EP_ENE        (0x40)
#define PROP_EX_SYS        (0x80)
#define PROP_DENSITY       (0x100)
#define PROP_PV_ISOTROPIC  (0x200)
#define PROP_PV_DIAGONAL   (0x400)
#define PROP_PV_OFFDIAG    (0x800)
#define PROP_ATOM_ENE      (0x1000)
#define PROP_CLEAN         (0x2000)

#define  TRJ_OUT_TXT  0
#define  TRJ_OUT_BIN  1
#define  TRJ_OUT_DCD  2

#ifndef _PT_CTRL_H_INCLUDED
#include "pt_ctrl.h"
#endif

#ifndef _EXTRA_POT_FUNC_H_INCLUDED
#include "extra_pot_func.h"
#endif


/* initialize and finalize routine */
void MD_SYSTEM_init(MD_SYSTEM *sys);
void MD_SYSTEM_finalize(MD_SYSTEM *sys);

/* file io routine */
void MD_SYSTEM_write_pdb_file(MD_SYSTEM *sys, char *fname, int group_no, int centering_solute);
void MD_SYSTEM_wrap_molecules(MD_SYSTEM *sys, int rigid_flag, int center_flag);
int  MD_SYSTEM_check_mdat_crd(MD_SYSTEM *sys);
unsigned int MD_SYSTEM_trjstr_to_flag(char *crdstr);
void MD_SYSTEM_flag_to_trjstr(char crdstr[10], unsigned int flag);
void MD_SYSTEM_xst_out(MD_SYSTEM *sys, int step);
void MD_SYSTEM_trj_out(MD_SYSTEM *sys);
void MD_SYSTEM_prop_header(MD_SYSTEM *sys);
void MD_SYSTEM_prop_out(MD_SYSTEM *sys);
void MD_SYSTEM_write_force(MD_SYSTEM *sys, char *fname);

/* setting property */
int MD_SYSTEM_set_dt(MD_SYSTEM *sys, double dt_ps);
int MD_SYSTEM_set_current_time(MD_SYSTEM *sys, double current_time);
int MD_SYSTEM_set_scnb(MD_SYSTEM *sys, double scnb);
int MD_SYSTEM_set_scee(MD_SYSTEM *sys, double scee);
int MD_SYSTEM_set_diel(MD_SYSTEM *sys, double diel);
void MD_SYSTEM_set_fixed_atom(MD_SYSTEM *sys, STR_LIST *group_str, int group_no);
void MD_SYSTEM_fixed_atom_correct_virial(MD_SYSTEM *sys);
void MD_SYSTEM_set_rigid_mol(MD_SYSTEM *sys, char *fname);
void MD_SYSTEM_check_rigid_mol(MD_SYSTEM *sys);
void MD_SYSTEM_set_trj_out(MD_SYSTEM *sys, char *fname, int step, int res_no, int flag, int type,
			   int total_step, int gzipped, int trj_wrap_mol, int trj_xst_flag);
void MD_SYSTEM_print_trj_info(MD_SYSTEM *sys);
int MD_SYSTEM_set_prop_out(MD_SYSTEM *sys, char *fname, int step, int flag);
int MD_SYSTEM_set_print_out_step(MD_SYSTEM *sys, int step);
void MD_SYSTEM_set_solute_mol(MD_SYSTEM *sys, int mol_no);
void MD_SYSTEM_set_solute_residue(MD_SYSTEM *sys, int res_no);
int MD_SYSTEM_set_density(MD_SYSTEM *sys, double density);
int MD_SYSTEM_set_box(MD_SYSTEM *sys, double box[3]);
void MD_SYSTEM_set_boxv(MD_SYSTEM *sys, double a[3], double b[3], double c[3]);
void MD_SYSTEM_shift(MD_SYSTEM *sys, double shift[3]);
void MD_SYSTEM_shift_center_solute(MD_SYSTEM *sys);
int MD_SYSTEM_set_initial_velocity(MD_SYSTEM *sys, double T);
void MD_SYSTEM_set_remove_momentum(MD_SYSTEM *sys, int flag, int step);

/* momentum */
void MD_SYSTEM_degree_of_freedom(MD_SYSTEM *sys);
void MD_SYSTEM_calc_momentum(MD_SYSTEM *sys);
void MD_SYSTEM_remove_momentum(MD_SYSTEM *sys);
void MD_SYSTEM_remove_momentum_all(MD_SYSTEM *sys);
void MD_SYSTEM_remove_momentum_partial(MD_SYSTEM *sys);
void MD_SYSTEM_remove_total_force(MD_SYSTEM *sys);
void MD_SYSTEM_remove_total_force_all(MD_SYSTEM *sys);
void MD_SYSTEM_remove_total_force_partial(MD_SYSTEM *sys);

/* print */
void MD_SYSTEM_clear_time(MD_SYSTEM *sys);
void MD_SYSTEM_print_energy(MD_SYSTEM *sys);
void MD_SYSTEM_print_md_status(MD_SYSTEM *sys, int step);
void MD_SYSTEM_print_time(MD_SYSTEM *sys, double total);
void MD_SYSTEM_print_time(MD_SYSTEM *sys, double total);
void STATISTICS_clear(STATISTICS *st);
void STATISTICS_sample(STATISTICS *st, MD_SYSTEM *sys);
void STATISTICS_sample2(STATISTICS *st, MD_SYSTEM *sys);
static double safe_sqrt(double c);
void STATISTICS_deviation(STATISTICS *st, STATISTICS *ave, MD_SYSTEM *sys);
void STATISTICS_average(STATISTICS *st);
void MD_SYSTEM_clear_statistics(MD_SYSTEM *sys);
void MD_SYSTEM_sample_statistics(MD_SYSTEM *sys);
void MD_SYSTEM_collect_statistics(MD_SYSTEM *sys);

/* calculation of kinetic energy, potential energy, total energy */
void MD_SYSTEM_calc_force(MD_SYSTEM *sys);

void MD_SYSTEM_calc_Pint(MD_SYSTEM *sys);
void MD_SYSTEM_make_nonbond_list(MD_SYSTEM *sys);
void MD_SYSTEM_calc_kene(MD_SYSTEM *sys);
void MD_SYSTEM_calc_kene_full(MD_SYSTEM *sys);
double MD_SYSTEM_calc_mol_kene(MD_SYSTEM *sys);
double MD_SYSTEM_atom_or_mol_kene(MD_SYSTEM *sys);
void MD_SYSTEM_sum_potential(MD_SYSTEM *sys);
void MD_SYSTEM_sum_total_energy(MD_SYSTEM *sys);

/*** Hybrid MC ***/
int MD_SYSTEM_set_Hybrid_MC(MD_SYSTEM *sys, double T, int step,
			    double P, int P_step, double dV);
void MD_SYSTEM_copy_to_x_ene_force(MD_SYSTEM *sys, X_ENE_FORCE *xx,
				   int rRESPA_flag);
void MD_SYSTEM_restore_from_x_ene_force(MD_SYSTEM *sys, X_ENE_FORCE *xx,
					int rRESPA_flag);
void MD_SYSTEM_Hybrid_MC_init(MD_SYSTEM *sys);
void MD_SYSTEM_Hybrid_MC_setup_before_run(MD_SYSTEM *sys, int rRESPA_flag);
void MD_SYSTEM_Hybrid_MC(MD_SYSTEM *sys, int P_flag, int rRESPA_flag);

/* test routines */
void MD_SYSTEM_test_force(MD_SYSTEM *sys, int iatom, double dx, double dy, double dz);
void MD_SYSTEM_test_force_all(MD_SYSTEM *sys, double dx);
void MD_SYSTEM_test_pert(MD_SYSTEM *sys, double l, double dl);
void MD_SYSTEM_test_momentum(MD_SYSTEM *sys);
void MD_SYSTEM_test_virial(MD_SYSTEM *sys, int type);
void MD_SYSTEM_check_bonds_angles(MD_SYSTEM *sys);

#define MD_SYSTEM_set_ene_out_step(s,d)  ((s)->ene_out_step=d)
#define MD_SYSTEM_set_ene_log_fp(s,d)  ((s)->log_fp=d)
#define MD_SYSTEM_set_target_temperature(s,d)  ((s)->target_temperature=d)
#define MD_SYSTEM_set_sampling_flag(s,d)  ((s)->sampling_flag=d)
#define MD_SYSTEM_set_const_temp_flag(s,d)  ((s)->const_temp_flag=d)
#define MD_SYSTEM_set_tau_t(s,d)  ((s)->tau_t=d)

#ifdef MD_SYSTEM_C
char *bin_header = "MD_SYSTEM_BIN_OUT";
int bin_version[2] = { 1, 1 };
char *trj_header = "MD_SYSTEM_TRJ_OUT";
int trj_version[2] = { 1, 0 };
#else /* MD_SYSTEM_C */
extern char *bin_header;
extern int bin_version[2];
extern char *trj_header;
extern int trj_version[2];
#endif


void MD_SYSTEM_md_check(MD_SYSTEM *sys, int step);
void MD_SYSTEM_scale_system(MD_SYSTEM *sys, double scale);
void MD_SYSTEM_scale_system_anisotropic(MD_SYSTEM *sys, double scale[3]);
void MD_SYSTEM_scale_velocity(MD_SYSTEM *sys, double scale_v);


/* mdat_file.c */
void MDAT_read_file(MD_SYSTEM *sys, char *fname);
int MDAT_read_file_master(MD_SYSTEM *sys, char *fname);
int MDAT_read_atom_data(ATOM_DATA *ad, FILE *fp, int mdat_format);
int MDAT_read_bond_data(BOND_DATA* bd, ATOM_DATA *ad, FILE *fp);
int MDAT_read_angle_data(ANGLE_DATA *ad, FILE *fp);
int MDAT_read_dihedral_data(DIHEDRAL_DATA* dd, FILE *fp);
int MDAT_read_cmap_data(DIHEDRAL_DATA* dd, FILE *fp);
int MDAT_read_rmol_data(RMOL_DATA *md, ATOM_DATA *ad, FILE *fp);
int MDAT_read_boundary_data(BOUNDARY *bc, FILE *fp);

void MDAT_read_file_comm(MD_SYSTEM *sys, int err);
void MDAT_comm_atom_data(ATOM_DATA *ad, int mdat_format);
void MDAT_comm_bond_data(BOND_DATA* bd, ATOM_DATA *ad);
void MDAT_comm_angle_data(ANGLE_DATA *ad);
void MDAT_comm_dihedral_data(DIHEDRAL_DATA* dd);
void MDAT_comm_cmap_data(DIHEDRAL_DATA* dd);
void MDAT_comm_rmol_data(RMOL_DATA *md, ATOM_DATA *ad);
void MDAT_comm_boundary_data(BOUNDARY *bc);

void MDAT_print_info(MD_SYSTEM *sys);

/* crd_file.c */
int CRD_read_file(MD_SYSTEM *sys, char *crdname, unsigned int user_type);
int CRD_read_file_master(MD_SYSTEM *sys, char *crdname, unsigned int user_type);
int CRD_write_file(MD_SYSTEM *sys, char *crdname, unsigned int type);
int CRD_read_bin_file(MD_SYSTEM *sys, char *fname);
int CRD_write_bin_file(MD_SYSTEM *sys, char *fname);
unsigned int CRD_crdstr_to_flag(char *crdstr);
void CRD_flag_to_crdstr(char crdstr[10], unsigned int flag);

#endif  /* _MD_SYSTEM_H_INCLUDED */
