/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#define MD_SYSTEM_C

#include "md_system.h"

#ifdef MPI_RDMD
#include "parallel.h"
#endif
#ifdef MPI_SDMD
#include "parallel.h"
#include "sdmd.h"
#endif

/********************************************************************/
/*   initialize and finalize routines                               */
/********************************************************************/

void MD_SYSTEM_init(MD_SYSTEM *sys)
{
  ATOM_DATA_init(&sys->atom);
  BOND_DATA_init(&sys->bond);
  ANGLE_DATA_init(&sys->angle);
  DIHEDRAL_DATA_init(&sys->dihed);
  RMOL_DATA_init(&(sys->rigid_mol));
  BOUNDARY_init(&(sys->boundary));

  NONBOND_LIST_init(&sys->nonbond, &(sys->boundary));
  LINKED_CELL_init(&sys->linked_cell);
  EWALD_init(&sys->ewald);
  EP_init(&sys->extra_pot);
  RATTLE_init(&sys->rattle);

  timer_init();

  sys->read_mdat_flag = 0;
  sys->read_crd_flag = 0;

  sys->current_time = 0.0;
  sys->diel = 1.0;
  MD_SYSTEM_set_dt(sys, 0.001);    /* 1 fs */

  /* rRESPA init
  NONBOND_LIST_init(&sys->nonbond1, &(sys->boundary));
  NONBOND_LIST_init(&sys->nonbond2, &(sys->boundary));
  NONBOND_LIST_init(&sys->nonbond3, &(sys->boundary));
  MD_SYSTEM_set_rRESPA(sys, 6, 2, 4, 4.3, 6.9, 9.0, 0.5, 1.0, 1);
  sys->nonbond1.outside_flag = 0;
  sys->nonbond2.outside_flag = 0;
  sys->nonbond3.outside_flag = 1;
  sys->rRESPA_far_near_ewald = 0;
  */
  /* Hybrid_MC 
  sys->Hybrid_MC = 0;
  sys->Hybrid_MC_P = 0;
  MD_SYSTEM_Hybrid_MC_init(sys);
  */

  /* Extended System Temperature and/or Pressure Control */
  sys->Ex_System_P_flag =  sys->Ex_System_T_flag = 0;
  sys->n_ex_system = 1;
  sys->n_chain_T = 0;
  sys->target_temperature = 300.0;
  sys->delta_T = 0.0;
  sys->gamma_ex = 0.0;
  /* sys->scale_system_method = MOL_BASED_SCALE; */
  sys->scale_system_method = ATOM_BASED_SCALE;

  /* Weak Coupling Temperature Control */
  sys->weak_coupling_T_flag = 0;
  sys->weak_coupling_P_flag = 0;
  sys->tau_t = 0.4;                     /* 0.4 ps */

  /* Constraint Temperature Control */
  sys->constraint_T_flag = 0;

  /* rescaling Temperature Control */
  sys->rescaling_T_flag = 0;
  
  /* gradual change T */
  sys->gradual_change_T_step = 0;
  sys->target_temperature0 = sys->target_temperature1 = 0.0;

  /* CPU TIMES */
  MD_SYSTEM_clear_time(sys);

  /* EWALD */
  sys->ewald.flag = FLAG_NO_EWALD;

  /* output control */
  sys->print_out_step = 1;
  sys->prop_out_step = sys->trj_out_step = 0;
  sys->prop_out_flag = 0;
  sys->trj_out_flag = TRJ_OUT_TXT;
  sys->trj_xst_flag = 0;
  sys->sample_step = 1;
  sys->prop_fp = sys->trj_fp = sys->trj_xst_fp = NULL;
  sys->ene_long_format = 0;
  sys->pdb_wrap_mol_flag = 0;

  sys->remove_momentum = 0;
  sys->remove_momentum_step = 1000;
  sys->remove_total_force = 0;

  sys->n_fixed_atom = 0;
  
#ifdef MPI_RDMD
  sys->rdmd_sync_step = 1000;
#endif  
}

void MD_SYSTEM_finalize(MD_SYSTEM *sys)
{
  if (sys->read_mdat_flag) {
    ATOM_DATA_finalize(&(sys->atom));
    BOND_DATA_finalize(&(sys->bond));
    ANGLE_DATA_finalize(&(sys->angle));
    DIHEDRAL_DATA_finalize(&(sys->dihed));
    RMOL_DATA_finalize(&(sys->rigid_mol));
  }
  if (sys->prop_fp) {
    fclose(sys->prop_fp);
  }
  if (sys->trj_fp) {
    fclose(sys->trj_fp);
  }
  if (sys->trj_xst_fp) {
    fclose(sys->trj_xst_fp);
  }
}

/********************************************************************/
/*   file io routines                                               */
/********************************************************************/


void MD_SYSTEM_write_pdb_file(MD_SYSTEM *sys, char *fname, int group_no, int centering_solute)
{
  int i;
  VEC *tmp;
  ATOM_DATA *ad;
  BOUNDARY *bc;
  FILE *fp;

  ad = &sys->atom;
  bc = &sys->boundary;

  if ((fp = safe_fopen(fname,"w")) == NULL) {
    lprintf("  ERROR: %s: No such file\n", fname);
    return;
  }
  
  if (bc->type == PERIODIC_BOUNDARY || centering_solute) {
    double a,b,c,alpha, beta, gamma;
    for (i=0;i<ad->natom;i++) {
      ad->px[i]=ad->x[i];
    }
    tmp = ad->x;
    ad->x = ad->px;
    if (centering_solute) {
      MD_SYSTEM_shift_center_solute(sys);
    }
    if (bc->type == PERIODIC_BOUNDARY) {
      MD_SYSTEM_wrap_molecules(sys, 0, sys->pdb_wrap_mol_flag);
      BOUNDARY_calc_abc(bc, &a, &b, &c, &alpha, &beta, &gamma);
      fprintf(fp,"CRYST1%9.3f%9.3f%9.3f%7.2f%7.2f%7.2f P 1           1\n",
	      a,b,c,alpha,beta,gamma);
    }
  }
  ATOM_DATA_write_pdb_file(ad, fp, group_no);
  fclose(fp);
  
  if (bc->type == PERIODIC_BOUNDARY) {
    ad->x = tmp;
  }
}

void MD_SYSTEM_wrap_molecules(MD_SYSTEM *sys, int rigid_flag, int center_flag)
{
  BOUNDARY *bc;
  ATOM_DATA *ad;
  RMOL_DATA *md;
  int i, j, iatom, imol;
  VEC center, offset, frac;
  double w;

  bc = &sys->boundary;
  if (bc->type != PERIODIC_BOUNDARY) return;
  ad = &sys->atom;
  md = &sys->rigid_mol;

  for (i=0;i<ad->nmol;i++) {
    /*if (sys->pdb_wrap_mol_flag == 1) {*/
    if (center_flag) {
      center.x = center.y = center.z = w = 0.0;
      for (j=0;j<ad->mol[i].natom;j++) {
	iatom = ad->mol[i].start_atom + j;
	center.x += ad->x[iatom].x;
	center.y += ad->x[iatom].y;
	center.z += ad->x[iatom].z;
	w += ad->w[iatom];
      }
      center.x /= w;
      center.y /= w;
      center.z /= w;
    } else {
      /* decision by the first atom coodinates */
      center = ad->x[ad->mol[i].start_atom];
    }

    
    frac.x = VEC_MUL_MAT_X(center,bc->recip);
    frac.y = VEC_MUL_MAT_Y(center,bc->recip);
    frac.z = VEC_MUL_MAT_Z(center,bc->recip);

    if (bc->origin_flag == 1) {
      frac.x += 0.5;
      frac.y += 0.5;
      frac.z += 0.5;
    }

    frac.x = floor(frac.x);
    frac.y = floor(frac.y);
    frac.z = floor(frac.z);
    
    offset.x = VEC_MUL_MAT_X(frac,bc->boxv);
    offset.y = VEC_MUL_MAT_Y(frac,bc->boxv);
    offset.z = VEC_MUL_MAT_Z(frac,bc->boxv);
    
    for (j=0;j<ad->mol[i].natom;j++) {
      iatom = ad->mol[i].start_atom + j;
      ad->x[iatom].x -= offset.x;
      ad->x[iatom].y -= offset.y;
      ad->x[iatom].z -= offset.z;

      if (rigid_flag &&
	  (ad->ex[iatom].flag & ATOM_RIGID) &&
	  (ad->ex[iatom].flag & ATOM_PARENT)) {
	imol = ad->ex[iatom].parent;
	md->mol[imol].rg.x -= offset.x;
	md->mol[imol].rg.y -= offset.y;
	md->mol[imol].rg.z -= offset.z;
      }
    }
  }
}

/* if already read mdat and crd return 0, otherwise return 1 */
int MD_SYSTEM_check_mdat_crd(MD_SYSTEM *sys)
{
  if (sys->read_mdat_flag && sys->read_crd_flag) return 0;
  lprintf("ERROR: mdat file or crd file has not read yet.\n");
  return 1;
}

unsigned int MD_SYSTEM_trjstr_to_flag(char *crdstr)
{
  char *p;
  unsigned int flag = 0;

  for (p=crdstr;*p!='\0';p++) {
    switch(*p) {
    case 'X':
      flag |= TRJ_X; break;
    case 'V':
      flag |= TRJ_V; break;
    case 'B':
      flag |= TRJ_BOUNDARY; break;
    case 'b':
      flag |= TRJ_AMBER_BOUNDARY; break;
    }
  }
  
  return flag;
}

void MD_SYSTEM_flag_to_trjstr(char crdstr[10], unsigned int flag)
{
  int i=0;
  if (flag & TRJ_X)               crdstr[i++] = 'X';
  if (flag & TRJ_V)               crdstr[i++] = 'V';
  if (flag & TRJ_BOUNDARY)        crdstr[i++] = 'B';
  if (flag & TRJ_AMBER_BOUNDARY)  crdstr[i++] = 'b';
  crdstr[i] = '\0';
}

void MD_SYSTEM_xst_out(MD_SYSTEM *sys, int step)
{
  int i, j;

  fprintf(sys->trj_xst_fp, "%d", step);
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      if (sys->boundary.boxv[i][j] == 0.0) {
	fprintf(sys->trj_xst_fp, " 0");
      } else {
	fprintf(sys->trj_xst_fp, " %.4f",sys->boundary.boxv[i][j]);
      }
    }
  }
  for (i=0;i<9;i++) {
    fprintf(sys->trj_xst_fp, " 0");
  }
  fprintf(sys->trj_xst_fp, "\n");
}

void MD_SYSTEM_trj_out(MD_SYSTEM *sys)
{
  int i;
  int out_int;
  VEC *tmp;
  ATOM_DATA *ad;

#ifdef MPI
  if (!mpi.master) return;
#endif
  ad = &sys->atom;

  if (sys->trj_wrap_mol) {
    for (i=0;i<ad->natom;i++) {
      ad->px[i]=ad->x[i];
    }
    tmp = ad->x;
    ad->x = ad->px;
    MD_SYSTEM_wrap_molecules(sys, 0, 0);
  }

  if (sys->trj_out_flag == TRJ_OUT_TXT) {
    if (sys->trj_out_type & TRJ_X)
      for (i=0;i<sys->trj_n_atom;i++)
	fprintf(sys->trj_fp, "%.3f %.3f %.3f\n",
		sys->atom.x[i].x, sys->atom.x[i].y, sys->atom.x[i].z);
    if (sys->trj_out_type & TRJ_V)
      for (i=0;i<sys->trj_n_atom;i++)
	fprintf(sys->trj_fp, "%.3f %.3f %.3f\n",
		sys->atom.v[i].x, sys->atom.v[i].y, sys->atom.v[i].z);
    if (sys->trj_out_type & TRJ_BOUNDARY) {
      for (i=0;i<3;i++)
	fprintf(sys->trj_fp, "%.3f %.3f %.3f\n",
		sys->boundary.boxv[i][0], sys->boundary.boxv[i][1],
		sys->boundary.boxv[i][2]);
    }
    if (sys->trj_out_type & TRJ_AMBER_BOUNDARY) {
      fprintf(sys->trj_fp, "%.3f %.3f %.3f\n",
	      sys->boundary.boxv[0][0], sys->boundary.boxv[1][1], sys->boundary.boxv[2][2]);
    }
  } else if (sys->trj_out_flag == TRJ_OUT_BIN) {
    if (sys->trj_out_type & TRJ_X)
      fwrite(&(sys->atom.x[0]), sizeof(VEC), sys->trj_n_atom, sys->trj_fp);
    if (sys->trj_out_type & TRJ_V)
      fwrite(&(sys->atom.v[0]), sizeof(VEC), sys->trj_n_atom, sys->trj_fp);
    if (sys->trj_out_type & TRJ_BOUNDARY || sys->trj_out_type & TRJ_AMBER_BOUNDARY) {
      fwrite(sys->boundary.boxv, sizeof(double), 9, sys->trj_fp);
    }
  } else if (sys->trj_out_flag == TRJ_OUT_DCD) {
    if (sys->trj_out_type & TRJ_BOUNDARY || sys->trj_out_type & TRJ_AMBER_BOUNDARY) {
      double tmp[6];
      out_int = 48;
      fwrite(&out_int,   sizeof(int), 1, sys->trj_fp);
      tmp[0]=sys->boundary.boxv[0][0];
      tmp[1]=sys->boundary.boxv[1][0];
      tmp[2]=sys->boundary.boxv[1][1];
      tmp[3]=sys->boundary.boxv[2][0];
      tmp[4]=sys->boundary.boxv[2][1];
      tmp[5]=sys->boundary.boxv[2][2];
      fwrite(tmp,   sizeof(double), 6, sys->trj_fp);
      fwrite(&out_int,   sizeof(int), 1, sys->trj_fp);
    }
    out_int = sys->trj_n_atom*4;
    for (i=0;i<sys->trj_n_atom;i++) sys->float_arr[i] = ad->x[i].x;
    fwrite(&out_int,   sizeof(int), 1, sys->trj_fp);
    fwrite(sys->float_arr, sizeof(float), sys->trj_n_atom, sys->trj_fp);
    fwrite(&out_int,   sizeof(int), 1, sys->trj_fp);
    
    for (i=0;i<sys->trj_n_atom;i++) sys->float_arr[i] = ad->x[i].y;
    fwrite(&out_int,   sizeof(int), 1, sys->trj_fp);
    fwrite(sys->float_arr, sizeof(float), sys->trj_n_atom, sys->trj_fp);
    fwrite(&out_int,   sizeof(int), 1, sys->trj_fp);
    
    for (i=0;i<sys->trj_n_atom;i++) sys->float_arr[i] = ad->x[i].z;
    fwrite(&out_int,   sizeof(int), 1, sys->trj_fp);
    fwrite(sys->float_arr, sizeof(float), sys->trj_n_atom, sys->trj_fp);
    fwrite(&out_int,   sizeof(int), 1, sys->trj_fp);
  }
  
  if (sys->trj_xst_flag) {
    MD_SYSTEM_xst_out(sys, sys->current_step+1);
  }

  if (sys->trj_wrap_mol) {
    ad->x = tmp;
  }
}

void MD_SYSTEM_prop_header(MD_SYSTEM *sys)
{
  int i, j, k;
  i=1;
  fprintf(sys->prop_fp, "#%dTIME        ",i++);

  if (sys->prop_out_flag & PROP_TEMPERATURE)
    fprintf(sys->prop_fp, "%2dTEMPERATURE ",i++);

  if (sys->prop_out_flag & PROP_TOTAL_ENE)
    fprintf(sys->prop_fp, "%2dTOTAL_ENE   ",i++);

  if (sys->prop_out_flag & PROP_POTENTIAL)
    fprintf(sys->prop_fp, "%2dPOTENTIAL   ",i++);

  if (sys->prop_out_flag & PROP_KINETIC)
    fprintf(sys->prop_fp, "%2dKINETIC     ",i++);

  if (sys->prop_out_flag & PROP_ENE_DETAIL) {
    fprintf(sys->prop_fp, "%2dBOND        ",i++);
    fprintf(sys->prop_fp, "%2dANGLE       ",i++);
    fprintf(sys->prop_fp, "%2dANGLE_UB    ",i++);
    fprintf(sys->prop_fp, "%2dDIHEDRAL    ",i++);
    fprintf(sys->prop_fp, "%2dVDW14       ",i++);
    fprintf(sys->prop_fp, "%2dELEC14      ",i++);
#ifdef HBOND    
    fprintf(sys->prop_fp, "%2dHBOND       ",i++);
#endif    
    fprintf(sys->prop_fp, "%2dVDW         ",i++);
    fprintf(sys->prop_fp, "%2dELEC        ",i++);
    fprintf(sys->prop_fp, "%2dBOUNDARY    ",i++);
    if (sys->fmm_flag) {
      fprintf(sys->prop_fp, "%2dFMM         ", i++);
    }
  }
  
  if (sys->prop_out_flag & PROP_EX_SYS) {
    if (sys->Ex_System_T_flag) {
      for (j=0;j<sys->n_ex_system;j++) {
	for (k=0;k<sys->n_chain_T;k++) {
	  fprintf(sys->prop_fp, "%2dETA[%d][%d]   ", i++, j, k);
	  fprintf(sys->prop_fp, "%2dETAV[%d][%d]  ", i++, j, k);
	}
      }
    }

    if (sys->Ex_System_P_flag == 1) {
      fprintf(sys->prop_fp, "%2dLOGV_V      ", i++);
    } else if (sys->Ex_System_P_flag >= 1) {
      for (j=0;j<3;j++) {
	for (k=j;k<3;k++) {
	  fprintf(sys->prop_fp, "%2dVg[%d][%d]    ", i++, j, k);
	}
      }
    }
  }
  
  if (sys->prop_out_flag & PROP_DENSITY) {
    fprintf(sys->prop_fp, "%2dDENSITY     ", i++);
  }
  
  if (sys->prop_out_flag & PROP_PV_ISOTROPIC) {
    fprintf(sys->prop_fp, "%2dPRESSURE    ", i++);
  }
  if (sys->prop_out_flag & (PROP_PV_DIAGONAL | PROP_PV_OFFDIAG)) {
    fprintf(sys->prop_fp, "%2dPxx         ", i++);
    fprintf(sys->prop_fp, "%2dPyy         ", i++);
    fprintf(sys->prop_fp, "%2dPzz         ", i++);
  }
  if (sys->prop_out_flag & PROP_PV_OFFDIAG) {
    fprintf(sys->prop_fp, "%2dPxy         ", i++);
    fprintf(sys->prop_fp, "%2dPxz         ", i++);
    fprintf(sys->prop_fp, "%2dPyz         ", i++);
  }
  
  if (sys->prop_out_flag & PROP_PV_ISOTROPIC) {
    fprintf(sys->prop_fp, "%2dVOLUME      ", i++);
  }
  if ( (sys->prop_out_flag & PROP_PV_DIAGONAL) &&
      !(sys->prop_out_flag & PROP_PV_OFFDIAG)) {
    fprintf(sys->prop_fp, "%2dBoxX        ", i++);
    fprintf(sys->prop_fp, "%2dBoxY        ", i++);
    fprintf(sys->prop_fp, "%2dBoxZ        ", i++);
  }
  if (sys->prop_out_flag & PROP_PV_OFFDIAG) {
    /* full flexible cell */
    for (j=0;j<3;j++)
      fprintf(sys->prop_fp,"%2da(%d)        ", i++, j);
    for (j=0;j<3;j++)
      fprintf(sys->prop_fp,"%2db(%d)        ", i++, j);
    for (j=0;j<3;j++)
      fprintf(sys->prop_fp,"%2dc(%d)        ", i++, j);
  }
    
  if (sys->prop_out_flag & PROP_EP_ENE) {
    EP_prop_header(&sys->extra_pot, sys->prop_fp, &i);
  }
  
  if (sys->prop_out_flag & PROP_ATOM_ENE) {
    ATOM_DATA_atom_ene_prop_header(&sys->atom, sys->prop_fp, &i);
  }
  
  fprintf(sys->prop_fp, "\n");
}

void MD_SYSTEM_prop_out(MD_SYSTEM *sys)
{
  int i, j, k;

#ifdef MPI
  if (!mpi.master) {
    
    if (sys->prop_out_flag & PROP_EP_ENE) {
      EP_prop_out(&sys->extra_pot, sys, sys->prop_fp);
    }
    /* for reduce !! */
    ATOM_DATA_atom_ene_prop_out(&sys->atom, NULL);
    return;
  }
#endif  

  fprintf(sys->prop_fp, "%13.6e",sys->current_time);
  
  if (sys->prop_out_flag & PROP_TEMPERATURE)
    fprintf(sys->prop_fp, " %13.6e",sys->temperature);

  if (sys->prop_out_flag & PROP_TOTAL_ENE)
    fprintf(sys->prop_fp, " %13.6e",sys->total_ene);

  if (sys->prop_out_flag & PROP_POTENTIAL)
    fprintf(sys->prop_fp, " %13.6e",sys->potential);

  if (sys->prop_out_flag & PROP_KINETIC)
    fprintf(sys->prop_fp, " %13.6e",sys->kene);
  
  if (sys->prop_out_flag & PROP_ENE_DETAIL) {
#ifdef HBOND    
    fprintf(sys->prop_fp, " %13.6e %13.6e %13.6e %13.6e %13.6e %13.6e %13.6e %13.6e %13.6e",
	    sys->ene[BOND_ENE], sys->ene[ANGLE_ENE],  sys->ene[DIHED_ENE], 
	    sys->ene[VDW14_ENE],sys->ene[ELEC14_ENE], sys->ene[HBOND_ENE],
	    sys->ene[VDW_ENE],  sys->ene[ELEC_ENE],   sys->ene[BOUNDARY_ENE]);
#else
    fprintf(sys->prop_fp, " %13.6e %13.6e %13.6e %13.6e %13.6e %13.6e %13.6e %13.6e %13.6e",
	    sys->ene[BOND_ENE], sys->ene[ANGLE_ENE], sys->ene[UB_ENE],
	    sys->ene[DIHED_ENE], sys->ene[VDW14_ENE],sys->ene[ELEC14_ENE],
	    sys->ene[VDW_ENE],  sys->ene[ELEC_ENE],   sys->ene[BOUNDARY_ENE]);
#endif    
    if (sys->fmm_flag) {
      fprintf(sys->prop_fp, " %13.6e", sys->ene[FMM_ENE]);
    }
  }
  
  if (sys->prop_out_flag & PROP_EX_SYS) {
    if (sys->Ex_System_T_flag) {
      for (j=0;j<sys->n_ex_system;j++) {
	for (k=0;k<sys->n_chain_T;k++) {
	  fprintf(sys->prop_fp, " %13.6e %13.6e",
		  sys->eta[j][k], sys->eta_v[j][k]);
	}
      }
    }
    
    if (sys->Ex_System_P_flag == 1) {
      fprintf(sys->prop_fp, " %13.6e", sys->logv_v);
    } else if (sys->Ex_System_P_flag >= 1) {
      for (j=0;j<3;j++) {
	for (k=j;k<3;k++) {
	  fprintf(sys->prop_fp, " %13.6e", sys->Vg[j][k]);
	}
      }
    }
  }

  if (sys->prop_out_flag & PROP_DENSITY) {
    fprintf(sys->prop_fp, " %13.6e",
	    sys->atom.total_w/sys->boundary.V/(MOL*1.0e-24));
  }

  if (sys->prop_out_flag & PROP_PV_ISOTROPIC) {
    fprintf(sys->prop_fp, " %13.6e", sys->Pint / ATM_TO_KCALMOLANG);
  }

  if (sys->prop_out_flag & (PROP_PV_DIAGONAL | PROP_PV_OFFDIAG)) {
    for (k=0;k<3;k++) {
      fprintf(sys->prop_fp, " %13.6e", sys->PintV[k]/ ATM_TO_KCALMOLANG);
    }
  }
      
  if (sys->prop_out_flag & PROP_PV_OFFDIAG) {
    for (k=3;k<6;k++) {
      fprintf(sys->prop_fp, " %13.6e", sys->PintV[k]/ ATM_TO_KCALMOLANG);
    }
  }

  if (sys->prop_out_flag & PROP_PV_ISOTROPIC) {
    fprintf(sys->prop_fp, " %13.6e", sys->boundary.V);
  }
  
  if ( (sys->prop_out_flag & PROP_PV_DIAGONAL) &&
      !(sys->prop_out_flag & PROP_PV_OFFDIAG)) {
    for (j=0;j<3;j++)
      fprintf(sys->prop_fp," %13.6e", sys->boundary.boxv[j][j]);
  }
  if (sys->prop_out_flag & PROP_PV_OFFDIAG) {
    /* full flexible cell */
    for (k=0;k<3;k++)
      for (j=0;j<3;j++)
	fprintf(sys->prop_fp," %13.6e", sys->boundary.boxv[k][j]);
  }
  
  if (sys->prop_out_flag & PROP_EP_ENE) {
    EP_prop_out(&sys->extra_pot, sys, sys->prop_fp);
  }

  if (sys->prop_out_flag & PROP_ATOM_ENE) {
    ATOM_DATA_atom_ene_prop_out(&sys->atom, sys->prop_fp);
  }
	  
  fprintf(sys->prop_fp, "\n");
}

void MD_SYSTEM_write_force(MD_SYSTEM *sys, char *fname)
{
  int i;
  ATOM_DATA *ad;
  FILE *fp;

#ifdef MPI_SDMD
  SDMD_gather_f(&sys->linked_cell, &sys->atom);
#endif

  ad = &sys->atom;
  fp = safe_fopen(fname,"w");
  for (i=0;i<ad->natom;i++) {
    fprintf(fp, "%e %e %e\n", ad->f[i].x,ad->f[i].y,ad->f[i].z);
  }
  fclose(fp);
}


/********************************************************************/
/*   set properties                                                 */
/********************************************************************/

int MD_SYSTEM_set_dt(MD_SYSTEM *sys, double dt_ps)
{
/* picosecand to internal unit  20.455 which is 10.0 * sqrt(4.184) */
  double c;
  c = 10.0 * sqrt(4.184);
  sys->dt_ps = dt_ps;
  sys->dt = dt_ps * c;
    
  sys->dt1_ps = sys->dt_ps  * sys->max_step0;
  sys->dt2_ps = sys->dt1_ps * sys->max_step1;
  sys->dt3_ps = sys->dt2_ps * sys->max_step2;
  
  sys->dt1 = sys->dt1_ps * c;
  sys->dt2 = sys->dt2_ps * c;
  sys->dt3 = sys->dt3_ps * c;

  return 0;
}

int MD_SYSTEM_set_current_time(MD_SYSTEM *sys, double current_time)
{
  sys->current_time = current_time;
  return 0;
}

int MD_SYSTEM_set_scnb(MD_SYSTEM *sys, double scnb)
{
  sys->atom.scnb = scnb;
  return 0;
}

int MD_SYSTEM_set_scee(MD_SYSTEM *sys, double scee)
{
  sys->atom.scee = scee;
  return 0;
}

int MD_SYSTEM_set_diel(MD_SYSTEM *sys, double diel)
{
  int i;
  if (MD_SYSTEM_check_mdat_crd(sys)) return 1;

  for (i=0;i<sys->atom.natom;i++) {
    sys->atom.q[i] *= sqrt(sys->diel/diel);
  }
  sys->diel = diel;

  return 0;
}

void MD_SYSTEM_set_fixed_atom(MD_SYSTEM *sys, STR_LIST *group_str, int group_no)
{
  int i;
  ATOM_DATA *ad;

  lprintf("FIXED ATOMS:\n");

  ad = &sys->atom;
  if (group_str) {
    group_no = 31;
    ATOM_DATA_set_group_str_list(ad, group_str, group_no, "  GROUP:");
  } else {
    lprintf("  Group No: %d\n", group_no);
  }
  ATOM_DATA_set_fixed_atom(ad, group_no);
  
  sys->n_fixed_atom = 0;
  for (i=0;i<ad->natom;i++) {
    if (ad->ex[i].flag & ATOM_FIXED) sys->n_fixed_atom++;
  }
  if (sys->n_fixed_atom > 0) {
    sys->atom.ex_force_flag = 1;
  }

  lprintf("  Number of atoms: %d\n", sys->n_fixed_atom);
  BOND_DATA_omit_data_for_rigid_mol(&sys->bond, &sys->atom);
  ANGLE_DATA_omit_data_for_rigid_mol(&sys->angle, &sys->atom);
  DIHEDRAL_DATA_omit_data_for_rigid_mol(&sys->dihed, &sys->atom);
  lprintf("\n");
}

void MD_SYSTEM_fixed_atom_correct_virial(MD_SYSTEM *sys)
{
  int i;
  ATOM_DATA *ad;

  ad = &sys->atom;

#ifdef MPI_SDMD  
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
#else
  for (i=0;i<ad->natom;i++) {
#endif    
    if (ad->ex[i].flag & ATOM_FIXED) {
      ad->virial[0] -= ad->f[i].x * ad->x[i].x;
      ad->virial[1] -= ad->f[i].y * ad->x[i].y;
      ad->virial[2] -= ad->f[i].z * ad->x[i].z;
      
      ad->virial[3] -= ad->f[i].x * ad->x[i].y;
      ad->virial[4] -= ad->f[i].x * ad->x[i].z;
      ad->virial[5] -= ad->f[i].y * ad->x[i].z;
      
      ad->virial[6] -= ad->f[i].y * ad->x[i].x;
      ad->virial[7] -= ad->f[i].z * ad->x[i].x;
      ad->virial[8] -= ad->f[i].z * ad->x[i].y;
    }
  }
}


/*
int MD_SYSTEM_set_rattle(MD_SYSTEM *sys, RATTLE_TYPE type, double tol)
{
  BOND_DATA_set_rattle(&sys->bond, type, tol);

  if (type==RATTLE_HYDR) {
    ATOM_DATA_set_hydrogen_group(&sys->atom); 
    BOND_DATA_set_hydrogen_group(&sys->bond,&sys->atom); 
  }

  MD_SYSTEM_degree_of_freedom(sys);
  return 0;
}
*/

void MD_SYSTEM_set_rigid_mol(MD_SYSTEM *sys, char *fname)
{
  lprintf("RIGID BODY SETUP\n");
  lprintf("  Library file name: %s\n",fname);
  RMOL_DATA_setup(&sys->rigid_mol,&sys->atom, &sys->bond, &sys->angle, fname);
  /*
    ATOM_DATA_check_hydrogen_rigid(&sys->atom, 1.2); */
  
  BOND_DATA_omit_data_for_rigid_mol(&sys->bond, &sys->atom);
  ANGLE_DATA_omit_data_for_rigid_mol(&sys->angle, &sys->atom);
  DIHEDRAL_DATA_omit_data_for_rigid_mol(&sys->dihed, &sys->atom);

  lprintf("\n");

  sys->rigid_mol_flag = sys->rigid_mol.n_mol;
}

void MD_SYSTEM_check_rigid_mol(MD_SYSTEM *sys)
{
  /* sys->atom.print_flag_pdb = 1; */

  RMOL_DATA_check_rigid_body(&sys->rigid_mol);
  ATOM_DATA_check_rigid_atom(&sys->atom);

}

void MD_SYSTEM_set_trj_out(MD_SYSTEM *sys, char *fname, int step, int res_no, int flag, int type,
			   int total_step, int gzipped, int trj_wrap_mol, int trj_xst_flag)
{
  extern char *_safe_fopen_fname;
  int out_int;
  char title[81],type_str[10];

  sys->trj_out_step = step;
  sys->trj_out_flag = flag;
  sys->trj_out_type = type;
  sys->gzip_trj = gzipped;
  sys->trj_wrap_mol = trj_wrap_mol;
  sys->trj_xst_flag = trj_xst_flag;

  if (res_no == -1) {
    sys->trj_n_atom = sys->atom.natom;
  } else if (res_no == -2) {
    sys->trj_n_atom = sys->atom.n_solute_atom;
  }else {
    sys->trj_n_atom = sys->atom.r[res_no-1].start_atom+sys->atom.r[res_no-1].natom;
  }
  if (step <= 0) {
    lprintf("ERROR: trj_out_step (%d) must be a positive value.\n", step);
    marble_exit(1);
  }
  
#ifdef MPI
  if (!mpi.master) return;
#endif
  
  if (sys->trj_fp != NULL) {
    if (!sys->gzip_trj)
      fclose(sys->trj_fp);
    else
      pclose(sys->trj_fp);
  }
  if (sys->trj_xst_fp != NULL) {
    fclose(sys->trj_xst_fp);
    sys->trj_xst_fp = NULL;
  }

  if (!sys->gzip_trj) {
    if ((sys->trj_fp = safe_fopen(fname,"w")) == NULL) {
      lprintf("ERROR: Can't open trj out file %s.\n", fname);
      marble_exit(1);
    }
    strcpy(sys->trj_out_fname, _safe_fopen_fname);
  } else {
    char buf[1000];
    if (strlen(fname) >= 2 && strcmp(&fname[strlen(fname)-2],"gz") == 0) {
      sprintf(buf, "gzip > %s", fname);
    } else {
      sprintf(buf, "gzip > %s.gz", fname);
    }
    if ((sys->trj_fp = popen(buf,"w")) == NULL) {
      lprintf("ERROR: Can't open trj out pipe command %s.\n", buf);
      marble_exit(1);
    }
    strcpy(sys->trj_out_fname, buf);
  }

  if (sys->trj_out_flag == TRJ_OUT_TXT) {
    MD_SYSTEM_flag_to_trjstr(type_str, sys->trj_out_type);
    fprintf(sys->trj_fp, "# TRAJECTORY BY MARBLE n_atom: %d, output: %s\n",
	    sys->trj_n_atom, type_str);
  } else if (sys->trj_out_flag == TRJ_OUT_BIN) {
    fwrite(trj_header,sizeof(char),strlen(trj_header)+1,sys->trj_fp);
    fwrite(trj_version,sizeof(int),2,sys->trj_fp);
    fwrite(&sys->trj_out_type,sizeof(int),1,sys->trj_fp);
    fwrite(&sys->trj_n_atom,sizeof(int),1,sys->trj_fp);
  } else if (sys->trj_out_flag == TRJ_OUT_DCD) {

    if (sys->trj_out_type & TRJ_V) {
      lprintf("ERROR: not support DCD output of velocities.\n");
      marble_abort(1);
    }

    sys->float_arr = emalloc("TRJ_OUT_DCD", sizeof(float)*sys->trj_n_atom);
    sys->n_float_arr = sys->trj_n_atom;
    
    out_int = 84;
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite("CORD",sizeof(char),4,sys->trj_fp);
    out_int = total_step/sys->trj_out_step;  /* this is number of snapshots */
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    out_int = sys->trj_out_step;  /* starting time step -- not zero */
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    out_int = sys->trj_out_step;  /* interval steps */
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    out_int = total_step;  /* n step  */
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    
    out_int = 0;
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    sys->float_arr[0] = sys->dt_ps;
    fwrite(sys->float_arr,sizeof(float),1,sys->trj_fp);
    if (sys->trj_out_type & TRJ_BOUNDARY || sys->trj_out_type & TRJ_AMBER_BOUNDARY) {
      out_int = 1;
      fwrite(&out_int,sizeof(int),1,sys->trj_fp);
      out_int = 0;
    } else {
      fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    }

    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    out_int = 24;
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    out_int = 84;
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    out_int = 164;
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    out_int = 2;
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    sprintf(title, "REMARKS FILENAME=%s, CREATED BY MARBLE", sys->trj_out_fname);
    padding(title, 80);
    fwrite(title,sizeof(char),80,sys->trj_fp);
    sprintf(title, "REMARKS DATA:%s",strtime());
    padding(title, 80);
    fwrite(title,sizeof(char),80,sys->trj_fp);
    out_int = 164;
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    out_int = 4;
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    out_int = sys->trj_n_atom;
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    out_int = 4;
    fwrite(&out_int,sizeof(int),1,sys->trj_fp);
    /* end of writing dcd header */
  }

  if (sys->trj_xst_flag) {
    change_extension(fname, sys->trj_xst_fname, "xst");
    if ((sys->trj_xst_fp = safe_fopen(sys->trj_xst_fname,"w")) == NULL) {
      lprintf("ERROR: Can't open xst file %s.\n", fname);
      marble_abort(1);
    }
    fprintf(sys->trj_xst_fp, "# MARBLE boundary trajectory file\n");
    fprintf(sys->trj_xst_fp, "# step ax ay az bx by bz cx cy cz\n");
    
    MD_SYSTEM_xst_out(sys, 0);
  }
}


void MD_SYSTEM_print_trj_info(MD_SYSTEM *sys)
{
  char type[10];
  char *format[] = {  "text", "binary", "DCD" };
  
  if (sys->trj_out_step <= 0) return;
  
  MD_SYSTEM_flag_to_trjstr(type, sys->trj_out_type);
  lprintf("  Trajectory Output = %s\n",sys->trj_out_fname);
  lprintf("     interval = %d steps\n",sys->trj_out_step);
  lprintf("     n_atom   = %d\n",sys->trj_n_atom);
  lprintf("     format   = %s\n", format[sys->trj_out_flag]);
  lprintf("     type     = %s\n",type);
  lprintf("     wrap_mol = %s\n",sys->trj_wrap_mol ? "on" : "off");
  if (sys->trj_xst_flag) {
    lprintf("     xst file = %s\n",sys->trj_xst_fname);
  }
}

int MD_SYSTEM_set_prop_out(MD_SYSTEM *sys, char *fname, int step, int flag)
{
  extern char *_safe_fopen_fname;
  
  sys->prop_out_step = step;
  
  /* prop_out_flag default settings */
  sys->prop_out_flag = PROP_TIME | PROP_TEMPERATURE | PROP_TOTAL_ENE | PROP_POTENTIAL;

  if (sys->extra_pot.head)
    sys->prop_out_flag |= PROP_EP_ENE;

  if (sys->atom.atom_ene_flag)
    sys->prop_out_flag |= PROP_ATOM_ENE;

  if (sys->Ex_System_P_flag || sys->weak_coupling_P_flag || sys->Hybrid_MC_P)
    sys->prop_out_flag |= PROP_DENSITY;

  if (sys->Ex_System_P_flag) {
    if (sys->Ex_System_P_flag == 1) {
      sys->prop_out_flag |= PROP_PV_ISOTROPIC;
    } else if (sys->Ex_System_P_flag == 2) {
      sys->prop_out_flag |= PROP_PV_DIAGONAL | PROP_PV_OFFDIAG;
    } else if (sys->Ex_System_P_flag >= 3) {
      sys->prop_out_flag |= PROP_PV_DIAGONAL;
    }
  }
  /* end of default settings */

  if (flag & PROP_CLEAN)
    sys->prop_out_flag = 0;
  
  sys->prop_out_flag |= flag;

#ifdef MPI
  if (!mpi.master) return 0;
#endif
  
  if (fname == NULL) {
    if (sys->prop_fp != NULL) {
      fclose(sys->prop_fp);
      sys->prop_fp = NULL;
      return 0;
    } else {
      lprintf("ERROR: prop_out_file has not been opened.\n");
      return 3;
    }
  }
  if (step <= 0) {
    lprintf("ERROR: prop_out_step (%d) must be a positive value.\n", step);
    return 2;
  }
  if (sys->prop_fp != NULL) {
    fclose(sys->prop_fp);
  }

  if ((sys->prop_fp = safe_fopen(fname,"w")) == NULL) {
    lprintf("ERROR: Can't open prop out file %s.\n", fname);
    return 1;
  }
  strcpy(sys->prop_out_fname, _safe_fopen_fname);
  MD_SYSTEM_prop_header(sys);

  return 0;
}

int MD_SYSTEM_set_print_out_step(MD_SYSTEM *sys, int step)
{
  sys->print_out_step = step;
  return 0;
}

void MD_SYSTEM_set_solute_mol(MD_SYSTEM *sys, int mol_no)
{
  int res_no;
  ATOM_DATA *ad;

  ad = &sys->atom;
  ad->n_solute_mol = mol_no;
  ATOM_DATA_setup_solute(ad);

  lprintf("Solute Molecules: Number of molecules %d, Number of atoms %d\n",
	  ad->n_solute_mol, ad->n_solute_atom);
  res_no = ad->a[ad->n_solute_atom-1].resno;
  if (res_no>0)
    lprintf("  Last  solute  residue (or molecule): %s (%d)\n", 
	    ad->r[res_no].name, res_no+1);
  if (ad->n_solute_atom < ad->natom)
    lprintf("  First solvent residue (or molecule): %s (%d)\n", 
	    ad->r[res_no+1].name, res_no+2);
  lprintf("\n");
}


void MD_SYSTEM_set_solute_residue(MD_SYSTEM *sys, int res_no)
{
  int i;
  int n_sol_atom;
  ATOM_DATA *ad;

  ad = &sys->atom;
  res_no--;
  n_sol_atom = sys->atom.r[res_no].start_atom + sys->atom.r[res_no].natom;
  
  for (i=0;i<ad->nmol;i++) {
    if (ad->mol[i].start_atom + ad->mol[i].natom >= n_sol_atom) {
      ad->n_solute_mol = i + 1;
      break;
    }
  }

  MD_SYSTEM_set_solute_mol(sys, ad->n_solute_mol);
}

int MD_SYSTEM_set_density(MD_SYSTEM *sys, double density)
{
  double vol, scale, box[3];
  int i;

  vol = sys->atom.total_w / density / (MOL*1.0e-24);
  scale = pow(vol/sys->boundary.V, 1.0/3.0);
  for (i=0;i<3;i++) {
    box[i] = sys->boundary.box[i]*scale;
  }

  return MD_SYSTEM_set_box(sys, box);
}

int MD_SYSTEM_set_box(MD_SYSTEM *sys, double box[3])
{
  double scale[3];
  int i;

  lprintf("PERIODIC BOX SIZE MODIFICATION:\n");
  if (sys->boundary.type != PERIODIC_BOUNDARY) {
    lprintf("  ERROR: PERIODIC BOUNDARY NOT SPECIFIED.\n");
    marble_exit(1);
  }
  
  for (i=0;i<3;i++) {
    scale[i] = box[i]/sys->boundary.box[i];
  }

  PTC_scale_system_anisotropic(sys, scale);

  lprintf("  SCALE FACTOR: (%f,%f,%f)\n", scale[0], scale[1], scale[2]);
  lprintf("  DENSITY:       %.4f g/cm3\n",
	  sys->atom.total_w/sys->boundary.V/(MOL*1.0e-24));
  lprintf("  BOX SIZE:     (%.2f,%.2f,%.2f)\n\n",
	  sys->boundary.box[0], sys->boundary.box[1], sys->boundary.box[2]);
  return 0;
}

void MD_SYSTEM_set_boxv(MD_SYSTEM *sys, double a[3], double b[3], double c[3])
{
  BOUNDARY_set_boxv(&(sys->boundary), a, b, c);
}

void MD_SYSTEM_shift(MD_SYSTEM *sys, double shift[3])
{
  int i;
  ATOM_DATA *ad;
  RMOL_DATA *md;

  ad = &sys->atom;
  md = &sys->rigid_mol;
  for (i=0;i<ad->natom;i++) {
    ad->x[i].x += shift[0];
    ad->x[i].y += shift[1];
    ad->x[i].z += shift[2];
  }
  for (i = 0; i < md->n_mol; i++) {
    md->mol[i].rg.x += shift[0];
    md->mol[i].rg.y += shift[1];
    md->mol[i].rg.z += shift[2];
  }
  RMOL_DATA_mol_to_room_all(md, ad);

  /*
  lprintf("Shift the whole system about (%.2f,%.2f,%.2f)\n\n",
	  shift[0],shift[1],shift[2]);
  */
}

void MD_SYSTEM_shift_center_solute(MD_SYSTEM *sys)
{
  int i;
  double w, x, y, z, shift[3];
  ATOM_DATA *ad;
  BOUNDARY *bc;
  VEC frac, center;

  ad = &sys->atom;
  bc = &sys->boundary;
  w = x = y = z = 0.0;
  for (i=0;i<ad->n_solute_atom;i++) {
    x += ad->x[i].x * ad->w[i];
    y += ad->x[i].y * ad->w[i];
    z += ad->x[i].z * ad->w[i];
    w += ad->w[i];
  }
  x /= w;
  y /= w;
  z /= w;
  if (bc->origin_flag == 1) {
    center.x = center.y = center.z = 0.0;
  } else {
    frac.x = frac.y = frac.z = 0.5;
    
    center.x = VEC_MUL_MAT_X(frac,bc->boxv);
    center.y = VEC_MUL_MAT_Y(frac,bc->boxv);
    center.z = VEC_MUL_MAT_Z(frac,bc->boxv);
  }
  shift[0] = center.x - x;
  shift[1] = center.y - y;
  shift[2] = center.z - z;

  MD_SYSTEM_shift(sys, shift);
}

int MD_SYSTEM_set_initial_velocity(MD_SYSTEM *sys, double T)
{
  int i;
  double lambda;
  
  ATOM_DATA_Maxwell_velocity(&sys->atom, T);
  RMOL_DATA_Maxwell_velocity(&sys->rigid_mol, T);

  if (sys->rattle.flag) {
    RATTLE_backup_x(&sys->atom);
    RATTLE_a(&sys->rattle, &sys->bond, &sys->atom, sys->dt);
    RATTLE_b(&sys->rattle, &sys->bond, &sys->atom);
  }

  lprintf("Velocity Initialization: T = %.2f [K]\n\n",T);
  
  return 0;
}

/*
 * remove_momentum:
 *    flag: -1 default behaviour
 *           0  don't remove mementum at all
 *           1  remove all translational momentum
 *           2  remove all translational and rotational momentum (not used)
 *           3  remove solute translational momentum
 *           4  remove solute translational and rotational momentum
 */
void MD_SYSTEM_set_remove_momentum(MD_SYSTEM *sys, int flag, int step)
{
  int ddf;

  if (flag == -1) {
    /* default:  if external force does not exists,
                 total momentum is removed */
    if (sys->atom.ex_force_flag == 1 || sys->Ex_System_T_flag == 0) {
      flag = 0;
    } else {
      flag = 1;
    }
  }
  
  sys->remove_momentum = flag;

  if (flag)
    sys->remove_momentum_step = step;
  else
    sys->remove_momentum_step = 0;

  sys->remove_total_force = flag;

  if (flag >= 1) {
    if (flag >= 3) {
      sys->remove_momentum_atom = sys->atom.n_solute_atom;
      sys->remove_momentum_w = sys->atom.solute_w;
    } else {
      sys->remove_momentum_atom = sys->atom.natom;
      sys->remove_momentum_w = sys->atom.total_w;
    }
  }
  
  lprintf("Remove Momentum of Center of Mass: %s\n", (flag) ? "on" : "off");
  lprintf("  External Force: %s\n", (sys->atom.ex_force_flag) ? "yes" : "no");
  
  if (flag) {
    if (flag < 3) {
      lprintf("  Target: all\n");
    } else {
      if (sys->remove_momentum == 0) {
	lprintf("ERROR: Number of atoms in solute is zero.\n");
	marble_exit(1);
      }
      lprintf("  Target: solute (n_atom = %d, n_residue = %d)\n",
	      sys->atom.n_solute_atom,
	      sys->atom.a[sys->atom.n_solute_atom-1].resno+1);
    }
    if (flag == 2 || flag == 4) {
      lprintf("  Stopping Translational and Rotational Motion\n");
    }
    lprintf("  Interval Step: %d\n",
	    sys->remove_momentum_step);
    if (sys->n_ex_system >= 1) {
      if (flag == 2 || flag == 4) {
	ddf = 6;
      } else {
	ddf = 3;
      }
      lprintf("  Decreasing degree of freedom: %d\n", ddf);
    }
  }
  lprintf("\n");
}

void MD_SYSTEM_degree_of_freedom(MD_SYSTEM *sys)
{
  int i,j;
  int n_flex_atom_ex[MAX_EX_SYSTEM];
  int n_rattle_group, n_rattle_degree_of_freedom_ex[MAX_EX_SYSTEM];
  int n_rigid_mol6_ex[MAX_EX_SYSTEM],n_rigid_mol5_ex[MAX_EX_SYSTEM];
  ATOM_DATA *ad;

  ad = &sys->atom;
  for (i=0;i<ad->natom;i++) {
    ad->ex[i].flag &= ~ATOM_EX_SYSTEM;
  }
  for (j=0;j<sys->n_ex_system;j++) {
    for (i=sys->atom_ex_system[j][0];i<=sys->atom_ex_system[j][1];i++) {
      ad->ex[i].flag |= EX_SYSTEM_ATOM_FLAG(j);
    }
  }

  ATOM_DATA_n_flex_atom(&sys->atom, &sys->n_flex_atom, n_flex_atom_ex);
  RATTLE_degree_of_freedom(&sys->rattle, &sys->bond, &sys->atom,
			   n_rattle_degree_of_freedom_ex,
			   &n_rattle_group);
  RMOL_DATA_n_rigid_mol(&sys->rigid_mol, &sys->atom,
			n_rigid_mol6_ex, n_rigid_mol5_ex);

  sys->degree_of_freedom = 0;
  sys->degree_of_freedom_tr = 0;
  for (i=0;i<sys->n_ex_system;i++) {
    sys->degree_of_freedom_arr[i] = 3*n_flex_atom_ex[i] + 6*n_rigid_mol6_ex[i] + 5*n_rigid_mol5_ex[i] + n_rattle_degree_of_freedom_ex[i];
    sys->degree_of_freedom += sys->degree_of_freedom_arr[i];
    sys->degree_of_freedom_tr += 3*(n_flex_atom_ex[i]+n_rigid_mol6_ex[i]+n_rigid_mol5_ex[i]);
  }
  sys->degree_of_freedom_tr += 3*n_rattle_group;

  if (sys->remove_momentum && sys->remove_total_force) {
    if (sys->n_ex_system >= 1) {
      int ddf;

      if (sys->remove_momentum == 2 || sys->remove_momentum == 4) {
	ddf = 6;
      } else {
	ddf = 3;
      }
	  
      sys->degree_of_freedom -= ddf;
      sys->degree_of_freedom_tr -= 3;
      sys->degree_of_freedom_arr[0] -= ddf;
    }
  }
}

void MD_SYSTEM_remove_momentum(MD_SYSTEM *sys)
{
  if (sys->remove_momentum == 1) {
    MD_SYSTEM_remove_momentum_all(sys);
  } else if (sys->remove_momentum >= 2) {
    MD_SYSTEM_remove_momentum_partial(sys);
  }
}


void MD_SYSTEM_remove_momentum_all(MD_SYSTEM *sys)
{
  int i;
  double gcv[4], gcv2[4];
  ATOM_DATA *ad;
  RMOL_DATA *md;
  
  ad = &sys->atom;
  md = &sys->rigid_mol;
  gcv[0] = gcv[1] = gcv[2] = gcv[3] = 0.0;
#ifdef MPI_SDMD  
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
#else
  for (i=0;i<ad->natom;i++) {
#endif    
    if (ad->ex[i].flag & ATOM_RIGID) continue;
    gcv[0] += ad->v[i].x * ad->w[i];
    gcv[1] += ad->v[i].y * ad->w[i];
    gcv[2] += ad->v[i].z * ad->w[i];
    gcv[3] += ad->w[i];
  }
#ifdef MPI_SDMD  
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i=0;i<md->n_mol;i++) {
#endif    
    gcv[0] += md->mol[i].vg.x * md->mol[i].type->weight;
    gcv[1] += md->mol[i].vg.y * md->mol[i].type->weight;
    gcv[2] += md->mol[i].vg.z * md->mol[i].type->weight;
    gcv[3] += md->mol[i].type->weight;
  }

#ifdef MPI_SDMD  
  MPI_Reduce(gcv, gcv2, 4, MPI_DOUBLE, MPI_SUM, mpi.master_pe, mpi.comm);

  if (mpi.master) {
#else
    for (i=0;i<4;i++) gcv2[i] = gcv[i];
#endif    
    gcv2[0] /= gcv2[3];
    gcv2[1] /= gcv2[3];
    gcv2[2] /= gcv2[3];
    lprintf("Removing Velocity of Center of Mass [A/ps]: (%f,%f,%f)\n",
	    gcv2[0]*sys->dt/sys->dt_ps, gcv2[1]*sys->dt/sys->dt_ps,
	    gcv2[2]*sys->dt/sys->dt_ps);
#ifdef MPI_SDMD  
  }
  MPI_Bcast(gcv2, 3, MPI_DOUBLE, mpi.master_pe, mpi.comm);
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
#else
  for (i=0;i<ad->natom;i++) {
#endif    
    if (ad->ex[i].flag & ATOM_RIGID) continue;
    ad->v[i].x -= gcv2[0];
    ad->v[i].y -= gcv2[1];
    ad->v[i].z -= gcv2[2];
  }
#ifdef MPI_SDMD  
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else
  for (i=0;i<md->n_mol;i++) {
#endif    
    md->mol[i].vg.x -= gcv2[0];
    md->mol[i].vg.y -= gcv2[1];
    md->mol[i].vg.z -= gcv2[2];
  }
}

void MD_SYSTEM_remove_momentum_partial(MD_SYSTEM *sys)
{
  double gcx, gcy, gcz;
  double gcvx, gcvy, gcvz;
  double lx, ly, lz, rx, ry, rz, vx, vy, vz, wpx, wpy, wpz;
  ATOM_DATA *ad;
  double tw, xx, yy, zz, xy, xz, yz, len;
  double tot_tr_ene, tot_rot_ene, T[3][3], T_inv[3][3];
  int i, n_r_atom;
#ifdef MPI_SDMD
  double tmp[9], tmp2[9];
#endif  

  ad = &sys->atom;
  n_r_atom = sys->remove_momentum_atom;
  tw = sys->remove_momentum_w;
  
  /* velocity of rigid molecules converts to atom velocities */
  RMOL_DATA_set_atom_velocity(&sys->rigid_mol, &sys->atom);
  
  /* Calculate the center of gravity and its velocity */
  gcx = gcy = gcz = 0.0;
  gcvx = gcvy = gcvz = 0.0;
  
#ifdef MPI_SDMD
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
    if (i>= n_r_atom) continue;
#else    
  for (i=0; i<n_r_atom; i++) {
#endif    
    gcx  += ad->x[i].x * ad->w[i];
    gcy  += ad->x[i].y * ad->w[i];
    gcz  += ad->x[i].z * ad->w[i];
    gcvx += ad->v[i].x * ad->w[i];
    gcvy += ad->v[i].y * ad->w[i];
    gcvz += ad->v[i].z * ad->w[i];
  }
#ifdef MPI_SDMD
  tmp[0] = gcx;  tmp[1]=gcy;  tmp[2] = gcz;
  tmp[3] = gcvx; tmp[4]=gcvy; tmp[5] = gcvz;
  MPI_Allreduce(tmp, tmp2, 6, MPI_DOUBLE, MPI_SUM, mpi.comm);
  gcx = tmp2[0]; gcy = tmp2[1]; gcz = tmp2[2];
  gcvx =tmp2[3]; gcvy =tmp2[4]; gcvz =tmp2[5];
#endif
  gcx /= tw;  gcy /= tw;  gcz /= tw;
  gcvx /= tw; gcvy /= tw; gcvz /= tw;
  lx = ly = lz = 0.0;
  xx = yy = zz = xy = xz = yz = 0.0;
#ifdef MPI_SDMD
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
    if (i>= n_r_atom) continue;
#else    
  for (i=0;i<n_r_atom;i++) {
#endif    
    rx = ad->x[i].x - gcx;
    ry = ad->x[i].y - gcy;
    rz = ad->x[i].z - gcz;
    vx = ad->v[i].x - gcvx;
    vy = ad->v[i].y - gcvy;
    vz = ad->v[i].z - gcvz;
    
    lx += (ry*vz-rz*vy)*ad->w[i];
    ly += (rz*vx-rx*vz)*ad->w[i];
    lz += (rx*vy-ry*vx)*ad->w[i];

    xx += ad->w[i]*rx*rx;
    yy += ad->w[i]*ry*ry;
    zz += ad->w[i]*rz*rz;
    xy += ad->w[i]*rx*ry;
    xz += ad->w[i]*rx*rz;
    yz += ad->w[i]*ry*rz;
  }
  
#ifdef MPI_SDMD
  tmp[0] = lx;  tmp[1] = ly;  tmp[2] = lz;
  tmp[3] = xx;  tmp[4] = yy;  tmp[5] = zz;
  tmp[6] = xy;  tmp[7] = xz;  tmp[8] = yz;
  MPI_Allreduce(tmp, tmp2, 9, MPI_DOUBLE, MPI_SUM, mpi.comm);
  lx = tmp2[0];  ly = tmp2[1];  lz = tmp2[2];
  xx = tmp2[3];  yy = tmp2[4];  zz = tmp2[5];
  xy = tmp2[6];  xz = tmp2[7];  yz = tmp2[8];
#endif
  
  T[0][0] =  yy+zz;  T[0][1] = -xy;    T[0][2] = -xz;
  T[1][0] = -xy;     T[1][1] =  zz+xx; T[1][2] = -yz;
  T[2][0] = -xz;     T[2][1] = -yz;    T[2][2] =  xx+yy;
  
  mat_inv33(T, T_inv);

  /*
  {
    int i,j,k; double kk[3][3];

    for (i=0;i<3;i++) 
      for (j=0;j<3;j++) {
	kk[i][j]=0.0;
	for (k=0;k<3;k++)
	  kk[i][j]+=T[i][k]*T_inv[k][j];
      }
    for (i=0;i<3;i++) 
      for (j=0;j<3;j++)
	lprintf("%f ", T[i][j]);
    lprintf("\n");
    
    for (i=0;i<3;i++) 
      for (j=0;j<3;j++)
	lprintf("%f ", kk[i][j]);
    lprintf("\n");
  }
  marble_exit(1);
  */

  wpx = T_inv[0][0]*lx + T_inv[0][1]*ly + T_inv[0][2]*lz;
  wpy = T_inv[1][0]*lx + T_inv[1][1]*ly + T_inv[1][2]*lz;
  wpz = T_inv[2][0]*lx + T_inv[2][1]*ly + T_inv[2][2]*lz;

  lprintf("Removing Velocity of Center of Mass [A/ps]: (%9.6f,%9.6f,%9.6f)\n",
	  gcvx*sys->dt/sys->dt_ps,
	  gcvy*sys->dt/sys->dt_ps,
	  gcvz*sys->dt/sys->dt_ps);

  if (sys->remove_momentum == 2 || sys->remove_momentum == 4) {
    lprintf("Removing Angular Velocity [deg/ps]:         (%9.6f,%9.6f,%9.6f)\n",
	    wpx*sys->dt/sys->dt_ps/M_PI*180.0,
	    wpy*sys->dt/sys->dt_ps/M_PI*180.0,
	    wpz*sys->dt/sys->dt_ps/M_PI*180.0);
  } else {
    wpx=wpy=wpz=0.0;
  }

#ifdef MPI_SDMD
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
    if (i>=n_r_atom) continue;
#else    
  for (i=0;i<n_r_atom;i++) {
#endif    
    ad->v[i].x -= gcvx;
    ad->v[i].y -= gcvy;
    ad->v[i].z -= gcvz;

    rx = ad->x[i].x - gcx;
    ry = ad->x[i].y - gcy;
    rz = ad->x[i].z - gcz;
    
    ad->v[i].x -= wpy*rz - wpz*ry;
    ad->v[i].y -= wpz*rx - wpx*rz;
    ad->v[i].z -= wpx*ry - wpy*rx;
  }
  
  RMOL_DATA_set_mol_velocity(&sys->rigid_mol, ad);
}

void MD_SYSTEM_remove_total_force(MD_SYSTEM *sys)
{
  if (sys->remove_total_force == 1)
    MD_SYSTEM_remove_total_force_all(sys);
  else
    MD_SYSTEM_remove_total_force_partial(sys);
}

void MD_SYSTEM_remove_total_force_all(MD_SYSTEM *sys)
{
  int i;
  double totf[3], totf2[3];
  ATOM_DATA *ad;

  ad = &sys->atom;
  totf[0]=totf[1]=totf[2]=0.0;
#ifdef MPI_SDMD  
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
#else
  for (i=0;i<ad->natom;i++) {
#endif
    totf[0]+=ad->f[i].x;
    totf[1]+=ad->f[i].y;
    totf[2]+=ad->f[i].z;
  }
#ifdef MPI_SDMD
  MPI_Allreduce(totf, totf2, 3, MPI_DOUBLE, MPI_SUM, mpi.comm);
  for (i=0;i<3;i++) totf[i]=totf2[i];
#endif

  for (i=0;i<3;i++) totf[i] /= ad->total_w;

#ifdef MPI_SDMD  
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
#else
  for (i=0;i<ad->natom;i++) {
#endif
    ad->f[i].x -= totf[0] * ad->w[i];
    ad->f[i].y -= totf[1] * ad->w[i];
    ad->f[i].z -= totf[2] * ad->w[i];
  }
}

void MD_SYSTEM_remove_total_force_partial(MD_SYSTEM *sys)
{
  int i,j;
  int n_r_atom;
  double totf[6], totf2[6];
  double T[3][3], T_inv[3][3], torq[3], A[3], tmp[9], tmp2[9];
  double tw, xx, yy, zz, xy, xz, yz;
  double rx, ry, rz, fx, fy, fz;
  ATOM_DATA *ad;

  ad = &sys->atom;
  
  n_r_atom = sys->remove_momentum_atom;
  tw = sys->remove_momentum_w;

  for (i=0;i<6;i++) totf[i] = 0.0;

#ifdef MPI_SDMD  
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
    if (i>=n_r_atom) continue;
#else
  for (i=0;i<n_r_atom;i++) {
#endif
    totf[0]+=ad->f[i].x;
    totf[1]+=ad->f[i].y;
    totf[2]+=ad->f[i].z;
    
    totf[3]+=ad->x[i].x * ad->w[i];
    totf[4]+=ad->x[i].y * ad->w[i];
    totf[5]+=ad->x[i].z * ad->w[i];
  }
#ifdef MPI_SDMD
  MPI_Allreduce(totf, totf2, 6, MPI_DOUBLE, MPI_SUM, mpi.comm);
  for (i=0;i<6;i++) totf[i]=totf2[i];
#endif
  for (i=0;i<6;i++) totf[i] /= tw;

  for (i=0;i<3;i++)
    torq[i] = 0.0;
  xx = yy = zz = xy = xz = yz = 0.0;

#ifdef MPI_SDMD
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
    if (i>= n_r_atom) continue;
#else    
  for (i=0;i<n_r_atom;i++) {
#endif    
    rx = ad->x[i].x - totf[3];
    ry = ad->x[i].y - totf[4];
    rz = ad->x[i].z - totf[5];

    torq[0] += ry * ad->f[i].z - rz * ad->f[i].y;
    torq[1] += rz * ad->f[i].x - rx * ad->f[i].z;
    torq[2] += rx * ad->f[i].y - ry * ad->f[i].x;
    
    xx += ad->w[i]*rx*rx;
    yy += ad->w[i]*ry*ry;
    zz += ad->w[i]*rz*rz;
    xy += ad->w[i]*rx*ry;
    xz += ad->w[i]*rx*rz;
    yz += ad->w[i]*ry*rz;
  }
#ifdef MPI_SDMD
  tmp[0] = torq[0];  tmp[1] = torq[1];  tmp[2] = torq[2];
  tmp[3] = xx;  tmp[4] = yy;  tmp[5] = zz;
  tmp[6] = xy;  tmp[7] = xz;  tmp[8] = yz;
  MPI_Allreduce(tmp, tmp2, 9, MPI_DOUBLE, MPI_SUM, mpi.comm);
  torq[0] = tmp2[0];  torq[1] = tmp2[1];  torq[2] = tmp2[2];
  xx = tmp2[3];  yy = tmp2[4];  zz = tmp2[5];
  xy = tmp2[6];  xz = tmp2[7];  yz = tmp2[8];
#endif
  
  T[0][0] =  yy+zz;  T[0][1] = -xy;    T[0][2] = -xz;
  T[1][0] = -xy;     T[1][1] =  zz+xx; T[1][2] = -yz;
  T[2][0] = -xz;     T[2][1] = -yz;    T[2][2] =  xx+yy;
  mat_inv33(T, T_inv);

  /*  A is derivative of angular velocity */
  for (i=0;i<3;i++) {
    A[i]=0.0;
    for (j=0;j<3;j++)
      A[i]+=T_inv[i][j]*torq[j];
  }
  if (sys->remove_momentum == 1 || sys->remove_momentum == 3) {
    for (i=0;i<3;i++) A[i]=0.0;
  }
  
#ifdef MPI_SDMD  
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
    if (i>=n_r_atom) continue;
#else
  for (i=0;i<n_r_atom;i++) {
#endif
    fx = -totf[0] * ad->w[i];
    fy = -totf[1] * ad->w[i];
    fz = -totf[2] * ad->w[i];
    
    rx = ad->x[i].x - totf[3];
    ry = ad->x[i].y - totf[4];
    rz = ad->x[i].z - totf[5];

    fx -= (A[1]*rz - A[2]*ry) * ad->w[i];
    fy -= (A[2]*rx - A[0]*rz) * ad->w[i];
    fz -= (A[0]*ry - A[1]*rx) * ad->w[i];

    ad->f[i].x += fx;
    ad->f[i].y += fy;
    ad->f[i].z += fz;

    /*
    ad->virial[0] += fx * ad->x[i].x;
    ad->virial[1] += fy * ad->x[i].y;
    ad->virial[2] += fz * ad->x[i].z;
    ad->virial[3] += (fx * ad->x[i].y + fy * ad->x[i].x) * 0.5;
    ad->virial[4] += (fx * ad->x[i].z + fz * ad->x[i].x) * 0.5;
    ad->virial[5] += (fy * ad->x[i].z + fz * ad->x[i].y) * 0.5;
    */
    
  }
}
  

void MD_SYSTEM_clear_time(MD_SYSTEM *sys)
{
  /* CPU TIMES */
  sys->time_bond = sys->time_angle =  sys->time_dihed = 0.0;
  sys->time_nonbond = 0.0;
  sys->time_ew_dir = sys->time_ew_rec = sys->time_ew_cor = 0.0;
  sys->time_boundary = 0.0;
  sys->time_fmm = 0.0;
  sys->time_comm1 = sys->time_comm2 = sys->time_comm3 = sys->time_comm4 = 0.0;
  sys->time_rattle = 0.0;
  
  sys->nonbond.time_list = sys->nonbond.time_fmm_setup = 0.0;
}

/*******************************************************************/
/*   MD print routines                                             */
/*******************************************************************/

void MD_SYSTEM_print_energy(MD_SYSTEM *sys)
{
  double elec;
  if (!sys->ene_long_format) {
    lprintf("BOND      %9.2f  ANGLE     %9.2f  DIHEDRAL  %9.2f\n",
	    sys->ene[BOND_ENE], sys->ene[ANGLE_ENE], sys->ene[DIHED_ENE]);
    if (sys->mdat_format == MDAT_CHARMM) {
      if (sys->dihed.cmap_flag)
	lprintf("ANGLE_UB  %9.2f  IMPROPER  %9.2f  CMAP      %9.2f\n",
		sys->ene[UB_ENE], sys->ene[IMPR_ENE], sys->ene[CMAP_ENE]);
      else
	lprintf("                     ANGLE_UB  %9.2f  IMPROPER  %9.2f\n",
		sys->ene[UB_ENE], sys->ene[IMPR_ENE]);
	
    }
    lprintf("VDW14  %12.2f  ELEC14 %12.2f  BOUNDARY  %9.2f\n",
	    sys->ene[VDW14_ENE],sys->ene[ELEC14_ENE],sys->ene[BOUNDARY_ENE]);
    
    if (sys->ewald.flag) {
      elec = sys->ene[ELEC_ENE] + sys->ene[EWREC_ENE] + sys->ene[EWCOR_ENE];
    } else if (sys->fmm_flag) {
      elec = sys->ene[ELEC_ENE] + sys->ene[FMM_ENE];
    } else {
      elec = sys->ene[ELEC_ENE];
    }
    
    lprintf("VDW    %12.2f  ELEC   %12.2f  EXTRA     %9.2f\n",
	    sys->ene[VDW_ENE], elec, sys->ene[EXTRA_ENE]);

    if (sys->ene[HBOND_ENE]!=0.0) {
      lprintf("HBOND     %9.2f  ", sys->ene[HBOND_ENE]);
      lprintf("\n");
    }
  } else {
    lprintf("BOND      %12.5f  ANGLE     %12.5f  DIHEDRAL  %12.5f\n",
	    sys->ene[BOND_ENE], sys->ene[ANGLE_ENE], sys->ene[DIHED_ENE]);
    if (sys->mdat_format == MDAT_CHARMM) {
      if (sys->dihed.cmap_flag)
	lprintf("ANGLE_UB  %12.5f  IMPROPER  %12.5f  CMAP      %12.5f\n",
		sys->ene[UB_ENE], sys->ene[IMPR_ENE], sys->ene[CMAP_ENE]);
      else
	lprintf("                        ANGLE_UB  %12.5f  IMPROPER  %12.5f\n",
		sys->ene[UB_ENE], sys->ene[IMPR_ENE]);
    }
    lprintf("VDW14     %12.5f  ELEC14    %12.5f  BOUNDARY  %12.5f\n",
	    sys->ene[VDW14_ENE],sys->ene[ELEC14_ENE],sys->ene[BOUNDARY_ENE]);

    if (sys->ewald.flag) {
      elec = sys->ene[ELEC_ENE] + sys->ene[EWREC_ENE] + sys->ene[EWCOR_ENE];
    } else if (sys->fmm_flag) {
      elec = sys->ene[ELEC_ENE] + sys->ene[FMM_ENE];
    } else {
      elec = sys->ene[ELEC_ENE];
    }

    lprintf("VDW       %12.5f  ELEC     %13.5f  EXTRA     %12.5f\n",
	    sys->ene[VDW_ENE], elec,sys->ene[EXTRA_ENE]);
    lprintf("VDW_SUM   %12.5f  ELEC_SUM %13.5f",
	    sys->ene[VDW_ENE]+sys->ene[VDW14_ENE], 
	    elec + sys->ene[ELEC14_ENE]);
    if (sys->boundary.long_range_correction_flag) {
      lprintf("  VDW+LRC  %13.5f\n",
	      sys->ene[VDW_ENE]+sys->ene[BOUNDARY_ENE]); 
    } else {
      lprintf("\n");
    }

    if (sys->ewald.flag) {
      lprintf("EW_DIR    %12.5f  ", sys->ene[ELEC_ENE]);
      lprintf("EW_REC    %12.5f  ", sys->ene[EWREC_ENE]);
      lprintf("EW_COR    %12.5f  ", sys->ene[EWCOR_ENE]);
      lprintf("\n");
    } else if (sys->fmm_flag) {
      lprintf("ELEC_DIR  %12.5f  ", sys->ene[ELEC_ENE]);
      lprintf("FMM       %12.5f  ", sys->ene[FMM_ENE]);
      lprintf("\n");
    }

    if (sys->ene[HBOND_ENE]!=0.0) {
      lprintf("HBOND     %12.5f  ", sys->ene[HBOND_ENE]);
      lprintf("\n");
    }
    lprintf("VIRIAL %12.5f %12.5f %12.5f %12.5f %12.5f %12.5f\n", sys->atom.virial[0], sys->atom.virial[1], sys->atom.virial[2], sys->atom.virial[3], sys->atom.virial[4], sys->atom.virial[5]);
  }
}

void MD_SYSTEM_print_md_status(MD_SYSTEM *sys, int step)
{
  lprintf("\n");
  lprintf("STEP      %9d  TIME   %9.3f ps  T       %9.2f K\n",
	  step, sys->current_time, sys->temperature);
    
  lprintf("TOTAL_E  %10.2f  POTENTIAL%10.2f  KINETIC  %10.2f\n",
	  sys->total_ene, sys->potential,sys->kene);
  if (sys->rigid_mol_flag || sys->rattle.flag)
    lprintf("KINETIC ENERGY:      K_TRANS  %10.2f  K_ROT    %10.2f\n",
	    sys->kene_tr, sys->kene_rot);
  
  if (sys->Ex_System_T_flag) {
    lprintf("EXTENDED SYSTEM T:   POTENTIAL%10.2f  KINETIC  %10.2f\n",
	    sys->e_eta, sys->e_eta_v);
  }
  if (sys->Ex_System_P_flag) {
    lprintf("EXTENDED SYSTEM P:   PV        %9.2f  KINETIC   %9.2f\n",
	    sys->PV, sys->PV_v);
  }
  
  MD_SYSTEM_print_energy(sys);
  if (sys->Ex_System_P_flag || sys->weak_coupling_P_flag ||
      sys->Hybrid_MC_P) {
    lprintf("PRESSURE  %9.2f  VOLUME %12.2f  DENSITY   %9.4f [g/cm3]\n",
	    sys->Pint / ATM_TO_KCALMOLANG, sys->boundary.V,
	    sys->atom.total_w/sys->boundary.V/(MOL*1.0e-24));
  }

  if (sys->n_ex_system==2&&sys->Ex_System_P_flag==0 ||
      sys->n_ex_system==3&&sys->Ex_System_P_flag>=1) {
    lprintf("SOLUTE_T %10.2f  SOLVENT_T%10.2f\n",
	    sys->temperature_arr[0],sys->temperature_arr[1]);
    lprintf("SOLUTE_K %10.2f  SOLVENT_K%10.2f\n",
	    sys->kene_arr[0],sys->kene_arr[1]);
  }
  
  if (sys->Ex_System_P_flag>=2) {
    BOUNDARY *bc;
    bc = &sys->boundary;
    lprintf("a = (%13e,%13e,%13e)\n",bc->boxv[0][0],bc->boxv[0][1],bc->boxv[0][2]);
    lprintf("b = (%13e,%13e,%13e)\n",bc->boxv[1][0],bc->boxv[1][1],bc->boxv[1][2]);
    lprintf("c = (%13e,%13e,%13e)\n",bc->boxv[2][0],bc->boxv[2][1],bc->boxv[2][2]);
  }

  if (sys->Hybrid_MC) {
    int tot;
    tot = sys->Hybrid_MC_accept+sys->Hybrid_MC_reject;
    if (tot == 0) tot = 1;
    lprintf("Hybrid MC: Acceptance Ratio %.2f (%d)\n",
	    (double) sys->Hybrid_MC_accept/tot*100, sys->Hybrid_MC_accept);
  }
  
  lprintf("\n-------------------------------------------------------------------------\n");
  
  lflush();
}

#if 1  /* ndef MPI  */
void MD_SYSTEM_print_time(MD_SYSTEM *sys, double total)
{
  double other;

  other = total - (sys->time_bond+sys->time_angle+sys->time_dihed+
		   sys->time_nonbond+sys->time_ew_dir+
		   sys->time_ew_rec+sys->time_ew_cor+
		   sys->time_boundary+sys->time_fmm+
		   sys->time_rattle+
		   sys->nonbond.time_list+sys->nonbond.time_fmm_setup);

  lprintf("----------  CPU TIME  -------------\n");
  lprintf("BOND         %10.2f sec\n" , sys->time_bond);
  lprintf("ANGLE        %10.2f sec\n" , sys->time_angle);
  lprintf("DIHEDRAL     %10.2f sec\n" , sys->time_dihed);
  if (sys->ewald.flag == FLAG_NO_EWALD) {
    lprintf("NONBOND      %10.2f sec\n" , sys->time_nonbond);
  } else {
    lprintf("EWALD DIRECT %10.2f sec\n" , sys->time_ew_dir);
    lprintf("EWALD REC    %10.2f sec\n" , sys->time_ew_rec);
    lprintf("EWALD CORR   %10.2f sec\n" , sys->time_ew_cor);
  }
  if (sys->fmm_flag) {
    lprintf("FMM SETUP    %10.2f sec\n" , sys->nonbond.time_fmm_setup);
    lprintf("FMM FORCE    %10.2f sec\n" , sys->time_fmm);
  }
  lprintf("NONBOND LIST %10.2f sec\n" , sys->nonbond.time_list);
  lprintf("BOUNDARY     %10.2f sec\n" , sys->time_boundary);
  lprintf("RATTLE       %10.2f sec\n" , sys->time_rattle);
#ifdef MPI_RDMD
  /* lprintf("COMM BARRIER %10.2f sec\n" , sys->time_comm3); */
  lprintf("COMM E       %10.2f sec\n" , sys->time_comm1);
  lprintf("COMM F       %10.2f sec\n" , sys->time_comm2);
  other -= sys->time_comm1 + sys->time_comm2 + sys->time_comm3;
#endif
#ifdef MPI_RDMD_ATOM
  lprintf("COMM X       %10.2f sec\n" , sys->time_comm4);
  other -= sys->time_comm4;
#endif
  
  lprintf("OTHER        %10.2f sec\n" , other);
  lprintf("-----------------------------------\n");
  lprintf("TOTAL        %10.2f sec\n\n", total);
}
#else
void MD_SYSTEM_print_time(MD_SYSTEM *sys, double total)
{
  double other;

  other = total - (sys->time_bond+sys->time_angle+sys->time_dihed+
		   sys->time_nonbond+sys->time_ew_dir+
		   sys->time_ew_rec+sys->time_ew_cor+
		   sys->time_boundary+sys->time_fmm+
		   sys->time_rattle+
		   sys->nonbond.time_list+sys->nonbond.time_fmm_setup);

  lprintf("----------  CPU TIME  -------------\n");
  lprintf("BOND         %10.2f sec\n" , sys->time_bond);
  lprintf("ANGLE        %10.2f sec\n" , sys->time_angle);
  lprintf("DIHEDRAL     %10.2f sec\n" , sys->time_dihed);
  if (sys->ewald.flag == FLAG_NO_EWALD) {
    plprintf("NONBOND      %10.2f sec\n" , sys->time_nonbond);
  } else {
    plprintf("EWALD DIRECT %10.2f sec\n" , sys->time_ew_dir);
    plprintf("EWALD REC    %10.2f sec\n" , sys->time_ew_rec);
    plprintf("EWALD CORR   %10.2f sec\n" , sys->time_ew_cor);
  }
  if (sys->fmm_flag) {
    plprintf("FMM SETUP    %10.2f sec\n" , sys->nonbond.time_fmm_setup);
    plprintf("FMM FORCE    %10.2f sec\n" , sys->time_fmm);
  }
  plprintf("NONBOND LIST %10.2f sec\n" , sys->nonbond.time_list);
  lprintf("BOUNDARY     %10.2f sec\n" , sys->time_boundary);
  lprintf("RATTLE       %10.2f sec\n" , sys->time_rattle);
#ifdef MPI_RDMD
  lprintf("COMM BARRIER %10.2f sec\n" , sys->time_comm3);
  plprintf("COMM E       %10.2f sec\n" , sys->time_comm1);
  plprintf("COMM F       %10.2f sec\n" , sys->time_comm2);
  other -= sys->time_comm1 + sys->time_comm2 + sys->time_comm3;
#endif
#ifdef MPI_RDMD_ATOM
  plprintf("COMM X       %10.2f sec\n" , sys->time_comm4);
  other -= sys->time_comm4;
#endif
  
  plprintf("OTHER        %10.2f sec\n" , other);
   lprintf("-----------------------------------\n");
  plprintf("TOTAL        %10.2f sec\n", total);
  lprintf("\n");
}
#endif

/********************************************************************/
/*   statistics                                                     */
/********************************************************************/

void STATISTICS_clear(STATISTICS *st)
{
  int i;
  st->count=0;
  st->potential=0.0;
  st->kene=0.0;
  st->total_ene=0.0;
  st->Pint=0.0;
  for (i=0;i<6;i++) st->PintV[i] = 0.0;
  st->V=0.0;
  st->HV=0.0;
  st->time=0.0;
  st->time_total_ene=0.0;
}

void STATISTICS_sample(STATISTICS *st, MD_SYSTEM *sys)
{
  int i;
  st->count++;
  /*
  if (sys->pert.pert_flag) {
    st->total_ene+=sys->total_ene + sys->pert.total_lambda_ene;
    st->potential+=sys->potential+sys->pert.potential;
    st->kene+=sys->kene+sys->pert.kene;
  } else {
    st->total_ene+=sys->total_ene;
    st->potential+=sys->potential;
    st->kene+=sys->kene;
  }
  */
  st->total_ene+=sys->total_ene;
  st->potential+=sys->potential;
  st->kene+=sys->kene;

  st->time+=sys->current_time;
  st->Pint+=sys->Pint;
  if (sys->gamma_ex == 0.0) {
    for (i=0;i<6;i++) st->PintV[i] += sys->PintV[i];
  } else {
    for (i=0;i<2;i++) st->PintV[i] += -(sys->PintV[i]-sys->Pex) * sys->boundary.boxv[2][2];
    for (i=2;i<6;i++) st->PintV[i] += sys->PintV[i];
  }
  st->V+=sys->boundary.V;
}

void STATISTICS_sample2(STATISTICS *st, MD_SYSTEM *sys)
{
  int i;
  st->count++;

  /*
  if (sys->pert.pert_flag) {
    st->total_ene+=SQR(sys->total_ene + sys->pert.total_lambda_ene);
    st->potential+=SQR(sys->potential+sys->pert.potential);
    st->kene+=SQR(sys->kene+sys->pert.kene);
    st->time_total_ene+=(sys->total_ene+sys->pert.total_lambda_ene)*sys->current_time;
  } else {
    st->total_ene+=SQR(sys->total_ene);
    st->potential+=SQR(sys->potential);
    st->kene+=SQR(sys->kene);
    st->time_total_ene+=sys->total_ene*sys->current_time;
  }
  */

  st->total_ene+=SQR(sys->total_ene);
  st->potential+=SQR(sys->potential);
  st->kene+=SQR(sys->kene);
  st->time_total_ene+=sys->total_ene*sys->current_time;

  st->time+=SQR(sys->current_time);
  st->Pint+=SQR(sys->Pint);
  if (sys->gamma_ex == 0.0) {
    for (i=0;i<6;i++) st->PintV[i] += SQR(sys->PintV[i]);
  } else {
    for (i=0;i<2;i++) st->PintV[i] += SQR(-(sys->PintV[i]-sys->Pex) * sys->boundary.boxv[2][2]);
    for (i=2;i<6;i++) st->PintV[i] += SQR(sys->PintV[i]);
  }
  st->V+=SQR(sys->boundary.V);
  st->HV+=sys->boundary.V*(sys->potential+sys->Pex*sys->boundary.V);
}

static double safe_sqrt(double c)
{
  if (c <= 0.0) return 0.0;
  return sqrt(c);
}

void STATISTICS_deviation(STATISTICS *st, STATISTICS *ave, MD_SYSTEM *sys)
{
  int i;
  st->potential=safe_sqrt(st->potential-SQR(ave->potential));
  st->kene=safe_sqrt(st->kene-SQR(ave->kene));
  st->total_ene=safe_sqrt(st->total_ene-SQR(ave->total_ene));
  st->Pint=safe_sqrt(st->Pint-SQR(ave->Pint));
  for (i=0;i<6;i++) st->PintV[i] += safe_sqrt(st->PintV[i]-SQR(ave->PintV[i]));
  st->V=safe_sqrt(st->V-SQR(ave->V));
  st->HV=st->HV - ave->V*(ave->potential+sys->Pex*ave->V);
}

void STATISTICS_average(STATISTICS *st)
{
  int i;
  if (st->count == 0) return;
  st->potential/=st->count;
  st->kene/=st->count;
  st->total_ene/=st->count;
  st->Pint/=st->count;
  for (i=0;i<6;i++) st->PintV[i] /= st->count;
  st->V/=st->count;
  st->HV/=st->count;
}

void MD_SYSTEM_clear_statistics(MD_SYSTEM *sys)
{
  STATISTICS_clear(&sys->ave);
  STATISTICS_clear(&sys->ave2);
  sys->start_total_ene = sys->total_ene;

  sys->total_ene_drift = 0.0;
  sys->total_ene_ave_block = sys->total_ene_var_block = 0.0;
  sys->kin_ene_ave_block   = sys->kin_ene_var_block = 0.0;
  sys->total_ene_R_ave = sys->total_ene_R_var = 0.0;
  
  sys->total_ene_fluct = 0.0;
  sys->total_ene_fluct_ave = sys->total_ene_fluct_var = 0.0;
  sys->start_total_ene_block = sys->total_ene;
  
  sys->total_ene_n_sample = 0;
  sys->total_ene_n_ave = 0;
  sys->total_ene_stime = sys->current_time;
}

void MD_SYSTEM_sample_statistics(MD_SYSTEM *sys)
{
  double total_ene_dev, kin_ene_dev, R;
  
  STATISTICS_sample(&sys->ave, sys);
  STATISTICS_sample2(&sys->ave2,sys);

  /* Estimation total energy fluctuation */
  sys->total_ene_drift += sys->total_ene - sys->start_total_ene;
  sys->total_ene_fluct += fabs((sys->total_ene - sys->start_total_ene_block)
			       /sys->start_total_ene_block);
  sys->total_ene_ave_block += sys->total_ene;
  sys->total_ene_var_block += SQR(sys->total_ene);
  sys->kin_ene_ave_block += sys->kene;
  sys->kin_ene_var_block += SQR(sys->kene);
  sys->total_ene_n_sample++;

  if (sys->current_time - sys->total_ene_stime >= 1.0 - 1.0e-8) {
    sys->total_ene_fluct /= sys->total_ene_n_sample;
    sys->total_ene_fluct_ave += sys->total_ene_fluct;
    sys->total_ene_fluct_var += sys->total_ene_fluct*sys->total_ene_fluct;
    sys->total_ene_fluct = 0.0;
    sys->start_total_ene_block = sys->total_ene;

    sys->total_ene_ave_block /= sys->total_ene_n_sample;
    sys->total_ene_var_block /= sys->total_ene_n_sample;
    sys->kin_ene_ave_block /= sys->total_ene_n_sample;
    sys->kin_ene_var_block /= sys->total_ene_n_sample;
    total_ene_dev = safe_sqrt(sys->total_ene_var_block-SQR(sys->total_ene_ave_block));
    kin_ene_dev   = safe_sqrt(sys->kin_ene_var_block  -SQR(sys->kin_ene_ave_block));
    R = total_ene_dev/kin_ene_dev;
    sys->total_ene_R_ave  += R;
    sys->total_ene_R_var  += R*R;
    sys->total_ene_ave_block = sys->total_ene_var_block = 0.0;
    sys->kin_ene_ave_block   = sys->kin_ene_var_block = 0.0;

    sys->total_ene_n_sample = 0;
    sys->total_ene_n_ave++;
    sys->total_ene_stime = sys->current_time;
  }
}

void MD_SYSTEM_collect_statistics(MD_SYSTEM *sys)
{
  double drift_a, drift_b, dd, noise;
  if (sys->ave.count == 0) return;

  dd = sys->ave.count * sys->ave2.time - SQR(sys->ave.time);

  if (dd==0.0) return;
  
  drift_a = (sys->ave2.time*sys->ave.total_ene-sys->ave.time*sys->ave2.time_total_ene) / dd;
  drift_b = (sys->ave.count*sys->ave2.time_total_ene-sys->ave.time*sys->ave.total_ene) / dd;

  
  noise = safe_sqrt((sys->ave2.total_ene + drift_a*drift_a*sys->ave.count
		     + drift_b*drift_b*sys->ave2.time
		     - 2.0*drift_a*sys->ave.total_ene - 2.0*drift_b*sys->ave2.time_total_ene
		     + 2.0*drift_a*drift_b*sys->ave.time)/sys->ave.count);

  STATISTICS_average(&sys->ave);
  STATISTICS_average(&sys->ave2);

  STATISTICS_deviation(&sys->ave2, &sys->ave, sys);
  lprintf("\n");
  lprintf("*** AVERAGE :                             T         %9.2f\n",
	  2.0*sys->ave.kene/(sys->degree_of_freedom*K*KCAL));
  lprintf("TOTAL_E  %10.2f  POTENTIAL%10.2f  KINETIC  %10.2f\n",
	  sys->ave.total_ene, sys->ave.potential, sys->ave.kene);
  if (sys->Ex_System_P_flag || sys->weak_coupling_P_flag || sys->Hybrid_MC_P) {
    lprintf("PRESSURE  %9.2f  VOLUME %12.2f  DENSITY   %9.4f [g/cm3]\n",
	    sys->ave.Pint / ATM_TO_KCALMOLANG, sys->ave.V, 
	    sys->atom.total_w/sys->ave.V/(MOL*1.0e-24));
  }
  if (sys->Ex_System_P_flag) {
    if (sys->gamma_ex == 0.0) {
      lprintf("Pxx       %9.2f  Pyy       %9.2f  Pzz       %9.2f\n",
	      sys->ave.PintV[0] / ATM_TO_KCALMOLANG,
	      sys->ave.PintV[1] / ATM_TO_KCALMOLANG,
	      sys->ave.PintV[2] / ATM_TO_KCALMOLANG);
    } else {
      lprintf("Gxx       %9.2f  Gyy       %9.2f  Pzz       %9.2f\n",
	      sys->ave.PintV[0] / DYNCM_TO_KCALMOLANG,
	      sys->ave.PintV[1] / DYNCM_TO_KCALMOLANG,
	      sys->ave.PintV[2] / ATM_TO_KCALMOLANG);
    }
    
    lprintf("Pxy       %9.2f  Pxz       %9.2f  Pyz       %9.2f\n",
	    sys->ave.PintV[3] / ATM_TO_KCALMOLANG,
	    sys->ave.PintV[4] / ATM_TO_KCALMOLANG,
	    sys->ave.PintV[5] / ATM_TO_KCALMOLANG);
  }

  lprintf("\n");
  lprintf("*** DEVIATION :                           T         %9.2f\n",
	  2.0*sys->ave2.kene/(sys->degree_of_freedom*K*KCAL));
  lprintf("TOTAL_E  %10.2f  POTENTIAL%10.2f  KINETIC  %10.2f\n",
	  sys->ave2.total_ene, sys->ave2.potential, sys->ave2.kene);
  if (sys->Ex_System_P_flag || sys->weak_coupling_P_flag || sys->Hybrid_MC_P) {
    lprintf("PRESSURE  %9.2f  VOLUME %12.2f  DENSITY   %9.4f [g/cm3]\n",
	    sys->ave2.Pint / ATM_TO_KCALMOLANG, sys->ave2.V, 
	    sys->atom.total_w/sys->ave.V/(MOL*1.0e-24)*sys->ave2.V/sys->ave.V);
  }
  if (sys->Ex_System_P_flag) {
    if (sys->gamma_ex == 0.0) {
      lprintf("Pxx       %9.2f  Pyy       %9.2f  Pzz       %9.2f\n",
	      sys->ave2.PintV[0] / ATM_TO_KCALMOLANG,
	      sys->ave2.PintV[1] / ATM_TO_KCALMOLANG,
	      sys->ave2.PintV[2] / ATM_TO_KCALMOLANG);
    } else {
      lprintf("Gxx       %9.2f  Gyy       %9.2f  Pzz       %9.2f\n",
	      sys->ave2.PintV[0] / DYNCM_TO_KCALMOLANG,
	      sys->ave2.PintV[1] / DYNCM_TO_KCALMOLANG,
	      sys->ave2.PintV[2] / ATM_TO_KCALMOLANG);
    }
    lprintf("Pxy       %9.2f  Pxz       %9.2f  Pyz       %9.2f\n",
	    sys->ave2.PintV[3] / ATM_TO_KCALMOLANG,
	    sys->ave2.PintV[4] / ATM_TO_KCALMOLANG,
	    sys->ave2.PintV[5] / ATM_TO_KCALMOLANG);
  }
  lprintf("\n");
  
  STATISTICS_clear(&sys->ave);
  STATISTICS_clear(&sys->ave2);
  if (sys->total_ene_n_ave == 0) return;

  /* lprintf("%d\n",sys->total_ene_n_ave);  DEBUG */
  lprintf("*** CONSERVATION OF TOTAL ENERGY (AVERAGED EVERY 1 ps) :\n");
  lprintf("NUMBER OF 1PS SEGMENTS: %d\n", sys->total_ene_n_ave);
  lprintf("RATIO (D_TOTAL/D_KINETIC):  %f (dev: %f)\n",
	  sys->total_ene_R_ave/sys->total_ene_n_ave,
	  safe_sqrt(sys->total_ene_R_var/sys->total_ene_n_ave-
		    SQR(sys->total_ene_R_ave/sys->total_ene_n_ave)));
  lprintf("TOTAL ENERGY FLUCTUATION:   log10(%e)=%f\n",
	  sys->total_ene_fluct_ave/sys->total_ene_n_ave,
	  log(sys->total_ene_fluct_ave/sys->total_ene_n_ave)/log(10.0));
  lprintf("                            (dev: %e)\n",
	  safe_sqrt(sys->total_ene_fluct_var/sys->total_ene_n_ave-
		    SQR(sys->total_ene_fluct_ave/sys->total_ene_n_ave)));
  lprintf("TOTAL ENERGY DRIFT:  %f [kcal/mol/ps] (noise %f [kcal/mol])\n\n",
	  drift_b, noise);
  lprintf("\n");
}

/*** calculation of force ***/

void MD_SYSTEM_calc_force(MD_SYSTEM *sys)
{
  double total_time = 0.0, start_time, end_time;
  int i;
  
  ATOM_DATA_clear_force(&(sys->atom));

  for (i=0;i<N_ENE;i++) sys->ene[i]=0.0;

  if (sys->atom.atom_ene_sample_flag) {
    ATOM_DATA_atom_ene_clear(&sys->atom);
  }
  
  dtime();
  bond_energy_force(&(sys->bond), &(sys->atom), &(sys->ene[BOND_ENE]));
  add_dtime(&sys->time_bond);
  
  angle_energy_force(&(sys->angle), &(sys->atom),
		     &(sys->ene[ANGLE_ENE]),&(sys->ene[UB_ENE]));
  add_dtime(&sys->time_angle);

  dihedral_energy_force(&(sys->dihed), &(sys->atom),
			&(sys->ene[DIHED_ENE]), &(sys->ene[IMPR_ENE]));
  
  nonbond14_energy_force(&(sys->dihed), &(sys->nonbond), &(sys->atom), 
                         &(sys->ene[VDW14_ENE]),&(sys->ene[ELEC14_ENE]));
  add_dtime(&sys->time_dihed);

  ATOM_DATA_calc_virial(&(sys->atom));

  dtime();
  if (sys->ewald.flag == FLAG_NO_EWALD) {
    nonbond_energy_force(&(sys->nonbond), &(sys->atom), &(sys->boundary),
	 &(sys->ene[VDW_ENE]),&(sys->ene[ELEC_ENE]),&(sys->ene[HBOND_ENE]));
    add_dtime(&sys->time_nonbond);
  } else {
#if 0
    /* EWALD */
    EW_direct_energy_force(&(sys->ewald), &(sys->linked_cell), 
			   &(sys->nonbond),&(sys->atom),&(sys->boundary),
			   &(sys->ene[VDW_ENE]),&(sys->ene[ELEC_ENE]),
			   &(sys->ene[HBOND_ENE]));
    add_dtime(&sys->time_ew_dir);

    if (sys->ewald.flag == FLAG_EWALD) {
      EW_rec_energy_force(&(sys->ewald), sys->boundary.box,
			  sys->boundary.min,
			  &(sys->atom), &(sys->ene[EWREC_ENE]));
    } else {
      /* PME */
      EW_pme_energy_force(&(sys->ewald),&(sys->atom),&(sys->boundary),
			  &(sys->ene[EWREC_ENE]));
    }
    add_dtime(&sys->time_ew_rec);

    EW_cor_energy_force(&(sys->ewald), &(sys->boundary), &(sys->atom),
			&(sys->ene[EWCOR_ENE]));
    add_dtime(&sys->time_ew_cor);
#endif
  }

  /*
  if (sys->fmm_flag) {
    calc_fmm(&(sys->atom));
    FMM_energy_force(&(sys->atom),&(sys->ene[FMM_ENE]));
    add_dtime(&sys->time_fmm);
  }
  */

  BOUNDARY_energy_force(&(sys->boundary), &(sys->atom), &(sys->ene[BOUNDARY_ENE]));
  add_dtime(&sys->time_boundary);

  EP_energy_force(&(sys->extra_pot), sys, &(sys->ene[EXTRA_ENE]));

  if (sys->remove_total_force)
    MD_SYSTEM_remove_total_force(sys);
  
#ifdef MPI_RDMD
  reduce_energy_force(sys);
#endif
  
  MD_SYSTEM_sum_potential(sys);
  
  for (i=3;i<=5;i++)
    sys->atom.virial[i+3] =  sys->atom.virial[i];
  if (sys->rigid_mol_flag)
    RMOL_DATA_correct_virial(&sys->rigid_mol, &sys->atom);
  if (sys->rattle.flag)
    RATTLE_correct_virial(&sys->rattle, &sys->atom);

  /*
  if (sys->scale_system_method == MOL_BASED_SCALE) {
    ATOM_DATA_molecular_virial(&sys->atom);
  }
  */

  if (sys->atom.atom_ene_sample_flag) {
    ATOM_DATA_atom_ene_sample(&sys->atom);
  }
}


  
/* Attention: This function uses kene. */
void MD_SYSTEM_calc_Pint(MD_SYSTEM *sys)
{
  int i, iex;
  
  if (sys->boundary.type != PERIODIC_BOUNDARY) {
    sys->Pint = 0.0;
    return;
  }

  /*
    double kene;
    kene = MD_SYSTEM_atom_or_mol_kene(sys);
    sys->Pint = (kene*2.0 + sys->atom.virial[0] + sys->atom.virial[1]
   	       + sys->atom.virial[2])
    / (3.0*sys->boundary.V);

  sys->Pint_k = kene*2.0/(3.0*sys->boundary.V);
  sys->Pint_v = (sys->atom.virial[0] + sys->atom.virial[1] + sys->atom.virial[2])/(3.0*sys->boundary.V);
  */
  
  sys->Pint_k = sys->kene_tr*2.0/(3.0*sys->boundary.V);
  sys->Pint_v = (sys->atom.virial[0] + sys->atom.virial[1] + sys->atom.virial[2])/(3.0*sys->boundary.V);
  sys->Pint = sys->Pint_k + sys->Pint_v;

  for (i=0;i<6;i++) {
    if (i<3)
      sys->PintV[i] = sys->atom.virial[i];
    else
      sys->PintV[i] = (sys->atom.virial[i] + sys->atom.virial[i+3]) * 0.5;
    for (iex=0;iex<sys->n_ex_system;iex++) {
      if (i<3)
	sys->PintV[i] += sys->kene_tr_arr[iex][i][i] * 2.0;
      else if (i==3)
	sys->PintV[3] += sys->kene_tr_arr[iex][0][1] + sys->kene_tr_arr[iex][1][0];
      else if (i==4)
	sys->PintV[4] += sys->kene_tr_arr[iex][0][2] + sys->kene_tr_arr[iex][2][0];
      else if (i==5)
	sys->PintV[5] += sys->kene_tr_arr[iex][1][2] + sys->kene_tr_arr[iex][2][1];
    }
    sys->PintV[i] /= sys->boundary.V;
  }
}

void MD_SYSTEM_make_nonbond_list(MD_SYSTEM *sys)
{
  make_nonbond_list(&(sys->nonbond), &(sys->atom), &(sys->linked_cell),
		    &(sys->boundary));
}

#if 0
void MD_SYSTEM_calc_kene(MD_SYSTEM *sys)
{
  int i, iex;
  int start_atom, end_atom;
  double kene = 0.0;
  double kmolv = 0.0, kmolr = 0.0;
  double tmp;
  double tmp_arr[MAX_EX_SYSTEM+3], tmp_arr2[MAX_EX_SYSTEM+3];
  ATOM_DATA *ad;

  ad = &(sys->atom);

  sys->kene_tr = sys->kene_rot = 0.0;
  for (iex=0;iex<sys->n_ex_system;iex++) {
    sys->kene_arr[iex] = 0.0;
  }

  /* for (i=0;i<sys->n_flex_atom;i++) */
  for (i=ad->node_fatom_h;i>=0;i=ad->node_fatom_n[i]) {
    tmp = ad->w[i] * Length2(ad->v[i].x, ad->v[i].y, ad->v[i].z);
    sys->kene_tr += tmp;
    
    iex = ATOM_FLAG_EX_SYSTEM(ad->ex[i].flag);
    sys->kene_arr[iex] += tmp;
    
  }
  if (sys->rigid_mol_flag)
    RMOL_DATA_kene(&sys->rigid_mol, &sys->kene_tr, &sys->kene_rot, sys->kene_arr);
  if (sys->rattle.flag)
    RATTLE_kene(&sys->rattle, &sys->atom,
		&sys->kene_tr, &sys->kene_rot, sys->kene_arr);
  
  sys->kene_tr  *= 0.5;
  sys->kene_rot *= 0.5;
  for (i=0;i<sys->n_ex_system;i++)
    sys->kene_arr[i] *= 0.5;
  
#if defined(MPI_RDMD_ATOM)||defined(MPI_SDMD)
  tmp_arr[0]=sys->kene_tr;
  tmp_arr[1]=sys->kene_rot;
  for (i=0;i<sys->n_ex_system;i++)
    tmp_arr[i+2]=sys->kene_arr[i];
  MPI_Allreduce(tmp_arr, tmp_arr2, 2+sys->n_ex_system, MPI_DOUBLE, MPI_SUM, mpi.comm);

  sys->kene_tr =tmp_arr2[0];
  sys->kene_rot=tmp_arr2[1];
  for (i=0;i<sys->n_ex_system;i++)
    sys->kene_arr[i]=tmp_arr2[i+2];
#endif
  
  sys->kene = sys->kene_tr + sys->kene_rot;
  sys->temperature = 2.0*sys->kene/(sys->degree_of_freedom*K*KCAL);
  
  for (iex=0;iex<sys->n_ex_system;iex++) {
    if (sys->degree_of_freedom_arr[iex]>0)
      sys->temperature_arr[iex] = 2.0*sys->kene_arr[iex]/(sys->degree_of_freedom_arr[iex]*K*KCAL);
    else
      sys->temperature_arr[iex] = 0.0;
  }
}
#endif

void MD_SYSTEM_calc_kene(MD_SYSTEM *sys)
{
  MD_SYSTEM_calc_kene_full(sys);
}

void MD_SYSTEM_calc_kene_full(MD_SYSTEM *sys)
{
  int i, iex, n, k, l;
  int start_atom, end_atom;
  double kene = 0.0;
  double kmolv = 0.0, kmolr = 0.0;
  double tmp;
  double tmp_arr[2+MAX_EX_SYSTEM*10], tmp_arr2[2+MAX_EX_SYSTEM*10];
  ATOM_DATA *ad;

  ad = &(sys->atom);

  sys->kene_tr = sys->kene_rot = 0.0;
  for (iex=0;iex<sys->n_ex_system;iex++) {
    for (k=0;k<3;k++)
      for (l=0;l<3;l++)
	sys->kene_tr_arr[iex][k][l] = 0.0;
    sys->kene_rot_arr[iex] = 0.0;
  }
  
  /* for (i=0;i<sys->n_flex_atom;i++) */
  for (i = ad->node_fatom_h; i>=0; i=ad->node_fatom_n[i]) {
    tmp = ad->w[i] * Length2(ad->v[i].x, ad->v[i].y, ad->v[i].z);
    sys->kene_tr += tmp;
    
    iex = ATOM_FLAG_EX_SYSTEM(ad->ex[i].flag);
    sys->kene_arr[iex] += tmp;

    sys->kene_tr_arr[iex][0][0] += ad->w[i]*ad->v[i].x*ad->v[i].x;
    sys->kene_tr_arr[iex][0][1] += ad->w[i]*ad->v[i].x*ad->v[i].y;
    sys->kene_tr_arr[iex][0][2] += ad->w[i]*ad->v[i].x*ad->v[i].z;
    sys->kene_tr_arr[iex][1][1] += ad->w[i]*ad->v[i].y*ad->v[i].y;
    sys->kene_tr_arr[iex][1][2] += ad->w[i]*ad->v[i].y*ad->v[i].z;
    sys->kene_tr_arr[iex][2][2] += ad->w[i]*ad->v[i].z*ad->v[i].z;
  }
  if (sys->rigid_mol_flag)
    RMOL_DATA_kene_full(&sys->rigid_mol, &sys->kene_tr, &sys->kene_rot,
			sys->n_ex_system, sys->kene_tr_arr, sys->kene_rot_arr);
  if (sys->rattle.flag)
    RATTLE_kene_full(&sys->rattle, &sys->atom, &sys->kene_tr, &sys->kene_rot,
		     sys->n_ex_system, sys->kene_tr_arr, sys->kene_rot_arr);

  sys->kene_tr  *= 0.5;
  sys->kene_rot *= 0.5;
  for (iex=0;iex<sys->n_ex_system;iex++) {
    sys->kene_rot_arr[iex] *= 0.5;
    sys->kene_tr_arr[iex][1][0] = sys->kene_tr_arr[iex][0][1];
    sys->kene_tr_arr[iex][2][0] = sys->kene_tr_arr[iex][0][2];
    sys->kene_tr_arr[iex][2][1] = sys->kene_tr_arr[iex][1][2];
    for (k=0;k<3;k++)
      for (l=0;l<3;l++)
	sys->kene_tr_arr[iex][k][l] *= 0.5;
  }
  
#if defined(MPI_RDMD_ATOM)||defined(MPI_SDMD)
  tmp_arr[0]=sys->kene_tr;
  tmp_arr[1]=sys->kene_rot;
  n=2;
  for (iex=0;iex<sys->n_ex_system;iex++) {
    for (k=0;k<3;k++)
      for (l=0;l<3;l++)
	tmp_arr[n++]=sys->kene_tr_arr[iex][k][l];
    tmp_arr[n++]=sys->kene_rot_arr[iex];
  }
  
  MPI_Allreduce(tmp_arr, tmp_arr2, 2+sys->n_ex_system*10, MPI_DOUBLE, MPI_SUM, mpi.comm);

  sys->kene_tr =tmp_arr2[0];
  sys->kene_rot=tmp_arr2[1];
  n=2;
  for (iex=0;iex<sys->n_ex_system;iex++) {
    for (k=0;k<3;k++) {
      for (l=0;l<3;l++) {
	sys->kene_tr_arr[iex][k][l]=tmp_arr2[n++];
      }
    }
    sys->kene_rot_arr[iex]=tmp_arr2[n++];
  }
#endif
  
  for (iex=0;iex<sys->n_ex_system;iex++) {
    sys->kene_arr[iex]=sys->kene_rot_arr[iex];
    for (k=0;k<3;k++) {
      sys->kene_arr[iex]+=sys->kene_tr_arr[iex][k][k];
    }
  }
  
  sys->kene = sys->kene_tr + sys->kene_rot;
  sys->temperature = 2.0*sys->kene/(sys->degree_of_freedom*K*KCAL);
  
  for (iex=0;iex<sys->n_ex_system;iex++) {
    if (sys->degree_of_freedom_arr[iex]>0)
      sys->temperature_arr[iex] = 2.0*sys->kene_arr[iex]
	                         / (sys->degree_of_freedom_arr[iex]*K*KCAL);
    else
      sys->temperature_arr[iex] = 0.0;
  }
}

  
double MD_SYSTEM_calc_mol_kene(MD_SYSTEM *sys)
{
  int i;
  double kene2 = 0.0;

  ATOM_DATA_mol_velocity(&sys->atom);
  
  for (i=0;i<sys->atom.nmol;i++) {
    kene2 += sys->atom.mol[i].w * (  SQR(sys->atom.mol[i].v.x)
				   + SQR(sys->atom.mol[i].v.y)
				   + SQR(sys->atom.mol[i].v.z));
  }
  return (kene2 * 0.5);
}

double MD_SYSTEM_atom_or_mol_kene(MD_SYSTEM *sys)
{
  if (sys->scale_system_method == ATOM_BASED_SCALE)  {
    MD_SYSTEM_calc_kene(sys);
    return sys->kene;
  } if (sys->scale_system_method == MOL_BASED_SCALE)  {
    return MD_SYSTEM_calc_mol_kene(sys);
  }
  return 0.0;
}


void MD_SYSTEM_sum_potential(MD_SYSTEM *sys)
{
  int i, j;
  int degree_of_freedom;
  
  sys->potential = 0.0;
  for (i=0;i<N_ENE;i++) sys->potential += sys->ene[i];

  sys->e_eta = sys->e_eta_v = 0.0;
  if (sys->Ex_System_T_flag) {
    for (i=0;i<sys->n_ex_system;i++) {
      for (j=0;j<sys->n_chain_T;j++) {
	if (j==0) degree_of_freedom = sys->degree_of_freedom_arr[i];
	else      degree_of_freedom = 1;
	sys->e_eta   += degree_of_freedom * KCAL * K * sys->target_temperature
	              * sys->eta[i][j];
	sys->e_eta_v += 0.5*sys->Q[i][j]*sys->eta_v[i][j]*sys->eta_v[i][j];
      }
    }
  }
  
  if (sys->Ex_System_P_flag == 1) {
    sys->PV = sys->Pex * sys->boundary.V;
    sys->PV_v = 0.5 * sys->W * sys->logv_v * sys->logv_v;
    sys->e_eta += KCAL * K * sys->target_temperature* sys->eta[sys->pc_ex_system][0];
  } else if (sys->Ex_System_P_flag >= 2) {
    sys->PV = sys->Pex * sys->boundary.V - sys->gamma_ex * sys->boundary.A;
    sys->PV_v = 0.0;
    for (i=0;i<3;i++) {
      for (j=0;j<3;j++) {
	sys->PV_v += 0.5 * sys->W * sys->Vg[i][j] * sys->Vg[i][j];
      }
    }
    sys->e_eta += sys->Ex_System_n_P_var * KCAL * K * sys->target_temperature* sys->eta[sys->pc_ex_system][0];
  }
}

void MD_SYSTEM_sum_total_energy(MD_SYSTEM *sys)
{
  sys->total_ene = sys->potential+sys->kene;
  
  if (sys->Ex_System_T_flag) {
    sys->total_ene += sys->e_eta+sys->e_eta_v;
  }
  if (sys->Ex_System_P_flag) {
    sys->total_ene += sys->PV+sys->PV_v;
  }
}



/*** test routines ***/

void MD_SYSTEM_test_force(MD_SYSTEM *sys, int iatom, double dx, double dy, double dz)
{
  ATOM_DATA *ad;
  double pre_pot;

  ad = &sys->atom;
#ifdef MPI_SDMD
  SDMD_setup(sys);
#else
  VERLET_setup(sys);
#endif
  pre_pot = sys->potential;

  ad->x[iatom].x += dx;
  ad->x[iatom].y += dy;
  ad->x[iatom].z += dz;

#ifdef MPI_SDMD
  SDMD_calc_force(sys);
#else
  MD_SYSTEM_calc_force(sys);           /* calc force at step = 0 */
#endif
  MD_SYSTEM_sum_potential(sys);
  lprintf("%d: %e =? %e\n",iatom+1, sys->potential - pre_pot,
	 -(ad->f[iatom].x * dx + ad->f[iatom].y * dy + ad->f[iatom].z * dz));

  ad->x[iatom].x -= dx;
  ad->x[iatom].y -= dy;
  ad->x[iatom].z -= dz;

}

void MD_SYSTEM_test_force_all(MD_SYSTEM *sys, double len)
{
  ATOM_DATA *ad;
  int i;
  double dx, dy, dz, dlen2, len2, r;
  
  ad = &sys->atom;

  for (i=0;i<ad->natom;i++) {
    do {
      dx = drand48()*2.0-1.0;
      dy = drand48()*2.0-1.0;
      dz = drand48()*2.0-1.0;
      dlen2 = dx*dx+dy*dy+dz*dz;
    } while (dlen2 > 1.0 || dlen2 < 1.0e-6);
    r = len/sqrt(dlen2);
    dx *= r;
    dy *= r;
    dz *= r;
    MD_SYSTEM_test_force(sys, i, dx, dy, dz);
  }
}


#if 0
void MD_SYSTEM_test_pert(MD_SYSTEM *sys, double l, double dl)
{
  ATOM_DATA *ad;
  double pre_pot, post_pot;

  ad = &sys->atom;
  ATOM_DATA_allocate_for_rRESPA(&(sys->atom)); /* for multi timestep */
  VERLET_setup(sys);

  sys->pert.l.lambda = l - dl*0.5;
  PERT_calc_force0_rRESPA(sys);  /* calc force at step == 1 */
  PERT_calc_force1_rRESPA(sys);  /* calc force at step == 1 */
  
  MD_SYSTEM_sum_potential(sys);
  PERT_sum_potential(&(sys->pert));
  PERT_sum_total_energy(sys);
  /* PERT_sum_potential(sys);*/
  
  pre_pot = sys->pert.potential;
  
  sys->pert.l.lambda = l+dl*0.5;
  PERT_calc_force0_rRESPA(sys);  /* calc force at step == 1 */
  PERT_calc_force1_rRESPA(sys);  /* calc force at step == 1 */
  
  MD_SYSTEM_sum_potential(sys);
  PERT_sum_potential(&(sys->pert));
  PERT_sum_total_energy(sys);
  /* PERT_sum_potential(sys);*/
  post_pot = sys->pert.potential;

  sys->pert.l.lambda = l;
  PERT_calc_force0_rRESPA(sys);  /* calc force at step == 1 */
  PERT_calc_force1_rRESPA(sys);  /* calc force at step == 1 */
  
  MD_SYSTEM_sum_potential(sys);
  PERT_sum_potential(&(sys->pert));
  PERT_sum_total_energy(sys);

  lprintf("(%e - %e) / %e = %e =? %e\n", post_pot, pre_pot, dl, (post_pot - pre_pot) / dl, - sys->pert.l.f_non);
}
#endif

void MD_SYSTEM_test_momentum(MD_SYSTEM *sys)
{
  VEC g, p;
  ATOM_DATA *ad;
  BOUNDARY *bc;
  VEC *fold_x;
  double tmp;
  int i, itmp;
  double w=0.0;

  ad = &sys->atom;
  bc = &sys->boundary;
  
  fold_x = malloc(sizeof(VEC)*ad->natom);
  for (i=0;i<ad->natom;i++) {
    tmp = ((ad->x[i].x - bc->min[0])/bc->box[0]);
    if (tmp < 0.0) { itmp = (int) tmp - 1; }
    else           { itmp = (int) tmp; }
    fold_x[i].x = (tmp - itmp) * bc->box[0] + bc->min[0];
    
    tmp = ((ad->x[i].y - bc->min[1])/bc->box[1]);
    if (tmp < 0.0) { itmp = (int) tmp - 1; }
    else           { itmp = (int) tmp; }
    fold_x[i].y = (tmp - itmp) * bc->box[1] + bc->min[1];
    
    tmp = ((ad->x[i].z - bc->min[2])/bc->box[2]);
    if (tmp < 0.0) { itmp = (int) tmp - 1; }
    else           { itmp = (int) tmp; }
    fold_x[i].z = (tmp - itmp) * bc->box[2] + bc->min[2];
  }

  g.x = g.y = g.z = 0.0;
  p.x = p.y = p.z = 0.0;
  for (i=0;i<ad->natom;i++) {
    g.x += fold_x[i].x * ad->w[i];
    g.y += fold_x[i].y * ad->w[i];
    g.z += fold_x[i].z * ad->w[i];
    p.x += ad->v[i].x * ad->w[i];
    p.y += ad->v[i].y * ad->w[i];
    p.z += ad->v[i].z * ad->w[i];
    w+=ad->w[i];
  }
  g.x /= w;  g.y /= w;  g.z /= w;
  printf("Gravity Center: %f %f %f\n", g.x, g.y, g.z);
  printf("Total Momentum: %f %f %f\n", p.x, p.y, p.z);
  printf("System Total Kene: %f\n",
	 (SQR(p.x) + SQR(p.y) + SQR(p.z))/(2.0*w));
  free(fold_x);
}

/* for check of calculation of virial */
void MD_SYSTEM_test_virial(MD_SYSTEM *sys, int type)
{
  int i,j,k,ii,jj;
  double pbox[3], pro[3];
  double pot0, boxv0[3][3], vmat0[3][3], vmat_recip[3][3];
  double dx = 1.0e-5;
  ATOM_DATA *ad;
  BOUNDARY *bc;
  RMOL_DATA *md;

  if (type == 0) {
    sys->scale_system_method = ATOM_BASED_SCALE;
  } else {
    sys->scale_system_method = MOL_BASED_SCALE;
  }
  sys->ewald.pressure_flag = 1;
  ad = &sys->atom;
  bc = &sys->boundary;
  md = &sys->rigid_mol;

#ifdef MPI_SDMD
  SDMD_setup(sys);
#else
  VERLET_setup(sys);
  md->crd = emalloc("test_virial", sizeof(RMOL_CRD)*md->n_mol);
#endif
  
  lprintf("potential = %e\n", sys->potential);
  

  /* for (i=0;i<sys->n_flex_atom;i++) */
  for (i=ad->node_fatom_h;i>=0;i=ad->node_fatom_n[i]) {
    ad->px[i].x = VEC_MUL_MAT_X(ad->x[i],bc->recip);
    ad->px[i].y = VEC_MUL_MAT_Y(ad->x[i],bc->recip);
    ad->px[i].z = VEC_MUL_MAT_Z(ad->x[i],bc->recip);
  }
  
#ifdef MPI_SDMD
  for (i=md->node_rmol_h; i>=0; i=md->mol[i].node_rmol_n) {
#else    
  for (i=0; i<md->n_mol; i++) {
#endif
    md->crd[i].rg.x = VEC_MUL_MAT_X(md->mol[i].rg,bc->recip);
    md->crd[i].rg.y = VEC_MUL_MAT_Y(md->mol[i].rg,bc->recip);
    md->crd[i].rg.z = VEC_MUL_MAT_Z(md->mol[i].rg,bc->recip);
  }

  
  for (i=0;i<3;i++)
    for (j=0;j<3;j++)
      boxv0[i][j] = bc->boxv[i][j];
  
  pot0 = sys->potential;
  
  vmat0[0][0] = ad->virial[0];
  vmat0[1][1] = ad->virial[1];
  vmat0[2][2] = ad->virial[2];

  vmat0[0][1] = ad->virial[3];
  vmat0[0][2] = ad->virial[4];
  vmat0[1][2] = ad->virial[5];
  vmat0[1][0] = ad->virial[6];
  vmat0[2][0] = ad->virial[7];
  vmat0[2][1] = ad->virial[8];
  
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      vmat_recip[i][j] = 0.0;
      for (k=0;k<3;k++) {
	vmat_recip[i][j] += vmat0[i][k]*bc->recip[k][j];
      }
    }
  }
  
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      
      for (ii=0;ii<3;ii++)
	for (jj=0;jj<3;jj++)
	  bc->boxv[ii][jj] = boxv0[ii][jj];
      
      bc->boxv[j][i] += dx;
      BOUNDARY_make_recip(bc);
      BOUNDARY_set_offset_v(bc);
      LINKED_CELL_calc_tr_x(&sys->linked_cell, ad, bc);
      
      /* for (i=0;i<sys->n_flex_atom;i++) */
      for (k=ad->node_fatom_h;k>=0;k=ad->node_fatom_n[k]) {
	ad->x[k].x = VEC_MUL_MAT_X(ad->px[k],bc->boxv);
	ad->x[k].y = VEC_MUL_MAT_Y(ad->px[k],bc->boxv);
	ad->x[k].z = VEC_MUL_MAT_Z(ad->px[k],bc->boxv);
      }
      
#ifdef MPI_SDMD
      for (k=md->node_rmol_h; k>=0; k=md->mol[k].node_rmol_n) {
#else
      for (k=0; k<md->n_mol; k++) {
#endif
        md->mol[k].rg.x = VEC_MUL_MAT_X(md->crd[k].rg,bc->boxv);
        md->mol[k].rg.y = VEC_MUL_MAT_Y(md->crd[k].rg,bc->boxv);
        md->mol[k].rg.z = VEC_MUL_MAT_Z(md->crd[k].rg,bc->boxv);
      }
      
      RMOL_DATA_mol_to_room(&sys->rigid_mol, &sys->atom);

#ifdef MPI_SDMD
      SDMD_dist_x(&sys->linked_cell, &sys->atom);
      SDMD_calc_force(sys);
#else      
      MD_SYSTEM_calc_force(sys);           /* calc force at step = 0 */
      MD_SYSTEM_sum_potential(sys);
#endif      

      lprintf("%d,%d: %e/%e = %e, vir=%e\n",i,j,
	     sys->potential - pot0, dx, (sys->potential - pot0)/dx,
	      -vmat_recip[i][j]);
    }
  }
}


void MD_SYSTEM_check_bonds_angles(MD_SYSTEM *sys)
{
  int i;
  ATOM_DATA *ad;
  BOND_DATA *bd;
  ANGLE_DATA *and;
  DIHEDRAL_DATA *did;
  DIHEDRAL *dihe;
  VEC *x;
  double r, r0;
  int type, atom1, atom2, atom3, atom4, nb, na, nd, ndz;
  double dx, dy, dz;
  double ax, ay, az;
  double bx, by, bz;
  double ab, aa, bb, len_ab, cc, th, th0;
  double eb, ea, ed;
  double x12,y12,z12,x23,y23,z23,x34,y34,z34,nx23,ny23,nz23,len_23;
  double gx,gy,gz,dd,gg,dg,len_dg;
  double ddx,ddy,ddz,dgx,dgy,dgz;
  double phi, psi0;
  double ened, kpsi;

  ad = &sys->atom;
  bd = &sys->bond;
  and = &sys->angle;
  did = &sys->dihed;
  x = ad->x;

  eb = ea = ed = 0.0;
  ened = 0.0;
  nb = na = nd = ndz = 0;

  for (i = 0; i < bd->n_bond; i++) {

    /* if (bd->bonds[i].flag & NO_CALC_BOND) continue; */

    atom1 = bd->bonds[i].atom1;
    atom2 = bd->bonds[i].atom2;
    
    if (atom1 >= ad->n_solute_atom || atom2 >= ad->n_solute_atom) continue;

    type = bd->bonds[i].type;
    r0    = bd->bond_type[type].r0;
    dx = x[atom2].x - x[atom1].x;
    dy = x[atom2].y - x[atom1].y;
    dz = x[atom2].z - x[atom1].z;
    r = sqrt(dx*dx+dy*dy+dz*dz);
    eb += (r - r0) * (r - r0);
    nb++;
  }

  for (i=0; i < and->n_angle; i++) {
    atom1 = and->angles[i].atom1;
    atom2 = and->angles[i].atom2;
    atom3 = and->angles[i].atom3;

    if (atom1 >= ad->n_solute_atom || 
	atom2 >= ad->n_solute_atom ||
	atom3 >= ad->n_solute_atom) continue;

    type = and->angles[i].type;
    th0  = and->angle_type[type].th0;

    ax = x[atom1].x - x[atom2].x;
    ay = x[atom1].y - x[atom2].y;
    az = x[atom1].z - x[atom2].z;
    bx = x[atom3].x - x[atom2].x;
    by = x[atom3].y - x[atom2].y;
    bz = x[atom3].z - x[atom2].z;

    ab = ax*bx + ay*by + az*bz;
    aa = ax*ax + ay*ay + az*az;
    bb = bx*bx + by*by + bz*bz;
    len_ab = sqrt(aa) * sqrt(bb);
    cc = ab/len_ab;
    th = acos(cc);
    ea += (th - th0) * (th - th0);
    na++;
  }

  for (i=0; i < did->n_dihedral; i++) {
    dihe = &(did->dihedrals[i]);
    type = dihe->type;

    if (dihe->flag & DIHED_ONLY14) continue;
    if (!(dihe->flag & DIHED_CHARMM_IMPR)) continue;

    if (did->impr_type[type].kpsi == 0.0) {
      ndz++;
      continue;
    }

    psi0  = did->impr_type[type].psi0;
    
    atom1 = dihe->atom1;
    atom2 = dihe->atom2;
    atom3 = dihe->atom3;
    atom4 = dihe->atom4;
    if (atom3 < 0) atom3 = -atom3;
    if (atom4 < 0) atom4 = -atom4;

    if (atom1 >= ad->n_solute_atom || 
	atom2 >= ad->n_solute_atom ||
	atom3 >= ad->n_solute_atom ||
	atom4 >= ad->n_solute_atom) continue;

    x12 = x[atom2].x - x[atom1].x;
    y12 = x[atom2].y - x[atom1].y;
    z12 = x[atom2].z - x[atom1].z;
    x23 = x[atom3].x - x[atom2].x;
    y23 = x[atom3].y - x[atom2].y;
    z23 = x[atom3].z - x[atom2].z;
    x34 = x[atom4].x - x[atom3].x;
    y34 = x[atom4].y - x[atom3].y;
    z34 = x[atom4].z - x[atom3].z;
    dx  = y12*z23 - z12*y23;
    dy  = z12*x23 - x12*z23;
    dz  = x12*y23 - y12*x23;
    gx  = y23*z34 - z23*y34;
    gy  = z23*x34 - x23*z34;
    gz  = x23*y34 - y23*x34;

    len_23 = sqrt(x23*x23 + y23*y23 + z23*z23);
    nx23 = x23 / len_23;
    ny23 = y23 / len_23;
    nz23 = z23 / len_23;
    dd  = dx*dx + dy*dy + dz*dz;
    gg  = gx*gx + gy*gy + gz*gz;
    dg  = dx*gx + dy*gy + dz*gz;
    len_dg = sqrt(dd) * sqrt(gg);

    phi = atan2((nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)
		 +nz23*(dx*gy-dy*gx))/len_dg,  dg/len_dg);

    if (phi - psi0 < -M_PI) phi += 2.0*M_PI;
    else if (phi - psi0 > M_PI) phi -= 2.0*M_PI;

    lprintf("%d %d %d %d %f %f %f %f\n", atom1+1,atom2+1, atom3+1, atom4+1, did->impr_type[type].kpsi, phi*180.0/M_PI, psi0*180.0/M_PI, fabs(phi-psi0)*180/M_PI);

    ed += (phi - psi0) * (phi - psi0);
    ened += did->impr_type[type].kpsi * (phi - psi0) * (phi - psi0);
    nd++;
  }

  lprintf("\n");
  lprintf("Bond deviations of solute molecules (%d molecule(s)):\n", ad->n_solute_mol);
  lprintf("  Number of bonds: %d\n", nb);
  if (nb > 0) 
    lprintf("  RMSD from equilibrium distances: %lf (angstrom)\n", sqrt(eb/nb));
  lprintf("\n");
  lprintf("Angle deviations of solute molecules:\n");
  lprintf("  Number of angles: %d\n", na);
  if (na > 0) 
    lprintf("  RMSD from equilibrium angles: %lf (degree)\n", sqrt(ea/na)*180.0/M_PI);
  lprintf("\n");

  lprintf("Improper dihedral angles deviations of solute molecules:\n");
  lprintf("  Number of improper dihedral angles: %d\n", nd);
  if (ndz > 0) {
    lprintf("    (excluding improper dihedral angles with k=0: %d)\n", ndz);
  }
  if (nd > 0) 
    lprintf("  RMSD from equilibrium angles: %lf (degree)\n", sqrt(ed/nd)*180.0/M_PI);

  /* lprintf("%lf\n", ened); */

  lprintf("\n");

}
