/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#define MAIN_C

/*
#define MAJOR_VERSION     x
#define MINOR_VERSION     x
#define SUBMINOR_VERSION  x
*/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef MPI
#include "mpi_global.h"
#endif

#ifdef _OPENMP
#include "omp.h"
#endif

#include "md_system.h"
#include "ctrl.h"
#ifdef STATTRJ
#include "stattrj.h"
#endif

void print_usage();

int main(int argc, char *argv[])
{
  int c;
  int log_flag = 0;
  int stdout_flag = 0;
  int multi_log_flag = 0;
  int test_flag = 0;
  int next_flag;
  int n_threads;
  char *ctrl_file = NULL;
  char *progname;

#ifdef MPI
  MPI_Init(&argc, &argv);
  mpi_setup();
#endif  
  
  progname = argv[0];

  while (--argc > 0 && (*++argv)[0] == '-') {
    next_flag = 0;
    while (!next_flag && (c = *++(argv[0]))) {
      switch (c) {
      case 's' :
	stdout_flag = 1;
	break;
#ifdef MPI
      case 'm' :
	multi_log_flag = 1;
	break;
#endif  /* ifdef MPI */
#ifdef _OPENMP
      case 't' :
	argc--;
	argv++;
	n_threads = atoi(argv[0]);
	if (n_threads <= 0) {
	  printf("rank %d: Number of threads must be positive.\n", mpi.rank);
	  marble_exit(1);
        }
	omp_set_num_threads(n_threads);
	mpi.n_threads = omp_get_max_threads();
	break;
#endif  /* ifdef _OPENMP */
      case 'n' :
        test_flag = 1;
	break;
      case 'v' :
	printf("marble: Version %d.%d.%d%s\n", MAJOR_VERSION, MINOR_VERSION, SUBMINOR_VERSION, PATCH_LEVEL);
	break;
      case 'h' :
      case '?' :
	print_usage();
      default:
	printf("marble: illegal option %c\n",c);
      }
    }
  }

  if (argc == 0) print_usage();

  ctrl_file = *argv;
  
#ifdef MPI
  /* print out to stdout */
  if (argc == 1 && !mpi.master) {
    set_log_stdout_flag(0);
  }
#endif

  /* print out to log */
  if (argc >= 2) {
#ifdef MPI
    if (multi_log_flag) {
      char buf[1000];
      sprintf(buf,"%s.%d",*(argv+1),mpi.rank);
      log_file_open(buf);
    } else if (mpi.master) {
      log_file_open(*(argv+1));      
    }
    if (!stdout_flag || !mpi.master) {
      set_log_stdout_flag(0);
    }
#else 
    log_file_open(*(argv+1));
    if (!stdout_flag) {
      set_log_stdout_flag(0);
    }
#endif
  }

  MD_SYSTEM_init(&_md_sys);
  opening(ctrl_file, progname);
  initialize_ctrl_section();
  read_ctrl_file(ctrl_file);
  if (test_flag) marble_exit(0);
  exec_ctrl_section();
  
  closing();
  marble_exit(0);
  return 0;
}

void print_usage()
{
#ifndef STATTRJ
  char *program = "marble";
#else
  char *program = "stattrj";
#endif  
  
#ifdef MPI	
#ifdef _OPENMP
  printf("%s [-smh] [-t n_threads] control_file [log_file]\n", program);
#else /* _OPENMP */
  printf("%s [-smh] control_file [log_file]\n", program);
#endif
#else	/* MPI */
  printf("%s [-sh] control_file [log_file]\n", program);
#endif  /* MPI */
  marble_exit(1);
}

static int _marble_exit_abort = 0;

void marble_exit_abort_flag(int arg)
{
  _marble_exit_abort = arg;
}

void marble_exit(int arg)
{
  if (_marble_exit_abort)
    marble_abort(arg);
#ifdef MPI
  MPI_Finalize();
#endif
  exit(arg);
}

void marble_abort(int arg)
{
  lflush();
#ifdef MPI
  MPI_Abort(MPI_COMM_WORLD, arg);
#else
  exit(arg);
#endif  
}

static double _opening_time;

void opening(char *ctrl_file, char *progname)
{
  static int printed = 0;
  char hostname[100], *log;
  int i;

  if (printed) return;
  printed = 1;
  lprintf("****************************************\n");
#ifndef STATTRJ  
  lprintf("     MARBLE (Version %d.%d.%d%s)\n", MAJOR_VERSION, MINOR_VERSION, SUBMINOR_VERSION, PATCH_LEVEL);
#else  
  lprintf("     STATTRJ (Version %d.%d.%d)\n", MAJOR_VERSION, MINOR_VERSION, SUBMINOR_VERSION);
#endif  

  lprintf("     Program: %s\n",progname);
  
  if (gethostname(hostname,100) == 0) {
    lprintf("     Host: %s\n",hostname);
  }

#ifdef MPI
  lprintf("     PE: %d\n", mpi.n_pe);
#endif  
#ifdef _OPENMP
  lprintf("     Threads: %d\n", mpi.n_threads);
#endif  
  lprintf("     Date: %s\n", strtime());
  
  lprintf("     Control File: %s\n", ctrl_file);
  lprintf("     Log File: %s\n",
	  ((log = get_log_file_name()) == NULL) ? "standard out" : log);
  lprintf("****************************************\n");
  lprintf("\n");

  _opening_time = get_user_time();
}

void closing()
{
  double time;
  int time_h, time_m;
  float time_s;

  /*user_system_times(&utime_h,&utime_m,&utime_s, &stime_h,&stime_m,&stime_s);*/
  time = get_user_time() - _opening_time;
  convert_times_d(time, &time_h, &time_m, &time_s);

  lprintf("\n");
  lprintf("****************************************\n");
  lprintf("     Date: %s\n",strtime());
  lprintf("     Elapsed Time:  %3dh %2dm %5.1fs\n",
	time_h,time_m,time_s);
  lprintf("****************************************\n");
  lprintf("\n\n");
  lflush();
}
