/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "misc.h"
#include "fit.h"

/* public routines */
void FIT_init(FIT *fit, int n_atom, double *w)
{
  int i;
  char *fname = "FIT_init";
  
  fit->n_atom = n_atom;
  
  fit->w = emalloc(fname, sizeof(double)*n_atom);
  fit->x0 = emalloc(fname, sizeof(VEC)*n_atom);
  fit->x1 = emalloc(fname, sizeof(VEC)*n_atom);

  fit->total_w = 0.0;
  for (i=0;i<n_atom;i++) {
    fit->w[i] = w[i];
    fit->total_w += w[i];
  }

  fit->flag = 1;
}

double FIT_calc_rmsd(FIT *fit, VEC *x0, VEC *x1)
{
  double rmsd;
  double dx, dy, dz;
  int i;
  
  rmsd = 0.0;
  
  for (i=0;i<fit->n_atom;i++) {
    dx = x0[i].x - x1[i].x;
    dy = x0[i].y - x1[i].y;
    dz = x0[i].z - x1[i].z;
    rmsd += fit->w[i] * (dx*dx + dy*dy + dz*dz);
    /*lprintf("%d %f (%f %f %f) (%f %f %f) %f\n", i, rmsd,
	    x0[i].x, x0[i].y,x0[i].z,
	    x1[i].x, x1[i].y,x1[i].z,fit->w[i]);*/

  }
  rmsd /= fit->total_w;
  fit->rmsd = sqrt(rmsd);
  return fit->rmsd;
}

void FIT_calc_com(FIT *fit, VEC *x0, VEC *com)
{
  int i;
  
  com->x = com->y = com->z = 0.0;

  for (i=0;i<fit->n_atom;i++) {
    com->x += fit->w[i] * x0[i].x;
    com->y += fit->w[i] * x0[i].y;
    com->z += fit->w[i] * x0[i].z;
  }

  com->x /= fit->total_w;
  com->y /= fit->total_w;
  com->z /= fit->total_w;
}

void FIT_copy_vec(FIT *fit, VEC *x0, VEC *x1)
{
  int i;

  for (i=0;i<fit->n_atom;i++) {
    x1[i].x = x0[i].x;
    x1[i].y = x0[i].y;
    x1[i].z = x0[i].z;
  }
}

void FIT_shift(FIT *fit, VEC *x0, VEC *shift)
{
  int i;

  for (i=0;i<fit->n_atom;i++) {
    x0[i].x -= shift->x;
    x0[i].y -= shift->y;
    x0[i].z -= shift->z;
  }
}

void FIT_only_translation(FIT *fit, VEC *x0, VEC *x1)
{
  VEC com0, com1;
  int i;
  
  FIT_calc_com(fit, x0, &com0);
  FIT_calc_com(fit, x1, &com1);

  com0.x -= com1.x;
  com0.y -= com1.y;
  com0.z -= com1.z;

  FIT_shift(fit, x0, &com0);
}

void FIT_best_fit(FIT *fit, VEC *x0, VEC *x1, VEC *x0r)
{
  int i;
  
  /* copy coordinates */
  FIT_copy_vec(fit, x0, fit->x0);
  FIT_copy_vec(fit, x1, fit->x1);

  /* move to origin */
  FIT_calc_com(fit, fit->x0, &(fit->com0));
  FIT_calc_com(fit, fit->x1, &(fit->com1));
  FIT_shift(fit, fit->x0, &(fit->com0));
  FIT_shift(fit, fit->x1, &(fit->com1));

  /* rotation */
  FIT_fit_rotation(fit, fit->x0, fit->x1);
  FIT_calc_rmsd(fit, fit->x0, fit->x1);

  if (x0r) {
    fit->com1.x *= -1;
    fit->com1.y *= -1;
    fit->com1.z *= -1;
    FIT_shift(fit, fit->x0, &(fit->com1));
    FIT_copy_vec(fit, fit->x0, x0r);
  }
}

/*
     int n;            The number of atoms 
     double *a, *b;    The atom coordinates of molecule A and B 
     double *w;        the NATOM arrey of weight 
     double R[3][3];   Rotation Matrix 
*/
void FIT_fit_rotation(FIT *fit, VEC *x0, VEC *x1)
{
  int i,j;
  double U[3][3];
  double Ut[3][3];
  double O[6][6];
  double J[6][6];
  double h[3][3],k[3][3],D[3];       /* Eigen Vector,Eigen Value */ 
  double ht[3][3],kt[3][3],Dm[3][3]; /* Eigen Vector,Eigen Value */ 
  double det;                        /* Determinant of U */

  FIT_make_U(fit,(double *)x0, (double *)x1, U);

  det = determinant33(U);

  transpose33(U,Ut);

  FIT_make_Omega(O,U,Ut);

  FIT_diag_Omega(det, O, J, h, k, D);

  FIT_make_R(h,k,fit->R);

  FIT_rotation(fit, x0, fit->R);
}

void FIT_make_U(FIT *fit, double *a, double *b, double U[3][3])
{
  int i,j,k;
  double sum;
  double ma,mb;
  double inv_total_w2;
  
  inv_total_w2 = 1.0/(fit->total_w*fit->total_w);

  for (i=0;i<3;i++) {
    for (j=0;j<3;++j) {
      U[i][j] = 0.0; 
      for (k=0;k<fit->n_atom;k++) {
	U[i][j] += fit->w[k] * a[k*3+i] * b[k*3+j];
      }
      U[i][j] *= inv_total_w2;
    }
  }
}

void FIT_make_Omega(double O[6][6],double U[3][3],double Ut[3][3])
{
  int i,j;

  for (i=0;i<6;i++) {
    for (j=0;j<6;j++) {

      O[i][j] = 0.0; 
      if ((3<=i)&&(i<=5)&&(0<=j)&&(j<=2)) {
	O[i][j] = Ut[i-3][j];
      }
      
      if ((0<=i)&&(i<=2)&&(3<=j)&&(j<=5)) {
	O[i][j] = U[i][j-3];
      }
    }
  }
}

void FIT_diag_Omega(double det, double O[6][6],double J[6][6],
		    double h[3][3],double k[3][3],double D[3])
{
  double dt[6];
  int num[6];
  double buff;
  int i,j,bnum,end;
  double r2;
  double cro[3];
  double dp;

  jacobi(6, &O[0][0], dt, &J[0][0]);
  sort_eigen(6, dt, &J[0][0], 1);
  
 /* ----------- SET EIGEN VECTOR ------------------------- */
  r2 = sqrt(2.0);
 
  for (i=0;i<3;++i) {
    for (j=0;j<3;++j) {
      h[i][j] = J[j][i]*r2;
      k[i][j] = J[j+3][i]*r2;
    }
  }
 
  D[0] = dt[0]; D[1] = dt[1]; D[2] = dt[2];

 /* --------- SET h3 SIGN ----------------------- */
  cross3(h[0],h[1],cro);
  if (dot3(h[2],cro)<0.0) {
    h[2][0] *=-1.0; h[2][1] *=-1.0; h[2][2] *=-1.0; 
    k[2][0] *=-1.0; k[2][1] *=-1.0; k[2][2] *=-1.0; 
  } 

  /* ------------ CASE of SINGULAR -------------- */
  if (fabs(det)<0.00001) {
    cross3(h[0],h[1],h[2]); 
    cross3(k[0],k[1],k[2]);
  }

  /* -------------CASE of NEGATIVE DET----------- */

  if (det<0.0) {
    k[2][0] *=-1.0; k[2][1] *= -1.0; k[2][2] *= -1.0;
  }
}


void FIT_make_R(double h[3][3],double k[3][3],double R[3][3])
{
  double kt[3][3];

  transpose33(k,kt);
  mul_mat33(kt,h,R);
}

void FIT_rotation(FIT *fit, VEC *x0, double R[3][3])
{
  VEC x1;
  int i;

  for (i=0;i<fit->n_atom;i++) {
    v_mat_mul(&x1, R, &x0[i]);
    x0[i] = x1;
  }
}


void FIT_append_weight_list(WEIGHT_LIST **headp, double weight)
{
  WEIGHT_LIST *cur, *p;

  cur = emalloc("FIT_append_weight_list", sizeof(WEIGHT_LIST));
  cur->weight = weight;
  cur->group_str = NULL;
  cur->next = NULL;

  if (*headp == NULL) {
    *headp = cur;
  } else {
    for (p=*headp;p->next!=NULL;p=p->next);
    p->next = cur;
  }
}

void FIT_set_weight_list_group_str(WEIGHT_LIST *headp, char *str)
{
  WEIGHT_LIST *cur, *p;

  if (headp == NULL) {
    lprintf("ERROR: set group string before setting weight value.\n");
    return;
  } else {
    for (p=headp;p->next!=NULL;p=p->next);
    append_str_list(&(p->group_str), str);
  }
}
