/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "md_system.h"

#ifdef MPI_SDMD
#include "parallel.h"
#include "sdmd.h"
#endif

void EP_init(EXTRA_POT *ep)
{
  ep->flag = 0;
  ep->head = NULL;
}

void EP_add_item(EXTRA_POT *ep, EP_ITEM *dat)
{
  EP_ITEM *p, *tail;
  dat->next = NULL;
  if (ep->head==NULL) {
    ep->flag = 1;
    ep->head = dat;
  } else {
    for (p=ep->head; p->next!=NULL; p=p->next);
    p->next = dat;
  }
}

void EP_energy_force(EXTRA_POT *ep, MD_SYSTEM *sys, double *energy)
{
  EP_ITEM *p;
  ATOM_DATA *ad;
  BOUNDARY *bc;
  LINKED_CELL *lc;

#ifdef MPI_RDMD
  if (!mpi.master) return;
#endif

  ad = &sys->atom;
  bc = &sys->boundary;
  lc = &sys->linked_cell;

  for (p=ep->head; p!=NULL; p=p->next) {
    switch(p->type) {
    case EP_T_DISTANCE :
      EP_DISTANCE_energy_force((EP_DISTANCE*)p, ad, bc, lc);
      break;
    case EP_T_GROUP_DISTANCE :
      EP_GROUP_DISTANCE_energy_force((EP_GROUP_DISTANCE*)p, ad, bc, lc);
      break;
    case EP_T_POSITION :
      EP_POSITION_energy_force((EP_POSITION*)p, ad);
      break;
/* yss-add */
    case EP_T_GROUP_POSITION :
      EP_GROUP_POSITION_energy_force((EP_GROUP_POSITION*)p, ad, bc, lc);
      break;
    case EP_T_MOLECULE_POSITION :
      EP_MOLECULE_POSITION_energy_force((EP_MOLECULE_POSITION*)p, ad, bc, lc);
      break;
/* yss-add (Mar09/2004) */
    case EP_T_TORSION :
      EP_TORSION_energy_force((EP_TORSION*)p, ad, bc, lc);
      break;
/* yse */

/* YAMANE_ADDED */
    /* for noe_restraint */ 
    case EP_T_NMR_DIS :
      EP_NMR_DIS_energy_force((EP_NMRDIS*)p, ad, lc);
      break;
    /*for noe_dihedral */
    case EP_T_NMR_DIH :
      EP_NMR_DIH_energy_force((EP_NMRDIH*)p, ad, lc);
      break;
    /*for noe_planarity */
    case EP_T_NMR_PLN :
     EP_NMR_PLN_energy_force((EP_NMRPLN*)p, ad, lc);
     break;
/* end of YAMANE_ADDED */

    case EP_T_SMD_CV :
     EP_SMD_CV_energy_force((EP_SMD*)p, ad, lc);
     break;

    case EP_T_RMSD :
     EP_RMSD_energy_force((EP_RMSD*)p, ad, lc);
     break;

    }
    *energy += p->energy;
  }
}

void EP_gradual_change(EXTRA_POT *ep, MD_SYSTEM *sys, int step)
{
  EP_ITEM *p;
  
  for (p=ep->head; p!=NULL; p=p->next) {
    switch(p->type) {
    case EP_T_DISTANCE :
      EP_DISTANCE_gradual_change((EP_DISTANCE*)p,step);
      break;
    case EP_T_GROUP_DISTANCE :
      EP_GROUP_DISTANCE_gradual_change((EP_GROUP_DISTANCE*)p,step);
      break;
    case EP_T_POSITION :
      EP_POSITION_gradual_change((EP_POSITION*)p,step);
      break;
/* yss-add */
    case EP_T_GROUP_POSITION :
      EP_GROUP_POSITION_gradual_change((EP_GROUP_POSITION*)p,step);
      break;
    case EP_T_MOLECULE_POSITION :
      EP_MOLECULE_POSITION_gradual_change((EP_MOLECULE_POSITION*)p,step);
      break;
/* yss-add (Mar09/2004) */
    case EP_T_TORSION :
      EP_TORSION_gradual_change((EP_TORSION*)p,step);
      break;
/* yse */
/* YAMANE_ADDED */
      /* for noe_restraint */ 
    case EP_T_NMR_DIS :
      EP_NMR_DIS_gradual_change((EP_NMRDIS*)p,step);
      break;
      /* for noe_dihedral */ 
    case EP_T_NMR_DIH :
      EP_NMR_DIH_gradual_change((EP_NMRDIH*)p,step);
      break;
      /* for noe_planarity */ 
    case EP_T_NMR_PLN :
      EP_NMR_PLN_gradual_change((EP_NMRPLN*)p,step);
      break;
/* End of YAMANE_ADDED */

    case EP_T_SMD_CV :
      EP_SMD_CV_gradual_change((EP_SMD*)p,sys,step);
      break;

    case EP_T_RMSD :
      EP_RMSD_gradual_change((EP_RMSD*)p, sys, step);
      break;

    }
  }
}

void EP_prop_header(EXTRA_POT *ep, FILE *fp, int *id)
{
  EP_ITEM *p;
  
  for (p=ep->head; p!=NULL; p=p->next) {
    switch(p->type) {
    case EP_T_DISTANCE :
      EP_DISTANCE_prop_header((EP_DISTANCE*)p, fp, id);
      break;
    case EP_T_GROUP_DISTANCE : 
      EP_GROUP_DISTANCE_prop_header((EP_GROUP_DISTANCE*)p, fp, id);
      break;
    case EP_T_POSITION :
      break;
/* yss-add */
    case EP_T_GROUP_POSITION : 
      EP_GROUP_POSITION_prop_header((EP_GROUP_POSITION*)p, fp, id);
      break;
    case EP_T_MOLECULE_POSITION : 
      break;
/* yss-add (Mar09/2004) */
    case EP_T_TORSION :
      EP_TORSION_prop_header((EP_TORSION*)p, fp, id);
      break;
/* yse */

/* YAMANE_ADDED */
    case EP_T_NMR_DIS :
      EP_NMR_DIS_prop_header((EP_NMRDIS*)p, fp, id);
      break;
    case EP_T_NMR_DIH :
      EP_NMR_DIH_prop_header((EP_NMRDIH*)p, fp, id);
      break;
/* End of YAMANE_ADDED */
      
    case EP_T_SMD_CV :
      EP_SMD_CV_prop_header((EP_SMD*)p, fp, id);
      break;

    case EP_T_RMSD :
      EP_RMSD_prop_header((EP_RMSD*)p, fp, id);
      break;
    }
  }
}

void EP_prop_out(EXTRA_POT *ep, MD_SYSTEM *sys, FILE *fp)
{
  EP_ITEM *p;
  
  for (p=ep->head; p!=NULL; p=p->next) {
    switch(p->type) {
    case EP_T_DISTANCE :
      EP_DISTANCE_prop_out((EP_DISTANCE*)p, sys, fp);
      break;
    case EP_T_GROUP_DISTANCE : 
      EP_GROUP_DISTANCE_prop_out((EP_GROUP_DISTANCE*)p, sys, fp);
      break;
    case EP_T_POSITION :
      break;
/* yss-add */
    case EP_T_GROUP_POSITION : 
      EP_GROUP_POSITION_prop_out((EP_GROUP_POSITION*)p, sys, fp);
      break;
    case EP_T_MOLECULE_POSITION : 
      break;
/* yss-add (Mar09/2004) */
    case EP_T_TORSION :
      EP_TORSION_prop_out((EP_TORSION*)p, sys, fp);
      break;
/* yse */

/* YAMANE_ADDED */
    case EP_T_NMR_DIS :
      EP_NMR_DIS_prop_out((EP_NMRDIS*)p, sys, fp);
      break;
    case EP_T_NMR_DIH :
      EP_NMR_DIH_prop_out((EP_NMRDIH*)p, sys, fp);
      break;
/* end of YAMANE_ADDED */

    case EP_T_SMD_CV :
      EP_SMD_CV_prop_out((EP_SMD*)p, sys, fp);
      break;

    case EP_T_RMSD :
      EP_RMSD_prop_out((EP_RMSD*)p, sys, fp);
      break;
      
    }
  }
}

void EP_output(EXTRA_POT *ep, MD_SYSTEM *sys)
{
  EP_ITEM *p;
  
  for (p=ep->head; p!=NULL; p=p->next) {
    switch(p->type) {
    case EP_T_DISTANCE :
      break;
    case EP_T_GROUP_DISTANCE : 
      break;
    case EP_T_POSITION :
      break;
    case EP_T_GROUP_POSITION : 
      break;
    case EP_T_MOLECULE_POSITION : 
      break;
    case EP_T_TORSION :
      break;

    case EP_T_NMR_DIS :
      EP_NMR_DIS_output((EP_NMRDIS*)p, sys);
      break;
    case EP_T_NMR_DIH :
      EP_NMR_DIH_output((EP_NMRDIH*)p, sys);
      break;

    case EP_T_SMD_CV :
      break;
    case EP_T_RMSD :
      break;
    }
  }
}


/*********************************/
/*                               */
/*        EP_DISTANCE            */
/*                               */
/*********************************/
static char *_ep_ptype_name[] = {
  "normal",
  "repulsive",
  "attractive"
};

void EP_DISTANCE_setup(EXTRA_POT *ep,
		       ATOM_DATA *ad,
		       BOUNDARY *bc,
		       LINKED_CELL *lc,
		       char *atom1, char *atom2,
		       double k, double r0,
		       int gradual_change_step, double k1, int virial_flag,
		       int potential_type,
		       int x_flag, int y_flag, int z_flag)
{
  EP_DISTANCE *ed;
  double dx, dy, dz, r;
  int ret1,ret2, atom_list[2];

  ed = emalloc("EP_DISTANCE_setup", sizeof(EP_DISTANCE));
  EP_add_item(ep, (EP_ITEM*) ed);
  
  ed->type = EP_T_DISTANCE;

  ed->atom1 = ATOM_DATA_str_to_atom_no(ad, atom1);
  ed->atom2 = ATOM_DATA_str_to_atom_no(ad, atom2);
  ed->k = k;
  ed->r0 = r0;
  ed->gradual_change_step = gradual_change_step;
  ed->k0 = k;
  ed->k1 = k1;
  ed->virial_flag = virial_flag;
  ed->potential_type = potential_type;
  ed->x_flag = x_flag;
  ed->y_flag = y_flag;
  ed->z_flag = z_flag;
  
  lprintf("Distance Harmonic Restraint:\n");
  lprintf("  "); ret1 = ATOM_DATA_print_atom(ad, ed->atom1);
  lprintf("  "); ret2 = ATOM_DATA_print_atom(ad, ed->atom2);
  if (ret1 != 0 || ret2 != 0) {
    lprintf("ERROR: No atom %s or %s\n", atom1, atom2);
    marble_exit(1);
  }

  dx = ad->x[ed->atom1].x - ad->x[ed->atom2].x;
  dy = ad->x[ed->atom1].y - ad->x[ed->atom2].y;
  dz = ad->x[ed->atom1].z - ad->x[ed->atom2].z;

  if (!ed->x_flag) dx = 0.0;
  if (!ed->y_flag) dy = 0.0;
  if (!ed->z_flag) dz = 0.0;

  /*
  if (bc->type == PERIODIC_BOUNDARY) {
     I should do something .. 
  }
  */
  r = sqrt(dx*dx+dy*dy+dz*dz);
  
  if (r0 < 0.0)
    ed->r0 = r;

  if (potential_type >= 3) {
    lprintf("ERROR:  Potential Type %d is unknown\n", potential_type);
    marble_exit(1);
  }
  lprintf("  Potential Type:  %s\n", _ep_ptype_name[potential_type]);
  lprintf("  Force Constant:    %lf [kcal/mol/Angstrom^2]\n", ed->k);
  lprintf("  Equilibrium Distance: %lf [Angstrom]\n", ed->r0);
  lprintf("  X-Restraint Flag: %s\n", (ed->x_flag) ? "ON" : "OFF"); 
  lprintf("  Y-Restraint Flag: %s\n", (ed->y_flag) ? "ON" : "OFF"); 
  lprintf("  Z-Restraint Flag: %s\n", (ed->z_flag) ? "ON" : "OFF"); 
  lprintf("  Virial Flag: %s\n", (ed->virial_flag) ? "ON" : "OFF");
  if (ed->gradual_change_step) {
    lprintf("  Gradual Change of Force Constant: %f -> %f in %d steps\n",
	    ed->k0, ed->k1, ed->gradual_change_step);
  }
  lprintf("  Current Distance: %lf [Angstrom]\n", r);
  lprintf("\n");
  
#ifdef MPI_SDMD
  atom_list[0] = ed->atom1;
  atom_list[1] = ed->atom2;
  SDMD_setup_ex_tr_atom(lc, 2, atom_list);
#endif  
}

void EP_DISTANCE_energy_force(EP_DISTANCE *ed, ATOM_DATA *ad,
			      BOUNDARY *bc, LINKED_CELL *lc)
{
  double dx, dy, dz;
  double r, F;
  int atom1, atom2;

#ifdef MPI_SDMD
  int pe;
  
  pe = ATOM_CPU(lc, ad, ed->atom1);
  if (pe != mpi.rank) {
    ed->energy = 0.0;
    ed->distance = 0.0;
    return;
  }
#endif  /* MPI_SDMD */

  atom1 = ed->atom1;
  atom2 = ed->atom2;

  dx = ad->x[atom1].x - ad->x[atom2].x;
  dy = ad->x[atom1].y - ad->x[atom2].y;
  dz = ad->x[atom1].z - ad->x[atom2].z;

  if (!ed->x_flag) dx = 0.0;
  if (!ed->y_flag) dy = 0.0;
  if (!ed->z_flag) dz = 0.0;

  /*
  if (bc->type == PERIODIC_BOUNDARY) {
  }
  */
  
  r = sqrt(dx*dx+dy*dy+dz*dz);
  ed->distance = r;

  if (ed->r0 == 0.0) {
    ed->energy = ed->k * r * r;
    F = -2.0 * ed->k;

  } else {

    switch (ed->potential_type) {
    case 0:  /* full */
      ed->energy = ed->k * (r - ed->r0) * (r - ed->r0);

      if (r < 1.0e-10) return; /* not good */
      F = -2.0 * ed->k * (r - ed->r0) / r;
      break;
    case 1:  /* repulsive */
      if (r <= ed->r0) {
	ed->energy = ed->k * (r - ed->r0) * (r - ed->r0);

	if (r < 1.0e-10) return; /* not good */
	F = -2.0 * ed->k * (r - ed->r0) / r;
      } else {
	ed->energy = 0.0;
	F = 0.0;
      }
      break;
    case 2:  /* attractive */
      if (r >= ed->r0) {
	ed->energy = ed->k * (r - ed->r0) * (r - ed->r0);

	if (r < 1.0e-10) return; /* not good */
	F = -2.0 * ed->k * (r - ed->r0) / r;
      } else {
	ed->energy = 0.0;
	F = 0.0;
      }
      break;
    }
  }
  
  ad->f[atom1].x += dx * F;
  ad->f[atom1].y += dy * F;
  ad->f[atom1].z += dz * F;
  ad->f[atom2].x -= dx * F;
  ad->f[atom2].y -= dy * F;
  ad->f[atom2].z -= dz * F;

  if (ed->virial_flag) {
    ad->virial[0] += F * dx * dx;
    ad->virial[1] += F * dy * dy;
    ad->virial[2] += F * dz * dz;
    ad->virial[3] += F * dx * dy;
    ad->virial[4] += F * dx * dz;
    ad->virial[5] += F * dy * dz;
  }
}

void EP_DISTANCE_prop_header(EP_DISTANCE *ed,
			     FILE *fp, int *id)
{
  fprintf(fp, "%2dDISTANCE    ", (*id)++);
  fprintf(fp, "%2dREST_ENE    ", (*id)++);
}

void EP_DISTANCE_prop_out(EP_DISTANCE *ed, MD_SYSTEM *sys, FILE *fp)
{
#ifdef MPI_SDMD
  int pe;
  LINKED_CELL *lc;
  ATOM_DATA *ad;
  MPI_Status stat;
  double buf[2];

  lc = &sys->linked_cell;
  ad = &sys->atom;
  pe = ATOM_CPU(lc, ad, ed->atom1);
  
  if (mpi.master) {
    if (pe != mpi.rank) {
      MPI_Recv(buf,2,MPI_DOUBLE, pe, SDMD_TAG_PROP, mpi.comm, &stat);
      ed->energy   = buf[0];
      ed->distance = buf[1];
    }
  } else {
    if (pe == mpi.rank) {
      buf[0] = ed->energy;
      buf[1] = ed->distance;
      MPI_Send(buf,2,MPI_DOUBLE, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
    }
    return;
  }
#endif /* MPI_SDMD */
  
  fprintf(fp, " %13.6e %13.6e", ed->distance, ed->energy);
  
}

void EP_DISTANCE_gradual_change(EP_DISTANCE *ed, int step)
{
  double lambda;
  if (ed->gradual_change_step) {
    if (step < ed->gradual_change_step) {
      lambda = (double)step/ed->gradual_change_step;
      ed->k = ed->k0*(1.0-lambda) + ed->k1*lambda;
    } else {
      ed->k = ed->k1;
    }
  }
}


/*********************************/
/*                               */
/*      EP_GROUP_DISTANCE        */
/*                               */
/*********************************/

void EP_GROUP_DISTANCE_setup(EXTRA_POT *ep,
			     ATOM_DATA *ad,
			     BOUNDARY *bc,
			     LINKED_CELL *lc,
			     STR_LIST *group1_str, STR_LIST *group2_str,
			     int group1, int group2,
			     double k, double r0,
			     int gradual_change_step, double k1, 
			     int gradual_change_step_r0, double r1, 
			     int virial_flag,
			     int potential_type,
			     int x_flag, int y_flag, int z_flag)
{
  EP_GROUP_DISTANCE *eg;
  double dx, dy, dz, r;
  double x1, y1, z1, x2, y2, z2;
  double w, w1, w2;
  int i, ret1, ret2;
  char *fname = "EP_GROUP_DISTANCE_setup";

  eg = emalloc(fname, sizeof(EP_GROUP_DISTANCE));
  EP_add_item(ep, (EP_ITEM*) eg);
  
  eg->type = EP_T_GROUP_DISTANCE;
  eg->group1 = group1;
  eg->group2 = group2;
  eg->k = k;
  eg->r0 = r0;
  eg->gradual_change_step = gradual_change_step;
  eg->k0 = k;
  eg->k1 = k1;
  eg->gradual_change_step_r0 = gradual_change_step_r0;
  eg->r00 = r0;
  eg->r01 = r1;
  eg->virial_flag = virial_flag;
  eg->potential_type = potential_type;
  eg->x_flag = x_flag;
  eg->y_flag = y_flag;
  eg->z_flag = z_flag;

  lprintf("Group Distance Harmonic Restraint:\n");

  if (group1_str!=NULL) {
    eg->group1 = group1 = 31;
    ATOM_DATA_set_group_str_list(ad, group1_str, group1, "  Group1:");
    eg->n_list1 = ATOM_DATA_count_group_atoms(ad, group1);
    eg->list1 = emalloc(fname,sizeof(int)*(eg->n_list1));
    ATOM_DATA_make_list_of_group_atoms(ad,group1,eg->list1);
  } else {
    eg->n_list1 = ATOM_DATA_count_group_atoms(ad, group1);
    eg->list1 = emalloc(fname,sizeof(int)*(eg->n_list1));
    ATOM_DATA_make_list_of_group_atoms(ad,group1,eg->list1);
    lprintf("  Group No. %d: %d atoms\n", eg->group1, eg->n_list1);
  }

  if (group2_str!=NULL) {
    eg->group2 = group2 = 31;
    ATOM_DATA_set_group_str_list(ad, group2_str, group2, "  Group2:");
    eg->n_list2 = ATOM_DATA_count_group_atoms(ad, group2);
    eg->list2 = emalloc(fname,sizeof(int)*(eg->n_list2));
    ATOM_DATA_make_list_of_group_atoms(ad,group2,eg->list2);
  } else {
    eg->n_list2 = ATOM_DATA_count_group_atoms(ad, group2);
    eg->list2 = emalloc(fname,sizeof(int)*(eg->n_list2));
    ATOM_DATA_make_list_of_group_atoms(ad,group2,eg->list2);
    lprintf("  Group No. %d: %d atoms\n", eg->group2, eg->n_list2);
  }

  if (eg->n_list1 == 0 || eg->n_list2 == 0) {
    lprintf("ERROR: No atom in the group1 or group2\n");
    marble_exit(1);
  }

  x1 = y1 = z1 = x2 = y2 = z2 = w1 = w2 = 0.0;
  for (i=0;i<eg->n_list1;i++) {
    w   = ad->w[eg->list1[i]];
    x1 += ad->x[eg->list1[i]].x * w;
    y1 += ad->x[eg->list1[i]].y * w;
    z1 += ad->x[eg->list1[i]].z * w;
    w1 += w;
  }
  x1 /= w1;
  y1 /= w1; 
  z1 /= w1;
  
  for (i=0;i<eg->n_list2;i++) {
    w   = ad->w[eg->list2[i]];
    x2 += ad->x[eg->list2[i]].x * w;
    y2 += ad->x[eg->list2[i]].y * w;
    z2 += ad->x[eg->list2[i]].z * w;
    w2 += w;
  }
  x2 /= w2;
  y2 /= w2;
  z2 /= w2;
  
  dx = x1 - x2;
  dy = y1 - y2;
  dz = z1 - z2;

  if (!eg->x_flag) dx = 0.0;
  if (!eg->y_flag) dy = 0.0;
  if (!eg->z_flag) dz = 0.0;

  /*
  if (bc->type == PERIODIC_BOUNDARY) {
  }
  */
  r = sqrt(dx*dx+dy*dy+dz*dz);

  if (r0 < 0.0) 
    eg->r00 = eg->r0 = r;
  
  if (potential_type >= 3) {
    lprintf("ERROR:  Potential Type %d is unknown\n", potential_type);
    marble_exit(1);
  }
  lprintf("  Potential Type:  %s\n", _ep_ptype_name[potential_type]);
  lprintf("  Force Constant:    %lf [kcal/mol/Angstrom^2]\n", eg->k);
  lprintf("  Equilibrium Distance: %lf [Angstrom]\n", eg->r0);
  lprintf("  X-Restraint Flag: %s\n", (eg->x_flag) ? "ON" : "OFF"); 
  lprintf("  Y-Restraint Flag: %s\n", (eg->y_flag) ? "ON" : "OFF"); 
  lprintf("  Z-Restraint Flag: %s\n", (eg->z_flag) ? "ON" : "OFF"); 
  lprintf("  Virial Flag: %s\n", (eg->virial_flag) ? "ON" : "OFF");
  if (eg->gradual_change_step) {
    lprintf("  Gradual Change of Force Constant: %f -> %f in %d steps\n",
	    eg->k0, eg->k1, eg->gradual_change_step);
  }
  if (eg->gradual_change_step_r0) {
    lprintf("  Gradual Change of Equilibrium Distance: %f -> %f in %d steps\n",
	    eg->r00, eg->r01, eg->gradual_change_step_r0);
  }
  lprintf("  Center of Mass of Group1: (%7.3f,%7.3f,%7.3f)\n", x1, y1, z1);
  lprintf("  Center of Mass of Group2: (%7.3f,%7.3f,%7.3f)\n", x2, y2, z2);
  lprintf("  Current Distance: %lf [Angstrom]\n", r);
  lprintf("\n");
  
#ifdef MPI_SDMD
  eg->list1 = erealloc(fname, eg->list1, sizeof(int)*(eg->n_list1+eg->n_list2));
  for (i=0;i<eg->n_list2;i++) {
    eg->list1[eg->n_list1+i] = eg->list2[i];
  }
  SDMD_setup_ex_tr_atom(lc, eg->n_list1+eg->n_list2, eg->list1);
#endif  
}

void EP_GROUP_DISTANCE_energy_force(EP_GROUP_DISTANCE *eg,
				    ATOM_DATA *ad,
				    BOUNDARY *bc,
				    LINKED_CELL *lc)
{
  double dx, dy, dz;
  double r, F;
  double w1, w2, w;
  double x1, y1, z1, x2, y2, z2;
  int i;

#ifdef MPI_SDMD
  int pe;
  
  pe = ATOM_CPU(lc, ad, eg->list1[0]);
  if (pe != mpi.rank) {
    eg->energy = 0.0;
    eg->distance = 0.0;
    return;
  }
#endif  /* MPI_SDMD */
  
  x1 = y1 = z1 = x2 = y2 = z2 = w1 = w2 = 0.0;
  for (i=0;i<eg->n_list1;i++) {
    w   = ad->w[eg->list1[i]];
    x1 += ad->x[eg->list1[i]].x * w;
    y1 += ad->x[eg->list1[i]].y * w;
    z1 += ad->x[eg->list1[i]].z * w;
    w1 += w;
  }
  x1 /= w1;
  y1 /= w1; 
  z1 /= w1;
  
  for (i=0;i<eg->n_list2;i++) {
    w   = ad->w[eg->list2[i]];
    x2 += ad->x[eg->list2[i]].x * w;
    y2 += ad->x[eg->list2[i]].y * w;
    z2 += ad->x[eg->list2[i]].z * w;
    w2 += w;
  }
  x2 /= w2;
  y2 /= w2;
  z2 /= w2;
  
  dx = x1 - x2;
  dy = y1 - y2;
  dz = z1 - z2;

  if (!eg->x_flag) dx = 0.0;
  if (!eg->y_flag) dy = 0.0;
  if (!eg->z_flag) dz = 0.0;

  /*
  if (bc->type == PERIODIC_BOUNDARY) {
  }
  */

  r = sqrt(dx*dx+dy*dy+dz*dz);
  eg->distance = r;

  if (eg->r0 == 0.0) {
    eg->energy = eg->k * r * r;
    F = -2.0 * eg->k;
  } else {

    switch (eg->potential_type) {
    case 0:  /* full */
      eg->energy = eg->k * (r - eg->r0) * (r - eg->r0);

      if (r < 1.0e-10) return; /* not good */
      F = -2.0 * eg->k * (r - eg->r0) / r;
      break;
    case 1:  /* repulsive */
      if (r <= eg->r0) {
	eg->energy = eg->k * (r - eg->r0) * (r - eg->r0);

	if (r < 1.0e-10) return; /* not good */
	F = -2.0 * eg->k * (r - eg->r0) / r;
      } else {
	eg->energy = 0.0;
	F = 0.0;
      }
      break;
    case 2:  /* attractive */
      if (r >= eg->r0) {
	eg->energy = eg->k * (r - eg->r0) * (r - eg->r0);

	if (r < 1.0e-10) return; /* not good */
	F = -2.0 * eg->k * (r - eg->r0) / r;
      } else {
	eg->energy = 0.0;
	F = 0.0;
      }
      break;
    }
  }

  for (i=0;i<eg->n_list1;i++) {
    w = ad->w[eg->list1[i]];
    ad->f[eg->list1[i]].x += dx * F * w / w1;
    ad->f[eg->list1[i]].y += dy * F * w / w1;
    ad->f[eg->list1[i]].z += dz * F * w / w1;
  }
  for (i=0;i<eg->n_list2;i++) {
    w = ad->w[eg->list2[i]];
    ad->f[eg->list2[i]].x -= dx * F * w / w2;
    ad->f[eg->list2[i]].y -= dy * F * w / w2;
    ad->f[eg->list2[i]].z -= dz * F * w / w2;
  }
  
  if (eg->virial_flag) {
    ad->virial[0] += F * dx * dx;
    ad->virial[1] += F * dy * dy;
    ad->virial[2] += F * dz * dz;
    ad->virial[3] += F * dx * dy;
    ad->virial[4] += F * dx * dz;
    ad->virial[5] += F * dy * dz;
  }
}

void EP_GROUP_DISTANCE_prop_header(EP_GROUP_DISTANCE *eg,
				    FILE *fp, int *id)
{
  fprintf(fp, "%2dDISTANCE    ", (*id)++);
  fprintf(fp, "%2dREST_ENE    ", (*id)++);
}

void EP_GROUP_DISTANCE_prop_out(EP_GROUP_DISTANCE *eg,
				MD_SYSTEM *sys, FILE *fp)
{
#ifdef MPI_SDMD
  int pe;
  LINKED_CELL *lc;
  ATOM_DATA *ad;
  MPI_Status stat;
  double buf[2];

  lc = &sys->linked_cell;
  ad = &sys->atom;
  pe = ATOM_CPU(lc, ad, eg->list1[0]);
  if (mpi.master) {
    if (pe != mpi.rank) {
      MPI_Recv(buf,2,MPI_DOUBLE, pe, SDMD_TAG_PROP, mpi.comm, &stat);
      eg->energy   = buf[0];
      eg->distance = buf[1];
    }
  } else {
    if (pe == mpi.rank) {
      buf[0] = eg->energy;
      buf[1] = eg->distance;
      MPI_Send(buf,2,MPI_DOUBLE, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
    }
    return;
  }
#endif /* MPI_SDMD */
  
  fprintf(fp, " %13.6e %13.6e", eg->distance, eg->energy);
}

void EP_GROUP_DISTANCE_gradual_change(EP_GROUP_DISTANCE *eg, int step)
{
  double lambda;
  if (eg->gradual_change_step) {
    if (step < eg->gradual_change_step) {
      lambda = (double)step/eg->gradual_change_step;
      eg->k = eg->k0*(1.0-lambda) + eg->k1*lambda;
    } else {
      eg->k = eg->k1;
    }
  }

  if (eg->gradual_change_step_r0) {
    if (step < eg->gradual_change_step_r0) {
      lambda = (double)step/eg->gradual_change_step_r0;
      eg->r0 = eg->r00*(1.0-lambda) + eg->r01*lambda;
    } else {
      eg->r0 = eg->r01;
    }
  }

}

/*********************************/
/*                               */
/*        EP_POSITION            */
/*                               */
/*********************************/

void EP_POSITION_setup(EXTRA_POT *ep,
		       ATOM_DATA *ad, STR_LIST *group_str, int group_no,
		       char *file_name, double k,
		       int gradual_change_step, double k1, int virial_flag,
		       int x_flag, int y_flag, int z_flag)
{
  EP_POSITION *epos;
  FILE *fp;
  char buf[100];
  int natom, i, g_id;
  char *fname = "EP_POSTION_setup";
  
  epos = emalloc(fname, sizeof(EP_POSITION));
  EP_add_item(ep, (EP_ITEM*) epos);

  epos->type = EP_T_POSITION;
  strcpy(epos->fname, file_name);
  epos->k = epos->k0 = k;
  epos->k1 = k1;
  epos->virial_flag = virial_flag;
  epos->gradual_change_step = gradual_change_step;
  epos->group_no = group_no;
  /* yss-add */
  epos->x_flag = x_flag;
  epos->y_flag = y_flag;
  epos->z_flag = z_flag;
  /* yse */

  ad->ex_force_flag = 1;
  
  lprintf("POSITION HARMONIC RESTRAINT:\n");

  if (group_str!=NULL) {
    epos->group_no = group_no = 31;
    ATOM_DATA_set_group_str_list(ad, group_str, group_no, "  GROUP:");
  } else {
    lprintf("  GROUP NO. %d\n", epos->group_no);
  }
  epos->atom_flag = emalloc(fname,sizeof(char)*(ad->natom));
  ATOM_DATA_set_flag_of_group_atoms(ad,group_no,epos->atom_flag);

  lprintf("  REFERENCE COORIDNATE FILE: %s\n", epos->fname);
  lprintf("  FORCE CONSTANT: %lf [kcal/mol/Angstrom^2]\n", epos->k);
  lprintf("  X-RESTRAINT FLAG: %s\n", (epos->x_flag) ? "ON" : "OFF"); 
  lprintf("  Y-RESTRAINT FLAG: %s\n", (epos->y_flag) ? "ON" : "OFF"); 
  lprintf("  Z-RESTRAINT FLAG: %s\n", (epos->z_flag) ? "ON" : "OFF"); 
  lprintf("  VIRIAL FLAG: %s\n", (epos->virial_flag) ? "ON" : "OFF");
  if (epos->gradual_change_step) {
    lprintf("  GRADUAL CHANGE OF FORCE CONSTANT: %f -> %f in %d steps\n", epos->k, epos->k1, epos->gradual_change_step);
  }

#if 0  
  if ((fp = fopen(file_name, "r")) == NULL) {
    lprintf("ERROR: %s: No such file\n", file_name);
    marble_exit(1);
  }
  epos->x0 = emalloc(fname,sizeof(VEC) * ad->natom);
  fgets(buf, 80, fp);  /* skip header */
  fscanf(fp,"%d", &(natom));
  if (natom != ad->natom) {
    lprintf("ERROR: Inconsistent atom number in crd file %s\n",fname);
    marble_exit(1);
  }
  for (i = 0; i < natom; i++) {
    if (fscanf(fp,"%lf%lf%lf", &(epos->x0[i].x),&(epos->x0[i].y),&(epos->x0[i].z)) != 3) {
      lprintf("ERROR: Invalid atom coordinate in crd file %s\n",fname);
      marble_exit(1);
    }
  }
  fclose(fp);
#endif

  epos->x0 = ATOM_DATA_get_ref_crd(ad, file_name);

  lprintf("\n");
}

void EP_POSITION_energy_force(EP_POSITION *epos,ATOM_DATA *ad)
{
  int i, j;
  unsigned int gid;
  double dx, dy, dz, k2, fx, fy, fz;

  gid = 1 << epos->group_no;
  k2 = epos->k * 2.0;
  epos->energy = 0.0;

#ifdef MPI_SDMD
  for (i=ad->node_atom_h;i>=0;i=ad->node_atom_n[i]) {
#else  
  for (i=0;i<ad->natom;i++) {
#endif    
    if (epos->atom_flag[i]) {
      /* yss-add */
      /*
      dx = ad->x[i].x - epos->x0[i].x;
      dy = ad->x[i].y - epos->x0[i].y;
      dz = ad->x[i].z - epos->x0[i].z;
      */
      if (epos->x_flag) {
	dx = ad->x[i].x - epos->x0[i].x;
      }else{
	dx = 0.0;
      }	
      if (epos->y_flag) {
	dy = ad->x[i].y - epos->x0[i].y;
      }else{
	dy = 0.0;
      }	
      if (epos->z_flag) {
	dz = ad->x[i].z - epos->x0[i].z;
      }else{
	dz = 0.0;
      }	
      /* yse */
      epos->energy += (SQR(dx) + SQR(dy) + SQR(dz))*epos->k;
      fx = -k2 * dx;
      fy = -k2 * dy;
      fz = -k2 * dz;
      ad->f[i].x += fx;
      ad->f[i].y += fy;
      ad->f[i].z += fz;

      if (epos->virial_flag) {
	ad->virial[0] += fx * ad->x[i].x;
	ad->virial[1] += fy * ad->x[i].y;
	ad->virial[2] += fz * ad->x[i].z;
	ad->virial[3] += (fx * ad->x[i].y + fy * ad->x[i].x) * 0.5;
	ad->virial[4] += (fx * ad->x[i].z + fz * ad->x[i].x) * 0.5;
	ad->virial[5] += (fy * ad->x[i].z + fz * ad->x[i].y) * 0.5;
      }
    }
  }
}
  
void EP_POSITION_gradual_change(EP_POSITION *epos, int step)
{
  double lambda;
  if (epos->gradual_change_step) {
    if (step < epos->gradual_change_step) {
      lambda = (double)step/epos->gradual_change_step;
      epos->k = epos->k0*(1.0-lambda) + epos->k1*lambda;
    } else {
      epos->k = epos->k1;
    }
  }
}

/* yss-add */

/*********************************/
/*                               */
/*      EP_GROUP_POSITION        */
/*                               */
/*********************************/

void EP_GROUP_POSITION_setup(EXTRA_POT *ep,
			     ATOM_DATA *ad,
			     BOUNDARY *bc,
			     LINKED_CELL *lc,
			     STR_LIST *group_str,
			     int group_no, char *file_name, double target[3], double k,
			     int gradual_change_step, double k1, int virial_flag,
			     int x_flag, int y_flag, int z_flag)
{
  EP_GROUP_POSITION *egpos;
  FILE *fp;
  char buf[100];
  int natom, i;
  char *fname = "EP_GROUP_POSITION_setup";
  double x1, y1, z1;
  double w, w1;
  int ret;
  VEC *x0;

  egpos = emalloc(fname, sizeof(EP_GROUP_POSITION));
  EP_add_item(ep, (EP_ITEM*) egpos);
  
  egpos->type = EP_T_GROUP_POSITION;
  strcpy(egpos->fname, file_name);

  egpos->group_no = group_no;
  egpos->k = egpos->k0 = k;
  egpos->k1 = k1;
  egpos->virial_flag = virial_flag;
  egpos->gradual_change_step = gradual_change_step;
  egpos->x_flag = x_flag;
  egpos->y_flag = y_flag;
  egpos->z_flag = z_flag;

  ad->ex_force_flag = 1;

   
  lprintf("GROUP POSITION HARMONIC RESTRAINT:\n");

  if (group_str!=NULL) {
    egpos->group_no = group_no = 31;
    ATOM_DATA_set_group_str_list(ad, group_str, group_no, "  GROUP:");
    egpos->n_list = ATOM_DATA_count_group_atoms(ad, group_no);
    egpos->list = emalloc(fname,sizeof(int)*(egpos->n_list));
    ATOM_DATA_make_list_of_group_atoms(ad,group_no,egpos->list);
  } else {
    egpos->n_list = ATOM_DATA_count_group_atoms(ad, group_no);
    egpos->list = emalloc(fname,sizeof(int)*(egpos->n_list));
    ATOM_DATA_make_list_of_group_atoms(ad,group_no,egpos->list);
    lprintf("  GROUP NO. %d, %d atoms\n", egpos->group_no, egpos->n_list);
  }

  if (file_name[0] != '\0') {
#if 0
    if ((fp = fopen(file_name, "r")) == NULL) {
      lprintf("ERROR: %s: No such file\n", file_name);
      marble_exit(1);
    } 
    x0 = emalloc(fname,sizeof(VEC) * ad->natom);
    fgets(buf, 80, fp);  /* skip header */
    fscanf(fp,"%d", &(natom));
    
    if (natom != ad->natom) {
      lprintf("ERROR: Inconsistent atom number in crd file %s\n",fname);
      marble_exit(1);
    }                              
    for (i = 0; i < natom; i++) {  
      if (fscanf(fp,"%lf%lf%lf", &(x0[i].x),&(x0[i].y),&(x0[i].z)) != 3) {
	lprintf("ERROR: Invalid atom coordinate in crd file %s\n",fname);
	marble_exit(1);
      }                  
    }                    
    fclose(fp);          
#else
    x0 = ATOM_DATA_get_ref_crd(ad, file_name);
#endif  

    x1 = y1 = z1 = w1 = 0.0;
    for (i=0;i<egpos->n_list;i++) {
      w   = ad->w[egpos->list[i]];
      x1 += x0[egpos->list[i]].x * w;
      y1 += x0[egpos->list[i]].y * w;
      z1 += x0[egpos->list[i]].z * w;
      w1 += w;
    }
    x1 /= w1;
    y1 /= w1; 
    z1 /= w1;
    free(x0);
  } else {
    x1 = target[0];
    y1 = target[1];
    z1 = target[2];
  }
  egpos->x0 = x1;
  egpos->y0 = y1;
  egpos->z0 = z1;
  
  lprintf("  FORCE CONSTANT:    %lf [kcal/mol/Angstrom^2]\n", egpos->k);
  /* yss-add */
  lprintf("  X-RESTRAINT FLAG: %s\n", (egpos->x_flag) ? "ON" : "OFF"); 
  lprintf("  Y-RESTRAINT FLAG: %s\n", (egpos->y_flag) ? "ON" : "OFF"); 
  lprintf("  Z-RESTRAINT FLAG: %s\n", (egpos->z_flag) ? "ON" : "OFF"); 
  /* yse */
  lprintf("  VIRIAL FLAG: %s\n", (egpos->virial_flag) ? "ON" : "OFF");
  if (egpos->gradual_change_step) {
    lprintf("  GRADUAL CHANGE OF FORCE CONSTANT: %f -> %f in %d steps\n",
	    egpos->k0, egpos->k1, egpos->gradual_change_step);
  }

  if (egpos->fname[0])
    lprintf("  REFERENCE COORDINATE FILE: %s\n", egpos->fname); 
  lprintf("  REFERENCE POINT:        (%7.3f,%7.3f,%7.3f)\n", x1, y1, z1);
  
  x1 = y1 = z1 = w1 = 0.0;
  for (i=0;i<egpos->n_list;i++) {
    w   = ad->w[egpos->list[i]];
    x1 += ad->x[egpos->list[i]].x * w;
    y1 += ad->x[egpos->list[i]].y * w;
    z1 += ad->x[egpos->list[i]].z * w;
    w1 += w;
  }
  x1 /= w1;
  y1 /= w1; 
  z1 /= w1;
  lprintf("  CURRENT CENTER OF MASS: (%7.3f,%7.3f,%7.3f)\n", x1, y1, z1);

  lprintf("\n");


#ifdef MPI_SDMD
  SDMD_setup_ex_tr_atom(lc, egpos->n_list, egpos->list);
#endif  
}

void EP_GROUP_POSITION_energy_force(EP_GROUP_POSITION *egpos,
				    ATOM_DATA *ad,
				    BOUNDARY *bc,
				    LINKED_CELL *lc)
{
  double dx, dy, dz;
  double fx, fy, fz;
  double w1, w;
  double x1, y1, z1;
  int i;

#ifdef MPI_SDMD
  int pe;
  
  pe = ATOM_CPU(lc, ad, egpos->list[0]);
  if (pe != mpi.rank) {
    egpos->energy = 0.0;
    return;
  }
#endif  /* MPI_SDMD */
  
  x1 = y1 = z1 = w1 = 0.0;
  for (i=0;i<egpos->n_list;i++) {
    w   = ad->w[egpos->list[i]];
    x1 += ad->x[egpos->list[i]].x * w;
    y1 += ad->x[egpos->list[i]].y * w;
    z1 += ad->x[egpos->list[i]].z * w;
    w1 += w;
  }
  x1 /= w1;
  y1 /= w1; 
  z1 /= w1;

  if (egpos->x_flag) {
    dx = x1 - egpos->x0;
  }else{
    dx = 0.0;
  }
  if (egpos->y_flag) {
    dy = y1 - egpos->y0;
  }else{
    dy = 0.0;
  }
  if (egpos->z_flag) {
    dz = z1 - egpos->z0;
  }else{
    dz = 0.0;
  }

  egpos->energy = (SQR(dx) + SQR(dy) + SQR(dz))*egpos->k;
  fx = -2.0 * egpos->k * dx;
  fy = -2.0 * egpos->k * dy;
  fz = -2.0 * egpos->k * dz;

  /* yss-debug 
   * lprintf("  Force for Group1: (%.3f,%.3f,%.3f)\n", fx, fy, fz);
   * lprintf("  Deviation for Group1: (%.3f,%.3f,%.3f)\n", dx, dy, dz);
   * yse */

  for (i=0;i<egpos->n_list;i++) {
    w = ad->w[egpos->list[i]];
    ad->f[egpos->list[i]].x += fx * w / w1;
    ad->f[egpos->list[i]].y += fy * w / w1;
    ad->f[egpos->list[i]].z += fz * w / w1;

    if (egpos->virial_flag) {
      ad->virial[0] += fx * ad->x[egpos->list[i]].x * w/w1;
      ad->virial[1] += fy * ad->x[egpos->list[i]].y * w/w1;
      ad->virial[2] += fz * ad->x[egpos->list[i]].z * w/w1;
      ad->virial[3] += (fx * ad->x[egpos->list[i]].y + fy * ad->x[egpos->list[i]].x) * 0.5 * w/w1;
      ad->virial[4] += (fx * ad->x[egpos->list[i]].z + fz * ad->x[egpos->list[i]].x) * 0.5 * w/w1;
      ad->virial[5] += (fy * ad->x[egpos->list[i]].z + fz * ad->x[egpos->list[i]].y) * 0.5 * w/w1;
    }
  }
}


void EP_GROUP_POSITION_prop_header(EP_GROUP_POSITION *egpos,
				   FILE *fp, int *id)
{
  fprintf(fp, "%2dGP_ENE      ", (*id)++);
}

void EP_GROUP_POSITION_prop_out(EP_GROUP_POSITION *egpos,
				MD_SYSTEM *sys, FILE *fp)
{
#ifdef MPI_SDMD
  int pe;
  LINKED_CELL *lc;
  ATOM_DATA *ad;
  MPI_Status stat;
  double buf[2];

  lc = &sys->linked_cell;
  ad = &sys->atom;
  pe = ATOM_CPU(lc, ad, egpos->list[0]);
  
  if (mpi.master) {
    if (pe != mpi.rank) {
      MPI_Recv(buf,1,MPI_DOUBLE, pe, SDMD_TAG_PROP, mpi.comm, &stat);
      egpos->energy   = buf[0];
    }
  } else {
    if (pe == mpi.rank) {
      buf[0] = egpos->energy;
      MPI_Send(buf,1,MPI_DOUBLE, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
    }
    return;
  }
#endif /* MPI_SDMD */
  
  fprintf(fp, " %13.6e", egpos->energy);
}


void EP_GROUP_POSITION_gradual_change(EP_GROUP_POSITION *egpos, int step)
{
  double lambda;
  if (egpos->gradual_change_step) {
    if (step < egpos->gradual_change_step) {
      lambda = (double)step/egpos->gradual_change_step;
      egpos->k = egpos->k0*(1.0-lambda) + egpos->k1*lambda;
    } else {
      egpos->k = egpos->k1;
    }
  }
}



/*********************************/
/*                               */
/*      EP_MOLECULE_POSITION     */
/*                               */
/*********************************/

#if 1

void EP_MOLECULE_POSITION_setup(EXTRA_POT *ep,
				ATOM_DATA *ad,
				BOUNDARY *bc,
				LINKED_CELL *lc,
				int group_no, char *file_name, double k,
				int gradual_change_step, double k1, int virial_flag,
				int mole_residue,
				int x_flag, int y_flag, int z_flag)
{
  EP_MOLECULE_POSITION *empos;
  FILE *fp;
  char buf[100];
  int natom, i, j, ii;
  char *fname = "EP_MOLECULE_POSITION_setup";
  double x1, y1, z1;
  double w, w1;
  int ret;
  VEC *x0;
  int *mole_list;

  empos = emalloc(fname, sizeof(EP_MOLECULE_POSITION));
  EP_add_item(ep, (EP_ITEM*) empos);
  
  empos->type = EP_T_MOLECULE_POSITION;
  strcpy(empos->fname, file_name);

  empos->group_no = group_no;
  empos->k = empos->k0 = k;
  empos->k1 = k1;
  empos->virial_flag = virial_flag;
  empos->gradual_change_step = gradual_change_step;
  empos->x_flag = x_flag;
  empos->y_flag = y_flag;
  empos->z_flag = z_flag;
  empos->mole_residue = mole_residue; 

  ad->ex_force_flag = 1;

  /*
  empos->n_list = ATOM_DATA_count_group_atoms(ad, group_no);
  empos->list = emalloc(fname,sizeof(int)*(empos->n_list));
  ATOM_DATA_make_list_of_group_atoms(ad,group_no,empos->list);
  */
  empos->n_mole = ATOM_DATA_count_group_molecules(ad,group_no);
  empos->mole = emalloc(fname,sizeof(EP_MOL_DATA)*(empos->n_mole));
  mole_list = emalloc(fname,sizeof(int)*(empos->n_mole));
  ATOM_DATA_make_list_of_group_molecules(ad,group_no,mole_list);

  lprintf("MOLECULE POSITION HARMONIC RESTRAINT:\n");
  lprintf("  REFERENCE COORDINATE FILE: %s\n", empos->fname); 
  lprintf("  GROUP NO: %d, ", empos->group_no);
  lprintf("%d molecules\n", empos->n_mole);

  for (i=0;i<empos->n_mole;i++) {
    empos->mole[i].no = mole_list[i];
    empos->mole[i].n_list = ATOM_DATA_count_group_atoms_in_mol(ad,group_no,mole_list[i]);
    empos->mole[i].list = emalloc(fname,sizeof(int)*empos->mole[i].n_list);
    ATOM_DATA_make_list_of_group_atoms_in_mol(ad,group_no,mole_list[i], empos->mole[i].list);
    /*
    lprintf("          : %d atoms in molecule No. %d\n", empos->mole[i].n_list, empos->mole[i].no+1);
    */
  }

#if 0
  if ((fp = fopen(file_name, "r")) == NULL) {
    lprintf("ERROR: %s: No such file\n", file_name);
        marble_exit(1);
  } 
  x0 = emalloc(fname,sizeof(VEC) * ad->natom);
  fgets(buf, 80, fp);  /* skip header */
  fscanf(fp,"%d", &(natom));
  if (natom != ad->natom) {
    lprintf("ERROR: Inconsistent atom number in crd file %s\n",fname);
    marble_exit(1);
  }                              
  for (i = 0; i < natom; i++) {  
    if (fscanf(fp,"%lf%lf%lf", &(x0[i].x),&(x0[i].y),&(x0[i].z)) != 3) {
      lprintf("ERROR: Invalid atom coordinate in crd file %s\n",fname);
            marble_exit(1);
    }                  
  }                    
  fclose(fp);          
#else
  x0 = ATOM_DATA_get_ref_crd(ad, file_name);
#endif

  lprintf("  FORCE CONSTANT:    %lf [kcal/mol/Angstrom^2]\n", empos->k);
  /* yss-add */
  lprintf("  X-RESTRAINT FLAG: %s\n", (empos->x_flag) ? "on" : "off"); 
  lprintf("  Y-RESTRAINT FLAG: %s\n", (empos->y_flag) ? "on" : "off"); 
  lprintf("  Z-RESTRAINT FLAG: %s\n", (empos->z_flag) ? "on" : "off"); 
  /* yse */
  lprintf("  VIRIAL FLAG: %s\n", (empos->virial_flag) ? "on" : "off");
  if (empos->gradual_change_step) {
    lprintf("  GRADUAL CHANGE OF FORCE CONSTANT: %f -> %f in %d steps\n",
	    empos->k0, empos->k1, empos->gradual_change_step);
  }

  empos->xm0 = emalloc(fname,sizeof(VEC) * empos->n_mole); 
  for (i=0;i<empos->n_mole;i++) {
    empos->xm0[i].x = 0.0;
    empos->xm0[i].y = 0.0;
    empos->xm0[i].z = 0.0;
    w1 = 0.0;
    for (j=0;j<empos->mole[i].n_list;j++){
      ii = empos->mole[i].list[j];
      w = ad->w[ii];
      empos->xm0[i].x += x0[ii].x * w;
      empos->xm0[i].y += x0[ii].y * w;
      empos->xm0[i].z += x0[ii].z * w;
      w1 += w;
    }
    empos->xm0[i].x /= w1;
    empos->xm0[i].y /= w1;
    empos->xm0[i].z /= w1;

    lprintf("  Gravity Center for Molecule %d (%d atoms) : (%.3f,%.3f,%.3f)\n", empos->mole[i].no+1, empos->mole[i].n_list, empos->xm0[i].x, empos->xm0[i].y, empos->xm0[i].z);
  }
  lprintf("\n");

#ifdef MPI_SDMD
  for (i=0;i<empos->n_mole;i++) {
    SDMD_setup_ex_tr_atom(lc, empos->mole[i].n_list, empos->mole[i].list);
  }
#endif  
#if 0
  free(x0);
#endif
  free(mole_list);
}

void EP_MOLECULE_POSITION_energy_force(EP_MOLECULE_POSITION *empos,
				       ATOM_DATA *ad,
				       BOUNDARY *bc,
				       LINKED_CELL *lc)
{
  double dx, dy, dz;
  double fx, fy, fz;
  double w1, w;
  double x1, y1, z1;
  double energy;
  int i, j, ii, ii1;

#ifdef MPI_SDMD
  int pe;
#endif  
  
  empos->energy = 0.0;

  for (i=0;i<empos->n_mole;i++) {
#ifdef MPI_SDMD
    pe = ATOM_CPU(lc, ad, empos->mole[i].list[0]);
    if (pe != mpi.rank) continue;
#endif  /* MPI_SDMD */

    x1 = y1 = z1 = w1 = 0.0;
    for (j=0;j<empos->mole[i].n_list;j++) {
      ii = empos->mole[i].list[j];
      
      w   = ad->w[ii];
      x1 += ad->x[ii].x * w;
      y1 += ad->x[ii].y * w;
      z1 += ad->x[ii].z * w;
      w1 += w;
    }
    x1 /= w1;
    y1 /= w1; 
    z1 /= w1;

    if (empos->x_flag) {
      dx = x1 - empos->xm0[i].x;
    }else{
      dx = 0.0;
    }
    if (empos->y_flag) {
      dy = y1 - empos->xm0[i].y;
    }else{
      dy = 0.0;
    }
    if (empos->z_flag) {
      dz = z1 - empos->xm0[i].z;
    }else{
      dz = 0.0;
    }

    energy = (SQR(dx) + SQR(dy) + SQR(dz))*empos->k;
    empos->energy += energy;
    fx = -2.0 * empos->k * dx;
    fy = -2.0 * empos->k * dy;
    fz = -2.0 * empos->k * dz;

  /* yss-debug 
   * printf("  Energy for Group: (%d,%d,%.3f)\n", mpi.rank, i , energy);
   * printf("  Energy for Total: (%d,%d,%.3f)\n", mpi.rank, i , empos->energy);
   * printf("  Force for Group: (%.3f,%.3f,%.3f)\n", fx, fy, fz);
   * printf("  Deviation for Group: (%.3f,%.3f,%.3f)\n", dx, dy, dz);
   * yse */

    for (j=0;j<empos->mole[i].n_list;j++) {
      ii = empos->mole[i].list[j];
      
      w   = ad->w[ii];
      ad->f[ii].x += fx * w / w1;
      ad->f[ii].y += fy * w / w1;
      ad->f[ii].z += fz * w / w1;

      if (empos->virial_flag) {
	ad->virial[0] += fx * ad->x[ii].x * w / w1;
	ad->virial[1] += fy * ad->x[ii].y * w / w1;
	ad->virial[2] += fz * ad->x[ii].z * w / w1;
	ad->virial[3] += (fx * ad->x[ii].y + fy * ad->x[ii].x) * 0.5 * w / w1;
	ad->virial[4] += (fx * ad->x[ii].z + fz * ad->x[ii].x) * 0.5 * w / w1;
	ad->virial[5] += (fy * ad->x[ii].z + fz * ad->x[ii].y) * 0.5 * w / w1;
      }
    }
  }
}

#else

void EP_MOLECULE_POSITION_setup(EXTRA_POT *ep,
				ATOM_DATA *ad,
				BOUNDARY *bc,
				LINKED_CELL *lc,
				int group_no, char *file_name, double k,
				int gradual_change_step, double k1, int virial_flag,
				int mole_residue,
				int x_flag, int y_flag, int z_flag)
{
  EP_MOLECULE_POSITION *empos;
  FILE *fp;
  char buf[100];
  int natom, i, j, ii;
  char *fname = "EP_MOLECULE_POSITION_setup";
  double x1, y1, z1;
  double w, w1;
  int ret;
  VEC *x0;

  empos = emalloc(fname, sizeof(EP_MOLECULE_POSITION));
  EP_add_item(ep, (EP_ITEM*) empos);
  
  empos->type = EP_T_MOLECULE_POSITION;
  strcpy(empos->fname, file_name);

  empos->group_no = group_no;
  empos->k = empos->k0 = k;
  empos->k1 = k1;
  empos->virial_flag = virial_flag;
  empos->gradual_change_step = gradual_change_step;
  empos->x_flag = x_flag;
  empos->y_flag = y_flag;
  empos->z_flag = z_flag;
  empos->mole_residue = mole_residue; 

  ad->ex_force_flag = 1;
  
  empos->n_list = ATOM_DATA_count_group_atoms(ad, group_no);
  empos->list = emalloc(fname,sizeof(int)*(empos->n_list));
  ATOM_DATA_make_list_of_group_atoms(ad,group_no,empos->list);
   
  lprintf("MOLECULE POSITION HARMONIC RESTRAINT:\n");
  lprintf("  REFERENCE COORDINATE FILE: %s\n", empos->fname); 
  lprintf("  GROUP NO: %d: %d atoms\n", empos->group_no, empos->n_list);

  empos->n_residue = ATOM_DATA_count_group_residues(ad,empos->n_list,empos->list);
  empos->n_mole = empos->n_residue/empos->mole_residue;
  lprintf("              : %d residues (%d)\n", empos->n_residue, empos->mole_residue);
  empos->natom_mole = empos->n_list/empos->n_mole;
  lprintf("              : %d molecules (%d atoms)\n", empos->n_mole, empos->natom_mole);

#if 0
  if ((fp = fopen(file_name, "r")) == NULL) {
    lprintf("ERROR: %s: No such file\n", file_name);
        marble_exit(1);
  } 
  x0 = emalloc(fname,sizeof(VEC) * ad->natom);
  fgets(buf, 80, fp);  /* skip header */
  fscanf(fp,"%d", &(natom));
  if (natom != ad->natom) {
    lprintf("ERROR: Inconsistent atom number in crd file %s\n",fname);
    marble_exit(1);
  }                              
  for (i = 0; i < natom; i++) {  
    if (fscanf(fp,"%lf%lf%lf", &(x0[i].x),&(x0[i].y),&(x0[i].z)) != 3) {
      lprintf("ERROR: Invalid atom coordinate in crd file %s\n",fname);
            marble_exit(1);
    }                  
  }                    
  fclose(fp);          
#else
  x0 = ATOM_DATA_get_ref_crd(ad, file_name);
#endif

  lprintf("  FORCE CONSTANT:    %lf [kcal/mol/Angstrom^2]\n", empos->k);
  /* yss-add */
  lprintf("  X-RESTRAINT FLAG: %s\n", (empos->x_flag) ? "on" : "off"); 
  lprintf("  Y-RESTRAINT FLAG: %s\n", (empos->y_flag) ? "on" : "off"); 
  lprintf("  Z-RESTRAINT FLAG: %s\n", (empos->z_flag) ? "on" : "off"); 
  /* yse */
  lprintf("  VIRIAL FLAG: %s\n", (empos->virial_flag) ? "on" : "off");
  if (empos->gradual_change_step) {
    lprintf("  GRADUAL CHANGE OF FORCE CONSTANT: %f -> %f in %d steps\n",
	    empos->k0, empos->k1, empos->gradual_change_step);
  }

  empos->xm0 = emalloc(fname,sizeof(VEC) * ad->natom);
  ii = 0;
  for (i=0;i<empos->n_mole;i++) {
    empos->xm0[i].x = 0.0;
    empos->xm0[i].y = 0.0;
    empos->xm0[i].z = 0.0;
    w1 = 0.0;
    for (j=0;j<empos->natom_mole;j++){
      w = ad->w[empos->list[ii]];
      empos->xm0[i].x += x0[empos->list[ii]].x * w;
      empos->xm0[i].y += x0[empos->list[ii]].y * w;
      empos->xm0[i].z += x0[empos->list[ii]].z * w;
      w1 += w;
      ii++;
    }
    empos->xm0[i].x /= w1;
    empos->xm0[i].y /= w1;
    empos->xm0[i].z /= w1;

    lprintf("  Gravity Center for Molecule %d : (%.3f,%.3f,%.3f)\n", i+1, empos->xm0[i].x, empos->xm0[i].y, empos->xm0[i].z);
  }
  lprintf("\n");

#ifdef MPI_SDMD
  SDMD_setup_ex_tr_atom(lc, empos->n_list, empos->list);
#endif  
#if 0
  free(x0);
#endif
}

void EP_MOLECULE_POSITION_energy_force(EP_MOLECULE_POSITION *empos,
				       ATOM_DATA *ad,
				       BOUNDARY *bc,
				       LINKED_CELL *lc)
{
  double dx, dy, dz;
  double fx, fy, fz;
  double w1, w;
  double x1, y1, z1;
  double energy;
  int i, j, ii, ii1;

#ifdef MPI_SDMD
  int pe;
  
  pe = ATOM_CPU(lc, ad, empos->list[0]);
  if (pe != mpi.rank) {
    empos->energy = 0.0;
    return;
  }
#endif  /* MPI_SDMD */
  
  ii=0;
  empos->energy = 0.0;

  for (i=0;i<empos->n_mole;i++) {
    x1 = y1 = z1 = w1 = 0.0;
    ii1 = ii;
    for (j=0;j<empos->natom_mole;j++) {
      w   = ad->w[empos->list[ii1]];
      x1 += ad->x[empos->list[ii1]].x * w;
      y1 += ad->x[empos->list[ii1]].y * w;
      z1 += ad->x[empos->list[ii1]].z * w;
      w1 += w;
      ii1++;
    }
    x1 /= w1;
    y1 /= w1; 
    z1 /= w1;

    if (empos->x_flag) {
      dx = x1 - empos->xm0[i].x;
    }else{
      dx = 0.0;
    }
    if (empos->y_flag) {
      dy = y1 - empos->xm0[i].y;
    }else{
      dy = 0.0;
    }
    if (empos->z_flag) {
      dz = z1 - empos->xm0[i].z;
    }else{
      dz = 0.0;
    }

    energy = (SQR(dx) + SQR(dy) + SQR(dz))*empos->k;
    empos->energy += energy;
    fx = -2.0 * empos->k * dx;
    fy = -2.0 * empos->k * dy;
    fz = -2.0 * empos->k * dz;

  /* yss-debug 
   * printf("  Energy for Group: (%d,%d,%.3f)\n", mpi.rank, i , energy);
   * printf("  Energy for Total: (%d,%d,%.3f)\n", mpi.rank, i , empos->energy);
   * printf("  Force for Group: (%.3f,%.3f,%.3f)\n", fx, fy, fz);
   * printf("  Deviation for Group: (%.3f,%.3f,%.3f)\n", dx, dy, dz);
   * yse */

    for (j=0;j<empos->natom_mole;j++) {
      w = ad->w[empos->list[ii]];
      ad->f[empos->list[ii]].x += fx * w / w1;
      ad->f[empos->list[ii]].y += fy * w / w1;
      ad->f[empos->list[ii]].z += fz * w / w1;

      if (empos->virial_flag) {
	ad->virial[0] += fx * ad->x[empos->list[ii]].x * w / w1;
	ad->virial[1] += fy * ad->x[empos->list[ii]].y * w / w1;
	ad->virial[2] += fz * ad->x[empos->list[ii]].z * w / w1;
	ad->virial[3] += (fx * ad->x[empos->list[ii]].y + fy * ad->x[empos->list[ii]].x) * 0.5 * w / w1;
	ad->virial[4] += (fx * ad->x[empos->list[ii]].z + fz * ad->x[empos->list[ii]].x) * 0.5 * w / w1;
	ad->virial[5] += (fy * ad->x[empos->list[ii]].z + fz * ad->x[empos->list[ii]].y) * 0.5 * w / w1;
      }
      ii++;
    }
  }
}

#endif

void EP_MOLECULE_POSITION_gradual_change(EP_MOLECULE_POSITION *empos, int step)
{
  double lambda;
  if (empos->gradual_change_step) {
    if (step < empos->gradual_change_step) {
      lambda = (double)step/empos->gradual_change_step;
      empos->k = empos->k0*(1.0-lambda) + empos->k1*lambda;
    } else {
      empos->k = empos->k1;
    }
  }
}

/* yse */

/* yss-add (Mar09/2004) */
/* multiple torsions Nov01/2006 ike */

/*********************************/
/*                               */
/*        EP_TORSION             */
/*                               */
/*********************************/

void EP_TORSION_setup(EXTRA_POT *ep,
		      ATOM_DATA *ad,
		      BOUNDARY *bc,
		      LINKED_CELL *lc,
		      char *atom1, char *atom2, char *atom3, char *atom4,
		      double k, double r0, double delta,
		      int gradual_change_step, double k1, int virial_flag)
{
  EP_TORSION *et;
  double phi;
  int ret1,ret2,ret3,ret4, atom_list[4];
  double x12,y12,z12,x23,y23,z23,x34,y34,z34,nx23,ny23,nz23,len_23;
  double dx,dy,dz,gx,gy,gz,dd,gg,dg,len_dg;
  double ddx,ddy,ddz,dgx,dgy,dgz;
  VEC *x;
  double cosp, sign;
  int i, n_atom[4];

  et = emalloc("EP_TORSION_setup", sizeof(EP_TORSION));
  EP_add_item(ep, (EP_ITEM*) et);
  
  et->type = EP_T_TORSION;

  lprintf("TORSION HARMONIC RESTRAINT:\n");
  
  et->n_torsion = n_atom[0] = ATOM_DATA_str_to_atom_list(ad, atom1, &(et->atom1));
  
  n_atom[1] = ATOM_DATA_str_to_atom_list(ad, atom2, &(et->atom2));
  if (et->n_torsion != n_atom[1]) {
    et->n_torsion = -1;
  }
  n_atom[2] = ATOM_DATA_str_to_atom_list(ad, atom3, &(et->atom3));
  if (et->n_torsion != n_atom[2]) {
    et->n_torsion = -1;
  }
  n_atom[3] = ATOM_DATA_str_to_atom_list(ad, atom4, &(et->atom4));
  if (et->n_torsion != n_atom[3]) {
    et->n_torsion = -1;
  }

  if (et->n_torsion == -1) {
    lprintf("  ERROR: Number of atoms is different: %s(%d),%s(%d),%s(%d),%s(%d)\n", 
	    atom1, n_atom[0], atom2, n_atom[1], atom3, n_atom[2], atom4, n_atom[3]);
    marble_exit(1);
  }  


  /* k      : (kcal/mol)
   * r0     : (degree)
   * et->r0 : (radian) & (minus) 
   * et->r1 : r0 - delta
   * et->r2 : r0 + delta
   */
  et->k = k;
  et->r0    = r0*M_PI/180.0;
  et->delta = delta*M_PI/180.0;
  et->r1 = et->r0 - et->delta;
  et->r2 = et->r0 + et->delta; 
  et->gradual_change_step = gradual_change_step;
  et->k0 = k;
  et->k1 = k1;
  et->virial_flag = virial_flag;
  et->torsion = emalloc("EP_TORSION_setup", sizeof(double)*et->n_torsion);

  lprintf("  FORCE CONSTANT:    %lf [kcal/mol/Radian^2]\n", et->k);
  lprintf("  STANDARD TORSION: %lf [Degree]\n", et->r0*(180.0/M_PI));
  lprintf("  ALLOWED DEVIATION: %lf [Degree]\n", et->delta*(180.0/M_PI));
  lprintf("  VIRIAL FLAG: %s\n", (et->virial_flag) ? "on" : "off");
  if (et->gradual_change_step) {
    lprintf("  GRADUAL CHANGE OF FORCE CONSTANT: %f -> %f in %d steps\n",
	    et->k0, et->k1, et->gradual_change_step);
  }
  lprintf("  NUMBER OF TORSIONS:    %d\n", et->n_torsion);

  
  /* dihedral angle */
  x = ad->x;

  for (i=0; i < et->n_torsion; i++) {
    x12 = x[et->atom2[i]].x - x[et->atom1[i]].x;
    y12 = x[et->atom2[i]].y - x[et->atom1[i]].y;
    z12 = x[et->atom2[i]].z - x[et->atom1[i]].z;
    x23 = x[et->atom3[i]].x - x[et->atom2[i]].x;
    y23 = x[et->atom3[i]].y - x[et->atom2[i]].y;
    z23 = x[et->atom3[i]].z - x[et->atom2[i]].z;
    x34 = x[et->atom4[i]].x - x[et->atom3[i]].x;
    y34 = x[et->atom4[i]].y - x[et->atom3[i]].y;
    z34 = x[et->atom4[i]].z - x[et->atom3[i]].z;
    dx  = y12*z23 - z12*y23;
    dy  = z12*x23 - x12*z23;
    dz  = x12*y23 - y12*x23;
    gx  = y23*z34 - z23*y34;
    gy  = z23*x34 - x23*z34;
    gz  = x23*y34 - y23*x34;

    len_23 = sqrt(x23*x23 + y23*y23 + z23*z23);
    nx23 = x23 / len_23;
    ny23 = y23 / len_23;
    nz23 = z23 / len_23;
    dd  = dx*dx + dy*dy + dz*dz;
    gg  = gx*gx + gy*gy + gz*gz;
    dg  = dx*gx + dy*gy + dz*gz;
    len_dg = sqrt(dd) * sqrt(gg);

    /* The sign of the next line is wrong. 
       phi = atan2(-(nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)
       +nz23*(dx*gy-dy*gx))/len_dg,  dg/len_dg);
    */
    phi = atan2((nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)
		 +nz23*(dx*gy-dy*gx))/len_dg,  dg/len_dg);

    /*
      cosp = dg/len_dg;
      phi  = acos(cosp);
      sign = nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)+nz23*(dx*gy-dy*gx);
      if (sign < 0.0) phi = 2.0*M_PI - phi;
    */

    if (phi < -M_PI) phi += 2.0*M_PI;
    else if (phi > M_PI) phi -= 2.0*M_PI;

    et->torsion[i] = phi;

    lprintf("  "); ret1 = ATOM_DATA_print_atom(ad, et->atom1[i]);
    lprintf("  "); ret2 = ATOM_DATA_print_atom(ad, et->atom2[i]);
    lprintf("  "); ret3 = ATOM_DATA_print_atom(ad, et->atom3[i]);
    lprintf("  "); ret4 = ATOM_DATA_print_atom(ad, et->atom4[i]);
    if (ret1 != 0 || ret2 != 0 || ret3 !=0 || ret4 !=0) {
      lprintf("ERROR: Invalid atom name %s, %s, %s or %s\n",
	      atom1, atom2, atom3, atom4);
      marble_exit(1);
    }
    lprintf("  CURRENT TORSION: %lf [Degree]\n", phi*180.0/M_PI); 

#ifdef MPI_SDMD
    atom_list[0] = et->atom1[i];
    atom_list[1] = et->atom2[i];
    atom_list[2] = et->atom3[i];
    atom_list[3] = et->atom4[i];
    SDMD_setup_ex_tr_atom(lc, 4, atom_list);
#endif  

  }

  lprintf("\n");
  
}

void EP_TORSION_energy_force(EP_TORSION *et, ATOM_DATA *ad,
			     BOUNDARY *bc, LINKED_CELL *lc)
{
  int atom1, atom2, atom3, atom4;
  double phi, e;
  double x12,y12,z12,x23,y23,z23,x34,y34,z34,nx23,ny23,nz23,len_23;
  double dx,dy,dz,gx,gy,gz,dd,gg,dg,len_dg;
  double ddx,ddy,ddz,dgx,dgy,dgz;
  double F, Fx1,Fy1,Fz1,Fx4,Fy4,Fz4,Fx2_d,Fy2_d,Fz2_d,Fx2_g,Fy2_g,Fz2_g;
  double cosp, sign;
  VEC *x;
  int i;

#ifdef MPI_SDMD
  int pe;
#endif

  et->energy = 0.0;
  x = ad->x;

  for (i=0; i < et->n_torsion; i++) {
    
#ifdef MPI_SDMD
    pe = ATOM_CPU(lc, ad, et->atom1[i]);
    if (pe != mpi.rank) {
      continue;
    }
#endif  /* MPI_SDMD */

    atom1 = et->atom1[i];
    atom2 = et->atom2[i];
    atom3 = et->atom3[i];
    atom4 = et->atom4[i];


    /* dihedral angle */
    x12 = x[atom2].x - x[atom1].x;
    y12 = x[atom2].y - x[atom1].y;
    z12 = x[atom2].z - x[atom1].z;
    x23 = x[atom3].x - x[atom2].x;
    y23 = x[atom3].y - x[atom2].y;
    z23 = x[atom3].z - x[atom2].z;
    x34 = x[atom4].x - x[atom3].x;
    y34 = x[atom4].y - x[atom3].y;
    z34 = x[atom4].z - x[atom3].z;
    dx  = y12*z23 - z12*y23;
    dy  = z12*x23 - x12*z23;
    dz  = x12*y23 - y12*x23;
    gx  = y23*z34 - z23*y34;
    gy  = z23*x34 - x23*z34;
    gz  = x23*y34 - y23*x34;

    len_23 = sqrt(x23*x23 + y23*y23 + z23*z23);
    nx23 = x23 / len_23;
    ny23 = y23 / len_23;
    nz23 = z23 / len_23;
    dd  = dx*dx + dy*dy + dz*dz;
    gg  = gx*gx + gy*gy + gz*gz;
    dg  = dx*gx + dy*gy + dz*gz;
    len_dg = sqrt(dd) * sqrt(gg);

    /*
      phi = atan2(-(nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)
      +nz23*(dx*gy-dy*gx))/len_dg,  dg/len_dg);
    */
    phi = atan2((nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)
		 +nz23*(dx*gy-dy*gx))/len_dg,  dg/len_dg);

    /* yss (Mar09/2004)
       cosp = dg/len_dg;
       phi  = acos(cosp);
       sign = nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)+nz23*(dx*gy-dy*gx);
       if (sign < 0.0) phi = 2.0*M_PI - phi;
    */
    
    if (phi - et->r0 < -M_PI) phi += 2.0*M_PI;
    else if (phi - et->r0 > M_PI) phi -= 2.0*M_PI;

    /* energy */
    if (phi < et->r1) {
      et->energy += et->k * (phi - et->r1) * (phi - et->r1);
      F = -2.0 * et->k * (phi - et->r1);
    }else if (phi > et->r2) {
      et->energy += et->k * (phi - et->r2) * (phi - et->r2);
      F = -2.0 * et->k * (phi - et->r2);
    }else{
      /*et->energy = 0.0; */
      F = 0.0;
    }

    et->torsion[i] = phi;

    /* force */  
    ddx = (dy*nz23 - dz*ny23)/dd;
    ddy = (dz*nx23 - dx*nz23)/dd;
    ddz = (dx*ny23 - dy*nx23)/dd;
    
    dgx = -(gy*nz23 - gz*ny23)/gg;
    dgy = -(gz*nx23 - gx*nz23)/gg;
    dgz = -(gx*ny23 - gy*nx23)/gg;

    Fx1 = F*(ddy * z23 - ddz * y23);
    Fy1 = F*(ddz * x23 - ddx * z23);
    Fz1 = F*(ddx * y23 - ddy * x23);
    Fx4 = F*(dgy * z23 - dgz * y23);
    Fy4 = F*(dgz * x23 - dgx * z23);
    Fz4 = F*(dgx * y23 - dgy * x23);
    Fx2_d = F*(ddy * z12 - ddz * y12);
    Fy2_d = F*(ddz * x12 - ddx * z12);
    Fz2_d = F*(ddx * y12 - ddy * x12);
    Fx2_g = F*(dgy * z34 - dgz * y34);
    Fy2_g = F*(dgz * x34 - dgx * z34);
    Fz2_g = F*(dgx * y34 - dgy * x34);
  
    ad->f[atom1].x += Fx1;
    ad->f[atom1].y += Fy1;
    ad->f[atom1].z += Fz1;
    ad->f[atom2].x += -Fx2_d + Fx2_g - Fx1;
    ad->f[atom2].y += -Fy2_d + Fy2_g - Fy1;
    ad->f[atom2].z += -Fz2_d + Fz2_g - Fz1;
    ad->f[atom3].x +=  Fx2_d - Fx2_g - Fx4;
    ad->f[atom3].y +=  Fy2_d - Fy2_g - Fy4;
    ad->f[atom3].z +=  Fz2_d - Fz2_g - Fz4;
    ad->f[atom4].x += Fx4;
    ad->f[atom4].y += Fy4;
    ad->f[atom4].z += Fz4;

    if (et->virial_flag) {
      ad->virial[0] += ( Fx1                   * ad->x[atom1].x + 
			 (-Fx2_d + Fx2_g - Fx1) * ad->x[atom2].x +
			 ( Fx2_d - Fx2_g - Fx4) * ad->x[atom3].x +
			 Fx4                   * ad->x[atom4].x );
      ad->virial[1] += ( Fy1                   * ad->x[atom1].y + 
			 (-Fy2_d + Fy2_g - Fy1) * ad->x[atom2].y +
			 ( Fy2_d - Fy2_g - Fy4) * ad->x[atom3].y +
			 Fy4                   * ad->x[atom4].y );
      ad->virial[2] += ( Fz1                   * ad->x[atom1].z + 
			 (-Fz2_d + Fz2_g - Fz1) * ad->x[atom2].z +
			 ( Fz2_d - Fz2_g - Fz4) * ad->x[atom3].z +
			 Fz4                   * ad->x[atom4].z );
      ad->virial[3] += ( Fx1                   * ad->x[atom1].y +
			 (-Fx2_d + Fx2_g - Fx1) * ad->x[atom2].y +
			 ( Fx2_d - Fx2_g - Fx4) * ad->x[atom3].y +
			 Fx4                   * ad->x[atom4].y );
      ad->virial[4] += ( Fx1                   * ad->x[atom1].z +
			 (-Fx2_d + Fx2_g - Fx1) * ad->x[atom2].z +
			 ( Fx2_d - Fx2_g - Fx4) * ad->x[atom3].z +
			 Fx4                   * ad->x[atom4].z );
      ad->virial[5] += ( Fy1                   * ad->x[atom1].z +
			 (-Fy2_d + Fy2_g - Fy1) * ad->x[atom2].z +
			 ( Fy2_d - Fy2_g - Fy4) * ad->x[atom3].z +
			 Fy4                   * ad->x[atom4].z );
    }
  }

}

void EP_TORSION_prop_header(EP_TORSION *et,
	 	            FILE *fp, int *id)
{
  if (et->n_torsion == 1) {
    fprintf(fp, "%2dTORSION     ", (*id)++);
  } else {
    fprintf(fp, "%2dTORSION_AVE ", (*id)++);
    fprintf(fp, "%2dTORSION_MIN ", (*id)++);
    fprintf(fp, "%2dTORSION_MAX ", (*id)++);
  }

  fprintf(fp, "%2dREST_ENE    ", (*id)++);
}

void EP_TORSION_prop_out(EP_TORSION *et, MD_SYSTEM *sys, FILE *fp)
{
  int i;
  double ave, min, max;
#ifdef MPI_SDMD
  int pe;
  LINKED_CELL *lc;
  ATOM_DATA *ad;
  MPI_Status stat;
  double buf[2], sum[2];

  lc = &sys->linked_cell;
  ad = &sys->atom;

  buf[0] = et->energy;

  MPI_Reduce(buf,sum, 1, MPI_DOUBLE, MPI_SUM, mpi.master_pe, mpi.comm);
  et->energy = sum[0];
  
  if (mpi.master) {
    for (i=0;i<et->n_torsion;i++) {
      pe = ATOM_CPU(lc, ad, et->atom1[i]);
      if (pe != mpi.rank) {
	MPI_Recv(buf,1,MPI_DOUBLE, pe, SDMD_TAG_PROP, mpi.comm, &stat);
	et->torsion[i] = buf[0];
      }
    }
  } else {
    for (i=0;i<et->n_torsion;i++) {
      pe = ATOM_CPU(lc, ad, et->atom1[i]);
      if (pe == mpi.rank) {
	buf[0] = et->torsion[i];
	MPI_Send(buf,1,MPI_DOUBLE, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
      }
    }
    return;
  }

#endif /* MPI_SDMD */
  
  if (et->n_torsion == 1) {
    fprintf(fp, " %13.6e %13.6e", et->torsion[0]*(180.0/M_PI), et->energy);
  } else {
    ave = 0.0;
    max = -1.0e10;
    min =  1.0e10;
    for (i=0;i<et->n_torsion;i++) {
      ave += et->torsion[i];
      if (max < et->torsion[i])
	max = et->torsion[i];
      if (min > et->torsion[i])
	min = et->torsion[i];
    }
      
    ave /= et->n_torsion;
    fprintf(fp, " %13.6e %13.6e %13.6e %13.6e", 
	    ave*(180.0/M_PI), 
	    min*(180.0/M_PI), 
	    max*(180.0/M_PI), 
	    et->energy);
  }
}

void EP_TORSION_gradual_change(EP_TORSION *et, int step)
{
  double lambda;
  if (et->gradual_change_step) {
    if (step < et->gradual_change_step) {
      lambda = (double)step/et->gradual_change_step;
      et->k = et->k0*(1.0-lambda) + et->k1*lambda;
    } else {
      et->k = et->k1;
    }
  }
}

/* yse */


/* Steered MD   2007/12/05                      */
/*   EP_SMD_CF:  SMD constant force             */
/*   EP_SMD_CV:  SMD constant velocity          */

void EP_SMD_CV_setup(EXTRA_POT *ep,
		     ATOM_DATA *ad, LINKED_CELL *lc, 
		     STR_LIST *group_str, int group_no,
		     char *crd_file,  double k,
		     double vel, double ang_vel,
		     double init_trans, double init_rot,
		     double dir[3],  double pos[3], 
		     int center_flag, int weight_flag,
		     int gradual_change_step, double k1, 
		     int virial_flag,
		     int x_flag, int y_flag, int z_flag)
{
  EP_SMD *smd;
  FILE *fp;
  char buf[100];
  int i, j, n_atom;
  double len, x, y, z;
  VEC *x0;
  char *fname = "EP_SMD_CV_setup";
  
  smd = emalloc(fname, sizeof(EP_SMD));
  EP_add_item(ep, (EP_ITEM*) smd);

  smd->type = EP_T_SMD_CV;
  smd->group_no = group_no;
  strcpy(smd->crd_file, crd_file);
  smd->k = smd->k0 = k;
  smd->k1 = k1;
  smd->vel = vel;
  smd->ang_vel = ang_vel;
  smd->init_trans = init_trans;
  smd->init_rot = init_rot;
  smd->center_flag = center_flag;
  smd->weight_flag = weight_flag;
  smd->virial_flag = virial_flag;
  smd->x_flag = x_flag;
  smd->y_flag = y_flag;
  smd->z_flag = z_flag;
  smd->gradual_change_step = gradual_change_step;

  len = 0.0;
  for (i=0;i<3;i++)
    len += dir[i]*dir[i];
  len = sqrt(len);
  for (i=0;i<3;i++) {
    smd->dir[i] = dir[i]/len;
    smd->pos[i] = pos[i];
  }

  ad->ex_force_flag = 1;
  
  lprintf("Steered MD: Constant Velocity\n");

  if (group_str!=NULL) {
    smd->group_no = group_no = 31;
    ATOM_DATA_set_group_str_list(ad, group_str, group_no, "  GROUP:");
  } else {
    lprintf("  Group No. %d\n", smd->group_no);
  }

  smd->n_atom = ATOM_DATA_count_group_atoms(ad, group_no);
  if (smd->n_atom == 0) {
    lprintf("  ERROR: no atom is specified!\n");
    marble_exit(1);
  }
  
  smd->atom_list = emalloc(fname,sizeof(int)*(smd->n_atom));
  ATOM_DATA_make_list_of_group_atoms(ad,group_no,smd->atom_list);

  if (smd->center_flag) {
    smd->x0 = emalloc(fname,sizeof(VEC));
    smd->n_x0 = 1;
    smd->x0[0].x = smd->x0[0].y = smd->x0[0].z = 0.0;
    smd->ene_list = NULL;
    smd->w = emalloc(fname,sizeof(double)*smd->n_atom);
    smd->total_w = 0.0;
  } else {
    smd->x0 = emalloc(fname,sizeof(VEC)*smd->n_atom);
    smd->n_x0 = smd->n_atom;
    smd->ene_list = emalloc(fname,sizeof(double)*smd->n_atom);
    smd->w = NULL;
  }

  lprintf("  Reference Cooridnate File: %s\n", smd->crd_file);
  lprintf("  Force Constant: %.3f [kcal/mol/Angstrom^2]\n", smd->k);
  lprintf("  Velocity: %.3f [Angstrom/ns]\n", smd->vel);
  lprintf("  Angular Velocity: %.3f [Degree/ns]\n", smd->ang_vel);
  lprintf("  Initial Translation: %.3f [Angstrom]\n", smd->init_trans);
  lprintf("  Initial Rotation: %.3f [Degree]\n", smd->init_rot);
  lprintf("  Direction:     (%.3f,%.3f,%.3f) \n", smd->dir[0], smd->dir[1], smd->dir[2]);
  lprintf("  Axis Position: (%.3f,%.3f,%.3f) \n", smd->pos[0], smd->pos[1], smd->pos[2]);
  lprintf("  Pulling Mode: %s\n", (smd->center_flag) ? "center_of_mass" : "atom" );
  if (smd->center_flag) {
    lprintf("  Weight Flag for Center : %s\n", (smd->weight_flag) ? "mass_weighted" : "uniform" );
  }
  lprintf("  X-Restraint Flag: %s\n", (smd->x_flag) ? "on" : "off"); 
  lprintf("  Y-Restraint Flag: %s\n", (smd->y_flag) ? "on" : "off"); 
  lprintf("  Z-Restraint Flag: %s\n", (smd->z_flag) ? "on" : "off"); 
  lprintf("  Virial Flag: %s\n", (smd->virial_flag) ? "on" : "off");
  if (smd->gradual_change_step) {
    lprintf("  Gradual Change of Force Constant: %f -> %f in %d steps\n", smd->k, smd->k1, smd->gradual_change_step);
  }

#if 0 
  if ((fp = fopen(crd_file, "r")) == NULL) {
    lprintf("ERROR: %s: No such file\n", crd_file);
    marble_exit(1);
  }

  fgets(buf, 80, fp);  /* skip header */
  fscanf(fp,"%d", &(n_atom));
  if (n_atom != ad->natom) {
    lprintf("ERROR: Inconsistent atom number in crd file %s\n",crd_file);
    marble_exit(1);
  }

  for (i = 0, j = 0; i < n_atom; i++) {
    if (fscanf(fp,"%lf%lf%lf", &x,&y,&z) != 3) {
      lprintf("ERROR: Invalid atom coordinate in crd file %s\n",crd_file);
      marble_exit(1);
    }
    if (j < smd->n_atom && i == smd->atom_list[j]) {
      if (smd->center_flag) {
	if (smd->weight_flag) {
	  smd->w[j] = ad->w[i];
	} else {
	  smd->w[j] = 1.0;
	}
	smd->x0[0].x += smd->w[j] * x;
	smd->x0[0].y += smd->w[j] * y;
	smd->x0[0].z += smd->w[j] * z;
	smd->total_w += smd->w[j];
      } else {
	smd->x0[j].x = x;
	smd->x0[j].y = y;
	smd->x0[j].z = z;
      }
      j++;
    }
  }
  fclose(fp);
#else
  x0 = ATOM_DATA_get_ref_crd(ad, crd_file);
  n_atom = ad->natom;

  for (i = 0, j = 0; i < n_atom; i++) {
    if (j < smd->n_atom && i == smd->atom_list[j]) {
      if (smd->center_flag) {
	if (smd->weight_flag) {
	  smd->w[j] = ad->w[i];
	} else {
	  smd->w[j] = 1.0;
	}
	smd->x0[0].x += smd->w[j] * x0[i].x;
	smd->x0[0].y += smd->w[j] * x0[i].y;
	smd->x0[0].z += smd->w[j] * x0[i].z;
	smd->total_w += smd->w[j];
      } else {
	smd->x0[j].x = x0[i].x;
	smd->x0[j].y = x0[i].y;
	smd->x0[j].z = x0[i].z;
      }
      j++;
    }
  }
#endif

  if (smd->center_flag) {
    smd->x0[0].x /= smd->total_w;
    smd->x0[0].y /= smd->total_w;
    smd->x0[0].z /= smd->total_w;
#ifdef MPI_SDMD
    SDMD_setup_ex_tr_atom(lc, smd->n_atom, smd->atom_list);
#endif  
  }

  /* initial translation and rotation */
  EP_SMD_CV_trans_rot(smd, smd->init_trans, smd->init_rot * M_PI / 180.0);

  lprintf("\n");
}

void EP_SMD_CV_test(EXTRA_POT *ep, ATOM_DATA *ad, char *base_file,
		    double timestep, int step, int interval)
{
  EP_ITEM *p;
  EP_SMD *smd;
  char pdb_file[MAX_FILE_NAME_LEN], trj_file[MAX_FILE_NAME_LEN];
  char aname[10], atom_no[10], res_no[10];
  FILE *fp;
  double trans, rot, dt_ns;
  int i, j, k;

#ifdef MPI_SDMD
  if (!mpi.master) return;
#endif

  for (p = ep->head; p->next!=NULL; p=p->next);
  smd = (EP_SMD*) p;

  if (smd->center_flag) {
    lprintf("Steered MD: Test Routine is not supported for pulling center of mass\n");
    return;
  }

  sprintf(pdb_file, "%s.pdb", base_file);
  sprintf(trj_file, "%s.trj", base_file);

  lprintf("Steered MD: Test Routine\n");
  lprintf("  Output Reference PDB and TRJ files.\n");
  lprintf("    PDB File: %s\n", pdb_file);
  lprintf("    TRJ File: %s\n", trj_file);
  lprintf("    Time Step: %.2f fs\n", timestep);
  lprintf("    Max Step:  %d ( %.2f ps ) \n", step, step*timestep/1000.0);
  lprintf("    Interval Step:  %d ( %.2f ps ) \n", interval, interval*timestep/1000.0);
  

  fp = fopen(pdb_file, "w");
  if (fp == NULL) {
    lprintf("ERROR: can't open file %s.\n", pdb_file);
    return;
  }
  for (k=0;k<smd->n_atom;k++) {
    i = smd->atom_list[k];
    
    if (i+1 < 100000) {
      sprintf(atom_no, "%5d ", i+1);
    } else {
      sprintf(atom_no, "%6d" , i+1);
    }
    j=ad->a[i].resno;
      
    if (ad->r[j].pdb_no < 10000) {
      sprintf(res_no, "%4d  ", ad->r[j].pdb_no);
    } else if (ad->r[j].pdb_no < 100000) {
      sprintf(res_no, "%5d ", ad->r[j].pdb_no);
    } else {
      sprintf(res_no, "%6d" , ad->r[j].pdb_no);
    }
    fprintf(fp,"ATOM  %-6s%-4s %-4s%c%-6s  %8.3f%8.3f%8.3f %5.2f %5.2f\n",
	    atom_no, make_PDB_atom_name(aname,ad->a[i].name),
	    ad->r[j].name, ad->r[j].pdb_chain, res_no,
	    smd->x0[k].x, smd->x0[k].y, smd->x0[k].z, 1.0, 1.0);
  }
  fclose(fp);
  fp = fopen(trj_file, "w");
  if (fp == NULL) {
    lprintf("ERROR: can't open file %s.\n", trj_file);
    return;
  }
  fprintf(fp, "# TRAJECTORY BY MARBLE n_atom: %d, output: %s\n",
	  smd->n_atom, "X");
  
  dt_ns = timestep / 1000000.0 * interval;

  trans = smd->vel * dt_ns;
  rot   = smd->ang_vel * dt_ns * M_PI / 180.0;
  
  for (j=interval;j<=step;j+=interval) {
    EP_SMD_CV_trans_rot(smd, trans, rot);
    
    for (k=0;k<smd->n_atom;k++) {    
      fprintf(fp,"%.3f %.3f %.3f\n", smd->x0[k].x, smd->x0[k].y, smd->x0[k].z);
    }
  }
  fclose(fp);
  
  lprintf("marble: going to exit\n");
}

void EP_SMD_CV_energy_force(EP_SMD *smd, ATOM_DATA *ad, LINKED_CELL *lc)
{
  int i, j;
  double dx, dy, dz, k2, fx, fy, fz;
  double cx, cy, cz, wr;
#ifdef MPI_SDMD
  int pe;
#endif

  k2 = smd->k * 2.0;
  smd->energy = 0.0;

  if (smd->center_flag) {
#ifdef MPI_SDMD
    pe = ATOM_CPU(lc, ad, smd->atom_list[0]);
    if (pe != mpi.rank) return;
#endif

    smd->center.x = smd->center.y = smd->center.z = 0.0;
    for (j=0;j<smd->n_atom;j++) {
      i = smd->atom_list[j];
      smd->center.x += smd->w[j] * ad->x[i].x;
      smd->center.y += smd->w[j] * ad->x[i].y;
      smd->center.z += smd->w[j] * ad->x[i].z;
    }
    smd->center.x /= smd->total_w;
    smd->center.y /= smd->total_w;
    smd->center.z /= smd->total_w;

    if (smd->x_flag) {
      dx = smd->center.x - smd->x0[0].x;
    } else {
      dx = 0.0;
    }	
    if (smd->y_flag) {
      dy = smd->center.y - smd->x0[0].y;
    } else {
      dy = 0.0;
    }	
    if (smd->z_flag) {
      dz = smd->center.z - smd->x0[0].z;
    } else {
      dz = 0.0;
    }	
    smd->energy = (SQR(dx) + SQR(dy) + SQR(dz))*smd->k;
    
    fx = -k2 * dx;
    fy = -k2 * dy;
    fz = -k2 * dz;

    for (j=0;j<smd->n_atom;j++) {
      i = smd->atom_list[j];
      
      wr = smd->w[j] / smd->total_w;

      ad->f[i].x += fx * wr;
      ad->f[i].y += fy * wr;
      ad->f[i].z += fz * wr;
      
      if (smd->virial_flag) {
	ad->virial[0] += fx * ad->x[i].x * wr;
	ad->virial[1] += fy * ad->x[i].y * wr;
	ad->virial[2] += fz * ad->x[i].z * wr;
	ad->virial[3] += (fx * ad->x[i].y + fy * ad->x[i].x) * 0.5 * wr;
	ad->virial[4] += (fx * ad->x[i].z + fz * ad->x[i].x) * 0.5 * wr;
	ad->virial[5] += (fy * ad->x[i].z + fz * ad->x[i].y) * 0.5 * wr;
      }
    }
      
  } else {
    for (j=0;j<smd->n_atom;j++) {
      smd->ene_list[j] = 0.0;

      i = smd->atom_list[j];
#ifdef MPI_SDMD
      pe = ATOM_CPU(lc, ad, i);
      if (pe != mpi.rank) continue;
#endif
    
      /*
	dx = ad->x[i].x - smd->x0[i].x;
	dy = ad->x[i].y - smd->x0[i].y;
	dz = ad->x[i].z - smd->x0[i].z;
      */
      if (smd->x_flag) {
	dx = ad->x[i].x - smd->x0[j].x;
      } else {
	dx = 0.0;
      }	
      if (smd->y_flag) {
	dy = ad->x[i].y - smd->x0[j].y;
      } else {
	dy = 0.0;
      }	
      if (smd->z_flag) {
	dz = ad->x[i].z - smd->x0[j].z;
      } else {
	dz = 0.0;
      }	
      smd->ene_list[j] = (SQR(dx) + SQR(dy) + SQR(dz))*smd->k;
      smd->energy += smd->ene_list[j];

      fx = -k2 * dx;
      fy = -k2 * dy;
      fz = -k2 * dz;
      ad->f[i].x += fx;
      ad->f[i].y += fy;
      ad->f[i].z += fz;

      if (smd->virial_flag) {
	ad->virial[0] += fx * ad->x[i].x;
	ad->virial[1] += fy * ad->x[i].y;
	ad->virial[2] += fz * ad->x[i].z;
	ad->virial[3] += (fx * ad->x[i].y + fy * ad->x[i].x) * 0.5;
	ad->virial[4] += (fx * ad->x[i].z + fz * ad->x[i].x) * 0.5;
	ad->virial[5] += (fy * ad->x[i].z + fz * ad->x[i].y) * 0.5;
      }
    }
  }
}

void EP_SMD_CV_prop_header(EP_SMD *smd,
			   FILE *fp, int *id)
{
  int j;
  if (smd->center_flag) {
    fprintf(fp, "%2dSMD_ENE     ", (*id)++);
    fprintf(fp, "%2dCOM_X       ", (*id)++);
    fprintf(fp, "%2dCOM_Y       ", (*id)++);
    fprintf(fp, "%2dCOM_Z       ", (*id)++);
    fprintf(fp, "%2dREF_COM_X   ", (*id)++);
    fprintf(fp, "%2dREF_COM_Y   ", (*id)++);
    fprintf(fp, "%2dREF_COM_Z   ", (*id)++);
  } else {
    fprintf(fp, "%2dSMD_ENE_ALL ", (*id)++);
    if (smd->n_atom > 1) {
      for (j=0;j<smd->n_atom;j++) {
	fprintf(fp, "%2dSMD%-9d", (*id)++, smd->atom_list[j]+1);
      }
    }
  }
}

void EP_SMD_CV_prop_out(EP_SMD *smd, MD_SYSTEM *sys, FILE *fp)
{
#ifdef MPI_SDMD
  int pe;
  LINKED_CELL *lc;
  ATOM_DATA *ad;
  MPI_Status stat;
  int i, j;
  double buf[4];

  lc = &sys->linked_cell;
  ad = &sys->atom;

  if (smd->center_flag) {
    /* puling com */
    i = smd->atom_list[0];
    pe = ATOM_CPU(lc, ad, i);

    if (mpi.master) {
      if (pe != mpi.rank) {
	MPI_Recv(buf, 4 ,MPI_DOUBLE, pe, SDMD_TAG_PROP, mpi.comm, &stat);
	smd->energy    = buf[0];
	smd->center.x  = buf[1];
	smd->center.y  = buf[2];
	smd->center.z  = buf[3];
      }
    } else {
      if (pe == mpi.rank) {
	buf[0] = smd->energy;
	buf[1] = smd->center.x;
	buf[2] = smd->center.y;
	buf[3] = smd->center.z;
	MPI_Send(buf,4,MPI_DOUBLE, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
      }
    }

    if (mpi.master) {
      fprintf(fp, " %13.6e", smd->energy);
      fprintf(fp, " %13.6e", smd->center.x);
      fprintf(fp, " %13.6e", smd->center.y);
      fprintf(fp, " %13.6e", smd->center.z);
      fprintf(fp, " %13.6e", smd->x0[0].x);
      fprintf(fp, " %13.6e", smd->x0[0].y);
      fprintf(fp, " %13.6e", smd->x0[0].z);
    } else {
      return;
    }
    
  } else {
    /* puling atoms */

    for (j = 0; j < smd->n_atom; j++) {
      i = smd->atom_list[j];
      pe = ATOM_CPU(lc, ad, i);

      if (mpi.master) {
	if (pe != mpi.rank) {
	  MPI_Recv(buf, 1 ,MPI_DOUBLE, pe, SDMD_TAG_PROP, mpi.comm, &stat);
	  smd->ene_list[j]  = buf[0];
	}
      } else {
	if (pe == mpi.rank) {
	  buf[0] = smd->ene_list[j];
	  MPI_Send(buf,1,MPI_DOUBLE, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
	}
      }
    }

    if (mpi.master) {
      double energy = 0.0;
      for (j = 0; j < smd->n_atom; j++) {
	energy += smd->ene_list[j];
      }
      fprintf(fp, " %13.6e", energy);
      if (smd->n_atom > 1) {
	for (j = 0; j < smd->n_atom; j++) {
	  fprintf(fp, " %13.6e", smd->ene_list[j]);
	}
      }
    } else {
      return;
    }
  }

#else /* MPI_SDMD */
  int j;
  
  fprintf(fp, " %13.6e", smd->energy);

  if (smd->n_atom > 1) {
    for (j = 0; j < smd->n_atom; j++) {
      fprintf(fp, " %13.6e", smd->ene_list[j]);
    }
  }
#endif /* MPI_SDMD */
 
}

void EP_SMD_CV_gradual_change(EP_SMD *smd, MD_SYSTEM *sys, int step)
{
  double trans, rot, dt_ns;
  double lambda;

  dt_ns = sys->dt_ps / 1000;

  trans = smd->vel * dt_ns;
  rot   = smd->ang_vel * dt_ns * M_PI / 180.0;

  EP_SMD_CV_trans_rot(smd, trans, rot);

  if (smd->gradual_change_step) {
    if (step < smd->gradual_change_step) {
      lambda = (double)step/smd->gradual_change_step;
      smd->k = smd->k0*(1.0-lambda) + smd->k1*lambda;
    } else {
      smd->k = smd->k1;
    }
  }
}

  /* trans: in Angstrom,
     rot:   in Radian   */
void EP_SMD_CV_trans_rot(EP_SMD *smd, double trans, double rot)
{
  int i, j, k, nx0;
  double S[3][3], R[3][3], S2[3][3], pos2[3], dx[3], dx2[3], co, si;

  if (trans == 0.0 && rot == 0.0) return;

  for (i=0;i<3;i++) {
    pos2[i] = smd->pos[i] + trans * smd->dir[i];
  }
  S[0][0] = 0.0;          S[0][1] = -smd->dir[2]; S[0][2] =  smd->dir[1];
  S[1][0] =  smd->dir[2]; S[1][1] = 0.0;          S[1][2] = -smd->dir[0];
  S[2][0] = -smd->dir[1]; S[2][1] =  smd->dir[0]; S[2][2] = 0.0;
  
  co = cos(rot);
  si = sin(rot);

  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      S2[i][j] = 0.0;
      for (k=0;k<3;k++) {
	S2[i][j] += S[i][k] * S[k][j];
      }
    }
  }
  
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      if (i==j) R[i][i] = 1.0;
      else      R[i][j] = 0.0;

      R[i][j] += si * S[i][j] + (1.0-co) * S2[i][j];
    }
  }

  /*
  printf("%f %f\n", si, co);

  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      printf(" %f", S[i][j]);
    }
    printf("\n");
  }

  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      printf(" %f", S2[i][j]);
    }
    printf("\n");
  }

  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      printf(" %f", R[i][j]);
    }
    printf("\n");
  }
  */

  for (i=0;i<smd->n_x0;i++) {
    dx[0] = smd->x0[i].x - smd->pos[0];
    dx[1] = smd->x0[i].y - smd->pos[1];
    dx[2] = smd->x0[i].z - smd->pos[2];

    for (j=0;j<3;j++) {
      dx2[j] = 0.0;
      for (k=0;k<3;k++) {
	dx2[j] += R[j][k] * dx[k];
      }
    }

    smd->x0[i].x = dx2[0] + pos2[0];    
    smd->x0[i].y = dx2[1] + pos2[1];    
    smd->x0[i].z = dx2[2] + pos2[2];
  }

  /*
  printf("gogo %f %f\n", trans, rot);
  printf("keke %f %f %f\n", smd->x0[0].x, smd->x0[0].y, smd->x0[0].z);
  */
}


/* RMSD restraint   2008/01/07                  */
/*   EP_RMSD:  RMSD restraint                   */

void EP_RMSD_setup(EXTRA_POT *ep,
		   ATOM_DATA *ad,
		   LINKED_CELL *lc,
		   STR_LIST *group_str, int group_no,
		   char *crd_file,  char *crd_file2,
		   int pdb_mode, int pdb2_mode,
		   STR_LIST *pdb_group_str, STR_LIST *pdb_group_str2, 
		   int check_pdb_group,
		   int best_fit, int weight_flag, WEIGHT_LIST *weight_list,
		   double k, double r0,
		   int gradual_change_step_k,  double k1, 
		   int gradual_change_step_r, double r1, 
		   int k_per_atom_flag,
		   int potential_type,
		   int virial_flag)
{
  EP_RMSD *rmsd;
  FILE *fp;
  char buf[100];
  int i, j, n_atom, check_group_no;
  double len, x, y, z;
  double rmsd0, rmsd1;
  unsigned int g_bit;
  WEIGHT_LIST *wl;
  char *fname = "EP_RMSD_setup";

  rmsd = emalloc(fname, sizeof(EP_RMSD));
  EP_add_item(ep, (EP_ITEM*) rmsd);

  rmsd->type = EP_T_RMSD;

  strcpy(rmsd->crd_file, crd_file);
  if (crd_file2[0]) {
    /* differences in RMSD for two coordinates */
    rmsd->mode = 2;
    strcpy(rmsd->crd_file2, crd_file2);
  } else {
    /* RMSD for one coordinates */
    rmsd->mode = 1;
    rmsd->crd_file2[0] = '\0';
  }
  rmsd->best_fit = best_fit;
  rmsd->weight_flag = weight_flag;
  rmsd->k = rmsd->k0 = k;
  rmsd->k1 = k1;
  rmsd->r = rmsd->r0 = r0;
  rmsd->r1 = r1;
  rmsd->virial_flag = virial_flag;
  rmsd->gradual_change_step_k = gradual_change_step_k;
  rmsd->gradual_change_step_r = gradual_change_step_r;
  rmsd->group_no = group_no;
  rmsd->potential_type = potential_type;

  ad->ex_force_flag = 1;

  lprintf("RMSD Restraint:\n");

  if (group_str!=NULL) {
    rmsd->group_no = group_no = 31;
    ATOM_DATA_set_group_str_list(ad, group_str, group_no, "    GROUP:");
  } else {
    lprintf("  GROUP NO. %d\n", rmsd->group_no);
  }

  rmsd->n_atom = ATOM_DATA_count_group_atoms(ad, group_no);
  if (rmsd->n_atom == 0) {
    lprintf("  ERROR: no atom is specified!\n");
    marble_exit(1);
  }
  
  rmsd->atom_list = emalloc(fname,sizeof(int)*rmsd->n_atom);
  rmsd->x  = emalloc(fname,sizeof(VEC)  *rmsd->n_atom);
  rmsd->x0 = emalloc(fname,sizeof(VEC)  *rmsd->n_atom);
  if (rmsd->best_fit) {
    rmsd->x0r = emalloc(fname,sizeof(VEC)  *rmsd->n_atom);
  } else {
    rmsd->x0r = rmsd->x0;
  }
  if (rmsd->mode == 2) {
    rmsd->x1 = emalloc(fname,sizeof(VEC)  *rmsd->n_atom);
    if (rmsd->best_fit) {
      rmsd->x1r = emalloc(fname,sizeof(VEC)  *rmsd->n_atom);
    } else {
      rmsd->x1r = rmsd->x1;
    }
  }
  rmsd->w = emalloc(fname,sizeof(double)*rmsd->n_atom);
  ATOM_DATA_make_list_of_group_atoms(ad,group_no,rmsd->atom_list);


  if (rmsd->weight_flag) {
    for (i = 0; i < rmsd->n_atom; i++) {
      rmsd->w[i] = ad->w[rmsd->atom_list[i]];
    }
    lprintf("  Weight: mass\n");
  } else {
    for (i = 0; i < rmsd->n_atom; i++) {
      rmsd->w[i] = 1.0;
    }
    lprintf("  Weight: 1.0\n");
  }
  for (wl = weight_list; wl != NULL; wl=wl->next) {
    lprintf("  Changing weights for selected atoms: %.1f\n", wl->weight);
    if (wl->group_str == NULL) {
      lprintf("    No atoms specified.\n");
      continue;
    }
    group_no = 30;
    g_bit = 1 << group_no;
    ATOM_DATA_set_group_str_list(ad, wl->group_str, group_no, "    GROUP:");
    
    for (i = 0; i < rmsd->n_atom; i++) {
      if (ad->ex[rmsd->atom_list[i]].group & g_bit) {
	rmsd->w[i] = wl->weight;
	/*
	ATOM_DATA_print_atom(ad, rmsd->atom_list[i]);
	*/
      }
    }
  }
  if (k_per_atom_flag == 1) {
    rmsd->k  *= rmsd->n_atom;
    rmsd->k1 *= rmsd->n_atom;
  }

  if (rmsd->mode == 1) {
    lprintf("  Harmonic Restraint for RMSD\n");
    if (pdb_mode) {
      lprintf("  Reference PDB File: %s\n", rmsd->crd_file);
    } else {
      lprintf("  Reference Cooridnate File: %s\n", rmsd->crd_file);
    }
  } else if (rmsd->mode == 2) {
    lprintf("  Harmonic Restraint for Difference between two RMSDs\n");
    if (pdb_mode) {
      lprintf("  Reference PDB File: %s\n", rmsd->crd_file);
    } else {
      lprintf("  Reference Cooridnate File: %s\n", rmsd->crd_file);
    }
  }

  if (check_pdb_group) {
    check_group_no = rmsd->group_no;
  } else {
    check_group_no = -1;
  }

  if (pdb_mode) {
    if (pdb_group_str == NULL) {
      pdb_group_str = group_str;
    }
  }

  ATOM_DATA_read_pdb_or_crd_file(ad, crd_file, pdb_mode, pdb_group_str,
				 rmsd->x0, rmsd->n_atom, 
				 "    GROUP:", check_group_no);

  if (rmsd->mode == 2) {
    if (pdb2_mode) {
      lprintf("  Reference PDB File2: %s\n", rmsd->crd_file2);
    } else {
      lprintf("  Reference Cooridnate File2: %s\n", rmsd->crd_file2);
    }

    if (pdb2_mode) {
      if (pdb_group_str2 == NULL) {
	pdb_group_str2 = pdb_group_str;
      }
    }
    
    ATOM_DATA_read_pdb_or_crd_file(ad, crd_file2, pdb2_mode, pdb_group_str2,
				   rmsd->x1, rmsd->n_atom, 
				   "    GROUP:",check_group_no);
  }

  lprintf("  Force Constant: %.3f (%.3f * n_atom) [kcal/mol/Angstrom^2]\n", rmsd->k, rmsd->k/rmsd->n_atom);
  if (rmsd->mode == 1) {
    lprintf("  Target RMSD: %.3f [Angstrom]\n", rmsd->r0);
  } else if (rmsd->mode == 2) {
    lprintf("  Target RMSD Difference: %.3f [Angstrom]\n", rmsd->r0);
  }
  lprintf("  Best Fit: %s\n", (rmsd->best_fit) ? "on" : "off");
  /*
  lprintf("  Weight Flag: %s\n", (rmsd->weight_flag) ? "mass_weighted" : "uniform");
  */
  {
    char *ptype_name[] = { "full", "lower", "upper" };
    lprintf("  Potential Type:  %s\n", ptype_name[rmsd->potential_type]);
  }
  lprintf("  Virial Flag: %s\n", (rmsd->virial_flag) ? "on" : "off");
  if (rmsd->gradual_change_step_k) {
    lprintf("  Gradual Change of Force Constant: %.3f (%.3f * n_atom) -> %.3f (%.3f * n_atom) in %d steps\n", 
	    rmsd->k, rmsd->k/rmsd->n_atom, rmsd->k1, rmsd->k1/rmsd->n_atom, rmsd->gradual_change_step_k);
  }
  if (rmsd->gradual_change_step_r) {
    lprintf("  Gradual Change of Target RMSD: %f -> %f in %d steps\n", 
	    rmsd->r0, rmsd->r1, rmsd->gradual_change_step_r);
  }
  
  rmsd->fit = emalloc(fname, sizeof(FIT));
  FIT_init(rmsd->fit, rmsd->n_atom, rmsd->w);

  ATOM_DATA_get_coord_of_atom_list(ad, rmsd->x, rmsd->atom_list, rmsd->n_atom);
  
  /*lprintf("%d %f %f\n", rmsd->atom_list[23579], ad->x[474694].x, rmsd->x[23579].x);*/

  /* RMSD Calculation */
  if (rmsd->best_fit) {
    FIT_best_fit(rmsd->fit, rmsd->x0, rmsd->x, NULL);
  } else {
    FIT_calc_rmsd(rmsd->fit, rmsd->x0, rmsd->x);
  }
  /*lprintf("%f %f %f, %f %f %f\n", rmsd->x0[0].x, rmsd->x0[0].y, rmsd->x0[0].z,
    rmsd->x[0].x, rmsd->x[0].y, rmsd->x[0].z);*/
  rmsd0 = rmsd->fit->rmsd;
  if (rmsd->mode == 1) {
    lprintf("  Current RMSD: %f [Angstrom]\n", rmsd0);
  } else {
    if (rmsd->best_fit) {
      FIT_best_fit(rmsd->fit, rmsd->x1, rmsd->x, NULL);
    } else {
      FIT_calc_rmsd(rmsd->fit, rmsd->x1, rmsd->x);
    }
    rmsd1 = rmsd->fit->rmsd;
    lprintf("  Current RMSD for crd_file1: %f [Angstrom]\n", rmsd0);
    lprintf("  Current RMSD for crd_file2: %f [Angstrom]\n", rmsd1);
    lprintf("  Current Difference between two RMSDs: %f [Angstrom]\n", rmsd0-rmsd1);
  }
  

#ifdef MPI_SDMD
  if (rmsd->best_fit || rmsd->mode == 2)
    SDMD_setup_ex_tr_atom(lc, rmsd->n_atom, rmsd->atom_list);
#endif  
  lprintf("\n");  
}

void EP_RMSD_energy_force(EP_RMSD *rmsd, ATOM_DATA *ad, LINKED_CELL *lc)
{
  int i, j;
  double dx, dy, dz;
  double dx1, dy1, dz1;
  double k2, k2w, k2_0, k2_1, k2_0w, k2_1w;
  double fx, fy, fz;
  double dd;

#ifdef MPI_SDMD
  int pe;
#endif

#ifdef MPI_SDMD
  if (rmsd->best_fit || rmsd->mode == 2) {
    pe = ATOM_CPU(lc, ad, rmsd->atom_list[0]);
    if (pe != mpi.rank) {
      rmsd->energy = 0.0;
      return;
    }
  }
#endif

  ATOM_DATA_get_coord_of_atom_list(ad, rmsd->x, rmsd->atom_list, rmsd->n_atom);

  if (rmsd->best_fit) {
    FIT_best_fit(rmsd->fit, rmsd->x0, rmsd->x, rmsd->x0r);
    rmsd->cur_rmsd = rmsd->fit->rmsd;
    if (rmsd->mode == 2) {
      FIT_best_fit(rmsd->fit, rmsd->x1, rmsd->x, rmsd->x1r);
      rmsd->cur_rmsd1 = rmsd->fit->rmsd;
    }
  } else {
#ifdef MPI_SDMD
    if (rmsd->mode == 1) {
      double msd_val = 0.0, msd_val2;
      for (i=0;i<rmsd->n_atom;i++) {
	j = rmsd->atom_list[i];
	if (ATOM_CPU(lc, ad, j) != mpi.rank) continue;
	dx = rmsd->x0[i].x - rmsd->x[i].x;
	dy = rmsd->x0[i].y - rmsd->x[i].y;
	dz = rmsd->x0[i].z - rmsd->x[i].z;
	msd_val += rmsd->fit->w[i] * (dx*dx + dy*dy + dz*dz);
      }
      MPI_Allreduce(&msd_val, &msd_val2, 1, MPI_DOUBLE, MPI_SUM, mpi.comm);
      msd_val2 /= rmsd->fit->total_w;
      rmsd->cur_rmsd = rmsd->fit->rmsd = sqrt(msd_val2);
      /*lprintf("%f\n", rmsd->cur_rmsd);*/
    } else {
      /* rmsd->mode == 2 */
      FIT_calc_rmsd(rmsd->fit, rmsd->x0, rmsd->x);
      rmsd->cur_rmsd = rmsd->fit->rmsd;
      FIT_calc_rmsd(rmsd->fit, rmsd->x1, rmsd->x);
      rmsd->cur_rmsd1 = rmsd->fit->rmsd;
    }
#else
    FIT_calc_rmsd(rmsd->fit, rmsd->x0, rmsd->x);
    rmsd->cur_rmsd = rmsd->fit->rmsd;
    if (rmsd->mode == 2) {
      FIT_calc_rmsd(rmsd->fit, rmsd->x1, rmsd->x);
      rmsd->cur_rmsd1 = rmsd->fit->rmsd;
    }
#endif
  }

  if (rmsd->mode == 1) {
    dd = rmsd->cur_rmsd - rmsd->r;

#ifdef MPI_SDMD
    if (rmsd->best_fit) {
      rmsd->energy = rmsd->k * dd * dd;
    } else {
      if (mpi.master)
	rmsd->energy = rmsd->k * dd * dd;
      else
	rmsd->energy = 0.0;
    }
#else
    rmsd->energy = rmsd->k * dd * dd;
#endif    

    if (rmsd->cur_rmsd < 1.0e-10) {
      k2 = 0.0;
    } else {
      k2 = rmsd->k * dd / rmsd->cur_rmsd * 2.0 / rmsd->fit->total_w;
    }

    if (rmsd->potential_type == 1) {
      /* repulsive or lower */
      if (rmsd->cur_rmsd > rmsd->r) {
	rmsd->energy = 0.0;
	k2 = 0.0;
      }
    } else if (rmsd->potential_type == 2) {
      /* attractive or upper */
      if (rmsd->cur_rmsd < rmsd->r) {
	rmsd->energy = 0.0;
	k2 = 0.0;
      }
    }

    for (i=0;i<rmsd->n_atom;i++) {
      j = rmsd->atom_list[i];
#ifdef MPI_SDMD
      if (!rmsd->best_fit) 
	if (ATOM_CPU(lc, ad, j) != mpi.rank) continue;
#endif

      dx = rmsd->x[i].x - rmsd->x0r[i].x;
      dy = rmsd->x[i].y - rmsd->x0r[i].y;
      dz = rmsd->x[i].z - rmsd->x0r[i].z;
      k2w = k2 * rmsd->w[i];

      fx = -k2w * dx;
      fy = -k2w * dy;
      fz = -k2w * dz;

      ad->f[j].x += fx;
      ad->f[j].y += fy;
      ad->f[j].z += fz;

      if (rmsd->virial_flag) {
	ad->virial[0] += fx * rmsd->x[i].x;
	ad->virial[1] += fy * rmsd->x[i].y;
	ad->virial[2] += fz * rmsd->x[i].z;
	ad->virial[3] += (fx * rmsd->x[i].y + fy * rmsd->x[i].x) * 0.5;
	ad->virial[4] += (fx * rmsd->x[i].z + fz * rmsd->x[i].x) * 0.5;
	ad->virial[5] += (fy * rmsd->x[i].z + fz * rmsd->x[i].y) * 0.5;
      }
    }
  } else {
    /* mode 2 */

    dd = rmsd->cur_rmsd - rmsd->cur_rmsd1 - rmsd->r;
    rmsd->energy = rmsd->k * dd * dd;

    k2 = rmsd->k * dd * 2.0 / rmsd->fit->total_w;

    if (rmsd->potential_type == 1) {
      /* repulsive or lower */
      if (dd > 0.0) {
	rmsd->energy = 0.0;
	k2 = 0.0;
      }
    } else if (rmsd->potential_type == 2) {
      /* attractive or upper */
      if (dd < 0.0) {
	rmsd->energy = 0.0;
	k2 = 0.0;
      }
    }

    if (rmsd->cur_rmsd < 1.0e-10) {
      k2_0 = 0.0;
    } else {
      k2_0 = k2 / rmsd->cur_rmsd;
    }

    if (rmsd->cur_rmsd1 < 1.0e-10) {
      k2_1 = 0.0;
    } else {
      k2_1 = k2 / rmsd->cur_rmsd1;
    }

    for (i=0;i<rmsd->n_atom;i++) {
      j = rmsd->atom_list[i];

      dx = rmsd->x[i].x - rmsd->x0r[i].x;
      dy = rmsd->x[i].y - rmsd->x0r[i].y;
      dz = rmsd->x[i].z - rmsd->x0r[i].z;
      k2_0w = k2_0 * rmsd->w[i];

      dx1 = rmsd->x[i].x - rmsd->x1r[i].x;
      dy1 = rmsd->x[i].y - rmsd->x1r[i].y;
      dz1 = rmsd->x[i].z - rmsd->x1r[i].z;
      k2_1w = k2_1 * rmsd->w[i];

      fx = - (k2_0w * dx - k2_1w * dx1);
      fy = - (k2_0w * dy - k2_1w * dy1);
      fz = - (k2_0w * dz - k2_1w * dz1);

      ad->f[j].x += fx;
      ad->f[j].y += fy;
      ad->f[j].z += fz;

      if (rmsd->virial_flag) {
	ad->virial[0] += fx * rmsd->x[i].x;
	ad->virial[1] += fy * rmsd->x[i].y;
	ad->virial[2] += fz * rmsd->x[i].z;
	ad->virial[3] += (fx * rmsd->x[i].y + fy * rmsd->x[i].x) * 0.5;
	ad->virial[4] += (fx * rmsd->x[i].z + fz * rmsd->x[i].x) * 0.5;
	ad->virial[5] += (fy * rmsd->x[i].z + fz * rmsd->x[i].y) * 0.5;
      }
    }
    
  }

}

void EP_RMSD_prop_header(EP_RMSD *rmsd, FILE *fp, int *id)
{
  if (rmsd->mode == 1) {
    fprintf(fp, "%2dRMSD_ENE    ", (*id)++);
    fprintf(fp, "%2dRMSD        ", (*id)++);
    if (rmsd->gradual_change_step_r)
      fprintf(fp, "%2dTARGET_RMSD ", (*id)++);
    if (rmsd->gradual_change_step_k)
      fprintf(fp, "%2dRMSD_K      ", (*id)++);
  } else {
    /* mode 2 */
    fprintf(fp, "%2dRMSD_ENE    ", (*id)++);
    fprintf(fp, "%2dRMSD_DIFF   ", (*id)++);
    fprintf(fp, "%2dRMSD0       ", (*id)++);
    fprintf(fp, "%2dRMSD1       ", (*id)++);
    if (rmsd->gradual_change_step_r)
      fprintf(fp, "%2dTARGET_DIFF ", (*id)++);
    if (rmsd->gradual_change_step_k)
      fprintf(fp, "%2dRMSD_K      ", (*id)++);
  }

}

void EP_RMSD_prop_out(EP_RMSD *rmsd, MD_SYSTEM *sys, FILE *fp)
{
#ifdef MPI_SDMD
  int pe;
  LINKED_CELL *lc;
  ATOM_DATA *ad;
  MPI_Status stat;
  double buf[3];

  lc = &sys->linked_cell;
  ad = &sys->atom;
  pe = ATOM_CPU(lc, ad, rmsd->atom_list[0]);
  
  if (mpi.master) {
    if (pe != mpi.rank) {
      if (rmsd->mode == 1) {
	MPI_Recv(buf,2,MPI_DOUBLE, pe, SDMD_TAG_PROP, mpi.comm, &stat);
	rmsd->energy   = buf[0];
	rmsd->cur_rmsd = buf[1];
      } else {
	MPI_Recv(buf,3,MPI_DOUBLE, pe, SDMD_TAG_PROP, mpi.comm, &stat);
	rmsd->energy    = buf[0];
	rmsd->cur_rmsd  = buf[1];
	rmsd->cur_rmsd1 = buf[2];
      }
    }
  } else {
    if (pe == mpi.rank) {
      if (rmsd->mode == 1) {
	buf[0] = rmsd->energy;
	buf[1] = rmsd->cur_rmsd;
	MPI_Send(buf,2,MPI_DOUBLE, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
      } else {
	buf[0] = rmsd->energy;
	buf[1] = rmsd->cur_rmsd;
	buf[2] = rmsd->cur_rmsd1;
	MPI_Send(buf,3,MPI_DOUBLE, mpi.master_pe, SDMD_TAG_PROP, mpi.comm);
      }
    }
    return;
  }
#endif /* MPI_SDMD */

  if (rmsd->mode == 1) {
    fprintf(fp, " %13.6e", rmsd->energy);
    fprintf(fp, " %13.6e", rmsd->cur_rmsd);
    if (rmsd->gradual_change_step_r)
      fprintf(fp, " %13.6e", rmsd->r);
    if (rmsd->gradual_change_step_k)
      fprintf(fp, " %13.6e", rmsd->k);
  } else {
    fprintf(fp, " %13.6e", rmsd->energy);
    fprintf(fp, " %13.6e", rmsd->cur_rmsd - rmsd->cur_rmsd1);
    fprintf(fp, " %13.6e", rmsd->cur_rmsd);
    fprintf(fp, " %13.6e", rmsd->cur_rmsd1);
    if (rmsd->gradual_change_step_r)
      fprintf(fp, " %13.6e", rmsd->r);
    if (rmsd->gradual_change_step_k)
      fprintf(fp, " %13.6e", rmsd->k);
  }
}

void EP_RMSD_gradual_change(EP_RMSD *rmsd, MD_SYSTEM *sys, int step)
{
  double lambda;
  
  if (rmsd->gradual_change_step_k) {
    if (step < rmsd->gradual_change_step_k) {
      lambda = (double)step/rmsd->gradual_change_step_k;
      rmsd->k = rmsd->k0*(1.0-lambda) + rmsd->k1*lambda;
    } else {
      rmsd->k = rmsd->k1;
    }
  }

  if (rmsd->gradual_change_step_r) {
    if (step < rmsd->gradual_change_step_r) {
      lambda = (double)step/rmsd->gradual_change_step_r;
      rmsd->r = rmsd->r0*(1.0-lambda) + rmsd->r1*lambda;
    } else {
      rmsd->r = rmsd->r1;
    }
  }
  
}

