/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifndef _DIHEDRAL_H_INCLUDED
#define _DIHEDRAL_H_INCLUDED

#ifndef _ATOM_H_INCLUDED
#include "atom.h"
#endif

#define DIHED_OTHER_NODE   1
#define DIHED_AMBER_IMPR   2
#define DIHED_CHARMM_IMPR  4
#define DIHED_RING         8
#define DIHED_MULTI       16
#define DIHED_OMIT14      30  /* 2+4+8+16 */
#define DIHED_ONLY14      32 

typedef struct s_dihedral {
  int atom1, atom2, atom3, atom4, type;
  int flag;
#ifdef MPI_SDMD
  int next;
#endif  
} DIHEDRAL;

typedef struct s_dihedral_type {
  double v, eta, ganma;
} DIHEDRAL_TYPE;

typedef struct s_improper_type {
  double kpsi, psi0;
} IMPROPER_TYPE;

typedef struct s_cmap {
  int atom[8], type;
  int flag;
#ifdef MPI_SDMD
  int next;
#endif  
} CMAP;

typedef struct s_cmap_type {
  int ndiv;
  double (*c)[4][4];
} CMAP_TYPE;


typedef struct s_dihedral_data {
  int n_dihedral, n_improper;
  int n_multi, n_only14, n_nb14;
  DIHEDRAL *dihedrals;

  int n_dihed_type;
  DIHEDRAL_TYPE *dihed_type;

  /* for CHARMM */
  int n_impr_type;
  IMPROPER_TYPE *impr_type;

  int cmap_flag;
  int n_cmap, n_cmap_type;
  CMAP *cmap;
  CMAP_TYPE *cmap_type;
  
#ifdef MPI_RDMD
  int start_task, end_task;
#endif

#ifdef MPI_SDMD
  int head, head_cmap;
#endif
  
} DIHEDRAL_DATA;

void DIHEDRAL_DATA_init(DIHEDRAL_DATA *dd);
void DIHEDRAL_DATA_finalize(DIHEDRAL_DATA *dd);
void dihedral_energy_force(DIHEDRAL_DATA *,ATOM_DATA *, double *, double *);
int read_dihedral_data(DIHEDRAL_DATA *, FILE*);
void print_dihedrals(FILE *, DIHEDRAL_DATA*);
void DIHEDRAL_DATA_scale_improper_k(DIHEDRAL_DATA *dd, double scale);
void DIHEDRAL_DATA_overwrite_improper_k(DIHEDRAL_DATA *dd, double val);
void nonbond14_energy_force(DIHEDRAL_DATA *dd, NONBOND_LIST *nl,
			    ATOM_DATA *ad,
			    double *vdw14,double *elec14);
void DIHEDRAL_DATA_omit_data_for_rigid_mol(DIHEDRAL_DATA *dd, ATOM_DATA *ad);

void DD_CMAP_make_derivatives(double *e, double *dedx, double *dedy, double *d2edxdy, 
                              double (*v)[4], int nx, int ny, double dx, double dy);
void DD_CMAP_cyc_cubic_spline(double (*y)[4], int n, double dx);
void DD_CMAP_make_cij(double *e, double *dedx, double *dedy, double *d2edxdy,
                      int nx, int ny, double dx, double dy, double (*c)[4][4]);
void DD_CMAP_energy_force(DIHEDRAL_DATA *dd, 
                          ATOM_DATA *ad,
                          double *energy);



#endif  /* _DIHEDRAL_H_INCLUDED */
