/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "misc.h"
#include "atom.h"
#include "boundary.h"
#include "linked_cell.h"
#include "nonbond.h"
#include "dihedral.h"

void DIHEDRAL_DATA_init(DIHEDRAL_DATA *dd)
{
  dd->n_dihedral = dd->n_improper = 0;
  dd->dihedrals = NULL;

  dd->n_dihed_type = 0;
  dd->dihed_type = NULL;

  dd->n_impr_type = 0;
  dd->impr_type = NULL;
  
  dd->n_cmap = 0;
  dd->n_cmap_type = 0;
  dd->cmap = NULL;
  dd->cmap_type = NULL;
}

void DIHEDRAL_DATA_finalize(DIHEDRAL_DATA *dd)
{
  if (dd->dihedrals) free(dd->dihedrals);
  if (dd->dihed_type) free(dd->dihed_type);
  DIHEDRAL_DATA_init;
}

void dihedral_energy_force(DIHEDRAL_DATA *dihedral_data, 
                           ATOM_DATA *atom_data,
                           double *energy, double *energy_impr)
{
  int i;
  DIHEDRAL *dihe;
  double v, eta, ganma;
  double kpsi, psi0;
  int atom1, atom2, atom3, atom4, type;
  double phi, e;
  double x12,y12,z12,x23,y23,z23,x34,y34,z34,nx23,ny23,nz23,len_23;
  double dx,dy,dz,gx,gy,gz,dd,gg,dg,len_dg;
  double ddx,ddy,ddz,dgx,dgy,dgz;
  double F, Fx1,Fy1,Fz1,Fx4,Fy4,Fz4,Fx2_d,Fy2_d,Fz2_d,Fx2_g,Fy2_g,Fz2_g;
  VEC *x, *f;
  
  x = atom_data->x;
  f = atom_data->f;
  
#ifdef MPI_RDMD
  for (i = dihedral_data->start_task; i <= dihedral_data->end_task; i++) {
#elif defined(MPI_SDMD) /* */
  for (i=dihedral_data->head; i>=0; i=dihedral_data->dihedrals[i].next) {
#else
  for (i=0; i < dihedral_data->n_dihedral; i++) {
#endif
#ifdef MPI_SDMD
    /*if (dihedral_data->dihedrals[i].flag & DIHED_OTHER_NODE) continue;*/
    /*if (dihedral_data->dihedrals[i].flag & DIHED_OTHER_NODE) {
      printf("%d %d\n", i, dihedral_data->dihedrals[i].flag & DIHED_OTHER_NODE);
      continue;
      }*/
#endif    
    dihe = &(dihedral_data->dihedrals[i]);
    type = dihe->type;

    if (dihe->flag & DIHED_ONLY14) continue;

    if (dihe->flag & DIHED_CHARMM_IMPR) {
      kpsi  = dihedral_data->impr_type[type].kpsi;
      psi0  = dihedral_data->impr_type[type].psi0;
    } else {
      v     = dihedral_data->dihed_type[type].v;
      eta   = dihedral_data->dihed_type[type].eta;
      ganma = dihedral_data->dihed_type[type].ganma;
    }
    atom1 = dihe->atom1;
    atom2 = dihe->atom2;
    atom3 = dihe->atom3;
    atom4 = dihe->atom4;
    if (atom3 < 0) atom3 = -atom3;
    if (atom4 < 0) atom4 = -atom4;
    x12 = x[atom2].x - x[atom1].x;
    y12 = x[atom2].y - x[atom1].y;
    z12 = x[atom2].z - x[atom1].z;
    x23 = x[atom3].x - x[atom2].x;
    y23 = x[atom3].y - x[atom2].y;
    z23 = x[atom3].z - x[atom2].z;
    x34 = x[atom4].x - x[atom3].x;
    y34 = x[atom4].y - x[atom3].y;
    z34 = x[atom4].z - x[atom3].z;
    dx  = y12*z23 - z12*y23;
    dy  = z12*x23 - x12*z23;
    dz  = x12*y23 - y12*x23;
    gx  = y23*z34 - z23*y34;
    gy  = z23*x34 - x23*z34;
    gz  = x23*y34 - y23*x34;

    len_23 = sqrt(x23*x23 + y23*y23 + z23*z23);
    nx23 = x23 / len_23;
    ny23 = y23 / len_23;
    nz23 = z23 / len_23;
    dd  = dx*dx + dy*dy + dz*dz;
    gg  = gx*gx + gy*gy + gz*gz;
    dg  = dx*gx + dy*gy + dz*gz;
    len_dg = sqrt(dd) * sqrt(gg);

    /*
    phi = atan2(-(nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)
		  +nz23*(dx*gy-dy*gx))/len_dg,  dg/len_dg);
    */
    phi = atan2((nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)
		 +nz23*(dx*gy-dy*gx))/len_dg,  dg/len_dg);

    /*
    lprintf("phi(%d,%d,%d,%d) %f %d\n", atom1+1,atom2+1, atom3+1, atom4+1,
	   phi/M_PI*180.0,dihe->flag);
    */
    
    if (dihe->flag & DIHED_CHARMM_IMPR) {
      if (phi - psi0 < -M_PI) phi += 2.0*M_PI;
      else if (phi - psi0 > M_PI) phi -= 2.0*M_PI;
      /* lprintf("%f %f\n", phi, psi0); */
      e = kpsi * (phi - psi0) * (phi - psi0);
      *energy_impr += e;
    } else {
      e = v * (1.0 + cos(eta * phi - ganma));
      *energy += e;
    }

    /*
    if (!(dihe->flag & DIHED_CHARMM_IMPR))
      lprintf("%d %d %d %d %e\n",atom1+1,atom2+1,atom3+1,atom4+1, e);
    */

#if 0
    if (atom_data->atom_ene_flag) {
      e /= 4.0;
      atom_data->atom_ene[atom1].type[ATOM_DIHED_ENE] += e;
      atom_data->atom_ene[atom2].type[ATOM_DIHED_ENE] += e;
      atom_data->atom_ene[atom3].type[ATOM_DIHED_ENE] += e;
      atom_data->atom_ene[atom4].type[ATOM_DIHED_ENE] += e;
    }
#endif

    /*
    ddx = -(dy*nz23 - dz*ny23)/dd;
    ddy = -(dz*nx23 - dx*nz23)/dd;
    ddz = -(dx*ny23 - dy*nx23)/dd;
    
    dgx = (gy*nz23 - gz*ny23)/gg;
    dgy = (gz*nx23 - gx*nz23)/gg;
    dgz = (gx*ny23 - gy*nx23)/gg;
    */

    /* dd[xyz] : dphi/dd[xyz], dg[xyz] : dphi/dg[xyz]  */
    
    ddx = (dy*nz23 - dz*ny23)/dd;
    ddy = (dz*nx23 - dx*nz23)/dd;
    ddz = (dx*ny23 - dy*nx23)/dd;
    
    dgx = -(gy*nz23 - gz*ny23)/gg;
    dgy = -(gz*nx23 - gx*nz23)/gg;
    dgz = -(gx*ny23 - gy*nx23)/gg;

    if (dihe->flag & DIHED_CHARMM_IMPR)
      F = -2.0 * kpsi * (phi-  psi0);
    else
      F = v * eta * sin(eta * phi - ganma);
    
    Fx1 = F*(ddy * z23 - ddz * y23);
    Fy1 = F*(ddz * x23 - ddx * z23);
    Fz1 = F*(ddx * y23 - ddy * x23);
    Fx4 = F*(dgy * z23 - dgz * y23);
    Fy4 = F*(dgz * x23 - dgx * z23);
    Fz4 = F*(dgx * y23 - dgy * x23);
    Fx2_d = F*(ddy * z12 - ddz * y12);
    Fy2_d = F*(ddz * x12 - ddx * z12);
    Fz2_d = F*(ddx * y12 - ddy * x12);
    Fx2_g = F*(dgy * z34 - dgz * y34);
    Fy2_g = F*(dgz * x34 - dgx * z34);
    Fz2_g = F*(dgx * y34 - dgy * x34);
    
    f[atom1].x += Fx1;
    f[atom1].y += Fy1;
    f[atom1].z += Fz1;
    f[atom2].x += -Fx2_d + Fx2_g - Fx1;
    f[atom2].y += -Fy2_d + Fy2_g - Fy1;
    f[atom2].z += -Fz2_d + Fz2_g - Fz1;
    f[atom3].x +=  Fx2_d - Fx2_g - Fx4;
    f[atom3].y +=  Fy2_d - Fy2_g - Fy4;
    f[atom3].z +=  Fz2_d - Fz2_g - Fz4;
    f[atom4].x += Fx4;
    f[atom4].y += Fy4;
    f[atom4].z += Fz4;
  }
}

void print_dihedrals(FILE *fp, DIHEDRAL_DATA *dd)
{
  int i;

  for (i = 0; i < dd->n_dihedral; i++) {
    fprintf(fp, "%d %d %d %d %d\n",
	    dd->dihedrals[i].atom1,
	    dd->dihedrals[i].atom2,
	    dd->dihedrals[i].atom3,
	    dd->dihedrals[i].atom4,
	    dd->dihedrals[i].type);
  }
}

void DIHEDRAL_DATA_scale_improper_k(DIHEDRAL_DATA *dd, double scale)
{
  int i;

  lprintf("Scale force constants of impropers: %lf\n\n", scale);
  if (dd->n_impr_type == 0) {
    lprintf("Warning: No charmm type improper diheral angles.\n\n");
    return;
  }
  
  for (i=0;i<dd->n_impr_type;i++) {
    dd->impr_type[i].kpsi *= scale;
  }
}

void DIHEDRAL_DATA_overwrite_improper_k(DIHEDRAL_DATA *dd, double val)
{
  int i;

  lprintf("Overwrite force constants of impropers: %lf\n", val);
  lprintf("  (except the case of k = 0)\n\n");

  if (dd->n_impr_type == 0) {
    lprintf("Warning: No charmm type improper diheral angles.\n\n");
    return;
  }
  
  for (i=0;i<dd->n_impr_type;i++) {
    if (dd->impr_type[i].kpsi == 0.0) continue;
    dd->impr_type[i].kpsi = val;
  }
}

void nonbond14_energy_force(DIHEDRAL_DATA *dd, NONBOND_LIST *nl,
			    ATOM_DATA *ad,
			    double *vdw14,double *elec14)
{
  int i,j,k;
  int vdw_index;
  double dx, dy, dz;
  double len, len2;
  double rlen2, rlen6, rlen12;
  double vdw12, vdw6, hb12, hb10, force, elec_tmp, vdw_tmp;
  double dSe, Se;
  /* CHARMM_VFSWITCH */
  double rl_ronoff6, rl_ronoff3;

  *vdw14 = *elec14 = 0.0;
  
  if (nl->vdw_method == NV_FSW) {
    rl_ronoff3  = nl->rl_on * nl->rl_on * nl->rl_on;
    rl_ronoff3 *= nl->rl_off * nl->rl_off * nl->rl_off;
    rl_ronoff3 = 1.0/rl_ronoff3;
    rl_ronoff6 = rl_ronoff3*rl_ronoff3;
  }
  
#ifdef MPI_RDMD
  for (k = dd->start_task; k <= dd->end_task; k++) {
#elif defined(MPI_SDMD)
  for (k = dd->head; k>=0; k=dd->dihedrals[k].next) {
#else
  for (k=0;k<dd->n_dihedral;k++) {
#endif
#ifdef MPI_SDMD
    /* if (dd->dihedrals[k].flag & DIHED_OTHER_NODE) continue; */
#endif    
    if (dd->dihedrals[k].atom3 < 0 || dd->dihedrals[k].atom4 < 0)
      continue;
    if (dd->dihedrals[k].flag & DIHED_OMIT14) continue;
    
    i = dd->dihedrals[k].atom1;
    j = dd->dihedrals[k].atom4;

    dx = ad->x[i].x - ad->x[j].x;
    dy = ad->x[i].y - ad->x[j].y;
    dz = ad->x[i].z - ad->x[j].z;

    len2 = dx * dx + dy * dy + dz * dz;
    rlen2 = 1.0/len2;
    rlen6 = rlen2 * rlen2 * rlen2;
    rlen12 = rlen6 * rlen6;

    vdw_index = ad->index[ad->vdw_type[i]+ad->vdw_type[j]*ad->ntype];

#ifdef HBOND      
    if (vdw_index >= 0) {
#endif
      
    if (nl->vdw_method == NV_FSW) {
      vdw12 = ad->vdw12_14[vdw_index];
      vdw6  = ad->vdw6_14[vdw_index];
      vdw_tmp = vdw12 * (rlen12 - rl_ronoff6) - vdw6 * (rlen6 - rl_ronoff3);
      *vdw14 += vdw_tmp;
      force = 12.0 * vdw12 * rlen12 - 6.0 * vdw6 * rlen6;
    } else {
      vdw12 = ad->vdw12_14[vdw_index] * rlen12;
      vdw6  = ad->vdw6_14[vdw_index]  * rlen6;
      vdw_tmp = vdw12 - vdw6;
      *vdw14 += vdw_tmp;
      force = 12.0 * vdw12 - 6.0 * vdw6;
    }
    
#ifdef HBOND
    } else {
      vdw_index = - vdw_index;
      hb12 = ad->hb12[vdw_index] / len12;
      hb10 = ad->hb10[vdw_index] / (len6 * len2 * len2);
      *hbond += hb12 - hb10;
      force = 12.0 * hb12 - 10.0 * hb10;
    }
#endif    
    
    len = sqrt(len2);
    elec_tmp = ad->q[i] * ad->q[j] / (len * ad->scee);

    /*
    lprintf("%s - %s %f %f\n", ad->a[i].name, ad->a[j].name, elec_tmp, vdw_tmp);
    */
    
    if (nl->elec_method == NE_FSH) {
      dSe = 1.0 - len/nl->rl_off;
      Se = dSe*dSe; dSe*=-2.0/nl->rl_off;
      *elec14 += elec_tmp*Se;
      force = (force + elec_tmp*Se) * rlen2 - (elec_tmp*dSe)/len;
    } else {
      force = (force + elec_tmp) * rlen2;
      *elec14 += elec_tmp;
    }

#ifdef ATOM_ENE
    if (ad->atom_ene_flag) {
      vdw_tmp *= 0.5; elec_tmp *= 0.5;
      ad->atom_ene[i].type[ATOM_VDW14_ENE]  += vdw_tmp;
      ad->atom_ene[j].type[ATOM_VDW14_ENE]  += vdw_tmp;
      ad->atom_ene[i].type[ATOM_ELEC14_ENE] += elec_tmp;
      ad->atom_ene[j].type[ATOM_ELEC14_ENE] += elec_tmp;
    }
#endif    

    ad->f[i].x += force * dx;
    ad->f[i].y += force * dy;
    ad->f[i].z += force * dz;
    
    ad->f[j].x -= force * dx;
    ad->f[j].y -= force * dy;
    ad->f[j].z -= force * dz;
  }
}

void DIHEDRAL_DATA_omit_data_for_rigid_mol(DIHEDRAL_DATA *dd, ATOM_DATA *ad)
{
  int i, atom1, atom2, atom3, atom4, mol1, mol2, mol3, mol4,pack_to;

  pack_to=0;
  for (i=0;i<dd->n_dihedral;i++) {
    atom1 = dd->dihedrals[i].atom1;
    atom2 = dd->dihedrals[i].atom2;
    atom3 = dd->dihedrals[i].atom3;
    atom4 = dd->dihedrals[i].atom4;
    if (atom3<0) atom3*=-1;
    if (atom4<0) atom4*=-1;

    if ((ad->ex[atom1].flag & ATOM_RIGID) &&
	(ad->ex[atom2].flag & ATOM_RIGID) &&
	(ad->ex[atom3].flag & ATOM_RIGID) &&
	(ad->ex[atom4].flag & ATOM_RIGID)) {

      mol1 = ATOM_RMOL(ad,atom1);
      mol2 = ATOM_RMOL(ad,atom2);
      mol3 = ATOM_RMOL(ad,atom3);
      mol4 = ATOM_RMOL(ad,atom4);

      if (mol1==mol2&&mol1==mol3&&mol1==mol4)
	continue;
    }

    if ((ad->ex[atom1].flag & ATOM_FIXED) &&
	(ad->ex[atom2].flag & ATOM_FIXED) &&
	(ad->ex[atom3].flag & ATOM_FIXED) &&
	(ad->ex[atom4].flag & ATOM_FIXED)) {
      continue;
    }
    
    if (i != pack_to) {
      dd->dihedrals[pack_to] = dd->dihedrals[i];
    }
    pack_to++;
  }
  lprintf("  Number of dihedrals: %d -> %d\n", dd->n_dihedral, pack_to);
  dd->n_dihedral = pack_to;
}
  


/*
   CMAP potential 
   Thu Nov 11 19:14:00 JST 2004  Mitsunori Ikeguchi
*/

#define E(i,j)        e[(i)*ny+(j)]
#define DEDX(i,j)     dedx[(i)*ny+(j)]
#define DEDY(i,j)     dedy[(i)*ny+(j)]
#define D2EDXDY(i,j)  d2edxdy[(i)*ny+(j)]

void DD_CMAP_make_derivatives(double *e, double *dedx, double *dedy, double *d2edxdy, 
			      double (*v)[4], int nx, int ny, double dx, double dy)
{
  int i, j;
  
  for (j=0;j<ny;j++) {
    for (i=0;i<nx;i++) {
      v[i][0] = E(i,j);
    }
    DD_CMAP_cyc_cubic_spline(v, nx, dx);
    for (i=0;i<nx;i++) {
      DEDX(i,j) = v[i][1];
    }
  }

  for (i=0;i<nx;i++) {
    for (j=0;j<ny;j++) {
      v[j][0] = E(i,j);
    }
    DD_CMAP_cyc_cubic_spline(v, ny, dy);
    for (j=0;j<ny;j++) {
      DEDY(i,j) = v[j][1];
    }
  }

  for (i=0;i<nx;i++) {
    for (j=0;j<ny;j++) {
      v[j][0] = DEDX(i,j);
    }
    DD_CMAP_cyc_cubic_spline(v, ny, dy);
    for (j=0;j<ny;j++) {
      D2EDXDY(i,j) = v[j][1];
    }
  }

  /*
  for (j=0;j<ny;j++) {
    for (i=0;i<nx;i++) {
      v[i][0] = DEDY(i,j);
    }
    DD_CMAP_cyc_cubic_spline(v, nx, dx);
    for (i=0;i<nx;i++) {
      D2EDYDX(i,j) = v[i][1];
    }
  }

  for (i=0;i<nx;i++) {
    for (j=0;j<ny;j++) {
      printf("%f %f\n", D2EDXDY(i,j),D2EDYDX(i,j));
    }
  }
  */
}


/*
 * calculate cyclic cubic spline
 * input:  n : number of input
 *         dx: interval of x
 *         y[0..n-1][0]: function
 *         y[0][1], y[n-1][1]: first derivatives at end point
 * output  y[0..n-1][m]: coefficient of spline function (m=1..3)
 *
 */
void DD_CMAP_cyc_cubic_spline(double (*y)[4], int n, double dx)
{
  int i;
  double dx2inv;
  
  /* y[i][1],y[i][2], y[i][3]: temporary use
     
     y[i][1]: beta(i,i)
     y[i][2]: u(i)
     y[i][3]: alpha(n, i)
  */
  
  dx2inv = 1.0/(dx*dx);

#define BETA(i)   y[i][1]
#define ALPHA(i)  y[i][3]

  /* end point */
  BETA(0)  = 4.0;
  ALPHA(0) = 1.0/BETA(0);
  y[0][2] = 6.0*(y[1][0]-2.0*y[0][0]+y[n-1][0])*dx2inv;

  /* forward substitution */
  for (i=1;i<n-2;i++) {
    BETA(i)  = 4.0 - 1.0/BETA(i-1);
    ALPHA(i) = -ALPHA(i-1)/BETA(i);
    y[i][2] = 6.0*(y[i+1][0]-2.0*y[i][0]+y[i-1][0])*dx2inv
      -y[i-1][2]/BETA(i-1);
  }

  BETA(n-2)  = 4.0 - 1.0/BETA(n-3);
  ALPHA(n-2) = (1.0 - ALPHA(n-3))/BETA(n-2);
  y[n-2][2] = 6.0*(y[n-1][0]-2.0*y[n-2][0]+y[n-3][0])*dx2inv
    -y[n-3][2]/BETA(n-3);

  BETA(n-1)  = 4.0;
  y[n-1][2] = 6.0*(y[0][0]-2.0*y[n-1][0]+y[n-2][0])*dx2inv;
  for (i=0;i<n-1;i++) {
    BETA(n-1) -= ALPHA(i)*ALPHA(i)*BETA(i);
    y[n-1][2] -= ALPHA(i)*y[i][2];
  }

  y[n-1][2] /= BETA(n-1);
  y[n-2][2] = y[n-2][2]/BETA(n-2) - ALPHA(n-2) * y[n-1][2];

  /* backsubstitution */
  for (i=n-3;i>=0;i--) {
    y[i][2] = (y[i][2] - y[i+1][2])/BETA(i) - ALPHA(i)*y[n-1][2];
  }

  /* calculation of y[i][1], y[i][2], y[i][3] */
  for (i=0;i<n;i++) {
    y[i][2] *= 0.5;
  }

  for (i=0;i<n-1;i++) {
    y[i][3] = (y[i+1][2]-y[i][2])/(3.0*dx);
    y[i][1] = (y[i+1][0]-y[i][0])/dx -y[i][2]*dx -y[i][3]*dx*dx;
  }
  y[n-1][3] = (y[0][2]-y[n-1][2])/(3.0*dx);
  y[n-1][1] = (y[0][0]-y[n-1][0])/dx -y[n-1][2]*dx -y[n-1][3]*dx*dx;
}

#undef ALPHA
#undef BETA

#define C(x,y,i,j)  c[(x)*ny+(y)][i][j]

void DD_CMAP_make_cij(double *e, double *dedx, double *dedy, double *d2edxdy,
		      int nx, int ny, double dx, double dy, double (*c)[4][4])
{
  static double e2c[16][16] = {
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
   -3, 0, 3, 0, 0, 0, 0, 0,-2, 0,-1, 0, 0, 0, 0, 0,
    2, 0,-2, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0,
    0, 0, 0, 0,-3, 0, 3, 0, 0, 0, 0, 0,-2, 0,-1, 0,
    0, 0, 0, 0, 2, 0,-2, 0, 0, 0, 0, 0, 1, 0, 1, 0,
   -3, 3, 0, 0,-2,-1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,-3, 3, 0, 0,-2,-1, 0, 0,
    9,-9,-9, 9, 6, 3,-6,-3, 6,-6, 3,-3, 4, 2, 2, 1,
   -6, 6, 6,-6,-4,-2, 4, 2,-3, 3,-3, 3,-2,-1,-2,-1,
    2,-2, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 2,-2, 0, 0, 1, 1, 0, 0,
   -6, 6, 6,-6,-3,-3, 3, 3,-4, 4,-2, 2,-2,-2,-1,-1,
    4,-4,-4, 4, 2, 2,-2,-2, 2,-2, 2,-2, 1, 1, 1, 1
  };
  int i,j, ix, iy, ixp[2], iyp[2];
  double f[16], *cl;

  for (ix=0;ix<nx;ix++) {
    ixp[0] = ix;
    if (ix == nx-1) ixp[1]=0;
    else            ixp[1]=ix+1;
    for (iy=0;iy<ny;iy++) {
      iyp[0] = iy;
      if (iy == ny-1) iyp[1]=0;
      else            iyp[1]=iy+1;
      for (j=0;j<2;j++) {
	for (i=0;i<2;i++) {
	  f[    j*2+i] = E(ixp[i], iyp[j]);
	  f[4 + j*2+i] = DEDX(ixp[i], iyp[j]) * dx;
	  f[8 + j*2+i] = DEDY(ixp[i], iyp[j]) * dy;
	  f[12+ j*2+i] = D2EDXDY(ixp[i], iyp[j]) * dx * dy;
	}
      }
      cl = &C(ix,iy,0,0);
      for (i=0;i<16;i++) {
	cl[i] = 0.0;
	for (j=0;j<16;j++) {
	  cl[i] += e2c[i][j] * f[j];
	}
      }
    }
  }
}
  
void DD_CMAP_energy_force(DIHEDRAL_DATA *dihed_data, 
			  ATOM_DATA *ad,
			  double *energy)
{
  CMAP *cmap;
  CMAP_TYPE *cmap_type;
  int atom1, atom2, atom3, atom4, n;
  int i,ii,jj;
  int iphi, ipsi;
  double phi, rphi, dphi, phi_ii, phi_ii_1;
  double psi, rpsi, dpsi, psi_jj, psi_jj_1;
  double e, dedx, dedy;
  double x12,y12,z12,x23,y23,z23,x34,y34,z34,nx23,ny23,nz23,len_23;
  double dx,dy,dz,gx,gy,gz,dd,gg,dg,len_dg;
  double ddx,ddy,ddz,dgx,dgy,dgz;
  double F_phi, Fx1_phi,Fy1_phi,Fz1_phi,Fx4_phi,Fy4_phi,Fz4_phi;
  double Fx2_d_phi,Fy2_d_phi,Fz2_d_phi,Fx2_g_phi,Fy2_g_phi,Fz2_g_phi;
  double F_psi, Fx1_psi,Fy1_psi,Fz1_psi,Fx4_psi,Fy4_psi,Fz4_psi;
  double Fx2_d_psi,Fy2_d_psi,Fz2_d_psi,Fx2_g_psi,Fy2_g_psi,Fz2_g_psi;
  VEC *x, *f;

  x = ad->x;
  f = ad->f;

  /*
  {
    void DD_CMAP_test(DIHEDRAL_DATA *);
    DD_CMAP_test(dihed_data);
  }
  */

#ifdef MPI_SDMD /* */
  for (i=dihed_data->head_cmap; i>=0; i=dihed_data->cmap[i].next) {
#else
  for (i=0; i < dihed_data->n_cmap; i++) {
#endif
    cmap = &(dihed_data->cmap[i]);
    cmap_type = &(dihed_data->cmap_type[cmap->type]);

    /* debug 
    {
      extern LINKED_CELL *_lc;
      int icell;

      jj = 0;
      for (ii=0;ii<8;ii++) {
	icell = _lc->atom_cell[dihed_data->cmap[i].atom[ii]];
	if (!(_lc->cell[icell].req & CELL_REQ_DIHED)) {
	  jj = 1;
	}
      }
      if (jj) {
	for (ii=0;ii<8;ii++) {
	  icell = _lc->atom_cell[dihed_data->cmap[i].atom[ii]];
	  printf("%d %d %d %d\n", ii, dihed_data->cmap[i].atom[ii], icell, _lc->cell[icell].req & CELL_REQ_DIHED);
	}
      }
    }
     end of debug part */
    

    /* phi */
    atom1 = cmap->atom[0];
    atom2 = cmap->atom[1];
    atom3 = cmap->atom[2];
    atom4 = cmap->atom[3];
    x12 = x[atom2].x - x[atom1].x;
    y12 = x[atom2].y - x[atom1].y;
    z12 = x[atom2].z - x[atom1].z;
    x23 = x[atom3].x - x[atom2].x;
    y23 = x[atom3].y - x[atom2].y;
    z23 = x[atom3].z - x[atom2].z;
    x34 = x[atom4].x - x[atom3].x;
    y34 = x[atom4].y - x[atom3].y;
    z34 = x[atom4].z - x[atom3].z;
    dx  = y12*z23 - z12*y23;
    dy  = z12*x23 - x12*z23;
    dz  = x12*y23 - y12*x23;
    gx  = y23*z34 - z23*y34;
    gy  = z23*x34 - x23*z34;
    gz  = x23*y34 - y23*x34;

    len_23 = sqrt(x23*x23 + y23*y23 + z23*z23);
    nx23 = x23 / len_23;
    ny23 = y23 / len_23;
    nz23 = z23 / len_23;
    dd  = dx*dx + dy*dy + dz*dz;
    gg  = gx*gx + gy*gy + gz*gz;
    dg  = dx*gx + dy*gy + dz*gz;
    len_dg = sqrt(dd) * sqrt(gg);

    phi = atan2((nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)
		 +nz23*(dx*gy-dy*gx))/len_dg,  dg/len_dg);
    
    while (phi >  M_PI) phi -= 2.0*M_PI;
    while (phi < -M_PI) phi += 2.0*M_PI;

    ddx = (dy*nz23 - dz*ny23)/dd;
    ddy = (dz*nx23 - dx*nz23)/dd;
    ddz = (dx*ny23 - dy*nx23)/dd;
    
    dgx = -(gy*nz23 - gz*ny23)/gg;
    dgy = -(gz*nx23 - gx*nz23)/gg;
    dgz = -(gx*ny23 - gy*nx23)/gg;
    
    Fx1_phi = (ddy * z23 - ddz * y23);
    Fy1_phi = (ddz * x23 - ddx * z23);
    Fz1_phi = (ddx * y23 - ddy * x23);
    Fx4_phi = (dgy * z23 - dgz * y23);
    Fy4_phi = (dgz * x23 - dgx * z23);
    Fz4_phi = (dgx * y23 - dgy * x23);
    Fx2_d_phi = (ddy * z12 - ddz * y12);
    Fy2_d_phi = (ddz * x12 - ddx * z12);
    Fz2_d_phi = (ddx * y12 - ddy * x12);
    Fx2_g_phi = (dgy * z34 - dgz * y34);
    Fy2_g_phi = (dgz * x34 - dgx * z34);
    Fz2_g_phi = (dgx * y34 - dgy * x34);

    /* psi */
    atom1 = cmap->atom[4];
    atom2 = cmap->atom[5];
    atom3 = cmap->atom[6];
    atom4 = cmap->atom[7];
    x12 = x[atom2].x - x[atom1].x;
    y12 = x[atom2].y - x[atom1].y;
    z12 = x[atom2].z - x[atom1].z;
    x23 = x[atom3].x - x[atom2].x;
    y23 = x[atom3].y - x[atom2].y;
    z23 = x[atom3].z - x[atom2].z;
    x34 = x[atom4].x - x[atom3].x;
    y34 = x[atom4].y - x[atom3].y;
    z34 = x[atom4].z - x[atom3].z;
    dx  = y12*z23 - z12*y23;
    dy  = z12*x23 - x12*z23;
    dz  = x12*y23 - y12*x23;
    gx  = y23*z34 - z23*y34;
    gy  = z23*x34 - x23*z34;
    gz  = x23*y34 - y23*x34;

    len_23 = sqrt(x23*x23 + y23*y23 + z23*z23);
    nx23 = x23 / len_23;
    ny23 = y23 / len_23;
    nz23 = z23 / len_23;
    dd  = dx*dx + dy*dy + dz*dz;
    gg  = gx*gx + gy*gy + gz*gz;
    dg  = dx*gx + dy*gy + dz*gz;
    len_dg = sqrt(dd) * sqrt(gg);

    psi = atan2((nx23*(dy*gz-dz*gy)+ny23*(dz*gx-dx*gz)
		 +nz23*(dx*gy-dy*gx))/len_dg,  dg/len_dg);
    
    while (psi >  M_PI) psi -= 2.0*M_PI;
    while (psi < -M_PI) psi += 2.0*M_PI;

    ddx = (dy*nz23 - dz*ny23)/dd;
    ddy = (dz*nx23 - dx*nz23)/dd;
    ddz = (dx*ny23 - dy*nx23)/dd;
    
    dgx = -(gy*nz23 - gz*ny23)/gg;
    dgy = -(gz*nx23 - gx*nz23)/gg;
    dgz = -(gx*ny23 - gy*nx23)/gg;
    
    Fx1_psi = (ddy * z23 - ddz * y23);
    Fy1_psi = (ddz * x23 - ddx * z23);
    Fz1_psi = (ddx * y23 - ddy * x23);
    Fx4_psi = (dgy * z23 - dgz * y23);
    Fy4_psi = (dgz * x23 - dgx * z23);
    Fz4_psi = (dgx * y23 - dgy * x23);
    Fx2_d_psi = (ddy * z12 - ddz * y12);
    Fy2_d_psi = (ddz * x12 - ddx * z12);
    Fz2_d_psi = (ddx * y12 - ddy * x12);
    Fx2_g_psi = (dgy * z34 - dgz * y34);
    Fy2_g_psi = (dgz * x34 - dgx * z34);
    Fz2_g_psi = (dgx * y34 - dgy * x34);
    
    n = cmap_type->ndiv;
    dphi = dpsi = 2.0*M_PI/n;

    phi += M_PI;
    iphi = (int)(phi / dphi);
    rphi = (phi - iphi*dphi)/dphi;
    while (iphi >= n) iphi -= n;
    while (iphi <  0) iphi += n;

    psi += M_PI;
    ipsi = (int)(psi / dpsi);
    rpsi = (psi - ipsi*dpsi)/dpsi;
    while (ipsi >= n) ipsi -= n;
    while (ipsi <  0) ipsi += n;

    e = dedx = dedy = 0.0;
    phi_ii = 1.0;
    phi_ii_1 = 0.0;
    for (ii=0;ii<4;ii++) {
      psi_jj = 1.0;
      psi_jj_1 = 0.0;
      for (jj=0;jj<4;jj++) {
	e += cmap_type->c[iphi*n+ipsi][ii][jj] * phi_ii * psi_jj;

	if (ii>0) {
	  dedx += cmap_type->c[iphi*n+ipsi][ii][jj] * ii * phi_ii_1 * psi_jj;
	}
	if (jj>0) {
	  dedy += cmap_type->c[iphi*n+ipsi][ii][jj] * jj * phi_ii * psi_jj_1;
	}
	psi_jj_1  = psi_jj;
	psi_jj   *= rpsi;
      }
      phi_ii_1  = phi_ii;
      phi_ii   *= rphi;
    }
    *energy += e;
    dedx /= dphi;
    dedy /= dpsi;
    F_phi = -dedx;
    F_psi = -dedy;
    
    /* phi */
    atom1 = cmap->atom[0];
    atom2 = cmap->atom[1];
    atom3 = cmap->atom[2];
    atom4 = cmap->atom[3];
    f[atom1].x += F_phi * Fx1_phi;
    f[atom1].y += F_phi * Fy1_phi;
    f[atom1].z += F_phi * Fz1_phi;
    f[atom2].x += F_phi * (-Fx2_d_phi + Fx2_g_phi - Fx1_phi);
    f[atom2].y += F_phi * (-Fy2_d_phi + Fy2_g_phi - Fy1_phi);
    f[atom2].z += F_phi * (-Fz2_d_phi + Fz2_g_phi - Fz1_phi);
    f[atom3].x += F_phi * ( Fx2_d_phi - Fx2_g_phi - Fx4_phi);
    f[atom3].y += F_phi * ( Fy2_d_phi - Fy2_g_phi - Fy4_phi);
    f[atom3].z += F_phi * ( Fz2_d_phi - Fz2_g_phi - Fz4_phi);
    f[atom4].x += F_phi * Fx4_phi;
    f[atom4].y += F_phi * Fy4_phi;
    f[atom4].z += F_phi * Fz4_phi;

    /* psi */
    atom1 = cmap->atom[4];
    atom2 = cmap->atom[5];
    atom3 = cmap->atom[6];
    atom4 = cmap->atom[7];
    f[atom1].x += F_psi * Fx1_psi;
    f[atom1].y += F_psi * Fy1_psi;
    f[atom1].z += F_psi * Fz1_psi;
    f[atom2].x += F_psi * (-Fx2_d_psi + Fx2_g_psi - Fx1_psi);
    f[atom2].y += F_psi * (-Fy2_d_psi + Fy2_g_psi - Fy1_psi);
    f[atom2].z += F_psi * (-Fz2_d_psi + Fz2_g_psi - Fz1_psi);
    f[atom3].x += F_psi * ( Fx2_d_psi - Fx2_g_psi - Fx4_psi);
    f[atom3].y += F_psi * ( Fy2_d_psi - Fy2_g_psi - Fy4_psi);
    f[atom3].z += F_psi * ( Fz2_d_psi - Fz2_g_psi - Fz4_psi);
    f[atom4].x += F_psi * Fx4_psi;
    f[atom4].y += F_psi * Fy4_psi;
    f[atom4].z += F_psi * Fz4_psi;
  }
}
   

/* for debug */
void DD_CMAP_eval2d(double phi, double psi, int n, double dx, 
		    double (*c)[4][4],
		    double *e, double *dedx, double *dedy)
{
  double dphi, dpsi, rphi, rpsi;
  double phi_ii, phi_ii_1, psi_jj, psi_jj_1;
  int iphi, ipsi;
  int ii, jj;

  dphi = dpsi = dx;

  while (phi >  M_PI) phi -= 2.0*M_PI;
  while (phi < -M_PI) phi += 2.0*M_PI;
  
  phi += M_PI;
  iphi = (int)(phi / dphi);
  rphi = (phi - iphi*dphi)/dphi;
  while (iphi >= n) iphi -= n;
  while (iphi <  0) iphi += n;

  while (psi >  M_PI) psi -= 2.0*M_PI;
  while (psi < -M_PI) psi += 2.0*M_PI;

  psi += M_PI;
  ipsi = (int)(psi / dpsi);
  rpsi = (psi - ipsi*dpsi)/dpsi;
  while (ipsi >= n) ipsi -= n;
  while (ipsi <  0) ipsi += n;

  *e = *dedx = *dedy = 0.0;
  phi_ii = 1.0;
  phi_ii_1 = 0.0;
  for (ii=0;ii<4;ii++) {
    psi_jj = 1.0;
    psi_jj_1 = 0.0;
    for (jj=0;jj<4;jj++) {
      *e += c[iphi*n+ipsi][ii][jj] * phi_ii * psi_jj;
      
      if (ii>0) {
	*dedx += c[iphi*n+ipsi][ii][jj] * ii * phi_ii_1 * psi_jj;
      }
      if (jj>0) {
	*dedy += c[iphi*n+ipsi][ii][jj] * jj * phi_ii * psi_jj_1;
      }
      psi_jj_1  = psi_jj;
      psi_jj   *= rpsi;
    }
    phi_ii_1  = phi_ii;
    phi_ii   *= rphi;
  }
  *dedx /= dphi;
  *dedy /= dpsi;
}


void DD_CMAP_test(DIHEDRAL_DATA *dd)
{
  double dx, x, y;
  double delta, valx1, valx2, valy1, valy2, val, dvaldx, dvaldy;
  double maxx, maxy, diffx, diffy;
  double maxx_phi, maxx_psi;
  double maxy_phi, maxy_psi;
  int i, j, n, ic;
  CMAP_TYPE *type;
  double (*c)[4][4];

  delta = 0.00001;
  for (ic = 0; ic < dd->n_cmap_type; ic++) {
    type = &(dd->cmap_type[ic]);
    n = type->ndiv;
    dx = 2.0*M_PI/n;
    c = type->c;
    
    maxx = maxy = 0.0;
    for (i=0;i<360;i++) {
      x = i*2*M_PI/360;
      for (j=0;j<360;j++) {
	y = j*2*M_PI/360;
	DD_CMAP_eval2d(x+0.5*delta,y,n,dx,c, &valx1, &dvaldx, &dvaldy);
	DD_CMAP_eval2d(x-0.5*delta,y,n,dx,c, &valx2, &dvaldx, &dvaldy);

	DD_CMAP_eval2d(x,y+0.5*delta,n,dx,c, &valy1, &dvaldx, &dvaldy);
	DD_CMAP_eval2d(x,y-0.5*delta,n,dx,c, &valy2, &dvaldx, &dvaldy);

	DD_CMAP_eval2d(x,y,n,dx, c, &val, &dvaldx, &dvaldy);

	diffx = fabs((valx1-valx2)/delta-dvaldx);
	if (maxx < diffx) {
	  maxx = diffx;
	  maxx_phi = x;
	  maxx_psi = y;
	}

	diffy = fabs((valy1-valy2)/delta-dvaldy);
	if (maxy < diffy) {
	  maxy = diffy;
	  maxy_phi = x;
	  maxy_psi = y;
	}
      }
    }
    lprintf("%d %e %f %f\n",ic, maxx, maxx_phi, maxx_psi);
    lprintf("%d %e %f %f\n",ic, maxy, maxy_phi, maxy_psi);
  }
  marble_exit(1);
}

