/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifndef _CTRL_H_INCLUDED
#define _CTRL_H_INCLUDED

#define FILENAME_LEN 1000
#define N_EXPAIR     10

#ifdef CTRL_C
SEL_DATA _sel_TF[] = {
  "true",  1,
  "false", 0,
  "on",    1,
  "off",   0,
   NULL,   0
};

#define RESTART_X        (0x1)      /*  00000000 00000001 */
#define RESTART_VEL      (0x2)      /*  00000000 00000010 */
#define RESTART_BOX      (0x4)      /*  00000000 00000100 */
#define RESTART_PT       (0x8)      /*  00000000 00001000 */
#define RESTART_LAMBDA   (0x10)     /*  00000000 00010000 */
#define RESTART_RMOL     (0x20)     /*  00000000 00100000 */
#define RESTART_ALL      (0xff)     /*  00000000 11111111 */


SEL_DATA _sel_restart[] = {
  "velocity",   RESTART_VEL,
  "PT_control", RESTART_PT,
  "PT",         RESTART_PT,
  "pt_control", RESTART_PT,
  "box",        RESTART_BOX,
  "VB",         RESTART_VEL | RESTART_BOX,
  "all",        RESTART_ALL,
  "on",         RESTART_VEL | RESTART_BOX | RESTART_PT,
  "off",        0,
   NULL,        0
};

#endif

struct sec_input {
  char mdat_file[FILENAME_LEN];
  char crd_file[FILENAME_LEN];
  unsigned int crd_type;
  char brd_file[FILENAME_LEN];
  unsigned int brd_type;
  int restart;
  
  int flag;
};

struct sec_output {
  char crd_file[FILENAME_LEN];
  unsigned int crd_type, crd_type_user_specified;
  char brd_file[FILENAME_LEN];
  unsigned int brd_type, brd_type_user_specified;
  char pdb_file[FILENAME_LEN];
  int pdb_group_no, pdb_centering_solute;
  char force_file[FILENAME_LEN];
  int wrap_periodic_boundary;
  int check_pdb;
  int check_bonds_angles;
  int flag;
};

typedef enum {
  GT_ATOM_NO, GT_RES_NO, GT_ATOM_NAME, GT_ATOM_NAME_RES_NO,
  GT_RES_NAME, GT_BACKBONE, GT_HEAVY, GT_SOLUTE_HEAVY
} group_type_t;

#ifdef CTRL_C
SEL_DATA _sel_gp_mode[] = {
  "or",      GM_OR,
  "add",     GM_OR,
  "and",     GM_AND,
  "or_not",  GM_OR_NOT,
  "and_not", GM_AND_NOT,
  "exclude", GM_AND_NOT,
  NULL,      GM_UNKNOWN
};

SEL_DATA _sel_gp_atype[] = {
  "backbone",            GT_BACKBONE,
  "heavy_atom",          GT_HEAVY,
  "solute_heavy_atom",   GT_SOLUTE_HEAVY,
  NULL,      -1
};
#endif

struct sec_group {
  group_type_t type;
  int mode;
  int group_no;
  int no[2];
  char resno[2][100];
  char chain;
  char name[100];
  struct sec_group *next;
  int flag;
};

struct sec_initialize {
  double cur_time;
  int cur_time_flag;
  long seed;
  double temperature;
  int solute_residue, solute_mol;
  int origin;
  int wrap_periodic_boundary;
  int remove_momentum;
  
  double a[3], b[3], c[3], shift[3];
  int a_flag, b_flag, c_flag, shift_flag;
  int bc_reset;
  
  double mass;
  int mass_g_no;
  
  double scale_bond_k, scale_angle_k, scale_impr_k;
  double overwrite_bond_k, overwrite_angle_k, overwrite_impr_k;

  int flag;
};

struct sec_nonbond {
  int list_method, vdw_method, elec_method;
  double cutoff, list_margin, smooth_margin, group_margin, cell_margin;
  double scee, scnb, diel;
  int cell_div;
  int vir_cor, update_step;
  int n_expair, expair[N_EXPAIR][2];
  int flag;
};

#ifdef CTRL_C
SEL_DATA _sel_nl_method[] = {
  "atom_based",    NL_ATOM_BASED,
  "residue_based", NL_RESIDUE_BASED,
  "linked_cell",   NL_LINKED_CELL,
  "FMM",           NL_FMM,
  NULL,            NL_UNKNOWN
};

SEL_DATA _sel_ne_method[] = {
  "potential_switch",           NE_PSW,
  "switch",                     NE_PSW,
  "shifted_force",              NE_FSH,
  "residue_potential_switch",   NE_RPSW,
  "ewald",                      NE_EWALD,
  "FMM",                        NE_FMM,
  NULL,                         NE_UNKNOWN
};

SEL_DATA _sel_nv_method[] = {
  "potential_switch", NV_PSW,
  "force_switch",     NV_FSW,
  "truncate",         NV_TRN,
  NULL,               NV_UNKNOWN
};

SEL_DATA _sel_cell_div[] = {
  "atom",    0,
  "group",   1,
  NULL,      1
};

#endif

typedef enum {
  rs_unknown, 
  rs_distance_harmonic, 
  rs_group_distance_harmonic, 
  rs_position_harmonic, 
  rs_group_position_harmonic, 
  rs_molecule_position_harmonic, 
  rs_torsion_harmonic, 
  rs_nmr_distance, 
  rs_nmr_dihedral, 
  rs_nmr_plane, 
  rs_steered_md,
  rs_rmsd,
  rs_rmsd2
} restraint_method_t;

typedef enum {
  smd_unknown, 
  smd_const_force,
  smd_const_vel
} smd_method_t;

typedef int smd_center_t;

struct sec_restraint {
  restraint_method_t method;
  
  double k, k1, r0;
  double xx[3];
  char fname[FILENAME_LEN];
  char atom[4][10];
  int group[2], group_no, gradual_change_step;
  STR_LIST *group_str, *group2_str;
  int virial_flag;
  
  int flag;

  /* yss-add */
  int x_flag, y_flag, z_flag;
  int potential_type;
  int mole_residue;
  double delta;
  /* yse */

/*YAMANE_ADDED*/
/* nmr dis*/
  double scl_dis, scl_dis1, Ceil;
  double sqof, sqcs, sqex;
  double soex, asym, rswi;
  double viol_dis;
  int num_viol_dis, n_output_viol; 
  char fname_dis[FILENAME_LEN];
/* nmr dih*/
  char fname_dih[FILENAME_LEN];
  int num_viol_dih;
  double scl_dih, scl_dih1;
  double viol_dih;
/* nmr plane*/
  char fname_pln[FILENAME_LEN];
  double scl_pln;
  double scl_pln1;
/*E_YAMANE_ADDED*/

/* steered MD */
  smd_method_t smd_method;
  int smd_center, smd_weight_flag;
  double smd_dir[3], smd_pos[3];
  double smd_vel, smd_ang_vel;
  double smd_force, smd_torque;
  double smd_init_trans, smd_init_rot;
  char smd_test_file[FILENAME_LEN];
  double smd_test_timestep;
  int smd_test_step, smd_test_interval;

/* RMSD */
  char fname2[FILENAME_LEN];
  int gradual_change_step_r0;
  double r1;
  int rmsd_best_fit, rmsd_weight_flag, rmsd_k_per_atom_flag;
  int pdb_mode, pdb2_mode, check_pdb_group;
  STR_LIST *pdb_group_str, *pdb_group_str2;
  WEIGHT_LIST *rmsd_weight_list;
  
  struct sec_restraint *next;
};

#ifdef CTRL_C
SEL_DATA _sel_rs_method[] =
{
  "distance_harmonic",       rs_distance_harmonic,
  "group_distance_harmonic", rs_group_distance_harmonic,
  "position_harmonic",       rs_position_harmonic,
/* yss-add */ 
  "group_position_harmonic", rs_group_position_harmonic,
  "molecule_position_harmonic", rs_molecule_position_harmonic,
  "mole_position_harmonic", rs_molecule_position_harmonic,
/* yss-add (Mar09/2004) */ 
  "torsion_harmonic", rs_torsion_harmonic, 
/* yse */
  "distance", rs_distance_harmonic,
  "group_distance", rs_group_distance_harmonic,
  "position", rs_position_harmonic,
/* yss-add */ 
  "group_position", rs_group_position_harmonic,
  "molecule_position", rs_molecule_position_harmonic,
  "mole_position", rs_molecule_position_harmonic,
/* yss-add (Mar09/2004) */ 
  "torsion", rs_torsion_harmonic, 
/* yse */
/* YAMANE_ADDED */
  "nmr_distance", rs_nmr_distance,
  "nmr_dihedral", rs_nmr_dihedral,
  "nmr_plane", rs_nmr_plane,
/* End of YAMANE_ADDED */
  "steered_md", rs_steered_md, 
  "rmsd",  rs_rmsd,
  "rmsd2", rs_rmsd2,
  "rmsd_diff", rs_rmsd2,
  NULL, rs_unknown
};

SEL_DATA _sel_potential_type[] = {
  "full",        0,
  "normal",      0,
  "repulsive",   1,
  "lower",       1,
  "attractive",  2,
  "upper",       2,
  NULL,          -1
};


SEL_DATA _sel_smd_method[] = {
  "constant_force",    smd_const_force,
  "constant_velocity", smd_const_vel,
  NULL,      smd_unknown
};

SEL_DATA _sel_smd_center[] = {
  "on",    1, 
  "off",   0,
  "weighted", 1,
  "uniform", 2,
  NULL,      0
};

SEL_DATA _sel_weight_flag[] = {
  "uniform",           0,
  "mass_weighted",     1,
  "mass",              1,
  NULL,                -1,
};

#else
extern SEL_DATA _sel_rs_method[];
#endif

#define MAX_N_RIGID_RES_TYPE 10

struct sec_constraint {
  RATTLE_TYPE rattle;
  double tolerance;
  int rigid_body_flag;
  int check_rigid_body;
  /* char rigid_res_name[MAX_N_RIGID_RES_TYPE][5]; */
  char rigid_body_fname[FILENAME_LEN];
  int fixed_group_flag, fixed_group_no;
  STR_LIST *fixed_group_str;
  
  int flag;
};

#ifdef CTRL_C
SEL_DATA _sel_cs_rattle[] =
{
  "ALL",     RATTLE_ALL,
  "HYDR",    RATTLE_HYDR,
  "WAT",     RATTLE_WAT,
  "all",     RATTLE_ALL,
  "hydrogen",RATTLE_HYDR,
  "water",   RATTLE_WAT,
  NULL,  RATTLE_NONE
};
#else
extern SEL_DATA _sel_cs_method[];
extern SEL_DATA _sel_cs_rattle_type[];
#endif

typedef enum { 
  pt_unknown, 
  pt_extended_system, 
  pt_weak_coupling, 
  pt_constraint,
  pt_rescaling
} PT_control_method_t;

typedef enum {
  en_unknown, 
  en_NVT, 
  en_NPT, 
  en_NPTflex, 
  en_NPTxyz, 
  en_NPTisoxy, 
  en_NPATz, 
  en_NVE
} PT_control_ensemble_t;

struct sec_PT_control {
  PT_control_method_t method;
  PT_control_ensemble_t ensemble;
  int coupling;
  int crd_data;
  double temperature, period_T, tau_T, delta_T;
  int n_chain_T;
  double pressure, period_P, period_PT, tau_P, compress;
  int n_chain_PT;
  int init_pt_control;
  int gradual_change_step;
  double temperature1;
  int flag;
};

#ifdef CTRL_C
SEL_DATA _sel_pt_method[] =
{
  "extended_system", pt_extended_system,
  "weak_coupling", pt_weak_coupling,
  "constraint", pt_constraint,
  "rescaling", pt_rescaling,
  NULL, pt_unknown
};

SEL_DATA _sel_pt_ensemble[] =
{
  "NVT",           en_NVT,
  "NPT",           en_NPT,
  "NPT_isotropic", en_NPT,
  "NPT_cubic",     en_NPT,
  "NPT_flexible",  en_NPTflex,
  "NPT_triclinic", en_NPTflex,
  "NPT_orthogonal",en_NPTxyz,
  "NPT_orthorhombic",en_NPTxyz,
  "NPT_xyz",       en_NPTxyz,
  "NPT_iso_xy",    en_NPTisoxy,
  "NPT_tetragonal",en_NPTisoxy,
  "NPAT_z",        en_NPATz,
  "NPAT",          en_NPATz,
  "NVE",           en_NVE,
  NULL,            en_unknown
};

SEL_DATA _sel_pt_coupling[] =
{
  "whole", 0,
  "solute/solvent", 1,
  "whole+pressure", 0,
  "solute/solvent/pressure", 1,
  "solute+pressure/solvent", 2,
  "whole/pressure", 3,
  NULL, -1
};
#else
extern SEL_DATA _sel_pt_method[];
extern SEL_DATA _sel_pt_ensemble[];
extern SEL_DATA _sel_pt_coupling[];
#endif

typedef enum { ew_unknown, ew_normal, ew_PME } ewald_method_t;

struct sec_ewald {
  ewald_method_t method;
  double tolerance;
  double beta;
  int tol_beta_flag;
  int grid[3];
  double d_grid;
  int max_m, n_spline;
  int erfc_resolution;
  double diel_sur;
#ifdef MPI_SDMD
  int pme_n_pe;
#endif  
  int flag;
};

#ifdef CTRL_C
SEL_DATA _sel_ew_method[] =
{
  "normal", ew_normal,
  "PME",    ew_PME,
  NULL, ew_unknown
};

#else
extern SEL_DATA _sel_ew_method[];
#endif

struct sec_FMM {
  int n_multipole;
  double unit_cell_size;
  int neighbor_cell, update_step;
  int flag;
};

struct sec_min {
  int step, cg_step, print_step;
  double init_step_len, largest_step_len, grad;
  double cg_init_step, cg_largest_step, cg_line_min_tol;
  int flag;
};

#define MAX_ATOM_ENE_GROUP  8

struct sec_leapfrog {
  char prop_out[FILENAME_LEN];
  int prop_out_step;
  char trj_out[FILENAME_LEN];
  int trj_out_step;
  int step, print_step;
  double time_step;
  char inter_crd[FILENAME_LEN], inter_crd_type[10];
  char inter_brd[FILENAME_LEN], inter_brd_type[10];

  int  targeted_MD_flag;
  char targeted_MD_fname[FILENAME_LEN];
  
  int heating_flag;
  double heating_temp[2];
  int flag;
};

struct sec_verlet {
  char prop_out[FILENAME_LEN];
  int prop_out_step, prop_out_flag;
  char trj_out[FILENAME_LEN];
  int trj_out_step, trj_out_flag, trj_out_type, trj_res_no, gzip_trj;
  int step, print_step;
  double time_step;
  int n_atom_ene_group, atom_ene_group[MAX_ATOM_ENE_GROUP];
  char atom_ene_file[FILENAME_LEN];
  
  char last_crd[FILENAME_LEN], last_crd_type[10];
  char last_brd[FILENAME_LEN], last_brd_type[10];
  int flag;
  struct sec_verlet *next;
};

struct sec_rRESPA {
  /* not set */
  char prop_out[FILENAME_LEN];
  int prop_out_step, prop_out_flag;
  char trj_out[FILENAME_LEN];
  int trj_out_step, trj_out_flag, trj_out_type, trj_res_no, gzip_trj;
  int step0,step1,step2, print_step;
  double time_step;
  int n_atom_ene_group, atom_ene_group[MAX_ATOM_ENE_GROUP];
  char atom_ene_file[FILENAME_LEN];
  
  char last_crd[FILENAME_LEN], last_crd_type[10];
  char last_brd[FILENAME_LEN], last_brd_type[10];
  
  int flag;
  struct sec_rRESPA *next;
};

#ifdef CTRL_C
SEL_DATA _sel_format[] = {
  "bin",   1,
  "binary",1,
  "text",  0,
  "ascii", 0,
  "dcd",   2,
  "DCD",   2,
   NULL,   -1
};

SEL_DATA _sel_prop_out[] = {
  "clean",       PROP_CLEAN,
  "time",        PROP_TIME,
  "temperature", PROP_TEMPERATURE,
  "total_energy",PROP_TOTAL_ENE,
  "potential",   PROP_POTENTIAL,
  "kinetic",     PROP_KINETIC,
  "energy_detail", PROP_ENE_DETAIL,
  "extra_potential", PROP_EP_ENE,
  "extended_system", PROP_EX_SYS,
  "density",         PROP_DENSITY,
  "PV_isotropic",    PROP_PV_ISOTROPIC,
  "PV_diagonal",     PROP_PV_DIAGONAL,
  "PV_offdiagonal",  PROP_PV_OFFDIAG,
  "atom_energy",     PROP_ATOM_ENE,
  NULL,  0
};

SEL_DATA _sel_remove_momentum[] = {
  "default",     -1,
  "off",         0,
  "on",          1,
  "all",         1,
  "all_rot",     2,
  "all_rotation",2,
  "solute",      3,
  "solute_tr",   3,
  "solute_translation",   3,
  "solute_rot",  4,
  "solute_rotation",  4,
  NULL,  0
};
#endif

struct sec_md {
  char prop_out[FILENAME_LEN];
  int  prop_out_step, prop_out_flag;
  char trj_out[FILENAME_LEN];
  int  trj_out_step, trj_out_format, trj_out_type, trj_res_no, gzip_trj;
  int  trj_wrap_mol, trj_xst_flag, trj_amber_boundary;
  int  step,step0,step1,step2,step3, print_step;
  double time_step;
  double init_velocity;
  int  remove_momentum, remove_momentum_step;
  int test_force_iatom;
  double test_force_dx, test_force_dy, test_force_dz;
  int test_virial;
  
  int flag;
};

struct sec_Hybrid_MC {
  /* not set */
  int flag;
};


#ifndef CTRL_C
extern
#endif
struct s_control {
  struct sec_input       input;
  struct sec_output      output;
  struct sec_initialize  initialize;
  struct sec_group       group;
  struct sec_nonbond     nonbond;
  struct sec_restraint   restraint;
  struct sec_constraint  constraint;
  struct sec_PT_control  PT_control;
  struct sec_ewald       ewald;
  struct sec_FMM         fmm;

  struct sec_min         min;
  struct sec_md          md;
  struct sec_leapfrog    leapfrog;
  struct sec_verlet      verlet;
  struct sec_Hybrid_MC   Hybrid_MC;
  struct sec_rRESPA      rRESPA;

} _ctrl;

#define CFILE_MAX_STACK 100
#define CFILE_DEF_WORD_LEN 1000
#define CFILE_BUF_LEN 1000

struct s_cfile_stack {
  FILE *fp;
  int depth;
  int if_block_depth;
  char fname[FILENAME_LEN];
  struct s_cfile_stack *next;
};

struct s_cfile_def_word {
  char *word, *val;
  struct s_cfile_def_word *next;
};

typedef struct s_cfile_fp {
  FILE *cur_fp;
  int  current_depth;
  int if_block_depth;
  struct s_cfile_stack *stack;
  struct s_cfile_def_word *def_word;
  int master;
} CFILE;


void initialize_ctrl_section();
void read_ctrl_file(char *fname);
void exec_ctrl_section();

void init_sec_input();
void init_sec_output();
void init_sec_group();
void init_sec_initialize();
void init_sec_restraint();
void init_sec_constraint();
void init_restraint(struct sec_restraint *current);
void init_sec_PT_control();
void init_sec_nonbond();
void init_sec_ewald();
void init_sec_fmm();
void init_sec_min();
void init_sec_leapfrog();
void init_sec_verlet();
void init_sec_rRESPA();
void init_sec_md();
void init_sec_pert();
void init_sec_g_ensemble();


void ctrl_input(CFILE *fp, char *);
void ctrl_output(CFILE *fp, char *);
void ctrl_group(CFILE *fp, char *);
void ctrl_initialize(CFILE *fp, char *);
void ctrl_nonbond(CFILE *fp, char *);
void ctrl_restraint(CFILE *fp, char *);
void ctrl_constraint(CFILE *fp, char *);
void ctrl_PT_control(CFILE *fp, char *);
void ctrl_ewald(CFILE *fp, char *);
void ctrl_fmm(CFILE *fp, char *);
void ctrl_min(CFILE *fp, char *);
void ctrl_leapfrog(CFILE *fp, char *);
void ctrl_verlet(CFILE *fp, char *);
void ctrl_Hybrid_MC(CFILE *fp, char *);
void ctrl_rRESPA(CFILE *fp, char *);
void ctrl_md(CFILE *fp, char *);
void ctrl_pert(CFILE *fp, char *);
void ctrl_g_ensemble(CFILE *fp, char *);
void ctrl_stattrj(CFILE *fp, char *);

void exec_input1();
void exec_input2();
void exec_output();
void exec_group();
void exec_initialize();
void exec_nonbond();
void exec_restraint();
void exec_constraint();
void exec_PT_control();
void exec_ewald();
void exec_fmm();
void exec_min();
void exec_leapfrog();
void exec_verlet();
void exec_Hybrid_MC();
void exec_rRESPA();
void exec_md();
void exec_pert();
void exec_g_ensemble();

void ctrl_check_error(char *);
void scan_double_list(char *buf1, double **pdata, int *pn);
void scan_string_list(char *buf1, char ***pdata, int *pn);
void free_string_list(char **pdata, int n);
void fill_space(char *d, int n);

#if 0
char *lfgets(char *buf, int n, FILE *fp);
#endif

CFILE *ctrl_fopen(char *fname, char *mode);
void ctrl_fclose(CFILE *cfp);
int ctrl_feof(CFILE *cfp);
char *ctrl_fgets(char *buf, int n, CFILE *cfp);
char *ctrl_fgets_master(char *buf, int n, CFILE *cfp);
int substitute_word(char *buf, int n, char *word, char *val);

/* functions in main.c */
void marble_exit(int arg);
void opening(char *ctrl_file, char *progname);
void closing();


#ifdef CTRL_C
struct {
  char *name;
  void (*func)(CFILE *, char*);
} section_data[] = {
  "input",      ctrl_input,
  "output",     ctrl_output,
  "group",      ctrl_group,
  "initialize", ctrl_initialize,
  "init",       ctrl_initialize,
  "nonbond",    ctrl_nonbond,
  "restraint",  ctrl_restraint,
  "constraint", ctrl_constraint,
  "PT_control", ctrl_PT_control,
  "ewald",      ctrl_ewald,
  "FMM",        ctrl_fmm,
  "min",        ctrl_min,
  "md",         ctrl_md,
  /*
  "leapfrog",   ctrl_leapfrog,
  "verlet",     ctrl_verlet,
  "rRESPA",     ctrl_rRESPA,
  "perturbation",ctrl_pert,
  "g_ensemble",  ctrl_g_ensemble,
  */
#ifdef STATTRJ  
  "stattrj",     ctrl_stattrj,
#endif  
  /*
  "Hybrid_MC",  ctrl_Hybrid_MC,
  "perturbation,ctrl_pert,
*/
  NULL,         NULL
};
MD_SYSTEM _md_sys;
#else
extern MD_SYSTEM _md_sys;
#endif /* CTRL_C */

#endif /* _CTRL_H_INCLUDED */
