/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#define CTRL_C

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>

#include "md_system.h"
#include "min.h"
#ifdef MPI_SDMD
#include "sdmd.h"
#endif
#include "ctrl.h"

void scan_double_list(char *buf1, double **pdata, int *pn);

void initialize_ctrl_section()
{
  init_sec_input();
  init_sec_output();
  init_sec_initialize();
  init_sec_nonbond();
  init_sec_group();
  init_sec_restraint();
  init_sec_constraint();
  init_sec_PT_control();
  init_sec_ewald();
  init_sec_fmm();
  init_sec_min();
  /*
  init_sec_verlet();
  init_sec_rRESPA();
  */
  init_sec_md();

#ifdef STATTRJ
  init_sec_stattrj();
#endif
}

void read_ctrl_file(char *fname)
{
  CFILE *cfp;
  char buf[CFILE_BUF_LEN], section_name[CFILE_BUF_LEN], *ret;
  int i;

  lprintf("**  STEP 1.  Read Control File\n\n");
  
  if ((cfp = ctrl_fopen(fname, "r")) == NULL) {
    lprintf("ERROR: %s: No such control file\n", fname);
    marble_exit(1);
  };

  while ((ret=ctrl_fgets(buf, CFILE_BUF_LEN, cfp)) != NULL)
    if (sscanf(buf, " [%[^]]", section_name) == 1) break;

  while (!ctrl_feof(cfp)) {
    for (i=0;section_data[i].name!=NULL;i++) {
      if (strcmp(section_data[i].name, section_name) == 0) {
	section_data[i].func(cfp, section_name);
	break;
      }
    }
    if (section_data[i].name==NULL) {
      lprintf("ERROR: invalid section name [%s].\n", section_name);
      marble_exit(1);
    }
  }
  lprintf("\n");
  lprintf_bar();
  lprintf("\n");
}

void exec_ctrl_section()
{
  lprintf("**  STEP 2.  Setup\n\n");
  
  exec_input1();

  exec_group();
  exec_constraint();
  exec_input2();
  exec_initialize();
  exec_nonbond();
  exec_ewald();
  exec_restraint();
  exec_PT_control();

  lprintf("\n");
  lprintf_bar();
  lprintf("\n");

  lprintf("**  STEP 3.  Execution\n\n");

  lflush();

  exec_min();

  /*
    exec_rRESPA();
    exec_leapfrog();
    exec_verlet();
    exec_g_ensemble();
  */
  exec_md();
#ifdef STATTRJ
  exec_stattrj();
#endif
  exec_output();
}

/***************************************
 * section [input]                     *
 ***************************************/

void init_sec_input()
{
  _ctrl.input.flag = 0;

  _ctrl.input.mdat_file[0] = '\0';
  _ctrl.input.crd_file[0] = '\0';
  _ctrl.input.brd_file[0] = '\0';
  _ctrl.input.crd_type = TCT_X;
  /* _ctrl.input.crd_pt_control = -2; */
  _ctrl.input.restart = 0;
}

void ctrl_input(CFILE *cfp, char *next_section)
{
  char buf[CFILE_BUF_LEN], buf1[CFILE_BUF_LEN], buf2[CFILE_BUF_LEN];
  int ret, n, np, d;

  _ctrl.input.flag = 1;

  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    if (sscanf(buf," mdat_file = %s",  buf1) == 1) {
      strcpy(_ctrl.input.mdat_file, buf1); continue;
    }
    if ((ret = sscanf(buf," crd_file = %s %10s",buf1,buf2))==1 || ret==2) {
      strcpy(_ctrl.input.crd_file, buf1);
      if (ret == 1) {
	_ctrl.input.crd_type = 0;
      } else {
	_ctrl.input.crd_type = CRD_crdstr_to_flag(buf2);
      }
      continue;
    }
    if (sscanf(buf," restart = %[^\n]", buf1) == 1) {
      n=0;
      while (sscanf(&buf1[n],"%s%n",buf2,&np) == 1) {
	n+=np;
	d = get_id_sel_data(_sel_restart, buf2);
	if (d==0)
	  _ctrl.input.restart = 0;
	else
	  _ctrl.input.restart |= d;
      }
      continue;
    }
    if (sscanf(buf," restart_off = %[^\n]", buf1) == 1) {
      n=0;
      while (sscanf(&buf1[n],"%s%n",buf2,&np) == 1) {
	n+=np;
	_ctrl.input.restart &= ~get_id_sel_data(_sel_restart, buf2);
      }
      continue;
    }
    if ((ret = sscanf(buf," brd_file = %s %10s",buf1,buf2))==1 || ret==2) {
      strcpy(_ctrl.input.brd_file, buf1);
      if (ret == 1) {
	_ctrl.input.brd_type = 0;
      } else {
	_ctrl.input.brd_type = CRD_crdstr_to_flag(buf2);
      }
      continue;
    }
    
    if (sscanf(buf," [%[^]]", next_section) == 1) break;
    ctrl_check_error(buf);
  }
}

void usage_input()
{
  lprintf("[input]\n");
  lprintf("  mdat_file = mdat_file_name\n");
  lprintf("  crd_file  = crd_file_name [XVBEL]\n");
  lprintf("  brd_file  = brd_file_name [XVBEL]\n");
}

void exec_input1()
{
  MDAT_read_file(&_md_sys, _ctrl.input.mdat_file);
}

void exec_input2()
{
  int user_type;

  user_type = TCT_ALL;
  
  if (!(_ctrl.input.restart & RESTART_PT)) {
    user_type &= ~TCT_EXT;
  }

  if (!(_ctrl.input.restart & RESTART_RMOL)) {
    user_type &= ~TCT_RMOL;
  }
  
  if (_ctrl.input.crd_file[0]) {
    if (CRD_read_file(&_md_sys, _ctrl.input.crd_file,
		      user_type) != 0) {
      marble_exit(1);
    }
    
    if (_md_sys.read_crd_flag & TCT_BOUNDARY)
      _ctrl.output.crd_type |= TCT_BOUNDARY;
    
    if (_md_sys.read_crd_flag & TCT_V)
      _ctrl.output.crd_type |= TCT_V;

  } else if (_ctrl.input.brd_file[0]) {
    /*
    if (MD_SYSTEM_read_brd(&_md_sys, _ctrl.input.brd_file,
			   _ctrl.input.brd_type) != 0) {
      marble_exit(1);
    }
    */
    lprintf("brd file: Currently not supported.\n");
    marble_exit(1);
  } else {
    marble_exit(1);
  }

  if (!(_ctrl.input.restart & RESTART_PT))
    PTC_set_Ex_System_off(&_md_sys);

#ifdef MPI_SDMD  
  SDMD_setup0(&_md_sys);
#endif

}


/***************************************
 * section [output]                    *
 ***************************************/

void init_sec_output()
{
  _ctrl.output.flag = 0;

  _ctrl.output.crd_file[0] = '\0';
  _ctrl.output.crd_type = TCT_X;
  _ctrl.output.crd_type_user_specified = 0;
  _ctrl.output.brd_file[0] = '\0';
  _ctrl.output.brd_type = TCT_X;
  _ctrl.output.brd_type_user_specified = 0;
  _ctrl.output.pdb_file[0] = '\0';
  _ctrl.output.pdb_group_no = -1;
  _ctrl.output.wrap_periodic_boundary = 0;
  _ctrl.output.check_pdb = 0;
  _ctrl.output.check_bonds_angles = 0;
  _ctrl.output.pdb_centering_solute = 0;
  _ctrl.output.force_file[0] = '\0';
}

void ctrl_output(CFILE *cfp, char *next_section)
{
  struct sec_output *p;
  char buf[CFILE_BUF_LEN], buf1[CFILE_BUF_LEN], buf2[CFILE_BUF_LEN];
  int ret, d;

  _ctrl.output.flag = 1;

  p = &_ctrl.output;

  /* read ... */
  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    if ((ret = sscanf(buf," crd_file = %s %10s",buf1, buf2)) == 1 || ret == 2) {
      strcpy(p->crd_file, buf1);
      if (ret == 2) {
	p->crd_type_user_specified = CRD_crdstr_to_flag(buf2);
      }
      continue;
    }
    if ((ret = sscanf(buf," brd_file = %s %10s", buf1, buf2)) == 1 || ret == 2) {
      strcpy(p->brd_file, buf1);
      if (ret == 2) {
	p->brd_type_user_specified = CRD_crdstr_to_flag(buf2);
      }
      continue;
    }
    if (sscanf(buf," pdb_file = %s %d",  buf1, &d) == 2) {
      strcpy(p->pdb_file,buf1);
      p->pdb_group_no = d;
      continue;
    }
    if (sscanf(buf," pdb_file = %s",  buf1) == 1) {
      strcpy(p->pdb_file,buf1);
      continue;
    }
    if (sscanf(buf," pdb_centering_solute = %s",  buf1) == 1) {
      p->pdb_centering_solute = get_id_sel_data(_sel_TF, buf1);
      continue;
    }
    if (sscanf(buf," pdb_wrap_molecules = %s",  buf1) == 1) {
      if (strcmp(buf1,"center_of_mass") == 0) {
	_md_sys.pdb_wrap_mol_flag = 1;
      } else {
	_md_sys.pdb_wrap_mol_flag = 0;
      }
      continue;
    }

    if (sscanf(buf," pdb_large_number = %s",  buf1) == 1) {
      if (strcmp(buf1,"vmd") == 0) {
	_md_sys.atom.pdb_long_no_flag = 1;
      } else {
	_md_sys.atom.pdb_long_no_flag = 0;
      }
      continue;
    }

    if (sscanf(buf," force_file = %s",  buf1) == 1) {
      strcpy(p->force_file,buf1);
      continue;
    }
    if (sscanf(buf," check_pdb = %s",  buf1) == 1) {
      p->check_pdb = get_id_sel_data(_sel_TF, buf1);
      continue;
    }
    if (sscanf(buf," wrap_molecules = %s",  buf1) == 1) {
      p->wrap_periodic_boundary = get_id_sel_data(_sel_TF, buf1);
      continue;
    }
    if (sscanf(buf," check_bonds_angles = %s",  buf1) == 1) {
      p->check_bonds_angles = get_id_sel_data(_sel_TF, buf1);
      continue;
    }
    
    if (sscanf(buf," [%[^]]", next_section) == 1) break;

    ctrl_check_error(buf);
  }
}

void exec_output()
{
  struct sec_output *p;
  char buf[100];
  extern char *_safe_fopen_fname;

  p = &_ctrl.output;

  if (!p->flag) return;

#ifdef MPI
  if (!mpi.master) return;
#endif

  if (p->brd_type_user_specified) {
    p->brd_type = p->brd_type_user_specified;
  } else {
    p->brd_type = p->crd_type;
  }
  if (p->crd_type_user_specified) {
    p->crd_type = p->crd_type_user_specified;
  }

  lprintf("OUTPUT:\n");
  if (p->wrap_periodic_boundary) {
    MD_SYSTEM_wrap_molecules(&_md_sys, 1, 0);
    lprintf("  Wrapping Molecules by Periodic Boundary Condition\n");
  }
  
  if (p->crd_file[0]) {
    CRD_flag_to_crdstr(buf, p->crd_type);
    CRD_write_file(&_md_sys, p->crd_file, p->crd_type);
    lprintf("  CRD FILE:     %s (%s)\n", _safe_fopen_fname, buf);
  }
  if (p->brd_file[0]) {
    CRD_write_bin_file(&_md_sys, p->brd_file);
    lprintf("  BIN CRD FILE: %s\n", _safe_fopen_fname);
  }
  if (p->pdb_file[0]) {
    if (p->check_pdb) {
      ATOM_DATA_set_print_flag_pdb(&_md_sys.atom, p->check_pdb);
    }
    MD_SYSTEM_write_pdb_file(&_md_sys, p->pdb_file, p->pdb_group_no, p->pdb_centering_solute);
    if (p->pdb_centering_solute) {
      lprintf("  PDB CENTERING SOLUTE: on\n");
    }
    lprintf("  PDB FILE:     %s\n", _safe_fopen_fname);
  }
  
  if (p->force_file[0]) {
    MD_SYSTEM_write_force(&_md_sys, p->force_file);
    lprintf("  FORCE FILE:     %s\n", _safe_fopen_fname);
  }

  EP_output(&_md_sys.extra_pot, &_md_sys);

  if (p->check_bonds_angles) {
    MD_SYSTEM_check_bonds_angles(&_md_sys);
  }
}

/***************************************
 * section [group]                     *
 ***************************************/
void init_sec_group()
{
  _ctrl.group.flag = 0;

  _ctrl.group.next = NULL;
}

void ctrl_group(CFILE *cfp, char *next_section)
{
  struct sec_group *p;
  char buf[CFILE_BUF_LEN], buf1[CFILE_BUF_LEN];
  char buf2[CFILE_BUF_LEN], buf3[CFILE_BUF_LEN];
  int d1, d2, g_no = 0;
  int ret;
  int current_mode = GM_OR;

  _ctrl.group.flag = 1;

  for (p = &_ctrl.group; p->next!=NULL; p = p->next);

  /* read ... */
  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    if (sscanf(buf," group_no = %d",&d1) == 1) {
      g_no = d1;
      if (g_no > MAX_USER_GROUP_NO) {
	lprintf("ERROR: user group number must be no more than %d\n", MAX_USER_GROUP_NO);
	marble_exit(1);
      }
      current_mode = GM_OR;
      continue;
    }
    if (sscanf(buf," operator = %s", buf1) == 1) {
      current_mode = get_id_sel_data(_sel_gp_mode, buf1);
      if (current_mode == GM_UNKNOWN) {
	lprintf("UNKNOWN OPERATOR: %s\n", buf1);
	marble_exit(1);
      }
      continue;
    }
    if (sscanf(buf," atom_no = %d %d", &d1, &d2) == 2) {
      p->type = GT_ATOM_NO;
      p->group_no = g_no;
      p->mode = current_mode;
      p->no[0] = d1;
      p->no[1] = d2;
      p->next = emalloc("ctrl_group", sizeof(struct sec_group));
      p = p->next; p->next = NULL;
      continue;
    }
    if (sscanf(buf," res_no = %9s %9s", buf1, buf2) == 2) {
      p->type = GT_RES_NO;
      p->group_no = g_no;
      p->mode = current_mode;
      strcpy(p->resno[0],buf1);
      strcpy(p->resno[1],buf2);
      p->next = emalloc("ctrl_group", sizeof(struct sec_group));
      p = p->next; p->next = NULL;
      continue;
    }
    if (sscanf(buf," atom_name = %99s %99s %99s", buf1, buf2, buf3) == 3) {
      p->type = GT_ATOM_NAME_RES_NO;
      p->group_no = g_no;
      p->mode = current_mode;
      strcpy(p->name, buf1);
      strcpy(p->resno[0],buf2);
      strcpy(p->resno[1],buf3);
      p->next = emalloc("ctrl_group", sizeof(struct sec_group));
      p = p->next; p->next = NULL;
      continue;
    }
    if (sscanf(buf," atom_name = %99s", buf1) == 1) { 
      p->type = GT_ATOM_NAME;
      p->group_no = g_no;
      p->mode = current_mode;
      strcpy(p->name, buf1);
      p->next = emalloc("ctrl_group", sizeof(struct sec_group));
      p = p->next; p->next = NULL;
      continue;
    }
    /* if (sscanf(buf," res_name = \"%[^\"]", buf1) == 1) { */
    if (sscanf(buf," res_name = %s", buf1) == 1 ||
        sscanf(buf," residue_name = %s", buf1) == 1) {
      p->type = GT_RES_NAME;
      p->group_no = g_no;
      p->mode = current_mode;
      strcpy(p->name, buf1);
      p->next = emalloc("ctrl_group", sizeof(struct sec_group));
      p = p->next;  p->next = NULL;
      continue;
    }
    
    if (sscanf(buf," atom_type = %s",buf1) == 1) {
      p->type = (group_type_t) get_id_sel_data(_sel_gp_atype, buf1);
      p->group_no = g_no;
      p->mode = current_mode;
      p->next = emalloc("ctrl_group", sizeof(struct sec_group));
      p = p->next;  p->next = NULL;
      continue;
    }
    
    if (sscanf(buf," %s",buf1) == 1) {
      if (strcmp(buf1,"next") == 0) {
	g_no++;
	current_mode = GM_OR;
	continue;
      }
    }
    
    if (sscanf(buf," [%[^]]", next_section) == 1) break;

    ctrl_check_error(buf);
  }
}

void exec_group()
{
  struct sec_group *p;

  if (!_ctrl.group.flag) return;
  
  for(p=&_ctrl.group;p->next!=NULL;p=p->next) {
    switch(p->type) {
    case GT_ATOM_NO:
      lprintf("Group[%d]: atom No. from %d to %d, Operator = %s\n",
	      p->group_no, p->no[0], p->no[1],
	      get_name_sel_data(_sel_gp_mode, p->mode));
      ATOM_DATA_group_atom_no(&_md_sys.atom, p->group_no, p->no[0], p->no[1], p->mode);
      break;
    case GT_RES_NO:
      lprintf("Group[%d]: residue No. from %s to %s, Operator = %s\n",
	      p->group_no, p->resno[0], p->resno[1],
	      get_name_sel_data(_sel_gp_mode, p->mode));
      ATOM_DATA_group_res_no(&_md_sys.atom, p->group_no, p->resno[0], p->resno[1], p->mode);
      break;
    case GT_ATOM_NAME:
      lprintf("Group[%d]: atom name \"%s\", Operator = %s\n", p->group_no,
	      p->name,
	      get_name_sel_data(_sel_gp_mode, p->mode));
      ATOM_DATA_group_atom_name(&_md_sys.atom, p->group_no, p->name, p->mode);
      break;
    case GT_ATOM_NAME_RES_NO:
      lprintf("Group[%d]: atom name \"%s\" in residues from %s to %s, Operator = %s\n", p->group_no, p->name, p->resno[0], p->resno[1], 
	      get_name_sel_data(_sel_gp_mode, p->mode));
      ATOM_DATA_group_atom_name_res_no(&_md_sys.atom, p->group_no, p->name, p->resno[0], p->resno[1], p->mode);
      break;
    case GT_RES_NAME:
      lprintf("Group[%d]: resiude name \"%s\", Operator = %s\n", p->group_no,
	      p->name,
	      get_name_sel_data(_sel_gp_mode, p->mode));
      ATOM_DATA_group_res_name(&_md_sys.atom, p->group_no, p->name, p->mode);
      break;
    case GT_BACKBONE:
      lprintf("Group[%d]: backbone, Operator = %s\n", p->group_no,
	      get_name_sel_data(_sel_gp_mode, p->mode));
      ATOM_DATA_group_atom_type(&_md_sys.atom, p->group_no, p->mode, 1);
      break;
    case GT_HEAVY:
      lprintf("Group[%d]: heavy atom, Operator = %s\n", p->group_no,
	      get_name_sel_data(_sel_gp_mode, p->mode));
      ATOM_DATA_group_atom_type(&_md_sys.atom, p->group_no, p->mode, 2);
      break;
      /*
    case GT_SOLUTE_HEAVY:
      lprintf("Group[%d]: solute heavy atom, Operator = %s\n", p->group_no,
	      get_name_sel_data(_sel_gp_mode, p->mode));
      ATOM_DATA_group_atom_type(&_md_sys.atom, p->group_no, p->mode, 3);
      break;
      */
    }
    lprintf("          %d atoms selected in total\n",
	    ATOM_DATA_count_group_atoms(&_md_sys.atom, p->group_no));
  }
  lprintf("\n");
}

/***************************************
 * section [initialize]                *
 ***************************************/

void init_sec_initialize()
{
  struct sec_initialize *p;
  
  p = &_ctrl.initialize;
  p->flag = 0;

  p->cur_time = -1.0;
  p->cur_time_flag = 0;
  p->seed = getpid();
  p->temperature = -1.0;
  p->solute_residue = -1;
  p->solute_mol = -1;
  p->origin = 0;
  p->a_flag = p->b_flag = p->c_flag = p->shift_flag = 0;
  p->wrap_periodic_boundary = 0;
  p->remove_momentum = 0;
  p->mass = 1.0;
  p->mass_g_no = -1;
  p->scale_bond_k = -1.0;
  p->scale_angle_k = -1.0;
  p->scale_impr_k = -1.0;
  p->overwrite_bond_k = -1.0;
  p->overwrite_angle_k = -1.0;
  p->overwrite_impr_k = -1.0;
  p->bc_reset = 0;
}

void ctrl_initialize(CFILE *cfp, char *next_section)
{
  char buf[CFILE_BUF_LEN], buf1[CFILE_BUF_LEN];
  double f, f1, f2, f3;
  double alpha, beta, gamma;
  long ld;
  int d, d1, d2, d3;
  struct sec_initialize *p;

  /* initialize... */
  p = &_ctrl.initialize;

  p->flag = 1;

  /* read ... */
  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    if (sscanf(buf," current_time = %lf", &f) == 1) {
      p->cur_time=f; p->cur_time_flag=1; continue; }
    if (sscanf(buf," seed = %ld", &ld) == 1) {p->seed=ld;continue;}
    if (sscanf(buf," temperature = %lf", &f) == 1) {p->temperature=f;continue;}
    if (sscanf(buf," solute_residue = %d", &d) == 1) {
      p->solute_residue=d;
      continue;
    }
    if (sscanf(buf," solute_molecule = %d", &d) == 1) {
      p->solute_mol=d;
      continue;
    }
    if (sscanf(buf," a = %lf %lf %lf", &f1,&f2,&f3) == 3) {
      p->a[0]=f1;
      p->a[1]=f2;
      p->a[2]=f3;
      p->a_flag = 1;
      continue;
    }
    if (sscanf(buf," b = %lf %lf %lf", &f1,&f2,&f3) == 3) {
      p->b[0]=f1;
      p->b[1]=f2;
      p->b[2]=f3;
      p->b_flag = 1;
      continue;
    }
    if (sscanf(buf," c = %lf %lf %lf", &f1,&f2,&f3) == 3) {
      p->c[0]=f1;
      p->c[1]=f2;
      p->c[2]=f3;
      p->c_flag = 1;
      continue;
    }
    if (sscanf(buf," box = %lf %lf %lf %lf %lf %lf",
	       &f1, &f2, &f3, &alpha, &beta, &gamma) == 6) {
      p->a[0] = f1;
      p->a[1] = 0.0;
      p->a[2] = 0.0;
      p->b[0] = f2*cos(gamma*M_PI/180.0);
      p->b[1] = f2*sin(gamma*M_PI/180.0);
      p->b[2] = 0.0;
      p->c[0] = f3*cos(beta*M_PI/180.0);
      p->c[1] = (f2*f3*cos(alpha*M_PI/180.0)-p->c[0]*p->b[0])/p->b[1];
      p->c[2] = sqrt(f3*f3-SQR(p->c[0])-SQR(p->c[1]));
      p->a_flag = p->b_flag = p->c_flag = 1;
      continue;
    }
    if (sscanf(buf," box = %lf %lf %lf", &f1, &f2, &f3) == 3) {
      p->a[0] = f1;
      p->a[1] = 0.0;
      p->a[2] = 0.0;
      p->b[0] = 0.0;
      p->b[1] = f2;
      p->b[2] = 0.0;
      p->c[0] = 0.0;
      p->c[1] = 0.0;
      p->c[2] = f3;
      p->a_flag = p->b_flag = p->c_flag = 1;
      continue;
    }
    if (sscanf(buf," boundary_condition = %s", buf1) == 1) {
      if (strcmp(buf1,"none") == 0) {
	p->bc_reset = 1;
      } else {
	lprintf("ERROR: unknown option: %s\n", buf1);
	marble_exit(1);
      }
      continue;
    }
    if (sscanf(buf," shift = %lf %lf %lf", &f1,&f2,&f3) == 3) {
      p->shift[0]=f1;
      p->shift[1]=f2;
      p->shift[2]=f3;
      p->shift_flag = 1;
      continue;
    }
    if (sscanf(buf," shift = %s", buf1) == 1) {
      if (strcmp(buf1,"centering_solute") == 0)
      p->shift_flag = 2;
      continue;
    }
    if (sscanf(buf," origin = %s", buf1) == 1) {
      if (strcmp(buf1,"minimum") == 0) {
	p->origin = 0;
      } else if (strcmp(buf1,"center") == 0) {
	p->origin = 1;
      }
      continue;
    }
    if (sscanf(buf," wrap_molecules = %s",  buf1) == 1) {
      p->wrap_periodic_boundary = get_id_sel_data(_sel_TF, buf1);
      continue;
    }
    if (sscanf(buf," remove_momentum = %s",  buf1) == 1) {
      p->remove_momentum = get_id_sel_data(_sel_remove_momentum, buf1);
      continue;
    }
    if (sscanf(buf," energy_long_format = %s",  buf1) == 1) {
      _md_sys.ene_long_format = get_id_sel_data(_sel_TF, buf1);
      continue;
    }
    if (sscanf(buf," scale_bond_k = %lf",  &f1) == 1) {
      p->scale_bond_k = f1;
      continue;
    }
    if (sscanf(buf," scale_angle_k = %lf",  &f1) == 1) {
      p->scale_angle_k = f1;
      continue;
    }
    if (sscanf(buf," scale_improper_k = %lf",  &f1) == 1) {
      p->scale_impr_k = f1;
      continue;
    }
    if (sscanf(buf," overwrite_bond_k = %lf",  &f1) == 1) {
      p->overwrite_bond_k = f1;
      continue;
    }
    if (sscanf(buf," overwrite_angle_k = %lf",  &f1) == 1) {
      p->overwrite_angle_k = f1;
      continue;
    }
    if (sscanf(buf," overwrite_improper_k = %lf",  &f1) == 1) {
      p->overwrite_impr_k = f1;
      continue;
    }

#ifdef MPI_SDMD

    if (sscanf(buf," overload = %lf %lf", &f, &f2) == 2) {
      _md_sys.linked_cell.start_overload = f;
      _md_sys.linked_cell.max_overload = f2;
      continue;
    }
    if (sscanf(buf," check_time_overflow = %s",  buf1) == 1) {
      _md_sys.linked_cell.check_time_overflow = get_id_sel_data(_sel_TF, buf1);
      continue;
    }
    if (sscanf(buf," tr_list_factor = %d", &d) == 1) {
      _md_sys.linked_cell.tr_list_factor = d;
      continue;
    }
#endif    

    /* not supported because of rigid body
    if (sscanf(buf," mass = %lf %d", &f1, &d) == 2) {
      p->mass = f1;
      p->mass_g_no = d;
      continue;
    }
    */
    
    if (sscanf(buf," [%[^]]", next_section) == 1) break;
    
    ctrl_check_error(buf);
  }

}

void exec_initialize()
{
  struct sec_initialize *p;
  p = &_ctrl.initialize;
  
  srand48(p->seed);
  lprintf("Random Seed: %ld\n\n", p->seed);
  
  if (p->temperature >= 0.0) {
    if (_ctrl.input.restart & RESTART_VEL)
      lprintf("Ignoring initial velocity/temperature setting for restarting.\n\n");    else
      MD_SYSTEM_set_initial_velocity(&_md_sys, p->temperature);
  }

  if (p->cur_time_flag) {
    MD_SYSTEM_set_current_time(&_md_sys, p->cur_time);
  }

  if (p->solute_residue >= 0) {
    MD_SYSTEM_set_solute_residue(&_md_sys, p->solute_residue);
  }
  if (p->solute_mol >= 0) {
    MD_SYSTEM_set_solute_mol(&_md_sys, p->solute_mol);
  }
  
  if (p->a_flag && p->b_flag && p->c_flag) {
    if (_ctrl.input.restart & RESTART_BOX) {
      lprintf("Ignoring box setting for restarting.\n\n");
    } else {
      MD_SYSTEM_set_boxv(&_md_sys, p->a, p->b, p->c);
      _ctrl.output.crd_type |= TCT_BOUNDARY;
    }
  } else if (p->a_flag || p->b_flag || p->c_flag) {
    lprintf("Warning: All a,b,c vectors must be set.\n");
    lprintf("         Partial setting is ignored.\n\n");
  }

  if (p->bc_reset) {
    lprintf("Boundary Condition is reset.\n");
    _md_sys.boundary.type = 0;
  }

  BOUNDARY_set_origin(&_md_sys.boundary, p->origin);

  if (p->shift_flag) {
    if (_ctrl.input.restart & RESTART_BOX)
      lprintf("Ignoring shift setting for restarting.\n\n");
    else if (p->shift_flag == 1) {
      MD_SYSTEM_shift(&_md_sys, p->shift);
    } if (p->shift_flag == 2) {
      MD_SYSTEM_shift_center_solute(&_md_sys);
    }
  }

  BOUNDARY_setup_and_print(&_md_sys.boundary, &_md_sys.atom);
  lprintf("\n");
  
  if (p->wrap_periodic_boundary) {
    MD_SYSTEM_wrap_molecules(&_md_sys, 1, 0);
    lprintf("Wrapping Molecules by Periodic Boundary Condition\n\n");
  }
  
  if (p->remove_momentum) {
    MD_SYSTEM_set_remove_momentum(&_md_sys, p->remove_momentum, 0);
    MD_SYSTEM_remove_momentum(&_md_sys);
    MD_SYSTEM_set_remove_momentum(&_md_sys, 0, 0);
    lprintf("\n");
  }
  
  if (p->scale_bond_k >= 0.0) {
    BOND_DATA_scale_bond_k(&_md_sys.bond, p->scale_bond_k);
  }
  if (p->scale_angle_k >= 0.0) {
    ANGLE_DATA_scale_angle_k(&_md_sys.angle, p->scale_angle_k);
  }
  if (p->scale_impr_k >= 0.0) {
    DIHEDRAL_DATA_scale_improper_k(&_md_sys.dihed, p->scale_impr_k);
  }
  if (p->overwrite_bond_k >= 0.0) {
    BOND_DATA_overwrite_bond_k(&_md_sys.bond, p->overwrite_bond_k);
  }
  if (p->overwrite_angle_k >= 0.0) {
    ANGLE_DATA_overwrite_angle_k(&_md_sys.angle, p->overwrite_angle_k);
  }
  if (p->overwrite_impr_k >= 0.0) {
    DIHEDRAL_DATA_overwrite_improper_k(&_md_sys.dihed, p->overwrite_impr_k);
  }

  /*
  if (p->mass_g_no >= 0) {
    ATOM_DATA_set_mass(&_md_sys.atom, p->mass_g_no, p->mass);
    lprintf("Mass of atoms belonging to group %d: %.4f\n\n", p->mass_g_no, p->mass);
  }
  */

}

/***************************************
 * section [restraint]                 *
 ***************************************/
void init_sec_restraint()
{
  init_restraint(&_ctrl.restraint);

  _ctrl.restraint.flag = 0;
}

void init_restraint(struct sec_restraint *current)
{
  current->next = NULL;
  current->method = rs_unknown;
  current->k  = 0.0;
  current->k1 = 0.0;
  current->gradual_change_step = 0;
  current->r0 = -1.0;
  current->atom[0][0] = '\0';
  current->group[0] = current->group[1] = -1;
  current->group_no = 0;
  current->group_str = current->group2_str = NULL;
  current->flag = 0;
  current->fname[0] = 0;
  current->xx[0] = current->xx[1] = current->xx[2] = 0.0;
  current->virial_flag = 1;

  /* yss-add */
  current->x_flag = 1; 
  current->y_flag = 1; 
  current->z_flag = 1;
  current->potential_type = 0;
  current->mole_residue = 1;
  /* yss-add (Mar09/2004) */
  current->delta = 0.0;
  /* yse */

  /* YAMANE_ADDED */
/* for nmr_distance */
  current->scl_dis  = 50.0;
  current->scl_dis1 = 0.0;
  current->Ceil = 1000;
  current->sqof = 0.0;
  current->sqcs = 1.0;
  current->sqex = 2.0;
  current->soex = 1.0;
  current->asym = 1.0;
  current->rswi = 0.5;
  current->viol_dis = 0;
  current->n_output_viol = 5;
/* for nmr_dihedral */
  current->scl_dih  = 50.0;
  current->scl_dih1 = 0.0;
  current->viol_dih = 0;
/* for nmr_plane */
  current->scl_pln = 50.0;
  current->scl_pln1 = 0;
/* End of YAMANE_ADDED */

  /* Steered MD */
  current->smd_method = smd_unknown;
  current->smd_center = 0;
  current->smd_weight_flag = 0;
  current->smd_dir[0] = 1.0;
  current->smd_dir[1] = 0.0;
  current->smd_dir[2] = 0.0;
  current->smd_pos[0] = 0.0;
  current->smd_pos[1] = 0.0;
  current->smd_pos[2] = 0.0;
  current->smd_vel = 0.0;
  current->smd_ang_vel = 0.0;
  current->smd_init_trans = 0.0;
  current->smd_init_rot = 0.0;
  current->smd_test_step = 0;

  /* RMSD */
  current->fname2[0] = '\0';
  current->gradual_change_step_r0 = 0;
  current->r1 = 0.0;
  current->rmsd_best_fit = 1;
  current->pdb_mode = current->pdb2_mode = 0;
  current->pdb_group_str = current->pdb_group_str2 = NULL;
  current->check_pdb_group = 1;
  current->rmsd_k_per_atom_flag = 0;
  current->rmsd_weight_flag = 0;
  current->rmsd_weight_list = NULL;
}

void ctrl_restraint(CFILE *cfp, char *next_section)
{
  char buf[CFILE_BUF_LEN], buf2[CFILE_BUF_LEN], buf3[CFILE_BUF_LEN], buf4[CFILE_BUF_LEN], buf5[CFILE_BUF_LEN];
  struct sec_restraint *p;
  char c;
  int d, d1;
  double f, f1, f2, f3;
  char *fname = "ctrl_restraint";
  static int first = 1;
  /* yss-add (Mar09/2004) */ 
  int da, da1, da2, da3;
  /* yse */

  _ctrl.restraint.flag = 1;

  for (p = &_ctrl.restraint; p->next!=NULL; p=p->next);
  
  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    if (sscanf(buf," method = %s", buf2) == 1) {
      if (first) {
	p->method = (restraint_method_t) get_id_sel_data(_sel_rs_method, buf2);
	first = 0;
      } else {
	p->next = emalloc(fname, sizeof(struct sec_restraint));
	p = p->next;
	init_restraint(p);
	p->method = (restraint_method_t) get_id_sel_data(_sel_rs_method, buf2);
      }
      continue;
    }
    switch (p->method) {
    case rs_distance_harmonic:
    case rs_group_distance_harmonic:
    case rs_torsion_harmonic:
      if (sscanf(buf," k = %lf", &f) == 1) {p->k=f;continue;}
      if (sscanf(buf," r0 = %lf",&f) == 1) {p->r0=f;continue;}
      if (sscanf(buf," gradual_change_k = %d %lf", &d,&f) == 2) {
	p->gradual_change_step=d; p->k1=f; continue;
      }
      if (p->method == rs_group_distance_harmonic) {
	if (sscanf(buf," gradual_change_r0 = %d %lf", &d,&f) == 2) {
	  p->gradual_change_step_r0=d; p->r1=f; continue;
	}
      }
      if (sscanf(buf," virial = %s", buf2) == 1) {
	p->virial_flag = get_id_sel_data(_sel_TF, buf2); continue;
      }
      if (sscanf(buf," potential_type = %s", buf2) == 1) {
	p->potential_type = get_id_sel_data(_sel_potential_type, buf2);
	continue;
      }
      if (p->method == rs_distance_harmonic ||
	  p->method == rs_group_distance_harmonic) {
	if (sscanf(buf," x_rest = %s", buf2) == 1) {
	  p->x_flag = get_id_sel_data(_sel_TF, buf2); continue;
	}
	if (sscanf(buf," y_rest = %s", buf2) == 1) {
	  p->y_flag = get_id_sel_data(_sel_TF, buf2); continue;
	}
	if (sscanf(buf," z_rest = %s", buf2) == 1) {
	  p->z_flag = get_id_sel_data(_sel_TF, buf2); continue;
	}
      }
      if (p->method == rs_distance_harmonic) {
	if (sscanf(buf," atom = %s %s", buf2, buf3) == 2) {
	  strcpy(p->atom[0],buf2);
	  strcpy(p->atom[1],buf3);
	  continue;
	}
      } else if (p->method == rs_group_distance_harmonic) {
	if (sscanf(buf," group_no = %d %d", &d,&d1) == 2) {
	  p->group[0]=d; p->group[1]=d1;
	  continue;
	}
	if (sscanf(buf," group1 = %[^\n]",buf2) == 1) {
	  append_str_list(&p->group_str, buf2);
	  continue;
	}
	if (sscanf(buf," group2 = %[^\n]",buf2) == 1) {
	  append_str_list(&p->group2_str, buf2);
	  continue;
	}
      } else if (p->method == rs_torsion_harmonic) {
	if (sscanf(buf," tor_atom = %s %s %s %s", buf2,buf3,buf4,buf5) == 4 ||
	    sscanf(buf," atom     = %s %s %s %s", buf2,buf3,buf4,buf5) == 4) {
	  strcpy(p->atom[0],buf2);
	  strcpy(p->atom[1],buf3);
	  strcpy(p->atom[2],buf4);
	  strcpy(p->atom[3],buf5);
	  continue;
	}
	if (sscanf(buf," torsion = %lf",&f) == 1) {p->r0=f;continue;}
	if (sscanf(buf," dihedral = %lf",&f) == 1) {p->r0=f;continue;}
	if (sscanf(buf," delta_r = %lf",&f) == 1) {p->delta=f;continue;}
	if (sscanf(buf," delta = %lf",&f) == 1) {p->delta=f;continue;}
      }
      break;
      
    case rs_position_harmonic:
    case rs_group_position_harmonic:
    case rs_molecule_position_harmonic:
      if (sscanf(buf," group_no = %d", &d) == 1) {p->group_no=d;continue;}
      if (sscanf(buf," group = %[^\n]",buf2) == 1) {
	append_str_list(&p->group_str, buf2);
	continue;
      }
      if (sscanf(buf," crd_file = %s", buf2) == 1) {
	strcpy(p->fname, buf2); continue;
      }
      if (sscanf(buf," k = %lf", &f) == 1) {p->k=f;continue;}
      if (sscanf(buf," gradual_change_k = %d %lf", &d,&f) == 2) {
	p->gradual_change_step=d; p->k1=f; continue;
      }
      if (sscanf(buf," virial = %s", buf2) == 1) {
	p->virial_flag = get_id_sel_data(_sel_TF, buf2); continue;
      }
      if (sscanf(buf," x_rest = %s", buf2) == 1) {
	p->x_flag = get_id_sel_data(_sel_TF, buf2); continue;
      }
      if (sscanf(buf," y_rest = %s", buf2) == 1) {
	p->y_flag = get_id_sel_data(_sel_TF, buf2); continue;
      }
      if (sscanf(buf," z_rest = %s", buf2) == 1) {
	p->z_flag = get_id_sel_data(_sel_TF, buf2); continue;
      }
      if (p->method == rs_molecule_position_harmonic) {
	if (sscanf(buf," mole_residue = %d", &d) == 1) {
	  p->mole_residue = d; continue;
	}
      } else if (p->method == rs_group_position_harmonic) {
	if (sscanf(buf," position = %lf %lf %lf", &f1, &f2, &f3) == 3 ||
	    sscanf(buf," reference_point = %lf %lf %lf", &f1, &f2, &f3) == 3) {
	  p->xx[0] = f1;
	  p->xx[1] = f2;
	  p->xx[2] = f3;
	  p->fname[0] = '\0';
	  continue;
	}
      }
      break;
      
    case rs_nmr_distance:
      if (sscanf(buf," nmr_dis_file = %s", buf2) == 1 ) {
	strcpy(p->fname_dis, buf2);continue; }
      if (sscanf(buf," scale_dis = %lf",&f) == 1) {p->scl_dis=f;continue;}
      /* if (sscanf(buf," scale_dis1 = %lf",&f) == 1) {p->scl_dis1=f;continue;} */
      if (sscanf(buf," ceiling = %lf", &f) == 1) {p->Ceil=f;continue;}
      if (sscanf(buf," sqoffset = %lf", &f) == 1) {p->sqof=f;continue;}
      if (sscanf(buf," sqconstant = %lf", &f) == 1) {p->sqcs=f;continue;}
      if (sscanf(buf," sqexponent = %lf", &f) == 1) {p->sqex=f;continue;}
      if (sscanf(buf," softexponent = %lf", &f) == 1) {p->soex=f;continue;}
      if (sscanf(buf," asymptote = %lf", &f) == 1) {p->asym=f;continue;}
      if (sscanf(buf," rswitch = %lf", &f) == 1) {p->rswi=f;continue;}
      if (sscanf(buf," viol_dis = %lf",&f) == 1) {p->viol_dis=f;continue;}
      if (sscanf(buf," n_output_viol = %d",&d) == 1) {p->n_output_viol=d;continue;}
      if (sscanf(buf," gradual_change_scale_dis = %d %lf", &d,&f) == 2) {
	p->gradual_change_step=d; p->scl_dis1=f; continue;
      }
      if (sscanf(buf," virial = %s", buf2) == 1) {
	p->virial_flag = get_id_sel_data(_sel_TF, buf2); continue;
      }
      break;

    case rs_nmr_dihedral:
      if (sscanf(buf," nmr_dih_file = %s", buf2) == 1) {
	strcpy(p->fname_dih, buf2);continue; }
      if (sscanf(buf," scale_dih = %lf",&f) == 1) {p->scl_dih=f;continue;}
      /*if (sscanf(buf," scale_dih1 = %lf",&f) == 1) {p->scl_dih1=f;continue;} */
      if (sscanf(buf," viol_dih = %lf",&f) == 1) {p->viol_dih=f;continue;}
      if (sscanf(buf," n_output_viol = %d",&d) == 1) {p->n_output_viol=d;continue;}
      if (sscanf(buf," gradual_change_scale_dih = %d %lf", &d,&f) == 2) {
	p->gradual_change_step=d; p->scl_dih1=f; continue;
      }
      if (sscanf(buf," virial = %s", buf2) == 1) {
	p->virial_flag = get_id_sel_data(_sel_TF, buf2); continue;
      }
      break;
    
    case rs_nmr_plane:
      if(sscanf(buf," nmr_pln_file = %s", buf2) == 1) {
	strcpy(p->fname_pln, buf2);continue; }
      if(sscanf(buf," scale_pln = %lf",&f) == 1) {p->scl_pln=f;continue;}
      /* if(sscanf(buf," scale_pln1 = %lf",&f) == 1) {p->scl_pln1=f;continue;} */
      if (sscanf(buf," gradual_change_scale_pln = %d %lf", &d,&f) == 2) {
	p->gradual_change_step=d; p->scl_pln1=f; continue;
      }
      if (sscanf(buf," virial = %s", buf2) == 1) {
	p->virial_flag = get_id_sel_data(_sel_TF, buf2); continue;
      }
      break;

    /* ikeguchi 2007/12/04 */
    case rs_steered_md:
      if (sscanf(buf," smd_method = %s", buf2) == 1) {
	p->smd_method = (smd_method_t) get_id_sel_data(_sel_smd_method, buf2);
	continue;
      }
      if (sscanf(buf," group_no = %d", &d) == 1) {
	p->group[0]=d;
	continue;
      }
      if (sscanf(buf," group = %[^\n]",buf2) == 1) {
	append_str_list(&p->group_str, buf2);
	continue;
      }

      switch (p->smd_method) {
      case smd_const_force:
	if (sscanf(buf," force = %lf",&f) == 1) {p->smd_force=f;continue;}
	if (sscanf(buf," torque = %lf",&f) == 1) {p->smd_torque=f;continue;}
	if (sscanf(buf," direction = %lf %lf %lf",&f1, &f2, &f3) == 3) {
	  p->smd_dir[0]=f1; p->smd_dir[1]=f2; p->smd_dir[2]=f3;
	  continue;
	}
	if (sscanf(buf," axis_position = %lf %lf %lf",&f1, &f2, &f3) == 3) {
	  p->smd_pos[0]=f1; p->smd_pos[1]=f2; p->smd_pos[2]=f3;
	  continue;
	}
	break;

      case smd_const_vel:
	if (sscanf(buf," crd_file = %s", buf2) == 1) {
	  strcpy(p->fname, buf2); continue;
	}
	if (sscanf(buf," velocity = %lf",&f) == 1) {p->smd_vel=f;continue;}
	if (sscanf(buf," angular_velocity = %lf",&f) == 1) {p->smd_ang_vel=f;continue;}
	if (sscanf(buf," direction = %lf %lf %lf",&f1, &f2, &f3) == 3) {
	  p->smd_dir[0]=f1; p->smd_dir[1]=f2; p->smd_dir[2]=f3;
	  continue;
	}
	if (sscanf(buf," axis_position = %lf %lf %lf",&f1, &f2, &f3) == 3) {
	  p->smd_pos[0]=f1; p->smd_pos[1]=f2; p->smd_pos[2]=f3;
	  continue;
	}
	if (sscanf(buf," k = %lf", &f) == 1) {p->k=f;continue;}
	if (sscanf(buf," initial_translation = %lf",&f) == 1) {
	  p->smd_init_trans=f;continue;
	}
	if (sscanf(buf," initial_rotation = %lf",&f) == 1) {
	  p->smd_init_rot=f;continue;
	}
	if (sscanf(buf," x_rest = %s", buf2) == 1) {
	  p->x_flag = get_id_sel_data(_sel_TF, buf2); continue;
	}
	if (sscanf(buf," y_rest = %s", buf2) == 1) {
	  p->y_flag = get_id_sel_data(_sel_TF, buf2); continue;
	}
	if (sscanf(buf," z_rest = %s", buf2) == 1) {
	  p->z_flag = get_id_sel_data(_sel_TF, buf2); continue;
	}
	if (sscanf(buf," test = %s %lf %d %d", buf2, &f, &d, &d1) == 4) {
	  strcpy(p->smd_test_file, buf2);
	  p->smd_test_timestep = f;
	  p->smd_test_step = d;
	  p->smd_test_interval = d1;
	  continue;
	}
	if (sscanf(buf," gradual_change_k = %d %lf", &d,&f) == 2) {
	  p->gradual_change_step=d; p->k1=f; continue;
	}
	if (sscanf(buf," center_of_mass = %s", buf2) == 1) {
	  p->smd_center = (smd_center_t) get_id_sel_data(_sel_smd_center, buf2);
	  continue;
	}
	if (sscanf(buf," center_weight_flag = %s", buf2) == 1) {
	  p->smd_weight_flag = get_id_sel_data(_sel_weight_flag, buf2); 
	  if (p->smd_weight_flag < 0) {
	    lprintf("ERROR: the weight flag must be \"uniform\" or \"mass_weighted\".\n");
	    marble_exit(1);
	  }
	  continue;
	}
	break;
      }
      break;
      
    case rs_rmsd:
    case rs_rmsd2:
      if (sscanf(buf," crd_file = %s", buf2) == 1) {
	strcpy(p->fname, buf2); 
	p->pdb_mode = 0;
	continue;
      }
      if (sscanf(buf," crd_file2 = %s", buf2) == 1) {
	strcpy(p->fname2, buf2); 
	p->pdb2_mode = 0;
	continue;
      }
      if (sscanf(buf," pdb_file = %s", buf2) == 1) {
	strcpy(p->fname, buf2); 
	p->pdb_mode = 1;
	continue;	
      }
      if (sscanf(buf," pdb_file2 = %s", buf2) == 1) {
	strcpy(p->fname2, buf2); 
	p->pdb2_mode = 1;
	continue;
      }
      if (sscanf(buf," pdb_group = %[^\n]",buf2) == 1) {
	append_str_list(&p->pdb_group_str, buf2);
	continue;
      }
      if (sscanf(buf," pdb_group2 = %[^\n]",buf2) == 1) {
	append_str_list(&p->pdb_group_str2, buf2);
	continue;
      }
      if (sscanf(buf," check_pdb_group = %s",buf2) == 1) {
	p->check_pdb_group = get_id_sel_data(_sel_TF, buf2);
	continue;
      }
      if (sscanf(buf," group_no = %d", &d) == 1) {
	p->group[0]=d;
	continue;
      }
      if (sscanf(buf," group = %[^\n]",buf2) == 1) {
	append_str_list(&p->group_str, buf2);
	continue;
      }
      if (sscanf(buf," k = %lf", &f) == 1) {p->k=f;continue;}
      if (sscanf(buf," kn = %lf", &f) == 1) {p->k=f;p->rmsd_k_per_atom_flag=1; continue;}
      if (sscanf(buf," k_per_atom = %lf", &f) == 1) {p->k=f;p->rmsd_k_per_atom_flag=1; continue;}
      if (sscanf(buf," rmsd = %lf",&f) == 1) {p->r0=f;continue;}
      if (sscanf(buf," rmsd_diff = %lf",&f) == 1) {p->r0=f;continue;}
      if (sscanf(buf," gradual_change_k = %d %lf", &d,&f) == 2) {
	p->gradual_change_step=d; p->k1=f; continue;
      }
      if (sscanf(buf," gradual_change_rmsd = %d %lf", &d,&f) == 2) {
	p->gradual_change_step_r0=d; p->r1=f; continue;
      }
      if (sscanf(buf," best_fit = %s", buf2) == 1) {
	p->rmsd_best_fit = get_id_sel_data(_sel_TF, buf2); continue;
      }
      if (sscanf(buf," weight_flag = %s", buf2) == 1) {
	p->rmsd_weight_flag = get_id_sel_data(_sel_weight_flag, buf2); 
	if (p->rmsd_weight_flag < 0) {
	  lprintf("ERROR: the weight flag must be \"uniform\" or \"mass_weighted\".\n");
	  marble_exit(1);
	}
	continue;
      }
      if (sscanf(buf," weight = %lf", &f) == 1) {
	FIT_append_weight_list(&(p->rmsd_weight_list), f);
	continue;
      }
      if (sscanf(buf," weight_group = %[^\n]",buf2) == 1) {
	if (p->rmsd_weight_list == NULL) {
	  lprintf("ERROR: set weight value before setting weight group.\n");
	  marble_exit(1);
	}
	FIT_set_weight_list_group_str(p->rmsd_weight_list, buf2);
	continue;
      }
      if (sscanf(buf," virial = %s", buf2) == 1) {
	p->virial_flag = get_id_sel_data(_sel_TF, buf2); continue;
      }
      if (sscanf(buf," potential_type = %s", buf2) == 1) {
	p->potential_type = get_id_sel_data(_sel_potential_type, buf2);
	continue;
      }
      break;
    }

    if (sscanf(buf," %s", buf2) == 1) {
      if (strcmp(buf2,"next") == 0) {
	p->next = emalloc(fname, sizeof(struct sec_restraint));
	p = p->next;
	init_restraint(p);
	continue;
      }
    }
    
    if (sscanf(buf," [%[^]]", next_section) == 1) break;

    ctrl_check_error(buf);
  }
}

void exec_restraint()
{
  struct sec_restraint *p;

  p = &_ctrl.restraint;

  if (!p->flag) return;
  
  for (;p!=NULL;p=p->next) {
    switch(p->method) {
    case rs_distance_harmonic:
      EP_DISTANCE_setup(&_md_sys.extra_pot,&_md_sys.atom,&_md_sys.boundary,
			&_md_sys.linked_cell,
			p->atom[0],p->atom[1],p->k,p->r0,
			p->gradual_change_step, p->k1, p->virial_flag,
			p->potential_type,
			p->x_flag, p->y_flag, p->z_flag);
      break;
    case rs_group_distance_harmonic:
      EP_GROUP_DISTANCE_setup(&_md_sys.extra_pot,&_md_sys.atom,&_md_sys.boundary,
			      &_md_sys.linked_cell,
			      p->group_str, p->group2_str,
			      p->group[0],p->group[1],p->k,p->r0,
			      p->gradual_change_step, p->k1, 
			      p->gradual_change_step_r0, p->r1, 
			      p->virial_flag,
			      p->potential_type,
			      p->x_flag, p->y_flag, p->z_flag);
      break;
/* yss-add */
    case rs_position_harmonic:
      EP_POSITION_setup(&_md_sys.extra_pot,&_md_sys.atom, p->group_str,
			p->group_no, p->fname, p->k,
			p->gradual_change_step, p->k1, p->virial_flag,
			p->x_flag, p->y_flag, p->z_flag);
      break;
    case rs_group_position_harmonic:
      EP_GROUP_POSITION_setup(&_md_sys.extra_pot,&_md_sys.atom,
		              &_md_sys.boundary,&_md_sys.linked_cell,	
			      p->group_str, p->group_no, p->fname, p->xx, p->k,
			      p->gradual_change_step, p->k1, p->virial_flag,
			      p->x_flag, p->y_flag, p->z_flag);
      break;
    case rs_molecule_position_harmonic:
      EP_MOLECULE_POSITION_setup(&_md_sys.extra_pot,&_md_sys.atom,
				 &_md_sys.boundary,&_md_sys.linked_cell,
				 p->group_no, p->fname, p->k,
				 p->gradual_change_step, p->k1, p->virial_flag,
				 p->mole_residue,
				 p->x_flag, p->y_flag, p->z_flag);
      break;
/* yss-add (Mar09/2004) */
    case rs_torsion_harmonic:
      EP_TORSION_setup(&_md_sys.extra_pot,&_md_sys.atom,&_md_sys.boundary,
		       &_md_sys.linked_cell,
		       p->atom[0],p->atom[1],
		       p->atom[2],p->atom[3],p->k,p->r0,p->delta,
		       p->gradual_change_step, p->k1, p->virial_flag);
      break;
/* yse */

/* YAMANE_ADDED */
/* for nmr_restraint */
    case rs_nmr_distance:
      EP_NMR_DIS_setup(&_md_sys.extra_pot,
		       &_md_sys.atom,
		       &_md_sys.linked_cell,
		       p->fname_dis, p->scl_dis, p->scl_dis1,
		       p->Ceil,
		       p->sqof, p->sqcs, p->sqex,
		       p->soex, p->asym, p->rswi, p->viol_dis, p->n_output_viol,
		       p->gradual_change_step, p->virial_flag);
      break;
/* for nmr_dihedral */
    case rs_nmr_dihedral:
      EP_NMR_DIH_setup(&_md_sys.extra_pot,
		       &_md_sys.atom,
                       &_md_sys.linked_cell,
		       p->fname_dih,p->scl_dih, p->scl_dih1,
		       p->viol_dih, p->n_output_viol,
		       p->gradual_change_step,
                       p->virial_flag);
      break;
/* for nmr_planarity */
    case rs_nmr_plane:
      EP_NMR_PLN_setup(&_md_sys.extra_pot,
		       &_md_sys.atom,
                       &_md_sys.linked_cell,
		       p->fname_pln,
		       p->scl_pln,
		       p->scl_pln1,
		       p->gradual_change_step,
                       p->virial_flag);
      break;
/* End of YAMANE_ADDED */

    case rs_steered_md:
      if (p->smd_method == smd_const_force) {
	lprintf("ERROR: Steered MD: constant force has not be implemented yet.\n");
	marble_exit(1);
      } else if (p->smd_method == smd_const_vel) {
	EP_SMD_CV_setup(&_md_sys.extra_pot,&_md_sys.atom,  &_md_sys.linked_cell,
			p->group_str, p->group_no, 
			p->fname, p->k,
			p->smd_vel, p->smd_ang_vel,
			p->smd_init_trans, p->smd_init_rot,
			p->smd_dir, p->smd_pos, p->smd_center, p->smd_weight_flag,
			p->gradual_change_step, p->k1, 
			p->virial_flag,
			p->x_flag, p->y_flag, p->z_flag);
	if (p->smd_test_step > 0) {
	  EP_SMD_CV_test(&_md_sys.extra_pot,&_md_sys.atom,
			 p->smd_test_file, p->smd_test_timestep,
			 p->smd_test_step, p->smd_test_interval);
	  marble_exit(1);
	}
      }
      break;

    case rs_rmsd:
    case rs_rmsd2:
      EP_RMSD_setup(&_md_sys.extra_pot,&_md_sys.atom, &_md_sys.linked_cell,
		    p->group_str, p->group_no, 
		    p->fname, p->fname2, 
		    p->pdb_mode, p->pdb2_mode, p->pdb_group_str, 
		    p->pdb_group_str2, p->check_pdb_group,
		    p->rmsd_best_fit, p->rmsd_weight_flag, p->rmsd_weight_list,
		    p->k, p->r0, 
		    p->gradual_change_step, p->k1, 
		    p->gradual_change_step_r0, p->r1, 
		    p->rmsd_k_per_atom_flag,
		    p->potential_type,
		    p->virial_flag);
      break;

    default:
      lprintf(" ERROR: unknown restraint method.\n");
      marble_exit(1);
    }
  }
}

/***************************************
 * section [constraint]                *
 ***************************************/
void init_sec_constraint()
{
  struct sec_constraint *p;

  p = &_ctrl.constraint;

  p->flag = 0;

  p->rigid_body_flag=0;
  p->check_rigid_body=0;
  p->rattle = RATTLE_NONE;
  p->tolerance=1.0e-6;
  p->fixed_group_flag = 0;
  p->fixed_group_no = 0;
  p->fixed_group_str = NULL;
}


void ctrl_constraint(CFILE *cfp, char *next_section)
{
  char buf[CFILE_BUF_LEN],buf2[CFILE_BUF_LEN],buf3[CFILE_BUF_LEN];
  int d, ret, no = 1;
  double f;
  struct sec_constraint *p;

  p = &_ctrl.constraint;
  
  p->flag = 1;
  
  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    if (sscanf(buf," rattle = %s", buf3) == 1) {
      p->rattle = (RATTLE_TYPE) get_id_sel_data(_sel_cs_rattle,buf3);
      continue;
    }
    if (sscanf(buf," rattle_tolerance = %lf", &f) == 1) {
      p->tolerance=f;
      continue;
    }

    if (sscanf(buf," rigid_body = %s", buf2) == 1) {
      if (p->rigid_body_flag) {
	lprintf("ERROR: rigid_body must be set only once.\n");
	marble_exit(1);
      }
      p->rigid_body_flag=1;
      strcpy(p->rigid_body_fname, buf2);
      continue;
    }
    if (sscanf(buf," check_rigid_body = %s", buf2) == 1) {
      p->check_rigid_body = get_id_sel_data(_sel_TF, buf2);
      continue;
    }

    if (sscanf(buf," fixed_atom_group_no = %d", &d) == 1) { 
      p->fixed_group_flag = 1;
      p->fixed_group_no = d;
      continue;
    }

    if (sscanf(buf," fixed_atom_group = %[^\n]", buf2) == 1) { 
      p->fixed_group_flag = 1;
      append_str_list(&p->fixed_group_str, buf2);
      continue;
    }

    
    if (sscanf(buf," [%[^]]", next_section) == 1) break;

    ctrl_check_error(buf);
  }
}

void exec_constraint()
{
  int i;
  struct sec_constraint *p;
  RATTLE_TYPE type;
  p = &_ctrl.constraint;

  if (!p->flag) return;

  if (p->fixed_group_flag) {
    MD_SYSTEM_set_fixed_atom(&_md_sys, p->fixed_group_str, p->fixed_group_no);
  }
  
  if (p->rigid_body_flag) {
    MD_SYSTEM_set_rigid_mol(&_md_sys, p->rigid_body_fname);
    _ctrl.output.crd_type |= TCT_RMOL;

    if (p->check_rigid_body)
      MD_SYSTEM_check_rigid_mol(&_md_sys);
  }

  if (p->rattle != RATTLE_NONE)
    RATTLE_setup(&_md_sys.rattle, &_md_sys.bond, &_md_sys.atom,
		 p->rattle, p->tolerance);

  if (p->rigid_body_flag || p->rattle != RATTLE_NONE || p->fixed_group_flag)
    ATOM_DATA_set_node_fatom(&_md_sys.atom);

  MD_SYSTEM_degree_of_freedom(&_md_sys);
}

/***************************************
 * section [PT_control]                *
 ***************************************/
void init_sec_PT_control()
{
  struct sec_PT_control *p;

  p = &_ctrl.PT_control;

  p->flag = 0;
  
  p->method = pt_extended_system;
  p->ensemble = en_NVT;
  p->coupling = get_id_sel_data(_sel_pt_coupling,"whole");
  p->crd_data = 2;   /* False */
  p->temperature = 298.15;
  p->period_T = 1.0;
  p->n_chain_T = 1;
  p->tau_T = 0.4;
  p->delta_T = 0.0;
  p->pressure = 1.0;
  p->compress = 1.0/(1890.0e6/101325.0);
  /* default bulk modules 1890 MPa  1 atm = 101325 Pa */
  p->period_P = 1.0;
  p->period_PT = 1.0;
  p->n_chain_PT = 1;
  p->tau_P = 0.4;
  p->gradual_change_step = 0;
  p->temperature1 = p->temperature;

  p->init_pt_control = 2;
}

void ctrl_PT_control(CFILE *cfp, char *next_section)
{
  char buf[CFILE_BUF_LEN],buf2[CFILE_BUF_LEN],buf3[CFILE_BUF_LEN];
  int d;
  double f;
  struct sec_PT_control *p;

  p = &_ctrl.PT_control;
  p->flag = 1;
  
  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    if (sscanf(buf," method = %s", buf2) == 1) {
      p->method = (PT_control_method_t) get_id_sel_data(_sel_pt_method,buf2);
      continue;
    }
    if (sscanf(buf," ensemble = %s", buf2) == 1) {
      p->ensemble = (PT_control_ensemble_t) get_id_sel_data(_sel_pt_ensemble,buf2);
      continue;
    }
    if (sscanf(buf," coupling = %s", buf2) == 1) {
      p->coupling = get_id_sel_data(_sel_pt_coupling,buf2);
      continue;
    }
    if (sscanf(buf," crd_data = %s", buf2) == 1) {
      p->crd_data = get_id_sel_data(_sel_TF, buf2);
      continue;
    }
    if (sscanf(buf," init_PT_control = %s", buf2) == 1 ||
    	sscanf(buf," init = %s", buf2) == 1 ||
    	sscanf(buf," initialize = %s", buf2) == 1) {
      p->init_pt_control = get_id_sel_data(_sel_TF, buf2);
      if (!(p->init_pt_control))
	_ctrl.input.restart |= RESTART_PT;
      else
	_ctrl.input.restart &= ~RESTART_PT;
      continue;
    }
    if (sscanf(buf," restart = %s", buf2) == 1) {
      if (get_id_sel_data(_sel_TF, buf2)) {
	p->init_pt_control = 0;
	_ctrl.input.restart |= RESTART_PT;
      } else {
	p->init_pt_control = 1;
	_ctrl.input.restart &= ~RESTART_PT;
      }
      continue;
    }
    if (sscanf(buf," restart_off = %s", buf2) == 1) {
      if (strcmp(buf2,"PT_control") == 0) {
	p->init_pt_control = 1;
	_ctrl.input.restart &= ~RESTART_PT;
	continue;
      } else {
	lprintf("ERROR: only \"restart_off = PT_contorol\" is available.\n");
	marble_exit(1);
      }
    }
    if (sscanf(buf," temperature = %lf", &f) == 1) {p->temperature=f;continue;}
    if (sscanf(buf," period_T = %lf", &f) == 1) {p->period_T=f;continue;}
    if (sscanf(buf," n_chain_T = %d", &d) == 1) { p->n_chain_T=d;continue;}
    if (sscanf(buf," tau_T = %lf", &f) == 1) {p->tau_T=f;continue;}
    if (sscanf(buf," pressure = %lf", &f) == 1) {p->pressure=f;continue;}
    if (sscanf(buf," period_P = %lf", &f) == 1) {p->period_P=f;continue;}
    if (sscanf(buf," surface_tension = %lf", &f) == 1) {
      PTC_set_gamma_ex(&_md_sys, f);
      continue;
    }
    /*
    if (sscanf(buf," period_PT = %lf", &f) == 1) {p->period_PT=f;continue;}
    if (sscanf(buf," n_chain_PT = %d", &d) == 1) {p->n_chain_PT=d;continue;}
    */
    if (sscanf(buf," tau_P = %lf", &f) == 1) {p->tau_P=f;continue;}

    if (sscanf(buf," gradual_change_T = %d %lf", &d, &f) == 2) {
      p->gradual_change_step=d;
      p->temperature1=f;
      continue;
    }
    if (sscanf(buf," delta_T = %lf", &f) == 1) {
      p->delta_T = f; continue;
    }

    if (sscanf(buf," [%[^]]", next_section) == 1) break;

    ctrl_check_error(buf);
  }
}

void exec_PT_control()
{
  struct sec_PT_control *p;

  p = &_ctrl.PT_control;

  /*
  lprintf("%d %d %d\n",p->init_pt_control,_md_sys.Ex_System_T_flag, _md_sys.Ex_System_P_flag);
  */
  
  if (p->init_pt_control == 2) {
    /* not specified : initialize = on or off */
    if (_ctrl.input.restart & RESTART_PT)
      p->init_pt_control = 0;
    else
      p->init_pt_control = 1;
  }

  if (!(p->init_pt_control) &&
      (_md_sys.Ex_System_T_flag || _md_sys.Ex_System_P_flag)) {
    PTC_Ex_System_print(&_md_sys);
    _ctrl.output.crd_type |= TCT_EXT;
    if (_md_sys.Ex_System_P_flag)
      _ctrl.output.crd_type |= TCT_BOUNDARY;
    lprintf("PT_CONTROL: Data in crd file are used.\n");
    lprintf("When you initialize settings,\nset \"initialize = on\" in [PT_control] section.\n");
    return;
  }
  
  if (!p->flag)
    return;
  
  switch(p->method) {
  case pt_extended_system:
    if (p->ensemble == en_NVT)
      PTC_set_Ex_System_T_cnst(&_md_sys, p->temperature,p->period_T,p->n_chain_T, p->coupling);
    else if (p->ensemble == en_NPT) {
      PTC_set_Ex_System_PT_cnst(&_md_sys, p->temperature,p->period_T,
				p->n_chain_T,p->coupling,
				p->pressure, p->period_P, 1);
      _ctrl.output.crd_type |= TCT_BOUNDARY;
    } else if (p->ensemble == en_NPTflex){
      PTC_set_Ex_System_PT_cnst(&_md_sys, p->temperature,p->period_T,
				p->n_chain_T,p->coupling,
				p->pressure, p->period_P, 2);
      _ctrl.output.crd_type |= TCT_BOUNDARY;
    } else if (p->ensemble == en_NPTxyz) {
      PTC_set_Ex_System_PT_cnst(&_md_sys, p->temperature,p->period_T,
				p->n_chain_T,p->coupling,
				p->pressure, p->period_P, 3);
      _ctrl.output.crd_type |= TCT_BOUNDARY;
    } else if (p->ensemble == en_NPATz) {
      PTC_set_Ex_System_PT_cnst(&_md_sys, p->temperature,p->period_T,
				p->n_chain_T,p->coupling,
				p->pressure, p->period_P, 4);
      _ctrl.output.crd_type |= TCT_BOUNDARY;
    } else if (p->ensemble == en_NPTisoxy) {
      PTC_set_Ex_System_PT_cnst(&_md_sys, p->temperature,p->period_T,
				p->n_chain_T,p->coupling,
				p->pressure, p->period_P, 5);
      _ctrl.output.crd_type |= TCT_BOUNDARY;
    } else if (p->ensemble == en_NVE) {
      PTC_set_Ex_System_off(&_md_sys);
      return;
    }
    PTC_Ex_System_print(&_md_sys);
    _ctrl.output.crd_type |= TCT_EXT;
    break;
  case pt_weak_coupling:
    if (p->ensemble == en_NVT) {
      PTC_set_weak_coupling_T(&_md_sys, p->temperature, p->tau_T);
    } else if (p->ensemble == en_NPT) {
      PTC_set_weak_coupling_T(&_md_sys, p->temperature, p->tau_T);
      PTC_set_weak_coupling_P(&_md_sys, p->pressure, p->tau_P, p->compress);
    }
    break;
  case pt_constraint:
    if (p->ensemble == en_NVT) {
      PTC_set_constraint_T(&_md_sys, p->temperature);
    } else if (p->ensemble == en_NPT) {
      lprintf("PT_CONTROL: ERROR: NPT ensemble not supported in constraint method.\n");
    }
    break;
  case pt_rescaling:
    if (p->ensemble == en_NVT) {
      PTC_set_rescaling_T(&_md_sys, p->temperature, p->delta_T);
    } else if (p->ensemble == en_NPT) {
      lprintf("PT_CONTROL: ERROR: NPT ensemble not supported in rescaling method.\n");
    }
    break;
  default:
    return;
  }

  if (p->gradual_change_step != 0) {
    PTC_set_gradual_change_T(&_md_sys, p->gradual_change_step, 
			     p->temperature1);
  }
  
  lprintf("\n");
}

/***************************************
 * section [nonbond]                   *
 ***************************************/
void init_sec_nonbond()
{
  char *env_npx;

  _ctrl.nonbond.flag = 0;

  _ctrl.nonbond.cutoff        = 9.0;
  _ctrl.nonbond.list_margin   = 1.5;
  _ctrl.nonbond.smooth_margin = 2.0;
  _ctrl.nonbond.group_margin  = 2.5;
  _ctrl.nonbond.cell_div = 0;
  _ctrl.nonbond.scee = 0.0;
  _ctrl.nonbond.scnb = 0.0;
  _ctrl.nonbond.diel = 1.0;
  _ctrl.nonbond.list_method = NL_LINKED_CELL;
  _ctrl.nonbond.vdw_method  = NV_PSW;
  _ctrl.nonbond.elec_method = NE_FSH;
  _ctrl.nonbond.update_step = 10;
  _ctrl.nonbond.n_expair = 0;
  _ctrl.nonbond.vir_cor = 0;  /* FALSE */

  if ((env_npx = getenv("MBL_NPX")) != NULL) {
    int d1, d2, d3;
    sscanf(env_npx,"%d %d %d", &d1, &d2, &d3);
    _md_sys.linked_cell.req_npx[0] = d1;
    _md_sys.linked_cell.req_npx[1] = d2;
    _md_sys.linked_cell.req_npx[2] = d3;
    _md_sys.linked_cell.request_npx = 1;
  }
}

void ctrl_nonbond(CFILE *cfp, char *next_section)
{
  char buf[CFILE_BUF_LEN], buf2[CFILE_BUF_LEN];
  double f, f2;
  int i, d, d1,d2,d3;

  /* initialize... */
  _ctrl.nonbond.flag = 1;
  
  /* read ... */
  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    if (sscanf(buf," list_method = %s", buf2) == 1) {
      _ctrl.nonbond.list_method = get_id_sel_data(_sel_nl_method, buf2);
      continue;
    }
    /*
    if (sscanf(buf," vdw_smoothing = %s", buf2) == 1) {
      _ctrl.nonbond.vdw_method = get_id_sel_data(_sel_nv_method, buf2);
      continue;
    }
    if (sscanf(buf," elec_smoothing = %s", buf2) == 1) {
      _ctrl.nonbond.elec_method = get_id_sel_data(_sel_ne_method, buf2);
      continue;
    }
    */
    if (sscanf(buf," vdw_method = %s", buf2) == 1) {
      _ctrl.nonbond.vdw_method = get_id_sel_data(_sel_nv_method, buf2);
      continue;
    }
    if (sscanf(buf," elec_method = %s", buf2) == 1) {
      _ctrl.nonbond.elec_method = get_id_sel_data(_sel_ne_method, buf2);
      continue;
    }
    if (sscanf(buf," virial_correction = %s", buf2) == 1 || 
        sscanf(buf," vdw_long_range_correction = %s", buf2) == 1) {
      _ctrl.nonbond.vir_cor = get_id_sel_data(_sel_TF, buf2);
      continue;
    }
    if (sscanf(buf," cutoff = %lf", &f) == 1) {_ctrl.nonbond.cutoff=f;continue;}
    if (sscanf(buf," list_margin = %lf", &f) == 1) {_ctrl.nonbond.list_margin=f;continue;}
    if (sscanf(buf," smooth_margin = %lf", &f) == 1) {_ctrl.nonbond.smooth_margin=f;continue;}
    if (sscanf(buf," group_margin = %lf", &f) == 1) {_ctrl.nonbond.group_margin=f;continue;}
    if (sscanf(buf," cell_margin = %lf", &f) == 1) {_ctrl.nonbond.cell_margin=f;continue;}
    if (sscanf(buf," cell_division = %s", buf2) == 1) {
      _ctrl.nonbond.cell_div = get_id_sel_data(_sel_cell_div, buf2);
      continue;
    }
    if (sscanf(buf," list_update_step = %d", &d) == 1) {_ctrl.nonbond.update_step=d;continue;}
    if (sscanf(buf," scee = %lf", &f) == 1) {_ctrl.nonbond.scee=f;continue;}
    if (sscanf(buf," scnb = %lf", &f) == 1) {_ctrl.nonbond.scnb=f;continue;}
    if (sscanf(buf," diel = %lf", &f) == 1) {_ctrl.nonbond.diel=f;continue;}
    if (sscanf(buf," excluded_pair = %d %d", &d, &d2) == 2) {
      _ctrl.nonbond.expair[_ctrl.nonbond.n_expair][0] = d;
      _ctrl.nonbond.expair[_ctrl.nonbond.n_expair][1] = d2;
      _ctrl.nonbond.n_expair++;
      continue;
    }

#ifdef MPI_SDMD
    if (sscanf(buf," n_cell = %d %d %d", &d1, &d2, &d3) == 3) {
      _md_sys.linked_cell.n_grid[0] = d1;
      _md_sys.linked_cell.n_grid[1] = d2;
      _md_sys.linked_cell.n_grid[2] = d3;
      _md_sys.linked_cell.request_grid = 1;
      continue;
    }

    if (sscanf(buf," npx = %d %d %d", &d1, &d2, &d3) == 3 ||
	sscanf(buf," n_pe = %d %d %d", &d1, &d2, &d3) == 3) {
      _md_sys.linked_cell.req_npx[0] = d1;
      _md_sys.linked_cell.req_npx[1] = d2;
      _md_sys.linked_cell.req_npx[2] = d3;
      _md_sys.linked_cell.request_npx = 1;
      continue;
    }

    if (sscanf(buf," n_neighbor = %d %d %d", &d1, &d2, &d3) == 3) {
      _md_sys.linked_cell.neighbor[0] = d1;
      _md_sys.linked_cell.neighbor[1] = d2;
      _md_sys.linked_cell.neighbor[2] = d3;
      _md_sys.linked_cell.request_grid = 0;
      continue;
    }

    if (sscanf(buf," load_balance = %s", buf2) == 1) {
      _md_sys.linked_cell.load_balancer_flag = get_id_sel_data(_sel_TF, buf2);
      continue;
    }

    if (sscanf(buf," n_partition = %d", &d) == 1) {
      _md_sys.linked_cell.n_part = d;
      continue;
    }

#endif
    
    if (sscanf(buf," [%[^]]", next_section) == 1) break;

    ctrl_check_error(buf);
  }
}

void exec_nonbond()
{
  int i;
  struct sec_nonbond *p;

  p = &_ctrl.nonbond;

  if (p->list_method == NL_RESIDUE_BASED) {
    p->elec_method  = NE_RPSW;
  }
  if (_ctrl.ewald.flag) {
    p->elec_method = NE_EWALD;
    /*
    if (_ctrl.fmm.flag) {
      lprintf("ERROR: Both ewald and FMM can not used.\n");
      marble_exit(1);
    }
    */
  }
  /*
  if (_ctrl.fmm.flag) {
    p->list_method =  NL_FMM;
    p->elec_method =  NE_FMM;
    if (p->update_step % _ctrl.fmm.update_step != 0) {
      lprintf("ERROR: Pairlist update step (%d) must be a multiple of FMM update step (%d).\n",
	      p->update_step, _ctrl.fmm.update_step);
      marble_exit(1);
    }
  }
  */
  /*
  if (_ctrl.constraint.flag &&
      (_ctrl.constraint.n_rigid_body > 0 ||
       _ctrl.constraint.rattle != RATTLE_NONE)) {
    p->cell_div = get_id_sel_data(_sel_cell_div,"group");
  }

  NONBOND_LIST_set_group_flag(&_md_sys.nonbond, &_md_sys.atom, p->cell_div);
  */
  
  NONBOND_LIST_set_param(&_md_sys.nonbond, &_md_sys.boundary, &_md_sys.atom,
			 p->list_method, p->vdw_method, p->elec_method,
			 p->cutoff, p->smooth_margin, p->list_margin,
			 p->group_margin, p->cell_margin, 
			 p->update_step, p->vir_cor);

  /* check consistency of list_method and elec_method */
  if ((p->list_method == NL_ATOM_BASED    && p->elec_method == NE_RPSW) ||
      (p->list_method == NL_LINKED_CELL   && p->elec_method == NE_RPSW) ||
      (p->list_method == NL_RESIDUE_BASED && p->elec_method != NE_RPSW)) {
    lprintf("  ERROR: \"%s\" is not consistent with \"%s\".\n",
	    get_name_sel_data(_sel_nl_method, p->list_method),
	    get_name_sel_data(_sel_ne_method, p->elec_method));
    marble_exit(1);
  }
  
  if (p->diel != 1.0) {
    if (p->diel < 0.0) {
      lprintf("ERROR: Dielectric constant must be positive.\n");
      marble_exit(1);
    }
    MD_SYSTEM_set_diel(&_md_sys, p->diel);
  }
  lprintf("Dielectric Constant: %.1f\n\n", p->diel);
  
  if (p->scee > 0.0) {
    MD_SYSTEM_set_scee(&_md_sys, p->scee);
    lprintf("1-4 Screen Factor of Elecrostatic Potential:   seee = %.1f\n\n", p->scee);
  }
  
  if (p->scnb > 0.0) {
    MD_SYSTEM_set_scnb(&_md_sys, p->scnb);
    lprintf("1-4 Screen Factor of van der Waals Potential:  scnb = %.1f\n\n", p->scnb);
  }
  
  if (p->n_expair > 0) {
    lprintf("Excluded Nonbond Pair:\n");
    for (i=0;i<p->n_expair;i++) {
      ATOM_DATA_add_excluded_pair(&_md_sys.atom, p->expair[i][0],
				  p->expair[i][1]);
    }
    lprintf("\n");
  }
}

/***************************************
 * section [ewald]                     *
 ***************************************/
void init_sec_ewald()
{
  _ctrl.ewald.flag = 0;

  _ctrl.ewald.method = ew_PME;
  _ctrl.ewald.diel_sur= 0.0;
  _ctrl.ewald.grid[0] = -1;
  _ctrl.ewald.d_grid = 0.0;
  _ctrl.ewald.n_spline = 4;
  _ctrl.ewald.tolerance = 1.0e-6;
  _ctrl.ewald.tol_beta_flag = 1;
  _ctrl.ewald.max_m = -1;
  _ctrl.ewald.erfc_resolution = 10000;
#ifdef MPI_SDMD  
  _ctrl.ewald.pme_n_pe = 0;
#endif  
}

void ctrl_ewald(CFILE *cfp, char *next_section)
{
  char buf[CFILE_BUF_LEN],buf2[CFILE_BUF_LEN],buf3[CFILE_BUF_LEN];
  int d, d0, d1, d2;
  double f;
  
  _ctrl.ewald.flag = 1;

  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    if (sscanf(buf," method = %s", buf2) == 1) {
      _ctrl.ewald.method = (ewald_method_t) get_id_sel_data(_sel_ew_method,buf2);
      continue;
    }
    if (sscanf(buf," surroundings = %s", buf3) == 1) {
      if (strcmp(buf3,"vacuum") == 0) {
	_ctrl.ewald.diel_sur = 1.0;
      } else if (strcmp(buf3,"metal") == 0) {
	_ctrl.ewald.diel_sur = 0.0;
      } else if (strcmp(buf3,"water") == 0) {
	_ctrl.ewald.diel_sur = 80.0;
      } else {
	_ctrl.ewald.diel_sur = atof(buf3);
      }
      continue;
    }
    if (sscanf(buf," beta = %lf", &f) == 1) {
      _ctrl.ewald.beta=f;
      _ctrl.ewald.tol_beta_flag = 0;
      continue;
    }
    if (sscanf(buf," tolerance = %lf", &f) == 1) {
      _ctrl.ewald.tolerance=f;
      _ctrl.ewald.tol_beta_flag = 1;
      continue;
    }
    if (sscanf(buf," grid = %d %d %d", &d0,&d1,&d2) == 3) {
      _ctrl.ewald.grid[0] = d0;
      _ctrl.ewald.grid[1] = d1;
      _ctrl.ewald.grid[2] = d2;
      continue;
    }
    if (sscanf(buf," d_grid = %lf", &f) == 1) {
      _ctrl.ewald.d_grid=f;
      continue;
    }
    if (sscanf(buf," n_spline = %d", &d) == 1) {_ctrl.ewald.n_spline=d;continue;}
    if (sscanf(buf," erfc_resolution = %d", &d) == 1) {
      _ctrl.ewald.erfc_resolution = d;
      continue;
    }
    
#ifdef MPI_SDMD
    if (sscanf(buf," pme_n_pe = %d", &d) == 1) {_ctrl.ewald.pme_n_pe=d;continue;}
#endif    
    
    if (sscanf(buf," max_m = %d", &d) == 1) {_ctrl.ewald.max_m=d;continue;}

    if (sscanf(buf," opt_infl_func = %s", buf2) == 1) {
      _md_sys.ewald.opt_infl = get_id_sel_data(_sel_TF, buf2);
      continue;}

    if (sscanf(buf," uniform_neutralizing_plasma = %s", buf2) == 1) {
      _md_sys.ewald.self_energy_v_flag = get_id_sel_data(_sel_TF, buf2);
      continue;}

    if (sscanf(buf," uniform_background = %s", buf2) == 1) {
      _md_sys.ewald.self_energy_v_flag = get_id_sel_data(_sel_TF, buf2);
      continue;}
    
    if (sscanf(buf," [%[^]]", next_section) == 1) break;

    ctrl_check_error(buf);
  }
}

void exec_ewald()
{
  struct sec_ewald *p;

  p = &_ctrl.ewald;
  if (!p->flag) return;
  if (_ctrl.nonbond.list_method != NL_ATOM_BASED && _ctrl.nonbond.list_method != NL_LINKED_CELL) {
    lprintf("ERROR: \"atom_based\" or \"linked_cell\" is required for nonbond list method in Ewald calculation.\n");
    marble_exit(1);
  }
  switch (p->method) {
  case ew_PME:
    if (p->grid[0]<0 && p->d_grid == 0.0) {
      lprintf("ERROR: Grid or Delta Grid (d_grid) for PME must be specified.\n");
      marble_exit(1);
    }

#ifdef MPI_SDMD    
    if (p->tol_beta_flag)
      SDMD_EW_pme_init(&_md_sys.ewald, &_md_sys.boundary, &_md_sys.nonbond, &_md_sys.atom,
		       p->tolerance, 1, p->grid, p->d_grid,
		       p->diel_sur, p->n_spline, p->erfc_resolution, p->pme_n_pe);
    else
      SDMD_EW_pme_init(&_md_sys.ewald, &_md_sys.boundary, &_md_sys.nonbond, &_md_sys.atom,
		       p->beta, 0, p->grid, p->d_grid,
		       p->diel_sur, p->n_spline, p->erfc_resolution, p->pme_n_pe);
#else
    if (p->tol_beta_flag)
      EW_pme_init(&_md_sys.ewald, &_md_sys.boundary, &_md_sys.nonbond, &_md_sys.atom,
		  p->tolerance, 1, p->grid[0], p->grid[1], p->grid[2],
		  p->diel_sur, p->n_spline, p->erfc_resolution);
    else
      EW_pme_init(&_md_sys.ewald, &_md_sys.boundary, &_md_sys.nonbond, &_md_sys.atom,
		  p->beta, 0, p->grid[0], p->grid[1], p->grid[2],
		  p->diel_sur, p->n_spline, p->erfc_resolution);
#endif    
    break;
  case ew_normal:
    lprintf(" ERROR: normal ewald is not supported yet.\n");
    marble_exit(1);
  default:
    lprintf(" ERROR: unknown ewald method\n");
    marble_exit(1);
  }
  lprintf("\n");
}

/***************************************
 * section [FMM]                       *
 ***************************************/
void init_sec_fmm()
{
  _ctrl.fmm.flag = 0;

  _ctrl.fmm.n_multipole = 4;
  _ctrl.fmm.unit_cell_size = 6.0;
  _ctrl.fmm.neighbor_cell = 2;
  _ctrl.fmm.update_step = 1;
}

void ctrl_fmm(CFILE *cfp, char *next_section)
{
  char buf[CFILE_BUF_LEN],buf2[CFILE_BUF_LEN],buf3[CFILE_BUF_LEN];
  int d;
  double f;
  
  _ctrl.fmm.flag = 1;

  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    
    if (sscanf(buf," n_multipole = %d", &d) == 1) {_ctrl.fmm.n_multipole=d;continue;}
    if (sscanf(buf," unit_cell_size = %lf", &f) == 1) {_ctrl.fmm.unit_cell_size=f;continue;}
    if (sscanf(buf," neighbor_cell = %d", &d) == 1) {_ctrl.fmm.neighbor_cell=d;continue;}
    if (sscanf(buf," update_step = %d", &d) == 1) {_ctrl.fmm.update_step=d;continue;}
    
    if (sscanf(buf," [%[^]]", next_section) == 1) break;

    ctrl_check_error(buf);
  }
}

void exec_fmm()
{
  struct sec_FMM *p;

  p = &_ctrl.fmm;
  
  if (!p->flag) return;

  _md_sys.fmm_flag = 1;
  /*
  if (fmm_init_from_boundary(&_md_sys.boundary, p->n_multipole, p->unit_cell_size, p->neighbor_cell, p->update_step) < 0) {
    marble_exit(1);
  }
  */
  lprintf("ERROR: FMM is not supported in this version.\n");
  lprintf("\n");
  marble_exit(1);
}

/***************************************
 * section [min]                       *
 ***************************************/
void init_sec_min()
{
  struct sec_min *p;

  p = &_ctrl.min;

  p->flag = 0;
  
  p->step = 0;
  p->init_step_len = 0.01;
  p->largest_step_len = 0.5;
  p->grad = 1.0e-4;
  p->print_step = 1;
  
  p->cg_step = 0;
  p->cg_init_step    = 1.0e-6;
  p->cg_largest_step = 1.0e-2;
  p->cg_line_min_tol = 1.0e-4;
}

void ctrl_min(CFILE *cfp, char *next_section)
{
  char buf[CFILE_BUF_LEN];
  int d;
  double f;
  struct sec_min *p;

  p = &_ctrl.min;
  
  p->flag = 1;

  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    
    if (sscanf(buf," step = %d", &d) == 1)    {p->step=d;continue;}
    if (sscanf(buf," sd_step = %d", &d) == 1) {p->step=d;continue;}
    if (sscanf(buf," print_step = %d", &d) == 1) {p->print_step=d;continue;}
    if (sscanf(buf," sd_init_step    = %lf", &f) == 1) {p->init_step_len=f;continue;}
    if (sscanf(buf," sd_largest_step = %lf", &f) == 1) {p->largest_step_len=f;continue;}
    if (sscanf(buf," grad = %lf", &f) == 1) {p->grad=f;continue;}
    
    if (sscanf(buf," cg_step = %d", &d) == 1) {p->cg_step=d;continue;}
    if (sscanf(buf," cg_init_step    = %lf", &f) == 1) {p->cg_init_step=f;continue;}
    if (sscanf(buf," cg_largest_step = %lf", &f) == 1) {p->cg_largest_step=f;continue;}
    if (sscanf(buf," cg_line_min_tolerance = %lf", &f) == 1) {p->cg_line_min_tol=f;continue;}
    
    if (sscanf(buf," [%[^]]", next_section) == 1) break;

    ctrl_check_error(buf);
  }
}

void exec_min()
{
  struct sec_min *p;

  p = &_ctrl.min;

  if (!p->flag) return;

  MIN_main(&_md_sys, p->step, p->init_step_len,p->largest_step_len, p->grad,
	   p->print_step, p->cg_step, p->cg_init_step, p->cg_largest_step,
	   p->cg_line_min_tol);
  
  _ctrl.md.flag = 0;
}


/***************************************
 * section [md]                        *
 ***************************************/
void init_sec_md()
{
  struct sec_md *p;

  p = &_ctrl.md;

  p->flag = 0;

  p->step = p->step0 = p->step1 = p->step2 = p->step3 = 0;
  p->time_step = 0.5;
  p->prop_out[0] = '\0';
  p->prop_out_step = 100;
  p->prop_out_flag = 0;
  p->trj_out[0] = '\0';
  p->trj_out_step = 100;
  p->trj_res_no = -1;
  p->trj_out_type = 0;  /* MD_SYSTEM_crdstr_to_flag("X"); */
  p->trj_out_format = get_id_sel_data(_sel_format, "text");
  p->gzip_trj = 0;
  p->trj_wrap_mol = 0;
  p->trj_amber_boundary = 0;
  p->init_velocity = -1.0;
  p->print_step = 1;
  p->remove_momentum = -1;
  p->remove_momentum_step = 1000;

  p->test_force_iatom = -1;
  p->test_virial = -1;
}

void ctrl_md(CFILE *cfp, char *next_section)
{
  struct sec_md *p;
  char buf[CFILE_BUF_LEN],buf1[CFILE_BUF_LEN],buf2[CFILE_BUF_LEN],buf3[CFILE_BUF_LEN],buf4[CFILE_BUF_LEN];
  int  i,n,np;
  int d, d2;
  double f, f1, f2, f3;
  
  p = &_ctrl.md;
  p->flag = 1;
    
  while (ctrl_fgets(buf, CFILE_BUF_LEN, cfp)!=NULL) {
    if (sscanf(buf," time_step = %lf", &f) == 1) {p->time_step=f;continue;}
    if (sscanf(buf," step = %d", &d) == 1) {p->step=d;continue;}
    if (sscanf(buf," step0 = %d", &d) == 1) {p->step0=d;continue;}
    if (sscanf(buf," step1 = %d", &d) == 1) {p->step1=d;continue;}
    if (sscanf(buf," step2 = %d", &d) == 1) {p->step2=d;continue;}
    if (sscanf(buf," step3 = %d", &d) == 1) {p->step3=d;continue;}
    if (sscanf(buf," prop_file = %s", buf1) == 1) {
      strcpy(p->prop_out,buf1); continue;
    }
    if (sscanf(buf," prop_step = %d", &d) == 1) {
      p->prop_out_step = d; continue;
    }
    if (sscanf(buf," prop_output = %[^\n]", buf1) == 1) {
      n=0;
      while (sscanf(&buf1[n],"%s%n",buf2,&np) == 1) {
	n+=np;
	p->prop_out_flag |= get_id_sel_data(_sel_prop_out, buf2);
      }
      continue;
    }
    if (sscanf(buf," trj_file = %s", buf1) == 1) {
      strcpy(p->trj_out,buf1); continue; }
    if (sscanf(buf," trj_step = %d", &d) == 1) {
      p->trj_out_step=d; continue; }
    if (sscanf(buf," trj_residue = %s", buf1) == 1) {
      if (buf1[0] == 'a') p->trj_res_no = -1;
      else if (buf1[0] == 's') p->trj_res_no = -2;
      else p->trj_res_no = atoi(buf1); continue; }
    if (sscanf(buf," trj_output = %s", buf1) == 1) {
      p->trj_out_type = MD_SYSTEM_trjstr_to_flag(buf1); continue; }
    if (sscanf(buf," trj_format = %s", buf1) == 1) {
      p->trj_out_format = get_id_sel_data(_sel_format, buf1);
      continue;
    }
    if (sscanf(buf," gzip_trj_file = %s", buf1) == 1) {
      p->gzip_trj = get_id_sel_data(_sel_TF, buf1);
      continue;
    }
    if (sscanf(buf," trj_wrap_molecules = %s", buf1) == 1) {
      p->trj_wrap_mol = get_id_sel_data(_sel_TF, buf1);
      continue;
    }
    if (sscanf(buf," trj_amber_boundary = %s", buf1) == 1) {
      p->trj_amber_boundary = get_id_sel_data(_sel_TF, buf1);
      continue;
    }
    if (sscanf(buf," trj_xst_output = %s", buf1) == 1) {
      p->trj_xst_flag = get_id_sel_data(_sel_TF, buf1);
      continue;
    }
    
    if (sscanf(buf," print_step = %d", &d) == 1) {p->print_step=d;continue;}
    if (sscanf(buf," init_velocity = %lf", &f) == 1 ||
	sscanf(buf," init_temperature = %lf", &f) == 1) {
      p->init_velocity=f; continue;
    }
    if (sscanf(buf," remove_momentum = %s", buf1) == 1) {
      p->remove_momentum = get_id_sel_data(_sel_remove_momentum, buf1);
      continue;
    }
    if (sscanf(buf," remove_momentum_step = %d", &d) == 1) {
      p->remove_momentum_step=d;
      continue;
    }
    if (sscanf(buf," test_force = %d %lf %lf %lf", &d, &f1, &f2, &f3) == 4) {
      p->test_force_iatom = d;
      p->test_force_dx = f1;
      p->test_force_dy = f2;
      p->test_force_dz = f3;
      continue;
    }
    if (sscanf(buf," test_virial = %d", &d) == 1) {
      p->test_virial = d;
      continue;
    }
    
#ifdef MPI_SDMD
    if (sscanf(buf," load_balancing_step = %d", &d) == 1) {
      _md_sys.linked_cell.load_balancer_step = d;
      continue;
    }
#endif
    
    if (sscanf(buf," [%[^]]", next_section) == 1) break;

    ctrl_check_error(buf);
  }
}

void exec_md()
{
  struct sec_md *p;

  p = &_ctrl.md;
  if (!p->flag) return;
  
  _ctrl.output.crd_type |= TCT_V;
  
  MD_SYSTEM_set_dt(&_md_sys, p->time_step*0.001);
  if (p->prop_out[0] != '\0')
    MD_SYSTEM_set_prop_out(&_md_sys, p->prop_out, p->prop_out_step,
			   p->prop_out_flag);
  if (p->trj_out[0] != '\0') {
    if (p->trj_out_type == 0) {
      if (_ctrl.output.crd_type & TCT_X)
	p->trj_out_type |= TRJ_X;
      if (_ctrl.output.crd_type & TCT_BOUNDARY) {
	if (p->trj_amber_boundary) {
	  p->trj_out_type |= TRJ_AMBER_BOUNDARY;
	} else {
	  p->trj_out_type |= TRJ_BOUNDARY;
	}
      }
    }
    if (p->trj_out_type & TRJ_BOUNDARY) {
      if (p->trj_amber_boundary) {
	p->trj_out_type &= ~TRJ_BOUNDARY;
	p->trj_out_type |=  TRJ_AMBER_BOUNDARY;
      }
    }
    
    MD_SYSTEM_set_trj_out(&_md_sys, p->trj_out, p->trj_out_step, p->trj_res_no,
			  p->trj_out_format, p->trj_out_type, p->step, 
			  p->gzip_trj, p->trj_wrap_mol, p->trj_xst_flag);
  }

  MD_SYSTEM_set_print_out_step(&_md_sys, p->print_step);


  if (p->test_force_iatom >= 0) {
    MD_SYSTEM_test_force(&_md_sys, p->test_force_iatom-1, 
			 p->test_force_dx,p->test_force_dy,p->test_force_dz);
    marble_exit(1);
  } else if (p->test_force_iatom == -2) {
    MD_SYSTEM_test_force_all(&_md_sys, p->test_force_dx);
    marble_exit(1);
  }

  if (p->test_virial >= 0) {
    MD_SYSTEM_test_virial(&_md_sys, p->test_virial);
    marble_exit(1);
  }

  if (p->init_velocity >= 0.0) {
    if (!_ctrl.input.restart)
      MD_SYSTEM_set_initial_velocity(&_md_sys, p->init_velocity);
    else
      lprintf("Ignoring initial velocity/temperature setting for restarting.\n\n");
  }
  
  MD_SYSTEM_set_remove_momentum(&_md_sys, p->remove_momentum,
				p->remove_momentum_step);

#ifdef MPI_SDMD  
  SDMD_main_loop(&_md_sys, p->step);
#else
  /*
  MTS_main_loop(&_md_sys, p->step0, p->step1, p->step2);
  */
  VERLET_main_loop(&_md_sys, p->step);
#endif  
}



/*
 *  end of routines for sections
 */

void ctrl_check_error(char *buf)
{
  char buf2[CFILE_BUF_LEN];
  buf2[0] = '\0';
  sscanf(buf,"%s",buf2);
  if (buf2[0] == '#' || buf2[0] == '\n' || buf2[0] == '\0') return;

  lprintf("  ERROR: unrecognized line: %s\n", buf);
  marble_exit(1);
}


void scan_double_list(char *buf1, double **pdata, int *pn)
{
  char *pbuf;
  double f, f1, f2, f3, *data;
  int n, pp;
  
  pbuf=buf1; n = 0;
      
  while (1) {
    /* count amount of data  */
    if (sscanf(pbuf,"%lf-%lf;%lf%n",&f1,&f2,&f3,&pp) == 3) {
      for (f=f1;f<=f2;f+=f3) n++;
      pbuf += pp;
    } else if (sscanf(pbuf,"%lf %n",&f1,&pp) == 1) {
      n++;
      pbuf += pp;
    } else break;
  }
  /* allocate memory */
  data = emalloc("scan_double_list",sizeof(double)*n);
  pbuf=buf1; n = 0;
  while (1) {
    /* input data */
    if (sscanf(pbuf,"%lf-%lf;%lf%n",&f1,&f2,&f3,&pp) == 3) {
      for (f=f1;f<=f2;f+=f3) data[n++] = f;
      pbuf += pp;
    } else if (sscanf(pbuf,"%lf %n",&f1,&pp) == 1) {
      data[n++]=f1;
      pbuf += pp;
    } else break;
  }
  *pdata = data;
  *pn = n;
}

void scan_int_list(char *buf1, int **pdata, int *pn)
{
  char *pbuf;
  int d, d1, d2, d3, *data;
  int n, pp;
  
  pbuf=buf1; n = 0;
      
  while (1) {
    /* count amount of data  */
    if (sscanf(pbuf,"%d-%d;%d%n",&d1,&d2,&d3,&pp) == 3) {
      for (d=d1;d<=d2;d+=d3) n++;
      pbuf += pp;
    } else if (sscanf(pbuf,"%d %n",&d1,&pp) == 1) {
      n++;
      pbuf += pp;
    } else break;
  }
  /* allocate memory */
  data = emalloc("scan_double_list",sizeof(double)*n);
  pbuf=buf1; n = 0;
  while (1) {
    /* input data */
    if (sscanf(pbuf,"%d-%d;%d%n",&d1,&d2,&d3,&pp) == 3) {
      for (d=d1;d<=d2;d+=d3) data[n++] = d;
      pbuf += pp;
    } else if (sscanf(pbuf,"%d %n",&d1,&pp) == 1) {
      data[n++]=d1;
      pbuf += pp;
    } else break;
  }
  *pdata = data;
  *pn = n;
}

void scan_string_list(char *buf1, char ***pdata, int *pn)
{
  char *pbuf, buf[CFILE_BUF_LEN], **data;
  int n, pp;
  
  pbuf=buf1; n = 0;
      
  while (1) {
    /* count amount of data  */
    if (sscanf(pbuf,"%s %n",buf,&pp) == 1) {
      n++;
      pbuf += pp;
    } else break;
  }
  /* allocate memory */
  data = emalloc("scan_string_list",sizeof(char*)*n);
  pbuf=buf1; n = 0;
  while (1) {
    /* input data */
    if (sscanf(pbuf,"%s %n",buf,&pp) == 1) {
      data[n] = emalloc("scan_string_list",strlen(buf)+1);
      strcpy(data[n],buf); n++;
      pbuf += pp;
    } else break;
  }
  *pdata = data;
  *pn = n;
}

void free_string_list(char **pdata, int n)
{
  int i;

  for (i=0;i<n;i++)
    free(pdata[i]);
  free(pdata);
}

#if 0
char *lfgets(char *buf, int n, FILE *fp)
{
  int i=1, stop_comma = 1;
  char *p;

  p = buf;
  if ((*p = getc(fp)) == EOF)
    return NULL;
  
  do {
    lputc(*p);
    if (*p == '\n') break;
    if (*p == ',' && stop_comma) break;
    if (*p == '#') stop_comma = 0;
    if (i >= n) break;
    p++; i++;
  } while ((*p = getc(fp)) != EOF);
  *p = '\0';
  return buf;
}
#endif

CFILE *ctrl_fopen(char *fname, char *mode)
{
#ifdef MPI

  CFILE *cfp;
  FILE *fp;
  int err;

  if (mpi.master) {

    if ((fp = fopen(fname, mode)) == NULL) {
      cfp = NULL;
      err = 1;
    } else {
      cfp = emalloc("ctrl_fopen", sizeof(CFILE));
      cfp->def_word = NULL;
      cfp->cur_fp = fp;
      cfp->current_depth = 0;
      cfp->if_block_depth = 0;
      
      cfp->stack = emalloc("ctrl_fopen", sizeof(struct s_cfile_stack));
      cfp->stack->next = NULL;
      cfp->stack->fp = fp;
      strcpy(cfp->stack->fname, fname);
      cfp->stack->depth = 0;
      cfp->stack->if_block_depth = 0;
      cfp->master = 1;
      err = 0;
    }
 
    MPI_Bcast(&err, 1, MPI_INT, mpi.master_pe, mpi.all_comm);
  } else {
    /* slave */
    MPI_Bcast(&err, 1, MPI_INT, mpi.master_pe, mpi.all_comm);
    if (err) {
      cfp = NULL;
    } else {
      cfp = emalloc("ctrl_fopen", sizeof(CFILE));
      cfp->master = 0;
    }
  }
   
  return cfp;

#else  /* def MPI */

  CFILE *cfp;
  FILE *fp;

  if ((fp = fopen(fname, mode)) == NULL) return NULL;
  
  cfp = emalloc("ctrl_fopen", sizeof(CFILE));
  cfp->def_word = NULL;
  cfp->cur_fp = fp;
  cfp->current_depth = 0;
  cfp->if_block_depth = 0;

  cfp->stack = emalloc("ctrl_fopen", sizeof(struct s_cfile_stack));
  cfp->stack->next = NULL;
  cfp->stack->fp = fp;
  strcpy(cfp->stack->fname, fname);
  cfp->stack->depth = 0;
  cfp->stack->if_block_depth = 0;
  
 return cfp;

#endif

}

void ctrl_fclose(CFILE *cfp)
{
  struct s_cfile_stack *stack,*pstack;

#ifdef MPI

  if (cfp->master) {
    stack = cfp->stack;
    while (stack != NULL) {
      fclose(stack->fp);
      pstack = stack;
      stack=stack->next;
      free(pstack);
    }
    free(cfp);
  } else {
    /* slave */
    free(cfp);
  }

#else  /* def MPI */

  stack = cfp->stack;
  while (stack != NULL) {
    fclose(stack->fp);
    pstack = stack;
    stack=stack->next;
    free(pstack);
  }
  free(cfp);
#endif

}

int ctrl_feof(CFILE *cfp)
{
#ifdef MPI
  int feof_flag;

  if (cfp->master) {
    if (cfp->current_depth == 0) 
      feof_flag = feof(cfp->cur_fp);
    else
      feof_flag = 0;
  }

  MPI_Bcast(&feof_flag, 1, MPI_INT, mpi.master_pe, mpi.all_comm);

  return feof_flag;
#else /* MPI */
  
  if (cfp->current_depth == 0) 
    return feof(cfp->cur_fp);
  else
    return 0;
#endif
}

struct s_cfile_def_word* search_def_word(CFILE *cfp, char *word)
{
  struct s_cfile_def_word* def_word;
  for (def_word = cfp->def_word; def_word != NULL; def_word=def_word->next) {
    if (strcmp(word, def_word->word) == 0)
      return def_word;
  }
  return NULL;
}

char *ctrl_fgets(char *buf, int n, CFILE *cfp)
{
#ifdef MPI
  char *ret;
  int len;

  if (cfp->master) {
    ret = ctrl_fgets_master(buf, n, cfp);

    if (ret == NULL) {
      len = -1;
    } else {
      len = strlen(buf);
    }
  }

  MPI_Bcast(&len, 1, MPI_INT, mpi.master_pe, mpi.all_comm);
  if (len != -1) {
    MPI_Bcast(buf, len+1, MPI_CHAR, mpi.master_pe, mpi.all_comm);
    return buf;
  } else {
    return NULL;
  }

#else  /* def MPI */
  return ctrl_fgets_master(buf, n, cfp);
#endif
}


char *ctrl_fgets_master(char *buf, int n, CFILE *cfp)
{
  char *fname = "ctrl_fgets_master";
  int i=1, stop_comma = 1, ret;
  char *p;
  struct s_cfile_stack *stack;
  struct s_cfile_def_word *def_word, *dw, *dw_p;
  double len, len2;
  FILE *cur_fp;
  char file_name[FILENAME_LEN];
  char word[CFILE_DEF_WORD_LEN], val[CFILE_DEF_WORD_LEN];
  int omit_if_block;
  int d;

  p = buf;
  cur_fp = cfp->cur_fp;
  if ((*p = getc(cur_fp)) == EOF) {
    if (cfp->current_depth == 0) {
      if (cfp->if_block_depth > 0) {
	lprintf("ERROR: unterminated #if\n");
	marble_exit(1);
      }
      return NULL;
    } else {
      cfp->current_depth--;
      stack = cfp->stack;
      fclose(stack->fp);
      lprintf("# end of including \"%s\"\n", stack->fname);
      if (cfp->if_block_depth > 0) {
	lprintf("ERROR: unterminated #if\n");
	marble_exit(1);
      }
      cfp->stack = cfp->stack->next;
      cfp->cur_fp = cfp->stack->fp;
      cfp->if_block_depth = cfp->stack->if_block_depth;
      free(stack);
      return ctrl_fgets_master(buf, n, cfp);
    }
  }
  
  do {
    if (*p == '\n') break;
    if (*p == ',' && stop_comma) break;
    if (*p == '#') stop_comma = 0;
    if (i >= n - 1) break;
    p++; i++;
  } while ((*p = getc(cur_fp)) != EOF);
  *(p+1) = '\0';

  omit_if_block = 0;

  if (sscanf(buf, " #include \"%[^\n\"]", file_name) == 1) {
    if (cfp->current_depth == CFILE_MAX_STACK) {
      lprintf("ERROR: too many included files\n");
      marble_exit(1);
    }

    stack = emalloc(fname, sizeof(struct s_cfile_stack));
    stack->next = cfp->stack;
    if ((cur_fp = fopen(file_name, "r")) == NULL) {
      lprintf("ERROR: no such file: %s\n", file_name);
      marble_exit(1);
    }
    strcpy(stack->fname, file_name);
    stack->fp = cur_fp;

    stack->next->if_block_depth = cfp->if_block_depth;
    cfp->if_block_depth = stack->if_block_depth = 0;

    cfp->cur_fp = stack->fp;
    cfp->stack = stack;
    cfp->current_depth++;
    stack->depth = cfp->current_depth;

  } else if ((ret = sscanf(buf, " #define %s %[^\n]", word, val)) == 2 ||
	     ret == 1) {
    if (ret==1) {
      val[0] = '\0';
    }
    def_word = emalloc(fname, sizeof(struct s_cfile_def_word));
    def_word->word = emalloc(fname,sizeof(char)*(strlen(word)+1));
    def_word->val = emalloc(fname,sizeof(char)*(strlen(val)+1));
    strcpy(def_word->word, word);
    strcpy(def_word->val, val);
    len = strlen(def_word->word);

    /* long word first */
    for (dw=cfp->def_word,dw_p=NULL; dw!=NULL;dw_p=dw, dw=dw->next) {
      len2 = strlen(dw->word);
      if (len > len2) break;
      if (len == len2) {
	if (strcmp(dw->word, def_word->word) == 0) {
	  lprintf("ERROR: two defined words (%s) are the same.\n", dw->word);
	  marble_exit(1);
	}
      }
    }
    if (dw_p!=NULL)
      dw_p->next = def_word;
    else
      cfp->def_word = def_word;
    def_word->next = dw;

  } else if (sscanf(buf," #ifdef %s", word) == 1) {
    cfp->if_block_depth++;
    if ((def_word = search_def_word(cfp,word)) == NULL) {
      omit_if_block = 1;
    }

  } else if (sscanf(buf," #ifndef %s", word) == 1) {
    cfp->if_block_depth++;
    if ((def_word = search_def_word(cfp,word)) != NULL) {
      omit_if_block = 1;
    }

  } else if (sscanf(buf," #if %s == %[^\n]", word, val) == 2) {
    cfp->if_block_depth++;
    if ((def_word = search_def_word(cfp,word)) != NULL) {
      if (strcmp(def_word->val, val) != 0)
	omit_if_block = 1;
    } else {
      lprintf(buf);
      lprintf("ERROR: undefined word: %s\n", word);
      marble_exit(1);
    }

  } else if (sscanf(buf," #if %s != %[^\n]", word, val) == 2) {
    cfp->if_block_depth++;
    if ((def_word = search_def_word(cfp,word)) != NULL) {
      if (strcmp(def_word->val, val) == 0)
	omit_if_block = 1;
    } else {
      lprintf(buf);
      lprintf("ERROR: undefined word: %s\n", word);
      marble_exit(1);
    }

  } else if (sscanf(buf," #if %s", word) == 1) {
    cfp->if_block_depth++;
    if (strcmp(word,"0") == 0) {
      omit_if_block = 1;
    } else if (strcmp(word,"1") == 0) {
      omit_if_block = 0;
    } else {
      lprintf(buf);
      lprintf("ERROR: unrecogized condition: %s\n", word);
      marble_exit(1);
    }

  } else if (sscanf(buf," #endif%c", &d) == 1) {
    cfp->if_block_depth--;
    if (cfp->if_block_depth < 0) {
      lprintf(buf);
      lprintf("ERROR: #endif without #if\n");
      marble_exit(1);
    }

  } else {
    for (def_word = cfp->def_word; def_word!=NULL; def_word=def_word->next) {
      substitute_word(buf, n, def_word->word, def_word->val);
    }
  }

  if (omit_if_block == 1) {
    lprintf(buf);
    while (1) {
      if (fgets(buf, n, cur_fp) == NULL) {
	lprintf("ERROR: unterminated #if\n");
	marble_exit(1);
      }
      if (sscanf(buf," #endif%c", &d) == 1) {
	cfp->if_block_depth--;
	break;
      } else {
	lprintf("### %s", buf);
      }
    }
  }

  lprintf(buf);
  p = buf + strlen(buf) - 1;
  if (p >= buf && (*p == '\n' || (*p == ','))) {
    *p = '\0';
  }

  return buf;
}



int substitute_word(char *buf, int n, char *word, char *val)
{
  char *p,*pp,*p2,*pword,*pval,*plast, *p2last;
  char buf2[CFILE_BUF_LEN];
  int ok;
  
  p = buf;
  plast = p + n;
  p2last = buf2 + CFILE_BUF_LEN;
  pword = word;
  ok = 0;

  if (*pword == '\0') return 0;

  while (*p) {
    if (*p == *pword) {
      /* start comparing */
      pp = p;
      do {
	pp++;
	pword++;
	if (pp >= plast) break;
      } while (*pword != '\0' && *pp == *pword) ;

      if (*pword == '\0') {
	ok = 1;
	break;
      } else {
	pword = word;
      }
    }
    p++;
    if (p >= plast) break;
  }

  if (!ok) return 0;

  /* start substituting */

  p  = buf;
  p2 = buf2;
  pword = word;
  
  while (*p) {
    if (*p == *pword) {
      /* start comparing */
      pp = p;
      do {
	pp++;
	pword++;
	if (pp >= plast) break;
      } while (*pword != '\0' && *pp == *pword);

      if (*pword == '\0') {
	pval = val;
	while (*pval) {
	  *p2++ = *pval++;
	  if (p2 >= p2last) break;
	}
	p = pp;
	pword = word;
	continue;
      }
      pword = word;
    }
    *p2++ = *p++;
    if (p >= plast || p2 >= p2last) break;
  }
  if (p2 >= buf2 + n) {
    p2 = buf2 + n - 1;
  }
  *p2 = '\0';
  
  strcpy(buf, buf2);
  
  return 1;
}
