/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "md_system.h"

#ifdef MPI_RDMD
#include "parallel.h"
#endif


/*
   Reading crd file :
     marble crd:
       if (user_type & TCT_RMOL)
         read RMOL_DATA if exist
       else
         skip RMOL_DATA

       if (user_type & TCT_EXT)
         read EXT_DATA if exist
       else {
         read EXT_DATA if exist
	 but off
       }
         
     amber crd:
         read using TXT_X
   
*/
int CRD_read_file(MD_SYSTEM *sys, char *crdname, unsigned int user_type)
{
  int ret;
#ifdef MPI
  ATOM_DATA *ad;
  RMOL_DATA *rd;
  
  ad = &sys->atom;
  rd = &sys->rigid_mol;

  if (mpi.master) {
    ret = CRD_read_file_master(sys, crdname, user_type);
  } 

  MPI_Bcast(&ret, 1, MPI_INT, mpi.master_pe, mpi.comm);

  if (ret)
    marble_exit(1);

  MPI_Bcast(&sys->current_time, 1, MPI_DOUBLE, mpi.master_pe, mpi.comm);
  MPI_Bcast(&sys->read_crd_flag, 1, MPI_INT, mpi.master_pe, mpi.comm);

  if (sys->read_crd_flag & TCT_X) {
    MPI_Bcast(ad->x, ad->natom*3, MPI_DOUBLE, mpi.master_pe, mpi.comm);
  }

  if (sys->read_crd_flag & TCT_V) {
    MPI_Bcast(ad->v, ad->natom*3, MPI_DOUBLE, mpi.master_pe, mpi.comm);
  }

  if (sys->read_crd_flag & TCT_BOUNDARY) {
    MPI_Bcast(&sys->boundary.type, 1, MPI_INT, mpi.master_pe, mpi.comm);
    if (sys->boundary.type == PERIODIC_BOUNDARY) {
      MPI_Bcast(sys->boundary.boxv, 9, MPI_DOUBLE, mpi.master_pe, mpi.comm);
      BOUNDARY_make_recip(&sys->boundary);
    }
  }

  if (sys->read_crd_flag & TCT_EXT) {
    PTC_Ex_System_comm_data(sys);
  }

  if (sys->read_crd_flag & TCT_RMOL) {
    if (mpi.master)
      RMOL_DATA_copy_rmolcrd_to_buf(rd);
    MPI_Bcast(rd->crd, rd->n_mol*13, MPI_DOUBLE, mpi.master_pe, mpi.comm);
    RMOL_DATA_copy_rmolcrd_from_buf(rd);
  }

#else
  ret = CRD_read_file_master(sys, crdname, user_type);
#endif

  BOUNDARY_calc_box(&sys->boundary, &sys->atom);
  MD_SYSTEM_degree_of_freedom(sys);
  
  return ret;
}

int CRD_read_file_master(MD_SYSTEM *sys, char *crdname, unsigned int user_type)
{
  FILE *fp;
  char buf[81], type_str[10], user_type_str[10];
  int major, minor;
  unsigned int type, file_type;
  int amber_orginal;
  int i;
  
  fp = fopen(crdname,"r");
  if (fp == NULL) {
    lprintf("ERROR: Can't open crd file \"%s\"\n", crdname);
    return 1;
  }
  
  fgets(buf, 80, fp);
  if (sscanf(buf, "MARBLE CRD FILE (%[A-Z]) Ver. %d.%d %lf",
	     type_str, &major, &minor, &sys->current_time) == 4) {
    /* case of marble crd */
    type = CRD_crdstr_to_flag(type_str);
    lprintf("MARBLE CRD FILE \"%s\" (TYPE:%s) Ver %d.%d\n",
	    crdname, type_str, major, minor);
    if (major != CRD_MAJOR_VERSION || minor != CRD_MINOR_VERSION) {
      lprintf("ERROR: Ver %d.%d required.\n",
	      CRD_MAJOR_VERSION, CRD_MINOR_VERSION);
      return 1;
    }
  } else {
    /* case of amber crd */
    type = TCT_X;
    CRD_flag_to_crdstr(type_str, type);
    lprintf("AMBER CRD FILE \"%s\" (TYPE:%s)\n", crdname, type_str);
  }

  if (type & TCT_X) {
    if (ATOM_DATA_read_coordinate(&(sys->atom), fp) != 0) {
      fclose(fp);
      return 1;
    }
  }
  if (type & TCT_V) {
    if (ATOM_DATA_read_velocity(&(sys->atom), fp) != 0) {
      fclose(fp);
      return 1;
    }
  }
  if (type & TCT_BOUNDARY) {
    if (BOUNDARY_read_data_from_crd_file(&(sys->boundary), fp) != 0) {
      fclose(fp);
      return 1;
    }
  }
  if (type & TCT_EXT) {
    PTC_Ex_System_read_data(sys, fp);
    if (!(user_type & TCT_EXT))
      PTC_set_Ex_System_off(sys);
  }
  if (type & TCT_RMOL) {
    int skip;

    if (user_type & TCT_RMOL)
      skip = 0;
    else
      skip = 1;
    
    if (RMOL_DATA_read_crd(&sys->rigid_mol, fp, skip) != 0) {
      /* the case of failure of reading data */
      RMOL_DATA_crd_to_rg_quan(&sys->rigid_mol, &sys->atom);
      RMOL_DATA_mol_to_room_with_check(&sys->rigid_mol, &sys->atom);
      RMOL_DATA_set_mol_velocity_all(&sys->rigid_mol, &sys->atom);
    } else {
      /* the case of success of reading data */
      RMOL_DATA_mol_to_room_all(&sys->rigid_mol, &sys->atom);
      RMOL_DATA_set_atom_velocity_all(&sys->rigid_mol, &sys->atom);
    }
  } else {
    RMOL_DATA_crd_to_rg_quan(&sys->rigid_mol, &sys->atom);
    RMOL_DATA_mol_to_room_with_check(&sys->rigid_mol, &sys->atom);
    RMOL_DATA_set_mol_velocity_all(&sys->rigid_mol, &sys->atom);
  }
  /*
  if (type & TCT_LAMBDA) {
    if (PERT_read_crd(&sys->pert, fp) != 0) {
      fclose(fp);
      return 1;
    }
  }
  */
  fclose(fp);

  sys->read_crd_flag = type;
  
  lprintf("\n");

  return 0;
}

int CRD_write_file(MD_SYSTEM *sys, char *crdname, unsigned int type)
{
  FILE *fp;
  char type_str[10];

  CRD_flag_to_crdstr(type_str, type);
  
  fp = safe_fopen(crdname,"w");
  if (fp == NULL) {
    lprintf("  ERROR: Can't open crd file %s.\n", crdname);
    return 1;
  }

  fprintf(fp, "MARBLE CRD FILE (%s) Ver. %d.%d %f\n", type_str,
	  CRD_MAJOR_VERSION, CRD_MINOR_VERSION, sys->current_time);

  if (type & TCT_X) {
    ATOM_DATA_write_coordinate(&(sys->atom), fp);
  }
  if (type & TCT_V) {
    RMOL_DATA_set_atom_velocity_all(&(sys->rigid_mol), &(sys->atom));
    ATOM_DATA_write_velocity(&(sys->atom), fp);
  }
  if (type & TCT_BOUNDARY) {
    BOUNDARY_write_data_to_crd_file(&(sys->boundary), fp);
  }
  if (type & TCT_EXT) {
    PTC_Ex_System_write_data(sys, fp);
  }
  if (type & TCT_RMOL) {
    RMOL_DATA_write_crd(&sys->rigid_mol, fp);
  }
  /*
  if (type & TCT_LAMBDA) {
    PERT_write_crd(&sys->pert, fp);
  }
  */
  
  fclose(fp);

  return 0;
}

int CRD_read_bin_file(MD_SYSTEM *sys, char *fname)
{
#if 0  
  FILE *fp;
  char buf[100];
  int n_flex_atom, wat_nmol;
  int i, j;
  int file_version[2];
  
  fp = fopen(fname,"r");
  if (fp == NULL) {
    lprintf("  ERROR: Can't open input binary file %s.\n",fname);
    return 1;
  }
  
  fread(buf,sizeof(char), strlen(bin_header)+1,fp);

  if (strcmp(buf,bin_header) != 0) {
    lprintf("BIN READ: This file is not md binary out format.\n");
    return 1;
  }
  
  fread(file_version,sizeof(int),2,fp);

  if (file_version[0] != bin_version[0] ||
      file_version[1] != bin_version[1]) {
    lprintf("BIN READ: File(%s) version (%d.%d) is not current version (%d.%d).\n", fname, file_version[0],file_version[1], bin_version[0], bin_version[1]);
    return 1;
  }
  
  fread(&sys->current_time, sizeof(double), 1, fp);
  
  fread(&n_flex_atom, sizeof(int), 1,fp);
  if (n_flex_atom != sys->n_flex_atom) {
    lprintf("BIN READ: n_flex_atom: %d (bin file), %d(top file)\n",
	    n_flex_atom, sys->n_flex_atom);
    return 1;
  }
  fread(&wat_nmol, sizeof(int), 1,fp);
  if (wat_nmol != sys->wat_nmol) {
    lprintf("BIN READ: wat_nmol: %d (bin file), %d(top file)\n",
	    wat_nmol, sys->wat_nmol);
    return 1;
  }
  
  fread(sys->atom.x, sizeof(VEC),sys->n_flex_atom, fp);
  fread(sys->atom.v, sizeof(VEC),sys->n_flex_atom, fp);
  for (i=0;i<sys->wat_nmol;i++) {
    fread(&(sys->rigid_mol.mol[i].rg), sizeof(VEC), 1, fp);
    fread(&(sys->rigid_mol.mol[i].vg), sizeof(VEC), 1, fp);
    fread(&(sys->rigid_mol.mol[i].q), sizeof(QUAN), 1, fp);
    fread(&(sys->rigid_mol.mol[i].l), sizeof(VEC), 1, fp);
    fread(&(sys->rigid_mol.mol[i].wp), sizeof(VEC), 1, fp);
    fread(&(sys->rigid_mol.mol[i].wp_half), sizeof(VEC), 1, fp);
  }
  
  fread(&(sys->boundary.box[0]), sizeof(double), 3, fp);
  fread(&(sys->boundary.min[0]), sizeof(double), 3, fp);
  lprintf("BOX: (%.2f, %.2f, %.2f)\n", sys->boundary.box[0],sys->boundary.box[1],sys->boundary.box[2]);
  
  fread(&(sys->target_temperature), sizeof(double), 1, fp);
  lprintf("TARGET TEMPERATURE: %.2f\n", sys->target_temperature);

  fread(&(sys->Ex_System_T_flag), sizeof(int), 1, fp);
  fread(&(sys->Ex_System_P_flag), sizeof(int), 1, fp);

  lprintf("Extended System T flag: %d\n", sys->Ex_System_T_flag);
  lprintf("Extended System P flag: %d\n", sys->Ex_System_P_flag);

  if (sys->Ex_System_T_flag) {
    fread(&(sys->n_ex_system), sizeof(int), 1, fp);
    fread(&(sys->n_chain_T), sizeof(int), 1, fp);
    for (i=0;i<sys->n_ex_system;i++) {
      fread(&(sys->atom_ex_system[i][0]), sizeof(int), 2, fp);
      for (j=0;j<sys->n_chain_T;j++) {
	fread(&(sys->eta[i][j]), sizeof(double), 1, fp);
	fread(&(sys->eta_v[i][j]), sizeof(double), 1, fp);
 	fread(&(sys->Q[i][j]), sizeof(double), 1, fp);
      }
    }
  }
  
  if (sys->Ex_System_P_flag) {
    fread(&(sys->pe), sizeof(double), 1, fp);
    fread(&(sys->W), sizeof(double), 1, fp);
    fread(&(sys->Pex), sizeof(double), 1, fp);
    
    if (sys->Ex_System_T_flag) {
      fread(&(sys->n_etaP), sizeof(int), 1, fp);
      fread(&(sys->etaP[0]), sizeof(double), sys->n_etaP, fp);  
      fread(&(sys->etaP_v[0]), sizeof(double), sys->n_etaP, fp);  
      fread(&(sys->QP[0]), sizeof(double), sys->n_etaP, fp);
    }
  }

  fclose(fp);
  
  RMOL_DATA_mol_to_room(&(sys->rigid_mol), &(sys->atom));
  sys->read_crd_flag = 1;
#endif  
  return 0;
}

int CRD_write_bin_file(MD_SYSTEM *sys, char *fname)
{
#if 0  
  FILE *fp;
  int i,j;

  if ((fp = safe_fopen(fname,"w")) == NULL) {
    lprintf("ERROR: Can't open binary output file %s.\n",fname);
    return 1;
  }
  fwrite(bin_header,sizeof(char),strlen(bin_header)+1,fp);
  fwrite(bin_version,sizeof(int),2,fp);
  fwrite(&sys->current_time, sizeof(double), 1, fp);
  fwrite(&sys->n_flex_atom, sizeof(int), 1,fp);
  fwrite(&sys->wat_nmol, sizeof(int),1, fp);
  fwrite(sys->atom.x, sizeof(VEC),sys->n_flex_atom, fp);
  fwrite(sys->atom.v, sizeof(VEC),sys->n_flex_atom, fp);
  for (i=0;i<sys->wat_nmol;i++) {
    fwrite(&(sys->rigid_mol.mol[i].rg), sizeof(VEC), 1,fp);
    fwrite(&(sys->rigid_mol.mol[i].vg), sizeof(VEC), 1,fp);
    fwrite(&(sys->rigid_mol.mol[i].q), sizeof(QUAN), 1,fp);
    fwrite(&(sys->rigid_mol.mol[i].l), sizeof(VEC), 1,fp);
    fwrite(&(sys->rigid_mol.mol[i].wp), sizeof(VEC), 1,fp);
    fwrite(&(sys->rigid_mol.mol[i].wp_half), sizeof(VEC), 1,fp);
  }
  fwrite(&(sys->boundary.box[0]), sizeof(double), 3, fp);
  fwrite(&(sys->boundary.min[0]), sizeof(double), 3, fp);
  fwrite(&(sys->target_temperature), sizeof(double), 1, fp);
  
  fwrite(&(sys->Ex_System_T_flag), sizeof(int), 1, fp);
  fwrite(&(sys->Ex_System_P_flag), sizeof(int), 1, fp);

  if (sys->Ex_System_T_flag) {
    fwrite(&(sys->n_ex_system), sizeof(int), 1, fp);
    fwrite(&(sys->n_chain_T), sizeof(int), 1, fp);
    for (i=0;i<sys->n_ex_system;i++) {
      fwrite(&(sys->atom_ex_system[i][0]), sizeof(int), 2, fp);
      for (j=0;j<sys->n_chain_T;j++) {
	fwrite(&(sys->eta[i][j]), sizeof(double), 1, fp);
	fwrite(&(sys->eta_v[i][j]), sizeof(double), 1, fp);
	fwrite(&(sys->Q[i][j]), sizeof(double), 1, fp);
      }
    }
  }
  
  if (sys->Ex_System_P_flag) {
    fwrite(&(sys->pe), sizeof(double), 1, fp);
    fwrite(&(sys->W), sizeof(double), 1, fp);
    fwrite(&(sys->Pex), sizeof(double), 1, fp);
    
    if (sys->Ex_System_T_flag) {
      fwrite(&(sys->n_etaP), sizeof(int), 1, fp);
      fwrite(&(sys->etaP[0]), sizeof(double), sys->n_etaP, fp);  
      fwrite(&(sys->etaP_v[0]), sizeof(double), sys->n_etaP, fp);  
      fwrite(&(sys->QP[0]), sizeof(double), sys->n_etaP, fp);
    }
  }
  
  fclose(fp);
#endif  
  
  return 0;
}

unsigned int CRD_crdstr_to_flag(char *crdstr)
{
  char *p;
  unsigned int flag = 0;

  for (p=crdstr;*p!='\0';p++) {
    switch(*p) {
    case 'X':
      flag |= TCT_X; break;
    case 'V':
      flag |= TCT_V; break;
    case 'B':
      flag |= TCT_BOUNDARY; break;
    case 'E':
      flag |= TCT_EXT; break;
    case 'L':
      flag |= TCT_LAMBDA; break;
    case 'R':
      flag |= TCT_RMOL; break;
    }
  }
  
  return flag;
}

void CRD_flag_to_crdstr(char crdstr[10], unsigned int flag)
{
  int i=0;
  if (flag & TCT_X)         crdstr[i++] = 'X';
  if (flag & TCT_V)         crdstr[i++] = 'V';
  if (flag & TCT_BOUNDARY)  crdstr[i++] = 'B';
  if (flag & TCT_EXT)       crdstr[i++] = 'E';
  if (flag & TCT_RMOL)      crdstr[i++] = 'R';
  if (flag & TCT_LAMBDA)    crdstr[i++] = 'L';
  crdstr[i] = '\0';
}
