/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifndef _BOUNDARY_H_INCLUDED
#define _BOUNDARY_H_INCLUDED

#define NO_BOUNDARY       0
#define PERIODIC_BOUNDARY 1
#define SPHERICAL_CAP     2
#define SPHERICAL_VDW     3
#define SSBP              4

/*
typedef enum {
  NO_BOUNDARY, PERIODIC_BOUNDARY, SPHERICAL_CAP, SPHERICAL_VDW, SSBP
} BOUNDARY_TYPE;
*/

typedef struct _s_water_data {
  char *name, *o_name, *h1_name, *h2_name, *m_name;
  int n_atom;
  double bond, angle, shift;
  double wo, wh;
} WATER_DATA;

#ifdef    BOUNDARY_C

WATER_DATA _water_data[] = {
  { "WAT", "O", "H1", "H2", NULL,3, 0.9572, 104.52, 0.0, 15.999, 1.008 },
  { "IP3", "O", "H1", "H2", NULL,3, 0.9572, 104.52, 0.0, 15.999, 1.008 },
  { "TIP3","OH2","H1","H2", NULL,3, 0.9572, 104.52, 0.0, 15.999, 1.008 },
  { "SPC", "O", "H1", "H2", NULL,3, 1.0   , 109.47, 0.0, 15.999, 1.008 },
  { "IP4", "OW","HW1","HW2","M", 4, 0.9572, 104.52, 0.15, 15.999, 1.008 }
};

int _n_water_data = sizeof(_water_data)/sizeof(WATER_DATA);

#else
extern WATER_DATA _water_data[];
extern int _n_water_data;
#endif    /* BOUNDARY_C */

typedef struct {
  int type;

  /* spherical boundary */
  double radius;
  double center[3];
  double ene;

  /* cap force constant */
  double cap_fc;

  /* wall vdw type */
  int vdw_type;
  double *vdw_radius;
  char vdw_sym[5];
  
  /* ssbp */
  double Whsr,Wvdw,Welec,Wfcos;

  /* periodic boundary */
  double beta, min[3], box[3], boxh[3];
  double boxv[3][3], recip[3][3], reclen[3], V, A;
  VEC offset_v[27];
  double long_range_corr;
  int long_range_correction_flag;

  int origin_flag;

  /* for water */
  WATER_DATA *wat_data;
  int wat_atom_order[4];
  int wat_start_atom, wat_start_res, wat_n_mol;

  double density;

} BOUNDARY;

#define CELL_BUFFER 10.0

void BOUNDARY_init(BOUNDARY *bc);
int BOUNDARY_read_data(BOUNDARY *bc, FILE *fp);
void BOUNDARY_set_boxv(BOUNDARY *bc, double a[3], double b[3], double c[3]);
void BOUNDARY_make_boxv(BOUNDARY *bc);
void BOUNDARY_make_recip(BOUNDARY *bc);
void BOUNDARY_scale_boxv(BOUNDARY *bc, double scale);
#ifdef MPI_SDMD
void BOUDNARY_bcast_boxv(BOUNDARY *bc);
#endif
void BOUNDARY_set_offset_v(BOUNDARY *bc);
void BOUNDARY_set_origin(BOUNDARY *bc, int flag);
int BOUNDARY_read_data_from_crd_file(BOUNDARY *bc, FILE *fp);
int BOUNDARY_write_data_to_crd_file(BOUNDARY *bc, FILE *fp);
void BOUNDARY_calc_box(BOUNDARY *bc, ATOM_DATA *ad);
void BOUNDARY_calc_density(BOUNDARY *bc, ATOM_DATA *ad);
void BOUNDARY_shift_box(BOUNDARY *bc, ATOM_DATA *ad);
void BOUNDARY_check_water(BOUNDARY *bc, ATOM_DATA *ad);
int is_water(char *name);

void BOUNDARY_VDW_init(BOUNDARY *bc, ATOM_DATA *ad, char *);
void BOUNDARY_SSBP_init(BOUNDARY *bc, ATOM_DATA *ad);
void BOUNDARY_calc_abc(BOUNDARY *bc, double *a, double *b, double *c,
		       double *alpha, double *beta, double *gamma);
void BOUNDARY_setup_and_print(BOUNDARY *bc, ATOM_DATA *ad);
void BOUNDARY_energy_force(BOUNDARY *bc, ATOM_DATA *ad, double *ene);
void BOUNDARY_CAP_energy_force(BOUNDARY *bc, ATOM_DATA *ad, double *ene);
void BOUNDARY_VDW_energy_force(BOUNDARY *bc, ATOM_DATA *ad, double *ene);

void BOUNDARY_long_range_correction_init(BOUNDARY *bc, ATOM_DATA *ad,
					 double cutoff);
void BOUNDARY_long_range_correction(BOUNDARY *bc, ATOM_DATA *ad, double *ene);

#endif    /* _BOUNDARY_H_INCLUDED */
