/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#define BOUNDARY_C

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "misc.h"
#include "atom.h"
#include "boundary.h"

void BOUNDARY_init(BOUNDARY *bc)
{
  bc->type = NO_BOUNDARY;
  
  bc->center[0] =  bc->center[1] = bc->center[2] = 0.0;
  bc->box[0] = bc->box[1] = bc->box[2] = 0.0;
  bc->boxh[0] = bc->boxh[1] = bc->boxh[2] = 0.0;
  bc->min[0] = bc->min[1] = bc->min[2] = 0.0;
  bc->beta = 90.0;
  bc->long_range_correction_flag = 0;
  bc->long_range_corr = 0.0;

  bc->cap_fc = 1.5;    /* cap constant default */
  bc->vdw_type = -1;
  bc->vdw_radius = NULL;
  
  bc->Whsr = bc->Wvdw = bc->Welec = bc->Wfcos = 0.0;

  bc->wat_n_mol = 0;
}


void BOUNDARY_set_boxv(BOUNDARY *bc, double a[3], double b[3], double c[3])
{
  int i;

  bc->type = PERIODIC_BOUNDARY;
  for (i=0;i<3;i++) {
    bc->boxv[0][i] = a[i];
    bc->boxv[1][i] = b[i];
    bc->boxv[2][i] = c[i];
  }
  bc->min[0] = bc->min[1] = bc->min[2] = 0.0;
  BOUNDARY_make_recip(bc);
  BOUNDARY_set_offset_v(bc);
}

void BOUNDARY_make_boxv(BOUNDARY *bc)
{
  int i, j;
  /* beta is assumed by 90.0 */
  for (i=0;i<3;i++)
    for (j=0;j<3;j++)
      bc->boxv[i][j]=0.0;
  bc->boxv[0][0]=bc->box[0];
  bc->boxv[1][1]=bc->box[1];
  bc->boxv[2][2]=bc->box[2];
}

/*
  boxv[i][xyz]:  box vector i
  recip[xyz][i]: reciprocal vector i
 */
void BOUNDARY_make_recip(BOUNDARY *bc)
{
  int i, j;
  double v01[3], v12[3], v20[3], tmp;

  cross3(bc->boxv[1],bc->boxv[2],v12);
  cross3(bc->boxv[2],bc->boxv[0],v20);
  cross3(bc->boxv[0],bc->boxv[1],v01);
  bc->V = dot3(bc->boxv[0],v12);
  bc->A = bc->boxv[0][0]*bc->boxv[1][1];
  
  for (j=0;j<3;j++) {
    bc->recip[j][0] = v12[j] / bc->V;
    bc->recip[j][1] = v20[j] / bc->V;
    bc->recip[j][2] = v01[j] / bc->V;
  }
  /* DEBUG
  {
    int k;
    for (i=0;i<3;i++)
      for (j=0;j<3;j++) {
	tmp = 0.0;
	for (k=0;k<3;k++) {
	  tmp += bc->boxv[i][k]*bc->recip[k][j];
	}
	lprintf("%d %d: %e\n", i,j,tmp);
      }
  }
  */
  for (i=0;i<3;i++) {
    tmp=0.0;
    for (j=0;j<3;j++)
      tmp+=bc->recip[j][i]*bc->recip[j][i];
    bc->reclen[i]=1.0/sqrt(tmp);
  }
}

void BOUNDARY_scale_boxv(BOUNDARY *bc, double scale)
{
  int i,j;
  for (i=0;i<3;i++) {
    for (j=0;j<3;j++) {
      bc->boxv[i][j] *= scale;
    }
  }
  BOUNDARY_make_recip(bc);
  BOUNDARY_set_offset_v(bc);
}

#ifdef MPI_SDMD
void BOUDNARY_bcast_boxv(BOUNDARY *bc)
{
  MPI_Bcast(bc->boxv, 9, MPI_DOUBLE, mpi.master_pe, mpi.comm);
  BOUNDARY_make_recip(bc);
  BOUNDARY_set_offset_v(bc);
}
#endif

void BOUNDARY_set_offset_v(BOUNDARY *bc)
{
  int i, j, k, id;
  
  for (i=-1;i<=1;i++) {
    for (j=-1;j<=1;j++) {
      for (k=-1;k<=1;k++) {
	id=k+1+3*(j+1+3*(i+1));
	bc->offset_v[id].x = bc->boxv[0][0]*i+bc->boxv[1][0]*j+bc->boxv[2][0]*k;
	bc->offset_v[id].y = bc->boxv[0][1]*i+bc->boxv[1][1]*j+bc->boxv[2][1]*k;
	bc->offset_v[id].z = bc->boxv[0][2]*i+bc->boxv[1][2]*j+bc->boxv[2][2]*k;
      }
    }
  }
}

void BOUNDARY_set_origin(BOUNDARY *bc, int flag)
{
  bc->origin_flag = flag;

}

int BOUNDARY_read_data_from_crd_file(BOUNDARY *bc, FILE *fp)
{
  int i;
  
  fscanf(fp,"%d", &(bc->type));
  
  switch(bc->type) {
  case PERIODIC_BOUNDARY:
    for (i=0;i<3;i++) {
      if (fscanf(fp,"%lf%lf%lf",
		 &(bc->boxv[i][0]),&(bc->boxv[i][1]),&(bc->boxv[i][2])) != 3) {
	lprintf("  ERROR: Format of Boundary box data is wrong.\n");
	return 1;
      }
    }
    BOUNDARY_make_recip(bc);
    lprintf("  Periodic boundary box data are read from crd file.\n");
    break;
  default :
    lprintf(" ERROR: THIS TYPE OF BOUNDARY CONDITION IS NOT SUPPORTED IN CRD FILE.\n");
    return 1;
  }
  return 0;
}

int BOUNDARY_write_data_to_crd_file(BOUNDARY *bc, FILE *fp)
{
  int i;
  fprintf(fp,"%d\n", bc->type);
  
  switch(bc->type) {
  case PERIODIC_BOUNDARY:
    for (i=0;i<3;i++) {
      fprintf(fp,"%lf %lf %lf\n",
	      bc->boxv[i][0],bc->boxv[i][1],bc->boxv[i][2]);
    }
    break;
  default :
    lprintf("ERROR: THIS TYPE OF BOUNDARY CONDITION IS NOT SUPPORTED IN CRD FILE.\n");
    return 1;
  }
  return 0;
}

void BOUNDARY_calc_box(BOUNDARY *bc, ATOM_DATA *ad)
{
  double min[3], max[3], box_width;
  int i, out;
  
  switch(bc->type) {
  case PERIODIC_BOUNDARY:
    return;
  case SPHERICAL_CAP :
  case SPHERICAL_VDW :
  case SSBP :
#if 0
    box_width = 2.0*(bc->radius + CELL_BUFFER);
    bc->min[0] = bc->center[0] - box_width*0.5;
    bc->min[1] = bc->center[1] - box_width*0.5;
    bc->min[2] = bc->center[2] - box_width*0.5;
    max[0] = bc->center[0] + box_width*0.5;
    max[1] = bc->center[1] + box_width*0.5;
    max[2] = bc->center[2] + box_width*0.5;
    bc->box[0] = box_width;
    bc->box[1] = box_width;
    bc->box[2] = box_width;
    out = 0;
    for (i=0;i<ad->natom;i++) {
      if (ad->x[i].x < bc->min[0] || ad->x[i].x > max[0] ||
	  ad->x[i].y < bc->min[1] || ad->x[i].y > max[1] ||
	  ad->x[i].z < bc->min[2] || ad->x[i].z > max[2]) {
	out = 1;
	break;
      }
    }
    if (!out) {
      lprintf("(%f %f %f) - (%f %f %f)\n",
	      bc->min[0],bc->min[1],bc->min[2],
	      max[0],max[1],max[2]);
      break;
    }

#endif    
  case NO_BOUNDARY :
    ATOM_DATA_calc_min_max(ad, min, max);
    lprintf("(%f %f %f) - (%f %f %f)\n", min[0],min[1],min[2],
	    max[0],max[1],max[2]);
    for (i=0;i<3;i++) {
      bc->min[i] = min[i] - CELL_BUFFER;
      bc->box[i] = max[i] - min[i] + CELL_BUFFER*2.0;
    }
    BOUNDARY_make_boxv(bc);
    BOUNDARY_make_recip(bc);
    break;
  }
}

void BOUNDARY_calc_density(BOUNDARY *bc, ATOM_DATA *ad)
{
  double volume;
  
  switch (bc->type) {
  case NO_BOUNDARY :
  case PERIODIC_BOUNDARY :
    volume = bc->box[0]*bc->box[1]*bc->box[2];
    bc->density = (double) ad->natom / volume;
    break;
  case SPHERICAL_CAP :
  case SPHERICAL_VDW :
  case SSBP :
    volume = 4.0/3.0*M_PI*bc->radius*bc->radius*bc->radius;
    bc->density = (double) ad->natom / volume;
    break;
  }
}

void BOUNDARY_shift_box(BOUNDARY *bc, ATOM_DATA *ad)
{
  int i;
  if (bc->min[0] == 0.0 && bc->min[1] == 0.0 && bc->min[2] == 0.0) {
    for (i=0;i<3;i++) {
      bc->min[i] = -bc->box[i]*0.5;
    }
    for (i=0;i<ad->natom;i++) {
      ad->x[i].x -= bc->box[0]*0.5;
      ad->x[i].y -= bc->box[1]*0.5;
      ad->x[i].z -= bc->box[2]*0.5;
    }
    lprintf("BOX is shifted.\n");
  }
}
				    
void BOUNDARY_check_water(BOUNDARY *bc, ATOM_DATA *ad)
{
  int i, j, first, type, iH, iatom, expected, wat_n_mol;
  char c;
  int O_vdw_tmp, H_vdw_tmp, OH_vdw_id, HH_vdw_id;

  first = 1;
  wat_n_mol = 0;
  for (i=0;i<ad->nres;i++) {
    if ((type = is_water(ad->r[i].name)) >= 0) {
      wat_n_mol++;
      ad->r[i].flag = (RES_FLAG_WATER | RES_FLAG_PERIODIC);
      if (first) {
	first = 0;
	bc->wat_start_atom = ad->r[i].start_atom;
	bc->wat_start_res = i;
	bc->wat_data = &_water_data[type];
	iH = 1;
	for (iatom = 0; iatom < ad->r[i].natom; iatom++) {
	  switch ((c = ad->a[ad->r[i].start_atom+iatom].sym[0])) {
	  case 'O' :
	    bc->wat_atom_order[0] = iatom; break;
	  case 'H' :
	    bc->wat_atom_order[iH++] = iatom; 
	    if (iH>3) {
	      lprintf("ERROR: CHECK_WATER: detecting more than 2 hydrogens in one water\n");
	    }
	    break;
	  case 'M' :
	    bc->wat_atom_order[3] = iatom; break;
	  default :
	    lprintf("ERROR: CHECK_WATER: unknown atom symbol %c in water\n", c);
	  }
	}
      } else {
	/* not first time */
	if (bc->wat_data != &_water_data[type]) {
	  lprintf("ERROR: CHECK_WATER: water type is not unique. %s (no:%d) - %s (no:%d)\n",
		  ad->r[bc->wat_start_res].name, bc->wat_start_res+1,
		  ad->r[i].name, i+1);
	}
	iH = 1;
	for (iatom = 0; iatom < ad->r[i].natom; iatom++) {
	  switch ((c = ad->a[ad->r[i].start_atom+iatom].sym[0])) {
	  case 'O' :
	    expected = 0; break;
	  case 'H' :
	    expected = iH++;
	    if (iH>3) {
	      lprintf("ERROR: CHECK_WATER: detecting more than 2 hydrogens in one water\n");
	    }
	    break;
	  case 'M' :
	    expected = 3; break;
	  default :
	    lprintf("ERROR: CHECK_WATER: unknown atom symbol %c in water\n", c);
	  }
	  if (iatom != bc->wat_atom_order[expected]) {
	    lprintf("ERROR: CHECK_WATER: order of atoms in water is not unique\n");
	  }
	}
      }
    }
  }
  bc->wat_n_mol = wat_n_mol;
  
  /* vdw_type is water oxigen */
  bc->vdw_type = ad->vdw_type[bc->wat_start_atom + bc->wat_atom_order[0]];
  strcpy(bc->vdw_sym, ad->a[bc->wat_start_atom + bc->wat_atom_order[0]].sym);

  /* Check OW-HW nonbond interaction: 12-10 -> 12-6 */
  O_vdw_tmp = ad->vdw_type[bc->wat_start_atom + bc->wat_atom_order[0]];
  H_vdw_tmp = ad->vdw_type[bc->wat_start_atom + bc->wat_atom_order[1]];

  OH_vdw_id = ad->index[O_vdw_tmp+H_vdw_tmp*ad->ntype];
  HH_vdw_id = ad->index[H_vdw_tmp+H_vdw_tmp*ad->ntype];
  if (OH_vdw_id < 0) {
    OH_vdw_id = -OH_vdw_id - 2;
    if (ad->hb12[OH_vdw_id] == 0.0 && ad->hb10[OH_vdw_id] == 0.0) {
      if (HH_vdw_id >= 0 &&
	  ad->vdw12[HH_vdw_id] == 0.0 && ad->vdw6[HH_vdw_id] == 0.0) {
	ad->index[O_vdw_tmp+H_vdw_tmp*ad->ntype] = HH_vdw_id;
	ad->index[H_vdw_tmp+O_vdw_tmp*ad->ntype] = HH_vdw_id;
      } else {
	lprintf("ERROR: unable to convert OW-HW -> HW-HW interaction\n");
	lprintf("HH vdw index = %d\n", HH_vdw_id);
      }
    }
  }
#ifndef HBOND
  for (i=0;i<ad->ntype;i++) {
    for (j=0;j<ad->ntype;j++) {
      if (ad->index[i+j*ad->ntype]<0) {
	lprintf("ERROR: This version is not supported 12-10 potential.\n");
	marble_exit(1);
      }
    }
  }
#endif  
}

int is_water(char *name)
{
  int i;
  
  for (i=0;i<_n_water_data;i++) {
    if (strcmp(name, _water_data[i].name) == 0) {
      return i;
    }
  }
  return -1;
}

void BOUNDARY_VDW_init(BOUNDARY *bc, ATOM_DATA *ad, char *sym)
{
  int i, len;
  int vdw_index;
  
  if (bc->vdw_radius) free(bc->vdw_radius);
  bc->vdw_radius = emalloc("BOUNDARY_VDW_init", sizeof(double)*ad->natom);

  if (sym) {
    len = strlen(sym);
    for (i=0;i<ad->natom;i++) {
      if (strncmp(ad->a[i].sym, sym, len) == 0) {
	strcpy(bc->vdw_sym,sym);
	bc->vdw_type = ad->vdw_type[i];
	goto sym_detected;
      }
    }
    lprintf("BOUNDARY VDW: SYMBOL %s was not found.\n",sym);
  }

  if (bc->vdw_type < 0)  {
    lprintf("BOUNDARY VDW: NOT SET\n");
    return;
  }
    
 sym_detected:

  for (i=0;i<ad->natom;i++) {
    vdw_index = ad->index[ad->vdw_type[i]+bc->vdw_type*ad->ntype];
    if (vdw_index >= 0) {
      if (fabs(ad->vdw6[vdw_index]) > EPS) {
	bc->vdw_radius[i] =
	  pow(ad->vdw12[vdw_index]/ad->vdw6[vdw_index], 1.0/6.0);
      } else {
	bc->vdw_radius[i] = 0.0;
      }
    }
  }
  bc->type = SPHERICAL_VDW;
  lprintf("BOUNDARY VDW SELECTED: WALL SYMBOL %s\n", bc->vdw_sym);
}

void BOUNDARY_SSBP_init(BOUNDARY *bc, ATOM_DATA *ad)
{
  /*
  solvent_boundary_init(bc->wat_n_mol,bc->center,ad->q,ad->natom,1);
  bc->type = SSBP;
  */
  lprintf("ERROR: SSBP is not supported in this version.\n");
  marble_exit(1);
}

void BOUNDARY_calc_abc(BOUNDARY *bc, double *a, double *b, double *c,
		       double *alpha, double *beta, double *gamma)
{
  double pbc,pca,pab;
  int i;
  
  *a = sqrt(SQR(bc->boxv[0][0])+SQR(bc->boxv[0][1])+SQR(bc->boxv[0][2]));
  *b = sqrt(SQR(bc->boxv[1][0])+SQR(bc->boxv[1][1])+SQR(bc->boxv[1][2]));
  *c = sqrt(SQR(bc->boxv[2][0])+SQR(bc->boxv[2][1])+SQR(bc->boxv[2][2]));
  pbc = pca = pab = 0.0;
  for (i=0;i<3;i++) {
    pbc += bc->boxv[1][i]*bc->boxv[2][i];
    pca += bc->boxv[2][i]*bc->boxv[0][i];
    pab += bc->boxv[0][i]*bc->boxv[1][i];
  }
  
  *alpha = acos(pbc/((*b)*(*c)))*180.0/M_PI;
  *beta  = acos(pca/((*c)*(*a)))*180.0/M_PI;
  *gamma = acos(pab/((*a)*(*b)))*180.0/M_PI;
}


void BOUNDARY_setup_and_print(BOUNDARY *bc, ATOM_DATA *ad)
{
  int i;
  double a, b, c, alpha, beta, gamma;
  
  switch (bc->type) {
  case SPHERICAL_CAP :
    ad->ex_force_flag = 1;
    lprintf("Spherical Cap Boundary Condition:\n");
    lprintf("  Radius %.2f, Center (%.2f,%.2f,%.2f)\n",
	    bc->radius,bc->center[0],bc->center[1],bc->center[2]);
    lprintf("  Cap Force Constant = %.2f\n",
	    bc->cap_fc);
    break;
  case SPHERICAL_VDW :
    ad->ex_force_flag = 1;
    lprintf("Spherical VDW Wall Boundary Condition:\n");
    lprintf("  Radius %.2f, Center (%.2f,%.2f,%.2f)\nVDW Symbol %s\n",
	    bc->radius,bc->center[0],bc->center[1],bc->center[2],bc->vdw_sym);
    break;
  case SSBP :
    ad->ex_force_flag = 1;
    lprintf("Solvent Boundary Condition:\n");
    lprintf("  Center (%.2f,%.2f,%.2f), Water Name %s\n",
	    bc->center[0],bc->center[1],bc->center[2],bc->wat_data->name);
    
    break;
  case PERIODIC_BOUNDARY :
    lprintf("Periodic Boundary Condition:\n");
    lprintf("  a vector (%7.2f,%7.2f,%7.2f)\n",bc->boxv[0][0],bc->boxv[0][1],bc->boxv[0][2]);
    lprintf("  b vector (%7.2f,%7.2f,%7.2f)\n",bc->boxv[1][0],bc->boxv[1][1],bc->boxv[1][2]);
    lprintf("  c vector (%7.2f,%7.2f,%7.2f)\n",bc->boxv[2][0],bc->boxv[2][1],bc->boxv[2][2]);
    BOUNDARY_calc_abc(bc, &a, &b, &c, &alpha, &beta, &gamma);
    lprintf("  Length: a = %.2f, b = %.2f, c = %.2f\n", a, b, c);
    lprintf("  alpha = %.2f, beta = %.2f, gamma = %.2f\n", alpha, beta, gamma);
    lprintf("  Origin: %s\n", (bc->origin_flag == 0) ? "minimum" : "center");
    break;
  case NO_BOUNDARY :
    lprintf("No Boundary Condition\n");
    break;
  }
}

void BOUNDARY_energy_force(BOUNDARY *bc, ATOM_DATA *ad, double *ene)
{
  switch (bc->type) {
  case SPHERICAL_CAP :
#ifdef MPI
    if (!mpi.master) break;
#endif    
    BOUNDARY_CAP_energy_force(bc, ad, ene);
    break;
  case SPHERICAL_VDW :
#ifdef MPI
    if (!mpi.master) break;
#endif    
    BOUNDARY_VDW_energy_force(bc, ad, ene);
    break;
  case SSBP :
#ifdef MPI
    if (!mpi.master) break;
#endif    
    /*
    solvent_boundary_energy_force(ad->x, ad->f, ad->q,
				  bc->wat_start_atom, bc->wat_n_mol,
				  &(bc->Whsr), &(bc->Wvdw),
				  &(bc->Welec),&(bc->Wfcos));
    *ene = bc->Whsr + bc->Wvdw + bc->Welec + bc->Wfcos;
    */
    break;
  case PERIODIC_BOUNDARY :
#ifdef MPI
    if (!mpi.master) break;
#endif    
    BOUNDARY_long_range_correction(bc, ad, ene);
    break;
  }
}

void BOUNDARY_CAP_energy_force(BOUNDARY *bc, ATOM_DATA *ad, double *ene)
{
  int i;
  double dx, dy, dz;
  double r, dr, r2, rad2;
  double force;

  *ene = 0.0;
  rad2 = bc->radius * bc->radius;
  for (i = 0; i < ad->natom; i++) {
    /* omit hydrogen */
    if (ad->a[i].sym[0] == 'H') continue;
    
    /* calculate vector */
    dx = ad->x[i].x - bc->center[0];
    dy = ad->x[i].y - bc->center[1];
    dz = ad->x[i].z - bc->center[2];

    r2 = Length2(dx, dy, dz);
    if (r2 <= rad2) continue;
    
    r = sqrt(r2);
    dr  = r - bc->radius;
    *ene += bc->cap_fc * SQR(dr);

    force = - 2.0 * bc->cap_fc * dr / r;
    ad->f[i].x += force * dx;
    ad->f[i].y += force * dy;
    ad->f[i].z += force * dz;
  }
}

void BOUNDARY_VDW_energy_force(BOUNDARY *bc, ATOM_DATA *ad, double *ene)
{
  int i;
  double dx, dy, dz;
  double len, len2, len6, len12, r;
  double vdw12, vdw6;
  double force;
  int vdw_index;

  *ene = 0.0;
  for (i = 0; i < ad->natom; i++) {
    /* calculate vector */
    dx = ad->x[i].x - bc->center[0];
    dy = ad->x[i].y - bc->center[1];
    dz = ad->x[i].z - bc->center[2];

    r = sqrt(Length2(dx, dy, dz));
    len  = bc->radius - r + bc->vdw_radius[i];
    len2 = len*len;
    len6 = len2 * len2 * len2;
    len12 = len6 * len6;
    
    vdw_index = ad->index[ad->vdw_type[i]+bc->vdw_type*ad->ntype];
    if (vdw_index >= 0) {
      vdw12 = ad->vdw12[vdw_index] / len12;
      vdw6 = ad->vdw6[vdw_index] / len6;
    
      *ene  += vdw12 - vdw6;
      force = (12.0 * vdw12 - 6.0 * vdw6)/(r*len);

      ad->f[i].x -= force * dx;
      ad->f[i].y -= force * dy;
      ad->f[i].z -= force * dz;
    }
  }
}

void BOUNDARY_long_range_correction_init(BOUNDARY *bc, ATOM_DATA *ad,
					 double cutoff)
{
  int i, j, vdw_index;
  double cutoff3, cutoff9;
  int *n_type_atom;

  bc->long_range_correction_flag = 1;
  cutoff9 = pow(cutoff, 9.0);
  cutoff3 = pow(cutoff, 3.0);
  bc->long_range_corr = 0.0;
#if 0
  for (i=0;i<ad->natom;i++) {
    for (j=0;j<ad->natom;j++) {
      vdw_index = ad->index[ad->vdw_type[i]+ad->vdw_type[j]*ad->ntype];
      if (vdw_index >= 0) {
	bc->long_range_corr += ad->vdw12[vdw_index] / (9.0*cutoff9) - ad->vdw6[vdw_index] / (3.0*cutoff3);
      } else {
	lprintf("12-10 pair is not supported in long range corrlection\n");
	marble_exit(1);
      }
    }
  }
  bc->long_range_corr *= 2.0 * M_PI;
#else
  n_type_atom = emalloc("BOUNDARY_long_range_correction_init",sizeof(int)*ad->ntype);
  for (i=0;i<ad->ntype;i++) {
    n_type_atom[i]=0;
  }
  
  for (i=0;i<ad->natom;i++) {
    n_type_atom[ad->vdw_type[i]]++;
  }
  for (i=0;i<ad->ntype;i++) {
    for (j=0;j<ad->ntype;j++) {
      vdw_index = ad->index[i+j*ad->ntype];
      if (vdw_index >= 0) {
	bc->long_range_corr += (/*ad->vdw12[vdw_index] / (9.0*cutoff9) amber */
	                      - ad->vdw6[vdw_index] / (3.0*cutoff3))
	                       * n_type_atom[i] * n_type_atom[j];
      } else {
	lprintf("12-10 pair is not supported in long range corrlection\n");
	marble_exit(1);
      }
    }
  }
  bc->long_range_corr *= 2.0 * M_PI;
  free(n_type_atom);
#endif
}

void BOUNDARY_long_range_correction(BOUNDARY *bc, ATOM_DATA *ad, double *ene)
{
  if (bc->long_range_correction_flag) {
    *ene = bc->long_range_corr / bc->V;
    

    ad->virial[0] += *ene;
    ad->virial[1] += *ene;
    ad->virial[2] += *ene;
    /*
    ad->virial[3] += 0.0;
    ad->virial[4] += 0.0;
    ad->virial[5] += 0.0;
    */
  }
}
