/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "misc.h"
#include "atom.h"
#include "bond.h"

void BOND_DATA_init(BOND_DATA *bd)
{
  bd->n_bond = bd->n_bond_h = 0;
  bd->bonds = NULL;

  bd->n_bond_type = 0;
  bd->bond_type = NULL;
}

void BOND_DATA_finalize(BOND_DATA *bd)
{
  if (bd->bonds) free(bd->bonds);
  if (bd->bond_type) free(bd->bond_type);
  BOND_DATA_init(bd);
}

void bond_energy_force(BOND_DATA *bond_data, ATOM_DATA *atom_data, 
                       double *Energy)
{
  int i;
  int atom1, atom2, type;
  double k, r0, dx, dy, dz, r, F, e;
  VEC *x, *f;

  x = atom_data->x;
  f = atom_data->f;

#ifdef MPI_RDMD
  for (i = bond_data->start_task; i <= bond_data->end_task; i++) {
#elif defined(MPI_SDMD)
  for (i = bond_data->head; i >=0; i=bond_data->bonds[i].next) {
#else    
  for (i = 0; i < bond_data->n_bond; i++) {
#endif 
    if (bond_data->bonds[i].flag & NO_CALC_BOND) continue;
    type = bond_data->bonds[i].type;
    k     = bond_data->bond_type[type].k;
    r0    = bond_data->bond_type[type].r0;
    atom1 = bond_data->bonds[i].atom1;
    atom2 = bond_data->bonds[i].atom2;
    dx = x[atom2].x - x[atom1].x;
    dy = x[atom2].y - x[atom1].y;
    dz = x[atom2].z - x[atom1].z;
    r = sqrt(dx*dx+dy*dy+dz*dz);
    e = k * (r - r0) * (r - r0);
    *Energy += e;

    /* debug
    if (atom1+1 == 5086 && atom2+1 == 5088) {
      printf("%d %e %e %e %e\n", atom2+1, x[atom2].x, x[atom2].y, x[atom2].z, e);
    }
    if (e > 1000.0) {
      printf("%d %d %e\n", atom1+1, atom2+1, e);
      printf("%d %e %e %e\n", atom1+1, x[atom1].x, x[atom1].y, x[atom1].z);
      printf("%d %e %e %e\n", atom2+1, x[atom2].x, x[atom2].y, x[atom2].z);
    }
    */
#if 0   /* no bond atom ene */   
    if (atom_data->atom_ene_flag) {
      e *= 0.5;
      atom_data->atom_ene[atom1].type[ATOM_BOND_ENE] += e;
      atom_data->atom_ene[atom2].type[ATOM_BOND_ENE] += e;
    }
#endif    
    F = 2.0 * k * (r - r0) / r;
    f[atom1].x += dx * F;
    f[atom1].y += dy * F;
    f[atom1].z += dz * F;
    f[atom2].x -= dx * F;
    f[atom2].y -= dy * F;
    f[atom2].z -= dz * F;
  }
}


void print_bonds(FILE *fp, BOND_DATA *bd)
{
  int i;

  for (i = 0; i < bd->n_bond; i++) {
    fprintf(fp, "%d %d %d\n", bd->bonds[i].atom1,bd->bonds[i].atom2,
	    bd->bonds[i].type);
  }
}

void set_all_bonds_flag(BOND_DATA *bd, ATOM_DATA *ad)
{
  char *p1,*p2;
  int i;
  
  for (i=0;i<bd->n_bond;i++) {
    bd->bonds[i].flag = 0;
    p1 = get_atom_sym(ad,get_bond_atom1(bd,i));
    p2 = get_atom_sym(ad,get_bond_atom2(bd,i));
    if (*p1 == 'H' || *p2 == 'H') {
      if (strcmp(p1,"HW") == 0 || strcmp(p2,"HW") == 0) {
	set_bond_flag(bd,i,WATER_BOND);
      } else {
	set_bond_flag(bd,i,HYDROGEN_INCLUDED);
      }
    }
  }
}


void BOND_DATA_omit_data_for_rigid_mol(BOND_DATA *bd, ATOM_DATA *ad)
{
  int i, atom1, atom2, mol1, mol2, pack_to;

  pack_to = 0;
  for (i=0;i<bd->n_bond;i++) {
    atom1 = bd->bonds[i].atom1;
    atom2 = bd->bonds[i].atom2;

    if ((ad->ex[atom1].flag & ATOM_RIGID) &&
	(ad->ex[atom2].flag & ATOM_RIGID)) {

      mol1 = ATOM_RMOL(ad,atom1);
      mol2 = ATOM_RMOL(ad,atom2);
      if (mol1==mol2) {
	bd->bonds[i].flag |= BOND_IN_RIGID;
	continue;
      }
    }
    
    if ((ad->ex[atom1].flag & ATOM_FIXED) &&
	(ad->ex[atom2].flag & ATOM_FIXED)) {
      bd->bonds[i].flag |= BOND_IN_RIGID;
      continue;
    }
    if (i != pack_to) {
      bd->bonds[pack_to] = bd->bonds[i];
    }
    pack_to++;
  }

  lprintf("  Number of bonds: %d -> %d\n", bd->n_bond, pack_to);
  bd->n_bond = pack_to;
}

void BOND_DATA_check_hydrogen(BOND_DATA *bd, ATOM_DATA *ad)
{
  int i, j, p, ok;
  
  for (i=0; i<ad->natom; i++) {
    if (ad->ex[i].flag & ATOM_PARENT) {
      p = i;
    } else if (ad->ex[i].flag & ATOM_CHILD) {
      for (j=0;j<bd->n_bond;j++) {
	ok = 0;
	if ((bd->bonds[j].atom1 == p && bd->bonds[j].atom2 == i) ||
	    (bd->bonds[j].atom1 == i && bd->bonds[j].atom2 == p)) {
	  ok = 1;
	  break;
	}
      }
      if (!ok) {
	lprintf("ERROR: Hydrogen %d and Parent atom %d are not connected.\n", i, p);
	lprintf("       Check topology file!\n");
	marble_exit(1);
      }
    }
  }
}

void BOND_DATA_set_hydrogen_group(BOND_DATA *bd, ATOM_DATA *ad)
{
  char *p1,*p2;
  int i, parent, child;
  
  for (i=0; i<ad->natom; i++) {
    ad->ex[i].child_list = -1;
  }
  
  for (i=0;i<bd->n_bond;i++) {
    p1 = get_atom_sym(ad,get_bond_atom1(bd,i));
    p2 = get_atom_sym(ad,get_bond_atom2(bd,i));
    if (*p1 == 'H' && *p2 == 'H') continue;
    if (*p1 != 'H' && *p2 != 'H') continue;

    if (*p1 != 'H') {
      parent = bd->bonds[i].atom1;
      child  = bd->bonds[i].atom2;
    } else {
      parent = bd->bonds[i].atom2;
      child  = bd->bonds[i].atom1;
    }
    
    ad->ex[parent].parent = parent;
    ad->ex[child].parent  = parent;

    ad->ex[child].child_list = ad->ex[parent].child_list;
    ad->ex[parent].child_list = child;
    
    ad->ex[parent].flag |= ATOM_PARENT;
    ad->ex[child].flag  |= ATOM_CHILD;
  }

  /* check routine */
  for (i=0;i<bd->n_bond;i++) {
    p1 = get_atom_sym(ad,get_bond_atom1(bd,i));
    p2 = get_atom_sym(ad,get_bond_atom2(bd,i));
    
    if (*p1 == 'H' && !(ad->ex[bd->bonds[i].atom1].flag & ATOM_CHILD)) {
      lprintf("ERROR: (internal) atom %d must be ATOM_CHILD\n",bd->bonds[i].atom1+1);
    }
    
    if (*p2 == 'H' && !(ad->ex[bd->bonds[i].atom2].flag & ATOM_CHILD)) {
      lprintf("ERROR: (internal) atom %d must be ATOM_CHILD\n",bd->bonds[i].atom2+1);
    }
  }
}

double BOND_DATA_standard_length(BOND_DATA *bd, int atom1, int atom2)
{
  int i;
  for (i=0;i<bd->n_bond;i++) {
    if ((bd->bonds[i].atom1 == atom1 &&
	 bd->bonds[i].atom2 == atom2) ||
        (bd->bonds[i].atom1 == atom2 &&
	 bd->bonds[i].atom2 == atom1)) {
      return bd->bond_type[bd->bonds[i].type].r0;
    }
  }
  return 0.0;
}

    
void BOND_DATA_scale_bond_k(BOND_DATA *bd, double scale)
{
  int i;

  lprintf("Scale force constants of bonds: %lf\n\n", scale);
  for (i=0;i<bd->n_bond_type;i++) {
    bd->bond_type[i].k *= scale;
  }
}

void BOND_DATA_overwrite_bond_k(BOND_DATA *bd, double val)
{
  int i;

  lprintf("Overwrite force constants of bonds: %lf\n\n", val);
  for (i=0;i<bd->n_bond_type;i++) {
    bd->bond_type[i].k = val;
  }
}
