/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#ifndef _ATOM_H_INCLUDED
#define _ATOM_H_INCLUDED

#define ATOM_ENE_VDW     0
#define ATOM_ENE_ELEC    1
#define MAX_ATOM_ENE     2

#define MAX_EX_SYSTEM    4

#define ATOM_PARENT      1
#define ATOM_CHILD       2
#define ATOM_REQ_INT     4   /* for use sdmd */
#define ATOM_REQ_NB      8   /* for use sdmd */
#define ATOM_REQ        12   /* ATOM_REQ_INT | ATOM_REQ_NB */
#define ATOM_EX_SYSTEM1 16   /* for use extended system */
#define ATOM_EX_SYSTEM2 32   /* for use extended system */
#define ATOM_EX_SYSTEM  48   /* ATOM_EX_SYSTEM1 | ATOM_EX_SYSTEM2 */
#define ATOM_RIGID      64   /* for use rigid body dynamics */
#define ATOM_RATTLE    128   /* for use RATTLE dynamics */
#define ATOM_FIXED     256   /* for frozen atoms */

#define ATOM_FLAG_EX_SYSTEM(f)   (((f)&ATOM_EX_SYSTEM)>>4)
#define EX_SYSTEM_ATOM_FLAG(f)   ((f)<<4)

#define ATOM_RMOL(ad,i)    (((ad)->ex[i].flag & ATOM_PARENT) ? (ad)->ex[i].parent : (ad)->ex[(ad)->ex[i].parent].parent)

#define ATOM_CPU(lc,ad,i)    (((ad)->ex[i].flag & ATOM_CHILD) ? (lc)->cell[(lc)->atom_cell[(ad)->ex[i].parent]].pe : (lc)->cell[(lc)->atom_cell[i]].pe)

#define MAX_USER_GROUP_NO 29

typedef struct _s_EXTRA_ATOM_DATA {
  /* excluded atom data */
  int n_exatom;
  int *exatom;
  int id;
  int flag;
  unsigned int group;
  int parent, child_list;
} EXTRA_ATOM_DATA;

typedef struct _s_ATOM_NAME {
  char name[5];
  char sym[5];
  int resno;
} ATOM_NAME;

typedef struct _s_RES_NAME {
  char name[5];
  int start_atom, natom, end_atom;
  int center_atom;
  unsigned int flag;
  int  pdb_no;
  char pdb_chain;
  VEC center;
} RESIDUE;

typedef struct _s_MOLECULE {
  char chain;
  int start_atom, natom;
  unsigned int flag;
  double w;
  VEC center, v;
} MOLECULE;

typedef struct _s_REF_CRD {
  char fname[MAX_FILE_NAME_LEN];
  VEC *x;
  
  struct _s_REF_CRD *next;
} REF_CRD;

#define RES_FLAG_HETATM    1
#define RES_FLAG_RIGID     2
#define RES_FLAG_WATER     4
#define RES_FLAG_AMINO     8
#define RES_FLAG_PERIODIC 16

typedef struct _s_ATOM_DATA {
  int natom;
  VEC *x, *v, *f;
  VEC *px;
  double virial[9];
  int   *vdw_type;
  double *q;
  double *w;
  double total_w, scnb, scee;
  EXTRA_ATOM_DATA *ex;
  
  int atom_ene_flag, atom_ene_sample_flag, n_atom_ene_group;
  int *atom_ene_group;  /* atom_ene_group[atom_no] == group_no */
  double ***atom_ene;     /* atom_ene[atom_no][group_no][ene_no] */
  double ***atom_ene_sum; /* atom_ene_sum[atom_no][group_no][ene_no] */
  int n_sample_atom_ene;

  /* FOR rRESPA */
  VEC *f0, *f1, *f2, *f3;
  double virial0[3], virial1[3], virial2[3], virial3[3];

  VEC *frac, *fold_x, *tr_x;
  
  VEC *fold_f;
  int *fold_id;
  double *fold_q;
  
  ATOM_NAME *a;
  
  int nres;
  RESIDUE *r;

  int nmol, n_solute_mol;
  int n_solute_atom;
  double solute_w;
  MOLECULE *mol;
  
  int  ntype, n_hbpair, *index;
  double *vdw12, *vdw6, *hb12, *hb10;
  double *eps, *rmin;
  double *vdw12_14, *vdw6_14;
  double *eps14, *rmin14;
  
  char (*sym)[5];

  int print_flag_pdb;
  int pdb_long_no_flag;  /* 0: original, 1: vmd format */
  int mdat_format;

  int ex_force_flag;

  int pdb_no_flag;

  REF_CRD *ref_crd;

#ifdef MPI_RDMD
  VEC *fa, *fb, *fd;
  int *group_no;
  VEC *f_reduce;
  int start_task, end_task;
  int *data_counts, *data_offsets;
#endif

  int node_fatom_h, *node_fatom_n;
#ifdef MPI_SDMD
  int node_atom_h, *node_atom_n;
#endif
  
} ATOM_DATA;

typedef enum {
  REWEIGHT_HYDROGEN
} ATOM_REWEIGHT;

#define get_atom_sym(ad,i)   ((ad)->a[i].sym)
#define get_atom_name(ad,i)  ((ad)->a[i].name)
#define get_res_natom(ad,i)  ((ad)->r[i].natom)
#define get_res_name(ad,i)   ((ad)->r[i].name)

#define GM_UNKNOWN -1
#define GM_OR       0
#define GM_AND      1
#define GM_OR_NOT   2
#define GM_AND_NOT  3

void ATOM_DATA_init(ATOM_DATA *ad);
void ATOM_DATA_finalize(ATOM_DATA *ad);
int ATOM_DATA_print_atom(ATOM_DATA *ad, int i);
int ATOM_DATA_print_atom_list(ATOM_DATA *ad, int *list, int n_list);
void print_atom_data(FILE *fp, ATOM_DATA *ad);
char* make_PDB_atom_name(char *pdb_name, char *name);
void ATOM_DATA_print_atom_pdb(FILE *fp, int atomno, char *atomname, char *resname, int resno,double x, double y, double z);
int ATOM_DATA_write_pdb_file(ATOM_DATA *ad, FILE *fp, int group_no);
void ATOM_DATA_allocate(ATOM_DATA *ad);
void ATOM_DATA_allocate_for_periodic_boundary(ATOM_DATA *ad);
void ATOM_DATA_allocate_atom_type(ATOM_DATA *ad);
void ATOM_DATA_allocate_for_rRESPA(ATOM_DATA *ad);
int ATOM_DATA_read_coordinate(ATOM_DATA *ad, FILE *fp);
int ATOM_DATA_read_velocity(ATOM_DATA *ad, FILE *fp);
void ATOM_DATA_write_coordinate(ATOM_DATA *ad, FILE *fp);
void ATOM_DATA_write_velocity(ATOM_DATA *ad, FILE *fp);
void ATOM_DATA_setup_solute(ATOM_DATA *ad);
void ATOM_DATA_set_print_flag_pdb(ATOM_DATA *ad, int flag);
void ATOM_DATA_set_mass(ATOM_DATA *ad, int g_no, double w);
void ATOM_DATA_set_hydrogen_group(ATOM_DATA *ad);
void ATOM_DATA_check_hydrogen_dist(ATOM_DATA *ad, double th);
void ATOM_DATA_check_hydrogen_rigid(ATOM_DATA *ad, double len);
void ATOM_DATA_check_rigid_atom(ATOM_DATA *ad);
void ATOM_DATA_clear_force(ATOM_DATA *ad);
void ATOM_DATA_calc_min_max(ATOM_DATA *ad, double min[3], double max[3]);
void ATOM_DATA_calc_virial(ATOM_DATA *ad);
void ATOM_DATA_mol_gravity_center(ATOM_DATA *ad);
void ATOM_DATA_mol_velocity(ATOM_DATA *ad);
void ATOM_DATA_molecular_virial(ATOM_DATA *ad);
void ATOM_DATA_Maxwell_velocity(ATOM_DATA *ad, double temperature);
void ATOM_DATA_Total_momentum_zero(ATOM_DATA *ad);
void ATOM_DATA_reweight(ATOM_DATA *ad, double w, ATOM_REWEIGHT flag);
void ATOM_DATA_set_center_atom_in_residue(ATOM_DATA *ad);
int  ATOM_DATA_add_excluded_pair(ATOM_DATA *ad, int na, int nb);
int  ATOM_DATA_change_lj_param(ATOM_DATA *ad, char *name, double eps, double r);

int ATOM_DATA_residue_range(ATOM_DATA *ad, int a_id, int no1, int no2, char chain);
int ATOM_DATA_count_group_atoms(ATOM_DATA *ad, int g_no);
int ATOM_DATA_count_group_residues(ATOM_DATA *ad, int num, int *list);
int ATOM_DATA_count_group_molecules(ATOM_DATA *ad, int g_no);
int ATOM_DATA_count_group_atoms_in_mol(ATOM_DATA *ad, int g_no, int imol);

void ATOM_DATA_set_flag_of_group_atoms(ATOM_DATA *ad, int g_no, char *flag);
void ATOM_DATA_make_list_of_group_atoms(ATOM_DATA *ad, int g_no, int *list);
void ATOM_DATA_make_list_of_group_molecules(ATOM_DATA *ad, int g_no, int *list);
void ATOM_DATA_make_list_of_group_atoms_in_mol(ATOM_DATA *ad, int g_no, int imol, int *list);
void ATOM_DATA_get_coord_of_atom_list(ATOM_DATA *ad, VEC *x0, int *atom_list, int n_atom);
void ATOM_DATA_set_group_str_list(ATOM_DATA *ad, STR_LIST *str_list, int g_no, char *header);
void ATOM_DATA_set_group_buf(ATOM_DATA *ad, char *buf, int g_no, char *header);
void ATOM_DATA_clear_group(ATOM_DATA *ad, int g_no);
void ATOM_DATA_set_g_bit(ATOM_DATA *ad, int atom_no, int g_bit, int ok, int mode);
void ATOM_DATA_group_atom_no(ATOM_DATA *ad, int g_no, int no1, int no2, int mode);
void ATOM_DATA_group_res_no(ATOM_DATA *ad, int g_no, char *resno1, char *resno2, int mode);

int ATOM_DATA_atom_name_to_flag(char *name);
int ATOM_DATA_match_non_hydrogen(char *name);
int ATOM_DATA_match_backbone(char *name);

void ATOM_DATA_group_atom_name(ATOM_DATA *ad, int g_no, char *name, int mode);
void ATOM_DATA_group_atom_name_res_no(ATOM_DATA *ad, int g_no, char *name, char *resno1, char *resno2, int mode);
void ATOM_DATA_group_res_name(ATOM_DATA *ad, int g_no, char *name, int mode);
void ATOM_DATA_group_atom_type(ATOM_DATA *ad, int g_no, int mode, int flag);
void ATOM_DATA_group_from_another_group(ATOM_DATA *ad, int g_no, int src_g_no, int mode);
void ATOM_DATA_set_atom_ene_group(ATOM_DATA *ad, int n_atom_ene_group, int *atom_ene_group);
void ATOM_DATA_atom_ene_clear(ATOM_DATA *ad);
void ATOM_DATA_atom_ene_sum_clear(ATOM_DATA *ad);
void ATOM_DATA_atom_ene_sample(ATOM_DATA *ad);
void ATOM_DATA_atom_ene_prop_header(ATOM_DATA *ad, FILE *prop_fp, int *no);
void ATOM_DATA_atom_ene_prop_out(ATOM_DATA *ad, FILE *prop_fp);
void ATOM_DATA_write_atom_ene(ATOM_DATA *ad, char *fname);
void ATOM_DATA_n_flex_atom(ATOM_DATA *ad, int *n_flex_atom, int *n_flex_atom_ex);
void ATOM_DATA_set_residue_type(ATOM_DATA *ad);
void ATOM_DATA_set_node_fatom(ATOM_DATA *ad);
int ATOM_DATA_str_to_atom_no(ATOM_DATA *ad, char *name);
int ATOM_DATA_str_to_res_no(ATOM_DATA *ad, char *name, int flag);
int ATOM_DATA_str_to_atom_list(ATOM_DATA *ad, char *name, int **plist);
int ATOM_DATA_str_to_res_list(ATOM_DATA *ad, char *name, int **plist, int flag);

void ATOM_DATA_set_fixed_atom(ATOM_DATA *ad, int group_no);
void ATOM_DATA_read_pdb_or_crd_file(ATOM_DATA *ad,
				    char *crd_file, int pdb_mode, 
				    STR_LIST *pdb_group_str,
				    VEC *x0, int x0_n_atom, char *header, int check_g_no);

/* ref_crd system */

VEC *ATOM_DATA_get_ref_crd(ATOM_DATA *ad, char *fname);
int  ATOM_DATA_read_ref_crd_master(ATOM_DATA *ad, char *fname, REF_CRD *rc);


#endif /* _ATOM_H_INCLUDED */
