/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <math.h>
#include <ctype.h>

#include "misc.h"
#include "atom.h"
#include "pdb.h"

void ATOM_DATA_init(ATOM_DATA *ad)
{
  ad->natom = 0;
  ad->x = NULL;  ad->v = NULL;  ad->f = NULL;  ad->q = NULL;
  ad->w = NULL;  ad->ex = NULL; ad->vdw_type = NULL; ad->a = NULL;
  ad->nres = 0;
  ad->r = NULL;

  ad->ntype = 0; ad->n_hbpair = 0;
  ad->index = NULL;
  ad->vdw12 = ad->vdw6 = ad->hb12 = ad->hb10 = NULL;

  ad->f0 = ad->f1 = ad->f2 = ad->f3 = NULL;

  ad->atom_ene_flag = 0;
  ad->atom_ene = NULL;

  ad->frac = ad->fold_x = ad->tr_x = NULL;

  ad->print_flag_pdb = 0;
  ad->scnb = 2.0;
  ad->scee = 1.2;

  ad->ex_force_flag = 0;

  ad->pdb_long_no_flag = 0;

  ad->ref_crd = NULL;
}

void ATOM_DATA_finalize(ATOM_DATA *ad)
{
  if (ad->x) free(ad->x);
  if (ad->v) free(ad->v);
  if (ad->f) free(ad->f);
  if (ad->q) free(ad->q);
  if (ad->w) free(ad->w);
  if (ad->ex) free(ad->ex);
  if (ad->vdw_type) free(ad->vdw_type);
  if (ad->a) free(ad->a);
  
  if (ad->r) free(ad->r);
  if (ad->index) free(ad->index);
  if (ad->vdw12) free(ad->vdw12);
  if (ad->vdw6) free(ad->vdw6);
  if (ad->hb12) free(ad->hb12);
  if (ad->hb10) free(ad->hb10);

#ifdef MPI_RDMD
  if (ad->f_reduce) free(ad->f_reduce);
  if (ad->data_counts) free(ad->data_counts);
  if (ad->data_offsets) free(ad->data_offsets);
#endif
  
  ATOM_DATA_init(ad);
}

int ATOM_DATA_print_atom(ATOM_DATA *ad, int i)
{
  char aname[10], atom_no[10], res_no[10];
  int j;

  if (i<0 || ad->natom <= i) {
    lprintf("ERROR: Specified number (%d) of atom is not valid.\n", i+1);
    return -1;
  }

  if (i+1 < 100000) {
    sprintf(atom_no, "%5d ", i+1);
  } else {
    sprintf(atom_no, "%6d" , i+1);
  }
  j=ad->a[i].resno;
  
  if (ad->r[j].pdb_no < 10000) {
    sprintf(res_no, "%4d  ", ad->r[j].pdb_no);
  } else if (ad->r[j].pdb_no < 100000) {
    sprintf(res_no, "%5d ", ad->r[j].pdb_no);
  } else {
    sprintf(res_no, "%6d" , ad->r[j].pdb_no);
  }
  lprintf("ATOM  %-6s%-4s %-4s%c%-6s  %8.3f%8.3f%8.3f %5.2f %5.2f\n",
	  atom_no, make_PDB_atom_name(aname,ad->a[i].name),
	  ad->r[j].name, ad->r[j].pdb_chain, res_no,
	  ad->x[i].x, ad->x[i].y, ad->x[i].z, 1.0, 1.0);
  return 0;
}

int ATOM_DATA_print_atom_list(ATOM_DATA *ad, int *list, int n_list)
{
  int i, ret;
  
  for (i=0;i<n_list;i++) {
    ret = ATOM_DATA_print_atom(ad, list[i]);
    if (ret < 0) return ret;
  }
  return 0;
}


void print_atom_data(FILE *fp, ATOM_DATA *ad)
{
  int i;
  for (i=0; i<ad->natom; i++) {
    fprintf(fp,"%d %f %f %f\n", i,ad->x[i].x, ad->x[i].y, ad->x[i].z);
  }
}

char* make_PDB_atom_name(char *pdb_name, char *name)
{
  /*
  if (strlen(name) <= 3) {
    if (isdigit(name[0])) {
      sprintf(pdb_name,"%s",name);
    } else {
      sprintf(pdb_name," %s",name);
    }
  } else {
    if (isdigit(name[0])) {
      sprintf(pdb_name,"%s",name);
    } else {
      pdb_name[0] = name[3];
      pdb_name[1] = name[0];
      pdb_name[2] = name[1];
      pdb_name[3] = name[2];
      pdb_name[4] = '\0';
    }
  }
  */
  if (strlen(name) <= 3)
    sprintf(pdb_name," %s",name);
  else
    sprintf(pdb_name,"%s",name);
  
  return pdb_name;
}

void ATOM_DATA_print_atom_pdb(FILE *fp, int atomno, char *atomname, char *resname, int resno,double x, double y, double z)
{
  fprintf(fp,"ATOM%7d %-4s %-4s%5d    %8.3f%8.3f%8.3f %5.2f %5.2f\n",
	  atomno,atomname,resname,resno,x,y,z,1.0,1.0);
}

int ATOM_DATA_write_pdb_file(ATOM_DATA *ad, FILE *fp, int group_no)
{
  int i, ia, j;
  unsigned int g_id;
  char aname[10], atom_no[10], res_no[10];

  if (group_no < 0) {
    g_id = 0;
  } else {
    g_id = 1 << group_no;
  }

  for (i=0;i<ad->natom;i++) {
    if (g_id == 0 || (ad->ex[i].group & g_id)) {
      j=ad->a[i].resno;
      
      if (ad->pdb_long_no_flag == 0) {
	if (i+1 < 100000) {
	  sprintf(atom_no, "%5d ", i+1);
	} else {
	  sprintf(atom_no, "%6d" , i+1);
	}
      
	if (ad->r[j].pdb_no < 10000) {
	  sprintf(res_no, "%4d  ", ad->r[j].pdb_no);
	} else if (ad->r[j].pdb_no < 100000) {
	  sprintf(res_no, "%5d ", ad->r[j].pdb_no);
	} else {
	  sprintf(res_no, "%6d" , ad->r[j].pdb_no);
	}

      } else {
	/* VMD format */
	sprintf(atom_no, "%5d ", (i+1)%100000);
	sprintf(res_no,  "%4d  ", ad->r[j].pdb_no % 10000);
      }

      fprintf(fp,"ATOM  %-6s%-4s %-4s%c%-6s  %8.3f%8.3f%8.3f %5.2f %5.2f",
	      atom_no, make_PDB_atom_name(aname,ad->a[i].name),
	      ad->r[j].name, ad->r[j].pdb_chain, res_no,
	      ad->x[i].x, ad->x[i].y, ad->x[i].z, 1.0, 1.0);
      
      /*
      if (ad->a[i].resno+1 < 10000) {
	sprintf(res_no, "%4d  ", ad->a[i].resno+1);
      } else if (ad->a[i].resno+1 < 100000) {
	sprintf(res_no, "%5d ", ad->a[i].resno+1);
      } else {
	sprintf(res_no, "%6d" , ad->a[i].resno+1);
      }
      fprintf(fp,"ATOM  %-6s%-4s %-4s %-6s  %8.3f%8.3f%8.3f %5.2f %5.2f",
	      atom_no, make_PDB_atom_name(aname,ad->a[i].name),
	      ad->r[ad->a[i].resno].name, res_no,
	      ad->x[i].x, ad->x[i].y, ad->x[i].z, 1.0, 1.0);
      */

      if (ad->print_flag_pdb) {
	fprintf(fp," g%d", ad->ex[i].group);
	if (ad->ex[i].flag & ATOM_RIGID) {
	  /* rigid_body */
	  if (ad->ex[i].flag & ATOM_PARENT)
	    fprintf(fp," R%d",ad->ex[i].parent+1);
	  else
	    fprintf(fp," r%d",ad->ex[i].parent+1);
	}
      }
      fprintf(fp,"\n");
      
      /*
      ATOM_DATA_print_atom_pdb(fp, i+1,
              make_PDB_atom_name(aname,ad->a[i].name),
			       ad->r[ad->a[i].resno].name,
			       ad->a[i].resno+1,
			       ad->x[i].x, ad->x[i].y, ad->x[i].z);
      */
    }
  }
  return 0;
}

void ATOM_DATA_allocate(ATOM_DATA *ad)
{
  char *fname = "ATOM_DATA_allocate";
  int i;
  
  ad->x  = emalloc(fname, sizeof(VEC)*ad->natom);
  ad->v  = emalloc(fname, sizeof(VEC)*ad->natom);
  ad->f  = emalloc(fname, sizeof(VEC)*ad->natom);
  ad->q = emalloc(fname, sizeof(double)*ad->natom);
  ad->w = emalloc(fname, sizeof(double)*ad->natom);
  ad->ex = emalloc(fname, sizeof(EXTRA_ATOM_DATA)*ad->natom);
  ad->vdw_type = emalloc(fname, sizeof(int)*ad->natom);
  ad->a = emalloc(fname, sizeof(ATOM_NAME)*ad->natom);
  
  ad->px  = emalloc(fname, sizeof(VEC)*ad->natom);

#ifdef MPI_RDMD
  ad->f_reduce = emalloc(fname, sizeof(VEC)*ad->natom);
  ad->data_counts = emalloc(fname, sizeof(int)*mpi.n_p);
  ad->data_offsets = emalloc(fname, sizeof(int)*mpi.n_p);
#endif
  
  ad->node_fatom_n = emalloc(fname, sizeof(int)*ad->natom);
  ad->node_fatom_h = 0;
  for (i=0;i<ad->natom-1;i++) {
    ad->node_fatom_n[i]=i+1;
  }
  ad->node_fatom_n[ad->natom-1]=-1;
  
#ifdef MPI_SDMD
  if (!mpi.master) {
    ad->node_fatom_h = -1;
  }
  
  ad->node_atom_n = ad->node_fatom_n;
  ad->node_atom_h = ad->node_fatom_h;
#endif  
}

void ATOM_DATA_allocate_for_periodic_boundary(ATOM_DATA *ad)
{
  char *fname = "ATOM_DATA_allocate_for_periodic_boundary";

  if (ad->frac==NULL) {
    ad->frac   = emalloc(fname, sizeof(VEC)*ad->natom);
    ad->fold_x = emalloc(fname, sizeof(VEC)*ad->natom);
    ad->tr_x   = emalloc(fname, sizeof(VEC)*ad->natom);

#if 1
    ad->fold_id = emalloc(fname, sizeof(int)*ad->natom);
    ad->fold_f  = emalloc(fname, sizeof(VEC)*ad->natom);
    ad->fold_q  = emalloc(fname, sizeof(double)*ad->natom);
#endif    
  }
}


void ATOM_DATA_allocate_atom_type(ATOM_DATA *ad)
{
  char *fname = "ATOM_DATA_allocate_atom_type";
  int n_table;
  
  ad->index = emalloc(fname,sizeof(int)*ad->ntype*ad->ntype);
  n_table   = (ad->ntype+1)*ad->ntype/2;
  
  ad->vdw12   = emalloc(fname,sizeof(double)*n_table);
  ad->vdw6    = emalloc(fname,sizeof(double)*n_table);
  ad->rmin    = emalloc(fname,sizeof(double)*n_table);
  ad->eps     = emalloc(fname,sizeof(double)*n_table);
  
  ad->vdw12_14 = emalloc(fname,sizeof(double)*n_table);
  ad->vdw6_14  = emalloc(fname,sizeof(double)*n_table);
  ad->rmin14   = emalloc(fname,sizeof(double)*n_table);
  ad->eps14    = emalloc(fname,sizeof(double)*n_table);
  
  if (ad->n_hbpair != 0) {
    ad->hb12  = emalloc(fname,sizeof(double)*ad->n_hbpair);
    ad->hb10   = emalloc(fname,sizeof(double)*ad->n_hbpair);
  } else {
    ad->hb12 = ad->hb10 = NULL;
  }
  
}

void ATOM_DATA_allocate_for_rRESPA(ATOM_DATA *ad)
{
  if (ad->f1 == NULL) {
    ad->f0 = ad->f;
    ad->f1 = emalloc("ATOM_DATA_allocate_for_RESPA",sizeof(VEC)*ad->natom);
    ad->f2 = emalloc("ATOM_DATA_allocate_for_RESPA",sizeof(VEC)*ad->natom);
    ad->f3 = emalloc("ATOM_DATA_allocate_for_RESPA",sizeof(VEC)*ad->natom);
  }
}

int ATOM_DATA_read_coordinate(ATOM_DATA *ad, FILE *fp)
{
  char buf[81];
  int  natom, i;
  
  /* fgets(buf, 80, fp);   skip header */

  /* lprintf("Reading coordinates.\n"); */
  
  fscanf(fp,"%d", &(natom));
  if (natom != ad->natom) {
    lprintf("ATOM DATA: inconsistent atom number in crd file\n");
    return 1;
  }

  for (i = 0; i < natom; i++) {
    if (fscanf(fp,"%lf%lf%lf", &(ad->x[i].x),&(ad->x[i].y),&(ad->x[i].z))
	!= 3) {
      return 1;
    };
    ad->v[i].x = ad->v[i].y = ad->v[i].z = 0.0;
  }
  return 0;
}

int ATOM_DATA_read_velocity(ATOM_DATA *ad, FILE *fp)
{
  int i;
  
  /* lprintf("Reading velocities.\n"); */

  for (i = 0; i < ad->natom; i++) {
    if (fscanf(fp,"%lf%lf%lf",
	       &(ad->v[i].x),&(ad->v[i].y),&(ad->v[i].z)) != 3) {
      return 1;
    }
  }
  return 0;
}

void ATOM_DATA_write_coordinate(ATOM_DATA *ad, FILE *fp)
{
  int  natom, i;
  
  fprintf(fp,"%5d\n", ad->natom);

  for (i = 0; i < ad->natom; i++) {
    fprintf(fp," %11.7f %11.7f %11.7f", ad->x[i].x,ad->x[i].y,ad->x[i].z); 
    /*
    fprintf(fp," %.15e %.15e %.15e", ad->x[i].x,ad->x[i].y,ad->x[i].z);
    */
    if (i % 2 == 1)
      fprintf(fp,"\n");
  }
  if (i % 2 == 1) {
    fprintf(fp,"\n");
  }
}

void ATOM_DATA_write_velocity(ATOM_DATA *ad, FILE *fp)
{
  int  natom, i;

  for (i = 0; i < ad->natom; i++) {
    fprintf(fp," %11.7f %11.7f %11.7f", ad->v[i].x,ad->v[i].y,ad->v[i].z);
    if (i % 2 == 1)
      fprintf(fp,"\n");
  }
  if (i % 2 == 1) {
    fprintf(fp,"\n");
  }
}

void ATOM_DATA_setup_solute(ATOM_DATA *ad)
{
  int i;
  if (ad->n_solute_mol > 0) {
    ad->n_solute_atom = ad->mol[ad->n_solute_mol-1].start_atom
      +ad->mol[ad->n_solute_mol-1].natom;
  } else {
    ad->n_solute_atom = ad->natom;
  }
  ad->solute_w = 0.0;
  for (i=0;i<ad->n_solute_atom;i++) {
    ad->solute_w += ad->w[i];
  }
}

void ATOM_DATA_set_print_flag_pdb(ATOM_DATA *ad, int flag)
{
  ad->print_flag_pdb = flag;
}

void ATOM_DATA_set_mass(ATOM_DATA *ad, int g_bit, double w)
{
  unsigned int i;
  
  for(i=0;i<ad->natom;i++) {
    if (ad->ex[i].group & g_bit) {
      ad->w[i] = w;
    }
  }
}


/* This does not work, because connected hydogens is not always placed just after a connected non-hyrogen atom in PDB files. */
void ATOM_DATA_set_hydrogen_group(ATOM_DATA *ad)
{
  int i, parent = -1;
  char *p;
  
  for (i=0; i<ad->natom; i++) {
    p = get_atom_sym(ad,i);
    if (*p != 'H') {
      parent = i;
      
      ad->ex[i].parent = parent;
      ad->ex[i].child_list = -1;
      
    } else {
      if (parent == -1) {
	lprintf("ERROR: The first atom of input data is Hydrogen.\n");
	marble_exit(1);
      }
      ad->ex[parent].flag |= ATOM_PARENT;
      ad->ex[i].flag      |= ATOM_CHILD;

      ad->ex[i].parent = parent;
      ad->ex[i-1].child_list = i;
      ad->ex[i].child_list = -1;
    }
  }
}

void ATOM_DATA_print_hydrogen_group(ATOM_DATA *ad)
{
  int i,j;
  for (i=0; i<ad->natom; i++) {
    if (ad->ex[i].flag & ATOM_CHILD) continue;
    lprintf("%d:",i);
    for (j=ad->ex[i].child_list;j>=0;j=ad->ex[j].child_list) {
      lprintf(" %d", j);
    }
    lprintf("\n");








  }
}

void ATOM_DATA_check_hydrogen_dist(ATOM_DATA *ad, double th)
{
  int i, p;
  double th2, dx, dy, dz;
  th2 = th*th;
  for (i=0; i<ad->natom; i++) {
    if (ad->ex[i].flag & ATOM_PARENT) {
      p = i;
    } else if (ad->ex[i].flag & ATOM_CHILD) {
      dx = ad->x[p].x - ad->x[i].x;
      dy = ad->x[p].y - ad->x[i].y;
      dz = ad->x[p].z - ad->x[i].z;
      if (dx*dx+dy*dy+dz*dz > th2) {
	printf("ERROR: Distance between hydrogen %d and parent %d is more than %f!\n", i, p, th);
	marble_abort(1);
      }
    }
  }
}

void ATOM_DATA_check_hydrogen_rigid(ATOM_DATA *ad, double len)
{
  int i,p,err=0;
  char *sym;
  double len2,dx,dy,dz;

  len2 = len*len;
  for (i=0; i<ad->natom; i++) {
    sym = get_atom_sym(ad,i);
    if (*sym == 'H') {
      if (!(ad->ex[i].flag & ATOM_RIGID)) {
	lprintf("Atom %d is not a rigid atom although it is hydrogen.\n", i+1);
	err=1;
      } else {
	p  = ad->ex[i].parent;
	dx = ad->x[p].x - ad->x[i].x;
	dy = ad->x[p].y - ad->x[i].y;
	dz = ad->x[p].z - ad->x[i].z;
	if (len > 0 && dx*dx+dy*dy+dz*dz > len2) {
	  lprintf("ERROR: Distance between hydrogen %d and parent %d is more than %f!\n", i+1, p+1, len);
	  err=1;
	}
      }
    }
  }
  if (err) {
    lprintf("ERROR: All hydrogen atoms are not assigned as rigid correctly.\n");
    marble_exit(1);
  }
  lprintf("All hydrogen atoms are assigned as rigid.\n");
}


void ATOM_DATA_check_rigid_atom(ATOM_DATA *ad)
{
  int i,j,n_group, count;
  int n_hydrogen, n_rigid_hydrogen, n_rigid_atom;
  int n_fixed_hydrogen, n_fixed_atom;
  char *sym;

  lprintf("Check Number of Atoms in Rigid Bodies:\n");
  
  n_hydrogen = n_rigid_hydrogen = n_rigid_atom = 0;
  n_fixed_hydrogen = n_fixed_atom = 0;
  for (j=0;j<ad->natom;j++) {
    sym = get_atom_sym(ad,j);
    if (*sym == 'H') {
      n_hydrogen++;
      if (ad->ex[j].flag & ATOM_RIGID) {
	n_rigid_hydrogen++;
      } else if (ad->ex[j].flag & ATOM_FIXED) {
	n_fixed_hydrogen++;
      } else {
	lprintf("NOT in rigid body: %d %s %s %d\n", j+1, ad->a[j].name,
		ad->r[ad->a[j].resno].name, ad->a[j].resno+1);
      }
    } 
    if (ad->ex[j].flag & ATOM_RIGID) {
      n_rigid_atom++;
    } else if (ad->ex[j].flag & ATOM_FIXED) {
      n_fixed_atom++;
    }
  }

  lprintf("  n_rigid_atom / n_atom = %d / %d\n",n_rigid_atom,ad->natom);
  lprintf("  n_fixed_atom / n_atom = %d / %d\n",n_fixed_atom,ad->natom);
  lprintf("  n_rigid_hydrogen / n_hydrogen = %d / %d\n",
	  n_rigid_hydrogen,n_hydrogen);
  lprintf("  n_fixed_hydrogen / n_hydrogen = %d / %d\n",
	  n_fixed_hydrogen,n_hydrogen);
  
  if (n_rigid_hydrogen+n_fixed_hydrogen == n_hydrogen) {
    lprintf("  All of hydrogens belong to rigid bodies or fixed regions.\n");
  } else {
    lprintf("Warning: All of hydrogens do not belong to rigid bodies.\n");
    marble_exit(1);
  } 
  lprintf("\n");
  
#if 0
  lprintf("Check Number of Rigid Groups in Each Residue:\n");
  for (i=0;i<ad->nres;i++) {
    n_group = 0;
    count = 0;
    for (j=ad->r[i].start_atom;j<=ad->r[i].end_atom;j++) {
      if (ad->ex[j].flag & ATOM_RIGID) {
	if (ad->ex[j].flag & ATOM_PARENT) {
	  n_group++;
	}
	count++;
      }
    }
    lprintf("%4d %-4s n_rigid_group= %2d, n_rigid_atom= %2d / %2d\n",
	    i+1, ad->r[i].name, n_group, count, ad->r[i].natom);
  }
  lprintf("\n");
#endif  
}

void ATOM_DATA_clear_force(ATOM_DATA *ad)
{
  int i;
  for (i=0; i<ad->natom; i++) {
    ad->f[i].x = ad->f[i].y = ad->f[i].z = 0.0;
  }
  for (i=0;i<6;i++) {
    ad->virial[i] = 0.0;
  }
}

void ATOM_DATA_calc_min_max(ATOM_DATA *ad, double min[3], double max[3])
{
  int i;
  min[0] = ad->x[0].x;
  min[1] = ad->x[0].y;
  min[2] = ad->x[0].z;
  max[0] = min[0];   max[1] = min[1];   max[2] = min[2];
  for (i=1;i<ad->natom;i++) {
    if (min[0] > ad->x[i].x) min[0] = ad->x[i].x;
    if (min[1] > ad->x[i].y) min[1] = ad->x[i].y;
    if (min[2] > ad->x[i].z) min[2] = ad->x[i].z;
    if (max[0] < ad->x[i].x) max[0] = ad->x[i].x;
    if (max[1] < ad->x[i].y) max[1] = ad->x[i].y;
    if (max[2] < ad->x[i].z) max[2] = ad->x[i].z;
  }
}

void ATOM_DATA_calc_virial(ATOM_DATA *ad)
{
  int i;

  for (i=0;i<ad->natom;i++) {
    ad->virial[0] += ad->f[i].x * ad->x[i].x;
    ad->virial[1] += ad->f[i].y * ad->x[i].y;
    ad->virial[2] += ad->f[i].z * ad->x[i].z;
    ad->virial[3] += ad->f[i].x * ad->x[i].y;
    ad->virial[4] += ad->f[i].x * ad->x[i].z;
    ad->virial[5] += ad->f[i].y * ad->x[i].z;
  }
}

void ATOM_DATA_mol_gravity_center(ATOM_DATA *ad)
{
  int i, j, atom_no;
  
  for (i=0;i<ad->nmol;i++) {
    ad->mol[i].center.x = ad->mol[i].center.y = ad->mol[i].center.z = 0.0;
    for (j=0;j<ad->mol[i].natom;j++) {
      atom_no = ad->mol[i].start_atom + j;
      ad->mol[i].center.x += ad->x[atom_no].x * ad->w[atom_no];
      ad->mol[i].center.y += ad->x[atom_no].y * ad->w[atom_no];
      ad->mol[i].center.z += ad->x[atom_no].z * ad->w[atom_no];
    }
    ad->mol[i].center.x /= ad->mol[i].w;
    ad->mol[i].center.y /= ad->mol[i].w;
    ad->mol[i].center.z /= ad->mol[i].w;
  }
}

void ATOM_DATA_mol_velocity(ATOM_DATA *ad)
{
  int i, j, atom_no;
  
  for (i=0;i<ad->nmol;i++) {
    ad->mol[i].v.x = ad->mol[i].v.y = ad->mol[i].v.z = 0.0;
    for (j=0;j<ad->mol[i].natom;j++) {
      atom_no = ad->mol[i].start_atom + j;
      ad->mol[i].v.x += ad->v[atom_no].x * ad->w[atom_no];
      ad->mol[i].v.y += ad->v[atom_no].y * ad->w[atom_no];
      ad->mol[i].v.z += ad->v[atom_no].z * ad->w[atom_no];
    }
    ad->mol[i].v.x /= ad->mol[i].w;
    ad->mol[i].v.y /= ad->mol[i].w;
    ad->mol[i].v.z /= ad->mol[i].w;
  }
}

void ATOM_DATA_molecular_virial(ATOM_DATA *ad)
{
  int i, j, atom_no;
  
  ATOM_DATA_mol_gravity_center(ad);
  for (i=0;i<ad->nmol;i++) {
    for (j=0;j<ad->mol[i].natom;j++) {
      atom_no = ad->mol[i].start_atom + j;
      ad->virial[0] -= ad->f[atom_no].x * (ad->x[atom_no].x - ad->mol[i].center.x);
      ad->virial[1] -= ad->f[atom_no].y * (ad->x[atom_no].y - ad->mol[i].center.y);
      ad->virial[2] -= ad->f[atom_no].z * (ad->x[atom_no].z - ad->mol[i].center.z);
    }
  }
}


void ATOM_DATA_Maxwell_velocity(ATOM_DATA *ad, double temperature)
{
  double rtemp;
  int i;

  for (i=0;i<ad->natom;i++) { 
    /* for (i=ad->node_fatom_h;i>=0;i=ad->node_fatom_n[i]) { */
    rtemp = sqrt(KCAL*K*temperature/ad->w[i]);
    ad->v[i].x = rtemp * gauss_rand();
    ad->v[i].y = rtemp * gauss_rand();
    ad->v[i].z = rtemp * gauss_rand();
  }
  /*
  sum.x /= ad->natom;  sum.y /= ad->natom;  sum.z /= ad->natom;
  for (i=0;i<ad->natom;i++) {
    ad->v[i].x -= sum.x;
    ad->v[i].y -= sum.y;
    ad->v[i].z -= sum.z;
  }
  */
}

void ATOM_DATA_Total_momentum_zero(ATOM_DATA *ad)
{
  int i;
  double w;
  double sum[3];
  
  w = sum[0] = sum[1] = sum[2] = 0.0;
  for (i=0;i<ad->natom;i++) {
    sum[0] += ad->v[i].x * ad->w[i];
    sum[1] += ad->v[i].y * ad->w[i];
    sum[2] += ad->v[i].z * ad->w[i];
    w += ad->w[i];
  }
  /* lprintf("sum (%f,%f,%f)\n", sum[0], sum[1], sum[2]); */
  
  sum[0] /= w;  sum[1] /= w;  sum[2] /= w;
  for (i=0;i<ad->natom;i++) {
    ad->v[i].x -= sum[0];
    ad->v[i].y -= sum[1];
    ad->v[i].z -= sum[2];
  }
}


void ATOM_DATA_reweight(ATOM_DATA *ad, double w, ATOM_REWEIGHT flag)
{
  int i;
  
  switch(flag) {
  case REWEIGHT_HYDROGEN:
    for (i=0;i<ad->natom;i++) {
      if (ad->a[i].sym[0] == 'H') {
	ad->w[i] += w;
      }
    }
    break;
  }
}

void ATOM_DATA_set_center_atom_in_residue(ATOM_DATA *ad)
{
  int i, j;
  VEC c;
  double weight, min_len2, len2;
  
  for (i=0;i<ad->nres;i++) {
    c.x = c.y = c.z = 0.0;
    weight=0.0;
    for (j=ad->r[i].start_atom;j<=ad->r[i].end_atom;j++) {
      c.x += ad->x[j].x * ad->w[j];
      c.y += ad->x[j].y * ad->w[j];
      c.z += ad->x[j].z * ad->w[j];
      weight += ad->w[j];
    }
    c.x /= weight;
    c.y /= weight;
    c.z /= weight;
    ad->r[i].center_atom = j = ad->r[i].start_atom;
    min_len2 = SQR(c.x-ad->x[j].x)+SQR(c.y-ad->x[j].y)+SQR(c.z-ad->x[j].z);
    for (j++;j<=ad->r[i].end_atom;j++) {
      len2=SQR(c.x-ad->x[j].x)+SQR(c.y-ad->x[j].y)+SQR(c.z-ad->x[j].z);
      if (min_len2 > len2) {
	ad->r[i].center_atom = j;
	min_len2 = len2;
      }
    }
  }
}

int ATOM_DATA_add_excluded_pair(ATOM_DATA *ad, int na, int nb)
{
  int i;

  na--; nb--;

  lprintf("  %s (%4d %s %4d) - %s (%4d %s %4d)\n",
	  ad->a[na].name, na+1, ad->r[ad->a[na].resno].name,ad->a[na].resno+1,
	  ad->a[nb].name, nb+1, ad->r[ad->a[nb].resno].name,ad->a[nb].resno+1);
  
  /* check */
  if (na == nb) {
    lprintf("  ERROR: Atom %d is indentical atom %d.\n",
	    na+1, nb+1);
    return 1;
  }
  
  if (na < 0) {
    lprintf("  ERROR: Atom number %d must be a positive value.\n",
	    na+1);
    return 1;
  }
  if (nb < 0) {
    lprintf("  ERROR: Atom number %d must be a positive value.\n",
	    nb*1);
    return 1;
  }
  
  for (i=0;i<ad->ex[na].n_exatom;i++) {
    if (ad->ex[na].exatom[i] == nb) {
      lprintf("  ERROR: Atom %d is already in the exclusive list of atom %d.\n",
	      nb+1, i+1);
      return 1;
    }
  }
  
  for (i=0;i<ad->ex[nb].n_exatom;i++) {
    if (ad->ex[nb].exatom[i] == na) {
      lprintf("  ERROR: Atom %d is already in the exclusive list of atom %d.\n",
	      na+1, i+1);
      return 1;
    }
  }

  /* add exclusive list! */
  ad->ex[na].n_exatom++;
  if (ad->ex[na].n_exatom == 1) {
    ad->ex[na].exatom = emalloc("ATOM_DATA_add_exclusive_pair",
				sizeof(int));
  } else {
    ad->ex[na].exatom = erealloc("ATOM_DATA_add_exclusive_pair",
				 ad->ex[na].exatom,
				 sizeof(int)*ad->ex[na].n_exatom);
  }
  ad->ex[na].exatom[ad->ex[na].n_exatom-1] = nb;
  
  ad->ex[nb].n_exatom++;
  if (ad->ex[nb].n_exatom == 1) {
    ad->ex[nb].exatom = emalloc("ATOM_DATA_add_exclusive_pair",
				sizeof(int));
  } else {
    ad->ex[nb].exatom = erealloc("ATOM_DATA_add_exclusive_pair",
				 ad->ex[nb].exatom,
				 sizeof(int)*ad->ex[nb].n_exatom);
  }
  ad->ex[nb].exatom[ad->ex[nb].n_exatom-1] = na;


  return 0;
}

int ATOM_DATA_change_lj_param(ATOM_DATA *ad, char *name, double eps, double r)
{
  int i, id, vdw_index;

  id = -1;
  for(i=0;i<ad->ntype;i++) {
    /* lprintf("\"%s\" \"%s\"\n", ad->sym[i], name); */
    if (strcmp(ad->sym[i], name) == 0) {
      id = i;
      break;
    }
  }
  if (id < 0) {
    lprintf("unknown atom symbol %s\n", name);
    return 1;
  }
  for(i=0;i<ad->ntype;i++) {
    vdw_index = ad->index[i+id*ad->ntype];
    if (eps > 0.0 && ad->eps[vdw_index]) {
      ad->eps[vdw_index] = eps * 4.0;
    }
    if (r > 0.0) {
      ad->rmin[vdw_index] = r;
    }
    lprintf("%-4s-%-4s %3d eps=%e, rmin=%e\n",ad->sym[id], ad->sym[i], vdw_index,
	    ad->eps[vdw_index]/4.0, ad->rmin[vdw_index]);
    ad->vdw12[vdw_index] = ad->eps[vdw_index] * pow(ad->rmin[vdw_index],6.0);
    ad->vdw6[vdw_index]  = ad->eps[vdw_index] * pow(ad->rmin[vdw_index],12.0);
  }
  return 0;
}

/*
int ATOM_DATA_residue_range(ATOM_DATA *ad, int a_id, int no1, int no2, char chain)
{
  int r_id;
  r_id = ad->a[a_id].resno;
  if (chain == '*' ||
      (chain == ad->r[r_id].pdb_chain)) {
    if (ad->r[r_id].pdb_no >= no1 &&
	ad->r[r_id].pdb_no <= no2) {
      return 1;
    }
  }
  return 0;
}
*/

int ATOM_DATA_count_group_atoms(ATOM_DATA *ad, int g_no)
{
  unsigned int g_bit, i, count;

  g_bit = 1 << g_no;
  count = 0;

  for (i=0;i<ad->natom;i++) {
    if (ad->ex[i].group & g_bit) {
      count++;
    }
  }
  return count;
}

/* yss-add */
int ATOM_DATA_count_group_residues(ATOM_DATA *ad, int num, int *list)
{
  int resno_ini, resno_fin;
  unsigned int i, count;

  resno_ini = ad->a[list[0]].resno+1;
  resno_fin = ad->a[list[num-1]].resno+1;

  count = resno_fin - resno_ini + 1;

  return count;
}
/* yse */

int ATOM_DATA_count_group_molecules(ATOM_DATA *ad, int g_no)
{
  unsigned int g_bit, i, j, count;

  g_bit = 1 << g_no;
  count = 0;

  for (i=0;i<ad->nmol;i++) {
    for (j=ad->mol[i].start_atom;
	 j<ad->mol[i].start_atom+ad->mol[i].natom;
	 j++) {
      if (ad->ex[j].group & g_bit) {
	count++;
	break;
      }
    }
  }
  return count;
}

int ATOM_DATA_count_group_atoms_in_mol(ATOM_DATA *ad, int g_no, int imol)
{
  unsigned int g_bit, i, j, count;

  g_bit = 1 << g_no;
  count = 0;

  for (j=ad->mol[imol].start_atom;
       j<ad->mol[imol].start_atom+ad->mol[imol].natom;
       j++) {
    if (ad->ex[j].group & g_bit) {
      count++;
    }
  }
  return count;
}

void ATOM_DATA_set_flag_of_group_atoms(ATOM_DATA *ad, int g_no, char *flag)
{
  unsigned int g_bit, i;

  g_bit = 1 << g_no;

  for (i=0;i<ad->natom;i++) {
    if (ad->ex[i].group & g_bit) {
      flag[i] = 1;
    } else {
      flag[i] = 0;
    }
  }
}

void ATOM_DATA_make_list_of_group_atoms(ATOM_DATA *ad, int g_no, int *list)
{
  unsigned int g_bit, i, count;

  g_bit = 1 << g_no;
  count = 0;

  for (i=0;i<ad->natom;i++) {
    if (ad->ex[i].group & g_bit) {
      list[count] = i;
      count++;
    }
  }
  return;
}



void ATOM_DATA_make_list_of_group_molecules(ATOM_DATA *ad, int g_no, int *list)
{
  unsigned int g_bit, i, j, count;

  g_bit = 1 << g_no;
  count = 0;

  for (i=0;i<ad->nmol;i++) {
    for (j=ad->mol[i].start_atom;
	 j<ad->mol[i].start_atom+ad->mol[i].natom;
	 j++) {
      if (ad->ex[j].group & g_bit) {
	list[count] = i;
	count++;
	break;
      }
    }
  }
  return;
}

void ATOM_DATA_make_list_of_group_atoms_in_mol(ATOM_DATA *ad, int g_no, int imol, int *list)
{
  unsigned int g_bit, i, j, count;

  g_bit = 1 << g_no;
  count = 0;

  for (j=ad->mol[imol].start_atom;
       j<ad->mol[imol].start_atom+ad->mol[imol].natom;
       j++) {
    if (ad->ex[j].group & g_bit) {
      list[count] = j;
      count++;
    }
  }
  return;
}

void ATOM_DATA_get_coord_of_atom_list(ATOM_DATA *ad, VEC *x0, int *atom_list, int n_atom)
{
  int i, j;
  for (i=0;i<n_atom;i++) {
    j = atom_list[i];
    x0[i].x = ad->x[j].x;
    x0[i].y = ad->x[j].y;
    x0[i].z = ad->x[j].z;
  }
}

#define GROUP_BUF_ATOM_NO          1
#define GROUP_BUF_RES_NO           2
#define GROUP_BUF_ATOM_NAME        3
#define GROUP_BUF_ATOM_NAME_RES_NO 4
#define GROUP_BUF_RES_NAME         5
#define GROUP_BUF_ATOM_TYPE        6
#define GROUP_BUF_GROUP_NO         7

SEL_DATA _sel_group_mode[] = {
  "or",      GM_OR,
  "add",     GM_OR,
  "and",     GM_AND,
  "or_not",  GM_OR_NOT,
  "and_not", GM_AND_NOT,
  "exclude", GM_AND_NOT,
  NULL,      GM_UNKNOWN
};

void ATOM_DATA_set_group_str_list(ATOM_DATA *ad, STR_LIST *str_list, int g_no, char *header)
{
  STR_LIST *s;

  ATOM_DATA_clear_group(ad, g_no);

  for (s=str_list;s != NULL; s=s->next) {
    ATOM_DATA_set_group_buf(ad, s->buf, g_no, header);
  }
}


void ATOM_DATA_set_group_buf(ATOM_DATA *ad, char *buf, int g_no, char *header)
{
  char mode_str[100];
  char buf1[100], buf2[100];
  char res1[100], res2[100];
  char atom1[100], atom2[100];
  int  mode_id, type, ret;
  int  atom1_no, atom2_no;
  int  d;
  int  src_g_no;

  if ((ret = sscanf(buf, " atom_no %99s %99s %99s", atom1, atom2, mode_str)) == 3) {
    type = GROUP_BUF_ATOM_NO;
  } else if (ret == 2) {
    type = GROUP_BUF_ATOM_NO;
    strcpy(mode_str,"add");

  } else if ((ret = sscanf(buf, " res_no %99s %99s %99s", atom1, atom2, mode_str)) == 3) {
    type = GROUP_BUF_RES_NO;
  } else if (ret == 2) {
    type = GROUP_BUF_RES_NO;
    strcpy(mode_str,"add");

  } else if ((ret = sscanf(buf, " atom_name %99s %99s %99s %99s", atom1, res1, res2, mode_str)) == 4) {
    type = GROUP_BUF_ATOM_NAME_RES_NO;
  } else if (ret == 3) {
    type = GROUP_BUF_ATOM_NAME_RES_NO;
    strcpy(mode_str,"add");
  } else if (ret == 2) {
    type = GROUP_BUF_ATOM_NAME;
    strcpy(mode_str,res1);
  } else if (ret == 1) {
    type = GROUP_BUF_ATOM_NAME;
    strcpy(mode_str,"add");

  } else if ((ret = sscanf(buf, " atom %99s %99s %99s %99s", atom1, res1, res2, mode_str)) == 4) {
    type = GROUP_BUF_ATOM_NAME_RES_NO;
  } else if (ret == 3) {
    type = GROUP_BUF_ATOM_NAME_RES_NO;
    strcpy(mode_str,"add");
  } else if (ret == 2) {
    type = GROUP_BUF_ATOM_NAME;
    strcpy(mode_str,res1);
  } else if (ret == 1) {
    type = GROUP_BUF_ATOM_NAME;
    strcpy(mode_str,"add");
    
  } else if ((ret = sscanf(buf, " res_name %99s %99s", res1, mode_str)) == 2) {
    type = GROUP_BUF_RES_NAME;
  } else if (ret == 1) {
    type = GROUP_BUF_RES_NAME;
    strcpy(mode_str,"add");

  } else if ((ret = sscanf(buf, " group_no %d %99s", &d, mode_str)) == 2) {
    type = GROUP_BUF_GROUP_NO;    
    src_g_no = d;
  } else if (ret == 1) {
    type = GROUP_BUF_GROUP_NO;    
    src_g_no = d;
    strcpy(mode_str,"add");

  } else if ((ret = sscanf(buf, " %d %99s", &d, mode_str)) == 2) {
    type = GROUP_BUF_GROUP_NO;    
    src_g_no = d;
  } else if (ret == 1) {
    type = GROUP_BUF_GROUP_NO;    
    src_g_no = d;
    strcpy(mode_str,"add");

  } else {
    lprintf("%s ERROR: unrecognized group definition: %s\n", header, buf);
    marble_exit(1);
  }

  mode_id = get_id_sel_data(_sel_group_mode, mode_str);
  
  switch (type) {
  case GROUP_BUF_ATOM_NO:
    lprintf("%s atoms from %s to %s, Operator = %s\n", header, atom1, atom2, mode_str);
    atom1_no = ATOM_DATA_str_to_atom_no(ad, atom1);
    atom2_no = ATOM_DATA_str_to_atom_no(ad, atom2);
    ATOM_DATA_group_atom_no(ad, g_no, atom1_no+1, atom2_no+1, mode_id);
    break;
  case GROUP_BUF_RES_NO:
    lprintf("%s residues from %s to %s, Operator = %s\n", header, res1, res2, mode_str);
    ATOM_DATA_group_res_no(ad, g_no, res1, res2, mode_id);
    break;
  case GROUP_BUF_ATOM_NAME:
    lprintf("%s atom name \"%s\", Operator = %s\n", header, atom1, mode_str);
    ATOM_DATA_group_atom_name(ad, g_no, atom1, mode_id);
    break;
  case GROUP_BUF_ATOM_NAME_RES_NO:
    lprintf("%s atom name \"%s\" in residues from %s to %s, Operator = %s\n", header, atom1, res1, res2, mode_str);
    ATOM_DATA_group_atom_name_res_no(ad, g_no, atom1, res1, res2, mode_id);
    break;
  case GROUP_BUF_RES_NAME:
    lprintf("%s residue name \"%s\", Operator = %s\n", header, res1, mode_str);
    ATOM_DATA_group_res_name(ad, g_no, res1, mode_id);
    break;
  case GROUP_BUF_GROUP_NO:
    lprintf("%s group no. %d, Operator = %s\n", header, src_g_no, mode_str);
    ATOM_DATA_group_from_another_group(ad, g_no, src_g_no, mode_id);
    break;
  }    
  lprintf("%s          %d atoms selected in total\n", header,ATOM_DATA_count_group_atoms(ad, g_no));
}

void ATOM_DATA_clear_group(ATOM_DATA *ad, int g_no)
{
  unsigned int g_bit;
  int i;
  
  g_bit = 1 << g_no;
  
  for (i=0;i<ad->natom;i++) {
    ad->ex[i].group &= ~g_bit;
  }
}

void ATOM_DATA_set_g_bit(ATOM_DATA *ad, int atom_no, int g_bit, int ok, int mode)
{
  switch (mode) {
  case GM_OR:
    if (ok)
      ad->ex[atom_no].group |= g_bit;
    break;
  case GM_AND:
    if (!ok)
      ad->ex[atom_no].group &= ~g_bit;
    break;
  case GM_OR_NOT:
    if (!ok)
      ad->ex[atom_no].group |= g_bit;
    break;
  case GM_AND_NOT:
    if (ok)
      ad->ex[atom_no].group &= ~g_bit;
    break;
  }
}

void ATOM_DATA_group_atom_no(ATOM_DATA *ad, int g_no, int no1, int no2, int mode)
{
  int i, ok;
  unsigned int g_bit;
  
  no1--;  no2--;
  g_bit = 1 << g_no;
  
  for (i=0;i<ad->natom;i++) {
    if (i >= no1 && i <= no2) ok = 1;
    else                      ok = 0;

    ATOM_DATA_set_g_bit(ad, i, g_bit, ok, mode);
  }
}

void ATOM_DATA_group_res_no(ATOM_DATA *ad, int g_no, char *resno1, char *resno2, int mode)
{
  int i, j, ok, no1, no2;
  unsigned int g_bit;
  
  no1 = ATOM_DATA_str_to_res_no(ad, resno1, 1);
  no2 = ATOM_DATA_str_to_res_no(ad, resno2, -1);
  g_bit = 1 << g_no;

  for (i=0;i<ad->natom;i++) {
    /*
    if (ATOM_DATA_residue_range(ad,i,no1,no2,chain)) ok = 1;
    else                                             ok = 0;
    */

    if (ad->a[i].resno >= no1 && ad->a[i].resno <= no2) ok = 1;
    else ok = 0;
    
    ATOM_DATA_set_g_bit(ad, i, g_bit, ok, mode);
  }
}

int ATOM_DATA_atom_name_to_flag(char *name)
{
  int flag;

  if (strcmp(name,"heavy_atom") == 0 || strcmp(name,"non_hydrogen") == 0)
    flag=1;
  else if (strcmp(name,"backbone") == 0)
    flag=2;
  else if (strcmp(name,"all") == 0)
    flag=3;
  else
    flag=0;
  
  return flag;
}

int ATOM_DATA_match_non_hydrogen(char *name)
{
  if (strmatch(name, "H*") == 0) return 0;
  if (isdigit(name[0]) && name[1] == 'H') return 0;
  return 1;
}

int ATOM_DATA_match_backbone(char *name)
{
  if (strmatch(name, "N")  == 0  ||
      strmatch(name, "CA") == 0  ||
      strmatch(name, "C")  == 0  ||
      strmatch(name, "O")  == 0) return 1;
  return 0;
}

void ATOM_DATA_group_atom_name(ATOM_DATA *ad, int g_no, char *name, int mode)
{
  int i, ok, flag;
  unsigned int g_bit;
  
  g_bit = 1 << g_no;

  flag = ATOM_DATA_atom_name_to_flag(name);
  for (i=0;i<ad->natom;i++) {
    ok = 0;

    if (flag==0) {
      if (strmatch(ad->a[i].name, name) == 0) ok = 1;
    } else if (flag==1) {
      /* heavy_atom or non_hydrogen */
      if (ATOM_DATA_match_non_hydrogen(ad->a[i].name)) ok = 1;
    } else if (flag==2) {
      /* backbone */
      if (ATOM_DATA_match_backbone(ad->a[i].name)) ok = 1;
    } else if (flag==3) {
	/* all */
      ok = 1;
    }
    
    ATOM_DATA_set_g_bit(ad, i, g_bit, ok, mode);
  }
}

void ATOM_DATA_group_atom_name_res_no(ATOM_DATA *ad, int g_no, char *name, char *resno1, char *resno2, int mode)
{
  int i, ok, flag, no1, no2;
  unsigned int g_bit;

  no1 = ATOM_DATA_str_to_res_no(ad, resno1, 1);
  no2 = ATOM_DATA_str_to_res_no(ad, resno2, -1);
  g_bit = 1 << g_no;
  
  flag = ATOM_DATA_atom_name_to_flag(name);
  
  for (i=0;i<ad->natom;i++) {
    ok = 0;
    /*if (ATOM_DATA_residue_range(ad,i,no1,no2,chain)) { */
    if (ad->a[i].resno >= no1 && ad->a[i].resno <= no2) { 

      if (flag==0) {
	if (strmatch(ad->a[i].name, name) == 0) ok = 1;
      } else if (flag==1) {
	/* heavy_atom or non_hydrogen */
	if (ATOM_DATA_match_non_hydrogen(ad->a[i].name)) ok = 1;
      } else if (flag==2) {
	/* backbone */
      if (ATOM_DATA_match_backbone(ad->a[i].name)) ok = 1;
      } else if (flag==3) {
	/* all */
	ok = 1;
      }
    }

    ATOM_DATA_set_g_bit(ad, i, g_bit, ok, mode);
  }
}

void ATOM_DATA_group_res_name(ATOM_DATA *ad, int g_no, char *name, int mode)
{
  int i, j, ok;
  unsigned int g_bit;
  
  g_bit = 1 << g_no;
  
  for (i=0;i<ad->natom;i++) {
    if (strmatch(ad->r[ad->a[i].resno].name, name) == 0) ok = 1;
    else ok = 0;

    ATOM_DATA_set_g_bit(ad, i, g_bit, ok, mode);
  }
}

void ATOM_DATA_group_atom_type(ATOM_DATA *ad, int g_no, int mode, int flag)
{
  int i, ok;
  unsigned int g_bit;
  
  g_bit = 1 << g_no;

  for (i=0;i<ad->natom;i++) {
    ok = 0;

    if (flag == 1) {
      /* backbone */
      if (ATOM_DATA_match_backbone(ad->a[i].name)) ok = 1;
    } else if (flag == 2) {
      /* heavy_atom or non_hydrogen */
      if (ATOM_DATA_match_non_hydrogen(ad->a[i].name)) ok = 1;
    } else if (flag == 3) {
      /* solute_heavy_atom */
      if (i < ad->n_solute_atom &&
	  ATOM_DATA_match_non_hydrogen(ad->a[i].name)) ok = 1;
    }
    
    ATOM_DATA_set_g_bit(ad, i, g_bit, ok, mode);
  }
}

void ATOM_DATA_group_from_another_group(ATOM_DATA *ad, int g_no, int src_g_no, int mode)
{
  int i, ok;
  unsigned int g_bit, src_g_bit;
  
  g_bit = 1 << g_no;
  src_g_bit = 1 << src_g_no;

  for (i=0;i<ad->natom;i++) {
    if (ad->ex[i].group & src_g_bit) ok = 1;
    else ok = 0;

    ATOM_DATA_set_g_bit(ad, i, g_bit, ok, mode);
  }
}


void ATOM_DATA_set_atom_ene_group(ATOM_DATA *ad, int n_atom_ene_group, int *atom_ene_group)
{
  double *p;
  int n[3],i,j, count;
  unsigned int flag;
  
  ad->atom_ene_flag = 1;
  n_atom_ene_group++;  /* for "others" */
  ad->n_atom_ene_group = n_atom_ene_group;
  ad->atom_ene_group = emalloc("ATOM_DATA_set_atom_ene_group", sizeof(int)*ad->natom);

  /* atom_ene */
  p = emalloc("ATOM_DATA_set_atom_ene_group", sizeof(double)*ad->natom*n_atom_ene_group*MAX_ATOM_ENE);
  n[0]=ad->natom; n[1]=n_atom_ene_group; n[2]=MAX_ATOM_ENE;
  ad->atom_ene = alloc_3d_index(p, sizeof(double), n);

  /* atom_ene_sum */
  p = emalloc("ATOM_DATA_set_atom_ene_group", sizeof(double)*ad->natom*n_atom_ene_group*MAX_ATOM_ENE);
  ad->atom_ene_sum = alloc_3d_index(p, sizeof(double), n);

  for (i=0;i<ad->natom;i++) {
    ad->atom_ene_group[i] = n_atom_ene_group-1;
               /* n_atom_ene_group-1 is for others */
    for (j=0;j<n_atom_ene_group-1;j++) {
      flag = 1 << atom_ene_group[j];
      if (ad->ex[i].group & flag) {
	ad->atom_ene_group[i] = j;
      }
    }
  }
  lprintf("ATOM ENERGY GROUP:\n");
  for (j=0;j<n_atom_ene_group;j++) {
    count = 0;
    for (i=0;i<ad->natom;i++) {
      if (ad->atom_ene_group[i] == j)
	count++;
    }
    if (j<n_atom_ene_group-1) {
      lprintf("  ATOM_ENE_GROUP %d :  ATOM_GROUP %d, NO OF ATOMS %d\n",
	      j, atom_ene_group[j], count);
    } else {
      lprintf("  ATOM_ENE_GROUP %d :  REST, NO OF ATOMS %d, INCLUDING FMM ENERGY\n",
	      j, count);
    }
  }
  lprintf("\n");
}

void ATOM_DATA_atom_ene_clear(ATOM_DATA *ad)
{
  int i, nitem;
  double *p;
  
  if (!ad->atom_ene_flag) return;

  nitem = ad->natom*ad->n_atom_ene_group*MAX_ATOM_ENE;
  p = &(ad->atom_ene[0][0][0]);
  for (i=0;i<nitem;i++)
    p[i]=0.0;
}

void ATOM_DATA_atom_ene_sum_clear(ATOM_DATA *ad)
{
  int i, nitem;
  double *p;
  
  if (!ad->atom_ene_flag) return;

  nitem = ad->natom*ad->n_atom_ene_group*MAX_ATOM_ENE;
  p = &(ad->atom_ene_sum[0][0][0]);
  
  for (i=0;i<nitem;i++)
    p[i]=0.0;
  
  ad->n_sample_atom_ene=0;
}

void ATOM_DATA_atom_ene_sample(ATOM_DATA *ad)
{
  int i, nitem;
  double *p1,*p2;
  
  if (!ad->atom_ene_flag) return;

  nitem = ad->natom*ad->n_atom_ene_group*MAX_ATOM_ENE;
  p1 = &(ad->atom_ene[0][0][0]);
  p2 = &(ad->atom_ene_sum[0][0][0]);
  for (i=0;i<nitem;i++)
    p2[i]+=p1[i];
  
  ad->n_sample_atom_ene++;
}


void ATOM_DATA_atom_ene_prop_header(ATOM_DATA *ad, FILE *prop_fp, int *no)
{
  int i,j;
  
  if (!ad->atom_ene_flag) return;
  
  for (i=0;i<ad->n_atom_ene_group;i++) {
    for (j=i;j<ad->n_atom_ene_group;j++) {
      fprintf(prop_fp, "%02dAT_VDW%02d-%02d ", (*no)++, i,j);
      fprintf(prop_fp, "%02dAT_ELC%02d-%02d ", (*no)++, i,j);
    }
  }
}

void ATOM_DATA_atom_ene_prop_out(ATOM_DATA *ad, FILE *prop_fp)
{
  double ene[MAX_ATOM_ENE], tmp[MAX_ATOM_ENE];
  int n_sample_atom_ene, i, j, ia;

  if (ad->n_sample_atom_ene == 0)
    n_sample_atom_ene = 1;   /* for avoiding zero devision */
  else
    n_sample_atom_ene = ad->n_sample_atom_ene;
    
  for (i=0;i<ad->n_atom_ene_group;i++) {
    for (j=i;j<ad->n_atom_ene_group;j++) {
      ene[ATOM_ENE_VDW]=ene[ATOM_ENE_ELEC]=0.0;
      for (ia=0;ia<ad->natom;ia++) {
	if (ad->atom_ene_group[ia] == i) {
	  ene[ATOM_ENE_VDW]  += ad->atom_ene[ia][j][ATOM_ENE_VDW];
	  ene[ATOM_ENE_ELEC] += ad->atom_ene[ia][j][ATOM_ENE_ELEC];
	} else if (ad->atom_ene_group[ia] == j) {
	  /* if i==j, do not come here */
	  ene[ATOM_ENE_VDW]  += ad->atom_ene[ia][i][ATOM_ENE_VDW];
	  ene[ATOM_ENE_ELEC] += ad->atom_ene[ia][i][ATOM_ENE_ELEC];
	}
      }
      /*
      ene[ATOM_ENE_VDW] /=n_sample_atom_ene;
      ene[ATOM_ENE_ELEC]/=n_sample_atom_ene;
      */
#ifdef MPI_RDMD
      MPI_Allreduce(ene, tmp, 2, MPI_DOUBLE, MPI_SUM, mpi.comm);
      ene[ATOM_ENE_VDW]  = tmp[ATOM_ENE_VDW];
      ene[ATOM_ENE_ELEC] = tmp[ATOM_ENE_ELEC];
#endif
      if (prop_fp)
	fprintf(prop_fp, " %13.6e %13.6e", ene[ATOM_ENE_VDW], ene[ATOM_ENE_ELEC]);
    }
  }
}

void ATOM_DATA_write_atom_ene(ATOM_DATA *ad, char *fname)
{
  FILE *fp;
  int n_sample_atom_ene, i, j;

#ifdef MPI_RDMD
  double *p1,*p2;
  int nitem;
  nitem = ad->natom*ad->n_atom_ene_group*MAX_ATOM_ENE;
  p1 = &ad->atom_ene_sum[0][0][0];
  p2 = &ad->atom_ene[0][0][0];
  MPI_Allreduce(p1,p2,nitem,MPI_DOUBLE,MPI_SUM, mpi.comm);
  for (i=0;i<nitem;i++)  p1[i] = p2[i];
  if (!mpi.master)
    return;
#endif
  
  fp = fopen(fname, "w");
  
  if (fp==NULL) {
    lprintf("ERROR: %s: No such atom energy file\n",fname);
    return;
  }
  if (ad->n_sample_atom_ene == 0)
    n_sample_atom_ene = 1;   /* for avoiding zero devision */
  else
    n_sample_atom_ene = ad->n_sample_atom_ene;
  
  for (i=0;i<ad->natom;i++) {
    fprintf(fp, "%6d",i+1);
    for (j=0;j<ad->n_atom_ene_group;j++) {
      ad->atom_ene_sum[i][j][ATOM_ENE_VDW]/=n_sample_atom_ene;
      ad->atom_ene_sum[i][j][ATOM_ENE_ELEC]/=n_sample_atom_ene;
      fprintf(fp," %13.6e %13.6e",
	      ad->atom_ene_sum[i][j][ATOM_ENE_VDW],
	      ad->atom_ene_sum[i][j][ATOM_ENE_ELEC]);
    }
    fprintf(fp,"\n");
  }
  fclose(fp);
}

void ATOM_DATA_n_flex_atom(ATOM_DATA *ad, int *n_flex_atom, int *n_flex_atom_ex)
{
  int i,iex;

  *n_flex_atom=0;
  for (i=0;i<MAX_EX_SYSTEM;i++) n_flex_atom_ex[i]=0;
  
  for (i=0;i<ad->natom;i++) {
    if (!(ad->ex[i].flag & (ATOM_RIGID|ATOM_RATTLE|ATOM_FIXED))) {
      iex = ATOM_FLAG_EX_SYSTEM(ad->ex[i].flag);
      n_flex_atom_ex[iex]++;
      (*n_flex_atom)++;
    }
  }
}

static char *_amino[] = {
  "ALA ",  "CYS ",  "ASP ",  "GLU ",  "PHE ",  "GLY ",  "HIS ",  "HID ",
  "HIE ",  "HIP ",  "ILE ",  "LYS ",  "LEU ",  "MET ",  "ASN ",  "PRO ",
  "GLN ",  "ARG ",  "SER ",  "THR ",  "VAL ",  "TRP ",  "TYR ",  NULL
};

void ATOM_DATA_set_residue_type(ATOM_DATA *ad)
{
  int i,j;

  for (i=0;i<ad->nres;i++) {
    for (j=0;_amino[j]!=NULL;j++) {
      if (strcmp(_amino[j],ad->r[i].name)==0) {
	ad->r[i].flag |= RES_FLAG_AMINO;
	break;
      }
    }
  }
}

void ATOM_DATA_set_node_fatom(ATOM_DATA *ad)
{
  int prev,j;
  
#ifdef MPI_SDMD  
  if (ad->node_atom_n == ad->node_fatom_n) {
    ad->node_fatom_n = emalloc("ATOM_DATA_set_node_fatom",
			       sizeof(int)*ad->natom);
  }
#endif    
  /* setting up fatom array */
  prev = -1;
#ifdef MPI_SDMD  
  for (j = ad->node_atom_h; j>=0; j=ad->node_atom_n[j]) {
#else
  for (j = 0; j<ad->natom; j++) {
#endif    
    if (!(ad->ex[j].flag & (ATOM_RIGID|ATOM_RATTLE|ATOM_FIXED))) {
	if (prev < 0)
	  ad->node_fatom_h = j;
	else
	  ad->node_fatom_n[prev] = j;
	prev = j;
    }
  }
  if (prev < 0)
    ad->node_fatom_h = -1;
  else
    ad->node_fatom_n[prev] = -1;
}


/*
    convert to atom name to atom number  
      e.g. CA.156A -> atom number
*/
int ATOM_DATA_str_to_atom_no(ATOM_DATA *ad, char *name)
{
  char aname[10], rname[10];
  int  i, j;

  if (isdigit(name[0])) {
    i = atoi(name);
    return (--i);
  }
  
  if (sscanf(name, "%[^ \t\n.].%s", aname, rname) != 2) {
    lprintf("  ERROR: %s: Invalid atom name format\n",name);
    marble_exit(1);
  }
  j = ATOM_DATA_str_to_res_no(ad, rname, 0);

  for (i=ad->r[j].start_atom;i<=ad->r[j].end_atom;i++) {
    if (strcmp(aname,ad->a[i].name) == 0) 
      break;
  }
  if (i>ad->r[j].end_atom) {
    lprintf("  ERROR: %s: No atom %s in Residue %s\n",name, aname,rname);
    marble_exit(1);
  }
  return i;
}

int ATOM_DATA_str_to_res_no(ATOM_DATA *ad, char *name, int flag)
{
  char aname[10], chain;
  int  resno;
  int  i, j, err=0;
  int nearest_j, diff;

  if (isdigit(name[0])) {
    if (sscanf(name, "%d%c", &resno, &chain) != 2) {
      if (sscanf(name, "%d", &resno) != 1) {
	err=1;
      }
      chain = ' ';
    }
  } else {
    if (sscanf(name, "%c%d", &chain, &resno) != 2) {
      err=1;
    }
  }
  if (err) {
    lprintf("ERROR: Residue Name Format Error \"%s\"\n",name);
    marble_exit(1);
  }

  nearest_j = -1;
  err = 1;
  for (j=0;j<ad->nres;j++) {
    if (ad->r[j].pdb_chain == chain) {
      if (ad->r[j].pdb_no == resno) return j;
      diff = ad->r[j].pdb_no - resno;
      if (flag == -1 && diff < 0 && (err || (nearest_j < j))) {
	nearest_j = j; 
	err = 0;
      }
      if (flag == 1 && diff > 0 && (err || (nearest_j > j))) {
	nearest_j = j; 
	err = 0;
      }
    }
  }

  if (err) {
    /*
    if (flag) {
      lprintf("  ERROR: Can't find chain \'%c\'",chain);
    } else {
      lprintf("  ERROR: Can't find residue %d in chain \'%c\'",resno,chain);
    }
    */
    lprintf("  ERROR: No residue %d (chain:%c)\n",resno,chain);
    marble_exit(1);
  }
    
  return nearest_j;
}

int ATOM_DATA_str_to_atom_list(ATOM_DATA *ad, char *name, int **plist)
{
  char *fname = "ATOM_DATA_str_to_atom_list";
  char aname[10], rname[10];
  int  i, j, k;
  int *list, n_list;

  if (isdigit(name[0])) {
    n_list = 1;
    list = *plist = emalloc(fname, sizeof(int));
    list[0] = atoi(name) - 1;
    return n_list;
  }
  
  if (sscanf(name, "%[^ \t\n.].%s", aname, rname) != 2) {
    lprintf("  ERROR: %s: Invalid atom name format\n",name);
    marble_exit(1);
  }
  n_list = ATOM_DATA_str_to_res_list(ad, rname, plist, 0);
  list = *plist;

  for (k=0; k < n_list; k++) {
    j = list[k];
    for (i=ad->r[j].start_atom;i<=ad->r[j].end_atom;i++) {
      if (strcmp(aname,ad->a[i].name) == 0) 
	break;
    }
    if (i>ad->r[j].end_atom) {
      lprintf("  ERROR: %s: No atom %s in Residue %s (%d%c)\n",name, aname, ad->r[j].name, ad->r[j].pdb_no, ad->r[j].pdb_chain);
      marble_exit(1);
    }
    list[k] = i;
  }
  return n_list;
}


int ATOM_DATA_str_to_res_list(ATOM_DATA *ad, char *name, int **plist, int flag)
{
  char *fname = "ATOM_DATA_str_to_res_list";
  int err = 0;
  int resno = -1, diff, *list, n_list;
  int i, j, nearest_j;
  char chain, res_name[10];

  if (isdigit(name[0])) {
    if (sscanf(name, "%d%c", &resno, &chain) != 2) {
      if (sscanf(name, "%d", &resno) != 1) {
	err=1;
      }
      chain = ' ';
    }
  } else if (isdigit(name[1])) {
    if (sscanf(name, "%c%d", &chain, &resno) != 2) {
      err=1;
    }
  } else {
    if (sscanf(name, "%s", res_name) != 1) {
      err=1;
    }
  }
	     
  if (err) {
    lprintf("ERROR: Residue Name Format Error \"%s\"\n",name);
    marble_exit(1);
  }

  if (resno >= 0) {
    nearest_j = -1;
    err = 1;
    for (j=0;j<ad->nres;j++) {
      if (ad->r[j].pdb_chain == chain) {
	if (ad->r[j].pdb_no == resno) {
	  nearest_j = j;
	  err=0;
	  break;
	}
	diff = ad->r[j].pdb_no - resno;
	if (flag == -1 && diff < 0 && (err || (nearest_j < j))) {
	  nearest_j = j; 
	  err = 0;
	}
	if (flag == 1 && diff > 0 && (err || (nearest_j > j))) {
	  nearest_j = j; 
	  err = 0;
	}
      }
    }
    if (err) {
      lprintf("  ERROR: No residue %d (chain:%c)\n",resno,chain);
      marble_exit(1);
    }
    n_list = 1;
    list = *plist = emalloc(fname, sizeof(int));
    list[0] = nearest_j;
    return n_list;
  } else {
    n_list = 0;
    for (j=0;j<ad->nres;j++) {
      if (strcmp(ad->r[j].name, res_name) == 0) {
	n_list++;
      }
    }
    if (n_list == 0) {
      lprintf("  ERROR: No such residue: %s\n",res_name);
      marble_exit(1);
    }
    list = *plist = emalloc(fname, sizeof(int)*n_list);
    
    n_list = 0;
    for (j=0;j<ad->nres;j++) {
      if (strcmp(ad->r[j].name, res_name) == 0) {
	list[n_list] = j;	
	n_list++;
      }
    }
    return n_list;
  }
  
}


void ATOM_DATA_set_fixed_atom(ATOM_DATA *ad, int group_no)
{
  int g_id, i;

  if (group_no < 0) return;

  g_id = 1 << group_no;

  for (i=0;i<ad->natom;i++) {
    if (ad->ex[i].group & g_id) {
      ad->ex[i].flag |= ATOM_FIXED;
      ad->v[i].x = 0.0;
      ad->v[i].y = 0.0;
      ad->v[i].z = 0.0;
    }
  }
}

#define PDB_GROUP_NO 31

void ATOM_DATA_check_name_in_group(ATOM_DATA *ad, int g_no,
				   PDB *pdb, int pdb_g_no);


void ATOM_DATA_read_pdb_or_crd_file(ATOM_DATA *ad,
				    char *crd_file, int pdb_mode, 
				    STR_LIST *pdb_group_str,
				    VEC *x0, int x0_n_atom, char *header, int check_g_no)
{
  PDB *pdb;
  FILE *fp;
  char *fname = "ATOM_DATA_read_pdb_or_crd_file";
  int n_atom;
  char buf[100];
  int i;
  double x, y, z;
  
  if (pdb_mode) {
#ifdef MPI
    if (mpi.master) {
#endif
      marble_exit_abort_flag(1);
      pdb = emalloc(fname,sizeof(PDB));
      PDB_init(pdb);
      PDB_read_file(pdb, crd_file);
      PDB_set_group_str_list(pdb, pdb_group_str, PDB_GROUP_NO, header);
      n_atom = PDB_count_group_atoms(pdb, PDB_GROUP_NO);
      if (n_atom != x0_n_atom) {
	lprintf("\nERROR: The number of selected atoms (%d) in pdb_file %s must be the same as the number of selected atoms (%d) in mdat_file.\n",n_atom, crd_file, x0_n_atom);
	marble_abort(1);
      }
      if (check_g_no < 0) {
	lprintf("    Consistency of atom/residue names in pdb file is not checked.\n");
      } else {
	ATOM_DATA_check_name_in_group(ad, check_g_no, pdb, PDB_GROUP_NO);
      }
      PDB_get_coord_in_group(pdb, PDB_GROUP_NO, x0, n_atom);
      PDB_free(pdb);
      marble_exit_abort_flag(0);
#ifdef MPI
    }
    MPI_Bcast(x0, x0_n_atom*3, MPI_DOUBLE, mpi.master_pe, mpi.comm);
#endif    
  } else {
    /* crd_mode */
#if 0
    if ((fp = fopen(crd_file, "r")) == NULL) {
      lprintf("\nERROR: %s: No such file\n", crd_file);
      marble_exit(1);
    }

    fgets(buf, 80, fp);  /* skip header */
    fscanf(fp,"%d", &(n_atom));
    
    if (n_atom != x0_n_atom) {
      lprintf("\nERROR: The atom number (%d) in crd_file %s must be the same as the number of selected atoms (%d) in mdat_file.\n",n_atom, crd_file, x0_n_atom);
      marble_exit(1);
    }
    for (i = 0; i < n_atom; i++) {
      if (fscanf(fp,"%lf%lf%lf", &x,&y,&z) != 3) {
	lprintf("\nERROR: Invalid atom coordinate in crd_file %s\n",crd_file);
	marble_exit(1);
      }
      x0[i].x = x;
      x0[i].y = y;
      x0[i].z = z;
    }
    fclose(fp);
#else
    x0 = ATOM_DATA_get_ref_crd(ad, crd_file);
#endif
  }
}

int ATOM_DATA_match_res_name(char *mdat_res, char *pdb_res)
{
  if (strcmp(mdat_res, pdb_res) == 0)
    return 0;

  if (strcmp(pdb_res, "HIS") == 0) {
    if (strcmp(mdat_res, "HSD") == 0 ||
	strcmp(mdat_res, "HSE") == 0 ||
	strcmp(mdat_res, "HSP") == 0 ||
	strcmp(mdat_res, "HID") == 0 ||
	strcmp(mdat_res, "HIE") == 0 ||
	strcmp(mdat_res, "HIP") == 0) {
      return 0;
    }
  }

  return 1;
}

void ATOM_DATA_check_name_in_group(ATOM_DATA *ad, int g_no,
				   PDB *pdb, int pdb_g_no)
{
  int g_id, i, j;
  int pdb_g_id, pdb_i;
  char aname[100];

  if (g_no < 0 || pdb_g_no < 0) return;

  g_id = 1 << g_no;
  pdb_g_id = 1 << pdb_g_no;

  pdb_i = 0;
  for (i=0;i<ad->natom;i++) {
    if (ad->ex[i].group & g_id) {
      while (pdb_i < pdb->n_atom) {
	if (pdb->al[pdb_i]->group & pdb_g_id) {
	  break;
	}
	pdb_i++;
      }
      if (pdb_i > pdb->n_atom) {
	lprintf("\nERROR: The numbers of selected atoms or residues are different.\n");
	marble_exit(1);
      }
      j=ad->a[i].resno;
      if (strcmp(ad->a[i].name, pdb->al[pdb_i]->name) != 0 ||
	  ATOM_DATA_match_res_name(ad->r[j].name, pdb->al[pdb_i]->res_name) != 0) {
	lprintf("\nERROR: The names of selected atoms or residues are different.\n");
	lprintf("  [%s:%s]!=[%s:%s]\n",ad->a[i].name, ad->r[j].name, 
		pdb->al[pdb_i]->name,pdb->al[pdb_i]->res_name);
	lprintf("mdat file:\n");
	ATOM_DATA_print_atom(ad, i);
	lprintf("\n");
	lprintf("PDB file:\n");
	PDB_print_atom(pdb, pdb_i);
	marble_exit(1);
      }
      pdb_i++;
    }
  }
}
  

VEC *ATOM_DATA_get_ref_crd(ATOM_DATA *ad, char *fname)
{
  REF_CRD *rc;
  int err = 0;
  char *func = "ATOM_DATA_get_ref_crd";
  
  for (rc = ad->ref_crd; rc != NULL; rc=rc->next) {
    if (strcmp(rc->fname, fname) == 0) {
      return rc->x;
    }
  }
  rc = emalloc(func, sizeof(REF_CRD));
  strcpy(rc->fname, fname);

#ifdef MPI
  if (mpi.master) {
    err = ATOM_DATA_read_ref_crd_master(ad, fname, rc);
  }
  MPI_Bcast(&err, 1, MPI_INT, mpi.master_pe, mpi.comm);
  if (err) 
    marble_exit(1);

  if (!mpi.master) 
    rc->x = emalloc(func, sizeof(VEC)*ad->natom);
  
  MPI_Bcast(rc->x, ad->natom*3, MPI_DOUBLE, mpi.master_pe, mpi.comm);
  
#else
  if (ATOM_DATA_read_ref_crd_master(ad, fname, rc))
    marble_exit(1);
#endif

  rc->next = ad->ref_crd;
  ad->ref_crd = rc;

  return rc->x;
}

int ATOM_DATA_read_ref_crd_master(ATOM_DATA *ad, char *fname, REF_CRD *rc)
{
  int n_atom, i;
  FILE *fp;
  char buf[100];
  char *func = "ATOM_DATA_read_get_ref_master";

  if ((fp = fopen(fname, "r")) == NULL) {
    lprintf("ERROR: no such file %s.\n", fname);
    return 1;
  }
  fgets(buf, 80, fp);  /* skip header */
  fscanf(fp,"%d", &(n_atom));
  if (n_atom != ad->natom) {
    lprintf("ERROR: Inconsistent atom number in crd file %s\n",fname);
    return 1;
  }

  rc->x = emalloc(func, sizeof(VEC)*n_atom);

  for (i = 0; i < n_atom; i++) {
    if (fscanf(fp,"%lf%lf%lf", &(rc->x[i].x),&(rc->x[i].y),&(rc->x[i].z)) != 3) {
      lprintf("ERROR: Invalid atom coordinate in crd file %s\n",fname);
      return 1;
    }
  }
  fclose(fp);
  
  return 0;
}
