/*
 * 
 * This source code is part of 
 *   MARBLE (MoleculAR simulation package for BiomoLEcules)
 * 
 * Written by Mitsunori Ikeguchi
 * Copyright (c) 2012 Yokohama City University
 *  
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "misc.h"
#include "atom.h"
#include "angle.h"

void ANGLE_DATA_init(ANGLE_DATA *and)
{
  and->n_angle = and->n_angle_h = 0;
  and->angles = NULL;

  and->n_angle_type = 0;
  and->angle_type = NULL;
}

void ANGLE_DATA_finalize(ANGLE_DATA *and)
{
  if (and->angles) free(and->angles);
  if (and->angle_type) free(and->angle_type);
  ANGLE_DATA_init(and);
}

void angle_energy_force(ANGLE_DATA *angle_data, ATOM_DATA *atom_data,
                        double *energy, double *energy_ub)
{
  int i;
  double k,th0,ax,ay,az,bx,by,bz,aa,bb,ab,len_ab,F, e, cc;
  double cx,cy,cz,len_c;
  double dx1,dy1,dz1,dx3,dy3,dz3,th;

  int atom1, atom2, atom3, type;
  VEC *x, *f;
  
  x = atom_data->x; 
  f = atom_data->f; 

#ifdef MPI_RDMD
  for (i = angle_data->start_task; i <= angle_data->end_task; i++) {
#elif defined(MPI_SDMD)
  for (i = angle_data->head; i >=0; i=angle_data->angles[i].next) {
#else  /* MPI_RDMD */
  for (i=0; i < angle_data->n_angle; i++) {
#endif /* MPI_RDMD */
#ifdef MPI_SDMD
    /* if (angle_data->angles[i].flag & ANGLE_OTHER_NODE) continue; */
#endif    
    type = angle_data->angles[i].type;
    k    = angle_data->angle_type[type].k;
    th0  = angle_data->angle_type[type].th0;
    atom1 = angle_data->angles[i].atom1;
    atom2 = angle_data->angles[i].atom2;
    atom3 = angle_data->angles[i].atom3;

    ax = x[atom1].x - x[atom2].x;
    ay = x[atom1].y - x[atom2].y;
    az = x[atom1].z - x[atom2].z;
    bx = x[atom3].x - x[atom2].x;
    by = x[atom3].y - x[atom2].y;
    bz = x[atom3].z - x[atom2].z;

    ab = ax*bx + ay*by + az*bz;
    aa = ax*ax + ay*ay + az*az;
    bb = bx*bx + by*by + bz*bz;
    len_ab = sqrt(aa) * sqrt(bb);
    cc = ab/len_ab;
    th = acos(cc);
    e = k * (th - th0) * (th - th0);
    *energy += e;

#if 0  /* no angle atom_ene */   
    if (atom_data->atom_ene_flag) {
      e /= 3.0;
      atom_data->atom_ene[atom1].type[ATOM_ANGLE_ENE] += e;
      atom_data->atom_ene[atom2].type[ATOM_ANGLE_ENE] += e;
      atom_data->atom_ene[atom3].type[ATOM_ANGLE_ENE] += e;
    }
#endif    
    /*
    printf("%d %e\n",i, k * (th - th0) * (th - th0));
    */

    if ( cc >= 0.999 || cc <= -0.999) {
      if ( th0 < 0.001 ) {
	F =  2.0*k*(1.0+(th - th0)*(th - th0)/6.0);
      } else if ( M_PI-th0 < 0.001 ) {
	F = -2.0*k*(1.0+(th - th0)*(th - th0)/6.0);
      } else {
	F = 2.0*k*(th - th0)/sqrt(1.0-cc*cc+2.22045e-16);
      }
      dx1 = F*(bx-ab*ax/aa)/len_ab;
      dy1 = F*(by-ab*ay/aa)/len_ab;
      dz1 = F*(bz-ab*az/aa)/len_ab;
      dx3 = F*(ax-ab*bx/bb)/len_ab;
      dy3 = F*(ay-ab*by/bb)/len_ab;
      dz3 = F*(az-ab*bz/bb)/len_ab;
    } else {
      cx = ay*bz - az*by;
      cy = az*bx - ax*bz;
      cz = ax*by - ay*bx;
      len_c = sqrt(cx*cx + cy*cy + cz*cz);
      cx /= len_c; 
      cy /= len_c; 
      cz /= len_c;

      F = -2.0*k*(th - th0);
      dx1 = F*(ay*cz-az*cy)/aa;
      dy1 = F*(az*cx-ax*cz)/aa;
      dz1 = F*(ax*cy-ay*cx)/aa;
      dx3 =-F*(by*cz-bz*cy)/bb;
      dy3 =-F*(bz*cx-bx*cz)/bb;
      dz3 =-F*(bx*cy-by*cx)/bb;
    }
    
    f[atom1].x += dx1;
    f[atom1].y += dy1;
    f[atom1].z += dz1;
    f[atom3].x += dx3;
    f[atom3].y += dy3;
    f[atom3].z += dz3;
    f[atom2].x -= dx1 + dx3;
    f[atom2].y -= dy1 + dy3;
    f[atom2].z -= dz1 + dz3;

    k    = angle_data->angle_type[type].kub;
    if (k!=0.0) {
      th0  = angle_data->angle_type[type].s0;
      
      dx1 = x[atom3].x - x[atom1].x;
      dy1 = x[atom3].y - x[atom1].y;
      dz1 = x[atom3].z - x[atom1].z;
      
      th = sqrt(dx1*dx1+dy1*dy1+dz1*dz1);
      e = k * (th - th0) * (th - th0);
      *energy_ub += e;

      /*
      lprintf("%f %f %f %d %d %d\n", k, th0, e, atom1+1, atom2+1, atom3+1);
      */
      
      F = 2.0 * k * (th - th0) / th;
      f[atom1].x += dx1 * F;
      f[atom1].y += dy1 * F;
      f[atom1].z += dz1 * F;
      f[atom3].x -= dx1 * F;
      f[atom3].y -= dy1 * F;
      f[atom3].z -= dz1 * F;
    }
  }
  
}


void print_angles(FILE *fp, ANGLE_DATA *ad)
{
  int i;

  for (i = 0; i < ad->n_angle; i++) {
    fprintf(fp, "%d %d %d %d\n", ad->angles[i].atom1,
            ad->angles[i].atom2, ad->angles[i].atom3,
	    ad->angles[i].type);
  }
}

void ANGLE_DATA_omit_data_for_rigid_mol(ANGLE_DATA *and, ATOM_DATA *ad)
{
  int i, atom1, atom2, atom3, mol1, mol2, mol3, pack_to;

  pack_to=0;
  for (i=0;i<and->n_angle;i++) {
    atom1 = and->angles[i].atom1;
    atom2 = and->angles[i].atom2;
    atom3 = and->angles[i].atom3;

    if ((ad->ex[atom1].flag & ATOM_RIGID) &&
	(ad->ex[atom2].flag & ATOM_RIGID) &&
	(ad->ex[atom3].flag & ATOM_RIGID)) {

      mol1 = ATOM_RMOL(ad,atom1);
      mol2 = ATOM_RMOL(ad,atom2);
      mol3 = ATOM_RMOL(ad,atom3);

      if (mol1==mol2&&mol1==mol3)
	continue;
    }

    if ((ad->ex[atom1].flag & ATOM_FIXED) &&
	(ad->ex[atom2].flag & ATOM_FIXED) &&
	(ad->ex[atom3].flag & ATOM_FIXED)) continue;
    
    if (i != pack_to) {
      and->angles[pack_to] = and->angles[i];
    }
    pack_to++;
  }
  lprintf("  Number of angles: %d -> %d\n", and->n_angle, pack_to);
  and->n_angle = pack_to;
}

double ANGLE_DATA_standard_angle(ANGLE_DATA *and, int atom1, int atom2, int atom3)
{
  int i;
  for (i=0;i<and->n_angle;i++) {
    if ((and->angles[i].atom1 == atom1 &&
	 and->angles[i].atom2 == atom2 &&
	 and->angles[i].atom3 == atom3) ||
	(and->angles[i].atom1 == atom3 &&
	 and->angles[i].atom2 == atom2 &&
	 and->angles[i].atom3 == atom1)) {
      return and->angle_type[and->angles[i].type].th0;
    }
  }
  return 0.0;
}

void ANGLE_DATA_scale_angle_k(ANGLE_DATA *and, double scale)
{
  int i;

  lprintf("Scale force constants of angles: %lf\n\n", scale);
  for (i=0;i<and->n_angle_type;i++) {
    and->angle_type[i].k *= scale;
  }
}

void ANGLE_DATA_overwrite_angle_k(ANGLE_DATA *and, double val)
{
  int i;

  lprintf("Overwrite force constants of angles: %lf\n\n", val);
  for (i=0;i<and->n_angle_type;i++) {
    and->angle_type[i].k = val;
  }
}
